#ifndef __ARCH_IO_H
#define __ARCH_IO_H

#include <SYS/TYPES.H>
#include <SPAD/LIST.H>

__BEGIN_DECLS

typedef __u16 io_t;

#define IO_ID		"IO_%X"
#define MEM_ID		"MEM_%lX"
#define IO_FORMAT	"%X"

static __finline__ __u8 io_inb(io_t port)
{
	__u8 ret;
	__asm__ volatile ("INB %1, %0":"=a"(ret):"Nd"((__u16)port));
	return ret;
}

static __finline__ __u16 io_inw(io_t port)
{
	__u16 ret;
	__asm__ volatile ("INW %1, %0":"=a"(ret):"Nd"((__u16)port));
	return ret;
}

static __finline__ __u32 io_inl(io_t port)
{
	__u32 ret;
	__asm__ volatile ("INL %1, %0":"=a"(ret):"Nd"((__u16)port));
	return ret;
}

static __finline__ void io_outb(io_t port, __u8 value)
{
	__asm__ volatile ("OUTB %0, %1": :"a"(value),"Nd"((__u16)port));
}

static __finline__ void io_outw(io_t port, __u16 value)
{
	__asm__ volatile ("OUTW %0, %1": :"a"(value),"Nd"((__u16)port));
}

static __finline__ void io_outl(io_t port, __u32 value)
{
	__asm__ volatile ("OUTL %0, %1": :"a"(value),"Nd"((__u16)port));
}

static __finline__ void __slowdown_io(void)
{
	__asm__ volatile ("OUTB %%AL, $0x80": :);
}

static __finline__ __u8 io_inb_p(io_t port)
{
	__u8 r = io_inb(port);
	__slowdown_io();
	return r;
}

static __finline__ __u16 io_inw_p(io_t port)
{
	__u16 r = io_inw(port);
	__slowdown_io();
	return r;
}

static __finline__ __u32 io_inl_p(io_t port)
{
	__u32 r = io_inl(port);
	__slowdown_io();
	return r;
}

static __finline__ void io_outb_p(io_t port, __u8 value)
{
	io_outb(port, value);
	__slowdown_io();
}

static __finline__ void io_outw_p(io_t port, __u16 value)
{
	io_outw(port, value);
	__slowdown_io();
}

static __finline__ void io_outl_p(io_t port, __u32 value)
{
	io_outl(port, value);
	__slowdown_io();
}

static __finline__ void io_insb(io_t port, void *ptr, unsigned count)
{
	__asm__ volatile ("REP ; INSB":"=D"(ptr),"=c"(count):"d"((__u16)port),"0"(ptr),"1"(count):"memory");
}

static __finline__ void io_insw(io_t port, void *ptr, unsigned count)
{
	__asm__ volatile ("REP ; INSW":"=D"(ptr),"=c"(count):"d"((__u16)port),"0"(ptr),"1"(count):"memory");
}

static __finline__ void io_insl(io_t port, void *ptr, unsigned count)
{
	__asm__ volatile ("REP ; INSL":"=D"(ptr),"=c"(count):"d"((__u16)port),"0"(ptr),"1"(count):"memory");
}

static __finline__ void io_outsb(io_t port, void *ptr, unsigned count)
{
	__asm__ volatile ("REP ; OUTSB":"=S"(ptr),"=c"(count):"d"((__u16)port),"0"(ptr),"1"(count):"memory");
}

static __finline__ void io_outsw(io_t port, void *ptr, unsigned count)
{
	__asm__ volatile ("REP ; OUTSW":"=S"(ptr),"=c"(count):"d"((__u16)port),"0"(ptr),"1"(count):"memory");
}

static __finline__ void io_outsl(io_t port, void *ptr, unsigned count)
{
	__asm__ volatile ("REP ; OUTSL":"=S"(ptr),"=c"(count):"d"((__u16)port),"0"(ptr),"1"(count):"memory");
}

#define io_to_cpu_barrier()	__asm__ ("" ::: "memory")
#define cpu_to_io_barrier()	__asm__ ("" ::: "memory")

#define IO_SPACE_LIMIT 0xffffU

/* __u16 will least likely alias anything */

static __finline__ __u8 mmio_inb(__u8 *port)
{
	__u8 val;
	__asm__ volatile ("MOVB %1, %0":"=q"(val):"m"(*(__u16 *)port));
	return val;
}

static __finline__ __u16 mmio_inw(__u8 *port)
{
	__u16 val;
	__asm__ volatile ("MOVW %1, %0":"=r"(val):"m"(*(__u16 *)port));
	return val;
}

static __finline__ __u32 mmio_inl(__u8 *port)
{
	__u32 val;
	__asm__ volatile ("MOVL %1, %0":"=r"(val):"m"(*(__u16 *)port));
	return val;
}

static __finline__ void mmio_outb(__u8 *port, __u8 val)
{
	__asm__ volatile ("MOVB %0, %1"::"q"(val),"m"(*(__u16 *)port));
}

static __finline__ void mmio_outw(__u8 *port, __u16 val)
{
	__asm__ volatile ("MOVW %0, %1"::"r"(val),"m"(*(__u16 *)port));
}

static __finline__ void mmio_outl(__u8 *port, __u32 val)
{
	__asm__ volatile ("MOVL %0, %1"::"r"(val),"m"(*(__u16 *)port));
}

#if 0
/* Is it really needed? Linux doesn't have it... */
#define mmio_to_cpu_barrier()	__asm__ ("" ::: "memory")
#define cpu_to_mmio_barrier()	__asm__ ("" ::: "memory")
#endif

void KERNEL$DI(void);
void KERNEL$EI(void);
int KERNEL$SAVE_INT_STATE(void);
void KERNEL$RESTORE_INT_STATE(int state);

typedef struct {
	LIST_ENTRY list;
	io_t start;
	io_t len;
	char *name;
} IO_RANGE;

int KERNEL$REGISTER_IO_RANGE(IO_RANGE *io);
void KERNEL$UNREGISTER_IO_RANGE(IO_RANGE *io);

__END_DECLS

#endif
