#include <STRING.H>
#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <ARCH/IO.H>
#include <ARCH/TIME.H>
#include <KERNEL/ASM.H>
#include <KERNEL/SEG.H>

#include <ARCH/IRQ.H>
#include <KERNEL/IRQ.H>

static unsigned read_4bytes(char *p)
{
	return (unsigned char)p[0] + ((unsigned char)p[1] << 8) + ((unsigned char)p[2] << 16) + ((unsigned char)p[3] << 24);
}

static void write_4bytes(char *p, unsigned v)
{
	p[0] = v & 0xff;
	p[1] = (v >> 8) & 0xff;
	p[2] = (v >> 16) & 0xff;
	p[3] = (v >> 24) & 0xff;
}

static void __REQUEST_RT_IRQ(int irq, IRQ_STUB *call, void *data)
{
	int intstate;
	if ((unsigned)irq >= NO_OF_IRQS || irq == 2) KERNEL$SUICIDE("__REQUEST_RT_IRQ: UNACCEPTABLE IRQ NUMBER: %d", irq);
	if (irq > 2) irq--;
	intstate = KERNEL$SAVE_INT_STATE();
	KERNEL$DI();
	memcpy(IRQ_STUBS + irq, RT_STUB_TEMPLATES + irq, sizeof(IRQ_TEMPLATE));
	write_4bytes(IRQ_STUBS[irq].c + RT_CALL_PATCHES[irq], read_4bytes(IRQ_STUBS[irq].c + RT_CALL_PATCHES[irq]) + (unsigned)call);
	write_4bytes(IRQ_STUBS[irq].c + RT_PTR_PATCHES[irq], (unsigned)data);
	KERNEL$RESTORE_INT_STATE(intstate);
}

static void __REQUEST_AST_IRQ(int irq, AST *ast)
{
	int intstate;
	if ((unsigned)irq >= NO_OF_IRQS || irq == 2) KERNEL$SUICIDE("__REQUEST_AST_IRQ: UNACCEPTABLE IRQ NUMBER: %d", irq);
	if (irq > 2) irq--;
	intstate = KERNEL$SAVE_INT_STATE();
	KERNEL$DI();
	memcpy(IRQ_STUBS + irq, AST_STUB_TEMPLATES + irq, sizeof(IRQ_TEMPLATE));
	write_4bytes(IRQ_STUBS[irq].c + AST_PATCHES[irq], (unsigned)ast);
	KERNEL$RESTORE_INT_STATE(intstate);
}

extern void I8259_DISABLE_0(void);
extern void I8259_DISABLE_1(void);
extern void I8259_DISABLE_3(void);
extern void I8259_DISABLE_4(void);
extern void I8259_DISABLE_5(void);
extern void I8259_DISABLE_6(void);
extern void I8259_DISABLE_7(void);
extern void I8259_DISABLE_8(void);
extern void I8259_DISABLE_9(void);
extern void I8259_DISABLE_10(void);
extern void I8259_DISABLE_11(void);
extern void I8259_DISABLE_12(void);
extern void I8259_DISABLE_13(void);
extern void I8259_DISABLE_14(void);
extern void I8259_DISABLE_15(void);
extern void I8259_ENABLE_0(void);
extern void I8259_ENABLE_1(void);
extern void I8259_ENABLE_3(void);
extern void I8259_ENABLE_4(void);
extern void I8259_ENABLE_5(void);
extern void I8259_ENABLE_6(void);
extern void I8259_ENABLE_7(void);
extern void I8259_ENABLE_8(void);
extern void I8259_ENABLE_9(void);
extern void I8259_ENABLE_10(void);
extern void I8259_ENABLE_11(void);
extern void I8259_ENABLE_12(void);
extern void I8259_ENABLE_13(void);
extern void I8259_ENABLE_14(void);
extern void I8259_ENABLE_15(void);

extern char I8259_PATCH_0[];
extern char I8259_PATCH_1[];
extern char I8259_PATCH_3[];
extern char I8259_PATCH_4[];
extern char I8259_PATCH_5[];
extern char I8259_PATCH_6[];
extern char I8259_PATCH_7[];
extern char I8259_PATCH_8[];
extern char I8259_PATCH_9[];
extern char I8259_PATCH_10[];
extern char I8259_PATCH_11[];
extern char I8259_PATCH_12[];
extern char I8259_PATCH_13[];
extern char I8259_PATCH_14[];
extern char I8259_PATCH_15[];

extern __u8 I8259MASK_1;
extern __u8 I8259MASK_2;

static __const__ IRQ_CONTROL IRQ_CONTROLS[NO_OF_IRQS] = {
	{ I8259_DISABLE_0, I8259_ENABLE_0 },
	{ I8259_DISABLE_1, I8259_ENABLE_1 },
	{ NULL, NULL },
	{ I8259_DISABLE_3, I8259_ENABLE_3 },
	{ I8259_DISABLE_4, I8259_ENABLE_4 },
	{ I8259_DISABLE_5, I8259_ENABLE_5 },
	{ I8259_DISABLE_6, I8259_ENABLE_6 },
	{ I8259_DISABLE_7, I8259_ENABLE_7 },
	{ I8259_DISABLE_8, I8259_ENABLE_8 },
	{ I8259_DISABLE_9, I8259_ENABLE_9 },
	{ I8259_DISABLE_10, I8259_ENABLE_10 },
	{ I8259_DISABLE_11, I8259_ENABLE_11 },
	{ I8259_DISABLE_12, I8259_ENABLE_12 },
	{ I8259_DISABLE_13, I8259_ENABLE_13 },
	{ I8259_DISABLE_14, I8259_ENABLE_14 },
	{ I8259_DISABLE_15, I8259_ENABLE_15 },
};

static char * __const__ IRQ_PATCHES[NO_OF_IRQS] = {
	I8259_PATCH_0,
	I8259_PATCH_1,
	NULL,
	I8259_PATCH_3,
	I8259_PATCH_4,
	I8259_PATCH_5,
	I8259_PATCH_6,
	I8259_PATCH_7,
	I8259_PATCH_8,
	I8259_PATCH_9,
	I8259_PATCH_10,
	I8259_PATCH_11,
	I8259_PATCH_12,
	I8259_PATCH_13,
	I8259_PATCH_14,
	I8259_PATCH_15,
};

static char intstate = -1;

void KERNEL$DI_HW(void)
{
	int i;
	if (__unlikely(intstate == -2)) return;
	if (__unlikely(intstate != -1))
		KERNEL$SUICIDE("KERNEL$DI_HW CALLED TWICE WITHOUT KERNEL$EI_HW");
	for (i = 0; i < NO_OF_IRQS; i++) if (i != 2) IRQ_CONTROLS[i].mask();
	/* This is not as easy as it seems: We must first mask interrupts on
	   the PIC and then disable processor interrupts, otherwise spurious
	   int 7 would be called in real mode (and that means crash, because
	   interrupt vectors differ).

	   If the interrupts were disabled on processor (that only happens
	   during crash screen restore), we must not restore hardware mask in
	   KERNEL$EI_HW() because that would crash. eoi() must be called with
	   enabled interrupts.

	   During crash screen restore we enable interrupts for a little moment
	   to flush spurious int 7, so that it won't be reflected to real mode.
	*/
	intstate = KERNEL$SAVE_INT_STATE();
	KERNEL$EI();
	KERNEL$DI();
}

void KERNEL$EI_HW(void)
{
	int i;
	if (__unlikely(intstate == -2)) return;
	if (__likely(intstate)) {
		intstate = -1;
		KERNEL$EI();
		for (i = 0; i < NO_OF_IRQS; i++) if (i != 2) IRQ_CONTROLS[i].eoi();
	} else {
		intstate = -2;
	}
}

static void IRQC_STARTUP(int irq)
{
	if ((unsigned)irq >= NO_OF_IRQS || irq == 2) KERNEL$SUICIDE("IRQC_STARTUP: BAD IRQ %d", irq);
	IRQ_PATCHES[irq][0] = 0xEB;
	IRQ_PATCHES[irq][1] = 0xFB;
}

static void IRQC_SHUTDOWN(int irq, int masked)
{
	if ((unsigned)irq >= NO_OF_IRQS || irq == 2) KERNEL$SUICIDE("IRQC_SHUTDOWN: BAD IRQ %d", irq);
	if (!masked) {
		if (irq < 8 ? I8259MASK_1 & (1 << irq) : I8259MASK_2 & (1 << (irq - 8))) KERNEL$SUICIDE("IRQC_SHUTDOWN: IRQ %d NOT UNMASKED BEFORE RELEASE", irq);
		IRQ_CONTROLS[irq].mask();
	}
}

/* Copied from Linux. Copyright (c) I don't know who. */

static void IRQC_INIT(void)
{
	static IO_RANGE iorange1;
	static IO_RANGE iorange2;
	int i;
	int r;
	iorange1.start = 0x20;
	iorange1.len = 2;
	iorange1.name = "IRQ MASTER";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&iorange1))) {
		__critical_printf("CAN'T REGISTER IO RANGE FOR IRQ MASTER: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	iorange2.start = 0xa0;
	iorange2.len = 2;
	iorange2.name = "IRQ MASTER";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&iorange2))) {
		__critical_printf("CAN'T REGISTER IO RANGE FOR IRQ SLAVE: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	/*
	 * io_outb_p - this has to work on a wide range of PC hardware.
	 */
	io_outb_p(0x20, 0x11);	/* ICW1: select 8259A-1 init */
	io_outb_p(0x21, 0x20 + 0);	/* ICW2: 8259A-1 IR0-7 mapped to 0x20-0x27 */
	io_outb_p(0x21, 0x04);	/* 8259A-1 (the master) has a slave on IR2 */
	io_outb_p(0x21, 0x01);	/* master expects normal EOI */

	io_outb_p(0xA0, 0x11);	/* ICW1: select 8259A-2 init */
	io_outb_p(0xA1, 0x20 + 8);	/* ICW2: 8259A-2 IR0-7 mapped to 0x28-0x2f */
	io_outb_p(0xA1, 0x02);	/* 8259A-2 is a slave on master's IR2 */
	io_outb_p(0xA1, 0x01);	/* (slave's support for AEOI in flat mode
				    is to be investigated) */

	KERNEL$UDELAY(100);	/* wait for 8259A to initialize */

	/* mask all of 8259A-1 and 8259A-2 */
	for (i = 0; i < NO_OF_IRQS; i++) if (i != 2) IRQ_CONTROLS[i].mask();
}

#define MAX_SHARED_IRQS	16

#define irq_str_len	64

#define irqh_no		0
#define irqh_ast	1
#define irqh_rt		2

struct irq_handler {
	int type;
	IRQ_CONTROL *ctrl;
	char string[irq_str_len];
};

struct irq_handler_2 {
	IRQ_STUB *call;
	void *data;
};

#define irqt_free	0
#define	irqt_exclusive	1
#define irqt_shared	2

struct irq_status {
	int type;
	int n_handlers;
	int pending_handlers;
};

static struct irq_handler handlers[NO_OF_IRQS][MAX_SHARED_IRQS];
static struct irq_handler_2 handlers_2[NO_OF_IRQS][MAX_SHARED_IRQS];

static struct irq_status irqs[NO_OF_IRQS];

DECL_RT_IRQ_HANDLER(UNHANDLED_IRQ)
{
	int irq = (int)DATA;
	/*if (__unlikely(irq != 7) && irq != 15) __critical_printf("UNHANDLED IRQ %d\n", irq);*/
	KERNEL$SUICIDE("UNHANDLED_IRQ: IRQ %d", irq);
	IRQ_RETURN;
}

void IRQ_INIT(void)
{
	int i;
	memset(handlers, 0, sizeof(handlers));
	memset(irqs, 0, sizeof(irqs));
	IRQC_INIT();
	for (i = 0; i < NO_OF_IRQS; i++) if (i != 2) {
		__REQUEST_RT_IRQ(i, &UNHANDLED_IRQ, (void *)i);
	}
	KERNEL$EI();
}

AST *multi_irq_asts[NO_OF_IRQS][MAX_SHARED_IRQS];
int multi_irq_n_asts[NO_OF_IRQS];
AST multi_irq_ast[NO_OF_IRQS];

DECL_RT_IRQ_HANDLER(dispatch_shirq)
{
	int irq = (unsigned long)DATA;
	int i;
	void *ast;
	int n_asts;
	n_asts = 0;
	for (i = 0; i < irqs[irq].n_handlers; i++) {
		if (__unlikely(handlers_2[irq][i].call != NULL)) {
			__u32 dummy1, dummy2;
			__asm__ volatile ("CALL *%3":"=a"(ast),"=d"(dummy1),"=c"(dummy2):"r"(handlers_2[irq][i].call),"a"(handlers_2[irq][i].data),"c"(SEG_KDATA):"cc","memory");
			if (ast) multi_irq_asts[irq][n_asts++] = ast;
		} else {
			IRQ_CONTROLS[irq].mask();
			multi_irq_asts[irq][n_asts++] = handlers_2[irq][i].data;
		}
	}
	if (n_asts) {
		multi_irq_n_asts[irq] = n_asts;
		IRQ_POST_AST(&multi_irq_ast[irq]);
	}
	IRQ_RETURN;
}

DECL_AST(MULTI_IRQ_AST, SPL_ALMOST_TOP, AST)
{
	int i;
	int irq = RQ - multi_irq_ast;
	for (i = 0; i < multi_irq_n_asts[irq] - 1; i++) {
		CALL_AST(multi_irq_asts[irq][i]);
	}
	RETURN_AST(multi_irq_asts[irq][i]);
}

char *KERNEL$REQUEST_IRQ(int irq, int flags, IRQ_STUB *call, void *data, IRQ_CONTROL *ctrl, __const__ char *desc)
{
	struct irq_handler *h;
	struct irq_handler_2 *h2;
	int spl;
	int enable;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), spl = KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$REQUEST_IRQ AT SPL %08X", KERNEL$SPL);
	/*if ((flags & (IRQ_SHARED | IRQ_MASK)) == (IRQ_SHARED | IRQ_MASK)) KERNEL$SUICIDE("KERNEL$REQUEST_IRQ: CAN'T USE BOTH IRQ_MASK AND IRQ_SHARED");*/
	if (__unlikely(irq < 0) || __unlikely(irq >= NO_OF_IRQS) || __unlikely(irq == 2)) return "UNACCEPTABLE IRQ NUMBER";
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(irqs[irq].type == irqt_exclusive) || (__likely(irqs[irq].type == irqt_shared) && __unlikely(!(flags & IRQ_SHARED)))) {
		LOWER_SPLX(spl);
		return "IRQ ALREADY ALLOCATED";
	}
	if (__unlikely(irqs[irq].n_handlers >= MAX_SHARED_IRQS)) {
		LOWER_SPLX(spl);
		return "TOO MANY SHARED IRQS";
	}
	if (__unlikely(!(flags & IRQ_SHARED))) {
		irqs[irq].type = irqt_exclusive;
	} else {
		irqs[irq].type = irqt_shared;
	}
	h = &handlers[irq][irqs[irq].n_handlers];
	h2 = &handlers_2[irq][irqs[irq].n_handlers];
	KERNEL$DI();
	if (!(flags & IRQ_AST_HANDLER)) {
		h->type = irqh_rt;
		h2->call = call;
		h2->data = data;
	} else {
		if (call) KERNEL$SUICIDE("KERNEL$REQUEST_IRQ: NO NULL CALL FOR AST IRQ");
		h->type = irqh_ast;
		h2->call = NULL;
		h2->data = data;
	}
	h->ctrl = ctrl;
	strlcpy(h->string, desc, irq_str_len);
	irqs[irq].n_handlers++;
	enable = 0;
	if (irqs[irq].n_handlers == 1) {
		if (!(flags & IRQ_AST_HANDLER)) __REQUEST_RT_IRQ(irq, call, data);
		else __REQUEST_AST_IRQ(irq, data);
		if (!(flags & IRQ_MASK)) enable = 1;
		IRQC_STARTUP(irq);
	} else {
		multi_irq_ast[irq].fn = MULTI_IRQ_AST;
		__REQUEST_RT_IRQ(irq, &dispatch_shirq, (void *)irq);
		if (flags & IRQ_MASK) IRQ_CONTROLS[irq].mask();
	}
	memcpy(ctrl, IRQ_CONTROLS + irq, sizeof(IRQ_CONTROL));
	KERNEL$EI();
	if (enable) IRQ_CONTROLS[irq].eoi();
	LOWER_SPLX(spl);
	return NULL;
}

void KERNEL$RELEASE_IRQ(IRQ_CONTROL *ctrl, int flags)
{
	void (*eoi)(void);
	struct irq_handler *h;
	struct irq_handler_2 *h2;
	int spl;
	int i, irq;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), spl = KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$RELEASE_IRQ AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_VSPACE);
	for (irq = 0; irq < NO_OF_IRQS; irq++) for (i = 0; i < irqs[irq].n_handlers; i++) {
		if (__unlikely((h = &handlers[irq][i])->ctrl == ctrl)) goto found;
	}
	KERNEL$SUICIDE("KERNEL$RELEASE_IRQ: AN ATTEMPT TO RELEASE UNHANDLED IRQ (CONTROL = %08lX, MASK = %08lX, EOI = %08lX)", (long)ctrl, (long)ctrl->mask, (long)ctrl->eoi);
	found:
	h2 = &handlers_2[irq][i];
	KERNEL$DI();
	eoi = NULL;
	if (__likely(irqs[irq].n_handlers == 1)) {
		IRQC_SHUTDOWN(irq, flags & IRQ_RELEASE_MASKED);
		__REQUEST_RT_IRQ(irq, &UNHANDLED_IRQ, (void *)irq);
		irqs[irq].type = irqt_free;
	} else {
		if (__unlikely(flags & IRQ_RELEASE_MASKED)) eoi = ctrl->eoi;
	}
	memmove(h, h + 1, sizeof(struct irq_handler) * (irqs[irq].n_handlers - i - 1));
	memmove(h2, h2 + 1, sizeof(struct irq_handler_2) * (irqs[irq].n_handlers - i - 1));
	irqs[irq].n_handlers--;
	KERNEL$EI();
	if (eoi) eoi();
	LOWER_SPLX(spl);
}



