#include <ARCH/CPU.H>
#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <SPAD/TIMER.H>
#include <SYS/TYPES.H>
#include <SPAD/SYSLOG.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/ASM.H>

#define MCA_CHECK_TIME		(10 * JIFFIES_PER_SECOND)
#define MCA_READ_RETRIES	5
#define MAX_MCES		5

void MACHINE_INIT(void);
int MCE_HANDLER(int kernel);

static int cpu_has_mca;
static int mce_first_read_bank;
static int mce_first_write_bank;
static int mce_banks;
static int mce_extended_msrs;

static TIMER MCA_TIMER;
static char addr[__MAX_STR_LEN];
static char misc[__MAX_STR_LEN];

static int MCA_CHECK_FOR_PCC(void)
{
	int i;
	for (i = mce_first_read_bank; i < mce_banks; i++) {
		__u64 st;
		if (__unlikely(KERNEL$READ_MSR(IA32_MC0_STATUS + i * 4, &st)))
			continue;
		if (__likely(!(st & IA32_MC_STATUS_VAL)))
			continue;
		if (__unlikely((st & IA32_MC_STATUS_PCC) != 0))
			return 1;
	}
	return 0;
}

static void MCA_CLEAR_BANKS(void)
{
	int i;
	for (i = mce_first_read_bank; i < mce_banks; i++) {
		__u64 st;
		if (__unlikely(KERNEL$READ_MSR(IA32_MC0_STATUS + i * 4, &st)))
			continue;
		if (__likely(!(st & IA32_MC_STATUS_VAL)))
			continue;
		KERNEL$WRITE_MSR(IA32_MC0_STATUS + i * 4, 0x0000000000000000ULL);
		if (KERNEL$CPU_FEATURES & CPU_HAS_CPUID) {
			__asm__ volatile ("			;\
				MOVL	%%EBX, %%ESI		;\
				XORL	%%EAX, %%EAX		;\
				CPUID				;\
				MOVL	%%ESI, %%EBX		;\
			":::"ax","cx","dx","si","cc");
		}
	}
}

static void MCA_SCAN_BANKS(int exc)
{
	int i;
	for (i = mce_first_read_bank; i < mce_banks; i++) {
		__u64 st;
		unsigned retries = MCA_READ_RETRIES;
		retry:
		if (__unlikely(KERNEL$READ_MSR(IA32_MC0_STATUS + i * 4, &st)))
			continue;
		if (__likely(!(st & IA32_MC_STATUS_VAL)))
			continue;
		addr[0] = 0;
		if (st & IA32_MC_STATUS_ADDRV) {
			__u64 ad;
			if (!KERNEL$READ_MSR(IA32_MC0_ADDR + i * 4, &ad))
				_snprintf(addr, sizeof addr, ", ADDR: %016LX", ad);
		}
		misc[0] = 0;
		if (st & IA32_MC_STATUS_MISCV) {
			__u64 ad;
			if (!KERNEL$READ_MSR(IA32_MC0_MISC + i * 4, &ad))
				_snprintf(misc, sizeof misc, ", MISC: %016LX", ad);
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "%s, BANK %d: STATUS: %016LX%s%s", !exc ? "NON FATAL CORRECTABLE ERROR" : st & IA32_MC_STATUS_PCC ? "FATAL ERROR" : exc == 1 ? "NON FATAL CORRECTABLE ERROR" : "ERROR", i, st, addr, misc);
		if (!exc) {
			KERNEL$WRITE_MSR(IA32_MC0_STATUS + i * 4, 0x0000000000000000ULL);
			if (KERNEL$CPU_FEATURES & CPU_HAS_CPUID) {
				__asm__ volatile ("			;\
					MOVL	%%EBX, %%ESI		;\
					XORL	%%EAX, %%EAX		;\
					CPUID				;\
					MOVL	%%ESI, %%EBX		;\
				":::"ax","cx","dx","si","cc");
			}
			if (--retries > 0) goto retry;
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "ERROR CANNOT BE CLEARED IN %d RETRIES", MCA_READ_RETRIES);
		}
	}
}

static jiffies_t last_mce = 0;
static int n_mces = 0;

static __finline__ void RESTORE_VIDEOMODE(void)
{
	if (KERNEL$SUICIDE_RESTORE_VIDEOMODE)
		KERNEL$SUICIDE_RESTORE_VIDEOMODE();
}

int MCE_HANDLER(int kernel)
{
	if (!cpu_has_mca) {
		__u64 ad, ty;
		int will_halt = 0;
		jiffies_t j = KERNEL$GET_JIFFIES();
		if (j == last_mce) {
			if (++n_mces >= MAX_MCES) {
				will_halt = 1;
			}
		} else {
			last_mce = j;
			n_mces = 0;
		}
		if (will_halt) {
			RESTORE_VIDEOMODE();
		}
		if (KERNEL$READ_MSR(IA32_P5_MC_ADDR, &ad) || KERNEL$READ_MSR(IA32_P5_MC_TYPE, &ty)) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION");
		else KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION, ADDR %016LX, TYPE %016LX", ad, ty);
		if (will_halt) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "TOO MANY MACHINE CHECK EXCEPTIONS - HALTING");
			return 2;
		}
		TLB_INVD_G();
		return 0;
	} else {
		__u64 st;
		int will_halt = 0;
		if (KERNEL$READ_MSR(IA32_MCG_STATUS, &st)) {
			RESTORE_VIDEOMODE();
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION, UNABLE TO READ STATUS");
			return 2;
		}
		if (MCA_CHECK_FOR_PCC()) will_halt = 2;
		else if (kernel && !(st & IA32_MCG_STATUS_RIPV)) will_halt = 1;
		if (will_halt) RESTORE_VIDEOMODE();
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "MACHINE CHECK EXCEPTION, STATUS %016LX", st);
		MCA_SCAN_BANKS(st & IA32_MCG_STATUS_RIPV ? 1 : 2);
		if (will_halt) {
			if (will_halt >= 2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "CPU CONTEXT CORRUPT - HALTING");
			else KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "UNRESTARTABLE ERROR IN KERNEL MODE - HALTING");
			return 2;
		}
		MCA_CLEAR_BANKS();
		st &= ~IA32_MCG_STATUS_MCIP;
		if (KERNEL$WRITE_MSR(IA32_MCG_STATUS, st)) {
			KERNEL$WRITE_MSR(IA32_MCG_STATUS, 0x0000000000000000ULL);
		}
		if (st & IA32_MCG_STATUS_RIPV) {
			TLB_INVD_G();
			return 0;
		} else {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CPU", "KILLING CURRENT PROCESS AND TRYING TO CONTINUE");
			TLB_INVD_G();
			return 1;
		}
	}
}

static void MCA_NONFATAL(TIMER *t)
{
	LOWER_SPL(SPL_TIMER);
	MCA_SCAN_BANKS(0);
	KERNEL$SET_TIMER(MCA_CHECK_TIME, &MCA_TIMER);
}

void MACHINE_INIT(void)
{

	__u32 cpuid = 0;
	__u32 cpuid2 = 0;
	__u32 maxc;
	__u32 version = 0;
	__u32 vendor[4] = { 0, 0, 0, 0 };
	if (KERNEL$CPU_FEATURES & CPU_HAS_CPUID) {
/* do not change EBX, because it can't be used when compiling with -fPIC */
		__asm__ volatile ("			;\
			MOVL	%%EBX, %%ESI		;\
			XORL	%%EAX, %%EAX		;\
			CPUID				;\
			XCHGL	%%EBX, %%ESI		;\
		":"=a"(maxc),"=S"(vendor[0]),"=d"(vendor[1]),"=c"(vendor[2])::"cc");
		__asm__ volatile ("			;\
			MOVL	%%EBX, %%ESI		;\
			MOVL	$1, %%EAX		;\
			CPUID				;\
			XCHGL	%%EBX, %%ESI		;\
		":"=d"(cpuid),"=c"(cpuid2),"=a"(version)::"si","cc");
	}

	if (!strcmp((char *)vendor, "GenuineIntel")) {
		/* Pentium 1 F00F bug */
		if ((version & 0xf00) == 0x500) {
			void *idt = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(KERNEL$VIRT_2_PHYS(IDT), IDT_END - IDT, PAT_RO);
			if (__unlikely(__IS_ERR(idt))) {
				__critical_printf("UNABLE TO CREATE MAPPING FOR IDT TO WORK AROUND PENTIUM BUG: %s", strerror(-__PTR_ERR(idt)));
				HALT_KERNEL();
			}
			IDTR.limit = IDT_END - IDT + 1;
			IDTR.base_lo = (__u32)idt;
			IDTR.base_hi = (__u32)idt >> 16;
			__asm__ volatile ("LIDTL IDTR");
		}

		/* XEON errata */
		if ((version & 0x0fff0fff) == 0xf11) {
			__u64 misc;
			if (!KERNEL$READ_MSR(IA32_MISC_ENABLE, &misc)) {
				misc |= 1ULL << 9;
				KERNEL$WRITE_MSR(IA32_MISC_ENABLE, misc);
			}
		}

		/* Pentium Pro errata */
		if ((version & 0xfff) >= 0x610 && (version & 0xfff) < 0x618) {
			VM_ADD_MEMORY_HOLE(0x70000000, 0x7003ffff);
		}
	}
	if (!strcmp((char *)vendor, "AuthenticAMD")) {
		if ((version & 0xf00) == 0x600 && (version & 0xff) >= 0x81) {
			__u64 clk;
			if (!KERNEL$READ_MSR(MSR_K7_CLK_CTL, &clk)) {
				clk &= ~0xfff00000ULL;
				clk |=  0x20000000ULL;
				KERNEL$WRITE_MSR(MSR_K7_CLK_CTL, clk);
			}
		}
	}

	/* enable MCE / MCA */
	cpu_has_mca = 0;
	if (cpuid & CPU_FEATURE_MCE) {
		if (cpuid & CPU_FEATURE_MCA) {
			__u64 cap;
			int i;
			if (KERNEL$READ_MSR(IA32_MCG_CAP, &cap)) {
				goto skip_mca;
			}
			if (cap & IA32_MCG_CAP_MCG_CTL_P) {
				KERNEL$WRITE_MSR(IA32_MCG_CTL, 0xffffffffffffffffULL);
			}
			mce_banks = cap & IA32_MCG_CAP_COUNT;
			if (cap & IA32_MCG_CAP_MCG_EXT_P) {
				mce_extended_msrs = (cap & IA32_MCG_CAP_MCG_EXT_CNT) >> P_IA32_MCG_CAP_MCG_EXT_CNT;
			}
			mce_first_read_bank = 0;
			mce_first_write_bank = 1;
			if (!strcmp((char *)vendor, "GenuineIntel") && ((version >> 8) & 0xf) == 0xf) mce_first_write_bank = 0;
			if (!strcmp((char *)vendor, "AuthenticAMD")) {
				mce_first_read_bank = 1;
			}
			if (mce_first_write_bank)
				KERNEL$WRITE_MSR(IA32_MC0_STATUS + 0 * 4, 0x0000000000000000ULL);
			for (i = mce_first_write_bank; i < mce_banks; i++) {
				KERNEL$WRITE_MSR(IA32_MC0_CTL + i * 4, 0xffffffffffffffffULL);
				KERNEL$WRITE_MSR(IA32_MC0_STATUS + i * 4, 0x0000000000000000ULL);
			}
			cpu_has_mca = 1;
			/*__debug_printf("first: %d, banks: %d, ext: %d\n", mce_first_bank, mce_banks, mce_extended_msrs);*/
			MCA_TIMER.fn = MCA_NONFATAL;
			KERNEL$SET_TIMER(MCA_CHECK_TIME, &MCA_TIMER);
		}
		skip_mca:
		if (!strcmp((char *)vendor, "CentaurHauls")) {
			__u64 fcr1;
			if (!KERNEL$READ_MSR(MSR_IDT_FCR1, &fcr1)) {
				fcr1 |= 1 << 2;
				fcr1 &= ~(1 << 4);
				KERNEL$WRITE_MSR(MSR_IDT_FCR1, fcr1);
			}
		}
		CR_4 |= CR4_MCE;
		__asm__ volatile ("MOVL	%0, %%CR4"::"r"(CR_4));
	}

	/* disable processor serial number */
	if (cpuid & CPU_FEATURE_PN) {
		__u64 sn;
		if (!KERNEL$READ_MSR(MSR_BBL_CR_CTL, &sn)) {
			sn |= MSR_BBL_CR_CTL_NUMBER_DISABLE;
			KERNEL$WRITE_MSR(MSR_BBL_CR_CTL, sn);
		}
	}
}
