#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <KERNEL/ASM.H>
#include <KERNEL/TIMER.H>

#include <KERNEL/TIME.H>
#include <ARCH/TIME.H>

#define I8254_FREQ	1193181

#define TIMER_DIV	((I8254_FREQ + __KERNEL_TICK / 2) / __KERNEL_TICK)

volatile jiffies_t __JIFFIES = 0;

extern __u32 TIME_MAX_FIXUP_1, TIME_MAX_FIXUP_2, TIME_MUL_FIXUP;
extern __u8 TIME_SHR_FIXUP;

static void WAIT_LOOP(int c)
{
	/* nops are for jmp prediction anomaly on Pentium... we won't get as cool bogomips values as on Linux :-) */
	__asm__ volatile(".ALIGN 16; 1: DECL %%EAX; NOP; NOP; NOP; NOP; NOP; NOP; NOP; NOP; JNS 1b":"=a"(c):"a"(c):"cc");
}

/* default values ... they are used before timer interrupt is enabled and real values calibrated */

static int cpu_mhz = 100;
static int bogomips = 160;

void KERNEL$UDELAY(unsigned u)
{
	while (__unlikely(u > 10000)) {
		KERNEL$UDELAY(10000);
		u -= 10000;
	}
	if (__likely(KERNEL$CPU_FEATURES & CPU_HAS_TSC)) {
		int t = u * cpu_mhz;
		unsigned a = RDTSC_LO();
		do {
			DO_NOP();
		} while ((int)(RDTSC_LO() - a) < t);
	} else {
		WAIT_LOOP((u * bogomips) >> 3);
	}
}

int GET_TIME_OF_DAY(struct timeval *tv)
{
	int r;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	__asm__ volatile ("CALL %c1":"=a"(r):"i"(APLACE(APAGE_GET_TIME_OF_DAY)),"0"(tv):"dx","cx","cc","memory");
	LOWER_SPLX(spl);
	return r;
}

time_t GET_TIME_SEC(void)
{
	time_t t;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	__asm__ volatile ("CALL %c1":"=A"(t):"i"(APLACE(APAGE_GET_TIME_SEC)):"cx","cc");
	LOWER_SPLX(spl);
	return t;
}

static volatile int timer_init_cnt;

static volatile __u64 rdtsc_calibrate;
extern __u8 *FINE_TICKS_SHIFT_FIXUP[];
static int FINE_TICKS_SHIFT;
sched_unsigned FINE_TICKS_PER_MINUTE;

DECL_RT_IRQ_HANDLER(TIMER_INIT_IRQ)
{
	if (__likely(KERNEL$CPU_FEATURES & CPU_HAS_TSC)) {
		__u64 r = RDTSC();
		if (timer_init_cnt == 1) rdtsc_calibrate = r;
		if (timer_init_cnt == 2) rdtsc_calibrate = r - rdtsc_calibrate;
	}
	timer_init_cnt++;
	IRQ_RETURN;
}

void FIXUP_TIMER_ASM(unsigned jiffies_step);

void TIMER_INIT(void)
{
	static IO_RANGE iorange;
	int i;
	int r;
	FIXUP_TIMER_ASM(JIFFIES_PER_SECOND / __KERNEL_TICK);
	iorange.start = 0x40;
	iorange.len = 4;
	iorange.name = "TIMER";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&iorange))) {
		__critical_printf("CAN'T REGISTER IO RANGE FOR TIMER: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	io_outb_p(0x43, 0x34);
	io_outb_p(0x40, TIMER_DIV & 0xff);
	io_outb_p(0x40, TIMER_DIV >> 8);
	timer_init_cnt = 0;
	KERNEL$REQUEST_IRQ(0, IRQ_RT_HANDLER, &TIMER_INIT_IRQ, NULL, &TIMER_IRQ_CONTROL, "KERNEL$TIMER_TEST");
	if (__likely(KERNEL$CPU_FEATURES & CPU_HAS_TSC)) {
/* FPU may be used here, because it's init code and no user processes
   run. Otherwise it would have to be saved and restored */
		long double cpu_mhz_fp;
		while (timer_init_cnt < 3) DO_NOP();
		if (rdtsc_calibrate < 1100000 / __KERNEL_TICK) {
			__critical_printf("BAD TIMER (%uHZ)\n", (unsigned)rdtsc_calibrate * __KERNEL_TICK);
			HALT_KERNEL();
		}
		cpu_mhz_fp = rdtsc_calibrate / ((long double)1000000 / __KERNEL_TICK);
		cpu_mhz = cpu_mhz_fp + 0.5;

			/* wrap around in about 1/2 - 1 hour */
		FINE_TICKS_SHIFT = 0;
		again_fine_ticks:
		if (cpu_mhz_fp / (1 << FINE_TICKS_SHIFT) >= 2) { /* this is needed so that we get smooth tv_usec */
			FINE_TICKS_SHIFT++;
			goto again_fine_ticks;
		}
		FINE_TICKS_PER_MINUTE = cpu_mhz_fp * 60000000. / (1 << FINE_TICKS_SHIFT);

		if (__unlikely(TIME_MAX_FIXUP_1 != 0x90909090) ||
		    __unlikely(TIME_MAX_FIXUP_2 != 0x90909090) ||
		    __unlikely(TIME_MUL_FIXUP != 0x90909090) ||
		    __unlikely(TIME_SHR_FIXUP != 0x90)) {
			KERNEL$SUICIDE("TIMER_INIT: INVALID INITIAL FIXUPS: TIME_MAX_FIXUP_1 == %08X, TIME_MAX_FIXUP_2 == %08X, TIME_MUL_FIXUP == %08X, TIME_SHR_FIXUP == %02X", TIME_MAX_FIXUP_1, TIME_MAX_FIXUP_2, TIME_MUL_FIXUP, TIME_SHR_FIXUP);
		}
		
		TIME_SHR_FIXUP = 31 - __BSR(1000000 / (JIFFIES_PER_SECOND >> KERNEL$JIFFIES_STEP_BITS) - 1);
		TIME_MUL_FIXUP = 60000000. * (1 << TIME_SHR_FIXUP) / FINE_TICKS_PER_MINUTE + .5;
		TIME_MAX_FIXUP_1 = TIME_MAX_FIXUP_2 = ((__u64)(1000000 / (JIFFIES_PER_SECOND >> KERNEL$JIFFIES_STEP_BITS) + 1) << TIME_SHR_FIXUP) / TIME_MUL_FIXUP - 1;

		for (i = 0; FINE_TICKS_SHIFT_FIXUP[i]; i++) {
			if (__unlikely(*FINE_TICKS_SHIFT_FIXUP[i] != 0x1f)) {
				KERNEL$SUICIDE("TIMER_INIT: FINE_TICKS_SHIFT_FIXUP DAMAGED: FINE_TICKS_SHIFT_FIXUP[%d] == %p, VALUE == %02X", i, FINE_TICKS_SHIFT_FIXUP[i], *FINE_TICKS_SHIFT_FIXUP[i]);
			}
			*FINE_TICKS_SHIFT_FIXUP[i] = FINE_TICKS_SHIFT;
		}
	} else {
#define SYNC_TIMER	\
do { cnt = timer_init_cnt; while (timer_init_cnt == cnt) DO_NOP(); } while (0)
		int cnt;
		int i;
		while (timer_init_cnt < 2) DO_NOP();
		for (i = 0; i < 17; i++) {
			bogomips = 1 << i;
			KERNEL$UDELAY(1000000 / __KERNEL_TICK);
			if (timer_init_cnt > 2) goto ool;
		}
		__critical_printf("TIMER STUCK\n");
		HALT_KERNEL();
		ool:
		while (1) {
			SYNC_TIMER;
			KERNEL$UDELAY(1000000 / __KERNEL_TICK);
			if (timer_init_cnt > cnt + 1) break;
			bogomips = 1 << ++i;
		}
		bogomips = 1 << --i;
		while (--i >= 0) {
			bogomips += 1 << i;
			SYNC_TIMER;
			KERNEL$UDELAY(1000000 / __KERNEL_TICK);
			if (timer_init_cnt > cnt + 1) bogomips -= 1 << i;
		}
		FINE_TICKS_PER_MINUTE = (JIFFIES_PER_SECOND * 60) << _FINE_TICKS_SHIFT_WITHOUT_TSC;
	}
	KERNEL$RELEASE_IRQ(&TIMER_IRQ_CONTROL, 0);
	KTIMER_INIT();
	KERNEL$REQUEST_IRQ(0, IRQ_RT_HANDLER, &TIMER_IRQ, NULL, &TIMER_IRQ_CONTROL, "KERNEL$TIMER");
}
