#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/DL.H>
#include <SPAD/SYSLOG.H>
#include <SYS/TYPES.H>
#include <ARCH/IO.H>

#include <SPAD/ISADMA.H>

#include "DMAREGS.H"

#define MAX_TRIES	10000

#define N_CHANS		8

static struct dma_state global_dma_state;

/* !!! SMPFIX: use locking & shared data */

static struct dma_state {
	char used[N_CHANS];
	char enabled[N_CHANS];
	IO_RANGE range1;
	IO_RANGE range2;
	IO_RANGE rangep;
} * __const__ dma_state = &global_dma_state;

static __const__ unsigned char dma_page_regs[N_CHANS] = { DMA_PAGE_0, DMA_PAGE_1, DMA_PAGE_2, DMA_PAGE_3, 0, DMA_PAGE_5, DMA_PAGE_6, DMA_PAGE_7 };

void ISADMA$SETUP_TRANSFER(int channel, int wr, int autoinit, unsigned long addr, unsigned long size)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DMA), spl))) KERNEL$SUICIDE("ISADMA$SETUP_TRANSFER AT SPL %08X", spl);
	RAISE_SPL(SPL_DMA);
	if (__unlikely((dma_state->enabled[channel] | (dma_state->used[channel] ^ 1)) != 0)) goto en;
	dma_state->enabled[channel] = 1;
	size--;
	if (channel < 4) {
		if (__unlikely(((addr & ~0x00FFFFFFUL) | (size & ~0x0000FFFFUL)) != 0)) goto invl;
		if (__unlikely((addr & ~0xFFFFUL) != ((addr + size) & ~0xFFFFUL))) goto invl;
		io_outb(DMA1_MODE_REG, DMA_MODE_READ + (wr << 2) + (autoinit << 4) + channel);
		io_outb(dma_page_regs[channel], addr >> 16);
		io_outb((channel << 1) + DMA_ADDR_0, addr & 0xff);
		io_outb((channel << 1) + DMA_ADDR_0, (addr >> 8) & 0xff);
		io_outb((channel << 1) + DMA_CNT_0, size & 0xff);
		io_outb((channel << 1) + DMA_CNT_0, size >> 8);
		io_outb(DMA1_MASK_REG, channel);
		LOWER_SPLX(spl);
		return;
	} else if (__likely(channel > 4)) {
		if (__unlikely(((addr & ~0x00FFFFFEUL) | (size & ~0x0001FFFEUL)) != 0)) goto invl;
		if (__unlikely((addr & ~0x1FFFFUL) != ((addr + size) & ~0x1FFFFUL))) goto invl;
		io_outb(DMA2_MODE_REG, DMA_MODE_READ + (wr << 2) + (autoinit << 4) + channel - 4);
		io_outb(dma_page_regs[channel], (addr >> 16) & ~1);
		io_outb(((channel - 4) << 2) + DMA_ADDR_4, (addr >> 1) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_ADDR_4, (addr >> 9) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_CNT_4, (size >> 1) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_CNT_4, size >> 9);
		io_outb(DMA2_MASK_REG, channel - 4);
		LOWER_SPLX(spl);
		return;
	} else KERNEL$SUICIDE("ISADMA$SETUP_TRANSFER: INVALID CHANNEL %d", channel);
	invl:
	KERNEL$SUICIDE("ISADMA$SETUP_TRANSFER: CHANNEL %d: INVALID ADDRESS/SIZE: %08lX, %08lX", channel, addr, size);
	en:
	if (!dma_state->used[channel]) KERNEL$SUICIDE("ISADMA$SETUP_TRANSFER: CHANNEL %d NOT REGISTERED", channel);
	else if (dma_state->used[channel] == 1) KERNEL$SUICIDE("ISADMA$SETUP_TRANSFER: CHANNEL %d HAS ALREADY TRANSFER IN PROGRESS", channel);
	else KERNEL$SUICIDE("ISADMA$SETUP_TRANSFER: JAMMED MEMORY, dma_state->used[%d] == %d", channel, dma_state->used[channel]);
}

void ISADMA$SETUP_AUTOINIT(int channel, unsigned long addr, unsigned long size)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DMA), spl))) KERNEL$SUICIDE("ISADMA$SETUP_AUTOINIT AT SPL %08X", spl);
	RAISE_SPL(SPL_DMA);
	if (__unlikely(!(dma_state->enabled[channel] & dma_state->used[channel]))) goto en;
	size--;
	if (channel < 4) {
		if (__unlikely(((addr & ~0x00FFFFFFUL) | (size & ~0x0000FFFFUL)) != 0)) goto invl;
		if (__unlikely((addr & ~0xFFFFUL) != ((addr + size) & ~0xFFFFUL))) goto invl;
		io_outb(dma_page_regs[channel], addr >> 16);
		io_outb((channel << 1) + DMA_ADDR_0, addr & 0xff);
		io_outb((channel << 1) + DMA_ADDR_0, (addr >> 8) & 0xff);
		io_outb((channel << 1) + DMA_CNT_0, size & 0xff);
		io_outb((channel << 1) + DMA_CNT_0, size >> 8);
		LOWER_SPLX(spl);
		return;
	} else if (__likely(channel > 4)) {
		if (__unlikely(((addr & ~0x00FFFFFEUL) | (size & ~0x0001FFFEUL)) != 0)) goto invl;
		if (__unlikely((addr & ~0x1FFFFUL) != ((addr + size) & ~0x1FFFFUL))) goto invl;
		io_outb(dma_page_regs[channel], (addr >> 16) & ~1);
		io_outb(((channel - 4) << 2) + DMA_ADDR_4, (addr >> 1) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_ADDR_4, (addr >> 9) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_CNT_4, (size >> 1) & 0xff);
		io_outb(((channel - 4) << 2) + DMA_CNT_4, size >> 9);
		LOWER_SPLX(spl);
		return;
	} else KERNEL$SUICIDE("ISADMA$SETUP_AUTOINIT: INVALID CHANNEL %d", channel);
	invl:
	KERNEL$SUICIDE("ISADMA$SETUP_AUTOINIT: CHANNEL %d: INVALID ADDRESS/SIZE: %08lX, %08lX", channel, addr, size);
	en:
	if (!dma_state->used[channel]) KERNEL$SUICIDE("ISADMA$SETUP_AUTOINIT: CHANNEL %d NOT REGISTERED", channel);
	else KERNEL$SUICIDE("ISADMA$SETUP_AUTOINIT: CHANNEL %d HAS NOT TRANSFER IN PROGRESS", channel);
}

unsigned long ISADMA$GET_ADDRESS(int channel)
{
	unsigned change_counter;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DMA), spl))) KERNEL$SUICIDE("ISADMA$GET_ADDRESS AT SPL %08X", spl);
	RAISE_SPL(SPL_DMA);
	if (__unlikely(!(dma_state->enabled[channel] & dma_state->used[channel]))) goto en;
	change_counter = 0;
	again:
	if (++change_counter == MAX_TRIES) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "ISADMA", "ADDRESS IS CONSTANTLY CHANGING, DMA CHANNEL %d", channel);
	}
	/*
	 * Note:
	 * on my VIA 586 mainboard, address is constantly changing under some
	 * workloads (sound card + disk simultaneously).
	 * We must compare only high byte --- low byte is unstable.
	 *	1. read low byte and discard it
	 *	2. read high byte and remember it
	 *	3. read low byte and remember it
	 *	4. read high byte --- if it matches, we know that low byte
	 *	   didn't wrap around. If it doesn't match, repeat
	 */
	if (channel < 4) {
		unsigned long address = io_inb(dma_page_regs[channel]) << 16;
		io_inb((channel << 1) + DMA_ADDR_0);
		address |= io_inb((channel << 1) + DMA_ADDR_0) << 8;
		address |= io_inb((channel << 1) + DMA_ADDR_0);
		if (__unlikely(((address >> 8) & 0xff) != io_inb((channel << 1) + DMA_ADDR_0)) && __likely(change_counter < MAX_TRIES)) goto again;
		LOWER_SPLX(spl);
		/*if (change_counter != 1) __debug_printf("changes: %d\n", change_counter);*/
		return address;
	} else {
		unsigned long address = (io_inb(dma_page_regs[channel]) << 16) & ~1;
		io_inb(((channel - 4) << 2) + DMA_ADDR_4);
		address |= io_inb(((channel - 4) << 2) + DMA_ADDR_4) << 9;
		address |= io_inb(((channel - 4) << 2) + DMA_ADDR_4) << 1;
		if (__unlikely(((address >> 9) & 0xff) != io_inb(((channel - 4) << 2) + DMA_ADDR_4)) && __likely(change_counter < MAX_TRIES)) goto again;
		LOWER_SPLX(spl);
		return address;
	}

	en:
	if (!dma_state->used[channel]) KERNEL$SUICIDE("ISADMA$GET_ADDRESS: CHANNEL %d NOT REGISTERED", channel);
	else KERNEL$SUICIDE("ISADMA$GET_ADDRESS: CHANNEL %d HAS NOT TRANSFER IN PROGRESS", channel);
}

void ISADMA$END_TRANSFER(int channel)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DMA), spl))) KERNEL$SUICIDE("ISADMA$END_TRANSFER AT SPL %08X", spl);
	RAISE_SPL(SPL_DMA);
	if (__unlikely(!dma_state->enabled[channel])) goto di;
	dma_state->enabled[channel] = 0;
	if (channel < 4) io_outb(DMA1_MASK_REG, channel + 4);
	else io_outb(DMA2_MASK_REG, channel - 4 + 4);
	LOWER_SPLX(spl);
	return;
	di:
	KERNEL$SUICIDE("ISADMA$SETUP_TRANSFER: CHANNEL %d IS DISABLED", channel);
}

char *ISADMA$REQUEST_CHANNEL(int channel)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DMA), spl))) KERNEL$SUICIDE("ISADMA$REQUEST_CHANNEL AT SPL %08X", spl);
	RAISE_SPL(SPL_DMA);
	if (__unlikely(channel < 0) || __unlikely(channel >= N_CHANS) || __unlikely(channel == 4)) {
		LOWER_SPLX(spl);
		return "UNACCEPTABLE CHANNEL NUMBER";
	}
	if (__unlikely(dma_state->used[channel])) {
		LOWER_SPLX(spl);
		return "CHANNEL IS ALREADY USED";
	}
	dma_state->used[channel] = 1;
	LOWER_SPLX(spl);
	return NULL;
}

void ISADMA$RELEASE_CHANNEL(int channel)
{
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DMA), spl))) KERNEL$SUICIDE("ISADMA$RELEASE_CHANNEL AT SPL %08X", spl);
	RAISE_SPL(SPL_DMA);
	if (__unlikely(channel < 0) || __unlikely(channel >= N_CHANS)) KERNEL$SUICIDE("ISADMA$RELEASE_CHANNEL: INVALID CHANNEL %d", channel);
	if (__unlikely(!dma_state->used[channel])) KERNEL$SUICIDE("ISADMA$RELEASE_CHANNEL: CHANNEL %d NOT REGISTERED", channel);
	if (__unlikely(dma_state->enabled[channel])) KERNEL$SUICIDE("ISADMA$RELEASE_CHANNEL: CHANNEL %d HAS TRANSFER IN PROGRESS", channel);
	dma_state->used[channel] = 0;
	LOWER_SPLX(spl);
	return;
}

DECL_IOCALL(DLL$LOAD, SPL_DEV, DLINITRQ)
{
	int r;
	memset(dma_state, 0, sizeof(struct dma_state));
	dma_state->range1.start = IO_DMA1_BASE;
	dma_state->range1.len = IO_DMA1_LEN;
	dma_state->range1.name = "DMA 0-3";
	dma_state->range2.start = IO_DMA2_BASE;
	dma_state->range2.len = IO_DMA2_LEN;
	dma_state->range2.name = "DMA 4-7";
	dma_state->rangep.start = IO_DMAP_BASE;
	dma_state->rangep.len = IO_DMAP_LEN;
	dma_state->rangep.name = "DMA PAGE";
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&dma_state->range1))) {
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER DMA 0-3 PORT RANGE: %s", strerror(-r));
		err0:
		RQ->status = r;
		RETURN_AST(RQ);
	}
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&dma_state->range2))) {
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER DMA 4-7 PORT RANGE: %s", strerror(-r));
		err1:
		KERNEL$UNREGISTER_IO_RANGE(&dma_state->range1);
		goto err0;
	}
	if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&dma_state->rangep))) {
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER DMA PAGE PORT RANGE: %s", strerror(-r));
		KERNEL$UNREGISTER_IO_RANGE(&dma_state->range2);
		goto err1;
	}
	io_outb(DMA1_CLEAR_FF_REG, 0);
	io_outb(DMA2_CLEAR_FF_REG, 0);
	RQ->status = 0;
	RETURN_AST(RQ);
}

DECL_IOCALL(DLL$UNLOAD, SPL_DEV, DLINITRQ)
{
	int i;
	for (i = 0; i < 8; i++) if (dma_state->used[i]) KERNEL$SUICIDE("ISADMA: CHANNEL %d WAS NOT RELEASED BY THE DRIVER", i);
	KERNEL$UNREGISTER_IO_RANGE(&dma_state->range1);
	KERNEL$UNREGISTER_IO_RANGE(&dma_state->range2);
	KERNEL$UNREGISTER_IO_RANGE(&dma_state->rangep);
	RQ->status = 0;
	RETURN_AST(RQ);
}



