#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SYS/TYPES.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/CMOS.H>
#include <SPAD/SYNC.H>
#include <SPAD/TIMER.H>
#include <SYS/TIME.H>
#include <TIME.H>

#define CMOS_SEC	0x00
#define CMOS_MIN	0x02
#define CMOS_HOUR	0x04
#define CMOS_DAY	0x07
#define CMOS_MONTH	0x08
#define CMOS_YEAR	0x09
#define CMOS_FLAG_B	0x0b
#define  FLAG_B_BIN	0x04
#define CMOS_CENTURY	0x32

static __const__ struct zone {
	int	offset;
	char	*stdzone;
} zonetab[] = {
	{-1*60,	"MET"},		/* Middle European */
	{-2*60,	"EET"},		/* Eastern European */
	{4*60,	"AST"},		/* Atlantic */
	{5*60,	"EST"},		/* Eastern */
	{6*60,	"CST"},		/* Central */
	{7*60,	"MST"},		/* Mountain */
	{8*60,	"PST"},		/* Pacific */
	{0,	"GMT"},		/* Greenwich */
	{0*60,	"WET"},		/* Western European */
	{-10*60,"EST"},		/* Aust: Eastern */
     {-10*60+30,"CST"},		/* Aust: Central */
	{-8*60,	"WST"},		/* Aust: Western */
	{-1,	NULL}
};

/* from linux/include/linux/time.h (C) someone who wrote it */

/* Converts Gregorian date to seconds since 1970-01-01 00:00:00.
 * Assumes input in normal date format, i.e. 1980-12-31 23:59:59
 * => year=1980, mon=12, day=31, hour=23, min=59, sec=59.
 *
 * [For the Julian calendar (which was used in Russia before 1917,
 * Britain & colonies before 1752, anywhere else before 1582,
 * and is still in use by some communities) leave out the
 * -year/100+year/400 terms, and add 10.]
 *
 * This algorithm was first published by Gauss (I think).
 *
 * WARNING: this function will overflow on 2106-02-07 06:28:16 on
 * machines were long is 32-bit! (However, as time_t is signed, we
 * will already get problems at other places on 2038-01-19 03:14:08)
 */
static time_t
maketime (unsigned int year, unsigned int mon,
	unsigned int day, unsigned int hour,
	unsigned int min, unsigned int sec)
{
	if (0 >= (int) (mon -= 2)) {	/* 1..12 -> 11,12,1..10 */
		mon += 12;		/* Puts Feb last since it has leap day */
		year -= 1;
	}

	return (((
		(time_t) (year/4 - year/100 + year/400 + 367*mon/12 + day) +
			year*365 - 719499
	    )*24 + hour /* now have hours */
	  )*60 + min /* now have minutes */
	)*60 + sec; /* finally seconds */
}

struct cmos_clock {
	unsigned sec, min, hour, day, month, year;
};

static unsigned debcd(unsigned bcd)
{
	unsigned lo = bcd & 0x0f;
	unsigned hi = bcd >> 4;
	if (lo >= 10 || hi >= 10) return -1;
	return lo + hi * 10;
}

static unsigned bcd(unsigned v)
{
	return (v % 10) + ((v / 10) << 4);
}

static int read_clock(struct cmos_clock *c)
{
	int i;
	unsigned sec, min, hour, day, month, year, century;
	unsigned bsec, bmin, bhour, bday, bmonth, byear, bcentury;
	unsigned flagb, sec2;
	i = 0;
	loop:
	sec = CMOS$READ(CMOS_SEC);
	min = CMOS$READ(CMOS_MIN);
	hour = CMOS$READ(CMOS_HOUR);
	day = CMOS$READ(CMOS_DAY);
	month = CMOS$READ(CMOS_MONTH);
	year = CMOS$READ(CMOS_YEAR);
	century = CMOS$READ(CMOS_CENTURY);
	flagb = CMOS$READ(CMOS_FLAG_B);
	sec2 = CMOS$READ(CMOS_SEC);
	if ((int)sec < 0 || (int)min < 0 || (int)hour < 0 || (int)day < 0 || (int)month < 0 || (int)year < 0 || (int)century < 0 || (int)flagb < 0 || (int)sec2 < 0) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CLOCK", "CAN'T READ CMOS");
		return -1;
	}
	if (sec != sec2) {
		if (i++ < 10) goto loop;
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CLOCK", "TIMER CONSTANTLY CHANGING, CANNOT READ");
		return -1;
	}
	if (!(flagb & FLAG_B_BIN)) {
		bsec = debcd(sec);
		bmin = debcd(min);
		bhour = debcd(hour);
		bday = debcd(day);
		bmonth = debcd(month);
		byear = debcd(year);
	} else {
		bsec = sec;
		bmin = min;
		bhour = hour;
		bday = day;
		bmonth = month;
		byear = year;
	}
	bcentury = debcd(century);
	if (bsec >= 60 || bmin >= 60 || bhour >= 24 || !bday || bday > 31 || !bmonth || bmonth > 12 || byear >= 256 || bcentury >= 100) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, "CLOCK", "INCORRECT DATA IN CMOS: SEC=%02X, MIN=%02X, HOUR=%02X, DAY=%02X, MONTH=%02X, YEAR=%02X, CENTURY=%02X, FLAG_B=%02X", sec, min, hour, day, month, year, century, flagb);
		return -1;
	}
	c->sec = bsec;
	c->min = bmin;
	c->hour = bhour;
	c->day = bday;
	c->month = bmonth;
	if (!bcentury) bcentury = 19;
	if (bcentury == 19 && byear < 70) bcentury = 20;
	c->year = byear + bcentury * 100;
	return 0;
}

static void set_clock_date(struct cmos_clock *c)
{
	int day = c->day;
	int month = c->month;
	int year = c->year % 100;
	int century = c->year / 100;
	int flagb = CMOS$READ(CMOS_FLAG_B);
	if (flagb < 0) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CLOCK", "CAN'T READ CMOS");
		return;
	}
	if (!(flagb & FLAG_B_BIN)) {
		day = bcd(day);
		month = bcd(month);
		year = bcd(year);
	}
	century = bcd(century);
	if (CMOS$WRITE(CMOS_DAY, day) ||
	    CMOS$WRITE(CMOS_MONTH, month) ||
	    CMOS$WRITE(CMOS_YEAR, year) ||
	    CMOS$WRITE(CMOS_CENTURY, century)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CLOCK", "CAN'T WRITE CMOS");
		return;
	}
}

static void set_clock_time(struct cmos_clock *c)
{
	int sec = c->sec;
	int min = c->min;
	int hour = c->hour;
	int flagb = CMOS$READ(CMOS_FLAG_B);
	if (flagb < 0) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CLOCK", "CAN'T READ CMOS");
		return;
	}
	if (!(flagb & FLAG_B_BIN)) {
		sec = bcd(sec);
		min = bcd(min);
		hour = bcd(hour);
	}
	if (CMOS$WRITE(CMOS_SEC, sec) ||
	    CMOS$WRITE(CMOS_MIN, min) ||
	    CMOS$WRITE(CMOS_HOUR, hour)) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, "CLOCK", "CAN'T WRITE CMOS");
		return;
	}
}

static void sync_clock(void)
{
	int i = 0;
	unsigned s = CMOS$READ(CMOS_SEC);
	do {
		KERNEL$SLEEP(1);
		if (++i >= 2 * JIFFIES_PER_SECOND >> KERNEL$JIFFIES_STEP_BITS) return;
	} while (s == CMOS$READ(CMOS_SEC));
}

int main(int argc, char *argv[])
{
	char *opt, *optend, *val;
	int set_tz;
	time_t t;
	struct cmos_clock c;
	struct cmos_clock s;
	struct timezone tz;
	char *tzname = NULL;
	int state = NULL;
	char **arg = argv;
	int dst = 0;
	int set_date = 0, set_time = 0;
	struct __param_table params[] = {
		"TZ", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"DST", __PARAM_BOOL, ~0, 1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &tzname;
	params[1].__p = &dst;
	memset(&tz, 0, sizeof tz);
	while (__parse_params(&arg, &state, params, &opt, &optend, &val)) {
		char *c;
		long v;
		if (opt != NULL || optend != NULL || !val) {
			badsyn:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETTIME: SYNTAX ERROR");
			return -EBADSYN;
		}
		if ((c = strchr(val, '.'))) {
			if (set_date) goto badsyn;
			if (__get_number(val, c, 0, &v)) goto badsyn;
			s.year = v;
			val = c + 1;
			if (!(c = strchr(val, '.'))) goto badsyn;
			if (__get_number(val, c, 0, &v)) goto badsyn;
			s.month = v;
			val = c + 1;
			if (__get_number(val, val + strlen(val), 0, &v)) goto badsyn;
			s.day = v;
			set_date = 1;
			continue;
		} else if ((c = strchr(val, ':')) != NULL) {
			if (set_time) goto badsyn;
			if (__get_number(val, c, 0, &v)) goto badsyn;
			s.hour = v;
			val = c + 1;
			if (!(c = strchr(val, ':'))) goto badsyn;
			if (__get_number(val, c, 0, &v)) goto badsyn;
			s.min = v;
			val = c + 1;
			if (__get_number(val, val + strlen(val), 0, &v)) goto badsyn;
			s.sec = v;
			set_time = 1;
			continue;
		} else goto badsyn;
	}
	if (tzname != NULL) {
		long z;
		int i;
		set_tz = 1;
		for (i = 0; zonetab[i].stdzone != NULL; i++) if (!_strcasecmp(zonetab[i].stdzone, tzname)) {
			tz.tz_minuteswest = zonetab[i].offset;
			goto zone_ok;
		}
		if (__get_number(tzname, tzname + strlen(tzname), 1, &z) || z <= 24 * 60 || z >= 24 * 60) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETTIME: INVALID TIME ZONE %s", tzname);
			return -EBADSYN;
		}
		tz.tz_minuteswest = z;
	} else if (dst) {
		gettimeofday(NULL, &tz);
		set_tz = 1;
	} else {
		set_tz = 0;
	}
	zone_ok:
	tz.tz_dsttime = dst;
	if (!set_date || !set_time) if (read_clock(&c)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETTIME: ERROR READING TIME");
		return -EINVAL;
	}
	if (set_date) if (s.year >= 10000 || !s.month || s.month > 12 || !s.day || s.day > 31) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETTIME: BAD DATE TO SET: YEAR %d, MONTH %d, DAY %d", s.year, s.month, s.day);
		return -EINVAL;
	}
	if (set_time) if (s.hour >= 24 || s.min >= 60 || s.sec >= 60) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETTIME: BAD TIME TO SET: HOUR %d, MINUTE %d, SECOND %d", s.hour, s.min, s.sec);
		return -EINVAL;
	}
	if (set_date || set_time) sync_clock();
	if (set_date) c.year = s.year, c.month = s.month, c.day = s.day, set_clock_date(&c);
	if (set_time) c.hour = s.hour, c.min = s.min, c.sec = s.sec, set_clock_time(&c);
	/*__debug_printf("SEC=%d, MIN=%d, HOUR=%d, DAY=%d, MONTH=%d, YEAR=%d\n", c.sec, c.min, c.hour, c.day, c.month, c.year);*/
	t = maketime(c.year, c.month, c.day, c.hour, c.min, c.sec);
	if (!set_tz) {
		if (gettimeofday(NULL, &tz)) tz.tz_minuteswest = 0;
		t = t + tz.tz_minuteswest * 60 - 3600 * tz.tz_dsttime;
		if (stime(&t)) {
			cant_set:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SETTIME: UNABLE TO SET TIME: %s", strerror(errno));
			return -errno;
		}
	} else {
		struct timeval tv;
		t = t + tz.tz_minuteswest * 60 - 3600 * tz.tz_dsttime;
		tv.tv_sec = t;
		tv.tv_usec = 0;
		if (settimeofday(&tv, &tz)) goto cant_set;
	}
	return 0;
}
