#include <SYS/TYPES.H>
#include <STRING.H>
#include <VALUES.H>
#include <STDARG.H>
#include <SPAD/LIBC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DL.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TIMER.H>
#include <SPAD/RANDOM.H>
#include <ARCH/IRQ.H>
#include <ARCH/IO.H>
#include <SPAD/BIO_KRNL.H>

#include <SPAD/SCSI.H>
#include <SPAD/ATA.H>
#include "ACTRL.H"
#include "ACTRLREG.H"

#define RESET_FF_TIMEOUT	(JIFFIES_PER_SECOND * 1)
#define POLL_TIME		(JIFFIES_PER_SECOND / 10)

#define ATAPI_XFER_LEN		(0x10000 - BIO_SECTOR_SIZE)

#define ATAPI_AUTOSENSE_LENGTH	18

static int (*__const__ controllers[])(ACTRL *a) = {
	DETECT_CMD640,
	DETECT_RZ1000,
	DETECT_VIA,
	DETECT_IT821,
	DETECT_TRIFLEX,
};

static __const__ struct pci_id_s ata_dev[] = {
	{ 0x1283, 0x8211, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "IT8211", ACTRL_ALWAYS_NATIVE },
	{ 0x1283, 0x8212, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "IT8212", ACTRL_ALWAYS_NATIVE },

	{ 0x1106, 0x5337, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1106, 0x0591, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1106, 0x3149, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1106, 0x5287, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1106, 0x5372, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },
	{ 0x1106, 0x7372, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VT6420", ACTRL_ALWAYS_NATIVE | ACTRL_SATA },

	{ PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_CLASS_STORAGE_IDE << 16, 0xffff0000, NULL, 0 },

	{ 0 }
};


static int ATA_UNLOAD(void *p, void **dlrq, char *argv[]);
static int ATA_DCALL(void *ptr, __const__ char *dcall_type, int cmd, void *param);
static int ATA_ATTACH(ACTRL *a, ATA_ATTACH_PARAM *ap, int atapi);
static void ATA_DETACH(ACTRL *a, ATA_ATTACH_PARAM *ap, int atapi);
static void ATA_LOCK(APORT *p);
static void ATA_UNLOCK(APORT *p);
static int ACTRL_AUX_CMD(ATA_ATTACH_PARAM *ap, int cmd, ...);
static int AD_POST(ATARQ *rq);
extern AST_STUB ACTRL_IRQ;
static void ACTRL_END_REQUEST(APORT *p, int status);
static void ACTRL_SELECT_DEVICE(APORT *p, ATA_ATTACH_PARAM *me, ATA_ATTACH_PARAM *other, BIORQ *b1);
static void ACTRL_NONDMA_IRQ(APORT *p, __u8 status);
static void ACTRL_PIO_POLL(TIMER *t);
static void ACTRL_TIMEOUT(TIMER *t);
static void AD_ERROR(APORT *p, int dmastatus, int status);
static void AD_UNMAP_REQUEST(ATARQ *rq);

static void ATAPI_INIT(SCSI_ATTACH_PARAM *sa);
static int ATAPI_POST(SCSIRQ *rq);
static void ATAPI_IRQ(APORT *p, __u8 dmastatus, __u8 status);
static void ATAPI_STATUS_ERROR(APORT *p, __u8 status);
static void ATAPI_END_SENSE_REQUEST(APORT *p);
static void ATAPI_ERROR_RESET_DRIVE(APORT *p);
static void ATAPI_CANCEL(SCSIRQ *rq);

static char *ATA_FIND_TEST(__const__ void *id_, pci_id_t id, unsigned long *rev_)
{
	__u32 dev_id = (__u32)(unsigned long)id_;
	__u8 rev_min = (__u8)(unsigned long)rev_;
	__u8 rev_max = (__u8)((unsigned long)rev_ >> 8);
	__u8 rv;
	if (dev_id != PCI$READ_CONFIG_DWORD(id, PCI_VENDOR_ID)) return NULL;
	rv = PCI$READ_CONFIG_BYTE(id, PCI_REVISION_ID);
	if (rv < rev_min || rv > rev_max) return NULL;
	return "";
}

int ATA_FIND_DEV(ACTRL *a, __u32 dev_id, __u8 rev_min, __u8 rev_max)
{
	return !PCI$FIND_DEVICE((__const__ void *)(unsigned long)dev_id, 0, 0, 0, ATA_FIND_TEST, NULL, NULL, (void *)(unsigned long)(rev_min + (rev_max << 8)), 1);
}

/* type: 1 - std (0x1f0, 0x3f4), 2 - nonstd, 3 - both */

#define ATA_PCI_TEST_STD	1
#define ATA_PCI_TEST_NON_STD	2

static char *ATA_PCI_TEST(__const__ void *type_p, pci_id_t id, unsigned long *actrl_flags)
{
	char *name;
	__u16 command;
	__u8 prg_info;
	int native;
	int type = (int)(unsigned long)type_p;
	if (!(name = PCI$TEST_LIST(ata_dev, id, actrl_flags))) return NULL;
	command = PCI$READ_CONFIG_WORD(id, PCI_COMMAND);
		/* Linux ata-generic.c: do not enable PCI_COMMAND_IO */
	if (!(command & PCI_COMMAND_IO)) return NULL;
	prg_info = PCI$READ_CONFIG_BYTE(id, PCI_CLASS_PROG);
	/*__debug_printf("class: %02x\n", prg_info);*/
	native = (*actrl_flags & ACTRL_ALWAYS_NATIVE) || (prg_info & 5);
	if (!(type & (native ? ATA_PCI_TEST_NON_STD : ATA_PCI_TEST_STD))) return NULL;
	return name;
}

int main(int argc, char *argv[])
{
	int r;
	union {
		MALLOC_REQUEST mrq;
		CONTIG_AREA_REQUEST car;
		DLLZRQ lz;
		DEVICE_REQUEST devrq;
	} u;
	ACTRL *a;
	char pci_dll;
	char on_pci = 0;
	pci_id_t pci_id = 0;
	char *chipset_name = NULL;
	unsigned long actrl_flags;
	pci_id_t id = 0, id_mask = 0;
	int order;
	char type;
	char some_enabled;
	int i;
	char pci_id_str[__MAX_STR_LEN];
	int nochipsetinit;
	int bus_clock = -1;
	char **arg;
	int state;
	char *opt, *optend, *str;
		/* min SPL --- SPL_TIMER + 1 */
	struct __param_table params[] = {
		"NO_CHIPSET_INIT", __PARAM_BOOL, ~0, 1, NULL,
		"BUS_CLOCK", __PARAM_INT, 20, 67, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &nochipsetinit;
	params[1].__p = &bus_clock;
	u.lz.handle = KERNEL$DLRQ();
	u.lz.caller = NULL;
	strcpy(u.lz.modname, "PCI");
	SYNC_IO_CANCELABLE(&u.lz, KERNEL$DL_LOAD_LAZY);
	pci_dll = u.lz.status >= 0;
	order = 0;
	type = 0;
	nochipsetinit = 0;
	arg = argv;
	state = 0;
	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (!pci_dll) {
			if (__strcasexcmp("PCI", opt, optend)) goto bads;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: PCI NOT ENABLED: %s", u.lz.error);
			r = -ENODEV;
			goto ret0;
		}
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			bads:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		} else type = 1;
	}
	if (type == 1) {
		if (pci_dll && !PCI$FIND_DEVICE((__const__ void *)(id_mask ? ATA_PCI_TEST_STD | ATA_PCI_TEST_NON_STD : ATA_PCI_TEST_NON_STD), id, id_mask, order, ATA_PCI_TEST, &pci_id, &chipset_name, &actrl_flags, 0)) {
			on_pci = 1;
		} else {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: NO PCI DEVICE FOUND");
			r = -ENODEV;
			goto ret0;
		}
	} else {
		if (pci_dll && !PCI$FIND_DEVICE((__const__ void *)ATA_PCI_TEST_STD, id, id_mask, order, ATA_PCI_TEST, &pci_id, &chipset_name, &actrl_flags, 0)) {
			on_pci = 1;
		}
	}
	if (on_pci) {
		PCI$ENABLE_DEVICE(pci_id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	}

	u.mrq.size = sizeof(ACTRL) + sizeof(APORT);
	SYNC_IO_CANCELABLE(&u.mrq, &KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret1;
	}
	a = u.mrq.ptr;
	memset(a, 0, u.mrq.size);

	a->pci_dll = pci_dll;
	a->pci_id = pci_id;
	a->on_pci = on_pci;
	a->chipset_name = chipset_name;
	a->actrl_flags = actrl_flags;

	if (a->on_pci) {
		a->dev_id = PCI$READ_CONFIG_DWORD(a->pci_id, PCI_VENDOR_ID);
		a->dev_rev = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_REVISION_ID);
		if (bus_clock == -1) bus_clock = 33;
	}
	if (bus_clock == -1) bus_clock = 50;
	if (bus_clock == 33) a->bus_clock = 33333333;
	else if (bus_clock == 37) a->bus_clock = 37500000;
	else if (bus_clock == 41) a->bus_clock = 41666666;
	else a->bus_clock = bus_clock * 1000000;

	/* Initialize ports */
	a->n_ports = 2;
	for (i = 0; i < 2; i++) {
		io_t base_port, alt_port, dma_port;
		int irq;
		__u8 prg_info;
		a->port[i].n = i;
		if (a->on_pci) {
			dma_port = PCI$READ_IO_RESOURCE(a->pci_id, 4);
			if (dma_port) {
				if (i && (io_inb(dma_port + DMAPORT_STATUS) & DMASTATUS_SIMPLEX)) dma_port = 0;
				else dma_port += i * 8;
			}
			prg_info = PCI$READ_CONFIG_BYTE(a->pci_id, PCI_CLASS_PROG);
			if (prg_info & (1 << (i << 1)) || a->actrl_flags & ACTRL_ALWAYS_NATIVE) {
				base_port = PCI$READ_IO_RESOURCE(a->pci_id, !i ? 0 : 2);
				alt_port = PCI$READ_IO_RESOURCE(a->pci_id, !i ? 1 : 3);
				irq = PCI$READ_INTERRUPT_LINE(a->pci_id);
			} else {
				goto legacy;
			}
			/*__debug_printf("pci read %d: base = %08x, alt = %08x, dma = %08x, irq=%d\n", i, base_port, alt_port, dma_port, irq);*/
		} else {
			dma_port = 0;
			legacy:
			base_port = !i ? 0x1f0 : 0x170;
			alt_port = base_port + 0x204;
			irq = !i ? 14 : 15;
		}
		if (!base_port || !alt_port) {
			a->port[i].aport_flags = APORT_DISABLED;
			continue;
		}
		a->port[i].io = base_port;
		a->port[i].alt_io = alt_port;
		a->port[i].dma_io = dma_port;
		a->port[i].aport_flags = 0;
		if (a->actrl_flags & ACTRL_SATA) a->port[i].aport_flags |= APORT_SATA | APORT_IMPLICIT_DMA;
		a->port[i].irq = irq;
		a->port[i].ctrl = a;
		INIT_TIMER(&a->port[i].timeout);
	}
	if (a->port[0].aport_flags & a->port[1].aport_flags & APORT_DISABLED) {
		r = -ENOENT;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACTRL: NO PORT IS ENABLED");
		goto ret2;
	}

	if (a->port[0].io == 0x1f0 || a->port[1].io == 0x170) _snprintf(a->dev_name, __MAX_STR_LEN, "ATA$ACTRL");
	else if (a->on_pci) _snprintf(a->dev_name, __MAX_STR_LEN, "ATA$ACTRL@" PCI_ID_FORMAT, a->pci_id);
	else _snprintf(a->dev_name, __MAX_STR_LEN, "ATA$ACTRL@" IO_ID, !(a->port[0].aport_flags & APORT_DISABLED) ? a->port[0].io : a->port[1].io);

	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		a->port[i].irq_ast.fn = &ACTRL_IRQ;
		a->port[i].atapi_placeholder.port = &a->port[i];
	}

	if (!nochipsetinit) {
		for (i = 0; i < sizeof(controllers) / sizeof(*controllers); i++) {
			r = controllers[i](a);
			if (r < 0) {
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CHIPSET DRIVER REFUSED TO INITIALIZE THE CONTROLLER", a->dev_name);
				goto ret2;
			}
			if (r > 0) break;
		}
	}

	for (i = 0; i < a->n_ports; i++) {
		ASGLIST *sg = ATA$ALLOC_SGLIST(&a->port[i].sglist_dev);
		if (__unlikely(__IS_ERR(sg))) {
			if (__PTR_ERR(sg) != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", a->dev_name, strerror(-__PTR_ERR(sg)));
			r = __PTR_ERR(sg);
			goto ret2;
		}
		a->port[i].sglist = sg;
		if (a->port[i].dma_io) {
			/* clear possible DMA errors */
			io_outb(a->port[i].dma_io + DMAPORT_CMD, 0);
			io_outb(a->port[i].dma_io + DMAPORT_STATUS, io_inb(a->port[i].dma_io + DMAPORT_STATUS) | DMASTATUS_IRQ | DMASTATUS_ERROR);
		}
	}

	some_enabled = 0;
	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		char *e;
		a->port[i].io_range.start = a->port[i].io;
		a->port[i].io_range.len = PORT_RANGE;
		a->port[i].io_range.name = a->dev_name;
		if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&a->port[i].io_range))) {
			KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET IO RANGE "IO_FORMAT" - "IO_FORMAT" FOR PORT %d: %s", a->port[i].io_range.start, a->port[i].io_range.start + a->port[i].io_range.len - 1, i, strerror(-r));
			goto disable0;
		}
		a->port[i].alt_io_range.start = a->port[i].alt_io + ALTPORT_OFFSET;
		a->port[i].alt_io_range.len = ALTPORT_RANGE;
		a->port[i].alt_io_range.name = a->dev_name;
		if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&a->port[i].alt_io_range))) {
			KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET ALT IO RANGE "IO_FORMAT" - "IO_FORMAT" FOR PORT %d: %s", a->port[i].alt_io_range.start, a->port[i].alt_io_range.start + a->port[i].alt_io_range.len - 1, i, strerror(-r));
			goto disable1;
		}
		if (a->port[i].dma_io) {
			a->port[i].dma_io_range.start = a->port[i].dma_io;
			a->port[i].dma_io_range.len = DMAPORT_RANGE;
			a->port[i].dma_io_range.name = a->dev_name;
			if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&a->port[i].dma_io_range))) {
				KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET DMA IO RANGE "IO_FORMAT" - "IO_FORMAT" FOR PORT %d: %s", a->port[i].dma_io_range.start, a->port[i].dma_io_range.start + a->port[i].dma_io_range.len - 1, i, strerror(-r));
				a->port[i].dma_io = 0;
			}
		}
		if ((e = KERNEL$REQUEST_IRQ(a->port[i].irq, IRQ_AST_HANDLER | IRQ_SHARED, NULL, &a->port[i].irq_ast, &a->port[i].irq_ctrl, a->dev_name))) {
			KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, a->dev_name, "COULD NOT GET IRQ %d FOR PORT %d: %s", a->port[i].irq, i, e);

			if (a->port[i].dma_io) KERNEL$UNREGISTER_IO_RANGE(&a->port[i].dma_io_range);
			KERNEL$UNREGISTER_IO_RANGE(&a->port[i].alt_io_range);
			disable1:
			KERNEL$UNREGISTER_IO_RANGE(&a->port[i].io_range);
			disable0:
			a->port[i].aport_flags |= APORT_DISABLED;
		} else {
			some_enabled = 1;
		}
	}

	if (!some_enabled) {
		r = -ENXIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ENABLE ANY PORT", a->dev_name);
		goto ret3;
	}

	if (a->chipset_name && !*a->chipset_name) a->chipset_name = NULL;

	if (a->on_pci) _printf("%s: ATA CONTROLLER ON PCI: %s\n", a->dev_name, PCI$ID(pci_id_str, a->pci_id));
	if (a->chipset_name) _printf("%s: %s\n", a->dev_name, a->chipset_name);
	for (i = 0; i < a->n_ports; i++) {
		if (a->port[i].aport_flags & APORT_DISABLED) _printf("%s: PORT %d DISABLED\n", a->dev_name, i);
		else if (!a->port[i].dma_io) _printf("%s: PORT %d: IO "IO_FORMAT", ALTIO "IO_FORMAT", IRQ %d\n", a->dev_name, i, a->port[i].io, a->port[i].alt_io, a->port[i].irq);
		else _printf("%s: PORT %d: IO "IO_FORMAT", ALTIO "IO_FORMAT", DMAIO "IO_FORMAT", IRQ %d\n", a->dev_name, i, a->port[i].io, a->port[i].alt_io, a->port[i].dma_io, a->port[i].irq);
	}

	u.devrq.name = a->dev_name;
	u.devrq.driver_name = "ACTRL.SYS";
	u.devrq.flags = 0;
	u.devrq.init_root_handle = NULL;
	u.devrq.dev_ptr = a;
	u.devrq.dcall = ATA_DCALL;
	u.devrq.dcall_type = "ATA,SCSI";
	u.devrq.dctl = NULL;
	u.devrq.unload = ATA_UNLOAD;
	SYNC_IO_CANCELABLE(&u.devrq, KERNEL$REGISTER_DEVICE);
	if (u.devrq.status < 0) {
		if (u.devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", a->dev_name);
		r = u.devrq.status;
		goto ret3;
	}
	a->lnte = u.devrq.lnte;
	a->dlrq = KERNEL$TSR_IMAGE();
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s", a->dev_name);
	return 0;

	ret3:
	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		KERNEL$RELEASE_IRQ(&a->port[i].irq_ctrl, 0);
		if (a->port[i].dma_io) KERNEL$UNREGISTER_IO_RANGE(&a->port[i].dma_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].alt_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].io_range);
	}
	ret2:
	for (i = 0; i < a->n_ports; i++) if (a->port[i].sglist) ATA$FREE_SGLIST(a->port[i].sglist);
	KERNEL$UNIVERSAL_FREE(a);
	ret1:
	if (on_pci) PCI$FREE_DEVICE(pci_id);
	ret0:
	return r;
}

static int ATA_UNLOAD(void *p, void **dlrq, char *argv[])
{
	int r;
	int i, j;
	ACTRL *a = p;
	RAISE_SPL(SPL_DEV);
	while (a->dcall_users) KERNEL$SLEEP(1);
	for (i = 0; i < a->n_ports; i++) for (j = 0; j < 2; j++) if (a->port[i].device[j].attached) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME DRIVERS ARE USING THIS CONTROLLER", a->dev_name);
		return -EBUSY;
	}
	if ((r = KERNEL$DEVICE_UNLOAD(a->lnte, argv))) {
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	LOWER_SPL(SPL_ZERO);
	for (i = 0; i < a->n_ports; i++) if (!(a->port[i].aport_flags & APORT_DISABLED)) {
		if (__unlikely(a->port[i].setup_pio != NULL)) {
			int j;
			for (j = 0; j < 2; j++)
				a->port[i].setup_pio(&a->port[i], j);
		}
		KERNEL$RELEASE_IRQ(&a->port[i].irq_ctrl, 0);
		if (a->port[i].dma_io) KERNEL$UNREGISTER_IO_RANGE(&a->port[i].dma_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].alt_io_range);
		KERNEL$UNREGISTER_IO_RANGE(&a->port[i].io_range);
	}
	if (a->on_pci) PCI$FREE_DEVICE(a->pci_id);
	for (i = 0; i < a->n_ports; i++) if (a->port[i].sglist) ATA$FREE_SGLIST(a->port[i].sglist);
	KERNEL$UNIVERSAL_FREE(a);
	*dlrq = a->dlrq;
	return 0;
}

static int ATA_DCALL(void *ptr, __const__ char *dcall_type, int cmd, void *param)
{
	ACTRL *a = ptr;
	int r;
	int atapi;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_ATA_SCSI), spl)))
		KERNEL$SUICIDE("ATA_DCALL AT SPL %08X", spl);
	RAISE_SPL(SPL_ATA_SCSI);
	a->dcall_users++;
	atapi = !strcmp(dcall_type, "ATAPI");
	if (!strcmp(dcall_type, "ATA") || atapi) {
		switch (cmd) {
			case ATA_CMD_GET:
				r = ATA_ATTACH(a, param, atapi);
				break;
			case ATA_CMD_FREE:
				ATA_DETACH(a, param, atapi);
				r = 0;
				break;
			default:
				r = -ENOOP;
				break;
		}
	} else if (!strcmp(dcall_type, "SCSI")) {
		switch (cmd) {
			case SCSI_CMD_GET:
				r = ATA$PI_ATTACH(a, ATA_DCALL, param, a->n_ports, 2);
				if (!r) ATAPI_INIT(param);
				break;
			case SCSI_CMD_FREE:
				ATA$PI_DETACH(a, ATA_DCALL, param);
				r = 0;
				break;
			default:
				r = -ENOOP;
				break;
		}
	} else KERNEL$SUICIDE("ATA_DCALL: UNKNOWN DCALL TYPE: %s", dcall_type);
	a->dcall_users--;
	LOWER_SPLX(spl);
	return r;
}

static int ATA_ATTACH(ACTRL *a, ATA_ATTACH_PARAM *ap, int atapi)
{
	APORT *p;
	int pio_mask = -1;
	int dma_mask = -1, dma_mode = -1;
	int i, j, n;
	int port = -1, device = -1;
	char *arg = ap->param;
	int state = 0;
	unsigned dev_flags = !atapi ? 0 : DEV_F_ATAPI;
	struct __param_table params[] = {
		"PORT", __PARAM_INT, 0, MAXINT, NULL,
		"DEVICE", __PARAM_INT, 0, 2, NULL,
		"MASKIRQ", __PARAM_BOOL, DEV_F_MASKIRQ, DEV_F_MASKIRQ, NULL,
		"NOMASKIRQ", __PARAM_BOOL, DEV_F_MASKIRQ, 0, NULL,
		"IO32", __PARAM_BOOL, DEV_F_IO32, DEV_F_IO32, NULL,
		"NOIO32", __PARAM_BOOL, DEV_F_IO32, 0, NULL,
		"PIO", __PARAM_INT, 0, 5, NULL,
		"SDMA", __PARAM_BOOL_X, ~0, IDE_XFER_SDMA_0, NULL,
		"SDMA", __PARAM_INT, 0, 3, NULL,
		"WDMA", __PARAM_BOOL_X, ~0, IDE_XFER_WDMA_0, NULL,
		"WDMA", __PARAM_INT, 0, 3, NULL,
		"UDMA", __PARAM_BOOL_X, ~0, IDE_XFER_UDMA_0, NULL,
		"UDMA", __PARAM_INT, 0, 7, NULL,
		NULL, 0, 0, 0, NULL,
	};
	params[0].__p = &port;
	params[1].__p = &device;
	params[2].__p = &dev_flags;
	params[3].__p = &dev_flags;
	params[4].__p = &dev_flags;
	params[5].__p = &dev_flags;
	params[6].__p = &pio_mask;
	params[7].__p = &dma_mode;
	params[8].__p = &dma_mask;
	params[9].__p = &dma_mode;
	params[10].__p = &dma_mask;
	params[11].__p = &dma_mode;
	params[12].__p = &dma_mask;
	if (__parse_extd_param(&arg, &state, params, NULL, NULL, NULL, NULL)) {
		ap->msg = "SYNTAX ERROR";
		return -EBADSYN;
	}
	if (pio_mask >= 0) dev_flags |= IDE_XFER_PIO_0 << pio_mask;
	if (dma_mask >= 0) dev_flags |= dma_mode << dma_mask;
	if (a->actrl_flags & ACTRL_FORCE_MASKIRQ) dev_flags |= ACTRL_FORCE_MASKIRQ;
	if (port >= a->n_ports) {
		ap->msg = "INVALID PORT";
		return -ERANGE;
	}
	if (port >= 0 && device >= 0 && a->port[port].device[device].attached) {
		ap->msg = "DRIVER ALREADY ATTACHED";
		return -EBUSY;
	}
	if (port >= 0 && a->port[port].aport_flags & APORT_DISABLED) {
		ap->msg = "PORT IS DISABLED";
		return -ENXIO;
	}
	for (i = 0; i < a->n_ports; i++) {
		if (port >= 0 && port != i) continue;
		if (a->port[i].aport_flags & APORT_DISABLED) continue;
		for (j = 0; j < 2; j++) {
			__u8 status;
			int slept, reset;

			if (device >= 0 && device != j) continue;
			if (a->port[i].device[j].attached) continue;
			if (ap->port) {
				if (ap->port > &a->port[i]) continue;
				if (ap->port == &a->port[i] && ap->device >= j) continue;
			}
			/* attach it */
			p = &a->port[i];
			p->device[j].attached = ap;
			ap->port = p;
			ap->device = j;

		/* Attached */

			p->device[j].dev_flags = dev_flags;
			n = i * 2 + j;
			if (!strcmp(a->dev_name, "ATA$ACTRL")) _snprintf(*ap->dev_name, sizeof *ap->dev_name, "%s%d", ap->dev_prefix, n);
			else _snprintf(*ap->dev_name, sizeof *ap->dev_name, "%s%d@%s", ap->dev_prefix, n, a->dev_name);
			ap->post = AD_POST;
			ap->aux_cmd = ACTRL_AUX_CMD;
			ap->queue_depth = 1;

		/* Drain pending requests */

			ATA_LOCK(p);

		/* Test for BUSY */

			reset = 0;
			after_reset:

	/* Linux source says that non-ready slave can break master detection */
			if (!p->device[j ^ 1].attached) {
				slept = 0;
				after_other_sleep:
				io_outb(p->io + PORT_DRIVE, (j ^ 1) * ATA_DEVICE_DRIVE);
				KERNEL$UDELAY(10);
				status = io_inb(p->io + PORT_STATUS);
				if (status & ATA_STATUS_BSY && status != 0xff) {
					if (slept >= ap->probe_timeout / POLL_TIME) goto other_not_ready;
					KERNEL$SLEEP(POLL_TIME);
					slept++;
					goto after_other_sleep;
				}
				other_not_ready:;
			}

			slept = 0;
			after_sleep:
			io_outb(p->io + PORT_DRIVE, j * ATA_DEVICE_DRIVE);
			KERNEL$UDELAY(10);
			status = io_inb(p->io + PORT_STATUS);
			if (status & ATA_STATUS_BSY) {
				if (slept >= ap->probe_timeout / POLL_TIME) goto detach_cont;
				if (status == 0xff && !reset && slept >= RESET_FF_TIMEOUT / POLL_TIME) goto detach_cont;
				KERNEL$SLEEP(POLL_TIME);
				slept++;
				goto after_sleep;
			}
			if (reset) {
				int error = io_inb(p->io + PORT_ERROR);
				if (error != 1) {
					KERNEL$SYSLOG(__SYSLOG_HW_ERROR, *ap->dev_name, "RESET FAILED, STATUS %02X, ERROR %02X", status, error);
					goto detach_cont;
				}
			}
			if (status & ATA_STATUS_DF) {
				detach_cont:
				ATA_UNLOCK(p);
				ATA_DETACH(a, ap, atapi);
				continue;
			}
			if (status & ATA_STATUS_DRQ) {
				if (reset) goto detach_cont;
				if (p->device[j ^ 1].attached && !atapi) {
					cant_reset:
					KERNEL$SYSLOG(__SYSLOG_HW_ERROR, *ap->dev_name, "DRIVE LEFT IN INCONSISTENT STATE (STATUS %02X), SKIPPING", status);
					goto detach_cont;
				}
				if (!a->get_avail_xfer && !(p->aport_flags & APORT_IMPLICIT_DMA) && !atapi) goto cant_reset;
				KERNEL$SYSLOG(__SYSLOG_HW_ERROR, *ap->dev_name, "DRIVE LEFT IN INCONSISTENT STATE (STATUS %02X), RESETTING", status);
				reset = 1;
				if (!atapi) {
					io_outb(p->alt_io + ALTPORT_CTRL, ATA_CTRL_ONE | ATA_CTRL_SRST);
					KERNEL$UDELAY(10);
					io_outb(p->alt_io + ALTPORT_CTRL, ATA_CTRL_ONE);
					KERNEL$UDELAY(10);
				} else {
					io_outb(p->io + PORT_CMD, ATA_CMD_DEVICE_RESET);
					KERNEL$UDELAY(10);
					goto after_reset;
				}
				goto after_reset;
			}
			if (!atapi && !(status & ATA_STATUS_DRDY)) {
				goto detach_cont;
			}
			ATA_UNLOCK(p);

		/* Now we are surely attached */

			if (p->aport_flags & APORT_SATA && p->device[j].dev_flags & IDE_XFER_MASK) {
				p->device[j].dev_flags &= ~IDE_XFER_MASK;
				KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, *ap->dev_name, "CAN'T SET TRANSFER MODE ON SATA");
			}

			return 0;
		}
	}
	return -ENODEV;

}

static void ATA_DETACH(ACTRL *a, ATA_ATTACH_PARAM *ap, int atapi)
{
	if (ap->port < a->port || ap->port >= a->port + a->n_ports)
		KERNEL$SUICIDE("ATA_DETACH: %s: DETACHING INVALID STRUCTURE, PORT %p, OUR PORT %p, N_PORTS %d", a->dev_name, ap->port, a->port, a->n_ports);
	if (ap->device >= 2)
		KERNEL$SUICIDE("ATA_DETACH: %s: INVALID DEVICE %d", a->dev_name, ap->device);
	if (ap->port->device[ap->device].attached != ap)
		KERNEL$SUICIDE("ATA_DETACH: %s: STRUCTURE %p NOT ATTACHED, CURRENTLY ATTACHED %p", a->dev_name, ap, ap->port->device[ap->device].attached);
	if (atapi != !!(ap->port->device[ap->device].dev_flags & DEV_F_ATAPI))
		KERNEL$SUICIDE("ATA_DETACH: %s: ATAPI FLAG (%d) DOES NOT MATCH, DEV_FLAGS %X", a->dev_name, atapi, ap->port->device[ap->device].dev_flags);
	ap->port->device[ap->device].attached = NULL;
}

static void ATA_LOCK(APORT *p)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ATA_LOCK AT SPL %08X", KERNEL$SPL);
	while (__unlikely(p->aport_flags & APORT_LOCK)) KERNEL$SLEEP(1);
	p->aport_flags |= APORT_LOCK;
	while (__unlikely(p->current_rq != 0)) KERNEL$SLEEP(1);
}

static void ATA_UNLOCK(APORT *p)
{
	int j;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ATA_UNLOCK AT SPL %08X", KERNEL$SPL);
	KERNEL$SLEEP(1);	/* Ignore a possible interrupt */
	p->aport_flags &= ~APORT_LOCK;
	for (j = 0; j < 2; j++) {
		ATA_ATTACH_PARAM *a = p->device[j].attached;
		if (a) while (a->dequeue(a)) ;
	}
}

static int ACTRL_AUX_CMD(ATA_ATTACH_PARAM *ap, int cmd, ...)
{
	APORT *p;
	int r;
	unsigned avail, avail_unsupp;
	va_list args;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ACTRL_AUX_CMD AT SPL %08X", KERNEL$SPL);
	p = ap->port;
	va_start(args, cmd);
	switch (cmd) {
		case ATA_AUX_GET_PROTOCOL_MASK: {
			__u16 *ident = va_arg(args, __u16 *);
			unsigned xfer_mask = va_arg(args, unsigned);
			ident = ident;	/* against warning */
			r = (1 << ATA_PROTOCOL_NODATA) | (1 << ATA_PROTOCOL_PIO);
			if (!p->dma_io) break;
			if (!(p->aport_flags & APORT_IMPLICIT_DMA)) {
				__u8 dmast = io_inb(p->dma_io + DMAPORT_STATUS);
				__u8 mask = !ap->device ? DMASTATUS_MASTER : DMASTATUS_SLAVE;
				if (!(dmast & mask)) {
					avail = 0, avail_unsupp = 0;
					if (p->ctrl->get_avail_xfer) p->ctrl->get_avail_xfer(p, ap->device, &avail, &avail_unsupp);
					if (!((avail | avail_unsupp) & xfer_mask & IDE_XFER_MASK_DMA)) {
						break;
					}
				}
			}
			r |= 1 << ATA_PROTOCOL_DMA;
			break;
		}
		case ATA_AUX_SETUP_PIO_XFER:
		case ATA_AUX_SETUP_DMA_XFER: {
			__u16 *ident = va_arg(args, __u16 *);
			unsigned xfer_mask = va_arg(args, unsigned);
			jiffies_lo_t timeout = va_arg(args, jiffies_lo_t);
			avail = 0, avail_unsupp = 0;
			if (p->ctrl->get_avail_xfer) p->ctrl->get_avail_xfer(p, ap->device, &avail, &avail_unsupp);
			r = (cmd == ATA_AUX_SETUP_PIO_XFER ? ATA$SETUP_PIO_XFER : ATA$SETUP_DMA_XFER)(ap, ident, p->device[ap->device].dev_flags & IDE_XFER_MASK, xfer_mask, avail, avail_unsupp, p->ctrl->set_xfer, timeout);
			break;
		}
		case ATA_AUX_FIXUP_IDENT: {
			__u16 *ident = va_arg(args, __u16 *);
			r = 0;
			if (p->ctrl->fixup_ident) r = p->ctrl->fixup_ident(p, ap->device, ident);
			break;
		}
		case ATA_AUX_GET_TRANSFER_SIZE: {
			int dma = va_arg(args, int);
			r = ATA$SGLIST_RECOMMENDED_SIZE(p->sglist, dma);
			if (p->aport_flags & APORT_SIZE_LIMIT && r > p->max_len) r = p->max_len;
			break;
		}
		case ATA_AUX_SET_ATAPI_FLAGS: {
			unsigned flags = va_arg(args, unsigned);
			unsigned new_flags = 0;
			if (flags & ~(ATA_AUX_ATAPI_FLAGS_PROTOCOL | ATA_AUX_ATAPI_FLAGS_DMADIR | ATA_AUX_ATAPI_FLAGS_DRQ_INT | ATA_AUX_ATAPI_FLAGS_LONG_DRQ | ATA_AUX_ATAPI_FLAGS_CMD_16)) {
				r = -EOPNOTSUPP;
				break;
			}
			if ((flags & ATA_AUX_ATAPI_FLAGS_PROTOCOL) == ATA_PROTOCOL_PIO) {
			} else if ((flags & ATA_AUX_ATAPI_FLAGS_PROTOCOL) == ATA_PROTOCOL_DMA) {
				new_flags |= DEV_F_ATAPI_DMA;
			} else {
				r = -EOPNOTSUPP;
				break;
			}
			if (flags & ATA_AUX_ATAPI_FLAGS_DMADIR) new_flags |= DEV_F_ATAPI_DMADIR;
			if (flags & ATA_AUX_ATAPI_FLAGS_DRQ_INT) new_flags |= DEV_F_ATAPI_DRQ_INT;
			if (flags & ATA_AUX_ATAPI_FLAGS_LONG_DRQ) new_flags |= DEV_F_ATAPI_LONG_DRQ;
			if (flags & ATA_AUX_ATAPI_FLAGS_CMD_16) new_flags |= DEV_F_ATAPI_CMD_16;
			p->device[ap->device].dev_flags &= ~(DEV_F_ATAPI_DMADIR | DEV_F_ATAPI_DRQ_INT | DEV_F_ATAPI_CMD_16 | DEV_F_ATAPI_DMA);
			p->device[ap->device].dev_flags |= new_flags;
			r = 0;
			break;
		}
		default: {
			r = -ENOOP;
			break;
		}
	}
	va_end(args);
	return r;
}

static char *RQ_2_DEV_NAME(ATARQ *rq)
{
	return *rq->port->device[rq->device].attached->dev_name;
}

static void ATA_INS(APORT *p, int bit32, void *data, unsigned len)
{
	if (bit32 && __likely(!(len & 1))) {
		io_insl(p->io + PORT_DATA, data, len >> 1);
	} else {
		io_insw(p->io + PORT_DATA, data, len);
	}
}

static void ATA_OUTS(APORT *p, int bit32, void *data, unsigned len)
{
	if (bit32 && __likely(!(len & 1))) {
		io_outsl(p->io + PORT_DATA, data, len >> 1);
	} else {
		io_outsw(p->io + PORT_DATA, data, len);
	}
}

static __finline__ void setup_pio(APORT *p, __u8 device)
{
	if (__unlikely(p->setup_pio != NULL)) p->setup_pio(p, device);
}

static __finline__ void setup_dma(APORT *p, ATARQ *rq)
{
	if (__unlikely(p->aport_flags & APORT_SPECIAL_DMA)) {
		p->setup_dma(p, rq);
	} else {
		setup_dma_generic(p, rq);
	}
}

static __finline__ void start_dma(APORT *p, ATARQ *rq)
{
	io_outb(p->dma_io + DMAPORT_CMD, (~rq->atarq_flags & DMACMD_READ) | DMACMD_BMEN);
}

static int AD_POST(ATARQ *rq)
{
	APORT *p = rq->port;
	int r;
	if (__unlikely(DMACMD_READ != ATARQ_TO_DEVICE))
		KERNEL$SUICIDE("AD_POST: BITS DON'T MATCH: DMACMD_READ %02X, ATARQ_TO_DEVICE %02X", DMACMD_READ, ATARQ_TO_DEVICE);
	if (__unlikely(rq->atarq_flags & ATARQ_RETRIED)) {
#if __DEBUG >= 1
		if (__unlikely(rq != p->current_rq))
			KERNEL$SUICIDE("AD_POST: RETRYING NON-CURRENT REQUEST %p, CURRENT %p", rq, p->current_rq);
#endif
		goto retrying;
	}
	if (__unlikely(p->aport_flags & APORT_LOCK)) return -EAGAIN;
	if (__unlikely(p->current_rq != NULL)) return -EAGAIN;
	if (__unlikely(p->aport_flags & APORT_SIZE_LIMIT)) {
		if (__likely(rq->atarq_flags & ATARQ_SET_SIZE))
			if (__unlikely(rq->len > p->max_len)) rq->len = p->max_len;
	}
	if (__likely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
		r = ATA$MAP_DMA(p->sglist, &rq->desc, rq->len, PF_WRITE << ((rq->atarq_flags / ATARQ_TO_DEVICE) & 1), 0);
		test_error_set_size:
		if (__unlikely(r < 0)) return r;
		if (__likely(rq->atarq_flags & ATARQ_SET_SIZE)) {
			rq->len = r;
			rq->fis.nsect = r >> BIO_SECTOR_SIZE_BITS;
		} else if (__unlikely(r != rq->len)) {
			AD_UNMAP_REQUEST(rq);
			return -EVSPACEFAULT;
		}
	} else if (__unlikely((rq->atarq_flags & ATARQ_PROTOCOL) != ATA_PROTOCOL_NODATA)) {
		r = ATA$MAP_PIO(p->sglist, &rq->desc, rq->len, PF_WRITE << ((rq->atarq_flags / ATARQ_TO_DEVICE) & 1), 0);
		p->pio_n_sect = r >> BIO_SECTOR_SIZE_BITS;
		p->pio_sg_pos = 0;
		goto test_error_set_size;
	}
	p->current_rq = rq;

	retrying:
	if (__likely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
		setup_dma(p, rq);
	} else {
		setup_pio(p, rq->device);
	}
	io_outb(p->io + PORT_DRIVE, rq->fis.device | (rq->device * ATA_DEVICE_DRIVE));
	if (__unlikely(rq->atarq_flags & ATARQ_VALID_FEATURE)) {
		if (rq->atarq_flags & ATARQ_VALID_48BIT) {
			io_outb(p->io + PORT_FEATURES, rq->fis.feature8);
		}
		io_outb(p->io + PORT_FEATURES, rq->fis.feature0);
	}
	if (rq->atarq_flags & ATARQ_VALID_48BIT) {
		io_outb(p->io + PORT_COUNT, rq->fis.nsect >> 8);
		io_outb(p->io + PORT_LBA_L, rq->fis.lba24);
		io_outb(p->io + PORT_LBA_M, rq->fis.lba24 >> 8);
		io_outb(p->io + PORT_LBA_H, rq->fis.lba40);
	}
	io_outb(p->io + PORT_COUNT, rq->fis.nsect);
	io_outb(p->io + PORT_LBA_L, rq->fis.lba0);
	io_outb(p->io + PORT_LBA_M, rq->fis.lba0 >> 8);
	io_outb(p->io + PORT_LBA_H, rq->fis.lba16);
	io_outb(p->io + PORT_CMD, rq->fis.command);
	if (__likely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
		start_dma(p, rq);
		p->timeout.fn = ACTRL_TIMEOUT;
		KERNEL$SET_TIMER(rq->timeout, &p->timeout);
		return 0;
	} else {
/* PIO is both interrupt driven and polled, there are reportedly some
   controllers that do not assert interrupt on PIO transfers.
   Also, I found one CD-ROM drive that magically disables PIO interrupt
   if it's reset at unexpected time.
*/
		p->pio_timeout = KERNEL$GET_JIFFIES_LO();
		p->timeout.fn = ACTRL_PIO_POLL;
		KERNEL$SET_TIMER(0, &p->timeout);
		return 0;
	}
}


DECL_AST(ACTRL_IRQ, SPL_ATA_SCSI, AST)
{
	ATARQ *rq;
	APORT *p = GET_STRUCT(RQ, APORT, irq_ast);
	__u8 dmastatus, status;
	dmastatus = 0;	/* against warning */
	if (__likely(p->dma_io != 0)) {
/* ICH SATA (and maybe others) need to clear DMA IRQ even on non-dma commands */
		dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
		if (__unlikely(!(dmastatus & (DMASTATUS_IRQ | DMASTATUS_ERROR)))) {
			rq = p->current_rq;
			if (rq && rq->atarq_flags & ATA_PROTOCOL_DMA) goto done;
	/* Maybe some controllers don't set DMASTATUS_IRQ on non-DMA transfers.
	   All controllers that I have set it. */
			goto test_pci_int;
		}
		io_outb(p->dma_io + DMAPORT_CMD, 0);
		io_outb(p->dma_io + DMAPORT_STATUS, dmastatus | DMASTATUS_IRQ | DMASTATUS_ERROR);
	} else if (p->ctrl->on_pci) {
	/* The interrupt may be shared and caused by something else...
	   I'm not sure if reading STATUS is atomic to test-and-clear
	   interrupt (i.e. may it return BSY bit and clear ongoing interrupt?)
	   ... Linux and BSD also read ALTSTATUS here.
	*/
		test_pci_int:
		status = io_inb(p->alt_io + ALTPORT_STATUS);
		if (__unlikely(status & ATA_STATUS_BSY)) goto done;
	}

	status = io_inb(p->io + PORT_STATUS);
	rq = p->current_rq;
	if (__unlikely(!rq)) {
		/*
		This may happen because one of these:
		- int 15 is asseted by 2nd PIC if it has to send masked irq
		- interrupt line is dropped too slowly
		- during disk probing/reset
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, p->ctrl->dev_name, "INTERRUPT WITH NO ACTIVE REQUEST. DMA STATUS %02X, DRIVE STATUS %02X", dmastatus, status);
		*/
		goto done;
	}
	if (__unlikely(status & ATA_STATUS_BSY)) {
		if (__unlikely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, RQ_2_DEV_NAME(rq), "DMA IS READY BUT DRIVE NOT. DMA STATUS %02X, DRIVE STATUS %02X", dmastatus, status);
		}
		goto done;
	}
	if (__unlikely(rq == &p->atapi_placeholder)) {
		ATAPI_IRQ(p, dmastatus, status);
	} else if (__likely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
		if (__unlikely(dmastatus & DMASTATUS_ERROR)) {
			AD_ERROR(p, dmastatus, status);
			goto done;
		}
		if (__unlikely((status & (ATA_STATUS_ERROR | ATA_STATUS_DRQ | ATA_STATUS_DF | ATA_STATUS_DRDY)) != ATA_STATUS_DRDY)) {
			AD_ERROR(p, dmastatus, status);
			goto done;
		}
		ATA$UNMAP_DMA(p->sglist);
		ACTRL_END_REQUEST(p, 0);
	} else {
		ACTRL_NONDMA_IRQ(p, status);
	}

	done:
	KERNEL$ADD_RANDOMNESS(&p->random_ctx, NULL, 0);
	p->irq_ctrl.eoi();
	RETURN;
}

static void ACTRL_END_REQUEST(APORT *p, int status)
{
	unsigned dev;
	ATARQ *rq;
	ATA_ATTACH_PARAM *me, *other;
	BIORQ *b1;
	rq = p->current_rq;
	rq->status = status;
	KERNEL$DEL_TIMER(&p->timeout);
	p->current_rq = NULL;
	rq->done(rq);
	dev = rq->device;
	me = p->device[dev].attached;
	if (__likely(!(other = p->device[dev ^ 1].attached)) || __likely(!(b1 = other->probe_queue(other)))) {
		int r = me->dequeue(me);
		if (__unlikely(!((long)me->port->current_rq | (r - 1))))
			while (me->dequeue(me)) ;
	} else {
		ACTRL_SELECT_DEVICE(me->port, me, other, b1);
	}
}

static void ACTRL_SELECT_DEVICE(APORT *p, ATA_ATTACH_PARAM *me, ATA_ATTACH_PARAM *other, BIORQ *b1)
{
	BIORQ *b0;
	long c;
	int r;
	b0 = me->probe_queue(me);
	compare_again:
	if (!b0) {
		deq_other:
		me = other;
		goto deq_me;
	}
	if (!b1) {
		deq_me:
		r = me->dequeue(me);
		if (__unlikely(!((long)p->current_rq | (r - 1))))
			while (me->dequeue(me)) ;
		return;
	}
	c = BIOQUE$COMPARE_REQUESTS(b0, me->biosched, b1, other->biosched);
	if (c < 0) {
		if (__unlikely(!me->dequeue(me))) goto deq_other;
		if (__likely(p->current_rq != NULL)) return;
		b0 = me->probe_queue(me);
		goto compare_again;
	} else {
		if (__unlikely(!other->dequeue(other))) goto deq_me;
		if (__likely(p->current_rq != NULL)) return;
		b1 = other->probe_queue(other);
		goto compare_again;
	}
}

static void ACTRL_NONDMA_IRQ(APORT *p, __u8 status)
{
	int n;
	ATARQ *rq = p->current_rq;
	p->pio_timeout = KERNEL$GET_JIFFIES_LO();
	if (__likely((rq->atarq_flags & ATARQ_PROTOCOL) == ATA_PROTOCOL_NODATA)) {
		__u8 drdy;
		end_test_status:
		drdy = p->device[rq->device].dev_flags & DEV_F_ATAPI ? 0 : ATA_STATUS_DRDY;
		if (__unlikely((status & (ATA_STATUS_ERROR | ATA_STATUS_DRQ | ATA_STATUS_DF | drdy)) != drdy)) {
			ad_error:
			AD_ERROR(p, -1, status);
			return;
		}
		AD_UNMAP_REQUEST(rq);
		ACTRL_END_REQUEST(p, 0);
		return;
	}
#if __DEBUG >= 1
	if (__likely((rq->atarq_flags & ATARQ_PROTOCOL) != ATA_PROTOCOL_PIO))
		KERNEL$SUICIDE("ACTRL_NONDMA_IRQ: UNKNOWN PROTOCOL, FLAGS %X", rq->atarq_flags);
#endif
	if (__unlikely(!p->pio_n_sect)) goto end_test_status;
	if (!(status & ATA_STATUS_DRQ)) goto ad_error;
	n = rq->multicount;
	if (__unlikely(!n)) KERNEL$SUICIDE("ACTRL_NONDMA_IRQ: ZERO MULTICOUNT");
	if (__unlikely(n > p->pio_n_sect)) n = p->pio_n_sect;
	ATA_DI(p, rq->device);
	do {
		void *data = ATA$MAP_PIO_SECTOR(p->sglist, p->pio_sg_pos);
		ATA$SET_NEXT_PIO_SECTOR(p->sglist, &p->pio_sg_pos);
		if (__likely(!(rq->atarq_flags & ATARQ_TO_DEVICE))) {
			ATA_INS(p, p->device[rq->device].dev_flags & DEV_F_IO32, data, 256);
		} else {
			ATA_OUTS(p, p->device[rq->device].dev_flags & DEV_F_IO32, data, 256);
		}
		ATA$UNMAP_PIO_SECTOR(p->sglist, data);
		p->pio_n_sect--;
	} while (--n > 0);
	ATA_EI(p, rq->device);
	if (__unlikely((status & (ATA_STATUS_ERROR | ATA_STATUS_DRQ | ATA_STATUS_DF)) != ATA_STATUS_DRQ)) goto ad_error;
	if (__unlikely(!p->pio_n_sect) && __likely(!(rq->atarq_flags & ATARQ_TO_DEVICE))) {
		status = io_inb(p->io + PORT_STATUS);
		if (__unlikely(status & (ATA_STATUS_BSY | ATA_STATUS_DRQ))) {
			KERNEL$DEL_TIMER(&p->timeout);
			p->pio_timeout = KERNEL$GET_JIFFIES_LO();
			p->timeout.fn = ACTRL_PIO_POLL;
			KERNEL$SET_TIMER(0, &p->timeout);
			return;
		}
		goto end_test_status;
	}
}

static void ACTRL_PIO_POLL(TIMER *t)
{
	__u8 status;
	APORT *p = GET_STRUCT(t, APORT, timeout);
	LOWER_SPL(SPL_ATA_SCSI);
	status = io_inb(p->io + PORT_STATUS);
	if (__likely(!(status & ATA_STATUS_BSY))) {
		if (status & ATA_STATUS_DRQ && (p->current_rq->atarq_flags & ATARQ_PROTOCOL) == ATA_PROTOCOL_PIO && !p->pio_n_sect) goto not_yet;
		VOID_LIST_ENTRY(&p->timeout.list);
		KERNEL$UDELAY(10);	/* Linux code has it because of some buggy disks */
		status = io_inb(p->io + PORT_STATUS);
		ACTRL_NONDMA_IRQ(p, status);
		return;
	}
	not_yet:
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - p->pio_timeout > p->current_rq->timeout)) {
		ACTRL_TIMEOUT(&p->timeout);
		return;
	}
	KERNEL$SET_TIMER(0, &p->timeout);
}

static int DONT_RETRY(int dmastatus, int status, int error)
{
	return status & ATA_STATUS_BSY || (status & ATA_STATUS_ERROR && error & (ATA_ERROR_NM | ATA_ERROR_ABORT | ATA_ERROR_IDNF | ATA_ERROR_UNC));
}

static void AD_RETRY(ATARQ *rq)
{
	KERNEL$DEL_TIMER(&rq->port->timeout);
	rq->atarq_flags |= ATARQ_RETRIED;
	AD_POST(rq);
}

static void ACTRL_TIMEOUT(TIMER *t)
{
	char dmastr[16];
	int dmastatus;
	__u8 status, error;
	ATARQ *rq;
	APORT *p = GET_STRUCT(t, APORT, timeout);
	LOWER_SPL(SPL_ATA_SCSI);
	VOID_LIST_ENTRY(&p->timeout.list);
	*dmastr = 0;
	dmastatus = -1;
	if (__likely(p->dma_io != 0)) {
		dmastatus = io_inb(p->dma_io + DMAPORT_STATUS);
		io_outb(p->dma_io + DMAPORT_CMD, 0);
		io_outb(p->dma_io + DMAPORT_STATUS, dmastatus | DMASTATUS_IRQ | DMASTATUS_ERROR);
		_snprintf(dmastr, sizeof dmastr, "DMA STATUS %02X, ", dmastatus);
	}
	status = io_inb(p->io + PORT_STATUS);
	error = io_inb(p->io + PORT_ERROR);
	rq = p->current_rq;
	if (DONT_RETRY(dmastatus, status, error)) rq->retries = 0;
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, RQ_2_DEV_NAME(rq), "TIMEOUT ON COMMAND %s. %sDRIVE STATUS %02X, ERROR %02X - %s", ATA$COMMAND_NAME(rq->fis.command, rq->fis.feature0), dmastr, status, error, rq->retries ? "RETRYING" : "ABORTING");
	if (rq->retries--) {
		AD_RETRY(rq);
	} else {
		AD_UNMAP_REQUEST(rq);
		ACTRL_END_REQUEST(p, -ETIMEDOUT);
	}
}

static void AD_ERROR(APORT *p, int dmastatus, int status)
{
	int code;
	char *description;
	char dmastr[16];
	ATARQ *rq = p->current_rq;
	__u8 error = io_inb(p->io + PORT_ERROR);
	__u8 drdy = p->device[rq->device].dev_flags & DEV_F_ATAPI ? 0 : ATA_STATUS_DRDY;
	if (DONT_RETRY(dmastatus, status, error)) rq->retries = 0;	/* do not try again */
	code = -EIO;
	if (status & ATA_STATUS_DF) {
		if (rq->atarq_flags & ATARQ_ERROR_ALLOW_ABORT) {
			description = NULL;
			code = -EOPNOTSUPP;
		} else {
			description = "DEVICE FAULT";
		}
	} else if (status & ATA_STATUS_ERROR && error & ATA_ERROR_UNC) {
		description = "UNCORRECTABLE ERROR";
	} else if (status & ATA_STATUS_ERROR && error & (ATA_ERROR_NM | ATA_ERROR_MC | ATA_ERROR_MCR)) {
		description = "MEDIA ERROR", code = -ENOMEDIA;
	} else if (status & ATA_STATUS_ERROR && error & ATA_ERROR_ABORT) {
		if (rq->atarq_flags & ATARQ_ERROR_ALLOW_ABORT) {
			description = NULL;
			code = -EOPNOTSUPP;
		} else {
			description = "ABORT";
		}
	} else if (dmastatus >= 0 && (dmastatus & DMASTATUS_ERROR || (status & ATA_STATUS_ERROR && error & ATA_ERROR_ICRC))) {
		description = "DMA CRC ERROR";
	} else if ((status & (ATA_STATUS_DRQ | drdy)) != drdy) {
		description = "PROTOCOL VIOLATION";
	} else {
		if (rq->atarq_flags & ATARQ_ERROR_ALLOW_ABORT) {
			description = NULL;
			code = -EOPNOTSUPP;
		} else {
			description = "ERROR";
		}
	}
	if (description) {
		if (dmastatus >= 0) {
			_snprintf(dmastr, sizeof dmastr, "DMA STATUS %02X, ", dmastatus);
		} else {
			*dmastr = 0;
		}
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, RQ_2_DEV_NAME(rq), "%s ON COMMAND %s, %sDRIVE STATUS %02X, ERROR %02X - %s", description, ATA$COMMAND_NAME(rq->fis.command, rq->fis.feature0), dmastr, status, error, rq->retries ? "RETRYING" : "ABORTING");
	}
	if (rq->retries--) {
		AD_RETRY(rq);
	} else {
		AD_UNMAP_REQUEST(rq);
		ACTRL_END_REQUEST(p, code);
	}
}

static void AD_UNMAP_REQUEST(ATARQ *rq)
{
	if (__likely(rq->atarq_flags & ATA_PROTOCOL_DMA)) {
		ATA$UNMAP_DMA(rq->port->sglist);
	} else if (__likely((rq->atarq_flags & ATARQ_PROTOCOL) != ATA_PROTOCOL_NODATA)) {
		ATA$UNMAP_PIO(rq->port->sglist);
	}
}


static void ATAPI_INIT(SCSI_ATTACH_PARAM *sa)
{
	ATA_ATTACH_PARAM *ap = sa->adapter;
	APORT *p = ap->port;
	sa->max_sectors = ATA$SGLIST_GUARANTEED_SIZE(p->sglist, !!(p->device[ap->device].dev_flags & DEV_F_ATAPI_DMA)) >> BIO_SECTOR_SIZE_BITS;
	sa->max_tags = 1;
	sa->max_fragments = 0;

	sa->post = ATAPI_POST;
	sa->cancel = ATAPI_CANCEL;
}

#define	PHASE_COMMAND		0x01
#define PHASE_MAPPED_DMA	0x02
#define PHASE_MAPPED_PIO	0x04
#define PHASE_RESET		0x08

#define atapi_phase		atapi_placeholder.status
#define atapi_device		atapi_placeholder.device
#define atapi_len		atapi_placeholder.len
#define atapi_sec_word		atapi_placeholder.retries
#define atapi_error_register	atapi_placeholder.fis.feature8
#define atapi_atarq_flags	atapi_placeholder.atarq_flags

#define IS_SENSE(p, rq)		((rq) == (SCSIRQ *)&(p)->sense_rq)

static char *ATAPI_NAME(APORT *p)
{
	return *p->device[p->atapi_device].attached->dev_name;
}

static int ATAPI_POLL_DRQ(APORT *p)
{
	__u8 status;
	u_jiffies_lo_t j, jj, timeout;
	timeout = __unlikely(p->device[p->atapi_device].dev_flags & (DEV_F_ATAPI_DRQ_INT | DEV_F_ATAPI_LONG_DRQ)) ? ATAPI_DRQ_LONG : ATAPI_DRQ_SHORT;
	timeout = TIMEOUT_JIFFIES(timeout);
	j = KERNEL$GET_JIFFIES_LO();
	do {
		jj = KERNEL$GET_JIFFIES_LO();
		status = io_inb(p->io + PORT_STATUS);
		if (__unlikely(status & ATA_STATUS_BSY)) continue;
		if (__unlikely(status & ATA_STATUS_ERROR)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "ERROR POLLING FOR DRQ, STATUS %02X, ERROR %02X", status, io_inb(p->io + PORT_ERROR));
			return -1;
		}
		if (__likely(status & ATA_STATUS_DRQ)) return 0;
	} while (jj - j <= timeout);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "TIMEOUT ON POLLING FOR DRQ, STATUS %02X", status);
	return -1;
}

static int ATAPI_COMMAND(APORT *p);
static void ATAPI_END_REQUEST(APORT *p);
static void ATAPI_DONE(ATARQ *rq);
static void ATAPI_UNMAP(APORT *p);
static void ATAPI_NONDMA_IRQ(APORT *p, __u8 status);
static void ATAPI_RESET_POLL(TIMER *t);

static int ATAPI_POST(SCSIRQ *rq)
{
	ATA_ATTACH_PARAM *ap = rq->adapter;
	APORT *p = ap->port;
	int r;
	BIODESC *desc;
	unsigned dev_flags;
	if (__unlikely(p->aport_flags & APORT_LOCK)) {
		r = -EAGAIN;
		goto ret0;
	}
	if (__unlikely(p->current_rq != NULL)) {
		if (__unlikely(IS_SENSE(p, rq))) goto ok;
		r = -EAGAIN;
		goto ret0;
	}
	ok:
	p->current_rq = &p->atapi_placeholder;
	p->atapi_phase = PHASE_COMMAND;
	p->atapi_device = ap->device;
	p->atapi_placeholder.done = ATAPI_DONE;
	io_outb(p->io + PORT_DRIVE, (p->atapi_device * ATA_DEVICE_DRIVE) | ATA_DEVICE_ATAPI_OR_BITS);

	p->atapi_placeholder.atarq_flags = ATA_PROTOCOL_PIO;
	p->scsirq = rq;
	dev_flags = p->device[p->atapi_device].dev_flags;
#if __DEBUG >= 1
	if (!(dev_flags & DEV_F_ATAPI))
		KERNEL$SUICIDE("ATAPI_POST: INVALID DEVICE FLAGS: %X", dev_flags);
#endif
	if (__unlikely(rq->cmdlen > 12 + ((dev_flags / (DEV_F_ATAPI_CMD_16 / 4)) & 4))) {
		r = -EOPNOTSUPP;
		goto ret1;
	}
	p->atapi_len = 0;
	if (__unlikely(!rq->direction)) {
		goto no_data;
	}
	desc = rq->desc;
	do {
		p->atapi_len += desc->v.len;
	} while ((desc = desc->next));
	if (__likely(dev_flags & DEV_F_ATAPI_DMA)) {
		if (__unlikely((((unsigned long)rq->desc->v.ptr | rq->desc->v.len) & (BIO_SECTOR_SIZE - 1)) != 0)) {
			goto unaligned__pio;
		}
		p->atapi_phase |= PHASE_MAPPED_DMA;
		r = ATA$MAP_DMA(p->sglist, &rq->desc, p->atapi_len, rq->direction, 1);
		if (__unlikely(r != p->atapi_len)) {
			map_failed:
			if (__unlikely(r >= 0)) {
				r = -EVSPACEFAULT;
				goto ret2;
			}
			goto ret1;
		}
		p->atapi_atarq_flags = (rq->direction & PF_WRITE) * ATARQ_TO_DEVICE;
		setup_dma(p, &p->atapi_placeholder);
		if (__unlikely(dev_flags & DEV_F_ATAPI_DMADIR)) {
			io_outb(p->io + PORT_FEATURES, ATA_FEATURE_PACKET_DMA | ((rq->direction & PF_WRITE) * ATA_FEATURE_PACKET_DMADIR));
		} else {
			io_outb(p->io + PORT_FEATURES, ATA_FEATURE_PACKET_DMA);
		}
	} else {
		unaligned__pio:
		p->atapi_phase |= PHASE_MAPPED_PIO;
		r = ATA$MAP_PIO(p->sglist, &rq->desc, p->atapi_len, rq->direction, 1);
		if (__unlikely(r != p->atapi_len)) goto map_failed;
		p->pio_sg_pos = 0;
		p->atapi_sec_word = 0;
		no_data:
		setup_pio(p, p->atapi_device);
		io_outb(p->io + PORT_FEATURES, 0);
	}
	io_outb(p->io + PORT_LBA_M, ATAPI_XFER_LEN & 0xff);
	io_outb(p->io + PORT_LBA_H, ATAPI_XFER_LEN >> 8);
	io_outb(p->io + PORT_CMD, ATA_CMD_PACKET);

	rq->status = p->atapi_len;
	rq->scsi_status = SCSI_GOOD;

	if (__unlikely(dev_flags & DEV_F_ATAPI_DRQ_INT)) {
		VOID_LIST_ENTRY(&p->timeout.list);
		return 0;
	}
	if (__unlikely(ATAPI_POLL_DRQ(p))) {
		ATAPI_ERROR_RESET_DRIVE(p);
		return 0;
	}
	VOID_LIST_ENTRY(&p->timeout.list);
	return ATAPI_COMMAND(p);

	ret2:
	ATAPI_UNMAP(p);
	ret1:
	p->current_rq = NULL;
	ret0:
	return r;
}

static int ATAPI_COMMAND(APORT *p)
{
	unsigned dev_flags;
	char atapi_packet[16];
	memset(atapi_packet, 0, sizeof atapi_packet);
	memcpy(atapi_packet, p->scsirq->cmd, p->scsirq->cmdlen);
	ATA_DI(p, p->atapi_device);
	dev_flags = p->device[p->atapi_device].dev_flags;
	ATA_OUTS(p, dev_flags & DEV_F_IO32, atapi_packet, 6 + ((dev_flags / (DEV_F_ATAPI_CMD_16 / 2)) & 2));
	ATA_EI(p, p->atapi_device);
	p->atapi_phase &= ~PHASE_COMMAND;
	if (__likely(p->atapi_phase & PHASE_MAPPED_DMA)) {
		p->atapi_placeholder.atarq_flags = ATA_PROTOCOL_DMA;
		start_dma(p, &p->atapi_placeholder);
	}
	return 0;
}

static void ATAPI_IRQ(APORT *p, __u8 dmastatus, __u8 status)
{
	if (__unlikely(p->atapi_phase & (PHASE_COMMAND | PHASE_RESET))) {
		if (__unlikely(p->atapi_phase & PHASE_RESET)) {
			KERNEL$DEL_TIMER(&p->timeout);
			ATAPI_RESET_POLL(&p->timeout);
			return;
		}
		if (__unlikely(ATAPI_POLL_DRQ(p))) {
			p->scsirq->status = -EIO;
			ATAPI_END_REQUEST(p);
			return;
		}
		ATAPI_COMMAND(p);
		return;
	}
	if (__likely(p->atapi_phase & PHASE_MAPPED_DMA)) {
		if (__unlikely(dmastatus & DMASTATUS_ERROR)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ATAPI_NAME(p), "DMA ERROR ON ATAPI COMMAND %02X, DMA STATUS %02X, DRIVE STATUS %02X, ERROR %02X", p->scsirq->cmd[0], dmastatus, status, io_inb(p->io + PORT_ERROR));
			p->scsirq->status = -EIO;
			ATAPI_END_REQUEST(p);
			return;
		}
		if (__unlikely(status & ATA_STATUS_ERROR)) {
			ATAPI_STATUS_ERROR(p, status);
			return;
		}
		ATAPI_END_REQUEST(p);
	} else {
		ATAPI_NONDMA_IRQ(p, status);
	}
}

static void ATAPI_END_REQUEST(APORT *p)
{
	if (__unlikely(IS_SENSE(p, p->scsirq))) {
		ATAPI_END_SENSE_REQUEST(p);
		return;
	}
	ACTRL_END_REQUEST(p, p->atapi_phase);
}

static void ATAPI_DONE(ATARQ *rq)
{
	APORT *p = GET_STRUCT(rq, APORT, atapi_placeholder);
	SCSIRQ *scsirq;
	ATAPI_UNMAP(p);
	scsirq = p->scsirq;
	/*__debug_printf("done(%02x) -> %d\n", scsirq->cmd[0], scsirq->status);*/
#if __DEBUG >= 1
	p->scsirq = NULL;
#endif
	scsirq->done(scsirq);
}

static void ATAPI_UNMAP(APORT *p)
{
	if (__likely(p->atapi_phase & PHASE_MAPPED_DMA)) {
		ATA$UNMAP_DMA(p->sglist);
	} else if (p->atapi_phase & PHASE_MAPPED_PIO) {
		ATA$UNMAP_PIO(p->sglist);
	}
}

static void ATAPI_NONDMA_IRQ(APORT *p, __u8 status)
{
	/*__debug_printf("nondma_irq: status: %02x\n", status);*/
	if (status & ATA_STATUS_DRQ) {
		unsigned devlen;
		__u8 ireason = io_inb(p->io + PORT_COUNT);
		__u8 lo = io_inb(p->io + PORT_LBA_M);
		__u8 hi = io_inb(p->io + PORT_LBA_H);
		devlen = (hi << 8) | lo;
		if (__unlikely(!(p->atapi_phase & PHASE_MAPPED_PIO))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TRANSFER FOR NON-DATA COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X", p->scsirq->cmd[0], status, ireason, devlen);
			end_reset:
			ATAPI_ERROR_RESET_DRIVE(p);
			return;
		}
		if (__unlikely((ireason & (ATA_QSTAT_CD | ATA_QSTAT_IO)) != (p->scsirq->direction & PF_WRITE) * ATA_QSTAT_IO)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TRANSFER DATA IN BAD DIRECTION, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X", p->scsirq->cmd[0], status, ireason, devlen);
			goto end_reset;
		}
		if (__unlikely(devlen - 1 >= ATAPI_XFER_LEN)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TO TRANSFER INVALID LENGTH, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X", p->scsirq->cmd[0], status, ireason, devlen);
			goto end_reset;
		}
		if (__unlikely(devlen > p->atapi_len)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TO TRANSFER MORE DATA THAN EXPECTED, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X (EXPECTED %X)", p->scsirq->cmd[0], status, ireason, devlen, p->atapi_len);
			goto end_reset;
		}
		if (__unlikely(devlen & 1) && __unlikely(devlen != p->atapi_len)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ATAPI_NAME(p), "DEVICE WANTS TO TRANSFER ODD NUMBER OF BYTES IN THE MIDDLE OF THE STREAM, COMMAND %02X, STATUS %02X, INTERRUPT REASON %02X, TRANSFER LENGTH %04X (REMAINING %X)", p->scsirq->cmd[0], status, ireason, devlen, p->atapi_len);
			goto end_reset;
		}
		p->atapi_len -= devlen;
		ATA_DI(p, p->atapi_device);
		do {
			__u16 *data = ATA$MAP_PIO_SECTOR(p->sglist, p->pio_sg_pos);
			unsigned words = (BIO_SECTOR_SIZE / 2) - p->atapi_sec_word;
			if (__unlikely(devlen == 1)) {
				if (__likely(p->scsirq->direction & PF_WRITE)) {
					*(__u8 *)(data + p->atapi_sec_word) = io_inb(p->io + PORT_DATA);
				} else {
					io_outb(p->io + PORT_DATA, *(__u8 *)(data + p->atapi_sec_word));
				}
				ATA$UNMAP_PIO_SECTOR(p->sglist, data);
				break;
			}
			if (__likely(words > (devlen >> 1))) words = devlen >> 1;
			if (__likely(p->scsirq->direction & PF_WRITE)) {
				ATA_INS(p, p->device[p->atapi_device].dev_flags & DEV_F_IO32, data + p->atapi_sec_word, words);
			} else {
				ATA_OUTS(p, p->device[p->atapi_device].dev_flags & DEV_F_IO32, data + p->atapi_sec_word, words);
			}
			p->atapi_sec_word = (p->atapi_sec_word + words) & (BIO_SECTOR_SIZE / 2 - 1);
			if (__unlikely(!p->atapi_sec_word)) ATA$SET_NEXT_PIO_SECTOR(p->sglist, &p->pio_sg_pos);
			devlen -= words << 1;
			ATA$UNMAP_PIO_SECTOR(p->sglist, data);
		} while (devlen);
		ATA_EI(p, p->atapi_device);
		if (__unlikely(status & ATA_STATUS_ERROR)) goto status_error;
		return;
	}
	p->scsirq->status -= p->atapi_len;
	if (__unlikely(status & ATA_STATUS_ERROR)) {
		status_error:
		ATAPI_STATUS_ERROR(p, status);
		return;
	}
	ATAPI_END_REQUEST(p);
}

static void ATAPI_STATUS_ERROR(APORT *p, __u8 status)
{
	__u8 error;
	if (__unlikely(IS_SENSE(p, p->scsirq))) {
		/* some CD-ROMs reportedly do not have sense */
		goto ret_eio;
	}
	error = io_inb(p->io + PORT_ERROR);
	if (__likely(error & ATA_ERROR_SENSE_KEY)) {
		int r;
		KERNEL$DEL_TIMER(&p->timeout);
		p->error_register = error;
		p->orig_scsirq = p->scsirq;
		p->scsirq = (SCSIRQ *)&p->sense_rq;
		p->sense_rq.status = 0;
		p->sense_rq.ch_id_lun = p->orig_scsirq->ch_id_lun;
		p->sense_rq.adapter = p->orig_scsirq->adapter;
		p->sense_rq.desc = &p->sense_desc;
		p->sense_rq.internal = 0;
		p->sense_rq.direction = PF_WRITE;
		p->sense_rq.cmdlen = 6;
		p->sense_rq.cmd[0] = SCMD_REQUEST_SENSE;
		p->sense_rq.cmd[1] = p->orig_scsirq->cmd[1] & 0xe0;
		p->sense_rq.cmd[4] = p->orig_scsirq->sense_size;
		if (p->orig_scsirq->sense_size > ATAPI_AUTOSENSE_LENGTH) {
			p->sense_rq.cmd[4] = ATAPI_AUTOSENSE_LENGTH;
		}
		p->sense_desc.v.ptr = (unsigned long)p->orig_scsirq->sense;
		p->sense_desc.v.len = p->sense_rq.cmd[4];
		p->sense_desc.v.vspace = &KERNEL$VIRTUAL;
		p->sense_desc.next = NULL;
		ATAPI_UNMAP(p);
		if (__unlikely(r = ATAPI_POST((SCSIRQ *)&p->sense_rq))) {
			KERNEL$SUICIDE("ATAPI_STATUS_ERROR: CANNOT POST REQUEST SENSE: %d", r);
		}
		return;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ATAPI_NAME(p), "DEVICE ERROR: COMMAND %02X, STATUS %02X, ERROR %02X", p->scsirq->cmd[0], status, error);
	ret_eio:
	p->scsirq->status = -EIO;
	ATAPI_END_REQUEST(p);
}

static void ATAPI_END_SENSE_REQUEST(APORT *p)
{
	SCSIRQ *scsirq = p->scsirq = p->orig_scsirq;
	__u8 *sense = scsirq->sense;
	unsigned sense_size = scsirq->sense_size;
	p->orig_scsirq = NULL;
	scsirq->scsi_status = SCSI_CHECK_CONDITION;
	if (__unlikely(p->sense_rq.status < 0)) {
		memset(sense, 0, sense_size);
		if (sense_size > 0) sense[0] = SENSE_RESPONSE_CODE_CURRENT;
		if (sense_size > 2) {
			sense[2] = p->error_register >> __BSF_CONST(ATA_ERROR_SENSE_KEY);
			if (p->error_register & ATA_ERROR_ILI) sense[2] |= SENSE_FLAGS_ILI;
			if (p->error_register & ATA_ERROR_EOM) sense[2] |= SENSE_FLAGS_EOM;
		}
	} else {
		memset(sense + p->sense_rq.status, 0, sense_size - p->sense_rq.status);
	}
	ATAPI_END_REQUEST(p);
}

static void ATAPI_ERROR_RESET_DRIVE(APORT *p)
{
	if (IS_SENSE(p, p->scsirq)) {
		p->scsirq = p->orig_scsirq;
		p->orig_scsirq = NULL;
	}
	p->scsirq->status = -EIO;
	ATAPI_CANCEL(p->scsirq);
}

static void ATAPI_CANCEL(SCSIRQ *rq)
{
	ATA_ATTACH_PARAM *ap = rq->adapter;
	APORT *p = ap->port;
	/*__debug_printf("atapi cancel\n");*/
	if (IS_SENSE(p, p->scsirq)) {
		p->scsirq = p->orig_scsirq;
		p->orig_scsirq = NULL;
	}
	if (__unlikely(rq != p->scsirq))
		KERNEL$SUICIDE("ATAPI_CANCEL: CANCELING WRONG REQUEST, %p != %p", rq, p->scsirq);
	if (p->atapi_phase & PHASE_RESET) return;
	io_outb(p->io + PORT_CMD, ATA_CMD_DEVICE_RESET);
	p->atapi_phase |= PHASE_RESET;
	KERNEL$DEL_TIMER(&p->timeout);
	p->pio_timeout = KERNEL$GET_JIFFIES_LO();
	p->timeout.fn = ATAPI_RESET_POLL;
	KERNEL$SET_TIMER(JIFFIES_PER_SECOND, &p->timeout);
}

static void ATAPI_RESET_POLL(TIMER *t)
{
	__u8 status;
	APORT *p = GET_STRUCT(t, APORT, timeout);
	LOWER_SPL(SPL_ATA_SCSI);
	status = io_inb(p->io + PORT_STATUS);
	if (!(status & (ATA_STATUS_BSY | ATA_STATUS_DRQ))) {
		VOID_LIST_ENTRY(&p->timeout.list);
		if (p->scsirq->status >= 0) p->scsirq->status = -EINTR;
		ATAPI_END_REQUEST(p);
		return;
	}
	if (__unlikely(KERNEL$GET_JIFFIES_LO() - p->pio_timeout > ATA$PI_TIMEOUT(p->device[p->atapi_device].attached))) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, ATAPI_NAME(p), "RESET TIMEOUT, STATUS %02X", status);
		VOID_LIST_ENTRY(&p->timeout.list);
		p->scsirq->status = -EIO;
		ATAPI_END_REQUEST(p);
		return;
	}
	KERNEL$SET_TIMER(JIFFIES_PER_SECOND, &p->timeout);
}
