#include <ARCH/BSF.H>
#include <ARCH/IO.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/ATA.H>

#include "ACTRLREG.H"
#include "ACTRL.H"

static int IT8212_FIXUP_IDENT(APORT *p, int drive, __u16 ident[256]);
static void IT_GET_AVAIL_XFER(APORT *p, int drive, unsigned *avail, unsigned *avail_unsupported);
static int IT_SET_DMA(APORT *p, int drive, unsigned dma);
static void IT_SETUP_DMA(APORT *p, ATARQ *rq);

/* chipset_flags */
#define IT_FLAGS_MODE_SHIFT		3
#define IT_FLAGS_MODE_0			0x00000007
#define IT_FLAGS_MODE_1			0x00000038
#define IT_FLAGS_MODE_2			0x000001c0
#define IT_FLAGS_MODE_3			0x00000e00
#define IT_FLAGS_REG_50			0x000ff000
#define IT_FLAGS_FIRMWARE_MODE		0x80000000

#define IT_PCI_MODE			0x50
#define  IT_PCI_MODE_FIRMWARE			0x01
#define  IT_PCI_MODE_PRIMARY_CLK_50		0x02
#define  IT_PCI_MODE_SECONDARY_CLK_50		0x04
#define  IT_PCI_MODE_PRIMARY_MASTER_MWDMA	0x08
#define  IT_PCI_MODE_PRIMARY_SLAVE_MWDMA	0x10
#define  IT_PCI_MODE_SECONDARY_MASTER_MWDMA	0x20
#define  IT_PCI_MODE_SECONDARY_SLAVE_MWDMA	0x40
#define  IT_PCI_MODE_RESET			0x80
#define IT_PRIMARY_MASTER_UDMA_TIMING	0x56
#define IT_PRIMARY_SLAVE_UDMA_TIMING	0x57
#define IT_SECONDARY_MASTER_UDMA_TIMING	0x5a
#define IT_SECONDARY_SLAVE_UDMA_TIMING	0x5b

static __const__ __u8 udma_timing[7][2] = {
	{ 0x44, 0x33 },
	{ 0x42, 0x31 },
	{ 0x31, 0x21 },
	{ 0x21, 0x21 },
	{ 0x11, 0x11 },
	{ 0xa2, 0x91 },
	{ 0x91, 0x91 }
};

static __const__ __s8 preferred_clock[7] = {
	-1, 1, -1, 0, 0, 1, 0
};

int DETECT_IT821(ACTRL *a)
{
	int i;
	__u8 mode;
	if (a->dev_id != 0x82111283 && 
	    a->dev_id != 0x82121283) return 0;
	mode = PCI$READ_CONFIG_BYTE(a->pci_id, IT_PCI_MODE);
	if (mode & IT_PCI_MODE_FIRMWARE) {
		a->chipset_name = "IT8212 (FIRMWARE MODE)";
		a->chipset_flags = IT_FLAGS_FIRMWARE_MODE;
		for (i = 0; i < a->n_ports; i++) {
			a->port[i].aport_flags |= APORT_IMPLICIT_DMA | APORT_SIZE_LIMIT;
			a->port[i].max_len = 255 << BIO_SECTOR_SIZE_BITS;
		}
	} else {
		/* chipset_name already set from PCI table */
		a->chipset_flags = IT_FLAGS_MODE_0 | IT_FLAGS_MODE_1 | IT_FLAGS_MODE_2 | IT_FLAGS_MODE_3;
		a->chipset_flags_2 = 0;
		a->get_avail_xfer = IT_GET_AVAIL_XFER;
		a->set_xfer = IT_SET_DMA;
		for (i = 0; i < a->n_ports; i++) {
			a->port[i].aport_flags |= APORT_SPECIAL_DMA;
			a->port[i].setup_dma = IT_SETUP_DMA;
		}
		mode &= ~(IT_PCI_MODE_PRIMARY_MASTER_MWDMA | IT_PCI_MODE_PRIMARY_SLAVE_MWDMA | IT_PCI_MODE_SECONDARY_MASTER_MWDMA | IT_PCI_MODE_SECONDARY_SLAVE_MWDMA);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PCI_MODE, mode);
		a->chipset_flags |= mode << __BSF_CONST(IT_FLAGS_REG_50);
	}
	a->fixup_ident = IT8212_FIXUP_IDENT;
	return 1;
}

static int IT8212_FIXUP_IDENT(APORT *p, int drive, __u16 ident[256])
{
	ACTRL *a = p->ctrl;
	int atapi = (ident[0] & 0xc000) == 0x8000;
	char model[41];
	if (atapi && (a->dev_rev == 0x10 || a->chipset_flags & IT_FLAGS_FIRMWARE_MODE)) {
		ident[49] &= ~0x0100;
		ident[62] = 0;
	}
	if (!(a->chipset_flags & IT_FLAGS_FIRMWARE_MODE)) return 0;
	ATA$GET_MODEL(ident, atapi, &model, NULL, NULL);
	if (strstr(model, "Integrated Technology Express")) {
		ident[49] |= 0x0300;
		ident[83] |= 0x0400;
		ident[86] |= 0x0400;
	} else {
		ident[53] &= 0x0001;
		ident[75] = 0x0000;
		ident[82] = 0x0000;
		ident[83] &= 0xc400;
		ident[84] &= 0xc000;
		ident[85] = 0x0000;
		ident[86] &= 0xc400;
		ident[87] &= 0xc000;
		ident[127] = 0x0000;
		ident[128] = 0x0000;
		ident[129] = 0x0000;
		ident[160] = 0x0000;
	}
	return 0;
}

static void IT_GET_AVAIL_XFER(APORT *p, int drive, unsigned *avail, unsigned *avail_unsupported)
{
	*avail = IDE_XFER_UDMA_0 | IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2 | IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4 | IDE_XFER_UDMA_5 | IDE_XFER_UDMA_6;
	*avail_unsupported = 0;
}

static int IT_SET_DMA(APORT *p, int drive, unsigned dma)
{
	ACTRL *a = p->ctrl;
	unsigned shift;
	drive |= p->n << 1;
	shift = drive * IT_FLAGS_MODE_SHIFT;
	a->chipset_flags = (a->chipset_flags & ~(IT_FLAGS_MODE_0 << shift)) | (ATA$XFER_GET_NUMBER(dma, IDE_XFER_MASK_UDMA) << shift);
	return 0;
}

static void IT_SETUP_DMA(APORT *p, ATARQ *rq)
{
	ACTRL *a = p->ctrl;
	int clock;
	int current_clock;
	__u8 timing;
	int drive = (p->n << 1) | rq->device;
	int udma = (a->chipset_flags >> (drive * IT_FLAGS_MODE_SHIFT)) & IT_FLAGS_MODE_0;
	if (__unlikely(udma > 6)) goto straight;
#define clock_shift	(__BSF_CONST(IT_PCI_MODE_PRIMARY_CLK_50) + __BSF_CONST(IT_FLAGS_REG_50) + p->n)
		/*
		a->chipset_flags ^= 1 << clock_shift;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PCI_MODE, a->chipset_flags >> __BSF_CONST(IT_FLAGS_REG_50));
		*/
	clock = preferred_clock[udma];
	current_clock = (a->chipset_flags >> clock_shift) & 1;
	if (__unlikely(clock < 0)) {
		clock = current_clock;
	} else if (__unlikely(clock != current_clock)) {
		a->chipset_flags ^= 1 << clock_shift;
		PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PCI_MODE, a->chipset_flags >> __BSF_CONST(IT_FLAGS_REG_50));
	}
#undef clock_shift
#define timing_shift	(8 * drive)
	timing = udma_timing[udma][clock];
	if (__unlikely(timing != ((a->chipset_flags_2 >> timing_shift) & 0xff))) {
		a->chipset_flags_2 = (a->chipset_flags & ~(0xff << timing_shift)) | (timing << timing_shift);
		PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PRIMARY_MASTER_UDMA_TIMING + (4 * p->n) + (drive & 1), timing);
		if (__unlikely(a->dev_rev <= 0x10)) {
			a->chipset_flags_2 = (a->chipset_flags & ~(0xff << (timing_shift ^ 8))) | (timing << (timing_shift ^ 8));
			PCI$WRITE_CONFIG_BYTE(a->pci_id, IT_PRIMARY_MASTER_UDMA_TIMING + (4 * p->n) + (~drive & 1), timing);
		}
	}
#undef timing_shift

	straight:
	setup_dma_generic(p, rq);
}

