#include <ARCH/IO.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/ATA.H>

#include "ACTRLREG.H"
#include "ACTRL.H"

static void TRIFLEX_GET_AVAIL_XFER(APORT *p, int drive, unsigned *avail, unsigned *avail_unsupported);
static int TRIFLEX_SET_XFER(APORT *p, int drive, unsigned xfer);
static void TRIFLEX_SETUP_DMA(APORT *p, ATARQ *rq);
static void TRIFLEX_TIMING(APORT *p, unsigned drive, unsigned tim);
static void TRIFLEX_SETUP_PIO(APORT *p, int drive);

#define PCI_TRIFLEX_TIMING	0x70
#define PCI_TRIFLEX_TIMING_NEXT	0x04

/* port_chipset_flags */
#define TRIFLEX_MODE_SHIFT	6
#define TRIFLEX_PIO_MODE_0	0x00000007
#define TRIFLEX_DMA_MODE_0	0x00000038
#define TRIFLEX_PIO_MODE_1	0x000001c0
#define TRIFLEX_DMA_MODE_1	0x00000e00

int DETECT_TRIFLEX(ACTRL *a)
{
	int i;
	if (a->dev_id != 0xae330e11) return 0;
	a->chipset_name = "COMPAQ TRIFLEX";
	a->get_avail_xfer = TRIFLEX_GET_AVAIL_XFER;
	a->set_xfer = TRIFLEX_SET_XFER;
	for (i = 0; i < 2; i++) {
		a->port[i].aport_flags |= APORT_SPECIAL_DMA;
		a->port[i].setup_dma = TRIFLEX_SETUP_DMA;
		a->port[i].setup_pio = TRIFLEX_SETUP_PIO;
		a->port[i].port_chipset_flags = TRIFLEX_PIO_MODE_0 | TRIFLEX_DMA_MODE_0 | TRIFLEX_PIO_MODE_1 | TRIFLEX_DMA_MODE_1;
		a->port[i].port_chipset_flags_2 = PCI$READ_CONFIG_DWORD(a->pci_id, PCI_TRIFLEX_TIMING + PCI_TRIFLEX_TIMING_NEXT * i);
	}
	return 1;
}

static void TRIFLEX_GET_AVAIL_XFER(APORT *p, int drive, unsigned *avail, unsigned *avail_unsupported)
{
	*avail = IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2 | IDE_XFER_PIO_3 | IDE_XFER_PIO_4 | IDE_XFER_SDMA_0 | IDE_XFER_SDMA_1 | IDE_XFER_SDMA_2 | IDE_XFER_WDMA_0 | IDE_XFER_WDMA_1 | IDE_XFER_WDMA_2;
	*avail_unsupported = 0;
}

static int TRIFLEX_SET_XFER(APORT *p, int drive, unsigned xfer)
{
	int shift;
	shift = drive * TRIFLEX_MODE_SHIFT;
	if (xfer & IDE_XFER_MASK_PIO) {
		unsigned pio = ATA$XFER_GET_NUMBER(xfer, IDE_XFER_MASK_PIO);
		if (__unlikely(pio > 4))
			KERNEL$SUICIDE("TRIFLEX_SET_XFER: BAD PIO MODE %d, XFER %X", pio, xfer);
		p->port_chipset_flags = (p->port_chipset_flags & ~(TRIFLEX_PIO_MODE_0 << shift)) | (pio << (shift + __BSF_CONST(TRIFLEX_PIO_MODE_0)));
	} else if (xfer & IDE_XFER_MASK_DMA) {
		unsigned dma = ATA$XFER_GET_NUMBER(xfer, IDE_XFER_MASK_DMA);
		if (__unlikely(dma > 5))
			KERNEL$SUICIDE("TRIFLEX_SET_XFER: BAD DMA MODE %d, XFER %X", dma, xfer);
		p->port_chipset_flags = (p->port_chipset_flags & ~(TRIFLEX_DMA_MODE_0 << shift)) | (dma << (shift + __BSF_CONST(TRIFLEX_DMA_MODE_0)));
	} else {
		KERNEL$SUICIDE("TRIFLEX_SET_XFER: BAD XFER %X", xfer);
	}
	return 0;
}

static int TRIFLEX_PIO_TIMING(APORT *p, int drive)
{
	static __u16 __const__ pio_timing[5] = { 0x0808, 0x0508, 0x0404, 0x0204, 0x0202 };
	unsigned pio = p->port_chipset_flags >> (drive * TRIFLEX_MODE_SHIFT + __BSF_CONST(TRIFLEX_PIO_MODE_0)) & (TRIFLEX_PIO_MODE_0 >> __BSF_CONST(TRIFLEX_PIO_MODE_0));
	if (__unlikely(pio > 4)) return -1;
	return pio_timing[pio];
}

static __finline__ int TRIFLEX_DMA_TIMING(APORT *p, int drive)
{
	static __u16 __const__ dma_timing[6] = { 0x0f0f, 0x0f0f, 0x0f0f, 0x0808, 0x0203, 0x0103 };
	unsigned dma = p->port_chipset_flags >> (drive * TRIFLEX_MODE_SHIFT + __BSF_CONST(TRIFLEX_DMA_MODE_0)) & (TRIFLEX_DMA_MODE_0 >> __BSF_CONST(TRIFLEX_DMA_MODE_0));
	if (__unlikely(dma > 5)) return -1;
	return dma_timing[dma];
}

static void TRIFLEX_SETUP_DMA(APORT *p, ATARQ *rq)
{
	int dma_tim = TRIFLEX_DMA_TIMING(p, rq->device);
	if (__unlikely(dma_tim < 0)) return;
	if (__unlikely(IS_ATAPI(p, rq))) {
		int pio_tim = TRIFLEX_PIO_TIMING(p, rq->device);
		if (__likely(pio_tim >= 0)) {
			if ((pio_tim & 0xff) > (dma_tim & 0xff)) dma_tim = (dma_tim & ~0xff) | pio_tim;
			if ((pio_tim & 0xff00) > (dma_tim & 0xff00)) dma_tim = (dma_tim & ~0xff00) | pio_tim;
		}
	}
	TRIFLEX_TIMING(p, rq->device, dma_tim);
	setup_dma_generic(p, rq);
}

static void TRIFLEX_TIMING(APORT *p, unsigned drive, unsigned tim)
{
	drive *= 16;
	if (__likely(((p->port_chipset_flags_2 >> drive) & 0xffff) == tim)) return;
	p->port_chipset_flags_2 = (p->port_chipset_flags_2 & ~(0xffff << drive)) | (tim << drive);
	/* __debug_printf("write timing: %02X -> %08X\n", PCI_TRIFLEX_TIMING + drive / (16 / PCI_TRIFLEX_TIMING_NEXT), p->port_chipset_flags_2); */
	PCI$WRITE_CONFIG_DWORD(p->ctrl->pci_id, PCI_TRIFLEX_TIMING + drive / (16 / PCI_TRIFLEX_TIMING_NEXT), p->port_chipset_flags_2);
}

static void TRIFLEX_SETUP_PIO(APORT *p, int drive)
{
	int tim = TRIFLEX_PIO_TIMING(p, drive);
	if (__unlikely(tim < 0)) return;
	TRIFLEX_TIMING(p, drive, tim);
}

