#include <ARCH/IO.H>
#include <STRING.H>
#include <SPAD/SYSLOG.H>

#include "ACTRL.H"

/* !!! FIXME: check cable */

#define VIA_IDE_ENABLE		0x40
#define VIA_IDE_CONFIG		0x41
#define VIA_FIFO_CONFIG		0x43
#define VIA_MISC_1		0x44
#define VIA_MISC_2		0x45
#define VIA_MISC_3		0x46
#define VIA_DRIVE_TIMING	0x48
#define VIA_ADDRESS_SETUP	0x4c
#define VIA_8BIT_TIMING		0x4e
#define VIA_UDMA_TIMING		0x50

#define VIA_UDMA_NONE	0x0001
#define VIA_UDMA_33	0x0002
#define VIA_UDMA_66	0x0004
#define VIA_UDMA_100	0x0008
#define VIA_UDMA_133	0x0010
#define VIA_SET_FIFO	0x0020
#define VIA_NO_UNMASK	0x0040
#define VIA_BAD_PREQ	0x0080
#define VIA_BAD_CLK66	0x0100
#define VIA_BAD_AST	0x0200
#define VIA_BRIDGE_BUG	0x0400
#define VIA_UDMA_MASK	(VIA_UDMA_NONE | VIA_UDMA_33 | VIA_UDMA_66 | VIA_UDMA_100 | VIA_UDMA_133)

static __const__ struct {
	char *name;
	__u32 id;
	__u8 rev_min;
	__u8 rev_max;
	unsigned short flags;
} via_bridges[] = {
	{ "CX700",	0x83241106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT8237S",	0x33721106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT6410",	0x31641106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT8251",	0x32871106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT8237",	0x32271106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT8237A",	0x33371106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT8235",	0x31771106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT8233A",	0x31471106, 0x00, 0x2f,	VIA_UDMA_133 | VIA_BAD_AST },
	{ "VT8233C",	0x31091106, 0x00, 0x2f,	VIA_UDMA_100 },
	{ "VT8233",	0x30741106, 0x00, 0x2f,	VIA_UDMA_100 },
	{ "VT8231",	0x82311106, 0x00, 0x2f,	VIA_UDMA_100 | VIA_BRIDGE_BUG },
	{ "VT82C686B",	0x06861106, 0x40, 0x4f,	VIA_UDMA_100 | VIA_BRIDGE_BUG },
	{ "VT82C686A",	0x06861106, 0x10, 0x2f,	VIA_UDMA_66 },
	{ "VT82C686",	0x06861106, 0x00, 0x0f,	VIA_UDMA_33 | VIA_BAD_CLK66 },
	{ "VT82C596B",	0x05961106, 0x10, 0x2f,	VIA_UDMA_66 },
	{ "VT82C596A",	0x05961106, 0x00, 0x0f,	VIA_UDMA_33 | VIA_BAD_CLK66 },
	{ "VT82C586B",	0x05861106, 0x47, 0x4f,	VIA_UDMA_33 | VIA_SET_FIFO },
	{ "VT82C586B",	0x05861106, 0x40, 0x46,	VIA_UDMA_33 | VIA_SET_FIFO | VIA_BAD_PREQ },
	{ "VT82C586B",	0x05861106, 0x30, 0x3f,	VIA_UDMA_33 | VIA_SET_FIFO },
	{ "VT82C586A",	0x05861106, 0x20, 0x2f,	VIA_UDMA_33 | VIA_SET_FIFO },
	{ "VT82C586",	0x05861106, 0x00, 0x0f,	VIA_UDMA_NONE | VIA_SET_FIFO },
	{ "VT82C576",	0x05761106, 0x00, 0x2f,	VIA_UDMA_NONE | VIA_SET_FIFO | VIA_NO_UNMASK },
	{ "VT82C576",	0x05761107, 0x00, 0x2f,	VIA_UDMA_NONE | VIA_SET_FIFO | VIA_NO_UNMASK },
};

static void VIA_GET_AVAIL_XFER(APORT *p, int drive, unsigned *avail, unsigned *avail_unsupported)
{
	ACTRL *a = p->ctrl;
	*avail = 0;
	*avail_unsupported = 0;
	switch (a->chipset_flags & VIA_UDMA_MASK) {
	case VIA_UDMA_133:
		*avail |= IDE_XFER_UDMA_6;
	case VIA_UDMA_100:
		*avail |= IDE_XFER_UDMA_5;
	case VIA_UDMA_66:
		*avail |= IDE_XFER_UDMA_3 | IDE_XFER_UDMA_4;
	case VIA_UDMA_33:
		*avail |= IDE_XFER_UDMA_0 | IDE_XFER_UDMA_1 | IDE_XFER_UDMA_2;
	}
	switch (a->chipset_flags & VIA_UDMA_MASK) {
	case VIA_UDMA_33:
		*avail_unsupported |= IDE_XFER_UDMA_3;
	}
}

static int VIA_SET_DMA(APORT *p, int drive, unsigned dma)
{
	unsigned udma_clock;
	unsigned supp, unofficial_supp;
	int val;
	ACTRL *a = p->ctrl;
	VIA_GET_AVAIL_XFER(p, drive, &supp, &unofficial_supp);
	if (__unlikely(!(dma & (supp | unofficial_supp)))) return -EOPNOTSUPP;
	drive |= p->n << 1;
	if (__unlikely((unsigned)drive >= 4))
		KERNEL$SUICIDE("VIA_SET_DMA: INVALID DRIVE %d", drive);
	udma_clock = a->bus_clock;
	switch (a->chipset_flags & VIA_UDMA_MASK) {
	case VIA_UDMA_133:
		udma_clock *= 4;
		break;
	case VIA_UDMA_100:
		udma_clock *= 3;
		break;
	case VIA_UDMA_66:
		udma_clock *= 2;
		break;
	}
	val = ((__u64)ATA$UDMA_TIMING[ATA$XFER_GET_NUMBER(dma, IDE_XFER_MASK_UDMA)] * udma_clock + 999999999) / 1000000000;
	if ((val -= 2) < 0) val = 0;
	if (val > 7) val = 7;
	if (val > 3 && a->chipset_flags & VIA_UDMA_33) val = 3;
	if (a->chipset_flags & VIA_UDMA_66) val |= 0xe8;
	else val |= 0xe0;
	/*__debug_printf("mode: %x, timing: %02x, port: %02x\n", dma, val, VIA_UDMA_TIMING + 3 - drive);*/
	PCI$WRITE_CONFIG_BYTE(a->pci_id, VIA_UDMA_TIMING + 3 - drive, val);
	return 0;
}

static __const__ struct pci_id_s southbridge_table[] = {
	{ 0x1106, 0x0305, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0x1106, 0x0391, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0x1106, 0x3102, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0x1106, 0x3112, PCI_ANY_ID, PCI_ANY_ID, 0, 0, NULL, 0 },
	{ 0 },
};

static void VIA_SOUTHBRIDGE_FIXUP(ACTRL *a)
{
	pci_id_t id;
	if (!PCI$FIND_DEVICE(southbridge_table, 0, 0, 0, PCI$TEST_LIST, &id, NULL, NULL, 1)) {
		__u8 reg76 = PCI$READ_CONFIG_BYTE(id, 0x76);
		if ((reg76 & 0xf0) != 0xd0) {
			PCI$WRITE_CONFIG_BYTE(id, 0x75, 0x80);
			PCI$WRITE_CONFIG_BYTE(id, 0x76, (reg76 & 0x0f) | 0xd0);
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, a->dev_name, "APPLYING VIA SOUTHBRIDGE DATA CORRUPTION FIX");
		}
	}
}

int DETECT_VIA(ACTRL *a)
{
	unsigned i;
	if (a->dev_id != 0x15711106 &&
	    a->dev_id != 0x05711106 &&
	    a->dev_id != 0x31641106 &&
	    a->dev_id != 0x53241106) return 0;
	for (i = 0; i < sizeof(via_bridges) / sizeof(*via_bridges); i++) {
		if (ATA_FIND_DEV(a, via_bridges[i].id, via_bridges[i].rev_min, via_bridges[i].rev_max)) goto got_it;
	}
	return 0;
	got_it:
	a->chipset_name = via_bridges[i].name;
	a->chipset_flags = via_bridges[i].flags;

	if (a->chipset_flags & VIA_BRIDGE_BUG) {
		VIA_SOUTHBRIDGE_FIXUP(a);
	}

	if (a->chipset_flags & (VIA_UDMA_66 | VIA_BAD_CLK66)) {
		__u32 u = PCI$READ_CONFIG_DWORD(a->pci_id, VIA_UDMA_TIMING);
		if (a->chipset_flags & VIA_UDMA_66) {
			PCI$WRITE_CONFIG_DWORD(a->pci_id, VIA_UDMA_TIMING, u | 0x80008);
		} else if (a->chipset_flags & VIA_BAD_CLK66) {
			PCI$WRITE_CONFIG_DWORD(a->pci_id, VIA_UDMA_TIMING, u & ~0x80008);
		}
	}
	if (a->chipset_flags & (VIA_SET_FIFO | VIA_BAD_PREQ)) {
		__u8 f = PCI$READ_CONFIG_BYTE(a->pci_id, VIA_FIFO_CONFIG);
		if (a->chipset_flags & VIA_BAD_PREQ) f &= ~0x80;
		if (a->chipset_flags & VIA_SET_FIFO) {
			f = f & ~0x60;
			if (!(a->port[0].aport_flags & APORT_DISABLED) && !(a->port[1].aport_flags & APORT_DISABLED)) f |= 0x20;
			else if (!(a->port[1].aport_flags & APORT_DISABLED)) f |= 0x60;
		}
		PCI$WRITE_CONFIG_BYTE(a->pci_id, VIA_FIFO_CONFIG, f);
	}
	if (a->chipset_flags & VIA_NO_UNMASK) {
		a->actrl_flags |= ACTRL_FORCE_MASKIRQ;
	}
	a->get_avail_xfer = VIA_GET_AVAIL_XFER;
	a->set_xfer = VIA_SET_DMA;
	return 1;
}
