#include <SYS/TYPES.H>
#include <ENDIAN.H>
#include <ARCH/BSF.H>
#include <SPAD/SYSLOG.H>

#include <SPAD/ATA.H>

static __const__ struct {
	__u8 cmd;
	short feature;
	char *name;
} command_names[] = {
{ ATA_CMD_NOP, -1, "NOP" },
{ ATA_CMD_NOP, ATA_FEATURE_NOP_NOP, "NOP" },
{ ATA_CMD_NOP, ATA_FEATURE_NOP_AUTO_POLL, "AUTO POLL" },
{ ATA_CMD_READ, -1, "READ" },
{ ATA_CMD_READ_EXT, -1, "READ EXT" },
{ ATA_CMD_READ_DMA_EXT, -1, "READ DMA EXT" },
{ ATA_CMD_READ_DMA_QUEUED_EXT, -1, "READ DMA QUEUED EXT" },
{ ATA_CMD_READ_NATIVE_MAX_ADDRESS_EXT, -1, "READ NATIVE MAX ADDRESS EXT" },
{ ATA_CMD_READ_MULTIPLE_EXT, -1, "READ MULTIPLE EXT" },
{ ATA_CMD_WRITE, -1, "WRITE" },
{ ATA_CMD_WRITE_EXT, -1, "WRITE EXT" },
{ ATA_CMD_WRITE_DMA_EXT, -1, "WRITE DMA EXT" },
{ ATA_CMD_WRITE_DMA_QUEUED_EXT, -1, "WRITE DMA QUEUED EXT" },
{ ATA_CMD_SET_MAX_ADDRESS_EXT, -1, "SET MAX ADDRESS EXT" },
{ ATA_CMD_WRITE_MULTIPLE_EXT, -1, "WRITE MULTIPLE EXT" },
{ ATA_CMD_PACKET, -1, "PACKET" },
{ ATA_CMD_IDENTIFY_PACKET_DEVICE, -1, "IDENTIFY PACKET DEVICE" },
{ ATA_CMD_SERVICE, -1, "SERVICE" },
{ ATA_CMD_CONFIGURATION, -1, "CONFIGURATION" },
{ ATA_CMD_CONFIGURATION, ATA_FEATURE_CONFIGURATION_RESTORE, "RESTORE" },
{ ATA_CMD_CONFIGURATION, ATA_FEATURE_CONFIGURATION_FREEZE_LOCK, "FREEZE LOCK" },
{ ATA_CMD_CONFIGURATION, ATA_FEATURE_CONFIGURATION_IDENTIFY, "IDENTIFY" },
{ ATA_CMD_CONFIGURATION, ATA_FEATURE_CONFIGURATION_SET, "SET" },
{ ATA_CMD_READ_MULTIPLE, -1, "READ MULTIPLE" },
{ ATA_CMD_WRITE_MULTIPLE, -1, "WRITE MULTIPLE" },
{ ATA_CMD_SET_MULTIPLE_MODE, -1, "SET MULTIPLE MODE" },
{ ATA_CMD_READ_DMA_QUEUED, -1, "READ DMA QUEUED" },
{ ATA_CMD_READ_DMA, -1, "READ DMA" },
{ ATA_CMD_WRITE_DMA, -1, "WRITE DMA" },
{ ATA_CMD_WRITE_DMA_QUEUED, -1, "WRITE DMA QUEUED" },
{ ATA_CMD_FLUSH_CACHE, -1, "FLUSH CACHE" },
{ ATA_CMD_FLUSH_CACHE_EXT, -1, "FLUSH CACHE EXT" },
{ ATA_CMD_IDENTIFY_DEVICE, -1, "IDENTIFY DEVICE" },
{ ATA_CMD_SET_FEATURES, -1, "SET FEATURES" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_ENABLE_WCACHE, "ENABLE WRITE CACHE" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_XFER, "XFER" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_SPIN_UP, "SPIN UP" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_DISABLE_RCACHE, "DISABLE READ CACHE" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_ENABLE_REL_INT, "ENABLE RELEASE INTERRUPT" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_ENABLE_SERV_INT, "ENABLE SERVICE INTERRUPT" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_DISABLE_WCACHE, "DISABLE WRITE CACHE" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_ENABLE_RCACHE, "ENABLE READ CACHE" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_DISABLE_REL_INT, "DISABLE RELEASE INTERRUPT" },
{ ATA_CMD_SET_FEATURES, ATA_FEATURE_SET_FEATURES_DISABLE_SERV_INT, "DISABLE SERVICE INTERRUPT" },
{ ATA_CMD_READ_NATIVE_MAX_ADDRESS, -1, "READ NATIVE MAX ADDRESS" },
{ ATA_CMD_SET_MAX_ADDRESS, -1, "SET MAX ADDRESS" },
};

char *ATA$COMMAND_NAME(__u8 cmd, __u8 feature)
{
	static char name[__MAX_STR_LEN];
	int i;
	char *cmdname = NULL, *ftname = NULL;
	for (i = 0; i < sizeof(command_names) / sizeof(*command_names); i++) {
		if (__unlikely(command_names[i].cmd == cmd) && __likely(command_names[i].feature == -1)) cmdname = command_names[i].name;
		if (__unlikely(command_names[i].cmd == cmd) && __unlikely(command_names[i].feature == feature)) ftname = command_names[i].name;
	}
	if (cmdname) {
		if (!ftname) return cmdname;
		_snprintf(name, sizeof name, "%s / %s", cmdname, ftname);
		return name;
	}
	_snprintf(name, sizeof name, "%02X", cmd);
	return name;
}

int ATA$DUMMY_DEQUEUE(ATA_ATTACH_PARAM *ap)
{
	return 0;
}

BIORQ *ATA$DUMMY_PROBE_QUEUE(ATA_ATTACH_PARAM *ap)
{
	return NULL;
}

void ATA$BSWAP_IDENTIFY(__u16 *id)
{
	int i;
	for (i = 0; i < 256; i++) *id = __16LE2CPU(*id);
}

void GET_MODEL_STRING(__u16 *id, char *str, unsigned len, int swap);

void ATA$GET_MODEL(__u16 *id, int atapi, char (*model)[41], char (*serial)[21], char (*revision)[9])
{
	int swap = 0;
	if (atapi && (
		id[27] == ('N' | ('E' << 8)) ||
		id[27] == ('F' | ('X' << 8)) ||
		id[27] == ('P' | ('i' << 8)))) swap = 1;
	if (model) GET_MODEL_STRING(id + 27, *model, 40, swap);
	if (serial) GET_MODEL_STRING(id + 10, *serial, 20, swap);
	if (revision) GET_MODEL_STRING(id + 23, *revision, 8, swap);
}

void GET_MODEL_STRING(__u16 *id, char *str, unsigned len, int swap)
{
	int i, ii;
	for (i = 0; i < len; i += 2) {
		str[i + swap] = id[i >> 1] >> 8;
		str[i + 1 - swap] = id[i >> 1];
	}
	str[i] = 0;
	for (i--; i >= 0; i--) if (str[i] == ' ') str[i] = 0;
			       else break;
	for (i = 0, ii = 0; str[i]; i++) if (str[i] != ' ' || (ii && str[ii - 1] != ' ')) str[ii++] = str[i];
	str[ii] = 0;
}

unsigned ATA$GET_XFER_MODE(__u16 *id, int atapi)
{
	unsigned avail_xfer = 0;
	if (id[53] & 2) {
		if (id[64] & 1) avail_xfer |= IDE_XFER_PIO_3;
		if (id[64] & 2) avail_xfer |= IDE_XFER_PIO_4;
	}
	if (avail_xfer & (IDE_XFER_PIO_3 | IDE_XFER_PIO_4)) {
		avail_xfer |= IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2;
	} else {
		if ((id[51] >> 8) >= 2) avail_xfer |= IDE_XFER_PIO_0 | IDE_XFER_PIO_1 | IDE_XFER_PIO_2;
		else if ((id[51] >> 8) == 1) avail_xfer |= IDE_XFER_PIO_0 | IDE_XFER_PIO_1;
		else avail_xfer |= IDE_XFER_PIO_0;
	}
	if (id[63] & 1) avail_xfer |= IDE_XFER_WDMA_0;
	if (id[63] & 2) avail_xfer |= IDE_XFER_WDMA_1;
	if (id[63] & 4) avail_xfer |= IDE_XFER_WDMA_2;
	if (id[53] & 4) {
		if (id[88] & 1) avail_xfer |= IDE_XFER_UDMA_0;
		if (id[88] & 2) avail_xfer |= IDE_XFER_UDMA_1;
		if (id[88] & 4) avail_xfer |= IDE_XFER_UDMA_2;
		if (id[88] & 8) avail_xfer |= IDE_XFER_UDMA_3;
		if (id[88] & 16) avail_xfer |= IDE_XFER_UDMA_4;
		if (id[88] & 32) avail_xfer |= IDE_XFER_UDMA_5;
		if (id[88] & 64) avail_xfer |= IDE_XFER_UDMA_6;
	}
	return avail_xfer;
}

__const__ unsigned char ATA$UDMA_TIMING[7] = {
	120, 80, 60, 45, 30, 20, 15
};

unsigned ATA$XFER_GET_NUMBER(unsigned val, unsigned mask)
{
	if (__unlikely(!(mask & val)) || __unlikely(val & (val - 1)))
		KERNEL$SUICIDE("ATA$XFER_GET_NUMBER: VALUE %X, MASK %X", val, mask);
	return __BSF(mask & val) - __BSF(mask);
}

unsigned ATA$XFER_BEST(unsigned mask)
{
	if (__unlikely(!mask)) return mask;
	return 1 << __BSR(mask);
}

int ATA$SETUP_PIO_XFER(ATA_ATTACH_PARAM *ap, __u16 *id, unsigned users_xfer, unsigned driver_xfer_mask, unsigned controller_xfer_mask, unsigned controller_xfer_mask_unsupported, int (*set_pio)(APORT *p, int drive, unsigned pio), jiffies_lo_t timeout)
{
	ATARQ rq;
	unsigned xfer;
	int pio, r;
	xfer = users_xfer & IDE_XFER_MASK_PIO;
	if (!xfer) {
		xfer = ATA$XFER_BEST(controller_xfer_mask & driver_xfer_mask & IDE_XFER_MASK_PIO);
	}
	if (!xfer) return 0;
	pio = ATA$XFER_GET_NUMBER(xfer, IDE_XFER_MASK_PIO);
	r = -EOPNOTSUPP;
	if (!set_pio || ((r = set_pio(ap->port, ap->device, xfer)))) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, *ap->dev_name, "ATA CONTROLLER REFUSED TO SET PIO MODE %d: %s", pio, strerror(-r));
		return r;
	}
	memset(&rq.fis, 0, sizeof rq.fis);
	rq.fis.feature0 = ATA_FEATURE_SET_FEATURES_XFER;
	rq.fis.nsect = ATA_SET_XFER_PIO | pio;
	rq.fis.command = ATA_CMD_SET_FEATURES;
	rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_FEATURE;
	ATA$SYNC_RQ(&rq, ap, NULL, 0, timeout, 0);
	return rq.status;
}

int ATA$SETUP_DMA_XFER(ATA_ATTACH_PARAM *ap, __u16 *id, unsigned users_xfer, unsigned driver_xfer_mask, unsigned controller_xfer_mask, unsigned controller_xfer_mask_unsupported, int (*set_dma)(APORT *p, int drive, unsigned dma), jiffies_lo_t timeout)
{
	ATARQ rq;
	unsigned xfer;
	int dma, r;
	xfer = users_xfer & IDE_XFER_MASK_DMA;
	if (!xfer) {
		xfer = ATA$XFER_BEST(controller_xfer_mask & driver_xfer_mask & IDE_XFER_MASK_DMA);
	}
	if (!xfer) return 0;
	if (xfer & IDE_XFER_MASK_SDMA) dma = ATA$XFER_GET_NUMBER(xfer, IDE_XFER_MASK_SDMA);
	else if (xfer & IDE_XFER_MASK_WDMA) dma = ATA$XFER_GET_NUMBER(xfer, IDE_XFER_MASK_WDMA);
	else if (xfer & IDE_XFER_MASK_UDMA) dma = ATA$XFER_GET_NUMBER(xfer, IDE_XFER_MASK_UDMA);
	else KERNEL$SUICIDE("ATA$SETUP_DMA_XFER: NO DMA FLAGS IN %X", xfer);
	r = -EOPNOTSUPP;
	if (!set_dma || ((r = set_dma(ap->port, ap->device, xfer)))) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCOMPATIBILITY, *ap->dev_name, "ATA CONTROLLER REFUSED TO SET %cDMA MODE %d: %s", xfer & IDE_XFER_MASK_SDMA ? 'S' : xfer & IDE_XFER_MASK_WDMA ? 'W' : 'U', dma, strerror(-r));
		return r;
	}
	memset(&rq.fis, 0, sizeof rq.fis);
	rq.fis.feature0 = ATA_FEATURE_SET_FEATURES_XFER;
	if (xfer & IDE_XFER_MASK_SDMA) rq.fis.nsect = ATA_SET_XFER_SDMA | dma;
	else if (xfer & IDE_XFER_MASK_WDMA) rq.fis.nsect = ATA_SET_XFER_WDMA | dma;
	else rq.fis.nsect = ATA_SET_XFER_UDMA | dma;
	rq.fis.command = ATA_CMD_SET_FEATURES;
	rq.atarq_flags = ATA_PROTOCOL_NODATA | ATARQ_VALID_FEATURE;
	ATA$SYNC_RQ(&rq, ap, NULL, 0, timeout, 0);
	return rq.status;
}
