#include <SYS/TYPES.H>
#include <SPAD/VM.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>

#include <SPAD/ATA.H>

/* maximum is 8192 --- the table can't cross 64k boundary */
#define SGLIST_ENTRIES		2046

#define UNLOCK(sg)		((sg)->s[SGLIST_ENTRIES].dma_unmap)
#define DEV_ADDR(sg)		((sg)->s[SGLIST_ENTRIES + 2].dma.addr)
#define MAGIC(sg)		((sg)->s[SGLIST_ENTRIES + 2].dma.len)
#define  MAGIC_IDLE			0x454c4449
#define  MAGIC_DMA			0x00414d44
#define  MAGIC_PIO			0x004f4950
#define UPPER(e)		((e) + SGLIST_ENTRIES + 3)

typedef union {
	struct {
		__u32 addr;
		__u32 len;
#define LEN_LENGTH		0x0000ffffU
#define LEN_EOF			0x80000000U
	} dma;
	vspace_dmaunlock_t *dma_unmap;
	__u64 phys_addr;
	vspace_physunlock_t *phys_unmap;
} SGLIST_ENTRY;

struct __asglist {
	SGLIST_ENTRY s[SGLIST_ENTRIES * 2 + 4];
};

#define SGLIST_SENTINEL		((vspace_dmaunlock_t *)1L)

/* Layout of ASGLIST:
	SGLIST_ENTRIES		--- hardware table
	vspace_dmaunlock_t	--- unlock for hw table
				--- pad
	__u32 addr		--- device address of hw table
	SGLIST_ENTRIES		--- unlock for individual entries
	SGLIST_SENTINEL		--- sentinel
*/

#define SGLIST_CLUSTERS		((sizeof(ASGLIST) + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS)

ASGLIST *ATA$ALLOC_SGLIST(__u32 *dev_addr)
{
	int spl = KERNEL$SPL;
	int i;
	ASGLIST *sg;
	union {
		CONTIG_AREA_REQUEST car;
		struct {
			VDESC desc;
			VDMA dma;
		} s;
	} u;
	if (sizeof(SGLIST_ENTRY) != 8)
		KERNEL$SUICIDE("ATA$ALLOC_SGLIST: WEIRD COMPILER PADDING, STRUCTURE SIZE %d", (int)sizeof(SGLIST_ENTRY));
	u.car.nclusters = SGLIST_CLUSTERS;
	u.car.flags = CARF_DATA | CARF_PHYSCONTIG | CARF_PCIDMA;
	u.car.align = 65535;
	SYNC_IO_CANCELABLE(&u.car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (u.car.status < 0) return __ERR_PTR(u.car.status);
	sg = u.car.ptr;
	for (i = 0; i < SGLIST_ENTRIES; i++) {
		UPPER(&sg->s[i])->dma_unmap = NULL;
	}
	UPPER(&sg->s[SGLIST_ENTRIES])->dma_unmap = SGLIST_SENTINEL;
	u.s.desc.ptr = (unsigned long)&sg->s;
	u.s.desc.len = SGLIST_ENTRIES * sizeof(SGLIST_ENTRY);
	u.s.desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	u.s.dma = KERNEL$VIRTUAL.op->vspace_dmalock(&u.s.desc, PF_RW, &UNLOCK(sg));
	LOWER_SPLX(spl);
	if (__unlikely(u.s.dma.len != u.s.desc.len))
		KERNEL$SUICIDE("ATA$ALLOC_SGLIST: CAN'T DMALOCK DESCRIPTOR LIST");
	*dev_addr = DEV_ADDR(sg) = u.s.dma.ptr;
	MAGIC(sg) = MAGIC_IDLE;
	return sg;
}

void ATA$FREE_SGLIST(ASGLIST *sg)
{
	int spl = KERNEL$SPL;
	if (__unlikely(MAGIC(sg) != MAGIC_IDLE))
		KERNEL$SUICIDE("ATA$FREE_SGLIST: BAD MAGIC ON SGLIST %p: %08X", sg, (unsigned)MAGIC(sg));
	if (__unlikely(UPPER(&sg->s[SGLIST_ENTRIES])->dma_unmap != SGLIST_SENTINEL))
		KERNEL$SUICIDE("ATA$FREE_SGLIST: SENTINEL DAMAGED: %p", UPPER(&sg->s[SGLIST_ENTRIES])->dma_unmap);
	RAISE_SPL(SPL_VSPACE);
	UNLOCK(sg)(DEV_ADDR(sg));
	LOWER_SPLX(spl);
	KERNEL$VM_RELEASE_CONTIG_AREA(sg, SGLIST_CLUSTERS);
}

int ATA$SGLIST_RECOMMENDED_SIZE(ASGLIST *sg, int dma)
{
	if (dma) return SGLIST_ENTRIES << __PAGE_CLUSTER_BITS;
	else return SGLIST_ENTRIES << BIO_SECTOR_SIZE_BITS;
}

int ATA$SGLIST_GUARANTEED_SIZE(ASGLIST *sg, int dma)
{
	if (dma) return SGLIST_ENTRIES << BIO_SECTOR_SIZE_BITS;
	else return SGLIST_ENTRIES << BIO_SECTOR_SIZE_BITS;
}

static VDMA ATA_MAP_DMA_SPLIT(VDMA dma, SGLIST_ENTRY *e)
{
	SGLIST_ENTRY *orig_e = e;
	unsigned total_len = 0;
	do {
		unsigned this_len = 0x10000 - (dma.ptr & 0xffff);
		total_len += this_len;
		e->dma.addr = __32CPU2LE(dma.ptr);
		e->dma.len = __32CPU2LE(this_len & 0xffff);
		e++;
		if (__unlikely(UPPER(e)->dma_unmap == SGLIST_SENTINEL)) {
			goto too_long;
		}
		UPPER(e)->dma_unmap = KERNEL$NULL_VSPACE_DMAUNLOCK;
		dma.len -= this_len;
		dma.ptr += this_len;
		TEST_SPL(SPL_ATA_SCSI, SPL_VSPACE);
	} while ((dma.ptr & 0xffff) + dma.len > 0x10000);
	e->dma.addr = __32CPU2LE(dma.ptr);
	total_len += dma.len;
	e->dma.len = __32CPU2LE(dma.len & 0xffff);
	e++;
	too_long:
	dma.len = total_len;
	dma.ptr = e - orig_e;
	return dma;
}

int ATA$MAP_DMA(ASGLIST *sg, BIODESC **pdesc, unsigned plen, int rw, int preserve)
{
	SGLIST_ENTRY *e = sg->s;
	BIODESC *desc = *pdesc;
	unsigned len = plen;
	int r;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ATA$MAP_DMA AT SPL %08X", KERNEL$SPL);
#endif
#if __DEBUG >= 2
	if (__unlikely(MAGIC(GET_STRUCT(e, ASGLIST, s)) != MAGIC_IDLE))
		KERNEL$SUICIDE("ATA$MAP_DMA: BAD MAGIC ON SGLIST %p: %08X", sg, (unsigned)MAGIC(GET_STRUCT(e, ASGLIST, s)));
	MAGIC(GET_STRUCT(e, ASGLIST, s)) = MAGIC_DMA;
#endif
	RAISE_SPL(SPL_VSPACE);
	do {
		unsigned long this_desc = 0;
		unsigned long l;
		VDMA dma;
		again:
		TEST_SPL(SPL_ATA_SCSI, SPL_VSPACE);
#if __DEBUG >= 1
		if (__unlikely(((unsigned)desc->v.ptr | (unsigned)desc->v.len) & (BIO_SECTOR_SIZE - 1)) || __unlikely(!desc->v.len))
			KERNEL$SUICIDE("ATA$MAP_DMA: UNALIGNED VDESC: %"__64_format"X, %lX", (__u64)desc->v.ptr, (unsigned long)desc->v.len);
#endif
		if (__unlikely(UPPER(e)->dma_unmap == SGLIST_SENTINEL)) {
			pres_too_long:
			if (preserve) {
				desc->v.ptr -= this_desc;
				desc->v.len += this_desc;
			}
			goto too_long;
		}
		if (__unlikely(!len)) {
			goto pres_too_long;
		}
		l = desc->v.len;
		if (__unlikely(l > len)) desc->v.len = len;
		dma = desc->v.vspace->op->vspace_dmalock(&desc->v, rw, &UPPER(e)->dma_unmap);
		desc->v.len = l;
		if (__unlikely(!dma.len)) {
			goto pres_too_long;
		}
		if (__unlikely((dma.ptr & 0xffff) + dma.len > 0x10000)) {
			dma = ATA_MAP_DMA_SPLIT(dma, e);
			e += dma.ptr;
		} else {
			e->dma.addr = __32CPU2LE(dma.ptr);
			e->dma.len = __32CPU2LE(dma.len & 0xffff);
			e++;
		}
		len -= dma.len;
		if (__unlikely(desc->v.len != dma.len)) {
			desc->v.len -= dma.len;
			desc->v.ptr += dma.len;
			this_desc += dma.len;
			goto again;
		}
		if (__unlikely(preserve)) {
			desc->v.ptr -= this_desc;
			desc->v.len += this_desc;
		}
	} while ((desc = desc->next));
#if __DEBUG >= 1
	if (__unlikely(len))
		KERNEL$SUICIDE("ATA$MAP_DMA: CHAIN HAD LESS DATA THAN REPORTED, LEFT %X", len);
#endif
	too_long:
	LOWER_SPL(SPL_ATA_SCSI);
	r = plen - len;
	if (__unlikely(!r)) {
#if __DEBUG >= 2
		if (__unlikely(MAGIC(GET_STRUCT(e, ASGLIST, s)) != MAGIC_DMA))
			KERNEL$SUICIDE("ATA$MAP_DMA: BAD MAGIC ON SGLIST %p IN PAGEFAULT: %08X", sg, (unsigned)MAGIC(GET_STRUCT(e, ASGLIST, s)));
		MAGIC(GET_STRUCT(e, ASGLIST, s)) = MAGIC_IDLE;
#endif
		return -EVSPACEFAULT;
	}
#if __DEBUG >= 2
	if (__unlikely(e - sg->s > SGLIST_ENTRIES))
		KERNEL$SUICIDE("ATA$MAP_DMA: RAN OUT OF SG TABLE: %ld > %d", (long)(e - sg->s), SGLIST_ENTRIES);
#endif
	e[-1].dma.len |= __32CPU2LE(LEN_EOF);
	if (__likely(!preserve)) *pdesc = desc;
	return r;
}

void ATA$UNMAP_DMA(ASGLIST *sg)
{
	SGLIST_ENTRY *e = sg->s;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ATA$UNMAP_DMA AT SPL %08X", KERNEL$SPL);
#endif
#if __DEBUG >= 2
	if (__unlikely(MAGIC(GET_STRUCT(e, ASGLIST, s)) != MAGIC_DMA))
		KERNEL$SUICIDE("ATA$UNMAP_DMA: BAD MAGIC ON SGLIST %p: %08X", sg, (unsigned)MAGIC(GET_STRUCT(e, ASGLIST, s)));
	MAGIC(GET_STRUCT(e, ASGLIST, s)) = MAGIC_IDLE;
	if (__unlikely(UPPER(&sg->s[SGLIST_ENTRIES])->dma_unmap != SGLIST_SENTINEL))
		KERNEL$SUICIDE("ATA$UNMAP_DMA: SENTINEL DAMAGED: %p", UPPER(&sg->s[SGLIST_ENTRIES])->dma_unmap);
#endif
	RAISE_SPL(SPL_VSPACE);
	do {
		TEST_SPL(SPL_ATA_SCSI, SPL_VSPACE);
		UPPER(e)->dma_unmap(__32LE2CPU(e->dma.addr));
	} while (!(e++->dma.len & __32CPU2LE(LEN_EOF)));
	LOWER_SPL(SPL_ATA_SCSI);
}

int ATA$MAP_PIO(ASGLIST *sg, BIODESC **pdesc, unsigned plen, int rw, int preserve)
{
	SGLIST_ENTRY *e = sg->s;
	BIODESC *desc = *pdesc;
	unsigned len = plen;
	int r;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ATA$MAP_PIO AT SPL %08X", KERNEL$SPL);
	if (__unlikely(MAGIC(GET_STRUCT(e, ASGLIST, s)) != MAGIC_IDLE))
		KERNEL$SUICIDE("ATA$MAP_PIO: BAD MAGIC ON SGLIST %p: %08X", sg, (unsigned)MAGIC(GET_STRUCT(e, ASGLIST, s)));
	MAGIC(GET_STRUCT(e, ASGLIST, s)) = MAGIC_PIO;
#endif
	RAISE_SPL(SPL_VSPACE);
	do {
		unsigned long this_desc = 0;
		unsigned long l;
		VPHYS phys;
		again:
		TEST_SPL(SPL_ATA_SCSI, SPL_VSPACE);
#if __DEBUG >= 1
		if ((__unlikely((unsigned)desc->v.len & (BIO_SECTOR_SIZE - 1)) && __unlikely(desc->next != NULL)) || __unlikely(!desc->v.len))
			KERNEL$SUICIDE("ATA$MAP_PIO: UNALIGNED VDESC: %"__64_format"X, %lX", (__u64)desc->v.ptr, (unsigned long)desc->v.len);
#endif
		if (__unlikely(UPPER(e)->dma_unmap == SGLIST_SENTINEL)) {
			pres_too_long:
			if (preserve) {
				desc->v.ptr -= this_desc;
				desc->v.len += this_desc;
			}
			goto too_long;
		}
		if (__unlikely(!len)) {
			goto pres_too_long;
		}
		l = desc->v.len;
		if (__likely(desc->v.len >= BIO_SECTOR_SIZE))
			desc->v.len = BIO_SECTOR_SIZE;
		if (__unlikely(desc->v.len > len))
			desc->v.len = len;
		desc->v.vspace->op->vspace_physlock(&desc->v, rw, &phys, &UPPER(e)->phys_unmap);
		desc->v.len = l - phys.len;
		desc->v.ptr += phys.len;
		this_desc += phys.len;
		if (__unlikely(!phys.len)) {
			goto pres_too_long;
		}
		len -= phys.len;
		e->phys_addr = phys.ptr;
		e++;
		if (__unlikely((phys.len & (BIO_SECTOR_SIZE - 1)) != 0)) {
			goto pres_too_long;
		}
		if (__likely(desc->v.len != 0)) {
			goto again;
		}
		if (__unlikely(preserve)) {
			desc->v.ptr -= this_desc;
			desc->v.len += this_desc;
		}
	} while ((desc = desc->next));
#if __DEBUG >= 1
	if (__unlikely(len))
		KERNEL$SUICIDE("ATA$MAP_PIO: CHAIN HAD LESS DATA THAN REPORTED, LEFT %X", len);
#endif
	too_long:
	LOWER_SPL(SPL_ATA_SCSI);
	r = plen - len;
	if (__unlikely(!r)) {
#if __DEBUG >= 1
		if (__unlikely(MAGIC(GET_STRUCT(e, ASGLIST, s)) != MAGIC_PIO))
			KERNEL$SUICIDE("ATA$MAP_PIO: BAD MAGIC ON SGLIST %p IN PAGEFAULT: %08X", sg, (unsigned)MAGIC(sg));
		MAGIC(GET_STRUCT(e, ASGLIST, s)) = MAGIC_IDLE;
#endif
		return -EVSPACEFAULT;
	}
#if __DEBUG >= 1
	if (__unlikely(e - sg->s > SGLIST_ENTRIES))
		KERNEL$SUICIDE("ATA$MAP_PIO: RAN OUT OF SG TABLE: %ld > %d", (long)(e - sg->s), SGLIST_ENTRIES);
#endif
	if (__likely(e - sg->s < SGLIST_ENTRIES)) UPPER(e)->phys_unmap = NULL;
	if (__likely(!preserve)) *pdesc = desc;
	return r;
}

void ATA$UNMAP_PIO(ASGLIST *sg)
{
	SGLIST_ENTRY *e = sg->s;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("ATA$UNMAP_PIO AT SPL %08X", KERNEL$SPL);
	if (__unlikely(MAGIC(GET_STRUCT(e, ASGLIST, s)) != MAGIC_PIO))
		KERNEL$SUICIDE("ATA$UNMAP_PIO: BAD MAGIC ON SGLIST %p: %08X", sg, (unsigned)MAGIC(GET_STRUCT(e, ASGLIST, s)));
	MAGIC(GET_STRUCT(e, ASGLIST, s)) = MAGIC_IDLE;
	if (__unlikely(UPPER(&sg->s[SGLIST_ENTRIES])->dma_unmap != SGLIST_SENTINEL))
		KERNEL$SUICIDE("ATA$UNMAP_PIO: SENTINEL DAMAGED: %p", UPPER(&sg->s[SGLIST_ENTRIES])->dma_unmap);
#endif
	RAISE_SPL(SPL_VSPACE);
	while (__likely(e < sg->s + SGLIST_ENTRIES) && __likely(UPPER(e)->phys_unmap != NULL)) {
		TEST_SPL(SPL_ATA_SCSI, SPL_VSPACE);
		UPPER(e)->phys_unmap(e->phys_addr);
		e++;
	}
	LOWER_SPL(SPL_ATA_SCSI);
}

void *ATA$MAP_PIO_SECTOR(ASGLIST *sg, unsigned pos)
{
#if __DEBUG >= 1
	if (__unlikely(MAGIC(sg) != MAGIC_PIO))
		KERNEL$SUICIDE("ATA$MAP_PIO_SECTOR: BAD MAGIC ON SGLIST %p: %08X", sg, (unsigned)MAGIC(sg));
	if (__unlikely(pos >= SGLIST_ENTRIES) || __unlikely(!UPPER(&sg->s[pos])->phys_unmap))
		KERNEL$SUICIDE("ATA$MAP_PIO_SECTOR: INVALID OFFSET %u", pos);
#endif
	return KERNEL$MAP_PHYSICAL_BANK(sg->s[pos].phys_addr);
}

void ATA$SET_NEXT_PIO_SECTOR(ASGLIST *sg, unsigned *pos)
{
	(*pos)++;
}

void ATA$UNMAP_PIO_SECTOR(ASGLIST *sg, void *data)
{
	KERNEL$UNMAP_PHYSICAL_BANK(data);
}
