#include <SYS/TYPES.H>
#include <SPAD/BIO.H>
#include <SPAD/AC.H>
#include <STRING.H>
#include <SPAD/LIBC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <ARCH/SETUP.H>
#include <SPAD/SLAB.H>
#include <SPAD/TIMER.H>
#include <SPAD/VM.H>
#include <SPAD/DL.H>
#include <SPAD/IOCTL.H>
#include <SYS/STAT.H>
#include <TIME.H>
#include <VALUES.H>
#include <SPAD/HASH.H>

#include <SPAD/BIO_KRNL.H>

/* just a soft limit --- handles are closed when you overflow it */
#define N_PARTITIONS		32

#define BUFFER_SECTORS		(PG_SIZE / BIO_SECTOR_SIZE - 2)

#define SBIORQ_ENTRIES		\
	BIORQ b;		\
	BIODESC desc;		\
	__sec_t sec;		\
	unsigned nsec;		\
	_u_off_t off;		\
	unsigned written;	\
	int flags;		\
	PARTITIONS *p;		\
	PROC_TAG tag;		\
	void *sio

typedef struct {
	SBIORQ_ENTRIES;
	__u8 pad[1];
} SBIORQ_FOR_ALIGN;

typedef struct {
	SBIORQ_ENTRIES;
	__u8 pad[BIO_SECTOR_SIZE - (__offsetof(SBIORQ_FOR_ALIGN, pad) & (BIO_SECTOR_SIZE - 1))];
	__u8 buffer[BIO_SECTOR_SIZE * BUFFER_SECTORS];
} SBIORQ;

#define SBIORQ_LOCK	1

struct __partitions {
	int (*ioctl)(IOCTLRQ *, PARTITION *, IORQ *);
	struct __slhead iorqs;
	WQ iorqs_wait;
	int lock;
	int n;
	int clock;
	void *dummy_rq;
	blksize_t tmp_blksize;
	char *dev_name;
	PARTITION p[1];
};

static void KILL_PARTITION_HANDLES(PARTITION *p)
{
	while (!XLIST_EMPTY(&p->list)) {
		HANDLE *h = LIST_STRUCT(p->list.next, HANDLE, fnode_entry);
		KERNEL$DETACH_HANDLE(h);
	}
}

static int DUMMY_IOCTL(IOCTLRQ *rq, PARTITION *part, IORQ *rq_to_queue)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV))) KERNEL$SUICIDE("DUMMY_IOCTL AT SPL %08X", KERNEL$SPL);
#endif
	return -1;
}

int BIO$ALLOC_PARTITIONS(PARTITIONS **pp, void *dev, __sec_t dev_blocx, int (*ioctl)(IOCTLRQ *, PARTITION *, IORQ *), char *dev_name)
{
	int i;
	PARTITIONS *p;
	MALLOC_REQUEST mrq;
	int r;
	mrq.size = sizeof(PARTITIONS) + N_PARTITIONS * sizeof(PARTITION);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) return mrq.status;
	p = mrq.ptr;
	p->ioctl = ioctl ? ioctl : DUMMY_IOCTL;
	p->n = N_PARTITIONS + 1;
	p->clock = 1;
	p->dev_name = dev_name;
	for (i = 0; i < p->n; i++) {
		p->p[i].start = p->p[i].len = 0;
		p->p[i].dev = dev;
		p->p[i].p = p;
		INIT_XLIST(&p->p[i].list);
	}
	p->p[0].len = dev_blocx;
	KERNEL$SLAB_INIT(&p->iorqs, sizeof(SBIORQ), BIO_SECTOR_SIZE, VM_TYPE_WIRED_MAPPED, NULL, NULL, NULL, "BIO$SBIORQ");
	if (__unlikely(r = KERNEL$SLAB_RESERVE(&p->iorqs, 2))) {
		KERNEL$SLAB_DESTROY(&p->iorqs);
		KERNEL$UNIVERSAL_FREE(p);
		return r;
	}
	p->dummy_rq = __slalloc(&p->iorqs);
	if (__unlikely(!p->dummy_rq))
		KERNEL$SUICIDE("BIO$ALLOC_PARTITIONS: CAN'T ALLOCATE DUMMY IO REQUEST");
	WQ_INIT(&p->iorqs_wait, "BIO$IORQS_WAIT");
	p->lock = 0;
	*pp = p;
	return 0;
}

	/* must not block */
void BIO$SET_PARTITIONS_SIZE(PARTITIONS *p, __sec_t dev_blocx)
{
	int i;
	p->p[0].len = dev_blocx;
	for (i = 0; i < p->n; i++) KILL_PARTITION_HANDLES(&p->p[i]);
}

void BIO$FREE_PARTITIONS(PARTITIONS *p)
{
	int i;
	for (i = 0; i < p->n; i++) if (__unlikely(!XLIST_EMPTY(&p->p[i].list)))
		KERNEL$SUICIDE("BIO$FREE_PARTITIONS: PARTITION (%016"__sec_t_format"X, %016"__sec_t_format"X) IS STILL USED", p->p[i].start, p->p[i].len);
	__slow_slfree(p->dummy_rq);
	while (__unlikely(!KERNEL$SLAB_EMPTY(&p->iorqs))) KERNEL$SLEEP(1);
	KERNEL$SLAB_DESTROY(&p->iorqs);
	WQ_WAKE_ALL(&p->iorqs_wait);
	KERNEL$UNIVERSAL_FREE(p);
}

void BIO$ROOT_PARTITION(HANDLE *h, PARTITIONS *p)
{
	h->fnode = &p->p[0];
}

void *BIO$LOOKUP_PARTITION(HANDLE *h, char *str, int open_flags)
{
	PARTITION *p = (PARTITION *)h->fnode;
	PARTITIONS *pp;
	PARTITION *ppp;
	__u64 start, len;
	char *mod;
	if (__unlikely(str[0] != '^')) return __ERR_PTR(-ENOTDIR);
	str++;
	if (__unlikely(_memcasecmp(str, "PART=", 5))) return __ERR_PTR(-EBADMOD);
	str += 5;
	if (__unlikely(!(mod = strchr(str, ',')))) return __ERR_PTR(-EBADSYN);
	if (__unlikely(__get_64_number(str, mod, 0, (__s64 *)&start))) return __ERR_PTR(-EBADSYN);
	if (__unlikely(__get_64_number(mod + 1, mod + 1 + strlen(mod + 1), 0, (__s64 *)&len))) return __ERR_PTR(-EBADSYN);
	if (__unlikely(start + len > p->len) || __unlikely(start + len < start)) return __ERR_PTR(-ERANGE);
	start += p->start;
	pp = p->p;
	for (ppp = pp->p; ppp < pp->p + pp->n; ppp++)
		if (ppp->start == start && ppp->len == len) goto ok2;
	for (ppp = pp->p + 1; ppp < pp->p + pp->n; ppp++)
		if (!ppp->start && !ppp->len && XLIST_EMPTY(&ppp->list)) goto ok;
	for (ppp = pp->p + 1; ppp < pp->p + pp->n; ppp++)
		if (XLIST_EMPTY(&ppp->list)) goto ok;
	ppp = pp->p + pp->clock;
	if (__unlikely(++pp->clock >= pp->n)) pp->clock = 1;
	KILL_PARTITION_HANDLES(ppp);
	ok:
	ppp->start = start;
	ppp->len = len;
	ok2:
	h->fnode = ppp;
	return NULL;
}

void *BIO$INSTANTIATE_PARTITION(HANDLE *h, IORQ *rq, int open_flags)
{
	PARTITION *p = h->fnode;
	if (__likely(h->file_addrspace != NULL)) ADD_TO_XLIST(&p->list, &h->fnode_entry);
	else VOID_LIST_ENTRY(&h->fnode_entry);
	return NULL;
}

void BIO$DETACH(HANDLE *h)
{
	DEL_FROM_LIST(&h->fnode_entry);
}

void BIO$FAULT(BIORQ *rq, __sec_t sec)
{
	if (__likely(sec != -1)) {
		if (__unlikely(sec < rq->partition)) KERNEL$SUICIDE("BIO$FAULT: SECTOR %016"__64_format"X, PARTITION %016"__64_format"X", (__u64)sec, (__u64)rq->partition);
		sec -= rq->partition;
	}
	if (__likely(!rq->status)) {
		rq->status = -EVSPACEFAULT;
		f: rq->fault_sec = sec;
	} else {
		if (__likely(rq->status == -EVSPACEFAULT) && __unlikely((__s64)sec < (__s64)rq->fault_sec)) goto f;
	}
}

void BIO$ERROR(BIORQ *rq, __sec_t sec)
{
	if (__likely(sec != -1)) {
		if (__unlikely(sec < rq->partition)) sec = -1;
		else sec -= rq->partition;
	}
	if (__likely(rq->status != -EIO) || __unlikely((__s64)sec < (__s64)rq->fault_sec)) rq->fault_sec = sec;
	rq->status = -EIO;
}

static void FREE_SB(SBIORQ *sb, PARTITIONS *p)
{
	WQ_WAKE_ALL(&p->iorqs_wait);
	RAISE_SPL(SPL_VSPACE);
	KERNEL$RELEASE_IO_TAG(&sb->tag);
	LOWER_SPL(SPL_DEV);
	__slfree(sb);
}

#define RW	0
#define AS	0
#include "SBIO.I"
#undef AS
#define AS	1
#include "SBIO.I"
#undef AS
#undef RW
#define RW	1
#define AS	0
#include "SBIO.I"
#undef AS
#define AS	1
#include "SBIO.I"
#undef AS
#undef RW

static __finline__ void VFS_FILLSTAT(struct stat *stat, PARTITION *pa)
{
#if 0
	char *c;
	ino_t hash;
#endif
	stat->st_mode = 0600 | S_IFBLK;
	stat->st_nlink = 1;
	stat->st_atime = stat->st_mtime = stat->st_ctime = time(NULL);
	if (pa->len != -1) {
		stat->st_size = pa->len << BIO_SECTOR_SIZE_BITS;
		stat->st_blocks = pa->len;
	}
#if 0
	hash = 0;
	c = pa->p->dev_name;
	quickcasehash(c, *c, hash);
	stat->st_dev = hash;
	c = (char *)&pa->start;
	quickcasehash(c, c < (char *)(&pa->start + 1), hash);
	c = (char *)&pa->len;
	quickcasehash(c, c < (char *)(&pa->len + 1), hash);
	stat->st_ino = stat->st_rdev = hash;
#endif
	stat->st_blksize = pa->p->tmp_blksize;
}

#define VFS_FN_SPL	SPL_DEV

#include <SPAD/VFS_FN.H>

extern AST_STUB FLUSH_FINISHED;

DECL_IOCALL(BIO$IOCTL, SPL_DEV, IOCTLRQ)
{
	PARTITION *pa;
	PARTITIONS *p;
	int r;
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op->ioctl != BIO$IOCTL)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_DEV));
	pa = h->fnode;
	p = pa->p;
	r = p->ioctl(RQ, pa, (IORQ *)RQ);
	if (__likely(!r)) {
		switch (RQ->ioctl) {
			case IOCTL_BIO_PHYSICAL_BLOCKSIZE: {
				unsigned n;
				if (__unlikely(RQ->status < 0)) break;
				if (!(unsigned)pa->start) break;
				n = 1 << __BSF((unsigned)pa->start);
				if (n < RQ->status) RQ->status = n;
				break;
			}
		}
		RETURN_AST(RQ);
	}
	if (__unlikely(r >= 0)) {
		RETURN;
	}
	switch (RQ->ioctl) {
		case IOCTL_STAT: {
			IOCTLRQ io;
			io.status = RQS_PROCESSING;
			io.h = -1;
			io.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
			io.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_READ;
			io.v.ptr = 0;
			io.v.len = 0;
			io.v.vspace = &KERNEL$VIRTUAL;
			io.handle = h;
			r = pa->p->ioctl(&io, pa, (IORQ *)RQ);
			if (__likely(!r) && __likely(io.status > 0)) p->tmp_blksize = BIO_SECTOR_SIZE << __BSR(io.status);
			else if (__unlikely(r > 0)) RETURN;
			else p->tmp_blksize = OPTIMAL_IO_SIZE;
			r = VFS_STAT(RQ, pa);
			if (__likely(r <= 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
		}
		case IOCTL_LSEEK: {
			r = VFS_LSEEK(RQ, 0);
			if (__likely(r <= 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (r == 2) RETURN;
			DO_PAGEIN(RQ, &RQ->v, PF_RW);
		}
		case IOCTL_FSYNC: {
			SBIORQ *sb;
			WQ *wq;
			if (RQ->param & PARAM_FSYNC_ASYNC) {
				/* disks sync their cache automatically, no
				   need to ask them */
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(sb = __slalloc(&p->iorqs)))) {
				WQ_WAIT_F(&p->iorqs_wait, RQ);
				RETURN;
			}
			RAISE_SPL(SPL_VSPACE);
			if (__unlikely((wq = KERNEL$ACQUIRE_IO_TAG(&sb->tag, h->name_addrspace)) != NULL)) {
				WQ_WAIT_F(wq, RQ);
				LOWER_SPL(SPL_DEV);
				__slow_slfree(sb);
				RETURN;
			}
			LOWER_SPL(SPL_DEV);
			sb->flags = 0;
			sb->sio = RQ;
			sb->p = p;
			sb->b.h = h->handle_num & HANDLE_NUM_MASK;
			sb->b.handle = h;
			sb->b.flags = BIO_FLUSH;
			sb->b.sec = 0;
			sb->b.nsec = 1;
			sb->b.proc = sb->tag.proc;
			sb->b.fault_sec = -1;
			sb->b.fn = FLUSH_FINISHED;
			RETURN_IORQ_CANCELABLE(&sb->b, KERNEL$WAKE_BIO, RQ);
		}
		case IOCTL_FCNTL_GETFL:
		case IOCTL_FCNTL_SETFL: {
			if (__likely(!VFS_GETSETFL(RQ))) RETURN_AST(RQ);
			RETURN;
		}
		case IOCTL_BIO_PHYSICAL_BLOCKSIZE:
		case IOCTL_BIO_PARTITION_BLOCKSIZE: {
			RQ->status = 512;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

DECL_AST(FLUSH_FINISHED, SPL_DEV, BIORQ)
{
	SBIORQ *sb = LIST_STRUCT(RQ, SBIORQ, b);
	IOCTLRQ *sio = sb->sio;
	PARTITIONS *p;
	IO_DISABLE_CHAIN_CANCEL(SPL_DEV, sio);
	sio->status = sb->b.status;
	p = sb->p;
	RAISE_SPL(SPL_VSPACE);
	KERNEL$RELEASE_IO_TAG(&sb->tag);
	LOWER_SPL(SPL_DEV);
	__slfree(sb);
	WQ_WAKE_ALL(&p->iorqs_wait);
	RETURN_AST(sio);
}

void BIO$CHECK_REQUEST(__const__ char *msg, BIORQ *rq)
{
	BIODESC *desc;
	unsigned long len;
	if (__unlikely(rq->flags & BIO_FLUSH)) {
		return;
	}
	if (__unlikely(rq->nsec <= 0)) {
		unsigned long off = 0;
		__const__ char *str = KERNEL$DL_GET_SYMBOL_NAME(rq->fn, &off, 0);
		KERNEL$SUICIDE("BIO$CHECK_REQUEST(%s): INVALID NUMBER OF SECTORS %X, SECTOR %"__sec_t_format"X, AST %s+%lX", msg, rq->nsec, rq->sec, str ? str : "?", off);
	}
	len = 0;
	desc = rq->desc;
	do {
		if (__unlikely(((unsigned)desc->v.ptr | (unsigned)desc->v.len) & (BIO_SECTOR_SIZE - 1)) || __unlikely(!desc->v.len)) {
			unsigned long off = 0;
			__const__ char *str = KERNEL$DL_GET_SYMBOL_NAME(rq->fn, &off, 0);
			KERNEL$SUICIDE("BIO$CHECK_REQUEST(%s): UNALIGNED DESCRIPTOR: %"__64_format"X, %lX, AST %s+%lX", msg, (__u64)desc->v.ptr, desc->v.len, str ? str : "?", off);
		}
		len += desc->v.len;
	} while ((desc = desc->next));
	if (__unlikely(len != ((unsigned long)rq->nsec << BIO_SECTOR_SIZE_BITS))) {
		unsigned long off = 0;
		__const__ char *str = KERNEL$DL_GET_SYMBOL_NAME(rq->fn, &off, 0);
		KERNEL$SUICIDE("BIO$CHECK_REQUEST(%s): LENGTH MISMATCH, LEN %lX, SECTORS %X, SECTOR %"__sec_t_format"X, AST %s+%lX", msg, len, rq->nsec, rq->sec, str ? str : "?", off);
	}
}
