#include <SPAD/TIMER.H>
#include <STDLIB.H>
#include <SPAD/SYSLOG.H>
#include <STRING.H>

#include <SPAD/SCSI.H>

#define SCSI_INQUIRY_TIMEOUT	(5 * JIFFIES_PER_SECOND)
#define SCSI_INQUIRY_RETRIES	3

int SCSI$INQUIRY(__const__ char *dev_name, int errorlevel, SCSI_ATTACH_PARAM *ap, __u8 lun_byte, void *data, unsigned datalen, jiffies_lo_t timeout)
{
	DECL_SCSIRQ(6) inq;
	SCSI_SENSE_DATA inq_sense;
	memset(data, 0, datalen);
	if (ap->scsi_flags & SCSI_FLAG_SHORT_INQUIRY && __likely(datalen > SCSI_SHORT_INQUIRY_SIZE)) datalen = SCSI_SHORT_INQUIRY_SIZE;
	inq.sense = (__u8 *)&inq_sense;
	inq.sense_size = SCSI_SIZEOF_SENSE_DATA;
	inq.direction = PF_WRITE;
	inq.cmdlen = 6;
	inq.cmd[0] = SCMD_INQUIRY;
	inq.cmd[1] = lun_byte;
	inq.cmd[2] = 0;
	inq.cmd[3] = datalen >> 8;
	inq.cmd[4] = datalen;
	inq.cmd[5] = 0;
	SCSI$SYNC_RQ(dev_name, errorlevel, "INQUIRY", (SCSIRQ *)&inq, ap, data, datalen, timeout ? timeout : SCSI_INQUIRY_TIMEOUT, SCSI_INQUIRY_RETRIES, SCSI$RECOMMENDED_ACTION);
	if (__unlikely(inq.status < SCSI_MIN_INQUIRY_SIZE)) {
		if (__likely(inq.status < 0)) return inq.status;
		if (inq.scsi_status != SCSI_GOOD) {
			if (inq.scsi_status == SCSI_CHECK_CONDITION) {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INQUIRY: %s", SCSI$SENSE_STRING(0, &inq_sense, SCSI_SIZEOF_SENSE_DATA));
			} else {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INQUIRY: %s", SCSI$STATUS_STRING(inq.scsi_status));
			}
			return SCSI$SENSE_ERROR_CODE(0, &inq_sense, SCSI_SIZEOF_SENSE_DATA);
		} else {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INQUIRY RETURNED ONLY %d BYTES", inq.status);
			return -EPROTO;
		}
	}
	return 0;
}

static void GET_MODEL_STRING(__u8 *src, char *str, unsigned len);

void SCSI$GET_MODEL(SCSI_INQUIRY_DATA *inq, char (*vendor)[9], char (*product)[17], char (*revision)[5])
{
	if (vendor) GET_MODEL_STRING(inq->vendor, *vendor, 8);
	if (product) GET_MODEL_STRING(inq->product, *product, 16);
	if (revision) GET_MODEL_STRING(inq->revision, *revision, 4);
}

static void GET_MODEL_STRING(__u8 *src, char *str, unsigned len)
{
	int i, ii;
	strlcpy(str, src, len + 1);
	for (i = strlen(str) - 1; i >= 0; i--) if (str[i] == ' ') str[i] = 0;
					       else break;
	for (i = 0, ii = 0; str[i]; i++) if (str[i] != ' ' || (ii && str[ii - 1] != ' ')) str[ii++] = str[i];
	str[ii] = 0;
}

SCSI_MODE_SENSE_HEADER *SCSI$MODE_SENSE(__const__ char *dev_name, int errorlevel, __const__ SCSI_INQUIRY_DATA *inq, __u8 scsi_version, SCSI_ATTACH_PARAM *ap, __u8 lun_byte, int modepage, unsigned len, jiffies_lo_t timeout, int retries)
{
	__u32 key;
	DECL_SCSIRQ(10) cmd;
	SCSI_SENSE_DATA sense;
	SCSI_MODE_SENSE_HEADER *data;
	int r;
	if (__unlikely(scsi_version == 1) && modepage != 0) return __ERR_PTR(-EOPNOTSUPP);
	len += sizeof(SCSI_MODE_SENSE_HEADER);
	data = __sync_malloc(len);
	if (__unlikely(!data)) return __ERR_PTR(-ENOMEM);
	memset(data, 0, len);
	cmd.sense = (__u8 *)&sense;
	cmd.sense_size = SCSI_SIZEOF_SENSE_DATA;
	if (__likely(ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_10)) {
		cmd.direction = PF_WRITE;
		cmd.cmdlen = 10;
		cmd.cmd[0] = SCMD_MODE_SENSE_10;
		cmd.cmd[1] = lun_byte | 0x08 | (scsi_version >= 4 ? 0x10 : 0);
		cmd.cmd[2] = modepage;
		cmd.cmd[3] = 0;
		cmd.cmd[4] = 0;
		cmd.cmd[5] = 0;
		cmd.cmd[6] = 0;
		cmd.cmd[7] = len >> 8;
		cmd.cmd[8] = len;
		cmd.cmd[9] = 0;
		SCSI$SYNC_RQ(dev_name, errorlevel, "MODE SENSE(10)", (SCSIRQ *)&cmd, ap, data, len, timeout, retries, SCSI$RECOMMENDED_ACTION);
		if (__likely(SCSI$RECOMMENDED_ACTION((SCSIRQ *)&cmd) == SCSI_ACTION_OK)) {
			unsigned data_len = __16BE2CPU(data->data_length) + 2;
			unsigned should_transfer = data_len < len ? data_len : len;
			if (__unlikely(cmd.status < should_transfer)) {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "MODE SENSE(10)/PAGE %02X RETURNED NOT ENOUGH DATA (TRANSFERED %d, DESCRIPTOR %d, ALLOCATED %d)", modepage, cmd.status, data_len, len);
				if (cmd.status < sizeof(SCSI_MODE_SENSE_HEADER)) {
					__slow_free(data);
					return __ERR_PTR(-EINVAL);
				}
				data->data_length = __16CPU2BE(cmd.status) + 2;
			}
			return data;
		}
		key = SCSI$GET_CMD_SENSE((SCSIRQ *)&cmd);
		if (__likely(!SCSI_SENSE_UNKNOWN(key)) && __likely(SCSI_SENSE_KEY(key) == SCSI_KEY_ILLEGAL_REQUEST) && __likely(SCSI_SENSE_ASC(key) == 0x20) && SCSI_SENSE_ASCQ(key) == 0x00) {
			if (ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_6) {
				ap->scsi_flags &= ~SCSI_FLAG_CAN_USE_MODE_10;
				goto try_mode_6;
			}
		}
	} else try_mode_6: if (__likely(ap->scsi_flags & SCSI_FLAG_CAN_USE_MODE_6)) {
		if (__unlikely(len - 4 >= 256)) {
			__slow_free(data);
			return __ERR_PTR(-ERANGE);
		}
		cmd.direction = PF_WRITE;
		cmd.cmdlen = 6;
		cmd.cmd[0] = SCMD_MODE_SENSE_6;
		cmd.cmd[1] = lun_byte | 0x08;
		cmd.cmd[2] = modepage;
		cmd.cmd[3] = 0;
		cmd.cmd[4] = len - 4;
		cmd.cmd[5] = 0;
		SCSI$SYNC_RQ(dev_name, errorlevel, "MODE SENSE(6)", (SCSIRQ *)&cmd, ap, (__u8 *)data + 4, len - 4, timeout, retries, SCSI$RECOMMENDED_ACTION);
		if (__likely(SCSI$RECOMMENDED_ACTION((SCSIRQ *)&cmd) == SCSI_ACTION_OK)) {
			SCSI_MODE_SENSE_6_HEADER *s6 = (SCSI_MODE_SENSE_6_HEADER *)(data + 1) - 1;
			unsigned data_len = s6->data_length + 1;
			unsigned should_transfer = data_len < len - 4 ? data_len : len - 4;
			if (__unlikely(cmd.status < should_transfer)) {
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "MODE SENSE(6)/PAGE %02X RETURNED NOT ENOUGH DATA (TRANSFERED %d, DESCRIPTOR %d, ALLOCATED %d)", modepage, cmd.status, data_len, len - 4);
				if (cmd.status < sizeof(SCSI_MODE_SENSE_6_HEADER)) {
					__slow_free(data);
					return __ERR_PTR(-EINVAL);
				}
				s6->data_length = cmd.status + 1;
			}
			data->data_length = __16CPU2BE(s6->data_length + 3);
			data->medium_type = s6->medium_type;
			data->device_specific = s6->device_specific;
			__barrier();
			data->reserved1 = 0;
			data->reserved2 = 0;
			data->block_descriptor_length = __16CPU2BE(s6->block_descriptor_length);
			return data;
		}
		key = SCSI$GET_CMD_SENSE((SCSIRQ *)&cmd);
	} else {
		free(data);
		return __ERR_PTR(-EOPNOTSUPP);
	}
	free(data);
	r = SCSI$CMD_ERROR_CODE(scsi_version, (SCSIRQ *)&cmd);
	if (SCSI_SENSE_UNKNOWN(key) ? scsi_version >= 2 || cmd.status < 0 || cmd.scsi_status != SCSI_CHECK_CONDITION : SCSI_SENSE_KEY(key) != SCSI_KEY_ILLEGAL_REQUEST)
		SCSI$LOG_ERROR(dev_name, errorlevel, inq, scsi_version, cmd.cmd[0] == SCMD_MODE_SENSE_10 ? "MODE SENSE(10)" : "MODE SENSE(6)", (SCSIRQ *)&cmd, r);
	return __ERR_PTR(r);
}

SCSI_MODE_SENSE_HEADER *SCSI$MODE_SENSE_PAGE(__const__ char *dev_name, int errorlevel, __const__ SCSI_INQUIRY_DATA *inq, __u8 scsi_version, SCSI_ATTACH_PARAM *ap, __u8 lun_byte, int modepage, unsigned *len, jiffies_lo_t timeout, int retries)
{
	SCSI_MODE_SENSE_HEADER *h;
	h = SCSI$MODE_SENSE(dev_name, errorlevel, inq, scsi_version, ap, lun_byte, modepage, 0, timeout, retries);
	if (__unlikely(__IS_ERR(h))) return h;
	*len = __16BE2CPU(h->data_length);
	free(h);
	if (__unlikely(*len <= 6)) return __ERR_PTR(-ENODATA);
	*len -= 6;
	h = SCSI$MODE_SENSE(dev_name, errorlevel, inq, scsi_version, ap, lun_byte, modepage, *len, timeout, retries);
	if (__unlikely(__IS_ERR(h))) return h;
	return h;
}

