#include <SPAD/LIBC.H>
#include <SPAD/SYSLOG.H>
#include <STRING.H>
#include <ARCH/BSF.H>
#include <SPAD/AC.H>
#include <SPAD/DL.H>

#include <SPAD/SCSI.H>

static __const__ struct sense_desc {
	__u8 asc;
	__u8 ascq_from;
	__u8 ascq_to;
	__const__ char *msg;
} senses[] = {

#include "SCSISENS.I"

};

#define N_SENSES	(sizeof(senses) / sizeof(*senses))

__const__ char *SCSI$SENSE_ASC_STRING(__u8 asc, __u8 ascq)
{
	static char string[__MAX_STR_LEN];
	int from = 0;
	int to = N_SENSES - 1;
	while (from <= to) {
		int middle = (from + to) >> 1;
		__const__ struct sense_desc *s = &senses[middle];
		if (s->asc < asc) {
			p2:
			from = middle + 1;
			continue;
		}
		if (s->asc > asc) {
			p1:
			to = middle - 1;
			continue;
		}
		if (s->ascq_to < ascq) {
			goto p2;
		}
		if (s->ascq_from > ascq) {
			goto p1;
		}
		if (__unlikely(s->ascq_from == s->ascq_to)) return s->msg;
		_snprintf(string, sizeof string, "%s%02X", s->msg, (unsigned)ascq);
		return string;
	}
	_snprintf(string, sizeof string, "ADDITIONAL CODE %02X, QUALIFIER %02X", (unsigned)asc, (unsigned)ascq);
	return string;
}

static __const__ char *__const__ sense_keys[16] = {
	"NO SENSE",
	"RECOVERED ERROR",
	"NOT READY",
	"MEDIUM ERROR",
	"HARDWARE ERROR",
	"ILLEGAL REQUEST",
	"UNIT ATTENTION",
	"DATA PROTECT",
	"BLANK CHECK",
	"VENDOR SPECIFIC",
	"COPY ABORTED",
	"ABORTED COMMAND",
	"EQUAL",
	"VOLUME OVERFLOW",
	"MISCOMPARE",
	"RESERVED SENSE KEY 15",
};

__u32 SCSI$GET_SENSE(__const__ SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	if (__unlikely(sense_size <= __offsetof(SCSI_SENSE_DATA, flags))) return -1;
	if (__likely((sense->response_code & (SENSE_RESPONSE_CODE & ~SENSE_RESPONSE_CODE_DEFERRED_BIT)) == SENSE_RESPONSE_CODE_CURRENT)) {
		__u32 ret = (sense->response_code & SENSE_RESPONSE_CODE_DEFERRED_BIT) << 4 | ((sense->flags & SENSE_FLAGS_SENSE_KEY) >> __BSF_CONST(SENSE_FLAGS_SENSE_KEY));
		if (__likely(sense_size > __offsetof(SCSI_SENSE_DATA, ascq)) && __likely(sense->additional_length > __offsetof(SCSI_SENSE_DATA, ascq) - __offsetof(SCSI_SENSE_DATA, additional_length) + 1)) return ret | (__16LE2CPU(*(__u16 *)&sense->asc) << 16);
		return ret;
	} else {
		return 0xffffffff;
	}
}

__u32 SCSI$GET_CMD_SENSE(__const__ SCSIRQ *rq)
{
	if (__unlikely(rq->status < 0) || __unlikely(rq->scsi_status != SCSI_CHECK_CONDITION)) return 0xffffffff;
	return SCSI$GET_SENSE((SCSI_SENSE_DATA *)rq->sense, rq->sense_size);
}

__const__ char *SCSI$SENSE_STRING(__u8 scsi_version, __const__ SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	static char string[__MAX_STR_LEN];
	if (__unlikely(sense_size < SCSI_MIN_REQUEST_SENSE_SIZE)) return "TOO SMALL SENSE SIZE";
	if (__likely((sense->response_code & (SENSE_RESPONSE_CODE & ~SENSE_RESPONSE_CODE_DEFERRED_BIT)) == SENSE_RESPONSE_CODE_CURRENT)) {
		__u8 key = (sense->flags & SENSE_FLAGS_SENSE_KEY) >> __BSF_CONST(SENSE_FLAGS_SENSE_KEY);
		if (__unlikely(scsi_version == 1) || __unlikely(sense_size <= __offsetof(SCSI_SENSE_DATA, ascq)) || __unlikely(sense->additional_length <= __offsetof(SCSI_SENSE_DATA, ascq) - __offsetof(SCSI_SENSE_DATA, additional_length) + 1) || __unlikely(!(sense->asc | sense->ascq))) return sense_keys[key];
		_snprintf(string, sizeof string, "%s: %s", sense_keys[key], SCSI$SENSE_ASC_STRING(sense->asc, sense->ascq));
		return string;
	}
	if (__unlikely(scsi_version >= 2)) {
		_snprintf(string, sizeof string, "INVALID SENSE CODE %02X", sense->response_code);
		return string;
	}
#define sense1	((SCSI1_SENSE_DATA *)sense)
	if (sense1->flags & SENSE1_FLAGS_ADVALID)
		_snprintf(string, sizeof string, "ERROR CLASS %02X, CODE %02X AT LBA %08X", (sense->flags & SENSE1_FLAGS_ERROR_CLASS) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CLASS), (sense->flags & SENSE1_FLAGS_ERROR_CODE) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CODE), ((sense1->lba3 & SENSE1_LBA3_LBA3) << 16) | (sense1->lba2 << 8) | sense1->lba1);
	else
		_snprintf(string, sizeof string, "ERROR CLASS %02X, CODE %02X", (sense->flags & SENSE1_FLAGS_ERROR_CLASS) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CLASS), (sense->flags & SENSE1_FLAGS_ERROR_CODE) >> __BSF_CONST(SENSE1_FLAGS_ERROR_CODE));
#undef sense1
	return string;
}

__const__ char *SCSI$STATUS_STRING(__u8 status)
{
	static char string[__MAX_STR_LEN];
	switch (status) {
		case SCSI_GOOD:
			return "GOOD";
		case SCSI_CHECK_CONDITION:
			return "CHECK CONDITION";
		case SCSI_CONDITION_MET:
			return "CONDITION MET";
		case SCSI_BUSY:
			return "BUSY";
		case SCSI_INTERMEDIATE:
			return "INTERMEDIATE";
		case SCSI_INTERMEDIATE_CONDITION_MET:
			return "INTERMEDIATE - CONDITION MET";
		case SCSI_RESERVATION_CONFLICT:
			return "RESERVATION CONFLICT";
		case SCSI_COMMAND_TERMINATED:
			return "COMMAND TERMINATED";
		case SCSI_QUEUE_FULL:
			return "QUEUE FULL";
		case SCSI_ACA_ACTIVE:
			return "ACA ACTIVE";
		case SCSI_TASK_ABORTED:
			return "TASK ABORTED";
		default:
			_snprintf(string, sizeof string, "UNKNOWN STATUS %02X", status);
			return string;
	}
}

int SCSI$RECOMMENDED_ACTION(SCSIRQ *srq)
{
	if (__unlikely(srq->status < 0)) return SCSI_ACTION_RETRY_FEW_TIMES;
	if (__likely(srq->scsi_status == SCSI_CHECK_CONDITION)) {
		__u32 key = SCSI$GET_SENSE((SCSI_SENSE_DATA *)srq->sense, srq->sense_size);
		if (__unlikely(SCSI_SENSE_UNKNOWN(key))) return SCSI_ACTION_RETRY_FEW_TIMES;
		if (__unlikely(SCSI_SENSE_DEFERRED(key))) return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		switch (SCSI_SENSE_KEY(key)) {
			case SCSI_KEY_NO_SENSE:
				return SCSI_ACTION_OK;
			case SCSI_KEY_RECOVERED_ERROR:
				return SCSI_ACTION_OK;
			case SCSI_KEY_NOT_READY:
				if (SCSI_SENSE_ASC(key) == 0x3A) return SCSI_ACTION_FAIL;
				if (SCSI_SENSE_ASC(key) == 0x04) {
					switch (SCSI_SENSE_ASCQ(key)) {
						case 0x01:
							return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
						case 0x02:
						case 0x03:
							return SCSI_ACTION_FAIL;
						case 0x04:
						case 0x05:
						case 0x06:
						case 0x07:
						case 0x08:
						case 0x09:
							return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
						case 0x11:
						case 0x12:
							return SCSI_ACTION_FAIL;
					}
				}
				if (SCSI_SENSE_ASC(key) == 0x06) return SCSI_ACTION_RETRY_FEW_TIMES;
				return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
			case SCSI_KEY_MEDIUM_ERROR:
				if (SCSI_SENSE_ASC(key) == 0x3A) return SCSI_ACTION_FAIL;
				return SCSI_ACTION_RETRY_FEW_TIMES;
			case SCSI_KEY_HARDWARE_ERROR:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_ILLEGAL_REQUEST:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_UNIT_ATTENTION:
				if (SCSI_SENSE_ASC(key) == 0x3A) return SCSI_ACTION_FAIL;
				return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
			case SCSI_KEY_DATA_PROTECT:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_BLANK_CHECK:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_VENDOR_SPECIFIC:
				return SCSI_ACTION_RETRY_FEW_TIMES;
			case SCSI_KEY_COPY_ABORTED:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_ABORTED_COMMAND:
				return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
			case SCSI_KEY_EQUAL:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_VOLUME_OVERFLOW:
				return SCSI_ACTION_FAIL;
			case SCSI_KEY_MISCOMPARE:
				return SCSI_ACTION_FAIL;
			default:
				return SCSI_ACTION_RETRY_FEW_TIMES;
		}
	}
	switch (srq->scsi_status) {
		case SCSI_GOOD:
			return SCSI_ACTION_OK;
		case SCSI_CONDITION_MET:
			return SCSI_ACTION_OK;
		case SCSI_BUSY:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_INTERMEDIATE:
			return SCSI_ACTION_FAIL;
		case SCSI_INTERMEDIATE_CONDITION_MET:
			return SCSI_ACTION_FAIL;
		case SCSI_RESERVATION_CONFLICT:
			return SCSI_ACTION_FAIL;
		case SCSI_COMMAND_TERMINATED:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_QUEUE_FULL:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_ACA_ACTIVE:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		case SCSI_TASK_ABORTED:
			return SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		default:
			return SCSI_ACTION_RETRY_FEW_TIMES;
	}
}

int SCSI$SENSE_ERROR_CODE(__u8 scsi_version, __const__ SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	__u32 key = SCSI$GET_SENSE(sense, sense_size);
	if (__unlikely(SCSI_SENSE_UNKNOWN(key))) {
		if (__likely(scsi_version == 1)) return -EIO;
		return -EPROTO;
	}
	switch (SCSI_SENSE_ASC(key)) {
		case 0x27:
			return -EWP;
		case 0x3A:
			return -ENOMEDIA;
	}
	switch (SCSI_SENSE_KEY(key)) {
		case SCSI_KEY_NO_SENSE:
			return -EPROTO;
		case SCSI_KEY_RECOVERED_ERROR:
			return -EIO;
		case SCSI_KEY_NOT_READY:
			return -EIO;
		case SCSI_KEY_MEDIUM_ERROR:
			return -EIO;
		case SCSI_KEY_HARDWARE_ERROR:
			return -EIO;
		case SCSI_KEY_ILLEGAL_REQUEST:
			return -EOPNOTSUPP;
		case SCSI_KEY_UNIT_ATTENTION:
			return -ENOMEDIA;
		case SCSI_KEY_DATA_PROTECT:
			return -EWP;
		case SCSI_KEY_BLANK_CHECK:
			return -ENOMEDIA;
		case SCSI_KEY_VENDOR_SPECIFIC:
			return -EIO;
		case SCSI_KEY_COPY_ABORTED:
			return -ECONNABORTED;
		case SCSI_KEY_ABORTED_COMMAND:
			return -ECONNABORTED;
		case SCSI_KEY_EQUAL:
			return -EPROTO;
		case SCSI_KEY_VOLUME_OVERFLOW:
			return -ENOSPC;
		case SCSI_KEY_MISCOMPARE:
			return -EPROTO;
		default:
			return -EPROTO;
	}
}

int SCSI$STATUS_ERROR_CODE(__u8 scsi_status)
{
	switch (scsi_status) {
		case SCSI_GOOD:
			return -EPROTO;
		case SCSI_CHECK_CONDITION:
			return -EIO;
		case SCSI_CONDITION_MET:
			return -EPROTO;
		case SCSI_BUSY:
			return -EBUSY;
		case SCSI_INTERMEDIATE:
			return -EPROTO;
		case SCSI_INTERMEDIATE_CONDITION_MET:
			return -EPROTO;
		case SCSI_RESERVATION_CONFLICT:
			return -EACCES;
		case SCSI_COMMAND_TERMINATED:
			return -ECONNABORTED;
		case SCSI_QUEUE_FULL:
			return -ENOSR;
		case SCSI_ACA_ACTIVE:
			return -EPERM;
		case SCSI_TASK_ABORTED:
			return -ECONNABORTED;
		default:
			return -EPROTO;
	}
}

int SCSI$CMD_ERROR_CODE(__u8 scsi_version, __const__ SCSIRQ *srq)
{
	if (srq->status < 0) return srq->status;
	if (srq->scsi_status == SCSI_CHECK_CONDITION) return SCSI$SENSE_ERROR_CODE(scsi_version, (SCSI_SENSE_DATA *)srq->sense, srq->sense_size);
	return SCSI$STATUS_ERROR_CODE(srq->scsi_status);
}

__sec_t SCSI$SENSE_LBA(__u8 scsi_version, __const__ SCSI_SENSE_DATA *sense, unsigned sense_size)
{
	if (__unlikely(sense_size < SCSI_MIN_REQUEST_SENSE_SIZE)) return -1;
	if (__likely((sense->response_code & (SENSE_RESPONSE_CODE & ~SENSE_RESPONSE_CODE_DEFERRED_BIT)) == SENSE_RESPONSE_CODE_CURRENT)) {
		if (__unlikely(sense_size < __offsetof(SCSI_SENSE_DATA, information[3]))) return -1;
		if (sense->response_code & SENSE_INFORMATION_VALID) return (sense->information[0] << 3) | (sense->information[1] << 2) | (sense->information[2] << 1) | sense->information[3];
	} else {
#define sense1	((SCSI1_SENSE_DATA *)sense)
		if (sense1->flags & SENSE1_FLAGS_ADVALID) return ((sense1->lba3 & SENSE1_LBA3_LBA3) << 16) | (sense1->lba2 << 8) | sense1->lba1;
	}
#undef sense1
	return -1;
}

__sec_t SCSI$CMD_LBA(__u8 scsi_version, __const__ SCSIRQ *srq)
{
	if (__unlikely(srq->status < 0) || __unlikely(srq->scsi_status != SCSI_CHECK_CONDITION)) return -1;
	return SCSI$SENSE_LBA(scsi_version, (SCSI_SENSE_DATA *)srq->sense, srq->sense_size);
}

void SCSI$LOG_ERROR(__const__ char *dev_name, int errorlevel, __const__ SCSI_INQUIRY_DATA *inq, __u8 scsi_version, __const__ char *cmdstring, __const__ SCSIRQ *rq, int error)
{
	int level, clas;
	__u32 key;
	if ((__likely(error == -ENOMEDIA) || __unlikely(error == -EWP)) && __likely(inq->rmb & INQ_RMB_RMB)) return;
	if (__unlikely(error == -EINTR) || __unlikely(error == -ECONNRESET)) return;
	if (__unlikely(rq->status < 0)) {
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "%s: SCSI TRANSPORT ERROR: %s", cmdstring, strerror(-rq->status));
		return;
	}
	if (__unlikely(rq->scsi_status != SCSI_CHECK_CONDITION)) {
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, dev_name, "%s: SCSI STATUS: %s", cmdstring, SCSI$STATUS_STRING(rq->scsi_status));
		return;
	}
	key = SCSI$GET_SENSE((SCSI_SENSE_DATA *)rq->sense, rq->sense_size);
	if (SCSI_SENSE_UNKNOWN(key)) {
		level = 1, clas = __SYSLOG_HW_ERROR;
	} else switch (SCSI_SENSE_KEY(key)) {
		case SCSI_KEY_NO_SENSE:
			level = 2, clas = __SYSLOG_HW_ERROR;
		case SCSI_KEY_RECOVERED_ERROR:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_NOT_READY:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_MEDIUM_ERROR:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_HARDWARE_ERROR:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_ILLEGAL_REQUEST:
/* A lot of CDROMs send Illegal Request when reading from non-data parts of
   a CD. So make it a level 2 error. */
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_UNIT_ATTENTION:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_DATA_PROTECT:
			level = 2, clas = __SYSLOG_HW_INCAPABILITY;
			break;
		case SCSI_KEY_BLANK_CHECK:
			level = 2, clas = __SYSLOG_HW_INCAPABILITY;
			break;
		case SCSI_KEY_VENDOR_SPECIFIC:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_COPY_ABORTED:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_ABORTED_COMMAND:
			level = 1, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_EQUAL:
			level = 1, clas = __SYSLOG_HW_BUG;
			break;
		case SCSI_KEY_VOLUME_OVERFLOW:
			level = 2, clas = __SYSLOG_HW_ERROR;
			break;
		case SCSI_KEY_MISCOMPARE:
			level = 1, clas = __SYSLOG_HW_BUG;
			break;
		default:
			level = 1, clas = __SYSLOG_HW_BUG;
			break;
	}
	if (errorlevel >= level)
		KERNEL$SYSLOG(clas, dev_name, "%s: %s%s", cmdstring, SCSI_SENSE_DEFERRED(key) ? "DEFERRED ERROR: " : "", SCSI$SENSE_STRING(scsi_version, (SCSI_SENSE_DATA *)rq->sense, rq->sense_size));
}

void SCSI$LOG_RECOVERED_ERROR(__const__ char *dev_name, int errorlevel, __const__ char *cmdstring, __const__ SCSIRQ *rq)
{
	__u32 key;
	if (__unlikely(rq->status < 0) || __likely(rq->scsi_status != SCSI_CHECK_CONDITION)) return;
	key = SCSI$GET_SENSE((SCSI_SENSE_DATA *)rq->sense, rq->sense_size);
	if (__unlikely(SCSI_SENSE_UNKNOWN(key))) return;
	if (key == SCSI_KEY_RECOVERED_ERROR) {
		if (errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_HW_WARNING, dev_name, "%s: %s%s", cmdstring, SCSI_SENSE_DEFERRED(key) ? "DEFERRED ERROR: " : "", SCSI$SENSE_STRING(5, (SCSI_SENSE_DATA *)rq->sense, rq->sense_size));
	}
}

DECL_IOCALL(DLL$LOAD, SPL_DEV, DLINITRQ)
{
	unsigned i;
	for (i = 0; i < N_SENSES; i++) {
		if (__unlikely(senses[i].ascq_from > senses[i].ascq_to)) KERNEL$SUICIDE("DLL$LOAD: SCSI SENSE TABLE ASCQ RANGE WRONG AT LINE %u", i);
		if (__unlikely(!i)) continue;
		if (__unlikely(senses[i - 1].asc > senses[i].asc)) KERNEL$SUICIDE("DLL$LOAD: SCSI SENSE TABLE ASC ORDER VIOLATION AT LINE %u", i);
		if (senses[i - 1].asc < senses[i].asc) continue;
		if (__unlikely(senses[i - 1].ascq_to >= senses[i].ascq_from)) KERNEL$SUICIDE("DLL$LOAD: SCSI SENSE TABLE ASCQ ORDER VIOLATION AT LINE %u", i);
	}
	RQ->status = 0;
	RETURN_AST(RQ);
}

