#include <SPAD/LIBC.H>
#include <STRING.H>
#include <VALUES.H>
#include <SPAD/SLAB.H>
#include <SPAD/AC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/BIO.H>
#include <SPAD/BIO_KRNL.H>

#include <SPAD/SCSI.H>
#include <SPAD/SCSISTDS.H>

#include "SD.H"

__const__ HANDLE_OPERATIONS SD_OPERATIONS = {
	SPL_X(SPL_ATA_SCSI),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	BIO$LOOKUP_PARTITION,	/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	BIO$INSTANTIATE_PARTITION,	/* instantiate */
	NULL,			/* leave */
	BIO$DETACH,		/* detach */
	NULL,			/* open */
	NULL,			/* close */
	BIO$READ,		/* READ */
	BIO$WRITE,		/* WRITE */
	BIO$AREAD,		/* AREAD */
	BIO$AWRITE,		/* AWRITE */
	BIO$IOCTL,		/* IOCTL */
	SD_REQUEST,		/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static void SD_INIT_ROOT(HANDLE *h, void *p)
{
	SD *sd = p;
	BIO$ROOT_PARTITION(h, sd->p);
	h->flags = 0;
	h->flags2 = 0;
	h->op = &SD_OPERATIONS;
}

static int SD_TRY_CONTROLLER(SD *sd, char *param);
static int SD_UNLOAD(void *p, void **release, char *argv[]);

static int SD_DUMMY_DEQUEUE(SCSI_ATTACH_PARAM *ap)
{
	return 0;
}

static BIORQ *SD_DUMMY_PROBE_QUEUE(SCSI_ATTACH_PARAM *ap)
{
	return NULL;
}

int main(int argc, char *argv[])
{
	int r, rr;
	int i;
	char **arg;
	int state;
	char *ctrl, *param;
	int errorlevel = -1;
	int errorlevel_set = 0;
	SD *sd;
	union {
		MALLOC_REQUEST mrq;
		LOGICAL_LIST_REQUEST llr;
		DEVICE_REQUEST devrq;
	} u;
	char *opt, *optend, *val;
	struct __param_table params[] = {
		"CTRL", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"LOG_ALL_ERRORS", __PARAM_BOOL, ~0, 2, NULL,
		"LOG_HW_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_NO_ERRORS", __PARAM_BOOL, ~0, 0, NULL,
		"TIMEOUT", __PARAM_INT, 1, MAXINT / JIFFIES_PER_SECOND, NULL,
		"PROBE_TIMEOUT", __PARAM_INT, 1, MAXINT / JIFFIES_PER_SECOND, NULL,
		NULL, 0, 0, 0, NULL,
	};
	u.mrq.size = sizeof(SD);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto err0;
	}
	sd = u.mrq.ptr;
	memset(sd, 0, sizeof(SD));

	ctrl = NULL;
	param = NULL;

	u.mrq.size = SD_MAX_SECSIZE;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto err1;
	}
	sd->pad_ptr = u.mrq.ptr;

	u.mrq.size = SD_MAX_SECSIZE;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(u.mrq.status < 0)) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto err1;
	}
	sd->write_buffer = u.mrq.ptr;

	sd->media_thread.thread = KERNEL$ALLOC_THREAD_SYNC();
	if (__unlikely(__IS_ERR(sd->media_thread.thread))) {
		r = __PTR_ERR(sd->media_thread.thread);
		sd->media_thread.thread = NULL;
		goto err1;
	}

	params[0].__p = &ctrl;
	params[1].__p = &errorlevel;
	params[2].__p = &errorlevel;
	params[3].__p = &errorlevel;
	params[4].__p = &sd->timeout;
	params[5].__p = &sd->probe_timeout;

	arg = argv;
	state = 0;
	while (__unlikely(__parse_params(&arg, &state, params, &opt, &optend, &val))) {
		if (opt) {
			r = __accumulate_params(&param, opt, optend, val, val ? strchr(val, 0) : NULL);
			if (__unlikely(r)) {
				if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", strerror(-r));
				goto err1;
			}
			continue;
		}
		bads:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: SYNTAX ERROR");
		r = -EBADSYN;
		goto err1;
	}
	if (errorlevel == -1) errorlevel = 2;
	else errorlevel_set = 1;

	sd->timeout *= JIFFIES_PER_SECOND;
	sd->probe_timeout *= JIFFIES_PER_SECOND;

	sd->errorlevel = errorlevel;
	sd->errorlevel_set = errorlevel_set;
	INIT_LIST(&sd->waiting_list);
	INIT_XLIST(&sd->in_progress);
	INIT_TIMER(&sd->timer);
	sd->timer.fn = SD_TIMER_FN;
	WQ_INIT(&sd->media_wq, "SD$MEDIA_WQ");

	if (__unlikely(r = BIOQUE$ALLOC_QUEUE(&sd->q))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: CAN'T ALLOC QUEUE: %s", strerror(-r));
		goto err1;
	}

	KERNEL$ERROR_MSG()[0] = 0;
	if (ctrl) {
		strcpy(sd->ctrl, ctrl);
		if (__unlikely(__check_logical_name(sd->ctrl, 1))) goto bads;
		r = SD_TRY_CONTROLLER(sd, param);
		if (__unlikely(r)) {
			goto nodev;
		}
	} else {
		u.llr.prefix = "";
		SYNC_IO_CANCELABLE(&u.llr, KERNEL$LIST_LOGICALS);
		if (__unlikely(u.llr.status < 0)) {
			if (u.llr.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: CAN'T LIST LOGICAL NAMES: %s", strerror(-u.llr.status));
			r = u.llr.status;
			goto err2;
		}
		for (i = 0; i < u.llr.n_entries; i++) {
			char *n = u.llr.entries[i]->name;
			if (_memcasecmp(n, "ATA$", 4) && _memcasecmp(n, "SCSI$", 5)) continue;
			strlcpy(sd->ctrl, n, __MAX_STR_LEN);
			/* __debug_printf("try: %s\n", sd->ctrl); */
			r = SD_TRY_CONTROLLER(sd, param);
			if (!r) {
				KERNEL$FREE_LOGICAL_LIST(&u.llr);
				goto got_it;
			}
		}
		KERNEL$FREE_LOGICAL_LIST(&u.llr);
		r = -ENODEV;
		nodev:
		if (!KERNEL$ERROR_MSG()[0]) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: NO DEVICE FOUND");
		goto err2;
	}

	got_it:

	SCSI$GET_MODEL(&sd->inq, &sd->vendor, &sd->product, &sd->revision);

	if (sd->scsi_version >= 3) sd->lun_byte = 0;
		/* the following line is confusing ... RBC specification says
		   that the device should support only MODE_SENSE_6 and not need
		   support MODE_SENSE_10.
		   However Windows do exactly the opposite: when the device has
		   something other than transparent SCSI, it uses only
		   MODE_SENSE_10. In that case USBMSD driver have already
		   disabled SCSI_FLAG_CAN_USE_MODE_6.
		*/
	if (sd->std == STD_RBC && __likely(sd->ap.scsi_flags & SCSI_FLAG_CAN_USE_MODE_6)) sd->ap.scsi_flags &= ~SCSI_FLAG_CAN_USE_MODE_10;
	if (__unlikely(sd->scsi_version == 1) && __likely(sd->ap.scsi_flags & SCSI_FLAG_CAN_USE_MODE_6)) sd->ap.scsi_flags &= ~SCSI_FLAG_CAN_USE_MODE_10;

	if (SD_REMOVABLE_MEDIA(sd)) {
		if (!errorlevel_set) sd->errorlevel = errorlevel = 1;
		sd->flags |= FLAGS_NO_MEDIA;
		if (STD_IS_DISK(sd->std)) {
/* USB sticks do not really have removable media, but they report so.
   So try to probe media now, so that their size can be reported.
   Don't do it for CD-ROMs, it is slow.
*/
			if (!SD_MEDIA_THREAD(sd)) sd->flags &= ~FLAGS_NO_MEDIA;
		}
	} else {
		r = SD_MEDIA_THREAD(sd);
		if (__unlikely(r)) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO ACCESS MEDIA: %s", sd->dev_name, strerror(-r));
			goto err3;
		}
		sd->flags &= ~FLAGS_NO_MEDIA;
	}

	if (__unlikely(r = BIO$ALLOC_PARTITIONS(&sd->p, sd, sd->flags & FLAGS_NO_MEDIA ? (__sec_t)-1 : sd->capacity, SD_IOCTL, sd->dev_name))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOC PARTITIONS: %s", sd->dev_name, strerror(-r));
		goto err3;
	}
	KERNEL$SLAB_INIT(&sd->tag_slab, sizeof(SD_TAG), sizeof(__sec_t), VM_TYPE_WIRED_MAPPED, SD_TAG_CTOR, sd, NULL, "SD$SD_TAG");

	if (__unlikely(r = KERNEL$SLAB_RESERVE(&sd->tag_slab, 2))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: %s", sd->dev_name, strerror(-r));
		goto err5;
	}
	sd->dummy_tag = __slalloc(&sd->tag_slab);
	if (__unlikely(!sd->dummy_tag))
		KERNEL$SUICIDE("SD: CAN'T ALLOCATE DUMMY TAG");

	if (__unlikely(!sd->ap.max_tags))
		KERNEL$SUICIDE("SD: ZERO FREE TAGS REPORTED BY SCSI HOST CONTROLLER DRIVER");
	if (__unlikely(sd->scsi_version == 1) || __unlikely(sd->std == STD_SCSI_1) || sd->std == STD_RBC) sd->ap.max_tags = 1;
	if ((sd->scsi_version == 0 || sd->scsi_version == 2 || sd->scsi_version == 3) && !(sd->inq.flags3 & INQ_FLAGS3_CMDQUE)) sd->ap.max_tags = 1;
	if (__unlikely(sd->scsi_version >= 4) && !(sd->inq.flags3 & INQ_FLAGS3_CMDQUE) && __unlikely(!(sd->inq.flags2 & INQ_FLAGS2_BQUE))) sd->ap.max_tags = 1;

	sd->free_tags = sd->ap.max_tags;
	
	_printf("%s: SCSI %s%s%s%s%s%s%s\n", sd->dev_name, sd->std < 0 ? "CD-ROM" : SD_REMOVABLE_MEDIA(sd) ? "REMOVABLE DISK" : "DISK", *sd->vendor || *sd->product || *sd->revision ? " " : "", sd->vendor, *sd->vendor && *sd->product ? "," : "", sd->product, *sd->revision && (*sd->vendor || *sd->product) ? "," : "", sd->revision);
	_printf("%s: ", sd->dev_name);
	/*{
		int i;
		for (i = 0; sd->vendor[i]; i++) __debug_printf("%d: %02x\n", i, sd->vendor[i]);
		for (i = 0; sd->product[i]; i++) __debug_printf("%d: %02x\n", i, sd->product[i]);
		for (i = 0; sd->revision[i]; i++) __debug_printf("%d: %02x\n", i, sd->revision[i]);
	}*/
	if (sd->std == -1) {
		_printf("MMC SCSI 2");
	} else if (sd->std <= -2) {
		_printf("MMC %d", STD_MMC_1 - sd->std + 1);
	} else if (sd->std <= STD_SCSI_2) {
		_printf("SCSI %d", sd->std - STD_SCSI_1 + 1);
	} else if (sd->std == STD_RBC) {
		_printf("RBC");
	} else {
		_printf("SBC %d", sd->std - STD_SBC_1 + 1);
	}
	if (!(sd->flags & FLAGS_NO_MEDIA)) {
		_printf(", ");
		if (sd->capacity < 2097152)
			_printf("%ld MiB", (long)(sd->capacity >> 11));
		else if (sd->capacity < (__u64)2097152 * 1024)
			_printf("%ld.%ld GiB", (long)(sd->capacity >> 21), ((long)sd->capacity & 2097151) / 209716);
		else
			_printf("%ld.%ld TiB", (long)(sd->capacity >> 31), (long)((sd->capacity >> 10) & 2097151) / 209716);
	}
	_printf("\n");

	sd->ap.dequeue = SD_DEQUEUE;
	sd->ap.probe_queue = SD_PROBE_QUEUE;

	u.devrq.name = sd->dev_name;
	u.devrq.driver_name = "SD.SYS";
	u.devrq.flags = 0;
	u.devrq.init_root_handle = SD_INIT_ROOT;
	u.devrq.dev_ptr = sd;
	u.devrq.dcall = NULL;
	u.devrq.dcall_type = NULL;
	u.devrq.dctl = NULL;
	u.devrq.unload = SD_UNLOAD;
	SYNC_IO_CANCELABLE(&u.devrq, KERNEL$REGISTER_DEVICE);
	if (__unlikely(u.devrq.status < 0)) {
		if (__unlikely(u.devrq.status != -EINTR)) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE: %s", sd->dev_name, strerror(-u.devrq.status));
		r = u.devrq.status;
		goto err6;
	}
	sd->lnte = u.devrq.lnte;
	sd->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), sd->dev_name);
	if (param) KERNEL$UNIVERSAL_FREE(param);
	return 0;

	err6:
	__slow_slfree(sd->dummy_tag);
	err5:
	KERNEL$SLAB_DESTROY(&sd->tag_slab);
	BIO$FREE_PARTITIONS(sd->p);
	err3:
	if (__unlikely(rr = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_FREE, &sd->ap)))
		KERNEL$SUICIDE("SD: CAN'T UNREGISTER AT CONTROLLER %s, DEVICE %x: %s", sd->ctrl, sd->ap.ch_id_lun, strerror(-rr));
	err2:
	BIOQUE$FREE_QUEUE(sd->q);
	err1:
	if (param) KERNEL$UNIVERSAL_FREE(param);
	if (sd->write_buffer) KERNEL$UNIVERSAL_FREE(sd->write_buffer);
	if (sd->media_thread.thread) KERNEL$FREE_THREAD(sd->media_thread.thread);
	if (sd->pad_ptr) KERNEL$UNIVERSAL_FREE(sd->pad_ptr);
	KERNEL$UNIVERSAL_FREE(sd);
	err0:
	return r;
}

static int SD_TRY_CONTROLLER(SD *sd, char *param)
{
	int r;
	unsigned type;

	memset(&sd->ap, 0, sizeof sd->ap);

	try_another:
	sd->ap.version = SCSI_VERSION;
	sd->ap.probe_timeout = sd->probe_timeout;
	sd->ap.timeout = sd->timeout;
	sd->ap.param = param;
	sd->ap.dev_prefix = "BIO$SD";
	sd->ap.dev_name = &sd->dev_name;
	sd->ap.dequeue = SD_DUMMY_DEQUEUE;
	sd->ap.probe_queue = SD_DUMMY_PROBE_QUEUE;
	sd->ap.biosched = BIOQUE$IOSCHED(sd->q);

	if (__unlikely(r = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_GET, &sd->ap))) {
		if (sd->ap.msg) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: %s", sd->ap.msg);
		else if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SD: CAN'T ATTACH TO HOST CONTROLLER DRIVER: %s", strerror(-r));
		return r;
	}
	sd->lun_byte = sd->ap.lun < 8 ? sd->ap.lun << 5 : 0;

	if (SCSI$INQUIRY(sd->dev_name, !sd->errorlevel_set ? 1 : sd->errorlevel, &sd->ap, sd->lun_byte, &sd->inq, sizeof sd->inq, sd->probe_timeout)) {
		/*__debug_printf("not me\n");*/
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INQUIRY FAILED", sd->dev_name);
		goto not_for_me;
	}

	/*__debug_printf("ok\n");*/
	/*{
		int i;
		__debug_printf("inquired: %.8s, %.16s, %.4s", sd->inq.vendor, sd->inq.product, sd->inq.revision);
		for (i = 0; i < sizeof(SCSI_INQUIRY_DATA); i++) {
			if (!(i & 15)) __debug_printf("\n%2d: ", i);
			__debug_printf(" %02X", ((unsigned char *)&sd->inq)[i]);
		}
		__debug_printf("\n");
	}*/

	sd->scsi_version = SCSI$VERSION(&sd->inq);

	type = sd->inq.type & INQ_TYPE_TYPE;

	if (!sd->timeout) sd->timeout = TIMEOUT_JIFFIES(__unlikely(type == SCSI_TYPE_OPTICAL_DA) ? SD_OPTICAL_TIMEOUT : SD_TIMEOUT);
	
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_SBC_2, SCSI_STD_SBC_2_ANSI_INCITS_405_2005, SCSI_STD_SBC_2_T10_1417_D_revision_16, SCSI_STD_SBC_2_T10_1417_D_revision_5a, SCSI_STD_SBC_2_T10_1417_D_revision_15, 0)) {
		sd->std = STD_SBC_2;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_SBC, SCSI_STD_SBC_ANSI_INCITS_306_1998, SCSI_STD_SBC_T10_0996_D_revision_08c, 0)) {
		sd->std = STD_SBC_1;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_RBC, SCSI_STD_RBC_ANSI_INCITS_330_2000, SCSI_STD_RBC_T10_1240_D_revision_10a, 0)) {
		sd->std = STD_RBC;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_5, 0)) {
		sd->std = STD_MMC_5;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_4, SCSI_STD_MMC_4_ANSI_INCITS_401_2005, SCSI_STD_MMC_4_T10_1545_D_revision_5, SCSI_STD_MMC_4_T10_1545_D_revision_3, SCSI_STD_MMC_4_T10_1545_D_revision_3d, 0)) {
		sd->std = STD_MMC_4;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_3, SCSI_STD_MMC_3_ANSI_INCITS_360_2002, SCSI_STD_MMC_3_T10_1363_D_revision_10g, SCSI_STD_MMC_3_T10_1363_D_revision_9, 0)) {
		sd->std = STD_MMC_3;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC_2, SCSI_STD_MMC_2_ANSI_INCITS_333_2000, SCSI_STD_MMC_2_T10_1228_D_revision_11a, SCSI_STD_MMC_2_T10_1228_D_revision_11, 0)) {
		sd->std = STD_MMC_2;
		goto mine;
	}
	if (SCSI$SUPPORTS_STD(&sd->inq, SCSI_STD_MMC, SCSI_STD_MMC_ANSI_INCITS_304_1997, SCSI_STD_MMC_T10_1048_D_revision_10a, 0)) {
		sd->std = STD_MMC_1;
		goto mine;
	}

	if (__likely(type == SCSI_TYPE_DA) || __unlikely(type == SCSI_TYPE_WORM) || __unlikely(type == SCSI_TYPE_OPTICAL_DA)) {
		if ((sd->inq.version & 7) == 1) sd->std = STD_SCSI_1;
		else if ((sd->inq.version & 7) == 2) sd->std = STD_SCSI_2;
		else if ((sd->inq.version & 7) == 3) sd->std = STD_SBC_1;
		else if (sd->inq.version == 4) sd->std = STD_SBC_1;
		else if (sd->inq.version >= 5 && sd->inq.version <= 7) sd->std = type == SCSI_TYPE_DA ? STD_SBC_2 : STD_SBC_1;
		else sd->std = STD_SCSI_2;	/* guess ... :-/ */
		goto mine;
	}
	if (__likely(type == SCSI_TYPE_REDUCED_DA)) {
		sd->std = STD_RBC;
		goto mine;
	}
	if (__likely(type == SCSI_TYPE_CD_DVD)) {
		if ((sd->inq.version & 7) < 3) sd->std = STD_MMC_SCSI_2;
		else sd->std = STD_MMC_5;
		goto mine;
	}
	not_for_me:
	if (__unlikely(r = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_FREE, &sd->ap)))
		KERNEL$SUICIDE("SD: CAN'T UNREGISTER AT CONTROLLER %s, DEVICE %x: %s", sd->ctrl, sd->ap.ch_id_lun, strerror(-r));
	goto try_another;

	mine:
	if (!sd->probe_timeout && sd->ap.probe_timeout) sd->probe_timeout = sd->ap.probe_timeout;
	return 0;
}

static int SD_UNLOAD(void *p, void **release, char *argv[])
{
	int r, rr;
	SD *sd = p;
	if ((r = KERNEL$DEVICE_UNLOAD(sd->lnte, argv))) return r;
	RAISE_SPL(SPL_ATA_SCSI);
	__slow_slfree(sd->dummy_tag);
	while (__unlikely(!BIOQUE$QUEUE_EMPTY(sd->q)) || __unlikely(!KERNEL$SLAB_EMPTY(&sd->tag_slab)) || __unlikely(sd->flags & FLAGS_MEDIA_PROBING)) {
		if (sd->flags & FLAGS_MEDIA_PROBING) KERNEL$CIO((IORQ *)&sd->media_thread);
		KERNEL$SLEEP(1);
	}
	LOWER_SPL(SPL_ZERO);
	if (!(sd->flags & (FLAGS_NO_MEDIA | FLAGS_NO_SYNCHRONIZE_CACHE | FLAGS_WP))) {
		DECL_SCSIRQ(16) cmd;
		SCSI_SENSE_DATA sense;
		char *cmdstring;
		cmd.sense = (__u8 *)&sense;
		cmd.sense_size = SCSI_SIZEOF_SENSE_DATA;
		cmd.direction = 0;
		if (__likely(!(sd->flags & FLAGS_COMMAND_SIZE_16))) {
			cmd.cmdlen = 10;
			cmd.cmd[0] = SCMD_SYNCHRONIZE_CACHE_10;
			*(__u64 *)&cmd.cmd[2] = __64CPU2BE(0);
			cmdstring = "SYNCHRONIZE CACHE(10)";
		} else {
			cmd.cmdlen = 16;
			cmd.cmd[0] = SCMD_SYNCHRONIZE_CACHE_16;
			*(__u64 *)&cmd.cmd[2] = __64CPU2BE(0);
			*(__u32 *)&cmd.cmd[10] = __32CPU2BE(0);
			*(__u16 *)&cmd.cmd[14] = __16CPU2BE(0);
			cmdstring = "SYNCHRONIZE CACHE(16)";
		}
		SCSI$SYNC_RQ(sd->dev_name, sd->errorlevel, cmdstring, (SCSIRQ *)&cmd, &sd->ap, NULL, 0, sd->timeout, SD_RETRIES, SCSI$RECOMMENDED_ACTION);
		if (__unlikely(SCSI$RECOMMENDED_ACTION((SCSIRQ *)&cmd) != SCSI_ACTION_OK)) {
			__u32 key = SCSI$GET_CMD_SENSE((SCSIRQ *)&cmd);
			if (!(!SCSI_SENSE_UNKNOWN(key) && SCSI_SENSE_KEY(key) == SCSI_KEY_ILLEGAL_REQUEST && __likely(SCSI_SENSE_ASC(key) == 0x20) && SCSI_SENSE_ASCQ(key) == 0x00)) {
				SCSI$LOG_ERROR(sd->dev_name, sd->errorlevel, &sd->inq, sd->scsi_version, cmdstring, (SCSIRQ *)&cmd, SCSI$CMD_ERROR_CODE(sd->scsi_version, (SCSIRQ *)&cmd));
			}
		}
	}
	if (__unlikely(sd->free_tags != sd->ap.max_tags))
		KERNEL$SUICIDE("SD_UNLOAD: FREE TAGS COUNTER SKEW: VALUE %u != ORIGINAL %u", (unsigned)sd->free_tags, (unsigned)sd->ap.max_tags);
	KERNEL$SLAB_DESTROY(&sd->tag_slab);
	BIO$FREE_PARTITIONS(sd->p);
	BIOQUE$FREE_QUEUE(sd->q);
	if (__unlikely(rr = KERNEL$DCALL(sd->ctrl, "SCSI", SCSI_CMD_FREE, &sd->ap)))
		KERNEL$SUICIDE("SD_UNLOAD: CAN'T UNREGISTER AT CONTROLLER %s, DEVICE %x: %s", sd->ctrl, sd->ap.ch_id_lun, strerror(-rr));
	*release = sd->dlrq;
	if (__unlikely(!LIST_EMPTY(&sd->waiting_list)))
		KERNEL$SUICIDE("SD_UNLOAD: WAITING_LIST CORRUPTED");
	if (__unlikely(!XLIST_EMPTY(&sd->in_progress)))
		KERNEL$SUICIDE("SD_UNLOAD: IN_PROGRESS LIST CORRUPTED");
	WQ_WAKE_ALL(&sd->media_wq);
	if (sd->write_buffer) KERNEL$UNIVERSAL_FREE(sd->write_buffer);
	KERNEL$FREE_THREAD(sd->media_thread.thread);
	KERNEL$UNIVERSAL_FREE(sd->pad_ptr);
	KERNEL$UNIVERSAL_FREE(sd);
	return 0;
}
