#include <SPAD/DEV_KRNL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/BIO.H>
#include <SPAD/BIO_KRNL.H>

#include "SD.H"

static void SD_REQUEST_DONE(SCSIRQ *cmd);

void SD_TAG_CTOR(void *sd_, void *tag_)
{
	SD *sd = sd_;
	SD_TAG *tag = tag_;
	tag->sd = sd;
	tag->cmd.ch_id_lun = sd->ap.ch_id_lun;
	tag->cmd.adapter = sd->ap.adapter;
	tag->cmd.done = SD_REQUEST_DONE;
	tag->cmd.sense = (__u8 *)&tag->sense;
	tag->cmd.sense_size = SCSI_SIZEOF_SENSE_DATA;
}

#if __DEBUG >= 2
#define SD_CHECK_REQUEST	BIO$CHECK_REQUEST
#else
#define SD_CHECK_REQUEST(m, p)	do { } while (0)
#endif

static void SD_POST_ERROR(SD *sd, BIORQ *rq, int error, __sec_t *lba_p)
{
	__sec_t lba;
	if (__unlikely(error > 0)) KERNEL$SUICIDE("SD_POST_ERROR: ERROR %d", error);
	if (lba_p) lba = *lba_p;
	else lba = -1;
	if (error == -EIO) {
		BIO$ERROR(rq, lba);
	} else if (error == -EVSPACEFAULT) {
		BIO$FAULT(rq, lba);
	} else if (__likely(!rq->status)) {
		rq->status = error;
	}
	if (__unlikely(!rq->tmp3))
		KERNEL$SUICIDE("SD_POST_ERROR: REFERENCE COUNT UNDERFLOW (ERROR %d)", error);
	if (__likely(!--rq->tmp3)) {
		CALL_AST(rq);
	}
}

int SD_RETRY_REQUEST(SD_TAG *tag)
{
	int r;
	SD *sd = tag->sd;
	if (__unlikely(sd->flags & FLAGS_NO_MEDIA)) {
		SD_MEDIA_PROBE(sd);
		goto notready;
	}
	if (__unlikely(tag->brq->flags & (BIO_WRITE | BIO_FLUSH)) && __unlikely(sd->flags & FLAGS_WP)) {
		r = -EWP;
		goto post_r;
	}
	if (__likely(XLIST_EMPTY(&sd->in_progress))) {
		KERNEL$SET_TIMER(sd->timeout, &sd->timer);
	}
	ADD_TO_XLIST(&sd->in_progress, &tag->list);
	tag->cmd.internal = 0;
	r = sd->ap.post((SCSIRQ *)&tag->cmd);
	if (__unlikely(r)) {
		BIORQ *rq;
		DEL_FROM_LIST(&tag->list);
		if (__likely(XLIST_EMPTY(&sd->in_progress))) KERNEL$DEL_TIMER(&sd->timer);
		post_r:
		if (__likely(r == -EAGAIN)) {
			notready:
			sd->flags |= FLAGS_WAITING_LIST_USED;
			ADD_TO_LIST_END(&sd->waiting_list, &tag->list);
			return 1;
		}
		rq = TAG_RQ(tag);
		SD_POST_ERROR(sd, rq, r, &tag->lba);
		sd->free_tags++;
		__slow_slfree(tag);
		if (__unlikely(sd->free_tags == 1)) SD_DEQUEUE(&sd->ap);
		return r;
	}
	return 0;
}

static int SD_SPLIT_REQUEST(SD_TAG *tag)
{
	SD *sd = tag->sd;
	unsigned max_sectors = sd->max_sectors;
	unsigned sectors;
	unsigned fragments;
	BIODESC **descp;
#if __DEBUG >= 1
	if (__unlikely(tag->brq == &tag->pw_rq)) KERNEL$SUICIDE("SD_SPLIT_REQUEST: SPLITTING PARTIAL WRITE REQUEST");
#endif
	SD_UNPAD_READ_REQUEST_END(tag, tag->brq);
	again:
	sectors = 0;
	fragments = 0;
	descp = &tag->brq->desc;
	while (1) {
		BIODESC *desc = *descp;
#if __DEBUG >= 1
		if (__unlikely(desc == &tag->split_desc))
			KERNEL$SUICIDE("SD_SPLIT_REQUEST: SPLITTING ALREADY SPLIT REQUEST");
#endif
		sectors += desc->v.len >> BIO_SECTOR_SIZE_BITS;
		if (__unlikely(sectors > max_sectors)) {
			BIORQ *rq;
			unsigned nsec;
			int r;
			tag->split_desc.v.ptr = desc->v.ptr;
			tag->split_desc.v.len = desc->v.len - ((sectors - max_sectors) << BIO_SECTOR_SIZE_BITS);
			tag->split_desc.v.vspace = desc->v.vspace;
			tag->split_desc.next = NULL;
			*descp = &tag->split_desc;
			rq = tag->brq;
			nsec = rq->nsec;
			rq->nsec = max_sectors;
		/* must be before SD_POST_REQUEST, because SD_POST_REQUEST can
		   change tag->split_desc.v.len */
			desc->v.ptr += tag->split_desc.v.len;
			desc->v.len -= tag->split_desc.v.len;
			rq->tmp3++;
			if (__unlikely(r = SD_POST_REQUEST(tag, rq))) {
				rq->tmp3--;
				if (r == 1) {
					desc->v.ptr -= tag->split_desc.v.len;
					desc->v.len += tag->split_desc.v.len;
					*descp = desc;
					rq->nsec = nsec;
				} else if (!rq->tmp3) {
					CALL_AST(rq);
				}
				return r;
			}
			rq->sec += max_sectors;
			rq->nsec = nsec - max_sectors;
			rq->desc = desc;
			SD_CHECK_REQUEST("SD_SPLIT_REQUEST 1", rq);
			return 0;
		}
#if __DEBUG >= 1
		if (!desc->next)
			KERNEL$SUICIDE("SD_SPLIT_REQUEST: NOTHING TO SPLIT");
#endif
		if (__unlikely(++fragments > (unsigned)(sd->ap.max_fragments - 1)) ||
		    __unlikely(sectors == max_sectors)) {
			BIORQ *rq;
			unsigned nsec;
			int r;
			BIODESC *next;
			if (__unlikely(sd->flags & FLAGS_LARGER_SECSIZE)) {
				if (__unlikely(sectors & ((1 << sd->secsize_shift) - 1))) {
					max_sectors = sectors & ~(unsigned)((1 << sd->secsize_shift) - 1);
#if __DEBUG >= 1
					if (__unlikely(!max_sectors))
						KERNEL$SUICIDE("SD_SPLIT_REQUEST: NOTHING LEFT AFTER PADDING");
#endif
					goto again;
				}
			}
			next = desc->next;
			desc->next = NULL;
			rq = tag->brq;
			nsec = rq->nsec;
			rq->nsec = sectors;
			rq->tmp3++;
			if (__unlikely(r = SD_POST_REQUEST(tag, rq))) {
				rq->tmp3--;
				if (r == 1) {
					desc->next = next;
					rq->nsec = nsec;
				} else if (!rq->tmp3) {
					CALL_AST(rq);
				}
				return r;
			}
			rq->desc = next;
			rq->sec += sectors;
			rq->nsec = nsec - sectors;
			SD_CHECK_REQUEST("SD_SPLIT_REQUEST 2", rq);
			return 0;
		}
		descp = &desc->next;
	}
}

int SD_POST_REQUEST(SD_TAG *tag, BIORQ *RQ)
{
	SD *sd = tag->sd;
	int r;
	__usec_t sec;
	unsigned nsec;
	SD_CHECK_REQUEST("SD_POST_REQUEST 1", RQ);
	tag->brq = RQ;
	again:
#ifndef SD_TEST_RMW
	if (__unlikely(sd->flags & (FLAGS_NO_MEDIA | FLAGS_LARGER_SECSIZE)))
#endif
	{
		if (__unlikely(sd->flags & FLAGS_NO_MEDIA)) {
			SD_MEDIA_PROBE(sd);
			r = -EAGAIN;
			goto post_r;
		}
#ifndef SD_TEST_RMW
		if (__unlikely(((unsigned)RQ->sec | RQ->nsec) & ((1 << sd->secsize_shift) - 1)))
#else
		if (RQ != &tag->pw_rq)
#endif
		{
			if (__unlikely(!(RQ->flags & (BIO_WRITE | BIO_FLUSH)))) {
				SD_PAD_READ_REQUEST(tag, RQ);
				SD_CHECK_REQUEST("SD_POST_REQUEST 2", RQ);
			} else if (__unlikely(RQ->flags & BIO_WRITE)) {
				return SD_PARTIAL_WRITE(tag, RQ);
			}
		}
	}
	tag->retries = SD_RETRIES;
	tag->cmd.desc = RQ->desc;
	tag->cmd.cmd[1] = sd->lun_byte;
	if (__unlikely(RQ->flags & BIO_FLUSH)) {
		if (__unlikely(sd->flags & FLAGS_WP)) goto wp;
		tag->cmd.cmd[1] = sd->lun_byte | (sd->std >= STD_SBC_2 ? 0x04 : 0x00);
		tag->expected_xfer = 0;
		tag->cmd.desc = NULL;
		tag->cmd.direction = 0;
		if (__unlikely(sd->flags & FLAGS_NO_SYNCHRONIZE_CACHE)) {
			r = 0;
			goto post_r;
		}
		if (__likely(!(sd->flags & FLAGS_COMMAND_SIZE_16))) {
			tag->cmd.cmdlen = 10;
			tag->cmd.cmd[0] = SCMD_SYNCHRONIZE_CACHE_10;
			*(__u64 *)&tag->cmd.cmd[2] = __64CPU2BE(0);
		} else {
			tag->cmd.cmdlen = 16;
			tag->cmd.cmd[0] = SCMD_SYNCHRONIZE_CACHE_16;
			*(__u64 *)&tag->cmd.cmd[2] = __64CPU2BE(0);
			*(__u32 *)&tag->cmd.cmd[10] = __32CPU2BE(0);
			*(__u16 *)&tag->cmd.cmd[14] = __16CPU2BE(0);
		}
		goto post_rq;
	}
	if (__unlikely(RQ->nsec > sd->max_sectors)) goto split;
	if (__unlikely(RQ->nsec >= (unsigned)(sd->ap.max_fragments - 1))) {
		BIODESC *desc = RQ->desc;
		int frag = sd->ap.max_fragments;
		do {
			frag--;
		} while ((desc = desc->next));
		if (__unlikely(frag < 0)) goto split;
	}
	if (__likely(!(RQ->flags & BIO_WRITE))) {
		tag->cmd.direction = PF_WRITE;
		tag->cmd.cmd[0] = SCMD_READ_10;
	} else {
		if (__unlikely(sd->flags & (FLAGS_WP | FLAGS_BLOCK_WRITES))) {
			if (!(sd->flags & FLAGS_WP)) {
				if (__unlikely(SD_WRITES_IN_PROGRESS(sd))) {
					r = 1;
					goto post_r;
				}
				goto wr_ok;
			}
			wp:
			r = -EWP;
			goto post_r;
		}
		wr_ok:
		tag->cmd.direction = PF_READ;
		tag->cmd.cmd[0] = SCMD_WRITE_10;
	}
	sec = RQ->sec;
	tag->lba = sec;
	if (__unlikely(RQ->desc == &tag->pad_desc1)) tag->lba += tag->pad_desc1.v.len >> BIO_SECTOR_SIZE_BITS;	/* undo read padding, so that VSPACE FAULT is not reported outside */
	/*if (__likely(!(RQ->flags & BIO_WRITE)) || __unlikely(sd->flags & FLAGS_NO_SYNCHRONIZE_CACHE))*/ BIOQUE$SET_HEAD(sd->q, sec);
	nsec = RQ->nsec;
	tag->expected_xfer = nsec << BIO_SECTOR_SIZE_BITS;
	if (__unlikely(sd->flags & (FLAGS_LARGER_SECSIZE | FLAGS_SMALLER_SECSIZE))) {
		if (__likely(sd->flags & FLAGS_LARGER_SECSIZE)) sec >>= sd->secsize_shift, nsec >>= sd->secsize_shift;
		else sec <<= sd->secsize_shift, nsec <<= sd->secsize_shift;
	}
/* it is required that read and write commands differ in exactly 2 and have the
   same data layout
   ... if you use other command here, fix it also in SD_PARTIAL_WRITE_DONE */
	if (__likely(sd->flags & FLAGS_COMMAND_SIZE_10)) {
		*(__u32 *)&tag->cmd.cmd[2] = __32CPU2BE(sec);
		*(__u32 *)&tag->cmd.cmd[6] = __32CPU2BE(nsec << 8);
		tag->cmd.cmdlen = 10;
	} else if (__likely(sd->flags & FLAGS_COMMAND_SIZE_16)) {
		*(__u64 *)&tag->cmd.cmd[2] = __64CPU2BE(sec);
		*(__u32 *)&tag->cmd.cmd[10] = __32CPU2BE(nsec);
		tag->cmd.cmdlen = 16;
		tag->cmd.cmd[0] += (SCMD_READ_16 - SCMD_READ_10);
		*(__u16 *)&tag->cmd.cmd[14] = __16CPU2BE(0);
	} else {
		tag->cmd.cmd[1] |= (unsigned)sec >> 16;
		*(__u16 *)&tag->cmd.cmd[2] = __16CPU2BE(sec);
		*(__u16 *)&tag->cmd.cmd[4] = __16CPU2LE(nsec);
		tag->cmd.cmdlen = 6;
		tag->cmd.cmd[0] -= (SCMD_READ_10 - SCMD_READ_6);
	}
	post_rq:
	if (__likely(XLIST_EMPTY(&sd->in_progress))) KERNEL$SET_TIMER(sd->timeout, &sd->timer);
	ADD_TO_XLIST(&sd->in_progress, &tag->list);
	tag->cmd.internal = 0;
	tag->t = KERNEL$GET_SCHED_TICKS();
	r = sd->ap.post((SCSIRQ *)&tag->cmd);
	if (__unlikely(r)) {
		DEL_FROM_LIST(&tag->list);
		if (__likely(XLIST_EMPTY(&sd->in_progress))) KERNEL$DEL_TIMER(&sd->timer);
		post_r:
		RQ = TAG_RQ(tag);
		if (r == -EAGAIN) {	/* otherwise, the request might be already modified */
			if (__unlikely(sd->flags & FLAGS_LARGER_SECSIZE) && __likely(!(RQ->flags & (BIO_WRITE | BIO_FLUSH)))) SD_UNPAD_READ_REQUEST(tag, RQ);
			sd->free_tags++;
			__slow_slfree(tag);
			return 1;
		}
		SD_POST_ERROR(sd, RQ, r, &tag->lba);
		sd->free_tags++;
		__slow_slfree(tag);
		if (__unlikely(sd->free_tags == 1)) SD_DEQUEUE(&sd->ap);
		return r;
	}
	return 0;

	split:
	r = SD_SPLIT_REQUEST(tag);
	if (__unlikely(r)) {
		return r;
	}
	if (__unlikely(!sd->free_tags)) {
		return 1;
	}
	sd->free_tags--;
	tag = __slalloc(&sd->tag_slab);
	if (__unlikely(!tag)) {
		sd->free_tags++;
		return 1;
	}
	tag->lba = -1;
	tag->brq = RQ;
	SD_CHECK_REQUEST("SD_POST_REQUEST 3", RQ);
	goto again;
}

static void SD_REQUEST_ERROR(SD_TAG *tag);

static void SD_REQUEST_DONE(SCSIRQ *cmd)
{
	SD_TAG *tag = GET_STRUCT(cmd, SD_TAG, cmd);
	sched_unsigned t1;
	sched_unsigned t = KERNEL$GET_SCHED_TICKS();
	BIORQ *RQ;
	SD *sd = tag->sd;
	PROC *proc;
	DEL_FROM_LIST(&tag->list);
	KERNEL$DEL_TIMER(&sd->timer);
	if (__unlikely(!XLIST_EMPTY(&sd->in_progress))) KERNEL$SET_TIMER(sd->timeout, &sd->timer);
	if (__unlikely(tag->cmd.status != tag->expected_xfer) || __unlikely(tag->cmd.scsi_status != SCSI_GOOD)) {
		/*__debug_printf("done: %d(%d) %d, pending %ld\n", tag->cmd.status, tag->expected_xfer, tag->cmd.scsi_status, tag->brq->tmp3); */
		RAISE_SPL(SPL_VSPACE);
		proc = tag->brq->proc;
		SWITCH_PROC_ACCOUNT_TICKS(proc, SPL_X(SPL_VSPACE), t);
		BIOQUE$ACCOUNT_IOSCHED(proc, sd->q, tag->t, t);
		LOWER_SPL(SPL_ATA_SCSI);
		SD_REQUEST_ERROR(tag);
		return;
	}
	RQ = tag->brq;
	if (__unlikely(RQ == &tag->pw_rq)) {
		RAISE_SPL(SPL_VSPACE);
		proc = tag->brq->proc;
		SWITCH_PROC_ACCOUNT_TICKS(proc, SPL_X(SPL_VSPACE), t);
		BIOQUE$ACCOUNT_IOSCHED(proc, sd->q, tag->t, t);
		LOWER_SPL(SPL_ATA_SCSI);
		SD_PARTIAL_WRITE_DONE(tag);
		return;
	}
	t1 = tag->t;
	__slfree(tag);
	RAISE_SPL(SPL_VSPACE);
	proc = RQ->proc;
	SWITCH_PROC_ACCOUNT_TICKS(proc, SPL_X(SPL_VSPACE), t);
	BIOQUE$ACCOUNT_IOSCHED(proc, sd->q, t1, t);
	LOWER_SPL(SPL_ATA_SCSI);
#if __DEBUG >= 1
	if (__unlikely(!RQ->tmp3))
		KERNEL$SUICIDE("SD_REQUEST_DONE: REFERENCE COUNT UNDERFLOW");
#endif
	if (__likely(!--RQ->tmp3)) {
		CALL_AST(RQ);
	}
	sd->free_tags++;
	if (__unlikely(sd->free_tags == 1)) SD_DEQUEUE(&sd->ap);
}

DECL_IOCALL(SD_REQUEST, SPL_ATA_SCSI, BIORQ)
{
	HANDLE *h = RQ->handle;
	SD *sd;
	SD_TAG *tag;
	if (__unlikely(h->op != &SD_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_BIO);
	/*__debug_printf("SD REQUEST: %Lx, %x, %x \n", RQ->sec, RQ->nsec, RQ->flags);*/
	SWITCH_PROC_ACCOUNT(RQ->proc, SPL_X(SPL_ATA_SCSI));
	BIO_TRANSLATE_PARTITION;

	if (__unlikely(RQ->flags & ~BIO_FLAG_MASK))
		KERNEL$SUICIDE("SD_REQUEST: BAD REQUEST: FLAGS %08X", RQ->flags);

	SD_CHECK_REQUEST("SD_REQUEST 1", RQ);

	sd = ((PARTITION *)h->fnode)->dev;

#if __DEBUG >= 2
	BIOQUE$CHECK_QUEUE("SD_REQUEST", sd->q, RQ);
#endif

	RQ->status = 0;
	RQ->tmp3 = 1;

	if (__unlikely(!sd->free_tags)) goto enq;
	sd->free_tags--;
	tag = __slalloc(&sd->tag_slab);
	if (__unlikely(!tag)) {
		sd->free_tags++;
		goto enq;
	}
	tag->lba = -1;
	SD_CHECK_REQUEST("SD_REQUEST 2", RQ);
	if (__unlikely(SD_POST_REQUEST(tag, RQ) == 1)) goto enq;
	ret:
	RETURN;

	enq:
	BIOQUE$ENQUEUE_REQUEST(sd->q, RQ);
	goto ret;

	out:
	RQ->status = -ERANGE;
	RETURN_AST(RQ);
}

int SD_DEQUEUE(SCSI_ATTACH_PARAM *ap)
{
	SD *sd = GET_STRUCT(ap, SD, ap);
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("SD_DEQUEUE AT SPL %08X", KERNEL$SPL);
#endif
	if (__unlikely(sd->flags & FLAGS_WAITING_LIST_USED)) {
		if (!LIST_EMPTY(&sd->waiting_list)) {
			SD_TAG *tag = LIST_STRUCT(sd->waiting_list.next, SD_TAG, list);
			DEL_FROM_LIST(&tag->list);
			if (__unlikely(SD_RETRY_REQUEST(tag) == 1)) {
				return 0;
			}
			return 1;
		}
		sd->flags &= ~FLAGS_WAITING_LIST_USED;
	}
	if (__unlikely(!BIOQUE$QUEUE_EMPTY(sd->q)) && __likely(sd->free_tags)) {
		SD_TAG *tag;
		BIORQ *RQ;
		sd->free_tags--;
		tag = __slalloc(&sd->tag_slab);
		if (__unlikely(!tag)) {
			sd->free_tags++;
			return 0;
		}
		tag->lba = -1;
		RQ = BIOQUE$DEQUEUE(sd->q);
		if (__unlikely(RQ->status < 0)) {
			sd->free_tags++;
			__slow_slfree(tag);
			SD_POST_ERROR(sd, RQ, 0, NULL);
			return 1;
		}
		if (__unlikely(SD_POST_REQUEST(tag, RQ) == 1)) {
			SD_CHECK_REQUEST("SD_DEQUEUE", RQ);
			BIOQUE$REQUEUE_DEQUEUED_REQUEST(sd->q, RQ);
			return 0;
		}
		BIOQUE$ACK_DEQUEUE(sd->q);
		return 1;
	}
	if (__unlikely(sd->flags & FLAGS_BLOCK_WRITES)) {
		if (BIOQUE$QUEUE_EMPTY(sd->q)) sd->flags &= ~FLAGS_BLOCK_WRITES;
	}
	return 0;
}

BIORQ *SD_PROBE_QUEUE(SCSI_ATTACH_PARAM *ap)
{
	SD *sd = GET_STRUCT(ap, SD, ap);
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ATA_SCSI)))
		KERNEL$SUICIDE("SD_PROBE_QUEUE AT SPL %08X", KERNEL$SPL);
#endif
	return BIOQUE$PROBE_QUEUE(sd->q);
}

static void SD_REQUEST_ERROR(SD_TAG *tag)
{
	SD *sd = tag->sd;
	int a;
	int r;
	char *cmdstring;
	BIORQ *RQ;
	__u32 key;
	if (tag->cmd.status == -EVSPACEFAULT) {
		r = -EVSPACEFAULT;
		goto err2;
	}
	switch (tag->cmd.cmd[0]) {
		case SCMD_READ_6: cmdstring = "READ(6)"; break;
		case SCMD_WRITE_6: cmdstring = "WRITE(6)"; break;
		case SCMD_READ_10: cmdstring = "READ(10)"; break;
		case SCMD_WRITE_10: cmdstring = "WRITE(10)"; break;
		case SCMD_SYNCHRONIZE_CACHE_10: cmdstring = "SYNCHRONIZE CACHE(10)"; break;
		case SCMD_READ_16: cmdstring = "READ(16)"; break;
		case SCMD_WRITE_16: cmdstring = "WRITE(16)"; break;
		case SCMD_SYNCHRONIZE_CACHE_16: cmdstring = "SYNCHRONIZE CACHE(16)"; break;
		default: cmdstring = "UNKNOWN"; break;
	}
	if (tag->retries == SD_RETRIES) tag->start_time = KERNEL$GET_JIFFIES_LO();
	if (__unlikely(tag->cmd.status == -EINTR)) {
		if (tag->retries > 1) tag->retries = 1;
		if (--tag->retries >= 0) goto retry;
		if (sd->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, sd->dev_name, "%s: CANCELED DUE TO TIMEOUT", cmdstring);
		r = -EIO;
		goto err2;
	}
	SCSI$LOG_RECOVERED_ERROR(sd->dev_name, sd->errorlevel, cmdstring, (SCSIRQ *)&tag->cmd);
	if (__likely(tag->cmd.status >= 0) && __unlikely(tag->cmd.scsi_status == SCSI_GOOD)) {
		if (sd->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, sd->dev_name, "%s: TRANSFERRED %u, EXPECTED %u", cmdstring, tag->cmd.status, tag->expected_xfer);
		if (--tag->retries >= 0) goto retry;
		r = -EIO;
		goto err2;
	}
	a = SCSI$RECOMMENDED_ACTION((SCSIRQ *)&tag->cmd);
	key = SCSI$GET_CMD_SENSE((SCSIRQ *)&tag->cmd);
	if (!SCSI_SENSE_UNKNOWN(key)) {
		__u8 asc = SCSI_SENSE_ASC(key);
		__u8 ascq = SCSI_SENSE_ASCQ(key);
		if (asc == 0x04 && (ascq == 0x02 || ascq == 0x11)) SD_MEDIA_PROBE(sd), a = SCSI_ACTION_RETRY_FEW_TIMES;
		if (asc == 0x28 && ascq == 0x00) SD_MEDIA_PROBE(sd), a = SCSI_ACTION_RETRY_FEW_TIMES;
		if (asc == 0x2A && (ascq == 0x00 || ascq == 0x01 || ascq == 0x09)) SD_MEDIA_PROBE(sd), a = SCSI_ACTION_RETRY_UNTIL_TIMEOUT;
		if (asc == 0x3A || (SCSI_SENSE_KEY(key) == SCSI_KEY_UNIT_ATTENTION && !asc && !ascq)) {
			sd->flags |= FLAGS_NO_MEDIA, a = SCSI_ACTION_FAIL;
			SD_RESET_FSTAT(sd);
		}
	}
	if (__likely(a == SCSI_ACTION_OK)) goto ok;
	if (__likely(a == SCSI_ACTION_RETRY_FEW_TIMES)) {
		if (--tag->retries >= 0) {
			retry:
			SD_RETRY_REQUEST(tag);
			return;
		}
		goto err;
	}
	if (__unlikely(a == SCSI_ACTION_RETRY_UNTIL_TIMEOUT)) {
		if (__unlikely(tag->retries == SD_RETRIES)) tag->retries = SD_RETRIES - 1;
		else if (__unlikely(KERNEL$GET_JIFFIES_LO() - tag->start_time > sd->timeout)) goto err;
		goto retry;
	}
	err:
	if (tag->cmd.cmd[0] == SCMD_SYNCHRONIZE_CACHE_10 || __unlikely(tag->cmd.cmd[0] == SCMD_SYNCHRONIZE_CACHE_16)) {
		if (!SCSI_SENSE_UNKNOWN(key) && SCSI_SENSE_KEY(key) == SCSI_KEY_ILLEGAL_REQUEST && (__likely(SCSI_SENSE_ASC(key) == 0x20) || __likely(SCSI_SENSE_ASC(key)) == 0x24) && SCSI_SENSE_ASCQ(key) == 0x00) {
			sd->flags |= FLAGS_NO_SYNCHRONIZE_CACHE;
			goto ok;
		}
	}
	r = SCSI$CMD_ERROR_CODE(sd->scsi_version, (SCSIRQ *)&tag->cmd);
	SCSI$LOG_ERROR(sd->dev_name, sd->errorlevel, &sd->inq, sd->scsi_version, cmdstring, (SCSIRQ *)&tag->cmd, r);
	if (r == -EIO) {
		__sec_t lba = SCSI$CMD_LBA(sd->scsi_version, (SCSIRQ *)&tag->cmd);
		if (lba >= 0) {
			if (__unlikely(sd->flags & (FLAGS_LARGER_SECSIZE | FLAGS_SMALLER_SECSIZE))) {
				if (__likely(sd->flags & FLAGS_LARGER_SECSIZE)) lba <<= sd->secsize_shift;
				else lba >>= sd->secsize_shift;
			}
			tag->lba = lba;
		}
	}
	err2:
	RQ = TAG_RQ(tag);
	SD_POST_ERROR(sd, RQ, r, &tag->lba);
	ret:
	sd->free_tags++;
	__slow_slfree(tag);
	if (__unlikely(sd->free_tags == 1)) SD_DEQUEUE(&sd->ap);
	return;
	ok:
	RQ = tag->brq;
	if (__unlikely(RQ == &tag->pw_rq)) {
		SD_PARTIAL_WRITE_DONE(tag);
		return;
	}
	if (__unlikely(!RQ->tmp3))
		KERNEL$SUICIDE("SD_REQUEST_ERROR: REFERENCE COUNT UNDERFLOW");
	if (__likely(!--RQ->tmp3)) {
		CALL_AST(RQ);
	}
	goto ret;
}

int SD_IOCTL(IOCTLRQ *rq, PARTITION *pa, IORQ *rq_to_queue)
{
	int r;
	SD *sd;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV))) KERNEL$SUICIDE("SD_IOCTL AT SPL %08X", KERNEL$SPL);
#endif
	sd = pa->dev;
	RAISE_SPL(SPL_ATA_SCSI);
	switch (rq->ioctl) {
		case IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE: {
			unsigned rqsize;
			if (__unlikely(sd->flags & FLAGS_NO_MEDIA)) {
				probe_media:
				WQ_WAIT_F(&sd->media_wq, rq_to_queue);
				SD_MEDIA_PROBE(sd);
				r = 1;
				break;
			}
			rqsize = sd->max_sectors;
			if (__unlikely(sd->ap.max_fragments)) {
				unsigned fsize = sd->ap.max_fragments * __SECTORS_PER_PAGE_CLUSTER - __SECTORS_PER_PAGE_CLUSTER_MINUS_1;
				if (__unlikely(fsize == 1)) fsize = __SECTORS_PER_PAGE_CLUSTER;
				/*__debug_printf("fsize %d, rqsize %d\n", fsize, rqsize);*/
				if (rqsize > fsize) rqsize = fsize;
			}
			rq->status = rqsize;
			r = 0;
			break;
		}
		case IOCTL_BIO_PHYSICAL_BLOCKSIZE:
		case IOCTL_BIO_PARTITION_BLOCKSIZE: {
			if (__unlikely(sd->flags & FLAGS_NO_MEDIA)) goto probe_media;
			rq->status = SD_SECTOR_SIZE(sd);
			r = 0;
		}
		default: {
			r = -1;
			break;
		}
	}
	LOWER_SPL(SPL_DEV);
	return r;
}

void SD_DEQUEUE_ERROR(SD *sd, int error)
{
	while (!LIST_EMPTY(&sd->waiting_list)) {
		SD_TAG *tag = LIST_STRUCT(sd->waiting_list.next, SD_TAG, list);
		DEL_FROM_LIST(&tag->list);
		SD_POST_ERROR(sd, TAG_RQ(tag), error, NULL);
		sd->free_tags++;
		__slow_slfree(tag);
	}
	sd->flags &= ~(FLAGS_WAITING_LIST_USED | FLAGS_BLOCK_WRITES);
	while (__unlikely(!BIOQUE$QUEUE_EMPTY(sd->q))) {
		BIORQ *RQ;
		RQ = BIOQUE$DEQUEUE(sd->q);
		SD_POST_ERROR(sd, RQ, error, NULL);
	}
}

void SD_TIMER_FN(TIMER *t)
{
	SD *sd = GET_STRUCT(t, SD, timer);
	SD_TAG *tag;
	LOWER_SPL(SPL_ATA_SCSI);
	VOID_LIST_ENTRY(&sd->timer.list);
	if (sd->errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, sd->dev_name, "TIMEOUT, CANCELING COMMANDS");
	XLIST_FOR_EACH(tag, &sd->in_progress, SD_TAG, list)
		sd->ap.cancel((SCSIRQ *)&tag->cmd);
}
