#include <SPAD/SYNC.H>
#include <SPAD/DEV.H>
#include <SPAD/BIO.H>
#include <SPAD/LIBC.H>
#include <SYS/TYPES.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYSLOG.H>
#include <STRING.H>

#include "PARTTYPE.H"

#define MAX_PARTS	256

static char *name;
static char msg[__MAX_STR_LEN + 10];


static int disk_handle_num;
static unsigned sector_size = 1;

static __sec_t sector;

static BIODESC biodesc;

static unsigned char buffer_data[1023];

static unsigned char *buffer;

static struct partition {
	__sec_t start;
	__sec_t len;
	__sec_t extstart;
	int used;
	int system;
} partitions [MAX_PARTS];

static int maxpart;
static int iteration;
static int extscan;

static struct partition_table {
	__u8 bootcode[0x1be];
	struct {
		__u8 boot;
		__u8 head, sec, cyl;
		__u8 system;
		__u8 head_end, sec_end, cyl_end;
		__u8 start[4];
		__u8 len[4];
	} part[4];
	__u8 magic[2];
} *table;

static char lname[__MAX_STR_LEN + 5];
static char lalias[__MAX_STR_LEN + 48];

static __finline__ __sec_t getpos(__const__ __u8 ptr[4])
{
	return ptr[0] + (ptr[1] << 8) + (ptr[2] << 16) + (ptr[3] << 24);
}

static __finline__ int is_extended(__u8 sys)
{
	return sys == DOS_EXTENDED_PARTITION || sys == LINUX_EXTENDED_PARTITION || sys == WIN98_EXTENDED_PARTITION;
}

static void get_part_desc(int log_ptr)
{
	_snprintf(lname, sizeof(lname), "%s", name);
	if (strchr(lname, ':')) *strchr(lname, ':') = 0;
	_snprintf(lname + strlen(lname), sizeof(lname) - strlen(lname), "-P%d", log_ptr + 1);
}

static void set_up_biorq(__sec_t sec, BIORQ *b)
{
	sector = sec;
	b->h = disk_handle_num;
	b->sec = sec * sector_size;
	b->nsec = 1;
	b->flags = BIO_READ;
	b->desc = &biodesc;
	b->proc = &KERNEL$PROC_KERNEL;
	b->fault_sec = -1;
	biodesc.v.ptr = (__v_off)(unsigned long)buffer;
	biodesc.v.len = 512;
	biodesc.v.vspace = &KERNEL$VIRTUAL;
}

int main(int argc, char *argv[])
{
	OPENRQ openrq;
	IOCTLRQ io;
	BIORQ biorq;
	int log_ptr;
	LOGICAL_REQUEST lrq;

	int i;
	int swp;
	char *p, *l;

	buffer = (unsigned char *)(((unsigned long)buffer_data + 511) & ~511UL);
	table = (void *)buffer;

	if (sizeof(struct partition_table) != 512)
		KERNEL$SUICIDE("SCANPART: MISCOMPILED: sizeof(struct partition_table) == %d != 512", (int)sizeof(struct partition_table));

	if (argc != 2) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SCANPART: SYNTAX ERROR");
		badss:
		return -EBADSYN;
	}
	name = argv[1];
	p = strchr(name, ':');
	l = strchr(name, '/');
	if (l || !p || p != name + strlen(name) - 1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SCANPART: DEVICE NAME MUST NOT CONTAIN DIRECTORIES AND MUST END WITH ':'");
		goto badss;
	}
	__upcase(name);
	_snprintf(msg, sizeof(msg), "SCANPART: %s", name);
	memset(partitions, 0, sizeof partitions);
	maxpart = 0;
	iteration = -1;
	extscan = -1;
	openrq.flags = O_RDONLY;
	openrq.path = name;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SCANPART: COULD NOT OPEN %s: %s", name, strerror(-openrq.status));
		return openrq.status;
	}
	disk_handle_num = openrq.status;

	io.h = disk_handle_num;
	io.ioctl = IOCTL_BIO_PARTITION_BLOCKSIZE;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__likely(io.status >= 0)) {
		if (__unlikely(!io.status) || __unlikely(io.status & 511)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID PARTITION SECTOR SIZE %ld", msg, io.status);
			KERNEL$FAST_CLOSE(disk_handle_num);
			return -EINVAL;
		}
		sector_size = io.status / 512;
	}

	set_up_biorq(0, &biorq);
	nextrq:
	SYNC_IO_CANCELABLE(&biorq, KERNEL$BIO);
	if (biorq.status < 0) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, msg, "ERROR READING SECTOR %016"__sec_t_format"X: %s", sector, strerror(-biorq.status));
		if (iteration) goto cont_ext;
		end:
		KERNEL$FAST_CLOSE(disk_handle_num);
		if (biorq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR AT SECTOR %08lX%08lX: %s", msg, (unsigned long)(sector >> 31 >> 1), (unsigned long)sector, strerror(-biorq.status));
		return biorq.status;
	}
	iteration++;
	if (table->magic[0] != 0x55 || table->magic[1] != 0xaa) {
		KERNEL$SYSLOG(__SYSLOG_SW_ERROR, msg, "BAD MAGIC ON PARTITION TABLE, SECTOR %08lX%08lX: MAGIC %02X%02X", (unsigned long)(sector >> 31 >> 1), (unsigned long)sector, table->magic[0], table->magic[1]);
		if (!iteration) goto end;
		goto cont_ext;
	}
	for (i = 0; i < 4; i++) {
		if (/* table->part[i].system linux akceptuje i 0 */ getpos(table->part[i].len)) {
			__sec_t pstart;
			int partnum;
			if (!iteration) partnum = i, maxpart = 4;
			else partnum = maxpart++;
			if (maxpart >= MAX_PARTS) {
				KERNEL$SYSLOG(__SYSLOG_SW_INCOMPATIBILITY, msg, "TOO MANY PARTITIONS (MAX %d)", MAX_PARTS);
				goto endscan;
			}
			if (extscan == -1) pstart = 0;
			else {
				if (!is_extended(table->part[i].system)) pstart = partitions[extscan].start;
				else pstart = partitions[extscan].extstart;
			}
			partitions[partnum].start = getpos(table->part[i].start) + pstart;
			if (extscan == -1) partitions[partnum].extstart = partitions[partnum].start;
			else partitions[partnum].extstart = partitions[extscan].extstart;
			partitions[partnum].len = getpos(table->part[i].len);
			partitions[partnum].used = 1;
			partitions[partnum].system = table->part[i].system;
			/*__debug_printf("PARTITION %d: %08X, %08X, %02X\n", partnum + 1, (unsigned)partitions[partnum].start, (unsigned)partitions[partnum].len, (unsigned)partitions[partnum].system);*/
		}
	}

	cont_ext:
	for (extscan++; extscan < maxpart; extscan++) {
		if (partitions[extscan].used) {
			int sys = partitions[extscan].system;
			if (is_extended(sys)) {
				set_up_biorq(partitions[extscan].start, &biorq);
				goto nextrq;
			}
		}
	}

	endscan:

	/* delete additional extended partitions in chains (to be compatible with Linux) */
	for (i = 4; i < maxpart; i++) {
		if (is_extended(partitions[i].system)) {
			memmove(&partitions[i], &partitions[i + 1], (maxpart - i - 1) * sizeof(struct partition));
			maxpart--;
			i--;
		}
	}

	/* sort partitions according to extstart (if there are more extended partitions (which shoudn't happen), get compatible results with Linux) */
	do {
		swp = 0;
		for (i = 4; i < maxpart - 1; i++)
			if (partitions[i].extstart > partitions[i + 1].extstart) {
				struct partition p;
				memcpy(&p, &partitions[i], sizeof(struct partition));
				memcpy(&partitions[i], &partitions[i + 1], sizeof(struct partition));
				memcpy(&partitions[i + 1], &p, sizeof(struct partition));
				swp = 1;
			}
	} while (swp);
	
	log_ptr = -1;
	next_part:
	log_ptr++;
	get_part_desc(log_ptr);
	if (log_ptr < maxpart) {
		if (!partitions[log_ptr].system || !partitions[log_ptr].len) {
			KERNEL$DELETE_LOGICAL(lname, 0);
			goto next_part;
		}
		lrq.name = lname;
		_snprintf(lalias, sizeof(lalias), "%s/^PART=%"__sec_t_format"d,%"__sec_t_format"d", name, partitions[log_ptr].start * sector_size, partitions[log_ptr].len * sector_size);
		lrq.alias = lalias;
		lrq.flags = 0;
		lrq.cwd = NULL;
		lrq.chh_option = NULL;
		lrq.blob = NULL;
		SYNC_IO_CANCELABLE(&lrq, KERNEL$CREATE_LOGICAL);
		if (lrq.status < 0) {
			KERNEL$SYSLOG(__SYSLOG_CONF_ERROR, msg, "CAN'T CREATE LOGICAL NAME %s->%s: %s", lname, lalias, strerror(-lrq.status));
			KERNEL$FAST_CLOSE(disk_handle_num);
			if (lrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "LOGICAL NAME ERROR");
			if (log_ptr) _printf("\n");
			return lrq.status;
		}
		_printf("%s%s (%"__sec_t_format"dMB, SYSID %02X)", log_ptr ? " " : "", lname, partitions[log_ptr].len * sector_size / (1024 * 1024 / 512), (unsigned)partitions[log_ptr].system);
		goto next_part;
	}
	if (!KERNEL$DELETE_LOGICAL(lname, 0) || log_ptr < 4) goto next_part;
	_printf("\n");
	KERNEL$FAST_CLOSE(disk_handle_num);
	return 0;
}



