#include <ARCH/SETUP.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV.H>
#include <SPAD/TIMER.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/VM.H>
#include <SPAD/DEV_KRNL.H>
#include <ARCH/IO.H>
#include <ARCH/MOV.H>
#include <SPAD/CMOS.H>
#include <SPAD/CONSOLE.H>
#include <SPAD/TTY.H>

#include "VESA.H"
#include "CONS.H"
#include "VGAREG.H"

#define DELAYED_UPDATE_FREQ	25

#define CARD_MDA	0
#define CARD_CGA	1
#define CARD_EGA	2
#define CARD_VGA	3

c_char *base;
int cols;
int rows;
static char card_type;
char can_do_color;
static char can_do_vesa;
static unsigned long port;

static int ring_size = 512;
int vcs;

static c_char apply_softcursor(c_char b);
static void cons_update(void);
static int cons_getattr(int color, int flags);
static void cons_write(int x, int y, int chr);
static void cons_setcursor(int x, int y, int type);
static void cons_clear_cached(void);

struct console_charset {
/* xlate: 0: normal, 1: VT100 graphics, 2: ESC[11m (cp437), 3: reverse normal, for mouse copying */
	unsigned char xlate[4][256];
	unsigned refcount;
	__u8 width;
	__u8 height;
	unsigned long data[1];
};

static struct console_charset *console_charsets[MAX_VCS];

static __const__ struct console_charset default_charset = {
#include "PCC_MAP.I"
	, 0, 0, 0, { 0 }
};

static void cons_hw_set_font(__const__ struct console_charset *c);

static __finline__ int on_kernel_cons(void)
{
	return base == KERNEL$CONSOLE_ADDR;
}

static __finline__ void cons_write(int x, int y, int chr)
{
	if (__unlikely(KERNEL$CONSOLE_GFX)) return;
#if __DEBUG >= 2
	if (__unlikely((unsigned)x >= cols) || __unlikely((unsigned)y >= rows)) KERNEL$SUICIDE("cons_write: OUT OF RANGE: x == %d, y == %d", x, y);
#endif
	base[x + y * cols] = __16CPU2LE(chr);
}

static __finline__ c_char invert_char(c_char b)
{
	if (__unlikely(!can_do_color)) return b ^ 0x0800;
	else return ((b) & 0x88ff) | (((b) & 0x7000) >> 4) | (((b) & 0x0700) << 4);
}

static __finline__ c_char mouse_invert_char(c_char b)
{
	return invert_char(b);
}

static __finline__ int cons_validate_font(struct font_blob *font)
{
	if (__unlikely(card_type != CARD_EGA) && __unlikely(card_type != CARD_VGA)) return 0;
	return (font->w == 8 || (__likely(font->w == 9) && __likely(card_type == CARD_VGA))) && __likely(font->h > 0) && __likely(font->h <= 32);
}

#include "CONS.I"

static c_char apply_softcursor(c_char b)
{
	int type = tt[avc].cursor_type;
	/* it is already assumed that type & CURSOR_SOFT */
	c_char orig = b;
	b |= (type >> 8) & 0xff00;
	b ^= type & 0xff00;
	if (type & 0x20 && (orig & 0x7000) == (b & 0x7000)) b ^= 0x7000;
	if (type & 0x40 && (b & 0x700) == ((b & 0x7000) >> 4)) b ^= 0x700;
	if (type & 0x80) b = invert_char(b);
	return b;
}

static void cons_update(void)
{
	int i;
	c_char *a;
	if (__unlikely(KERNEL$CONSOLE_GFX)) return;
	for (i = 0, a = base; i < rows; i++, a += cols) {
#ifdef __LITTLE_ENDIAN
		if ((__unlikely(mouse_visible) && __unlikely(i == mouse_y))
		 || (__unlikely(drag_visible) && __unlikely(i >= drag_start_y) && __unlikely(i <= drag_end_y))
		 || (__unlikely(tt[avc].cursor_type & CURSOR_SOFT) && __unlikely(i == tt[avc].y + scrollback[avc]))) {
#endif
			update_line_slow(i);
#ifdef __LITTLE_ENDIAN
		} else {
			c_char *line = lines[avc][RING_MASK(scrpos[avc] - scrollback[avc] + i)];
			memcpy(a, line, cols * sizeof(c_char));
		}
#endif
	}
}


/*
 * color: low 4 bits - fg, high 4 bits - bg
 * attr: attributes: TTYF_xxx
 */
static int cons_getattr(int color, int flags)
{
	int attr;
	if (!can_do_color) {
		if (!(flags & TTYF_REVERSE)) {
			if (!(flags & TTYF_BRIGHT)) {
				if (!(flags & TTYF_UNDERLINE))
					return 0x07;
				else
					return 0x01;
			} else {
				if (!(flags & TTYF_UNDERLINE))
					return 0x0F;
				else
					return 0x09;
			}
		} else {
			if (!(flags & TTYF_BRIGHT))
				return 0x70;
			else
				return 0x7F;
		}
	}
	attr = color;
	if (flags & TTYF_UNDERLINE) attr = (attr & 0xf0) | t->ul_color;
	else if (flags & TTYF_DIM) attr = (attr & 0xf0) | t->half_color;
	if (flags & TTYF_REVERSE) attr = ((attr >> 4) & 0x0f) | ((attr << 4) & 0xf0);
	if (flags & TTYF_BLINK) attr |= 0x80;
	if (flags & TTYF_BRIGHT) attr |= 0x08;
	return attr;
}

static int lv = -1;
static int lt = -1;

/* x can be == cols */
static void cons_setcursor(int x, int y, int type)
{
	int val;
	if (__unlikely(KERNEL$CONSOLE_GFX)) return;
	if (on_kernel_cons()) {
		KERNEL$CONSOLE_X = x;
		KERNEL$CONSOLE_Y = y;
	}
	if (__unlikely(x == cols)) x--;
	if (__unlikely((unsigned)y >= (unsigned)rows)) y = rows + 1;
	val = x + y * cols;
	if (__unlikely(type != lt)) {
		int height;
		int from, to;
		io_outb_p(port, VGA_CRTC_MAXIMUM_SCAN_LINE);
		height = (io_inb(port + 1) & VGA_CRTC_MAXIMUM_SCAN_LINE_LINE) + 1;
		lt = type;
		to = height - (height < 10 ? 1 : 2);
		if (type == CURSOR_NONE) {
			from = 31;
			to = 30;
		} else if (type == CURSOR_UNDERLINE) {
			from = height - (height < 10 ? 2 : 3);
		} else if (type == CURSOR_LOWER_THIRD) {
			from = height * 2 / 3;
		} else if (type == CURSOR_LOWER_HALF) {
			from = height / 2;
		} else if (type == CURSOR_TWO_THIRDS) {
			from = height / 3;
		} else {
			from = 1;
			to = height;
		}
		io_outb_p(port, VGA_CRTC_CURSOR_START);
		io_outb_p(port + 1, (io_inb_p(port + 1) & 0xc0) | from);
		io_outb_p(port, VGA_CRTC_CURSOR_END);
		io_outb_p(port + 1, (io_inb_p(port + 1) & 0xe0) | to);
	}
	if (val != lv) {
		lv = val;
		io_outb_p(port, VGA_CRTC_CURSOR_HI);
		io_outb_p(port + 1, val >> 8);
		io_outb_p(port, VGA_CRTC_CURSOR_LO);
		io_outb_p(port + 1, val);
	}
}

static void cons_clear_cached(void)
{
	lv = -1;
	lt = -1;
}

static void cons_hw_set_font(__const__ struct console_charset *c)
{
#define USER_FONT_ADDR	0xa8000
	__const__ unsigned long *fontdata;
	unsigned fontslot, px, skipmask, i;
	__u32 *addr;
	__u8 misc, current_pixels, wanted_pixels, changing_width;
	if (__unlikely(KERNEL$CONSOLE_GFX)) return;
	if (__unlikely(card_type != CARD_EGA) && __unlikely(card_type != CARD_VGA)) return;
	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, VGA_TS_RESET_AR);
	io_outb_p(VGA_TS_IDX, VGA_TS_PLANE_WRITE_ENABLE);
	io_outb_p(VGA_TS_REG, 0x04);
	io_outb_p(VGA_TS_IDX, VGA_TS_MEMORY_MODE);
	io_outb_p(VGA_TS_REG, 1 | VGA_TS_MEMORY_MODE_EXT_MEM | VGA_TS_MEMORY_MODE_OE_DIS);
	io_outb_p(VGA_TS_IDX, VGA_TS_CLOCKING_MODE);
	current_pixels = io_inb_p(VGA_TS_REG);
	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, VGA_TS_RESET_AR | VGA_TS_RESET_SR);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_READ_MAP);
	io_outb_p(VGA_GDC_REG, 0x02);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_GRAPHICS_MODE);
	io_outb_p(VGA_GDC_REG, 0x00);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_MISC);
	misc = io_inb_p(VGA_GDC_REG);
	io_outb_p(VGA_GDC_REG, VGA_GDC_MISC_MEMORY_MAP_A000_BFFF);

	if (c != &default_charset) for (i = 0,
			fontdata = c->data,
			px = c->width * c->height,
			fontslot = LONGS_PER_CHAR(c->width, c->height),
			addr = (__u32 *)(KERNEL$ZERO_BANK + USER_FONT_ADDR),
			skipmask = c->width == 9 ? 7 : -1
			; i < 256; i++, fontdata += fontslot) {
		unsigned bitpos = 0;
		__u32 value = 0;
		unsigned x;
		for (x = 0; x < px; x++) {
			if (__unlikely(bitpos == 32)) *addr++ = __32CPU2LE(value), value = 0, bitpos = 0;
			value |= __BT(fontdata, x) << (bitpos ^ 7);
			bitpos++;
			x += __unlikely(!(bitpos & skipmask));
		}
		*addr = __32CPU2LE(value);
		addr = (__u32 *)(((unsigned long)addr + 32) & ~31UL);
	}
	__write_barrier2();
	wanted_pixels = c->width == 8;
	changing_width = (wanted_pixels ^ current_pixels) & VGA_TS_CLOCKING_MODE_98DM;
	if (__unlikely(card_type != CARD_VGA)) changing_width = 0;

	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, __unlikely((wanted_pixels ^ current_pixels) & VGA_TS_CLOCKING_MODE_98DM) ? 0 : VGA_TS_RESET_AR);
	io_outb_p(VGA_TS_IDX, VGA_TS_PLANE_WRITE_ENABLE);
	io_outb_p(VGA_TS_REG, 0x03);
	io_outb_p(VGA_TS_IDX, VGA_TS_MEMORY_MODE);
	io_outb_p(VGA_TS_REG, 1 | VGA_TS_MEMORY_MODE_EXT_MEM);
	io_outb_p(VGA_TS_IDX, VGA_TS_CHARACTER_MAP);
	io_outb_p(VGA_TS_REG, c == &default_charset ? 0x00 : 0x0a);
	if (__unlikely(changing_width)) {
		io_outb_p(VGA_TS_IDX, VGA_TS_CLOCKING_MODE);
		io_outb_p(VGA_TS_REG, (current_pixels & ~VGA_TS_CLOCKING_MODE_98DM) | wanted_pixels);
	}
	io_outb_p(VGA_TS_IDX, VGA_TS_RESET);
	io_outb_p(VGA_TS_REG, VGA_TS_RESET_AR | VGA_TS_RESET_SR);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_READ_MAP);
	io_outb_p(VGA_GDC_REG, 0x00);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_GRAPHICS_MODE);
	io_outb_p(VGA_GDC_REG, VGA_GDC_GRAPHICS_MODE_HOST_OE);
	io_outb_p(VGA_GDC_IDX, VGA_GDC_MISC);
	io_outb_p(VGA_GDC_REG, misc);
	if (__unlikely(changing_width)) {
		__u8 atc_ctrl, output;
		output = io_inb_p(VGA_OUTPUT_REG);
		if (__likely(!(output & VGA_OUTPUT_REG_CLOCK_2))) {
			output = (output | VGA_OUTPUT_REG_CLOCK_1) - (wanted_pixels * VGA_OUTPUT_REG_CLOCK_1);
			io_outb_p(VGA_OUTPUT_REG_WRITE, output);
		}
		io_inb_p(VGA_BASE_COLOR + VGA_INPUT_REG);
		io_outb_p(VGA_ATC_IDX_REG, VGA_ATC_CONTROL | VGA_ATC_IDX_REG_PAS);
		atc_ctrl = io_inb_p(VGA_ATC_REG_READ);
		atc_ctrl = (atc_ctrl | VGA_ATC_CONTROL_LGE) ^ (wanted_pixels * VGA_ATC_CONTROL_LGE);
		io_outb_p(VGA_ATC_IDX_REG, atc_ctrl);
		io_outb_p(VGA_ATC_IDX_REG, VGA_ATC_H_PIXEL_PANNING | VGA_ATC_IDX_REG_PAS);
		io_outb_p(VGA_ATC_IDX_REG, 8 ^ (wanted_pixels * 8));
	}
#undef USER_FONT_ADDR
}

static void probe_hw_console(void)
{
	int equip;
	can_do_vesa = 0;
	can_do_color = *(__u8 *)(KERNEL$ZERO_BANK + 0x449) != 7;
	base = (void *)(*(__u8 *)(KERNEL$ZERO_BANK + 0x449) == 7 ? 0xb0000 : 0xb8000);
	port = (*(__u8 *)(KERNEL$ZERO_BANK + 0x449) == 7 ? 0x3b4 : 0x3d4);
	cols = *(__u16 *)(KERNEL$ZERO_BANK + 0x44a);
	rows = *(__u8 *)(KERNEL$ZERO_BANK + 0x484) + 1;
	if (rows <= 1 || rows >= 101) rows = 25;
	if (cols <= 1 || cols >= 301) cols = 80;
		/* test for MDA/CGA/EGA/VGA, method from FreeBSD>=4 */
	equip = CMOS$READ(0x14);
	if (equip >= 0 && (equip >> 4) & 3) {
		if (!can_do_color) card_type = CARD_MDA;
		else card_type = CARD_CGA;
	} else {
		__u8 cursor;
		card_type = CARD_EGA;
		io_outb_p(port, VGA_CRTC_CURSOR_START);
		cursor = io_inb_p(port + 1);
		io_outb_p(port + 1, cursor | 0x20);
		if (io_inb_p(port + 1) != (cursor | 0x20)) goto ega;
		io_outb_p(port + 1, cursor & ~0x20);
		if (io_inb_p(port + 1) != (cursor & ~0x20)) goto ega;
		card_type = CARD_VGA;
		can_do_vesa = 1;
		ega:
		io_outb_p(port + 1, cursor);
	}
}

/*
ARGS:	[BASE=#B8000]
	[PORT=#3D4]
	[COLS=80]
	[ROWS=25]
	[COLOR]
	[MONO]
*/

char dev_name[__MAX_STR_LEN];

static char port_locked = 0;
static char vga_locked = 0;
static IO_RANGE port_range;
static IO_RANGE vga_range;

static __const__ struct __param_table console_params[] = {
	"BASE", __PARAM_PTR, 0xa0000, 0x100000, &base,
	"PORT", __PARAM_UNSIGNED_LONG, 0, IO_SPACE_LIMIT - 1, &port,
	"COLS", __PARAM_INT, 2, 301, &cols,
	"ROWS", __PARAM_INT, 2, 101, &rows,
	"COLOR", __PARAM_BOOL, ~0, 1, &can_do_color,
	"MONO", __PARAM_BOOL, ~0, 0, &can_do_color,
	"NOVESA", __PARAM_BOOL, ~0, 0, &can_do_vesa,
	"SCROLLBACK", __PARAM_INT, 1, MAX_RING_SIZE + 1, &ring_size,
	"VC", __PARAM_INT, 1, MAX_VCS + 1, &vcs,
	"MAX_V_REFRESH", __PARAM_UNSIGNED_INT, 50, 10000, &max_v_refresh,
	"MAX_H_REFRESH", __PARAM_UNSIGNED_INT, 30, 10000, &max_h_refresh,
	NULL, 0, 0, 0, NULL,
};


int main(int argc, char *argv[])
{
	int r;
	int i, j;
	unsigned x, y;
	DEVICE_REQUEST console_dev;
	MALLOC_REQUEST mrq;
	char **arg = argv;
	int state = 0;
	vcs = MAX_VCS;
	probe_hw_console();
	if (__parse_params(&arg, &state, console_params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PCCONS: SYNTAX ERROR");
		return -EBADSYN;
	}
	_snprintf(dev_name, __MAX_STR_LEN, "PCCONS@" MEM_ID, (unsigned long)base);

	port_range.start = port;
	port_range.len = 8;
	port_range.name = dev_name;
/* port may be already locked by the kernel --- so allow failure */
	if (!KERNEL$REGISTER_IO_RANGE(&port_range)) port_locked = 1;

	if (can_do_vesa) {
		vga_range.start = VGA_BASE;
		vga_range.len = 0x10;
		vga_range.name = dev_name;
		if (KERNEL$REGISTER_IO_RANGE(&vga_range)) goto skip_vesa;
		vga_locked = 1;
		if (PROBE_VESA()) {
			can_do_vesa = 0;
			KERNEL$UNREGISTER_IO_RANGE(&vga_range);
			vga_locked = 0;
		}
	}
	skip_vesa:
	if (ring_size < rows || (ring_size & (ring_size - 1))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PCCONS: BAD SCROLLBACK (MUST BE POWER OF 2 AND GREATER THAN NUMBER OF ROWS)");
		r = -EBADSYN;
		goto ret0;
	}

	memset(lines, 0, sizeof(lines));
	for (i = 0; i < vcs; i++) for (j = 0; j < ring_size; j++) {
		mrq.size = sizeof(c_char) * cols;
		SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
		if (__unlikely(mrq.status < 0)) {
			if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PCCONS: COULD NOT ALLOCATE LINES BUFFER");
			r = mrq.status;
			goto ret1;
		}
		memset(mrq.ptr, 0, mrq.size);
		lines[i][j] = mrq.ptr;
	}

	avc = 0;
	for (i = 0; i < MAX_VCS; i++) console_charsets[i] = (struct console_charset *)&default_charset;
	for (i = 0; i < vcs; i++) {
		TEST_LOCKUP_SYNC;
		set_write_vc(i), tty_init();
	}
	set_write_vc(0);
	base = (void *)((char *)base + (unsigned long)KERNEL$ZERO_BANK);
	for (y = 0; y < rows; y++)
		for (x = 0; x < cols; x++)
			lines[0][RING_MASK(scrpos[0] + y)][x] = __16LE2CPU(base[x + y * cols]);
	io_outb_p(port, VGA_CRTC_CURSOR_HI);
	x = io_inb_p(port + 1);
	io_outb_p(port, VGA_CRTC_CURSOR_LO);
	x = (x << 8) + io_inb_p(port + 1);
	t->x = x % cols;
	if ((t->y = x / cols) >= rows) t->y = 0;
	validate_tty_state();

	last_cons_update = KERNEL$CONSOLE_BUFFER_PTR;

	console_dev.flags = 0;
	console_dev.init_root_handle = console_init_root;
	console_dev.dev_ptr = NULL;
	console_dev.name = dev_name;
	console_dev.driver_name = "PCCONS.SYS";
	console_dev.dcall = VESA_VIDEOMODE_DCALL;
	console_dev.dcall_type = "CONSOLE";
	console_dev.dctl = NULL;
	console_dev.unload = NULL;	/* !!! FIXME */

	SYNC_IO_CANCELABLE(&console_dev, KERNEL$REGISTER_DEVICE);

	if (console_dev.status < 0) {
		if (console_dev.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PCCONS: COULD NOT REGISTER DEVICE");
		r = console_dev.status;
		goto ret1;
	}
	strlcpy(KERNEL$ERROR_MSG(), dev_name, __MAX_STR_LEN);
	KERNEL$TSR_IMAGE();
	return 0;
	ret1:
	for (i = 0; i < vcs; i++) for (j = 0; j < ring_size; j++) {
		if (lines[i][j]) KERNEL$UNIVERSAL_FREE(lines[i][j]);
	}
	ret0:
	if (port_locked) KERNEL$UNREGISTER_IO_RANGE(&port_range);
	if (vga_locked) KERNEL$UNREGISTER_IO_RANGE(&vga_range);
	return r;
}

long PCCONS_VIDEOMODE(int cmd, int vc, va_list args)
{
	switch (cmd) {
		case VIDEOMODE_GET_FONT_MODE: {
			if (card_type == CARD_EGA) return RESULT_GET_FONT_MODE_EGA;
			if (card_type == CARD_VGA) return RESULT_GET_FONT_MODE_VGA;
			return RESULT_GET_FONT_MODE_NONE;
		}
		case VIDEOMODE_SET_FONT: {
			PROC *proc = va_arg(args, PROC *);
			FBLOB *blob_addr = va_arg(args, FBLOB *);
			size_t blob_size = va_arg(args, size_t);
			IORQ *rq = va_arg(args, IORQ *);
			return CONS_SET_FONT(vc, proc, blob_addr, blob_size, rq);
		}
		case VIDEOMODE_GET_DEFAULT_CHARSET: {
			return (long)"ISO-8859-1";
		}
		default: {
			return -ENXIO;
		}
	}
}

