#include <SPAD/SYNC.H>
#include <SPAD/LIBC.H>
#include <SYS/TYPES.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TTY.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/TIMER.H>
#include <SPAD/DL.H>
#include <SPAD/PCI.H>
#include <VALUES.H>
#include <ASM/IO.H>
#include <ARCH/MOV.H>

#include <VGA.H>
#include "VGAREG.H"

#define S3_VIRGE_FIFO_TIMEOUT	(JIFFIES_PER_SECOND)
#define RETRACE_TIMEOUT		(JIFFIES_PER_SECOND / 20)

static char *cons = "PCCONS@MEM_B8000:";
static __const__ char dev_name[] = "VIDEO$S3";

static io_t io_base;

#define VGA_DWORD_IO

#include "VGA.I"
#include "DAC.I"

#define S3_911		0
#define S3_924		1
#define S3_801		2
#define S3_805		3
#define S3_928		4
#define S3_864		5
#define S3_964		6
#define S3_TRIO32	7
#define S3_TRIO64	8
#define S3_866		9
#define S3_868		10
#define S3_968		11
#define S3_765		12

static const char *s3_chipname[] = {
"911", "924", "801", "805", "928",
"864", "964", "Trio32", "Trio64", "866", "868", "968", "Trio64V+"};

static int s3_chiptype;

#define S3_OLD_STEPPING		0x04

static int s3_flags;
static struct dac *dac;
static int accel = 1;
static pci_id_t pci_id;
static int pci_valid = 0;
static long pci_chip_type = 0;
static char *pci_regs = NULL;

static __const__ struct __param_table params[] = {
	"CONSOLE", __PARAM_STRING, 1, __MAX_STR_LEN, &cons,
	"NOACCEL", __PARAM_BOOL, ~0, 0, &accel,
	NULL, 0, 0, 0, NULL,
};

#define VIRGE			0
#define VIRGE_VX		1
#define VIRGE_DX_GX		2
#define VIRGE_GX_2		3
#define VIRGE_MX		4
#define VIRGE_MXP		5
#define VIRGE_TRIO_3D		6
#define VIRGE_TRIO_3D_2X	7

static __const__ struct pci_id_s s3v_pci_cards[] = {
	{ 0x5333, 0x5631, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE", VIRGE },
	{ 0x5333, 0x883D, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE VX", VIRGE_VX },
	{ 0x5333, 0x8A01, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE DX/GX", VIRGE_DX_GX },
	{ 0x5333, 0x8A10, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE GX 2", VIRGE_GX_2 },
	{ 0x5333, 0x8C01, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE MX", VIRGE_MX },
	{ 0x5333, 0x8C03, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 ViRGE MX+", VIRGE_MXP },
	{ 0x5333, 0x8904, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Trio3D", VIRGE_TRIO_3D },
	{ 0x5333, 0x8A13, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "S3 Trio3D/2X", VIRGE_TRIO_3D_2X },
	{ 0 },
};

static void s3_set_timing(struct video_mode *v, struct tty_videomode_params *p);
static long s3_get_pixel_clock(struct video_mode *v, unsigned px);
static int s3_set_display_start(struct video_mode *v, unsigned addr, int waitretrace);

static long s3v_accel_avail(struct video_mode *v);
static void s3v_accel_init(struct video_mode *v);
static void *s3v_accel_do(struct video_mode *v, unsigned op, union accel_param *ap);

static struct driver_function s3_functions[] = {
	"ACCEL_AVAIL", NULL,
	"ACCEL_INIT", NULL,
	"ACCEL_DO", NULL,
	"SET_VIDEO_TIMING", s3_set_timing,
	"GET_PIXEL_CLOCK", s3_get_pixel_clock,
	"SET_DISPLAY_START", s3_set_display_start,
	NULL, NULL,
};

static int test_wr(unsigned reg, __u8 idx, __u8 mask)
{
	__u8 old, new1, new2;
	io_outb(io_base + reg, idx);
	old = io_inb(io_base + reg + 1);
	io_outb(io_base + reg + 1, old & ~mask);
	new1 = io_inb(io_base + reg + 1) & mask;
	io_outb(io_base + reg + 1, old | mask);
	new2 = io_inb(io_base + reg + 1) & mask;
	io_outb(io_base + reg + 1, old);
	return !new1 && new2 == mask;
}

static int s3_probe(void)
{
	int r;
	__u8 old_11, old_38, config;
	unsigned id, rev;
	RAISE_SPL(SPL_TTY);
	io_base = io_inb(VGA_OUTPUT_REG) & 1 ? VGA_BASE_COLOR : VGA_BASE_MONO;
	old_11 = in_cr(0x11);
	out_cr(0x11, 0x00);
	old_38 = in_cr(0x38);
	out_cr(0x38, 0x00);
	if (test_wr(VGA_CRTC_IDX, 0x35, 0x0f)) {
		not_found:
		r = -ENODEV;
		ret:
		out_cr(0x11, old_11);
		out_cr(0x38, old_38);
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	out_cr(0x38, 0x48);
	if (!test_wr(VGA_CRTC_IDX, 0x35, 0x0f)) goto not_found;
	out_cr(0x39, 0xa5);
	id = in_cr(0x30);
	rev = id & 0x0f;
	if (id >= 0xe0) {
		id |= in_cr(0x2e) << 8;
		rev |= in_cr(0x2f) << 4;
	}
	config = in_cr(0x36);
	s3_chiptype = -1;
	s3_flags = 0;
	switch (id & 0xfff0) {
		case 0x0080:
			if (rev == 1) s3_chiptype = S3_911;
			if (rev == 2) s3_chiptype = S3_924;
			break;
		case 0x00a0:
			switch (config & 0x03) {
				case 0x00:
				case 0x01:
					s3_chiptype = S3_805;
					if ((rev & 0x0f) < 2) s3_flags |= S3_OLD_STEPPING;
					break;
				case 0x03:
					s3_chiptype = S3_801;
					if ((rev & 0x0f) < 2) s3_flags |= S3_OLD_STEPPING;
					break;
			}
			break;
		case 0x0090:
			s3_chiptype = S3_928;
			if ((rev & 0x0f) < 4) s3_flags |= S3_OLD_STEPPING;
			break;
		case 0x00b0:
			s3_chiptype = S3_928;
			break;
		case 0x00c0:
			s3_chiptype = S3_864;
			break;
		case 0x00d0:
			s3_chiptype = S3_964;
			break;
		case 0x10e0:
			s3_chiptype = S3_TRIO32;
			break;
		case 0x3de0:
		case 0x31e0:
		case 0x01e0:
		case 0x04e0:
		case 0x11e0:
			if (rev & 0x0400) s3_chiptype = S3_765;
			else s3_chiptype = S3_TRIO64;
			break;
		case 0x80e0:
			s3_chiptype = S3_866;
			break;
		case 0x90e0:
			s3_chiptype = S3_868;
			break;
		case 0xf0e0:
		case 0xb0e0:
			s3_chiptype = S3_968;
			break;
	}
	if (__unlikely(s3_chiptype == -1)) {
		r = -ENXIO;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: UNKNOWN CHIP TYPE. ID %04X, REV %03X", id, rev);
		goto ret;
	}
	if (s3_chiptype == S3_TRIO64 || s3_chiptype == S3_765) {
		if ((r = s3_trio64_dac.probe())) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: UNABLE TO INITIALIZE DAC. CHIP %s", s3_chipname[s3_chiptype]);
			goto ret;
		}
		dac = &s3_trio64_dac;
	} else {
		if (!s3_sdac.probe()) dac = &s3_sdac;
		else if (!s3_gendac.probe()) dac = &s3_gendac;
		else {
			r = -ENXIO;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: UNKNOWN DAC. CHIP %s", s3_chipname[s3_chiptype]);
			goto ret;
		}
	}

	r = 0;
	goto ret;
}

static void s3_set_timing(struct video_mode *v, struct tty_videomode_params *p)
{
	struct tty_videomode_params pp = *p;
	if (__unlikely(!dac)) return;
	out_cr(0x38, 0x48);
	out_cr(0x39, 0xa5);
	if (__unlikely(!(pp.pixel_clock = dac->set_pixel_clock(v->bpp, pp.pixel_clock)))) return;
	if (s3_chiptype >= S3_911) out_cr(0x40, in_cr(0x40) | 0x01);
	pp.h = dac->map_horizontal_crtc(v->bpp, pp.pixel_clock, pp.h);
	pp.h_sync_start = dac->map_horizontal_crtc(v->bpp, pp.pixel_clock, pp.h_sync_start);
	pp.h_sync_end = dac->map_horizontal_crtc(v->bpp, pp.pixel_clock, pp.h_sync_end);
	pp.h_total = dac->map_horizontal_crtc(v->bpp, pp.pixel_clock, pp.h_total);
	/*__debug_printf("h: %d, h_sync_start: %d, h_sync_end: %d, h_total: %d\n", pp.h, pp.h_sync_start, pp.h_sync_end, pp.h_total);
	__debug_printf("v: %d, v_sync_start: %d, v_sync_end: %d, v_total: %d\n", pp.v, pp.v_sync_start, pp.v_sync_end, pp.v_total);*/
	if (pp.h_total > 4096) pp.h_total = 4096;
	setup_vga_timing(v, &pp);
	if (s3_chiptype >= S3_801) {
		int m, n;
		int i, j;
		n = 0xff;
		if (s3_chiptype >= S3_864 || s3_chiptype == S3_801 || s3_chiptype == S3_805) {
	/* How to calculate these values correctly? I don't know ...
	   ... in 864 they have these timings wrong in their own vesa bios! */
			int clock = (pp.pixel_clock * ((v->bpp + 7) >> 3)) / 1000;
			int mc;
			if (v->total_mem < 2048 * 1024 || s3_chiptype == S3_TRIO32) clock *= 2;
			if (__likely(mclk)) mc = mclk;
			else if (s3_chiptype == S3_864) mc = 40000;
			else if (s3_chiptype == S3_801 || s3_chiptype == S3_805) mc = 50000;
			else mc = 60000;
			m = (((__s64)mc * 720 / 1000 + 16867) * 89736 / (clock + 39000) - 21154) / 1000;
			if (v->total_mem < 2048 * 1024 || s3_chiptype == S3_TRIO32) m /= 2;
			/*__debug_printf("%d, %d %d: %d %d\n", v->total_mem, clock, s3_chiptype, m, n);*/
			if (m > 31) m = 31;
			if (m < 0) {
				m = 0;
				if (s3_chiptype != S3_864) {
/* 864 will trash screen on 800x600x16m32@75 with this enabled.
   Maybe more cards will need to be excluded here */
					n = 16;
				}
			}
		} else if (v->total_mem == 512 * 1024 || v->x > 1200) {
			m = 0;
		} else if (v->total_mem == 1024 * 1024) {
			m = 2;
		} else {
			m = 20;
		}
		out_cr(0x54, m << 3);
		out_cr(0x60, n);
		out_cr(0x55, (in_cr(0x55) & 0x08) | 0x40);

		out_cr(0x5e, (((pp.v_total - 2) & 0x400) >> 10) |
			     (((pp.v - 1) & 0x400) >> 9) |
			     (((pp.v_sync_start) & 0x400) >> 8) |
			     (((pp.v_sync_start) & 0x400) >> 6) | 0x40);
		i = ((((pp.h_total >> 3) - 5) & 0x100) >> 8) |
		    ((((pp.h >> 3) - 1) & 0x100) >> 7) |
		    ((((pp.h_sync_start >> 3) - 1) & 0x100) >> 6) |
		    ((pp.h_sync_start & 0x800) >> 7);
		if ((pp.h_sync_end >> 3) - (pp.h_sync_start >> 3) > 64) i |= 0x08;
		if ((pp.h_sync_end >> 3) - (pp.h_sync_start >> 3) > 32) i |= 0x20;
		j = (in_cr(0x00) + ((i & 0x01) << 8) + in_cr(0x04) + ((i & 0x10) << 4) + 1) / 2;
		if (j - (in_cr(0x04) + ((i & 0x10) << 4)) < 4) {
			if (in_cr(0x04) + ((i & 0x10) << 4) + 4 <= in_cr(0x00) + ((i & 0x01) << 8)) j = in_cr(0x04) + ((i & 0x10) << 4) + 4;
			else j = in_cr(0x00) + ((i & 0x01) << 8) + 1;
		}
		out_cr(0x3b, j);
		i |= (j & 0x100) >> 2;
		out_cr(0x3c, (in_cr(0x00) + ((i & 0x01) << 8)) / 2);
		out_cr(0x5d, (in_cr(0x5d) & 0x80) | i);
	}
	if (pp.flags & VIDEO_PARAMS_INTERLACED) out_cr(0x42, in_cr(0x42) | 0x20);
	else out_cr(0x42, in_cr(0x42) & ~0x20);
	/*{
		out_cr(0x38, 0x48);
		out_cr(0x39, 0xa5);
		out_cr(0x58, in_cr(0x58) & ~0x14);
		out_cr(0x40, 0x01);
		out_cr(0x53, in_cr(0x53) & ~0x0f);
		out_cr(0x54, 0xa0);
		io_outw(0xbee8, 0xe000);
		io_outw(0xaae8, 0xffff);
		io_outw(0xaee8, 0xffff);
		__debug_printf("%04x\n", io_inw(0x9ae8));
	}*/
}

static long s3_get_pixel_clock(struct video_mode *v, unsigned px)
{
	unsigned clk;
	if (__unlikely(!dac)) return -ENXIO;
	clk = dac->get_pixel_clock(v->bpp, px);
	if (__unlikely(!clk) || __unlikely(clk > MAXINT)) return -EINVAL;
	return clk;
}

/*
   Some S3 Vesa bioses have this function seriously broken --- waitretrace flips
   the image immediatelly AFTER the retrace, so the old image is displayed for
   one more frame. If the CPU is fast enough to write to the old page, screen
   flicker happens in mplayer. --- I reimplemented it correctly.
*/

static int s3_set_display_start(struct video_mode *v, unsigned addr, int waitretrace)
{
	KERNEL$DI();	/* Minimize the probability that incorrect values are
			sampled by the card */
	out_cr(0x38, 0x48);
	out_cr(0x39, 0xa5);
	out_cr(0x0d, (addr & 0x3fc) >> 2);
	out_cr(0x0c, (addr & 0x3fc00) >> 10);
#if 0		/* that trick with ATC doesn't seem to work */
	io_inb(VGA_BASE_COLOR + VGA_INPUT_REG);
	io_outb(VGA_ATC_IDX_REG, VGA_ATC_H_PIXEL_PANNING | VGA_ATC_IDX_REG_PAS);
	io_outb(VGA_ATC_IDX_REG, (io_inb(VGA_ATC_REG_READ) & 0xf0) | ((addr & 3) << 1));
#endif
	io_outb(io_base + VGA_CRTC_REG, (in_cr(0x31) & 0xcf) | (addr & 0xc0000) >> 14);
	if (__likely(s3_chiptype >= S3_801)) {
		io_outb(io_base + VGA_CRTC_REG, (in_cr(0x51) & 0xfc) | (addr & 0x300000) >> 20);
	}
	KERNEL$EI();
	return waitretrace;
}


#define S3V_REGOFFS	0x01000000
#define S3V_REGSIZE	0x00010000

#define INREG(r)	__32LE2CPU(mmio_inl(pci_regs + (r)))
#define OUTREG(r, v)	mmio_outl(pci_regs + (r), v)

static void s3v_error(char *str)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "%s, STATUS %08X, ADV CTRL %08X", str, (unsigned)INREG(0x8504), (unsigned)INREG(0x850c));
}

#define WAITFIFO(n, err)						\
do {									\
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;			\
	while (jj = KERNEL$GET_JIFFIES_LO(), __likely(pci_chip_type != VIRGE_GX_2) ? __unlikely(((INREG(0x8504) >> 8) & 0x1f) < (n)) : __unlikely(!(INREG(0x8504 >> 9) & 0x60))) {							\
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_VIRGE_FIFO_TIMEOUT))) {									\
			s3v_error("TIMEOUT WAITING FOR FIFO");		\
			err;						\
			break;						\
		}							\
	}								\
} while (0)

#define WAITCMD(err)							\
do {									\
	if (__unlikely(pci_chip_type == VIRGE_GX_2)) {			\
		u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;		\
		while (jj = KERNEL$GET_JIFFIES_LO(), __unlikely((INREG(0x850c) & 0x07c0) != 0x0400)) {							\
			if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_VIRGE_FIFO_TIMEOUT))) {								\
				s3v_error("TIMEOUT WAITING FOR COMMAND");\
				err;					\
				break;					\
			}						\
		}							\
	}								\
} while (0)

#define WAITIDLE(err)							\
do {									\
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO(), jj;			\
	while (jj = KERNEL$GET_JIFFIES_LO(), __unlikely((INREG(0x8504) & 0x3f00) < 0x3000)) {								\
		if (__unlikely(jj - j > TIMEOUT_JIFFIES(S3_VIRGE_FIFO_TIMEOUT))) {							\
			s3v_error("TIMEOUT WAITING FOR IDLE");		\
			err;						\
			break;						\
		}							\
	}								\
} while (0)

static int s3v_cmd_bpp;
static int s3v_patch_32;
/*static int s3v_last_color;*/

static long s3v_accel_avail(struct video_mode *v)
{
	if ((v->x << (v->bpp == 32)) >= 2048 || v->y >= 2048) return 0;
	if (v->bpp != 8 && v->bpp != 15 && v->bpp != 16 && v->bpp != 24 && v->bpp != 32) return 0;
	return (pci_chip_type != VIRGE_TRIO_3D ? ACCELFLAG_FILLBOX | ACCELFLAG_SETFGCOLOR : 0) | ACCELFLAG_SCREENCOPY | ACCELFLAG_SETMODE | ACCELFLAG_SYNC;
}

static void s3v_accel_init(struct video_mode *v)
{
	unsigned stride, scis;
	int cb;
	s3v_cmd_bpp = -1;
	s3v_patch_32 = 0;
	if ((v->x << (v->bpp == 32)) >= 2048 || v->y >= 2048) return;
	if (v->bpp == 8) cb = 0;
	else if (v->bpp == 15 || v->bpp == 16) cb = 4;
	else if (v->bpp == 24) cb = 8;
	else if (v->bpp == 32) cb = 4, s3v_patch_32 = 1;
	else return;
	out_cr(0x53, 0x08);
	WAITFIFO(6, return);
	OUTREG(0xa4d4, 0);
	OUTREG(0xa4d8, 0);
	stride = v->scanline;
	OUTREG(0xa4e4, stride | (stride << 16));
	OUTREG(0xa4dc, v->x << s3v_patch_32);
	scis = v->y * v->images;
	if (scis > 2047) scis = 2047;
	OUTREG(0xa4e0, scis);
	WAITCMD(return);
	OUTREG(0xa500, 0x78000000);
	s3v_cmd_bpp = cb;
	/*s3v_last_color = -1;*/
}

static void *s3v_accel_do(struct video_mode *v, unsigned op, union accel_param *ap)
{
	int command = s3v_cmd_bpp;
	if (__unlikely(command == -1)) return __ERR_PTR(-EOPNOTSUPP);
	__write_barrier2();
	if (__likely((op & _ACCEL_OP) == ACCEL_FILLBOX)) {
		if (__unlikely(pci_chip_type == VIRGE_TRIO_3D)) return __ERR_PTR(-EOPNOTSUPP);
		command |= 0x17e00120;
		if (__unlikely(s3v_patch_32)) {
			ap->fillbox.w <<= 1;
			ap->fillbox.x <<= 1;
			if (__unlikely(((ap->fillbox.c >> 16) & 0xff) != (ap->fillbox.c & 0xff))) return __ERR_PTR(-EINVAL);
		}
		/*if (__unlikely(ap->fillbox.c != s3v_last_color)) {*/
/* Without this, rare pixel corruption will occur. (for example links -g -mode
 * 800x600x16m32 www.usenix.org) Is it a bug in the engine? I don't know, I have
 * no docs for it. But pixel corruption also happens under X sometimes. */
/* Update: microscopic pixel corruption (when horizontally scrolling on
 * comix.spaceport.cz) happens even when there are more requests pending with
 * the same color. Only this unconditional WAITIDLE fixes it (and kills
 * performance :-( ). Attepmpts to add WAITIDLE on scrolling were unsuccessful.
 */
			WAITIDLE(return __ERR_PTR(-EIO));
		/*	s3v_last_color = ap->fillbox.c;
		} else {
			WAITFIFO(4, return __ERR_PTR(-EIO));
		}*/
		OUTREG(0xa4f4, ap->fillbox.c);
		OUTREG(0xa504, ((ap->fillbox.w - 1) << 16) | ap->fillbox.h);
		OUTREG(0xa50c, (ap->fillbox.x << 16) | ap->fillbox.y);
		WAITCMD(return __ERR_PTR(-EIO));
		OUTREG(0xa500, command);
	} else if (__likely((op & _ACCEL_OP) == ACCEL_SCREENCOPY)) {
		command |= 0x07980020;
		if (__unlikely(s3v_patch_32)) {
			ap->screencopy.w <<= 1;
			ap->screencopy.x1 <<= 1;
			ap->screencopy.x2 <<= 1;
		}
		if (ap->screencopy.x1 < ap->screencopy.x2) {
			ap->screencopy.x1 += ap->screencopy.w - 1;
			ap->screencopy.x2 += ap->screencopy.w - 1;
			command &= ~0x02000000;
		}
		if (ap->screencopy.y1 < ap->screencopy.y2) {
			ap->screencopy.y1 += ap->screencopy.h - 1;
			ap->screencopy.y2 += ap->screencopy.h - 1;
			command &= ~0x04000000;
		}
		WAITFIFO(4, return __ERR_PTR(-EIO));
		OUTREG(0xa504, ((ap->screencopy.w - 1) << 16) | ap->screencopy.h);
		OUTREG(0xa508, (ap->screencopy.x1 << 16) | ap->screencopy.y1);
		OUTREG(0xa50c, (ap->screencopy.x2 << 16) | ap->screencopy.y2);
		WAITCMD(return __ERR_PTR(-EIO));
		OUTREG(0xa500, command);
	} else if (__likely((op & _ACCEL_OP) == ACCEL_SYNC)) {
		WAITIDLE(return __ERR_PTR(-EIO));
		return NULL;
	} else {
		return __ERR_PTR(-EOPNOTSUPP);
	}
	if (!(op & _ACCEL_ASYNC)) {
		WAITIDLE(return NULL);
	}
	return NULL;
}

static int s3_unload(void *p, void **release, char *argv[]);

static void *dlrq, *lnte;

int main(int argc, char *argv[])
{
	int r;
	DEVICE_REQUEST devrq;
	char **arg = argv;
	int state = 0;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}
	*KERNEL$ERROR_MSG() = 0;
	if (__unlikely(r = s3_probe())) {
		if (!*KERNEL$ERROR_MSG()) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: S3 CARD NOT FOUND");
		goto ret0;
	}
	if (accel) {
		DLLZRQ lz;
		char *chip_name;
		char gstr[__MAX_STR_LEN];
		__u64 mem_res;
		lz.handle = KERNEL$DLRQ();
		lz.caller = NULL;
		strcpy(lz.modname, "PCI");
		SYNC_IO_CANCELABLE(&lz, KERNEL$DL_LOAD_LAZY);
		if (__unlikely(lz.status < 0)) {
			goto accel_done;
		}
		if (PCI$FIND_DEVICE(s3v_pci_cards, 0, 0, 0, PCI$TEST_LIST, &pci_id, &chip_name, &pci_chip_type, 0)) goto no_s3v;
		PCI$ENABLE_DEVICE(pci_id, PCI_COMMAND_IO | PCI_COMMAND_MEMORY);
		pci_valid = 1;
		mem_res = PCI$READ_MEM_RESOURCE(pci_id, 0, NULL);
		if (!mem_res) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
			goto accel_done;
		}
		pci_regs = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(mem_res + S3V_REGOFFS, S3V_REGSIZE, PAT_UC);
		if (!pci_regs) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "UNABLE TO MAP MEM RESOURCE");
			goto accel_done;
		}
		_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, pci_id));
		_printf("%s: FRAMEBUFFER %"__64_format"X\n", dev_name, mem_res);
		s3_functions[0].ptr = s3v_accel_avail;
		s3_functions[1].ptr = s3v_accel_init;
		s3_functions[2].ptr = s3v_accel_do;
		goto accel_done;
		no_s3v:;
	}

	accel_done:

	if (!pci_valid) {
		_printf("%s: %s\n", dev_name, s3_chipname[s3_chiptype]);
	}

	if ((r = KERNEL$DCALL(cons, "CONSOLE", CONSOLE_CMD_INSTALL_DRIVER, (void *)s3_functions))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "S3: CAN'T ATTACH TO CONSOLE DEVICE %s", cons);
		goto ret0;
	}
	devrq.name = dev_name;
	devrq.driver_name = "S3.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = NULL;
	devrq.dev_ptr = NULL;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = s3_unload;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = devrq.status;
		goto ret1;
	}
	lnte = devrq.lnte;
	dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;

	ret1:
	KERNEL$DCALL(cons, "CONSOLE", CONSOLE_CMD_RELEASE_DRIVER, NULL);
	ret0:
	if (pci_regs) KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(pci_regs, S3V_REGSIZE);
	if (pci_valid) PCI$FREE_DEVICE(pci_id);
	return r;
}

static int s3_unload(void *p, void **release, char *argv[])
{
	int r;
	if ((r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	KERNEL$DCALL(cons, "CONSOLE", CONSOLE_CMD_RELEASE_DRIVER, NULL);
	if (pci_regs) KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(pci_regs, S3V_REGSIZE);
	if (pci_valid) PCI$FREE_DEVICE(pci_id);
	*release = dlrq;
	return 0;
}
