#include <SPAD/VFS.H>
#include <SPAD/SYSLOG.H>

#include "STRUCT.H"
#include "CDFS.H"

static __const__ FSOPS cdfsops = {
	DEFAULT_VFSID,
	"CDFS",
	FS_RO_ONLY,
	sizeof(CDFS),
	sizeof(CDFNODE),
	VFS$FNODE_CTOR,
	sizeof(CDPAGEINRQ),
	VFS$PAGEINRQ_CTOR,
	CDFS_PROCESS_OPTION,
	CDFS_MOUNT,
	NULL,			/* sync_background_init */
	CDFS_UMOUNT,
	NULL,			/* set_dirty */
	NULL,			/* validate_filename */
	CDFS_LOOKUP,
	sizeof(unsigned),
	CDFS_INIT_READDIR_COOKIE,
	CDFS_READDIR,
	VFS$DEFAULT_WRITEPAGES,	/* writepages */
	VFS$DEFAULT_COMMIT,
	NULL,			/* init_fnode */
	NULL,			/* create_fnode */
	NULL,			/* write_fnode_data */
	NULL,			/* add_fnode_to_directory */
	NULL,			/* delete_fnode */
	NULL,			/* remove_fnode_from_directory */
	VFS$NULL_FNODE_CALL,	/* free_fnode */
	CDFS_BMAP,
	CDFS_SYNC_BMAP,
	NULL,			/* init_last_page */
	NULL,			/* account */
	NULL,			/* unaccount */
	CDFS_STAT,
	CDFS_STATFS,
};

int main(int argc, char *argv[])
{
	return VFS$MOUNT(argc, argv, &cdfsops, "CDFS.SYS");
}

int CDFS_PROCESS_OPTION(FS *fs_, char *opt, char *optend, char *str)
{
	CDFS *fs = (CDFS *)fs_;
	if (!__strcasexcmp("NOJOLIET", opt, optend) && !str) {
		fs->flags |= CDFS_DISABLE_JOLIET;
		fs->flags &= ~CDFS_PREFER_JOLIET;
	} else if (!__strcasexcmp("JOLIET", opt, optend) && !str) {
		fs->flags &= ~CDFS_DISABLE_JOLIET;
		fs->flags |= CDFS_PREFER_JOLIET;
	} else if (!__strcasexcmp("NOROCK", opt, optend) && !str) {
		fs->flags |= CDFS_DISABLE_ROCK_RIDGE;
		fs->flags |= CDFS_PREFER_JOLIET;
	} else if (!__strcasexcmp("ROCK", opt, optend) && !str) {
		fs->flags &= ~CDFS_DISABLE_ROCK_RIDGE;
		fs->flags &= ~CDFS_PREFER_JOLIET;
	} else {
		return 1;
	}
	return 0;
}

int CDFS_MOUNT(FS *fs_)
{
	CDFS *fs = (CDFS *)fs_;
	unsigned volume_sector;
	int r;
	ISO_DESCRIPTOR *pri, *supp;
	HS_DESCRIPTOR *hs;
	int have_pri, have_supp, have_hs;
	int joliet_level;
	unsigned root_sector;
	unsigned root_length;
	CDFNODE *f;
	time_t root_ctime, root_mtime;
	if (__unlikely(sizeof(ISO_DESCRIPTOR) != CDFS_SECTOR_SIZE) || __unlikely(sizeof(HS_DESCRIPTOR) != CDFS_SECTOR_SIZE))
		KERNEL$SUICIDE("CDFS: MISCOMPILED DRIVER, SIZEOF(ISO_DESCRIPTOR) %d, SIZEOF(HS_DESCRIPTOR) %d, SECTOR SIZE %d", (int)sizeof(ISO_DESCRIPTOR), (int)sizeof(HS_DESCRIPTOR), CDFS_SECTOR_SIZE);
	pri = __sync_malloc(sizeof(ISO_DESCRIPTOR));
	if (__unlikely(!pri)) {
		r = -ENOMEM;
		goto ret0;
	}
	supp = __sync_malloc(sizeof(ISO_DESCRIPTOR));
	if (__unlikely(!supp)) {
		r = -ENOMEM;
		goto ret1;
	}
	hs = __sync_malloc(sizeof(HS_DESCRIPTOR));
	if (__unlikely(!hs)) {
		r = -ENOMEM;
		goto ret2;
	}
	have_pri = 0;
	have_supp = 0;
	have_hs = 0;
	joliet_level = 0;
	for (volume_sector = CDFS_VOLUME_START; volume_sector < CDFS_VOLUME_MAX; volume_sector++) {
		ISO_DESCRIPTOR *desc;
		HS_DESCRIPTOR *hs_desc;
		desc = VFS$READ_BUFFER_SYNC((FS *)fs, CDFS_GET_SECTOR(volume_sector), CDFS_SECTOR_N, fs->buffer_readahead, (void *)&KERNEL$LIST_END);
		if (__unlikely(__IS_ERR(desc))) {
			r = __PTR_ERR(desc);
			goto ret3;
		}
		hs_desc = (HS_DESCRIPTOR *)desc;
		if (__likely(!memcmp(desc->id, ISO_ID, sizeof desc->id))) {
			if (__likely(desc->type == ISO_TYPE_PRIMARY)) {
				if (!have_pri) {
					memcpy(pri, desc, sizeof(ISO_DESCRIPTOR));
					have_pri = 1;
				}
			} else if (__likely(desc->type == ISO_TYPE_SUPPLEMENTARY) && __likely(!(fs->flags & CDFS_DISABLE_JOLIET))) {
				if (!have_supp && desc->supplementary_escape_sequences[0] == '%' && desc->supplementary_escape_sequences[1] == '/') {
					if (desc->supplementary_escape_sequences[2] == '@') joliet_level = 1;
					if (desc->supplementary_escape_sequences[2] == 'C') joliet_level = 2;
					if (desc->supplementary_escape_sequences[2] == 'E') joliet_level = 3;
					if (joliet_level) {
						memcpy(supp, desc, sizeof(ISO_DESCRIPTOR));
						have_supp = 1;
					}
				}
			} else if (__likely(desc->type == ISO_TYPE_END)) {
				VFS$PUT_BUFFER(desc);
				break;
			}
		} else if (!memcmp(hs_desc->id, HS_ID, sizeof hs_desc->id)) {
			if (hs_desc->type == HS_TYPE_PRIMARY) {
				if (!have_hs) {
					memcpy(hs, hs_desc, sizeof(HS_DESCRIPTOR));
					have_hs = 1;
				}
			}
		}
		VFS$PUT_BUFFER(desc);
		if ((have_pri && have_supp) || have_hs) break;
	}
	if (__unlikely(!have_pri) && __unlikely(!have_hs)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO DESCRIPTOR FOUND", fs->filesystem_name);
		r = -EINVAL;
		goto ret3;
	}
	if (have_pri) {
		int rock_ok = 0;
		SUSP *susp;
		DIRECTORY_RECORD *dir1 = (DIRECTORY_RECORD *)&pri->root_directory;
		DIRECTORY_RECORD *dir;
		SUSP_STATE rock;
		dir = VFS$READ_BUFFER_SYNC((FS *)fs, CDFS_GET_SECTOR(DIRENT_SECTOR(dir1)), CDFS_SECTOR_N, fs->buffer_readahead, (void *)&KERNEL$LIST_END);
		if (__unlikely(__IS_ERR(dir))) {
			if (__PTR_ERR(dir) == -EINTR) goto ret_dir;
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNABLE TO READ PRIMARY ROOT DIRECTORY AT %"__d_off_format"X: %s", CDFS_GET_SECTOR(DIRENT_SECTOR(dir1)), strerror(-__PTR_ERR(dir)));
			if (!have_supp) {
				ret_dir_snprintf:
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO READ ROOT DIRECTORY", fs->filesystem_name);
				ret_dir:
				r = __PTR_ERR(dir);
				goto ret3;
			}
			have_pri = 0;
			goto try_supp;
		}
		root_ctime = root_mtime = CDFS_TIME(dir->date, CDFS_TIME_SHORT);
		INIT_SUSP_STATE(fs, &rock, dir, DIRENT_SECTOR(dir), 0);
		VFS$PUT_BUFFER(dir);
		susp_next_area:
		while ((susp = GET_SUSP_ENTRY(fs, &rock))) {
			if (TEST_SUSP(susp, 'S', 'P', 1)) {
				SUSP_SP *susp_sp = (SUSP_SP *)susp;
				if (__unlikely(susp_sp->len != sizeof(SUSP_SP))) continue;
				if (__unlikely(susp_sp->check_1 != SUSP_SP_CHECK_1) || __unlikely(susp_sp->check_2 != SUSP_SP_CHECK_2)) continue;
				fs->susp_offset = susp_sp->susp_offset;
			} else if (TEST_SUSP(susp, 'T', 'F', 1)) {
				if (__unlikely(ROCK_TIME(susp, &root_ctime, &root_mtime)))
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD ROCK RIDGE TIMESTAMP IN PRIMARY ROOT DIRECTORY AT %"__d_off_format"X", CDFS_GET_SECTOR(DIRENT_SECTOR(dir1)));
				rock_ok = 1;
			} else if (TEST_SUSP(susp, 'P', 'X', 1) || TEST_SUSP(susp, 'R', 'R', 1) || TEST_SUSP(susp, 'N', 'M', 1)) {
				rock_ok = 1;
			}
		}
		if (rock.cont_sector) {
			__u8 *sec = VFS$READ_BUFFER_SYNC((FS *)fs, CDFS_GET_SECTOR(rock.cont_sector), CDFS_SECTOR_N, fs->buffer_readahead, (void *)&KERNEL$LIST_END);
			if (__unlikely(__IS_ERR(sec))) {
				if (__PTR_ERR(sec) == -EINTR) {
					r = __PTR_ERR(sec);
					goto ret3;
				}
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNABLE TO READ ROOT DIRECTORY'S SUSP AREA AT %"__d_off_format"X: %s", CDFS_GET_SECTOR(rock.cont_sector), strerror(-__PTR_ERR(dir)));
				goto no_more_susp;
			}
			SUSP_CONTINUE(fs, &rock, sec);
			VFS$PUT_BUFFER(sec);
			goto susp_next_area;
		}
		no_more_susp:
		if (rock_ok || !have_supp) {
			if (__likely(have_supp) && __unlikely(fs->flags & CDFS_PREFER_JOLIET)) goto try_supp;
			try_pri:
			root_sector = DIRENT_SECTOR(dir1);
			root_length = GET_LE_32_16(dir1->size_le);
			if (rock_ok) fs->flags |= CDFS_ROCK_RIDGE;
			fs->size = CDFS_GET_SECTOR(__32LE2CPU(pri->volume_space_size_le));
			if (__unlikely(fs->size >> CDFS_SECTOR_SHIFT != __32LE2CPU(pri->volume_space_size_le))) {
				too_large:
				_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FILESYSTEM TOO LARGE, RECOMPILE VFS WITH 64-BIT __d_off", fs->filesystem_name);
				r = -EINVAL;
				goto ret3;
			}
		} else {
			DIRECTORY_RECORD *dir2;
			try_supp:
			dir2 = (DIRECTORY_RECORD *)&supp->root_directory;
			dir = VFS$READ_BUFFER_SYNC((FS *)fs, CDFS_GET_SECTOR(DIRENT_SECTOR(dir2)), CDFS_SECTOR_N, fs->buffer_readahead, (void *)&KERNEL$LIST_END);
			if (__unlikely(__IS_ERR(dir))) {
				if (__PTR_ERR(dir) == -EINTR) goto ret_dir;
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNABLE TO READ SUPPLEMENTARY ROOT DIRECTORY AT %"__d_off_format"X: %s", CDFS_GET_SECTOR(DIRENT_SECTOR(dir2)), strerror(-__PTR_ERR(dir)));
				if (have_pri) goto try_pri;
				goto ret_dir_snprintf;
			}
			root_sector = DIRENT_SECTOR(dir2);
			root_length = GET_LE_32_16(dir2->size_le);
			fs->flags |= CDFS_JOLIET;
			root_ctime = root_mtime = CDFS_TIME(dir->date, CDFS_TIME_SHORT);
			VFS$PUT_BUFFER(dir);
			fs->size = CDFS_GET_SECTOR(__32LE2CPU(supp->volume_space_size_le));
			if (__unlikely(fs->size >> CDFS_SECTOR_SHIFT != __32LE2CPU(supp->volume_space_size_le))) {
				goto too_large;
			}
		}
	} else {
		/* High Sierra */
		DIRECTORY_RECORD *dir3 = (DIRECTORY_RECORD *)&hs->root_directory;
		DIRECTORY_RECORD *dir;
		dir = VFS$READ_BUFFER_SYNC((FS *)fs, CDFS_GET_SECTOR(DIRENT_SECTOR(dir3)), CDFS_SECTOR_N, fs->buffer_readahead, (void *)&KERNEL$LIST_END);
		if (__unlikely(__IS_ERR(dir))) {
			r = __PTR_ERR(dir);
			if (__PTR_ERR(dir) == -EINTR) goto ret3;
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNABLE TO READ HIGH SIERRA ROOT DIRECTORY AT %"__d_off_format"X: %s", CDFS_GET_SECTOR(DIRENT_SECTOR(dir3)), strerror(-__PTR_ERR(dir)));
			goto ret3;
		}
		root_sector = DIRENT_SECTOR(dir3);
		root_length = GET_LE_32_16(dir3->size_le);
		fs->flags |= CDFS_HIGH_SIERRA;
		root_ctime = root_mtime = CDFS_TIME(dir->date, CDFS_TIME_HIGH_SIERRA);
		VFS$PUT_BUFFER(dir);
		fs->size = CDFS_GET_SECTOR(__32LE2CPU(hs->volume_space_size_le));
		if (__unlikely(fs->size >> CDFS_SECTOR_SHIFT != __32LE2CPU(hs->volume_space_size_le))) {
			goto too_large;
		}
	}
	/*__debug_printf("got desc..., flags %08x, root %08x, length %08x\n", fs->flags, root_sector, root_length);*/

	fs->max_filesize = 0xFFFFFFFFU & __NOT_PAGE_CLUSTER_SIZE_MINUS_1;
	fs->pageio_mask = CDFS_SECTOR_SIZE - 1;
	fs->pageio_bits = CDFS_SECTOR_SIZE_BITS;
	fs->block_mask = CDFS_SECTOR_N - 1;

	f = (CDFNODE *)VFS$GET_ROOT_FNODE((FS *)fs);
	f->flags |= FNODE_DIRECTORY;
	VFS_INIT_DIR((FNODE *)f);
	f->ctime = root_ctime;
	f->mtime = root_mtime;
	f->sector = root_sector;
	f->offset = 0;
	f->size = root_length;
	f->disk_size = root_length;

	r = 0;

	ret3:
	free(hs);
	ret2:
	free(supp);
	ret1:
	free(pri);
	ret0:
	return r;
}

void CDFS_UMOUNT(FS *fs_, int nosync)
{
}

void CDFS_STAT(FNODE *f_, struct stat *stat)
{
}

WQ *CDFS_STATFS(FS *fs_, struct statfs *stat)
{
	CDFS *fs = (CDFS *)fs_;
	stat->f_blocks = fs->size >> CDFS_SECTOR_SHIFT;
	stat->f_namelen = fs->flags & CDFS_ROCK_RIDGE ? 255 : fs->flags & CDFS_JOLIET ? 63 : 30;
	return NULL;
}
