#include <SPAD/SYSLOG.H>
#include <VALUES.H>

#include "STRUCT.H"
#include "CDFS.H"

extern AST_STUB CDFS_AST;

static __finline__ unsigned CDFS_UNICODE_XLATE(CDFS *fs, __u8 *unicode, int unicode_len, char *result, int result_len)
{
	unsigned ret = 0;
	/* !!! TODO: translate unicode here */
	while (ret < result_len && unicode_len > 1) {
		result[ret] = unicode[1];
		if (__unlikely(IS_FORBIDDEN_CHAR(result[ret]))) result[ret] += 0x80;
		unicode += 2;
		unicode_len -= 2;
		ret++;
	}
	return ret;
}

static unsigned CDFS_REMOVE_VERSION(char *name, unsigned namelen)
{
	unsigned i = namelen;
	while (i) {
		i--;
		if (name[i] == ';') return i;
		if (name[i] < '0' || name[i] > '9') return namelen;
	}
	return namelen;
}

static __finline__ void CDFS_GET_FNODE(CDFNODE *f, CDPAGEINRQ *p, DIRECTORY_RECORD *dirent, __u8 dflags, int nametype)
{
	f->ctime = p->ctime;
	f->mtime = p->mtime;
	FNODE_OUT_OF_WANTFREE((FNODE *)f);
	if (__unlikely(nametype == DT_DIR)) {
		f->sector = DIRENT_SECTOR(dirent);
		f->size = f->disk_size = GET_LE_32_16(dirent->size_le);
		f->offset = 0;
		if (!(f->flags & FNODE_DIRECTORY)) {
			VFS_INIT_DIR((FNODE *)f);
		}
		f->flags = FNODE_DIRECTORY;
	} else {
		f->sector = p->first_extent_sector;
		f->offset = p->first_extent_offset;
		f->file_desc_length = ((p->sector - p->first_extent_sector) << CDFS_SECTOR_SIZE_BITS) + p->offset + dirent->length - p->first_extent_offset;
		f->size = f->disk_size = p->accumulated_size;
		f->run_length = 0;
		f->flags = FNODE_FILE;
	}
}

static __finline__ void CDPAGEINRQ_INIT_COMMON(CDPAGEINRQ *p)
{
	p->dirent_residue_length = 0;
	p->multi_extent = 0;
	p->susp_active = 0;
	p->reading_dir = 0;
}

void CDFS_LOOKUP(PAGEINRQ *p_)
{
	CDPAGEINRQ *p = (CDPAGEINRQ *)p_;
	CDFNODE *f = (CDFNODE *)p->fnode;
	p->fn = CDFS_AST;
	p->type = PG_LOOKUP;
	p->sector = f->sector;
	p->offset = 0;
	p->remaining = (unsigned)f->disk_size;
	CDPAGEINRQ_INIT_COMMON(p);
	CALL_AST(p);
}

void CDFS_INIT_READDIR_COOKIE(struct readdir_buffer *rb, FNODE *f_)
{
	unsigned *c = (unsigned *)rb->cookie;
	rb->cookie_size = sizeof(unsigned);
	*c = 0;
}

void CDFS_READDIR(PAGEINRQ *p_, struct readdir_buffer *rb)
{
	CDPAGEINRQ *p = (CDPAGEINRQ *)p_;
	CDFNODE *f = (CDFNODE *)p->fnode;
	unsigned c;
	if (__unlikely(!rb))
		KERNEL$SUICIDE("CDFS_READDIR: READDIR CALLED WITH NULL BUFFER");
	c = *(unsigned *)rb->cookie;
	if (__unlikely(c >= (unsigned)f->disk_size)) {
		IORQ *caller;
		rb->end = 1;
		if (__unlikely(VFS$DONE_READDIR((PAGEINRQ *)p)))
			KERNEL$SUICIDE("CDFS_READDIR: VFS$DONE_READDIR RETURNED NON-ZERO");
		caller = VFS$FREE_PAGEIN((PAGEINRQ *)p);
		caller->status = 0;
		CALL_AST(caller);
		return;
	}
	p->fn = CDFS_AST;
	p->type = PG_READDIR;
	p->sector = f->sector + (c >> CDFS_SECTOR_SIZE_BITS);
	p->offset = c & (CDFS_SECTOR_SIZE - 1);
	p->remaining = (unsigned)f->disk_size - c;
	CDPAGEINRQ_INIT_COMMON(p);
	CALL_AST(p);
}

void CDFS_BMAP(PAGEINRQ *p_)
{
	CDPAGEINRQ *p = (CDPAGEINRQ *)p_;
	CDFNODE *f = (CDFNODE *)p->fnode;
	p->fn = CDFS_AST;
	p->type = PG_BMAP;
	p->sector = f->sector;
	p->offset = f->offset;
	p->remaining = f->file_desc_length;
	p->accumulated_size = 0;
	CDPAGEINRQ_INIT_COMMON(p);
	CALL_AST(p);
}

static void PROCESS_SUSP_ENTRIES(CDPAGEINRQ *RQ);

DECL_AST(CDFS_AST, SPL_FS, CDPAGEINRQ)
{
#define f ((CDFNODE *)RQ->fnode)
#define fs ((CDFS *)RQ->fs)
	__u8 *data;
	unsigned want_sector, advance;
	DIRECTORY_RECORD *dirent;
	int err;
	IORQ *caller;
	__u8 dflags;
	unsigned size;
	void *map;
	unsigned maplen;
	void (*unmap)(void *ptr);

	IO_DISABLE_CHAIN_CANCEL(SPL_FS, RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status < 0)) {
		err = RQ->status;
		goto error;
	}

	if (0) {
		next:
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
			lockup:
			WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
			RETURN;
		}
	}

	if (__unlikely(RQ->susp_active == 1)) {
		if (__unlikely(!(data = VFS$GET_BUFFER((FS *)fs, CDFS_GET_SECTOR(RQ->st.cont_sector), CDFS_SECTOR_N, (void *)&KERNEL$LIST_END, RQ->tag.proc)))) {
			VFS$READ_BUFFER(CDFS_GET_SECTOR(RQ->st.cont_sector), (PAGEINRQ *)RQ, fs->buffer_readahead);
			RETURN;
		}
		SUSP_CONTINUE(fs, &RQ->st, data);
		VFS$PUT_BUFFER(data);
		PROCESS_SUSP_ENTRIES(RQ);
		if (__unlikely(RQ->st.cont_sector != 0)) {
			goto next;
		}
		RQ->susp_active = 2;
	}

	want_sector = RQ->sector;
	if (__unlikely(RQ->dirent_residue_length)) {
		want_sector++;
		if (__unlikely(!want_sector)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "SECTOR NUMBER WRAP-AROUND");
			err = -EFSERROR;
			goto error;
		}
	}
	if (__unlikely(!(data = VFS$GET_BUFFER((FS *)fs, CDFS_GET_SECTOR(want_sector), CDFS_SECTOR_N, (void *)&KERNEL$LIST_END, RQ->tag.proc)))) {
		VFS$READ_BUFFER(CDFS_GET_SECTOR(want_sector), (PAGEINRQ *)RQ, fs->buffer_readahead);
		RETURN;
	}
	if (__unlikely(RQ->dirent_residue_length)) {
		unsigned dir_len = ((DIRECTORY_RECORD *)RQ->dirent_residue)->length;
		memcpy(RQ->dirent_residue + RQ->dirent_residue_length, data, dir_len - RQ->dirent_residue_length);
		dirent = (DIRECTORY_RECORD *)RQ->dirent_residue;
	} else {
		dirent = (DIRECTORY_RECORD *)(data + RQ->offset);
		if (__unlikely(dirent->length & 1) || (__likely(dirent->length) && __unlikely(dirent->length < sizeof(DIRECTORY_RECORD)))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIRECTORY ENTRY AT %"__d_off_format"X:%X HAS INVALID LENGTH %02X", CDFS_GET_SECTOR(RQ->sector), RQ->offset, dirent->length);
			err = -EFSERROR;
			goto error_put_buffer;
		}
		if (__unlikely(dirent->length > RQ->remaining)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIRECTORY ENTRY AT %"__d_off_format"X:%X ENDS BEYOND DIRECTORY END", CDFS_GET_SECTOR(RQ->sector), RQ->offset);
			err = -EFSERROR;
			goto error_put_buffer;
		}
		if (__unlikely(RQ->offset + dirent->length > CDFS_SECTOR_SIZE)) {
			RQ->dirent_residue_length = CDFS_SECTOR_SIZE - RQ->offset;
			memcpy(RQ->dirent_residue, dirent, RQ->dirent_residue_length);
			VFS$PUT_BUFFER(data);
			goto next;
		}
	}
	if (!dirent->length) {
		advance = CDFS_SECTOR_SIZE - RQ->offset;
		goto do_advance_put_buffer;
	}
	if (__unlikely(__offsetof(DIRECTORY_RECORD, name) + dirent->name_len > dirent->length)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIRECTORY ENTRY AT %"__d_off_format"X:%X HAS TOO LONG NAME LENGTH, LENGTH %02X, NAME LENGTH %02X", CDFS_GET_SECTOR(RQ->sector), RQ->offset, dirent->length, dirent->name_len);
		err = -EFSERROR;
		goto error_put_buffer;
	}

	dflags = DIRENT_FLAGS(fs, dirent);

	if (RQ->type != PG_BMAP) {
		char *name;
		unsigned namelen;
		int nametype;
		unsigned i;

		if (__unlikely(dflags & DIR_FLAG_MULTI_EXTENT)) {
			if (__unlikely(dflags & DIR_FLAG_DIRECTORY)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "MULTI-EXTENT DIRECTORY ENTRY AT %"__d_off_format"X:%X, FLAGS %02X", CDFS_GET_SECTOR(RQ->sector), RQ->offset, dflags);
				err = -EFSERROR;
				goto error_put_buffer;
			}
			if (!RQ->multi_extent) {
				RQ->multi_extent = 1;
				RQ->first_extent_sector = RQ->sector;
				RQ->first_extent_offset = RQ->offset;
				RQ->accumulated_size = 0;
			}
			size = GET_LE_32_16(dirent->size_le);
			if (__unlikely(size & (CDFS_SECTOR_SIZE - 1))) {
				unaligned_size:
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "MULTI-EXTENT FILE ENTRY AT %"__d_off_format"X:%X IS NOT SECTOR-ALIGNED, LENGTH %X", CDFS_GET_SECTOR(RQ->sector), RQ->offset, size);
				err = -EFSERROR;
				goto error_put_buffer;
			}
			if (__unlikely(RQ->accumulated_size + size < RQ->accumulated_size)) {
				size_wrap:
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "SIZE OF MULTI-EXTENT FILE ENTRY AT %"__d_off_format"X:%X WRAPPED AROUND", CDFS_GET_SECTOR(RQ->sector), RQ->offset);
				err = -EFSERROR;
				goto error_put_buffer;
			}
			RQ->accumulated_size += size;
			goto skip_this_dirent;
		}

		if (!RQ->susp_active) {
			RQ->rock_used_flags = 0;
			RQ->ctime = RQ->mtime = CDFS_TIME(dirent->date, __unlikely(fs->flags & CDFS_HIGH_SIERRA) ? CDFS_TIME_HIGH_SIERRA : CDFS_TIME_SHORT);
			INIT_SUSP_STATE(fs, &RQ->st, dirent, RQ->sector, RQ->offset);
			PROCESS_SUSP_ENTRIES(RQ);
			if (__unlikely(RQ->st.cont_sector != 0)) {
				RQ->susp_active = 1;
				VFS$PUT_BUFFER(data);
				goto next;
			}
		}
		RQ->susp_active = 0;

		if (__likely(!RQ->multi_extent)) {
			RQ->first_extent_sector = RQ->sector;
			RQ->first_extent_offset = RQ->offset;
			RQ->accumulated_size = 0;
		} else {
			RQ->multi_extent = 0;
		}
		size = GET_LE_32_16(dirent->size_le);
		if (__unlikely(RQ->accumulated_size + size < RQ->accumulated_size)) {
			goto size_wrap;
		}
		RQ->accumulated_size += size;

		if (RQ->reading_dir) {
			RQ->reading_dir = 0;
			nametype = DT_DIR;
			goto name_is_ok;
		}

		if (__unlikely(RQ->rock_used_flags & ROCK_USED_HIDE)) goto skip_this_dirent;

		if (__unlikely(!dirent->name_len) || (__unlikely(dirent->name_len == 1) && __likely((__u8)dirent->name[0] <= 1))) goto skip_this_dirent;
		if (!(fs->flags & CDFS_JOLIET)) {
			if (__likely(RQ->rock_used_flags & ROCK_USED_NAME)) {
				name = RQ->name;
				namelen = RQ->namelen;
			} else {
				name = dirent->name;
				namelen = dirent->name_len;
				namelen = CDFS_REMOVE_VERSION(name, namelen);
				if (__likely(namelen) && __unlikely(name[namelen - 1] == '.')) namelen--;
			}
		} else {
			namelen = CDFS_UNICODE_XLATE(fs, dirent->name, dirent->name_len, RQ->name, sizeof RQ->name);
			name = RQ->name;
			namelen = CDFS_REMOVE_VERSION(name, namelen);
		}
		if (__unlikely(!namelen)) goto skip_this_dirent;
		for (i = 0; i < namelen; i++) if (__unlikely(IS_FORBIDDEN_CHAR(name[i]))) goto skip_this_dirent;

		if (__likely(!(RQ->rock_used_flags & ROCK_USED_CHILD_LINK))) {
			nametype = __unlikely(dflags & DIR_FLAG_DIRECTORY) ? DT_DIR : DT_REG;
		} else {
			nametype = DT_DIR;
		}

		if (__unlikely(RQ->type == PG_READDIR)) {
			map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, (FS *)fs, &maplen, &unmap);
			if (__unlikely(!map)) goto pagefault_put_buffer;
			if (__unlikely(map == (void *)1)) {
				err = 0;
				goto error_put_buffer;
			}
			if (__unlikely(__IS_ERR(map))) {
				err = __PTR_ERR(map);
				goto error_put_buffer;
			}
			err = VFS$DO_READDIR_NAME((PAGEINRQ *)RQ, map, maplen, name, namelen, nametype);
			if (__unlikely(err != 0)) {
				VFS$UNMAP_READDIR(map, unmap);
				goto error_put_buffer;
			}
			*(unsigned *)((struct readdir_buffer *)map)->cookie = ((RQ->sector - f->sector) << CDFS_SECTOR_SIZE_BITS) + RQ->offset + dirent->length;
			VFS$UNMAP_READDIR(map, unmap);
			if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS)) {
				VFS$PUT_BUFFER(data);
				if (RQ->tag.proc != &KERNEL$PROC_KERNEL) {
					err = -EINTR;
					goto error;
				}
				goto lockup;
			}
		} else {	/* PG_LOOKUP */
			CDFNODE *fn = (CDFNODE *)RQ->new_fnode;
			int was_dir;
			if (namelen != fn->namelen || _memcasecmp(name, fn->name, namelen)) goto skip_this_dirent;

			if (__unlikely(RQ->rock_used_flags & ROCK_USED_CHILD_LINK)) {
				RQ->sector = RQ->child_link;
				RQ->offset = 0;
				RQ->remaining = CDFS_SECTOR_SIZE;
				RQ->reading_dir = 1;
				VFS$PUT_BUFFER(data);
				goto next;
			}
			/* we don't need to read directory root unless it was created via rock-ridge child link
			if (nametype == DT_DIR) {
				RQ->sector = DIRENT_SECTOR(dirent);
				RQ->offset = 0;
				RQ->remaining = CDFS_SECTOR_SIZE;
				RQ->reading_dir = 1;
				VFS$PUT_BUFFER(data);
				goto next;
			}*/

			name_is_ok:
			fn = (CDFNODE *)RQ->new_fnode;
			was_dir = 0;
			if (fn->flags & FNODE_DIRECTORY) {
				was_dir = 1;
				if (__unlikely(nametype != DT_DIR)) {
					err = -ENOTDIR;
					goto error_put_buffer;
				}
			}
			RQ->fnode->readers--;
			WQ_WAKE_ALL(&RQ->fnode->wait);
			if (__unlikely(RQ->fnode->readers < 0))
				KERNEL$SUICIDE("CDFS_AST: %s: LOCK UNDERRUN: %d", VFS$FNODE_DESCRIPTION(RQ->fnode), RQ->fnode->readers);
			CDFS_GET_FNODE(fn, RQ, dirent, dflags, nametype);
			WQ_WAKE_ALL(&fn->wait);
			VFS$PUT_BUFFER(data);
			if (was_dir) {
				RQ->new_fnode = LIST_STRUCT(fn->u.d.clean.next, FNODE, dirent_entry);
				RQ->fnode = (FNODE *)fn;
				RQ->fnode->readers++;
				RQ->sector = fn->sector;
				RQ->offset = 0;
				RQ->remaining = (unsigned)fn->disk_size;
				CDPAGEINRQ_INIT_COMMON(RQ);
				goto next;
			}
			caller = VFS$FREE_EMPTY_PAGEIN((PAGEINRQ *)RQ);
			RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);
		}
	} else {	/* PG_BMAP */
		size = GET_LE_32_16(dirent->size_le);
		if (__likely(RQ->off < ((RQ->accumulated_size + size + CDFS_SECTOR_SIZE - 1) & ~(CDFS_SECTOR_SIZE - 1)) >> BIO_SECTOR_SIZE_BITS)) {
			unsigned stride, used, lbn;
			unsigned sectors_back, sectors_forward;
			unsigned extent_sectors, sector;
			if (__likely(!dirent->interleave_gap)) {
				stride = MAXINT;
				used = MAXINT;
			} else {
				if (__unlikely(!dirent->file_unit_size)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FILE ENTRY AT %"__d_off_format"X:%X HAS NON-ZERO INTERLAVE GAP (%02X) AND ZERO FILE UNIT SIZE", CDFS_GET_SECTOR(f->sector), f->offset, dirent->interleave_gap);
					err = -EFSERROR;
					goto error_put_buffer;
				}
				stride = dirent->file_unit_size + dirent->interleave_gap;
				used = dirent->file_unit_size;
			}
			lbn = (unsigned)(RQ->off >> CDFS_SECTOR_SHIFT) + dirent->ea_sectors;

			sectors_back = lbn % used;
			sectors_forward = used - sectors_back;
			if (__unlikely(lbn - sectors_back < dirent->ea_sectors)) sectors_back = lbn - dirent->ea_sectors;
			extent_sectors = (size + CDFS_SECTOR_SIZE - 1) >> CDFS_SECTOR_SIZE_BITS;
			if (__unlikely(lbn - dirent->ea_sectors + sectors_forward > extent_sectors)) sectors_forward = extent_sectors - (lbn - dirent->ea_sectors);
			sector = (lbn / used) * stride + lbn % used + GET_LE_32_16(dirent->extent_le);

			f->file_blk = RQ->off - ((__d_off)sectors_back << CDFS_SECTOR_SHIFT);
			f->disk_blk = (__d_off)(sector - sectors_back) << CDFS_SECTOR_SHIFT;
			f->run_length = (__d_off)(sectors_back + sectors_forward) << CDFS_SECTOR_SHIFT;
			/*__debug_printf("RQ:%LX,lbn:%X,back:%X,fwd:%X,st:%X,sec:%X,size:%X\n", RQ->off, lbn, sectors_back, sectors_forward, GET_LE_32_16(dirent->extent_le), sector, size);*/

			/*f->file_blk = RQ->accumulated_size >> BIO_SECTOR_SIZE_BITS;
			f->disk_blk = CDFS_GET_SECTOR(DIRENT_SECTOR(dirent));
			f->run_length = ((size + CDFS_SECTOR_SIZE - 1) & ~(CDFS_SECTOR_SIZE - 1)) >> BIO_SECTOR_SIZE_BITS;*/
			VFS$PUT_BUFFER(data);
			if (__unlikely((__u64)f->disk_blk + (__u64)f->run_length > (__u64)fs->size)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID BMAP: %"__d_off_format"X, %"__d_off_format"X, %"__d_off_format"X", f->disk_blk, f->file_blk, f->run_length);
				f->run_length = 0;
				RQ->status = -EFSERROR;
			}
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		if (__unlikely(!(dflags & DIR_FLAG_MULTI_EXTENT))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "PREMATURE END OF EXTENTS FOR %"__d_off_format"X:%X", CDFS_GET_SECTOR(f->sector), f->offset);
			err = -EFSERROR;
			goto error_put_buffer;
		}
		if (__unlikely(size & (CDFS_SECTOR_SIZE - 1))) goto unaligned_size;
		RQ->accumulated_size += size;
	}

	skip_this_dirent:
	advance = dirent->length;

	do_advance_put_buffer:
	if (__unlikely(RQ->reading_dir)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIRECTORY'S ROOT ENTRY NOT FOUND AT %"__d_off_format"X", CDFS_GET_SECTOR(RQ->sector));
		err = -EFSERROR;
		goto error_put_buffer;
	}
	VFS$PUT_BUFFER(data);
	RQ->offset += advance;
	if (__unlikely(RQ->offset >= CDFS_SECTOR_SIZE)) {
		RQ->offset -= CDFS_SECTOR_SIZE;
		RQ->sector++;
	}
	if (__unlikely(RQ->remaining <= advance)) goto eof;
	RQ->remaining -= advance;
	goto next;

	eof:
	switch (RQ->type) {
		case PG_LOOKUP:
			err = -ENOENT;
			goto error;
		case PG_READDIR:
			map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, (FS *)fs, (void *)&KERNEL$LIST_END, &unmap);
			if (__unlikely(!map)) goto pagefault;
			if (__unlikely(map == (void *)1)) {
				err = 0;
				goto error;
			}
			if (__unlikely(__IS_ERR(map))) {
				err = __PTR_ERR(map);
				goto error;
			}
			((struct readdir_buffer *)map)->end = 1;
			VFS$UNMAP_READDIR(map, unmap);
			if (__unlikely(VFS$DONE_READDIR((PAGEINRQ *)RQ)))
				KERNEL$SUICIDE("CDFS_AST: VFS$DONE_READDIR RETURNED NON-ZERO");
			err = 0;
			goto error;
		case PG_BMAP:
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "RAN OUT OF EXTENTS FOR %"__d_off_format"X:%X", CDFS_GET_SECTOR(f->sector), f->offset);
			err = -EFSERROR;
			goto error;
		default:
			KERNEL$SUICIDE("CDFS_AST: %s: TYPE %d (2)", VFS$FNODE_DESCRIPTION((FNODE *)f), RQ->type);
	}

	error_put_buffer:
	VFS$PUT_BUFFER(data);
	error:
	switch (RQ->type) {
		case PG_LOOKUP:
			if (__unlikely(err != -ENOENT) || __likely(!(RQ->wr & (O_CREAT | _O_RENAME)))) {
				caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, err == -ENOENT);
				caller->status = err;
				RETURN_AST(caller);
			}
			caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, 1);
			RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);
		case PG_READDIR:
			caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
			caller->status = err;
			RETURN_AST(caller);
		case PG_BMAP:
			RQ->status = err;
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		default:
			KERNEL$SUICIDE("CDFS_AST: %s: TYPE %d (1)", VFS$FNODE_DESCRIPTION((FNODE *)f), RQ->type);
	}

	pagefault_put_buffer:
	VFS$PUT_BUFFER(data);
	pagefault:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	DO_PAGEIN(caller, &((IOCTLRQ *)caller)->v, PF_RW);
#undef fs
#undef f
}

static void PROCESS_SUSP_ENTRIES(CDPAGEINRQ *RQ)
{
#define fs ((CDFS *)RQ->fs)
#define susp_nm ((SUSP_NM *)susp)
#define susp_cl ((SUSP_CL *)susp)
	SUSP *susp;
	while ((susp = GET_SUSP_ENTRY(fs, &RQ->st))) {
		unsigned namelen;
		if (TEST_SUSP(susp, 'N', 'M', 1)) {
			if (__unlikely(susp->len < SUSP_NM_OFFSET)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "SUSP ERROR AT %"__d_off_format"X:%X: NM ENTRY HAS INVALID LENGTH %X", CDFS_GET_SECTOR(RQ->st.current_sector), RQ->st.current_offset + RQ->st.pos, susp->len);
				continue;
			}
			if (susp_nm->flags & (SUSP_NM_CURRENT | SUSP_NM_PARENT)) continue;
			namelen = susp_nm->len - SUSP_NM_OFFSET;
			if (__unlikely(!namelen)) continue;
			if (__likely(!(RQ->rock_used_flags & ROCK_USED_NAME))) RQ->rock_used_flags |= ROCK_USED_NAME, RQ->namelen = 0;
			if (__unlikely(RQ->namelen + namelen > 255)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "SUSP ERROR AT %"__d_off_format"X:%X: ROCK RIDGE NAME IS LONGER THAN 255 CHARACTERS", CDFS_GET_SECTOR(RQ->st.current_sector), RQ->st.current_offset + RQ->st.pos);
				namelen = 255 - RQ->namelen;
			}
			memcpy(RQ->name + RQ->namelen, susp_nm->name, namelen);
			RQ->namelen += namelen;
		} else if (TEST_SUSP(susp, 'C', 'L', 1)) {
			if (__unlikely(susp->len != sizeof(SUSP_CL))) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "SUSP ERROR AT %"__d_off_format"X:%X: CL ENTRY HAS INVALID LENGTH %X", CDFS_GET_SECTOR(RQ->st.current_sector), RQ->st.current_offset + RQ->st.pos, susp->len);
				continue;
			}
			RQ->rock_used_flags |= ROCK_USED_CHILD_LINK;
			RQ->child_link = GET_LE_32_8(susp_cl->sector_le);
		} else if (TEST_SUSP(susp, 'R', 'E', 1)) {
			RQ->rock_used_flags |= ROCK_USED_HIDE;
		} else if (TEST_SUSP(susp, 'T', 'F', 1)) {
			if (__unlikely(ROCK_TIME(susp, &RQ->ctime, &RQ->mtime)))
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "SUSP ERROR AT %"__d_off_format"X:%X: BAD ROCK RIDGE TIMESTAMP", CDFS_GET_SECTOR(RQ->st.current_sector), RQ->st.current_offset + RQ->st.pos);
		}
	}
#undef fs
#undef susp_nm
#undef susp_cl
}

int CDFS_SYNC_BMAP(FNODE *f_, __d_off off, int try)
{
	return -ENOENT;
}

