#include <SPAD/SYSLOG.H>
#include <TIME.H>

#include "EXT2.H"

/* parts from linux/fs/ext2/super.c, (C) Remy Card & others */

static __const__ FSOPS ext2fsops = {
	DEFAULT_VFSID,
	"EXT2",
	FS_SPARSE_FILES,
	sizeof(EXT2FS),
	sizeof(EXT2FNODE),
	VFS$FNODE_CTOR,
	sizeof(EXT2PAGEINRQ),
	VFS$PAGEINRQ_CTOR,
	EXT2_PROCESS_OPTION,
	EXT2_MOUNT,
	NULL,
	EXT2_UMOUNT,
	EXT2_SET_DIRTY,
	EXT2_VALIDATE_FILENAME,
	EXT2_LOOKUP,
	sizeof(EXT2_READDIR_COOKIE),
	EXT2_INIT_READDIR_COOKIE,
	EXT2_READDIR,
	VFS$DEFAULT_WRITEPAGES,
	EXT2_COMMIT,
	EXT2_INIT_FNODE,
	EXT2_CREATE_FNODE,
	EXT2_WRITE_FNODE_DATA,
	EXT2_ADD_FNODE_TO_DIRECTORY,
	EXT2_DELETE_FNODE,
	EXT2_REMOVE_FNODE_FROM_DIRECTORY,
	EXT2_FREE_FNODE,
	EXT2_BMAP,
	EXT2_SYNC_BMAP,
	VFS$INIT_LAST_PAGE,
	EXT2_ACCOUNT,
	EXT2_UNACCOUNT,
	EXT2_STAT,
	EXT2_STATFS,
};


int main(int argc, char *argv[])
{
	return VFS$MOUNT(argc, argv, &ext2fsops, "EXT2.SYS");
}

void *EXT2_READ_BUFFER_SYNC(EXT2FS *fs, ext2_blk_t block, int dirty)
{
	void *p;
	BUFFER *b;
	if (__unlikely(dirty < 0)) {
		p = VFS$GET_BUFFER((FS *)fs, block << fs->sectors_per_block_bits, fs->sectors_per_block, &b, &KERNEL$PROC_KERNEL);
		if (__unlikely(!p)) return __ERR_PTR(-ENOENT);
		return p;
	}
	p = VFS$READ_BUFFER_SYNC((FS *)fs, block << fs->sectors_per_block_bits, fs->sectors_per_block, 0, &b);
	if (__likely(!dirty) || __unlikely(__IS_ERR(p))) return p;
	VFS$MARK_BUFFER_DIRTY(b, (block << fs->sectors_per_block_bits) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, fs->sectors_per_block);
	return p;
}

void *EXT2_ALLOC_BUFFER_SYNC(EXT2FS *fs, ext2_blk_t block)
{
	return VFS$ALLOC_BUFFER_SYNC((FS *)fs, block << fs->sectors_per_block_bits, fs->sectors_per_block, (void *)&KERNEL$LIST_END);
}

int EXT2_PROCESS_OPTION(FS *fs_, char *opt, char *optend, char *str)
{
	return 1;
}

static void ext2_commit_super(struct super_block *sb, int sync)
{
	struct ext2_super_block *xs;
	sb->sblk.s_wtime = cpu_to_le32(time(NULL));
	xs = EXT2_READ_BUFFER_SYNC(sb, sb->sb_block, 1);
	xs = (struct ext2_super_block *)((char *)xs + sb->sb_block_offset);
	if (__unlikely(__IS_ERR(xs))) {
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sb->filesystem_name, "ERROR READING SUPERBLOCK");
		return;
	}
	memcpy(xs, &sb->sblk, sizeof(struct ext2_super_block));
	VFS$PUT_BUFFER(xs);
	if (__likely(!sync)) return;
	VFS$WRITE_BUFFERS((FS *)sb, 1);
	VFS$WAIT_FOR_BUFFERS((FS *)sb);
	VFS$SYNC_DISK((FS *)sb);
}


void ext2_write_super(struct super_block *sb)
{
	if (__likely(!(sb->flags & FS_RO))) {
		ext2_commit_super(sb, 0);
	}
}

void ext2_update_dynamic_rev(struct super_block *sb)
{
	if (__likely(le32_to_cpu(sb->sblk.s_rev_level) > EXT2_GOOD_OLD_REV)) return;
	sb->sblk.s_first_ino = cpu_to_le32(EXT2_GOOD_OLD_FIRST_INO);
	sb->sblk.s_inode_size = cpu_to_le16(EXT2_GOOD_OLD_INODE_SIZE);
	sb->sblk.s_rev_level = cpu_to_le32(EXT2_DYNAMIC_REV);
}

void ext2_error(struct super_block *sb, int sync)
{
	if (!(sb->flags & FS_RO)) {
		sb->s_mount_state |= EXT2_ERROR_FS;
		sb->sblk.s_state = cpu_to_le16(le16_to_cpu(sb->sblk.s_state) | EXT2_ERROR_FS);
		if (sync) ext2_commit_super(sb, 1);
	}
	if (test_opt (sb, ERRORS_PANIC)) goto ro;
		/*panic ("EXT2-fs panic (device %s): %s: %s\n",
		       sb->s_id, function, error_buf);*/
	if (test_opt (sb, ERRORS_RO)) {
		ro:
		if (sync) sb->flags |= FS_RO;
	}
}

static int ext2_setup_super (struct super_block * sb,
			      struct ext2_super_block * es,
			      int read_only)
{
	int res = 0;

	if (read_only)
		return res;
	if ((sb->s_mount_state & EXT2_ERROR_FS))
		KERNEL$SYSLOG (__SYSLOG_SW_WARNING, sb->filesystem_name, "MOUNTING FS WITH ERRORS, RUNNING E2FSCK IS RECOMMENDED");
	else if (!(sb->s_mount_state & EXT2_VALID_FS))
		KERNEL$SYSLOG (__SYSLOG_SW_WARNING, sb->filesystem_name, "MOUNTING UNCHECKED FS, RUNNING E2FSCK IS RECOMMENDED");
	/*
	else if ((__s16) le16_to_cpu(es->s_max_mnt_count) >= 0 &&
		 le16_to_cpu(es->s_mnt_count) >=
		 (unsigned short) (__s16) le16_to_cpu(es->s_max_mnt_count))
		KERNEL$SYSLOG (__SYSLOG_SW_WARNING, sb->filesystem_name, "MAXIMAL MOUNT COUNT REACHED, RUNNING E2FSCK IS RECOMMENDED");
	else if (le32_to_cpu(es->s_checkinterval) &&
		(le32_to_cpu(es->s_lastcheck) + le32_to_cpu(es->s_checkinterval) <= get_seconds()))
		KERNEL$SYSLOG (__SYSLOG_SW_WARNING, sb->filesystem_name, "CHECKTIME REACHED, RUNNING E2FSCK IS RECOMMENDED");
	*/
	if (!(__s16) le16_to_cpu(es->s_max_mnt_count))
		es->s_max_mnt_count = (__s16) cpu_to_le16(EXT2_DFL_MAX_MNT_COUNT);
	es->s_mnt_count=cpu_to_le16(le16_to_cpu(es->s_mnt_count) + 1);
	es->s_mtime = cpu_to_le32(time(NULL));
	ext2_write_super(sb);
	if (test_opt (sb, DEBUG))
		KERNEL$SYSLOG (__SYSLOG_SW_INFO, sb->filesystem_name, "[EXT II FS %s, %s, bs=%u, fs=%u, gc=%lu, "
			"bpg=%lu, ipg=%lu, mo=%04x]\n",
			EXT2FS_VERSION, EXT2FS_DATE, sb->s_blocksize,
			sb->s_frag_size,
			(unsigned long)sb->s_groups_count,
			(unsigned long)EXT2_BLOCKS_PER_GROUP(sb),
			(unsigned long)EXT2_INODES_PER_GROUP(sb),
			sb->s_mount_opt);
#ifdef CONFIG_EXT2_CHECK
	if (test_opt (sb, CHECK)) {
		ext2_check_blocks_bitmap (sb);
		ext2_check_inodes_bitmap (sb);
	}
#endif
	return res;
}

static int ext2_check_descriptors (struct super_block * sb)
{
	int i;
	ext2_blk_t block = le32_to_cpu(sb->sblk.s_first_data_block);
	struct ext2_group_desc * gdp = sb->block_groups;

	ext2_debug ("Checking group descriptors");

	for (i = 0; i < sb->s_groups_count; i++)
	{
		/*
		if ((i % EXT2_DESC_PER_BLOCK(sb)) == 0)
			gdp = (struct ext2_group_desc *) sb->s_group_desc[desc_block++]->b_data;
		*/
		if (le32_to_cpu(gdp->bg_block_bitmap) < block ||
		    le32_to_cpu(gdp->bg_block_bitmap) >= block + EXT2_BLOCKS_PER_GROUP(sb))
		{
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "ext2_check_descriptors: Block bitmap for group %d not in group (block %lu)!",
				    i, (unsigned long) le32_to_cpu(gdp->bg_block_bitmap));
			ext2_error(sb, 1);
			return 0;
		}
		if (le32_to_cpu(gdp->bg_inode_bitmap) < block ||
		    le32_to_cpu(gdp->bg_inode_bitmap) >= block + EXT2_BLOCKS_PER_GROUP(sb))
		{
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "ext2_check_descriptors: Inode bitmap for group %d not in group (block %lu)!",
				    i, (unsigned long) le32_to_cpu(gdp->bg_inode_bitmap));
			ext2_error(sb, 1);
			return 0;
		}
		if (le32_to_cpu(gdp->bg_inode_table) < block ||
		    le32_to_cpu(gdp->bg_inode_table) + sb->s_itb_per_group >=
		    block + EXT2_BLOCKS_PER_GROUP(sb))
		{
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "ext2_check_descriptors: Inode table for group %d not in group (block %lu)!",
				    i, (unsigned long) le32_to_cpu(gdp->bg_inode_table));
			ext2_error(sb, 1);
			return 0;
		}
		block += EXT2_BLOCKS_PER_GROUP(sb);
		gdp++;
	}
	return 1;
}

static __finline__ int test_root(int a, int b)
{
	if (a == 0)
		return 1;
	while (1) {
		if (a == 1)
			return 1;
		if (a % b)
			return 0;
		a = a / b;
	}
}

static int ext2_group_sparse(int group)
{
	return (test_root(group, 3) || test_root(group, 5) ||
		test_root(group, 7));
}


/**
 *      ext2_bg_has_super - number of blocks used by the superblock in group
 *      @sb: superblock for filesystem
 *      @group: group number to check
 *
 *      Return the number of blocks used by the superblock (primary or backup)
 *      in this group.  Currently this will be only 0 or 1.
 */
int ext2_bg_has_super(struct super_block *sb, int group)
{
	if (EXT2_HAS_RO_COMPAT_FEATURE(sb,EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER)&&
	    !ext2_group_sparse(group))
		return 0;
	return 1;
}

static ext2_blk_t descriptor_loc(struct super_block *sb,
				    ext2_blk_t logic_sb_block,
				    int nr)
{
	ext2_blk_t bg, first_data_block, first_meta_bg;
	int has_super = 0;
	
	first_data_block = le32_to_cpu(sb->sblk.s_first_data_block);
	first_meta_bg = le32_to_cpu(sb->sblk.s_first_meta_bg);

	if (!EXT2_HAS_INCOMPAT_FEATURE(sb, EXT2_FEATURE_INCOMPAT_META_BG) ||
	    nr < first_meta_bg)
		return (logic_sb_block + nr + 1);
	bg = sb->s_desc_per_block * nr;
	if (ext2_bg_has_super(sb, bg))
		has_super = 1;
	return (first_data_block + has_super + (bg * sb->s_blocks_per_group));
}

struct ext2_group_desc * ext2_get_group_desc(struct super_block * sb, unsigned int block_group)
{
	return &sb->block_groups[block_group];
}

void ext2_write_group(struct super_block *sb, int grpno)
{
	struct ext2_group_desc *grp = &sb->block_groups[grpno], *dst;
	ext2_blk_t block = descriptor_loc(sb, sb->sb_block, grpno / (sb->s_blocksize / sizeof(struct ext2_group_desc)));
	int off = grpno * sizeof(struct ext2_group_desc) & (sb->s_blocksize - 1);
	void *p = EXT2_READ_BUFFER_SYNC(sb, block, 1);
	if (__unlikely(__IS_ERR(p))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "ERROR UPDATING BLOCK GROUP %d, BLOCK %lu: %s", grpno, (unsigned long)block, strerror(__PTR_ERR(p)));
		ext2_error(sb, 1);
		return;
	}
	dst = (struct ext2_group_desc *)((char *)p + off);
	if (__unlikely(dst->bg_block_bitmap != grp->bg_block_bitmap) || __unlikely(dst->bg_inode_bitmap != grp->bg_inode_bitmap) || __unlikely(dst->bg_inode_table != grp->bg_inode_table)) {
		VFS$PUT_BUFFER(p);
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "BAD GROUP DESCRIPTOR READ, GROUP %d, BLOCK %lu", grpno, (unsigned long)block);
		ext2_error(sb, 1);
		return;
	}
	memcpy(dst, grp, sizeof(struct ext2_group_desc));
	VFS$PUT_BUFFER(p);
}

/* Called at mount-time, super-block is locked */
static ext2_blk_t ext2_count_dirs (struct super_block * sb)
{
	ext2_blk_t count = 0;
	int i;

	for (i = 0; i < EXT2_SB(sb)->s_groups_count; i++) {
		struct ext2_group_desc *gdp = ext2_get_group_desc (sb, i);
		if (!gdp)
			continue;
		count += le16_to_cpu(gdp->bg_used_dirs_count);
	}
	return count;
}

int EXT2_MOUNT(FS *fs_)
{
	int i, r;
	int db_count;
	int blocksize;
#define fs ((EXT2FS *)fs_)
#define sb fs
#define sbi sb
	struct ext2_super_block *es;
	CONTIG_AREA_REQUEST car;

	fs->flags |= FS_MORE_SAME_NAMES;

	for (i = 0; i < INODE_HASH_SIZE; i++) INIT_XLIST(&fs->inode_hash[i]);

	if (!fs->sb_block) fs->sb_block = 1;
	sb->s_blocksize = blocksize = BLOCK_SIZE;
	sb->s_log_block_size = BLOCK_SIZE_BITS;

	sb->sectors_per_block_bits = BLOCK_SIZE_BITS - BIO_SECTOR_SIZE_BITS;
	sb->sectors_per_block = 1 << sb->sectors_per_block_bits;

	if (__IS_ERR(es = EXT2_READ_BUFFER_SYNC(fs, fs->sb_block, 0))) {
		r = __PTR_ERR(es);
		goto ret0;
	}
	memcpy(&sb->sblk, es, sizeof(struct ext2_super_block));
	VFS$PUT_BUFFER(es);
	es = &sb->sblk;

	if (le16_to_cpu(es->s_magic) != EXT2_SUPER_MAGIC) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD EXT2 SUPERBLOCK SIGNATURE %04X", (unsigned)__32LE2CPU(es->s_magic));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: EXT2 SUPERBLOCK SIGNATURE NOT FOUND", fs->filesystem_name);
		r = -EINVAL;
		goto ret0;
	}

	sbi->s_resuid = (ext2_uid_t)le16_to_cpu(es->s_def_resuid);
	sbi->s_resgid = (ext2_gid_t)le16_to_cpu(es->s_def_resgid);
	
	/*
		this check fails on kernel 2.2 with ext3
	if (le32_to_cpu(es->s_rev_level) == EXT2_GOOD_OLD_REV &&
	    (EXT2_HAS_COMPAT_FEATURE(sb, ~0U) ||
	     EXT2_HAS_RO_COMPAT_FEATURE(sb, ~0U) ||
	     EXT2_HAS_INCOMPAT_FEATURE(sb, ~0U))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FEATURE FLAGS SET ON REV 0 FS");
		goto failed_mount;
	}
	*/

	if (le32_to_cpu(es->s_rev_level) > EXT2_MAX_SUPP_REV && !(sb->flags & FS_RO)) {
		KERNEL$SYSLOG (__SYSLOG_SW_INCOMPATIBILITY, sb->filesystem_name, "REVISION LEVEL %d TOO HIGH (MAX %d SUPPORTED)", es->s_rev_level, EXT2_MAX_SUPP_REV);
		goto f1;
	}
	/*
	 * Check feature flags regardless of the revision level, since we
	 * previously didn't change the revision level when setting the flags,
	 * so there is a chance incompat flags are set on a rev 0 filesystem.
	 */
	if ((i = EXT2_HAS_INCOMPAT_FEATURE(sb, ~EXT2_FEATURE_INCOMPAT_SUPP))) {
		KERNEL$SYSLOG(__SYSLOG_SW_INCOMPATIBILITY, fs->filesystem_name, "UNSUPPORTED FEATURE %X", i);
		f1:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNSUPPORTED FEATURES", fs->filesystem_name);
		r = -EINVAL;
		goto ret0;
	}
	if (!(sb->flags & FS_RO) &&
	    (i = EXT2_HAS_RO_COMPAT_FEATURE(sb, ~EXT2_FEATURE_RO_COMPAT_SUPP))){
		KERNEL$SYSLOG(__SYSLOG_SW_INCOMPATIBILITY, fs->filesystem_name, "UNSUPPORTED FEATURE %X, CAN ONLY MOUNT READ-ONLY", i);
		goto f1;
	}

	if (le32_to_cpu(sbi->sblk.s_log_block_size) >= 8 - (BLOCK_SIZE_BITS - BIO_SECTOR_SIZE_BITS)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNSUPPORTED BLOCK SIZE ID %u", (unsigned)le32_to_cpu(sbi->sblk.s_log_block_size));
		goto f1;
	}
	sb->s_blocksize = blocksize = BLOCK_SIZE << le32_to_cpu(sbi->sblk.s_log_block_size);
	sb->s_log_block_size = le32_to_cpu(sbi->sblk.s_log_block_size);

	sb->sectors_per_block_bits = le32_to_cpu(sbi->sblk.s_log_block_size) + BLOCK_SIZE_BITS - BIO_SECTOR_SIZE_BITS;
	sb->sectors_per_block = 1 << sb->sectors_per_block_bits;
	sb->sb_block_offset = BLOCK_SIZE * (sb->sb_block & ((1 << le32_to_cpu(sbi->sblk.s_log_block_size)) - 1));
	sb->sb_block >>= le32_to_cpu(sbi->sblk.s_log_block_size);

	if (!blocksize || blocksize > __PAGE_CLUSTER_SIZE) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNSUPPORTED BLOCK SIZE %u", blocksize);
		goto f1;
	}

	sb->block_mask = sb->sectors_per_block - 1;
	sb->pageio_mask = blocksize - 1;
	sb->pageio_bits = sb->sectors_per_block_bits + BIO_SECTOR_SIZE_BITS;
	sb->size = __32LE2CPU(sb->sblk.s_blocks_count) * sb->sectors_per_block;

	if ((__u64)(__d_off)(sb->size * 2) != (sb->size * 2)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FILESYSTEMS > 1TB NOT SUPPORTED BY VFS LAYER --- CHANGE SIZE OF __d_off IN VFS.H AND RECOMPILE VFS.DLL AND ALL FILESYSTEM DRIVERS", fs->filesystem_name);
		r = -EINVAL;
		VFS$PUT_BUFFER(sb);
		goto ret0;
	}

	/* we turn the flag dynamically like Linux. There is no other way to turn it on. sb->max_filesize = EXT2_HAS_RO_COMPAT_FEATURE(sb, EXT2_FEATURE_RO_COMPAT_LARGE_FILE) ? ((_u_off_t)__D_OFF_TOP_BIT << BIO_SECTOR_SIZE_BITS) - __PAGE_CLUSTER_SIZE : 0x80000000 - __PAGE_CLUSTER_SIZE; */
	sb->max_filesize = ((_u_off_t)__D_OFF_TOP_BIT << BIO_SECTOR_SIZE_BITS) - __PAGE_CLUSTER_SIZE;

	if (le32_to_cpu(es->s_rev_level) == EXT2_GOOD_OLD_REV) {
		sbi->s_inode_size = EXT2_GOOD_OLD_INODE_SIZE;
		sbi->s_first_ino = EXT2_GOOD_OLD_FIRST_INO;
	} else {
		sbi->s_inode_size = le16_to_cpu(es->s_inode_size);
		sbi->s_first_ino = le32_to_cpu(es->s_first_ino);
		if ((sbi->s_inode_size < EXT2_GOOD_OLD_INODE_SIZE) ||
		    (sbi->s_inode_size & (sbi->s_inode_size - 1)) ||
		    (sbi->s_inode_size > blocksize)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNSUPPORTED INODE SIZE %d", sbi->s_inode_size);
			failed_mount:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BAD SUPERBLOCK", fs->filesystem_name);
			r = -EFSERROR;
			goto ret0;
		}
	}

	sbi->s_frag_size = EXT2_MIN_FRAG_SIZE <<
				   le32_to_cpu(es->s_log_frag_size);
	if (sbi->s_frag_size)
		sbi->s_frags_per_block = sb->s_blocksize /
						  sbi->s_frag_size;
	/*else
		sb->s_magic = 0; ???? */
	sbi->s_blocks_per_group = le32_to_cpu(es->s_blocks_per_group);
	sbi->s_frags_per_group = le32_to_cpu(es->s_frags_per_group);
	sbi->s_inodes_per_group = le32_to_cpu(es->s_inodes_per_group);
	sbi->s_inodes_per_block = sb->s_blocksize /
					   EXT2_INODE_SIZE(sb);
	sbi->s_itb_per_group = sbi->s_inodes_per_group /
					sbi->s_inodes_per_block;
	sbi->s_desc_per_block = sb->s_blocksize /
					 sizeof (struct ext2_group_desc);
	sbi->s_mount_state = le16_to_cpu(es->s_state);
	sbi->s_addr_per_block_bits =
		log2 (EXT2_ADDR_PER_BLOCK(sb));
	sbi->s_desc_per_block_bits =
		log2 (EXT2_DESC_PER_BLOCK(sb));

	if (sb->s_blocksize != sbi->s_frag_size) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "EXT2-fs: fragsize %u != blocksize %u (not supported yet)",
			sbi->s_frag_size, sb->s_blocksize);
		goto failed_mount;
	}

	if (sbi->s_blocks_per_group > sb->s_blocksize * 8 || !sbi->s_blocks_per_group) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "EXT2-fs: #blocks per group too big: %lu",
			(unsigned long)sbi->s_blocks_per_group);
		goto failed_mount;
	}
	if (sbi->s_frags_per_group > sb->s_blocksize * 8 || !sbi->s_frags_per_group) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "EXT2-fs: #fragments per group invalid: %lu",
			(unsigned long)sbi->s_frags_per_group);
		goto failed_mount;
	}
	if (sbi->s_inodes_per_group > sb->s_blocksize * 8 || !sbi->s_inodes_per_group) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "EXT2-fs: #inodes per group invalid: %lu",
			(unsigned long)sbi->s_inodes_per_group);
		goto failed_mount;
	}

	sbi->s_groups_count = (le32_to_cpu(es->s_blocks_count) -
				       le32_to_cpu(es->s_first_data_block) +
				       EXT2_BLOCKS_PER_GROUP(sb) - 1) /
				       EXT2_BLOCKS_PER_GROUP(sb);

	db_count = (sbi->s_groups_count + EXT2_DESC_PER_BLOCK(sb) - 1) /
		   EXT2_DESC_PER_BLOCK(sb);
	sbi->s_gdb_count = db_count;
	car.flags = CARF_DATA;
	car.nclusters = (sbi->s_gdb_count * sb->s_blocksize + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS;
	car.align = 0;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (car.status < 0) {
		r = car.status;
		goto ret0;
	}
	fs->block_groups = (void *)car.ptr;
	for (i = 0; i < db_count; i++) {
		ext2_blk_t block = descriptor_loc(sb, sb->sb_block, i);
		void *p = EXT2_READ_BUFFER_SYNC(fs, block, 0);
		if (__IS_ERR(p)) {
			r = __PTR_ERR(p);
			goto ret1;
		}
		memcpy((char *)fs->block_groups + i * blocksize, p, blocksize);
		VFS$PUT_BUFFER(p);
	}

	if (!ext2_check_descriptors (sb)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: EXT2-fs: group descriptors corrupted!", fs->filesystem_name);
		r = -EFSERROR;
		goto ret1;
	}
	sbi->s_dir_count = ext2_count_dirs(sb);

	{
		EXT2FNODE *root;
		char *blk;
		ext2_blk_t root_blk;
		int root_off;
		r = EXT2_FIND_INODE(sb, EXT2_ROOT_INO, &root_blk, &root_off);
		if (r) {
			cng:
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET ROOT INODE", fs->filesystem_name);
			goto ret1;
		}
		blk = EXT2_READ_BUFFER_SYNC(fs, root_blk, 0);
		if (__IS_ERR(blk)) {
			r = __PTR_ERR(blk);
			goto ret1;
		}
		blk += root_off;
		root = (EXT2FNODE *)VFS$GET_ROOT_FNODE((FS *)fs);
		r = EXT2_GET_INODE(root, (struct ext2_inode *)blk, EXT2_ROOT_INO);
		VFS$PUT_BUFFER(blk);
		if (r) goto cng;
	}

/*
	if (EXT2_HAS_COMPAT_FEATURE(sb, EXT3_FEATURE_COMPAT_HAS_JOURNAL))
		ext2_warning(sb, __FUNCTION__,
			"mounting ext3 filesystem as ext2\n");
*/
	ext2_setup_super (sb, es, sb->flags & FS_RO);
	return 0;

	ret1:
	KERNEL$VM_RELEASE_CONTIG_AREA(fs->block_groups, (sbi->s_gdb_count * sb->s_blocksize + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS);
	ret0:
	return r;
#undef fs
#undef sb
#undef sbi
}

void EXT2_SET_DIRTY(FS *fs_, int dirty)
{
	EXT2FS *fs = (EXT2FS *)fs_;
	fs->sblk.s_state = le16_to_cpu(fs->s_mount_state);
	if (dirty) fs->sblk.s_state &= le16_to_cpu(~EXT2_VALID_FS);
	ext2_commit_super(fs, 1);
}

void EXT2_COMMIT(FS *fs)
{
	EXT2_DISCARD_PREALLOC((EXT2FS *)fs);
	fs->flags &= ~FS_COMMIT_FREES_DATA;
	ext2_write_super((EXT2FS *)fs);
	VFS$DEFAULT_COMMIT(fs);
}

void EXT2_UMOUNT(FS *fs_, int nosync)
{
	int i;
	EXT2FS *fs = (EXT2FS *)fs_;
	for (i = 0; i < INODE_HASH_SIZE; i++) if (!XLIST_EMPTY(&fs->inode_hash[i]))
		KERNEL$SUICIDE("EXT2_UMOUNT: %s: ENTRIES LEAKED IN INODE HASH: %d", fs->filesystem_name, i);
	if (!nosync && (fs->overallocated_avg_blocks || fs->overallocated_blocks || fs->overallocated_inodes)) KERNEL$SUICIDE("EXT2_UMOUNT: %s: ACCOUNTING SKEW (%ld AVG BLOCKS, %ld BLOCKS, %ld INODES LEAKED)", fs->filesystem_name, (unsigned long)fs->overallocated_avg_blocks, (unsigned long)fs->overallocated_blocks, (unsigned long)fs->overallocated_inodes);
	KERNEL$VM_RELEASE_CONTIG_AREA(fs->block_groups, (fs->s_gdb_count * fs->s_blocksize + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS);
}

int EXT2_VALIDATE_FILENAME(FS *fs, __const__ char *filename)
{
	if (__unlikely(!*filename) || __unlikely(strlen(filename) > 255)) return -EINVAL;
	return 0;
}

void *EXT2_ACCOUNT(FNODE *f, int dir, PAGE *p)
{
	EXT2FS *fs = (EXT2FS *)f->fs;
	unsigned avg_blocks, blocks, inodes;
	if (__likely(dir == ACCOUNT_PAGE)) {
		avg_blocks = __PAGE_CLUSTER_SIZE >> EXT2_BLOCK_SIZE_BITS(fs);
		blocks = avg_blocks + 5;
		inodes = 0;
	} else if (__likely(dir == ACCOUNT_FILE)) {
		avg_blocks = 0;
		blocks = 4;
		inodes = 1;
	} else {
		avg_blocks = 1;
		blocks = 5;
		inodes = 1;
	}
	if (__unlikely(fs->overallocated_blocks + blocks > __32LE2CPU(fs->sblk.s_free_blocks_count))) return VFS$OVERACCOUNT(fs->root);
	if (__unlikely(fs->overallocated_inodes + inodes > __32LE2CPU(fs->sblk.s_free_inodes_count))) return VFS$OVERACCOUNT(fs->root);
	fs->overallocated_avg_blocks += avg_blocks;
	fs->overallocated_blocks += blocks;
	fs->overallocated_inodes += inodes;
	return NULL;
}

void EXT2_UNACCOUNT(FNODE *f, int dir, PAGE *p)
{
	EXT2FS *fs = (EXT2FS *)f->fs;
	unsigned avg_blocks, blocks, inodes;
	if (__likely(dir == ACCOUNT_PAGE)) {
		avg_blocks = __PAGE_CLUSTER_SIZE >> EXT2_BLOCK_SIZE_BITS(fs);
		blocks = avg_blocks + 5;
		inodes = 0;
	} else if (__likely(dir == ACCOUNT_FILE)) {
		avg_blocks = 0;
		blocks = 4;
		inodes = 1;
	} else {
		avg_blocks = 1;
		blocks = 5;
		inodes = 1;
	}
	fs->overallocated_avg_blocks -= avg_blocks;
	fs->overallocated_blocks -= blocks;
	fs->overallocated_inodes -= inodes;
}

void EXT2_STAT(FNODE *f_, struct stat *stat)
{
	EXT2FS *fs = (EXT2FS *)f_->fs;
	stat->st_blocks = EXT2_BLOCKS(fs, stat->st_size) << fs->sectors_per_block_bits;
}

ext2_blk_t EXT2_BLOCKS(EXT2FS *fs, __u64 size)
{
	ext2_blk_t blocks = (size + (fs->sectors_per_block << BIO_SECTOR_SIZE_BITS) - 1) >> (fs->sectors_per_block_bits + BIO_SECTOR_SIZE_BITS);
	ext2_blk_t indir = (blocks - EXT2_NDIR_BLOCKS + EXT2_ADDR_PER_BLOCK(fs) - 1) >> (EXT2_BLOCK_SIZE_BITS(fs) - 2);
	blocks += indir;
	indir = (indir - 1 + EXT2_ADDR_PER_BLOCK(fs) - 1) >> (EXT2_BLOCK_SIZE_BITS(fs) - 2);
	blocks += indir;
	blocks += __unlikely(indir > 1);
	return blocks;
}

WQ *EXT2_STATFS(FS *fs_, struct statfs *stat)
{
#define fs ((EXT2FS *)fs_)
	stat->f_blocks = __32LE2CPU(fs->sblk.s_blocks_count);
	stat->f_bfree = __likely(__32LE2CPU(fs->sblk.s_free_blocks_count) + fs->prealloc_blk_len > fs->overallocated_avg_blocks) ? __32LE2CPU(fs->sblk.s_free_blocks_count) + fs->prealloc_blk_len - fs->overallocated_avg_blocks : 0;
	stat->f_bavail = stat->f_bfree;
	stat->f_files = fs->s_inodes_per_group * fs->s_groups_count;
	stat->f_ffree = __likely(__32LE2CPU(fs->sblk.s_free_inodes_count) + fs->prealloc_ino_len > fs->overallocated_inodes) ? __32LE2CPU(fs->sblk.s_free_inodes_count) + fs->prealloc_ino_len - fs->overallocated_inodes : 0;
	stat->f_favail = stat->f_ffree;
	stat->f_namelen = 255;
	return NULL;
#undef fs
}

