#include <SPAD/SYSLOG.H>

#include "EXT2.H"

static void *EXT2_MAP_BITMAP(EXT2FS *sb, int group_no, int in)
{
	ext2_blk_t block = __32LE2CPU(__unlikely(in) ? sb->block_groups[group_no].bg_inode_bitmap : sb->block_groups[group_no].bg_block_bitmap);
	return EXT2_READ_BUFFER_SYNC(sb, block, 1);
}

static __finline__ int test_bit(__u32 d, int bit)
{
	return d & (1 << (bit & 31));
}

static __finline__ __u32 set_bit(__u32 d, int bit)
{
	return d | 1 << (bit & 31);
}

static __finline__ __u32 clear_bit(__u32 d, int bit)
{
	return d & ~(1 << (bit & 31));
}

static int find_string(__u32 *bmp, unsigned start, unsigned end, unsigned n)
{
	unsigned run = 0;
	__u32 d;
	do {
		d = __32LE2CPU(bmp[start >> 5]);
		if (__likely(!(start & 31))) {
			if (__likely(d == 0xffffffffu)) {
				plus32:
				start += 32;
				run = 0;
				continue;
			}
			if (n >= 64) {
				if (__likely(!run)) {
					if (__unlikely(start + n > end)) break;
					if (__likely(bmp[(start >> 5) + 1] != __32CPU2LE(0))) {
						goto plus32;
					}
				}
				if (!d && run + 32 < n) {
					run += 32;
					start += 32;
					continue;
				}
			}
			if (!run) {
				start |= __BSF(~d);
				goto bit_test_ok;
			}
		}
		if (!test_bit(d, start)) {
			bit_test_ok:
			if (++run == n) {
				goto found;
			}
		} else {
			run = 0;
		}
		start++;
	} while (start <= end);
	return -1;
	found:
	set_allocated:
	d = __32LE2CPU(bmp[start >> 5]);
	d = set_bit(d, start);
	bmp[start >> 5] = __32CPU2LE(d);
	if (__likely(--n)) {
		if (__unlikely(!(start & 31))) {
			while (__likely(n > 32)) {
				start -= 32;
				bmp[start >> 5] = __32CPU2LE(0xffffffffu);
				n -= 32;
			}
		}
		start--;
		goto set_allocated;
	}
	return start;
}

static int clear_string(__u32 *bmp, unsigned start, unsigned n)
{
	do {
		__u32 d = __32LE2CPU(bmp[start >> 5]);
		if (__unlikely(!test_bit(d, start))) return 1;
		d = clear_bit(d, start);
		bmp[start >> 5] = __32CPU2LE(d);
		start++;
	} while (--n);
	return 0;
}

ext2_blk_t EXT2_ALLOC(EXT2FS *sb, ext2_blk_t goal, unsigned n, int in)
{
	__u32 *data;
	unsigned group_size = __unlikely(in) ? EXT2_INODES_PER_GROUP(sb) : EXT2_BLOCKS_PER_GROUP(sb);
	unsigned first = __unlikely(in) ? 1 : le32_to_cpu(sb->sblk.s_first_data_block);
	unsigned group_no;
	int nloops = sb->s_groups_count + 1;
	int free;
	int found;
	if (__unlikely(goal < first)) {
		if (__unlikely(!n)) return 0;
		goal = first;
	}
	goal -= first;
	group_no = goal / group_size;
	goal = goal % group_size;
	if (__unlikely(group_no >= sb->s_groups_count)) {
		if (__unlikely(!n)) return 0;
		group_no = 0;
	}

	new_loop:
	free = __16LE2CPU(in ? sb->block_groups[group_no].bg_free_inodes_count : sb->block_groups[group_no].bg_free_blocks_count);
	if (free < n) goto newgrp;
	data = EXT2_MAP_BITMAP(sb, group_no, in);
	if (__unlikely(__IS_ERR(data))) {
		ext2_error(sb, 1);
		if (__unlikely(!n)) return 0;
		goto newgrp;
	}
	if (__unlikely(!n)) {
		__u32 d = __32LE2CPU(data[goal >> 5]);
		if (__unlikely(test_bit(d, goal))) {
			VFS$PUT_BUFFER(data);
			return 0;
		}
		d = set_bit(d, goal);
		data[goal >> 5] = __32CPU2LE(d);
		found = goal;
		n = 1;
		goto put_update_count;
	}
	if (__likely((found = find_string(data, goal, group_size, n)) >= 0)) {
		ext2_blk_t count;
		put_update_count:
		VFS$PUT_BUFFER(data);
		free -= n;
		if (__unlikely(in)) {
			sb->block_groups[group_no].bg_free_inodes_count = __16CPU2LE(free);
			if (__unlikely(in == 2)) {
				count = __16LE2CPU(sb->block_groups[group_no].bg_used_dirs_count);
				count += n;
				if (__unlikely(count > group_size)) {
					KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "DIRECTORY COUNT FOR GROUP %d OVERFLOW: %lu", group_no, (unsigned long)count);
					ext2_error(sb, 1);
					count = group_size;
				}
				sb->block_groups[group_no].bg_used_dirs_count = __16CPU2LE(count);
			}
			count = __32LE2CPU(sb->sblk.s_free_inodes_count);
			if (__unlikely(count < n)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "FREE INODE COUNT UNDERFLOW: %lu, ALLOCATED %u", (unsigned long)count, n);
				ext2_error(sb, 1);
				count = 0;
			} else {
				count -= n;
			}
			sb->sblk.s_free_inodes_count = __32CPU2LE(count);
		} else {
			sb->block_groups[group_no].bg_free_blocks_count = __16CPU2LE(free);
			count = __32LE2CPU(sb->sblk.s_free_blocks_count);
			if (__unlikely(count < n)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "FREE BLOCK COUNT UNDERFLOW: %lu, ALLOCATED %u", (unsigned long)count, n);
				ext2_error(sb, 1);
				count = 0;
			} else {
				count -= n;
			}
			sb->sblk.s_free_blocks_count = __32CPU2LE(count);
		}
		ext2_write_group(sb, group_no);
		return found + group_no * group_size + first;
	}
	VFS$PUT_BUFFER(data);

	if (__unlikely(!goal) && n == 1) {
		unsigned was_free;
		if (__unlikely(in)) {
			was_free = __16LE2CPU(sb->block_groups[group_no].bg_free_inodes_count);
			sb->block_groups[group_no].bg_free_inodes_count = __16CPU2LE(0);
		} else {
			was_free = __16LE2CPU(sb->block_groups[group_no].bg_free_blocks_count);
			sb->block_groups[group_no].bg_free_blocks_count = __16CPU2LE(0);
		}
		ext2_write_group(sb, group_no);
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "NO FREE %s IN GROUP %d (FREE COUNTER WAS %u)", in ? "INODES" : "BLOCKS", group_no, was_free);
		ext2_error(sb, 1);
	}

	newgrp:
	group_no++;
	if (__unlikely(group_no >= sb->s_groups_count)) group_no = 0;
	nloops--;
	goal = 0;
	if (__likely(nloops >= 0)) goto new_loop;
	return 0;
}

void EXT2_FREE(EXT2FS *sb, ext2_blk_t goal, unsigned n, int in)
{
	if (!goal) return;
	if (__likely(!in)) VFS$CLEAN_BUFFERS_SYNC((FS *)sb, goal << sb->sectors_per_block_bits, sb->sectors_per_block);
	{__u32 *data;
	unsigned group_size = __unlikely(in) ? EXT2_INODES_PER_GROUP(sb) : EXT2_BLOCKS_PER_GROUP(sb);
	unsigned first = in ? 1 : le32_to_cpu(sb->sblk.s_first_data_block);
	unsigned group_no;
	unsigned to_clear;
	unsigned free;
	ext2_blk_t count;
	if (__unlikely(goal < first)) {
		oor:
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "%s %lu OUT OF RANGE", in ? "INODE" : "BLOCK", (unsigned long)goal);
		ext2_error(sb, 1);
		return;
	}
	goal -= first;
	group_no = goal / group_size;
	goal = goal % group_size;

	again:
	if (__unlikely(group_no >= sb->s_groups_count)) {
		goal = goal + group_no * group_size + first;
		goto oor;
	}

	data = EXT2_MAP_BITMAP(sb, group_no, in);
	if (__unlikely(__IS_ERR(data))) {
		ext2_error(sb, 1);
		return;
	}
	to_clear = group_size - goal;
	if (__likely(to_clear > n)) to_clear = n;
	n -= to_clear;
	if (clear_string(data, goal, to_clear)) {
		VFS$PUT_BUFFER(data);
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "BIT ALREADY CLEARED FOR %s RUN %lu, %u", in ? "INODE" : "BLOCK", (unsigned long)goal, to_clear);
		ext2_error(sb, 1);
		return;
	}
	VFS$PUT_BUFFER(data);
	free = __16LE2CPU(in ? sb->block_groups[group_no].bg_free_inodes_count : sb->block_groups[group_no].bg_free_blocks_count);
	free += to_clear;
	if (__unlikely(free > group_size)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "%s COUNT FOR GROUP %d OVERFLOW: %u", in ? "INODE" : "BLOCK", group_no, free);
		ext2_error(sb, 1);
		free = group_size;
	}
	if (__unlikely(in)) {
		sb->block_groups[group_no].bg_free_inodes_count = __16CPU2LE(free);
		if (__unlikely(in == 2)) {
			count = __16LE2CPU(sb->block_groups[group_no].bg_used_dirs_count);
			if (__unlikely(count < to_clear)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "DIRECTORY COUNT FOR GROUP %d UNDERFLOW: %lu, FREED %u", group_no, (unsigned long)count, to_clear);
				ext2_error(sb, 1);
				count = 0;
			} else {
				count -= to_clear;
			}
			sb->block_groups[group_no].bg_used_dirs_count = __16CPU2LE(count);
		}
		sb->sblk.s_free_inodes_count = __32CPU2LE(__32LE2CPU(sb->sblk.s_free_inodes_count) + to_clear);
	} else {
		sb->block_groups[group_no].bg_free_blocks_count = __16CPU2LE(free);
		sb->sblk.s_free_blocks_count = __32CPU2LE(__32LE2CPU(sb->sblk.s_free_blocks_count) + to_clear);
	}
	ext2_write_group(sb, group_no);
	if (__unlikely(n)) {
		group_no++;
		goal = 0;
		goto again;
	}
	}
}
