#include <SPAD/SYSLOG.H>
#include <SYS/STAT.H>
#include <TIME.H>

#include "EXT2.H"

void EXT2_INIT_FNODE(FNODE *f_)
{
#define f ((EXT2FNODE *)f_)
	f->i_ino = 0;
#undef f
}

int EXT2_FIND_INODE(EXT2FS *sb, ext2_ino_t ino, ext2_blk_t *block, int *offset)
{
	ext2_blk_t block_group;
	struct ext2_group_desc * gdp;

	if ((ino != EXT2_ROOT_INO && ino < EXT2_FIRST_INO(sb)) ||
	    ino > le32_to_cpu(EXT2_SB(sb)->sblk.s_inodes_count))
		goto Einval;

	block_group = (ino - 1) / EXT2_INODES_PER_GROUP(sb);
	gdp = ext2_get_group_desc(sb, block_group);
	if (!gdp)
		goto Egdp;
	/*
	 * Figure out the offset within the block group inode table
	 */
	*offset = ((ino - 1) % EXT2_INODES_PER_GROUP(sb)) * EXT2_INODE_SIZE(sb);
	*block = le32_to_cpu(gdp->bg_inode_table) +
		((*offset) >> EXT2_BLOCK_SIZE_BITS(sb));
	*offset &= (EXT2_BLOCK_SIZE(sb) - 1);
	return 0;

Einval:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, sb->filesystem_name, "BAD INODE NUMBER: %lu", (unsigned long) ino);
	ext2_error(sb, 0);
	return -EFSERROR;

Egdp:
	return -EIO;
}

int EXT2_GET_INODE(EXT2FNODE *inode, struct ext2_inode *raw_inode, ext2_ino_t ino)
{
	int n;
	struct ext2_inode_info *ei;
	if (__unlikely(!ino)) return -EINVAL;
	ei = EXT2_I(inode);
	inode->i_mode = le16_to_cpu(raw_inode->i_mode);
	inode->i_uid = (ext2_uid_t)le16_to_cpu(raw_inode->i_uid_low);
	inode->i_gid = (ext2_gid_t)le16_to_cpu(raw_inode->i_gid_low);
	if (!(test_opt ((EXT2FS *)inode->fs, NO_UID32))) {
		inode->i_uid |= (ext2_uid_t)le16_to_cpu(raw_inode->i_uid_high) << 16;
		inode->i_gid |= (ext2_gid_t)le16_to_cpu(raw_inode->i_gid_high) << 16;
	}
	inode->i_nlink = le16_to_cpu(raw_inode->i_links_count);
	inode->i_atime = le32_to_cpu(raw_inode->i_atime);
	inode->ctime = __time32_2_time_t(le32_to_cpu(raw_inode->i_ctime));
	inode->mtime = __time32_2_time_t(le32_to_cpu(raw_inode->i_mtime));
	ei->i_dtime = le32_to_cpu(raw_inode->i_dtime);
	/* We now have enough fields to check if the inode was active or not.
	 * This is needed because nfsd might try to access dead inodes
	 * the test is that same one that e2fsck uses
	 * NeilBrown 1999oct15
	 */
	if (inode->i_nlink == 0 && (inode->i_mode == 0 || ei->i_dtime)) {
		/* this inode is deleted */
		goto bad_inode;
	}
	inode->i_blocks = le32_to_cpu(raw_inode->i_blocks);
	ei->i_flags = le32_to_cpu(raw_inode->i_flags);
	ei->i_faddr = le32_to_cpu(raw_inode->i_faddr);
	ei->i_frag_no = raw_inode->i_frag;
	ei->i_frag_size = raw_inode->i_fsize;
	ei->i_file_acl = le32_to_cpu(raw_inode->i_file_acl);
	ei->i_dir_acl = 0;
	FNODE_OUT_OF_WANTFREE((FNODE *)inode);
	if (LINUX_S_ISREG(inode->i_mode)) {
		inode->size = inode->disk_size = __make64(le32_to_cpu(raw_inode->i_size), le32_to_cpu(raw_inode->i_size_high));
		inode->flags = FNODE_FILE;
		inode->run_length = 0;
		/*if (__unlikely(inode->i_nlink != 1)) return -EINVAL;*/
	} else if (LINUX_S_ISDIR(inode->i_mode)) {
		inode->size = inode->disk_size = le32_to_cpu(raw_inode->i_size);
		ei->i_dir_acl = le32_to_cpu(raw_inode->i_dir_acl);
		if (!(inode->flags & FNODE_DIRECTORY)) {
			VFS_INIT_DIR((FNODE *)inode);
		}
		inode->flags = FNODE_DIRECTORY;
	} else return -EINVAL;
	ei->i_dtime = 0;
	inode->i_generation = le32_to_cpu(raw_inode->i_generation);
	/*
	ei->i_next_alloc_block = 0;
	ei->i_next_alloc_goal = 0;
	ei->i_prealloc_count = 0;
	ei->i_block_group = (ino - 1) / EXT2_INODES_PER_GROUP((EXT2FS *)inode->fs);
	ei->i_dir_start_lookup = 0;*/

	/*
	 * NOTE! The in-memory inode i_data array is in little-endian order
	 * even on big-endian machines: we do NOT byteswap the block numbers!
	 */
	for (n = 0; n < EXT2_N_BLOCKS; n++)
		ei->i_data[n] = raw_inode->i_block[n];

	/*
	if (ei->i_flags & EXT2_SYNC_FL)
		inode->i_flags |= S_SYNC;
	if (ei->i_flags & EXT2_APPEND_FL)
		inode->i_flags |= S_APPEND;
	if (ei->i_flags & EXT2_IMMUTABLE_FL)
		inode->i_flags |= S_IMMUTABLE;
	if (ei->i_flags & EXT2_NOATIME_FL)
		inode->i_flags |= S_NOATIME;
	*/
	inode->i_ino = ino;
	if (__likely(!(inode->flags & FNODE_DIRECTORY)) && __likely(inode->fs->root != (FNODE *)inode))
		ADD_TO_XLIST(&((EXT2FS *)inode->fs)->inode_hash[INODE_HASH(inode->i_ino)], &inode->inode_hash);
	else
		VOID_LIST_ENTRY(&inode->inode_hash);
	return 0;
	
bad_inode:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, inode->fs->filesystem_name, "BAD INODE %lu", (unsigned long)ino);
	ext2_error((EXT2FS *)inode->fs, 0);
	return -EFSERROR;
}

void EXT2_FREE_FNODE(FNODE *f_)
{
	EXT2FNODE *f = (EXT2FNODE *)f_;
	if (__unlikely(!f->i_ino))
		KERNEL$SUICIDE("EXT2_FREE_FNODE: %s: FNODE (FLAGS %08X) HAS NOT BEEN ADDED TO LIST", VFS$FNODE_DESCRIPTION((FNODE *)f), f->flags);
	DEL_FROM_LIST(&f->inode_hash);
}

static EXT2FNODE *EXT2_FIND_INO(EXT2FS *fs, __u32 ino)
{
	EXT2FNODE *inode;
	XLIST_FOR_EACH_UNLIKELY(inode, &fs->inode_hash[INODE_HASH(ino)], EXT2FNODE, inode_hash)
		if (__unlikely(inode->i_ino == ino)) return inode;
	return NULL;
}

void EXT2_INIT_READDIR_COOKIE(struct readdir_buffer *buf, FNODE *f)
{
	EXT2_READDIR_COOKIE *c = (EXT2_READDIR_COOKIE *)buf->cookie;
	c->block = 0;
	c->offs = 0;
	buf->cookie_size = sizeof(EXT2_READDIR_COOKIE);
}

extern AST_STUB EXT2_AST;

void EXT2_READDIR(PAGEINRQ *readdir, struct readdir_buffer *buf)
{
	EXT2PAGEINRQ *r = (EXT2PAGEINRQ *)readdir;
	r->fn = EXT2_AST;
	if (buf) {
		r->off = ((EXT2_READDIR_COOKIE *)buf->cookie)->block;
		r->blk = r->off >> ((EXT2FS *)r->fs)->sectors_per_block_bits;
	} else {
		r->blk = r->off = 0;
	}
	r->type = PG_READDIR;
	r->depth = 0;
	CALL_AST(r);
}

void EXT2_LOOKUP(PAGEINRQ *p)
{
	EXT2PAGEINRQ *r = (EXT2PAGEINRQ *)p;
	r->fn = EXT2_AST;
	r->off = 0;
	r->blk = 0;
	r->type = PG_LOOKUP;
	r->depth = 0;
	CALL_AST(r);
}

void EXT2_BMAP(PAGEINRQ *p)
{
	EXT2PAGEINRQ *r = (EXT2PAGEINRQ *)p;
	r->fn = EXT2_AST;
	r->blk = r->off >> ((EXT2FS *)r->fs)->sectors_per_block_bits;
	r->type = PG_BMAP;
	r->depth = 0;
	CALL_AST(r);
}

DECL_AST(EXT2_AST, SPL_FS, EXT2PAGEINRQ)
{
#define f ((EXT2FNODE *)RQ->fnode)
#define ext2fs ((EXT2FS *)RQ->fs)
	IORQ *caller;
	long error;
	__u32 *run_start, *run_end, *run_c;
	__u32 *blk;
	IO_DISABLE_CHAIN_CANCEL(SPL_FS, RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status < 0)) {
		error = RQ->status;
		goto err;
	}
	if (RQ->depth == -2) {
		ext2_blk_t block;
		int pos;
		char *iblk;
		int was_dir;
		EXT2FNODE *fn;
		get_inode:
		if (__unlikely((fn = EXT2_FIND_INO(ext2fs, RQ->indir)) != NULL)) {
			WQ *wq;
			if (__likely(!(wq = VFS$FREE_FNODE((FNODE *)fn)))) goto get_inode;
			if (__likely(!PAGEIN_IS_PREFETCH(RQ, RQ->caller))) WQ_WAIT_F(wq, RQ->caller);
			LOWER_SPL(SPL_FS);
			VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, 0);
			RETURN;
		}
		fn = (EXT2FNODE *)RQ->new_fnode;
		error = EXT2_FIND_INODE(ext2fs, RQ->indir, &block, &pos);
		if (__unlikely(error != 0)) goto err;
		if (__unlikely(!(iblk = VFS$GET_BUFFER((FS *)ext2fs, block << ext2fs->sectors_per_block_bits, ext2fs->sectors_per_block, (void *)&KERNEL$LIST_END, RQ->tag.proc)))) {
			VFS$READ_BUFFER(block << ext2fs->sectors_per_block_bits, (PAGEINRQ *)RQ, 0);
			RETURN;
		}
		iblk += pos;
		was_dir = 0;
		if (fn->flags & FNODE_DIRECTORY) {
			was_dir = 1;
			if (__unlikely(!LINUX_S_ISDIR(((struct ext2_inode *)iblk)->i_mode))) {
				error = -ENOTDIR;
				VFS$PUT_BUFFER(iblk);
				goto err;
			}
		}
		error = EXT2_GET_INODE((EXT2FNODE *)RQ->new_fnode, (struct ext2_inode *)iblk, RQ->indir);
		WQ_WAKE_ALL(&RQ->new_fnode->wait);
		VFS$PUT_BUFFER(iblk);
		if (__unlikely(error != 0)) goto err;
		RQ->fnode->readers--;
		WQ_WAKE_ALL(&RQ->fnode->wait);
		if (__unlikely(RQ->fnode->readers < 0))
			KERNEL$SUICIDE("EXT2_AST: %s: LOCK UNDERRUN: %d", VFS$FNODE_DESCRIPTION(RQ->fnode), RQ->fnode->readers);
		if (was_dir) {
			RQ->new_fnode = LIST_STRUCT(fn->u.d.clean.next, FNODE, dirent_entry);
			RQ->fnode = (FNODE *)fn;
			RQ->fnode->readers++;
			RQ->off = 0;
			RQ->blk = 0;
			RQ->depth = 0;
			goto search_next_block;
		}
		if (__likely((RQ->wr & (O_CREAT | O_EXCL)) != (O_CREAT | O_EXCL))) {
			caller = VFS$FREE_EMPTY_PAGEIN((PAGEINRQ *)RQ);
			RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);
		} else {
			caller = VFS$FREE_EMPTY_PAGEIN((PAGEINRQ *)RQ);
			caller->status = -EEXIST;
			RETURN_AST(caller);
		}
	}
	if (!RQ->depth) {
		if (0) {
			search_next_block:
			if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
				WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
				RETURN;
			}
		}
		if (RQ->blk < EXT2_NDIR_BLOCKS) {
			run_start = &f->i_data[0];
			run_end = &f->i_data[EXT2_NDIR_BLOCKS];
			run_c = &f->i_data[RQ->blk];
			blk = NULL;
			goto found;
		}
		RQ->blk -= EXT2_NDIR_BLOCKS;
		if (RQ->blk < EXT2_ADDR_PER_BLOCK(ext2fs)) {
			RQ->depth = 1;
			RQ->indir = __32LE2CPU(f->i_data[EXT2_IND_BLOCK]);
			goto again2;
		}
		RQ->blk -= EXT2_ADDR_PER_BLOCK(ext2fs);
		if (RQ->blk < EXT2_ADDR_PER_BLOCK(ext2fs) * EXT2_ADDR_PER_BLOCK(ext2fs)) {
			RQ->depth = 2;
			RQ->indir = __32LE2CPU(f->i_data[EXT2_DIND_BLOCK]);
			goto again2;
		}
		RQ->blk -= EXT2_ADDR_PER_BLOCK(ext2fs) * EXT2_ADDR_PER_BLOCK(ext2fs);
		if (RQ->blk < EXT2_ADDR_PER_BLOCK(ext2fs) * EXT2_ADDR_PER_BLOCK(ext2fs) * EXT2_ADDR_PER_BLOCK(ext2fs)) {
			RQ->depth = 3;
			RQ->indir = __32LE2CPU(f->i_data[EXT2_TIND_BLOCK]);
			goto again2;
		}
		goto z;
	}
	again2:
	if (!RQ->indir) {
		z:
		if (__likely(RQ->type == PG_BMAP)) {
			RQ->status = 1;
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		goto skip_blk;
	}
	if (__unlikely(!(blk = VFS$GET_BUFFER((FS *)ext2fs, RQ->indir << ext2fs->sectors_per_block_bits, ext2fs->sectors_per_block, (void *)&KERNEL$LIST_END, RQ->tag.proc)))) {
		VFS$READ_BUFFER(RQ->indir << ext2fs->sectors_per_block_bits, (PAGEINRQ *)RQ, 0);
		RETURN;
	}
	if (__unlikely(RQ->depth > 1)) {
		if (__unlikely(RQ->depth == 3)) {
			RQ->indir = __32LE2CPU(blk[RQ->blk / (EXT2_ADDR_PER_BLOCK(ext2fs) * EXT2_ADDR_PER_BLOCK(ext2fs))]);
			RQ->blk %= EXT2_ADDR_PER_BLOCK(ext2fs) * EXT2_ADDR_PER_BLOCK(ext2fs);
		} else {
			RQ->indir = __32LE2CPU(blk[RQ->blk / EXT2_ADDR_PER_BLOCK(ext2fs)]);
			RQ->blk %= EXT2_ADDR_PER_BLOCK(ext2fs);
		}
		RQ->depth--;
		VFS$PUT_BUFFER(blk);
		goto again2;
	}
	if (RQ->depth == 1) {
		run_start = &blk[0];
		run_end = &blk[EXT2_ADDR_PER_BLOCK(ext2fs)];
		run_c = &blk[RQ->blk];

		found:
		if (__unlikely(run_c < run_start) || __unlikely(run_c >= run_end))
			KERNEL$SUICIDE("EXT2_AST: %s: OUT OF ARRAY", VFS$FNODE_DESCRIPTION((FNODE *)f));
		if (__likely(RQ->type == PG_BMAP)) {
			__u32 *r;
			if (__unlikely(*run_c == __32CPU2LE(0))) {
				if (blk) VFS$PUT_BUFFER(blk);
				goto z;
			}
			f->file_blk = RQ->off & -(__d_off)ext2fs->sectors_per_block;
			f->disk_blk = __32LE2CPU(*run_c) << ext2fs->sectors_per_block_bits;
			f->run_length = ext2fs->sectors_per_block;
			r = run_c - 1;
			while (r >= run_start && __32LE2CPU(r[0]) && __32LE2CPU(r[0]) + 1 == __32LE2CPU(r[1])) {
				f->file_blk -= ext2fs->sectors_per_block;
				f->disk_blk -= ext2fs->sectors_per_block;
				f->run_length += ext2fs->sectors_per_block;
				r--;
			}
			r = run_c + 1;
			while (r < run_end && __32LE2CPU(r[0]) - 1 == __32LE2CPU(r[-1])) {
				f->run_length += ext2fs->sectors_per_block;
				r++;
			}
			if (blk) VFS$PUT_BUFFER(blk);
			if (is_bmap_invalid(f)) {
				EXT2_INVALID_BMAP_MSG(f);
				RQ->status = -EFSERROR;
			}
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		RQ->indir = __32LE2CPU(*run_c);
		RQ->depth = -1;
		if (blk) VFS$PUT_BUFFER(blk);
		goto again2;
	}

	{
		struct ext2_dir_entry_2 *next;
		struct ext2_dir_entry_2 *e;
		struct ext2_dir_entry_2 *end;
		void *map;
		unsigned maplen;
		void (*unmap)(void *ptr);
#define rb ((struct readdir_buffer *)map)

		if (__unlikely(RQ->type != PG_LOOKUP) && __unlikely(RQ->type != PG_READDIR))
			KERNEL$SUICIDE("EXT2_AST: %s: TYPE %d (2)", VFS$FNODE_DESCRIPTION((FNODE *)f), RQ->type);

		e = (struct ext2_dir_entry_2 *)blk;
		end = (struct ext2_dir_entry_2 *)((char *)blk + ext2fs->s_blocksize);

		if (__unlikely(RQ->type == PG_READDIR)) {
			map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, (FS *)ext2fs, &maplen, &unmap);
			if (__unlikely(!map)) {
				VFS$PUT_BUFFER(blk);
				goto pagefault;
			}
			if (__unlikely(map == (void *)1)) {
				error = 0;
				goto err_put;
			}
			if (__unlikely(__IS_ERR(map))) {
				error = __PTR_ERR(map);
				goto err_put;
			}
		} else map = NULL;	/* not needed, but warning go away */

		next_de:
		next = (struct ext2_dir_entry_2 *)((char *)e + __16LE2CPU(e->rec_len));
		/*__debug_printf("%p - (rec %d, namel %d, inode %x) END - %p\n", e, e->rec_len, e->name_len, e->inode, end);*/
		if (__unlikely((int)(ext2_blk_t)next & 3) || __unlikely(next > end) || __unlikely((char *)next < &e->name[e->name_len])) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, ext2fs->filesystem_name, "INVALID DIR ENTRY AT BLOCK %lu", (unsigned long)RQ->indir);
			ext2_error(ext2fs, 0);
			error = -EFSERROR;
			if (__unlikely(RQ->type == PG_READDIR)) VFS$UNMAP_READDIR(map, unmap);
			err_put:
			VFS$PUT_BUFFER(blk);
			goto err;
		}
		if (__likely(__32LE2CPU(e->inode))) {
			if ((e->name_len == 1 && e->name[0] == '.') ||
			    (e->name_len == 2 && e->name[0] == '.' && e->name[1] == '.')) goto skip;
			if (__unlikely(RQ->type == PG_READDIR)) {
				if (__unlikely((char *)e - (char *)blk < ((EXT2_READDIR_COOKIE *)rb->cookie)->offs)) goto skip_readdir;
				error = VFS$DO_READDIR_NAME((PAGEINRQ *)RQ, map, maplen, e->name, e->name_len, e->file_type == EXT2_FT_REG_FILE ? DT_REG : e->file_type == EXT2_FT_DIR ? DT_DIR : DT_UNKNOWN);
				if (__unlikely(error != 0)) {
					err_put_unmap_readdir:
					VFS$UNMAP_READDIR(map, unmap);
					goto err_put;
				}
				((EXT2_READDIR_COOKIE *)rb->cookie)->offs = (char *)next - (char *)blk;
				if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS)) {
					if (__unlikely(RQ->tag.proc == &KERNEL$PROC_KERNEL) || __unlikely(RQ->wr < 0)) {
						VFS$UNMAP_READDIR(map, unmap);
						VFS$PUT_BUFFER(blk);
						WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
						RETURN;
					} else {
						error = -EINTR;
						goto err_put_unmap_readdir;
					}
				}
				skip_readdir:;
			} else {
				EXT2FNODE *fn = (EXT2FNODE *)RQ->new_fnode;
				if (e->name_len == fn->namelen && !_memcasecmp(e->name, fn->name, e->name_len)) {
					RQ->depth = -2;
					RQ->indir = __32LE2CPU(e->inode);
					VFS$PUT_BUFFER(blk);
					goto get_inode;
				}
			}
			skip:;
		}
		if ((e = next) != end) goto next_de;
		VFS$PUT_BUFFER(blk);
		if (__unlikely(RQ->type == PG_READDIR)) {
			((EXT2_READDIR_COOKIE *)rb->cookie)->block = RQ->off + ext2fs->sectors_per_block_bits;
			((EXT2_READDIR_COOKIE *)rb->cookie)->offs = 0;
			VFS$UNMAP_READDIR(map, unmap);
		}
		skip_blk:
		RQ->off += ext2fs->sectors_per_block;
		RQ->blk = RQ->off >> ext2fs->sectors_per_block_bits;
		RQ->depth = 0;
		if (RQ->off << BIO_SECTOR_SIZE_BITS >= f->size) {
			if (RQ->type == PG_LOOKUP) {
				error = -ENOENT;
				goto err;
			} else {
				void *map;
				void (*unmap)(void *ptr);
				map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, (FS *)ext2fs, (void *)&KERNEL$LIST_END, &unmap);
				if (__unlikely(!map)) goto pagefault;
				if (__unlikely(map == (void *)1)) {
					error = 0;
					goto err;
				}
				if (__unlikely(__IS_ERR(map))) {
					error = __PTR_ERR(map);
					goto err;
				}
				((struct readdir_buffer *)map)->end = 1;
				VFS$UNMAP_READDIR(map, unmap);
				if (VFS$DONE_READDIR((PAGEINRQ *)RQ)) goto repost;
				error = 0;
				goto err;
			}
					
		}
		goto search_next_block;
#undef rb
	}

	err:
	switch (RQ->type) {
		case PG_LOOKUP:
			if (__unlikely(error != -ENOENT) || __likely(!(RQ->wr & (O_CREAT | _O_RENAME)))) {
				caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, error == -ENOENT);
				caller->status = error;
				RETURN_AST(caller);
			}
			caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, 1);
			RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);
		case PG_READDIR:
			caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
			caller->status = error;
			RETURN_AST(caller);
		case PG_BMAP:
			RQ->status = error;
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		default:
			KERNEL$SUICIDE("EXT2_AST: %s: TYPE %d (1)", VFS$FNODE_DESCRIPTION((FNODE *)f), RQ->type);
	}

	repost:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);

	pagefault:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	DO_PAGEIN(caller, &((IOCTLRQ *)caller)->v, PF_RW);
#undef f
#undef ext2fs
}

