#include <SPAD/SYSLOG.H>

#include "FAT.H"

extern AST_STUB FAT_BMAP_AST;

#define MAX_READAHEAD 1024

void FAT_BMAP(PAGEINRQ *pgin_)
{
	FATPAGEINRQ *pgin = (FATPAGEINRQ *)pgin_;
	FATFNODE *f = (FATFNODE *)pgin->fnode;
#define fatfs ((FATFS *)f->fs)
	if (__unlikely(!f->start_cluster)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "NO CLUSTER CHAIN");
		efserror:
		pgin->status = -EFSERROR;
		VFS$BMAP_DONE((PAGEINRQ *)pgin);
		return;
	}
	pgin->fn = FAT_BMAP_AST;
	pgin->c[1] = 0;
	pgin->current_fat = 0;
	f->file_blk = 0;
	f->disk_blk = FAT_CLUSTER_2_SECTOR(fatfs, f->start_cluster);
	f->run_length = fatfs->sectors_per_cluster;
	pgin->dirptr = f->disk_blk + fatfs->sectors_per_cluster - 1;
	if (is_bmap_invalid(f)) {
		FAT_INVALID_BMAP_MSG(f);
		goto efserror;
	}
	if (f->disk_size <= fatfs->sectors_per_cluster << BIO_SECTOR_SIZE_BITS) {
		if (__likely(VFS$BMAP_OK(pgin->fnode, pgin->off))) {
			VFS$BMAP_DONE((PAGEINRQ *)pgin);
			return;
		}
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "ATTEMPTING TO BMAP OUT OF FILE: %016"__d_off_format"X", pgin->off);
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "ATTEMPTING TO BMAP OUT OF FILE: %016"__d_off_format"X (FILE %s)", pgin->off, f->name);
		goto efserror;
	}
	CALL_AST(pgin);
#undef fatfs
}

DECL_AST(FAT_BMAP_AST, SPL_FS, FATPAGEINRQ)
{
	unsigned long old, to_read;
#define f ((FATFNODE *)RQ->fnode)
#define fatfs ((FATFS *)RQ->fs)
	IO_DISABLE_CHAIN_CANCEL(SPL_FS, RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status != 0)) {
		if (__unlikely(++RQ->current_fat >= fatfs->fats)) {
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		RQ->status = 0;
	}
	if (0) {
		again:
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
			WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
			RETURN;
		}
	}
	old = RQ->dirptr;
	if (__unlikely(FAT_GET_NEXT_SECTOR(fatfs, &RQ->dirptr, &to_read, RQ->current_fat))) {
		if (VFS$BMAP_OK(RQ->fnode, RQ->off)) {
			ok:
			VFS$BMAP_DONE((PAGEINRQ *)RQ);
			RETURN;
		}
		VFS$READ_BUFFER(to_read, (PAGEINRQ *)RQ, 0);
		RETURN;
	}
	RQ->current_fat = 0;
	if (!RQ->dirptr) {
		if (__likely(VFS$BMAP_OK(RQ->fnode, RQ->off))) goto ok;
		goto out;
	}
	if (__likely(RQ->dirptr == old + 1)) {
		f->run_length++;
	} else {
		if (__likely(VFS$BMAP_OK(RQ->fnode, RQ->off))) goto ok;
		f->file_blk += f->run_length;
		f->disk_blk = RQ->dirptr;
		f->run_length = 1;
	}
	if (is_bmap_invalid(f)) {
		FAT_INVALID_BMAP_MSG(f);
		goto fserr;
	}
	if (__unlikely(FAT_STOP_CYCLES(RQ->dirptr, &RQ->c))) goto cycle;
	goto again;

	cycle:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (BMAP)", RQ->dirptr);
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (BMAP %s:%016"__d_off_format"X)", RQ->dirptr, RQ->fnode->name, RQ->off);
	fserr:
	RQ->status = -EFSERROR;
	VFS$BMAP_DONE((PAGEINRQ *)RQ);
	RETURN;

	out:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "EOF DETECTED");
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "EOF DETECTED (BMAP %s:%016"__d_off_format"X)", RQ->fnode->name, RQ->off);
	goto fserr;
#undef f
#undef fatfs
}

int FAT_SYNC_BMAP(FNODE *f_, __d_off off, int try)
{
#define f ((FATFNODE *)f_)
#define fatfs ((FATFS *)f->fs)
	unsigned long dirptr, old;
	unsigned long c[2];
	c[1] = 0;
	if (__unlikely(!f->start_cluster)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "NO CLUSTER CHAIN");
		return -EFSERROR;
	}
	dirptr = FAT_CLUSTER_2_SECTOR(fatfs, f->start_cluster);
	f->file_blk = 0;
	f->disk_blk = dirptr;
	f->run_length = 1;
	if (is_bmap_invalid(f)) {
		FAT_INVALID_BMAP_MSG(f);
		return -EFSERROR;
	}
	again:
	old = dirptr;
	if (__likely(!try)) {
		dirptr = FAT_GET_NEXT_SECTOR_SYNC(fatfs, dirptr);
	} else {
		unsigned long dp = dirptr;	/* allow dirptr to be in register */
		if (__unlikely(FAT_GET_NEXT_SECTOR(fatfs, &dp, (void *)&KERNEL$LIST_END, 0))) {
			if (VFS$BMAP_OK(f_, off)) return 0;
			return -ENOENT;
		}
		dirptr = dp;
	}
	if (!dirptr) {
		if (__likely(VFS$BMAP_OK(f_, off))) return 0;
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "EOF DETECTED ON SYNC BMAP");
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "EOF DETECTED (SYNC BMAP %s:%016"__d_off_format"X)", f->name, off);
		return -EFSERROR;
	}
	if (__likely(dirptr == old + 1)) {
		f->run_length++;
	} else {
		if (__likely(VFS$BMAP_OK(f_, off))) return 0;
		f->file_blk += f->run_length;
		f->disk_blk = dirptr;
		f->run_length = 1;
	}
	if (is_bmap_invalid(f)) {
		FAT_INVALID_BMAP_MSG(f);
		return -EFSERROR;
	}
	if (__unlikely(FAT_STOP_CYCLES(dirptr, &c))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (SYNC BMAP)", dirptr);
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, fatfs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (SYNC BMAP %s:%016"__d_off_format"X)", dirptr, f->name, off);
		return -EFSERROR;
	}
	goto again;
#undef f
#undef fatfs
}

int FAT_GROW(FATFNODE *f, unsigned long n_clusters)
{
#define fatfs ((FATFS *)f->fs)
	unsigned long ncl, new_chain;
	unsigned long ocl = 0;
	unsigned long cl = f->start_cluster;
	unsigned long c[2];
	c[1] = 0;
	while (cl >= 2) {
		ocl = cl;
		cl = FAT_READ(fatfs, cl);
		if (IS_EOF(cl)) break;
		if (__unlikely(cl < 2) || __unlikely(cl >= fatfs->clusters)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "BOGUS CLUSTER %08lX AT POSITION %08lX WHEN EXTENDING FILE", cl, ocl);
			return -EFSERROR;
		}
		if (__unlikely(FAT_STOP_CYCLES(cl, &c))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "CYCLE DETECTED ON CLUSTER %08lX WHEN EXTENDING FILE", cl);
			return -EFSERROR;
		}
	}
	new_chain:
	ncl = n_clusters;
	new_chain = FAT_ALLOC(fatfs, ocl ? ocl + 1 : FAT_SECTOR_2_CLUSTER(fatfs, ((FATFNODE *)f->parent)->dirent_sec), &ncl, ocl ? 0 : 1);
	if (!new_chain) return -ENOSPC;
	if (ocl) FAT_WRITE(fatfs, ocl, new_chain);
	else f->start_cluster = new_chain;
	ocl = new_chain + ncl - 1;
	if ((n_clusters -= ncl)) goto new_chain;
	return 0;
#undef fatfs
}

int FAT_SHRINK(FATFNODE *f, unsigned long leave)
{
#define fatfs ((FATFS *)f->fs)
	unsigned long ocl = 0;	/* warning, go away */
	unsigned long cl;
	f->run_length = 0;
	cl = f->start_cluster;
	while (leave--) {
		ocl = cl;
		cl = FAT_READ(fatfs, cl);
		if (__unlikely(cl < 2) || __unlikely(cl >= fatfs->clusters)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "BOGUS CLUSTER %08lX AT POSITION %08lX WHEN TRUNCATING FILE", cl, ocl);
			return -EFSERROR;
		}
	}
	if (__unlikely(!cl)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fatfs->filesystem_name, "FILE HAS NO CLUSTERS");
		return -EFSERROR;
	}
	FAT_FREE(fatfs, cl);
	if (ocl) FAT_WRITE(fatfs, ocl, CLUSTER_EOF);
	else f->start_cluster = 0;
	return 0;
#undef fatfs
}

void FAT_INVALID_BMAP_MSG(FATFNODE *f)
{
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, f->fs->filesystem_name, "%s: BMAP OUT OF FILESYSTEM: %"__d_off_format"X, %"__d_off_format"X, %"__d_off_format"X", VFS$FNODE_DESCRIPTION((FNODE *)f), f->disk_blk, f->file_blk, f->run_length);
	f->disk_blk = 0;
	f->file_blk = 0;
	f->run_length = 0;
}
