#include <ENDIAN.H>
#include <STDLIB.H>
#include <VALUES.H>
#include <SPAD/SYSLOG.H>
#include <ARCH/ROL.H>

#include "FAT.H"
#include "STRUCT.H"

/* parts from linux/fs/fat/ */

static __finline__ int not_allowed_char(unsigned char c)
{
	return	__unlikely(c < ' ') ||
		__unlikely(c == '"') ||
		__unlikely(c == '*') ||
		__unlikely(c == '/') ||
		__unlikely(c == ':') ||
		__unlikely(c == '<') ||
		__unlikely(c == '>') ||
		__unlikely(c == '?') ||
		__unlikely(c == '\\') ||
		__unlikely(c == '|');
}

static __finline__ int no_8_3_char(unsigned char c)
{
	return	__unlikely(c == ' ') ||
		__unlikely(c == '+') ||
		__unlikely(c == ',') ||
		__unlikely(c == ';') ||
		__unlikely(c == '=') ||
		__unlikely(c == '[') ||
		__unlikely(c == ']') ||
		__unlikely(c == '.');
}

static int VALIDATE_FILENAME(__const__ char *name, int vfat)
{
	int i;
	unsigned char c;
	if (!vfat) {
		int dp = -1;
		for (i = 0; (c = name[i]); i++) {
			if (c == '.') {
				if (__unlikely(dp != -1)) return -ENAMETOOLONG;
				if (__unlikely(!i) || __unlikely(!name[i + 1])) return -EINVAL;
				dp = i;
				continue;
			}
			if (__unlikely(not_allowed_char(c)) || __unlikely(no_8_3_char(c)))
				return -EINVAL;
		}
		if (__unlikely(dp == -1)) {
			if (__unlikely(!i)) return -EINVAL;
			if (__unlikely(i > 8)) return -ENAMETOOLONG;
		} else {
			if (__unlikely(dp > 8) || __unlikely(i - dp > 4)) return -ENAMETOOLONG;
		}
		return 0;
	} else {
		for (i = 0; (c = name[i]); i++) {
			if (__unlikely(not_allowed_char(c))) return -EINVAL;
		}
		if (__unlikely(!i)) return -EINVAL;
		if (__unlikely(name[i - 1] == '.') || __unlikely(name[i - 1] == ' ')) return -EINVAL;
		if (__unlikely(i >= 256)) return -ENAMETOOLONG;
		return 0;
	}
}

int FAT_VALIDATE_FILENAME(FS *fs_, __const__ char *name)
{
#define fs ((FATFS *)fs_)
	return VALIDATE_FILENAME(name, fs->flags & FAT_VFAT);
#undef fs
}

static __finline__ __u8 shortnamesum(__u8 *chr)
{
	int i;
	__u8 sum;
	for (sum = 0, i = 0; i < 11; i++)
		sum = __ROR8(sum, 1) + chr[i];
	return sum;
}

static void put_char(__const__ char **s, __u8 *p)
{
	if (!*s) p[0] = p[1] = 0xff;
	else if (!(p[0] = *(*s)++)) *s = NULL;
}

static int FIND_SHORT_NAME(FATFNODE *dir, char *name)
{
#define fs ((FATFS *)dir->fs)
	unsigned long c[2];
	struct fat_dir_entry *e;
	int i;
	unsigned long sec = FAT_CLUSTER_2_SECTOR(fs, dir->start_cluster);
	c[1] = 0;
	again:
	e = FAT_READ_BUFFER_SYNC(fs, sec, 0);
	if (__unlikely(__IS_ERR(e))) return __PTR_ERR(e);
	for (i = 0; i < BIO_SECTOR_SIZE / FAT_DIRENT_SIZE; i++) {
		if (!memcmp(e[i].name, name, 11)) {
			VFS$PUT_BUFFER(e);
			return 1;
		}
		if (!e[i].name[0]) {
			VFS$PUT_BUFFER(e);
			return 0;
		}
	}
	VFS$PUT_BUFFER(e);
	sec = FAT_GET_NEXT_SECTOR_SYNC(fs, sec);
	if (sec) {
		if (__unlikely(FAT_STOP_CYCLES(sec, &c))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (FIND SHORT NAME)", sec);
			return -EFSERROR;
		}
		goto again;
	}
	return 0;
#undef fs
}

int FAT_CREATE_SLOTS_FROM_NAME(FATFNODE *dir, __const__ char *name, struct fat_dir_entry slots[32])
{
	__const__ char *k;
	int n_slots;
	int i, j, dot, r;
	if (dir->fs->flags & FAT_VFAT) {
		if (!VALIDATE_FILENAME(name, 0)) {
			for (i = 0; name[i]; i++)
				if (name[i] >= 'a' && name[i] <= 'z') goto long_;
			goto short_;
		}
		long_:
		n_slots = (strlen(name) + 12) / 13;
		if (__unlikely(n_slots > 31)) KERNEL$SUICIDE("FAT_CREATE_SLOTS_FROM_NAME: %s: NAME LEN %d", VFS$FNODE_DESCRIPTION((FNODE *)dir), (int)strlen(name));
		memset(slots, 0, (n_slots + 1) * FAT_DIRENT_SIZE);
		k = name;
		for (i = n_slots - 1; i >= 0; i--) {
			struct fat_dir_slot *slot = (struct fat_dir_slot *)&slots[i];
			slot->id = n_slots - i;
			if (!i) slot->id |= DIRSLOT_START;
			put_char(&k, &slot->name0_4[0]);
			put_char(&k, &slot->name0_4[2]);
			put_char(&k, &slot->name0_4[4]);
			put_char(&k, &slot->name0_4[6]);
			put_char(&k, &slot->name0_4[8]);
			slot->attr = ATTR_EXT;
			put_char(&k, &slot->name5_10[0]);
			put_char(&k, &slot->name5_10[2]);
			put_char(&k, &slot->name5_10[4]);
			put_char(&k, &slot->name5_10[6]);
			put_char(&k, &slot->name5_10[8]);
			put_char(&k, &slot->name5_10[10]);
			put_char(&k, &slot->name11_12[0]);
			put_char(&k, &slot->name11_12[2]);
		}
		goto make_short;
	} else {
		struct fat_dir_entry *slot;
		short_:
		n_slots = 0;
		memset(slots, 0, FAT_DIRENT_SIZE);

		make_short:
		slot = &slots[n_slots];
		memset(slot->name, ' ', 11);

		for (i = 0; name[i] == '.' || name[i] == ' '; i++) ;
		j = 0;
		for (; name[i] && name[i] != '.' && j < 8; i++) {
			if (__unlikely(name[i] == ' ')) continue;
			if (__unlikely(no_8_3_char(name[i]))) slot->name[j++] = '_';
			else slot->name[j++] = __upcasechr(name[i]);
		}
		if (__unlikely(slot->name[0] == DELETED_FLAG)) slot->name[0] = DELETED_REPLACE;
		dot = 0;
		for (; name[i]; i++) if (name[i] == '.') dot = i + 1;
		if (dot) {
			j = 8;
			for (i = dot; name[i] && j < 11; i++) {
				if (__unlikely(name[i] == ' ')) continue;
				if (__unlikely(no_8_3_char(name[i]))) slot->name[j++] = '_';
				else slot->name[j++] = __upcasechr(name[i]);
			}
		}

		if (!(dir->fs->flags & FAT_VFAT)) goto ret;
		if (n_slots) {
			for (i = 0; i < 6; i++) if (__unlikely(slot->name[i] == ' ')) break;
			slot->name[i] = '~';
			slot->name[i + 1] = '1';
		}
		if (__unlikely((r = FIND_SHORT_NAME(dir, slot->name)))) {
			unsigned number;
			char numstr[7];
			if (__unlikely(r < 0)) return r;

			if (__unlikely(!n_slots)) goto long_;

			slot->name[1] = '~';
			number = arc4random() % 900000;
			for (i = 0; i < 900000; i++) {
				_snprintf(numstr, sizeof numstr, "%06d", number + 100000);
				memcpy(&slot->name[2], numstr, 6);
				if (__likely(!(r = FIND_SHORT_NAME(dir, slot->name)))) goto ret;
				if (__unlikely(r < 0)) return r;
				if (__unlikely(++number == 900000)) number = 0;
			}
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, dir->fs->filesystem_name, "TOO MANY SHORT NAME COLISIONS");
			return -ENOSPC;
		}
	}
	ret:
	if (n_slots) {
		__u8 sum = shortnamesum(slots[n_slots].name);
		for (i = 0; i < n_slots; i++) ((struct fat_dir_slot *)&slots[i])->alias_checksum = sum;
	}
	return n_slots;
}

void FAT_INIT_FNODE(FNODE *f_)
{
#define f ((FATFNODE *)f_)
	f->dirent_sec = 0;
	f->dirent_pos = 0;
	f->n_slots = 0;
	f->start_cluster = 0;
#undef f
}

static int day_n[] = { 0,31,59,90,120,151,181,212,243,273,304,334,0,0,0,0 };
		  /* JanFebMarApr May Jun Jul Aug Sep Oct Nov Dec */

static int date_dos2unix(unsigned short time,unsigned short date)
{
	int month,year,secs;

	/* first subtract and mask after that... Otherwise, if
	   date == 0, bad things happen */
	month = ((date >> 5) - 1) & 15;
	year = date >> 9;
	secs = (time & 31)*2+60*((time >> 5) & 63)+(time >> 11)*3600+86400*
	    ((date & 31)-1+day_n[month]+(year/4)+year*365-((year & 3) == 0 &&
	    month < 2 ? 1 : 0)+3653);
			/* days since 1.1.70 plus 80's leap day */
	/*secs += sys_tz.tz_minuteswest*60;*/
	return secs;
}

void fat_date_unix2dos(int unix_date,unsigned short *time,
    unsigned short *date)
{
	int day,year,nl_day,month;

	/*unix_date -= sys_tz.tz_minuteswest*60;*/

	/* Jan 1 GMT 00:00:00 1980. But what about another time zone? */
	if (unix_date < 315532800)
		unix_date = 315532800;

	*time = (unix_date % 60)/2+(((unix_date/60) % 60) << 5)+
	    (((unix_date/3600) % 24) << 11);
	day = unix_date/86400-3652;
	year = day/365;
	if ((year+3)/4+365*year > day) year--;
	day -= (year+3)/4+365*year;
	if (day == 59 && !(year & 3)) {
		nl_day = day;
		month = 2;
	}
	else {
		nl_day = (year & 3) || day <= 59 ? day : day-1;
		for (month = 0; month < 12; month++)
			if (day_n[month] > nl_day) break;
	}
	*date = nl_day-day_n[month-1]+1+(month << 5)+(year << 9);
}

static __finline__ void FAT_GET_FNODE(FATFNODE *f, struct fat_dir_entry *dirblk, int slots, unsigned long off, int i)
{
	f->mtime = date_dos2unix(__16LE2CPU(dirblk->time), __16LE2CPU(dirblk->date));
	if (f->fs->flags & FAT_VFAT) {
		f->ctime = date_dos2unix(__16LE2CPU(dirblk->ctime), __16LE2CPU(dirblk->cdate));
	} else {
		f->ctime = f->mtime;
	}
	f->dirent_sec = off;
	f->dirent_pos = i;
	f->n_slots = slots;
	FNODE_OUT_OF_WANTFREE((FNODE *)f);
	if (__likely(!(dirblk->attr & ATTR_DIR))) {
		f->flags = FNODE_FILE;
		f->run_length = 0;
		f->size = __32LE2CPU(dirblk->size);
	} else {
		if (!(f->flags & FNODE_DIRECTORY)) {
			VFS_INIT_DIR((FNODE *)f);
		}
		f->flags = FNODE_DIRECTORY;
		f->size = 0;
	}
	f->disk_size = f->size;
	f->start_cluster = __16LE2CPU(dirblk->start);
	if (((FATFS *)f->fs)->fat_bits == 32)
		f->start_cluster |= __16LE2CPU(dirblk->starthi) << 16;
}

extern AST_STUB FAT_READDIR_AST;

void FAT_INIT_READDIR_COOKIE(struct readdir_buffer *buf, FNODE *f)
{
	int *c = (int *)buf->cookie;
	buf->cookie_size = sizeof(int);
	*c = 0;
}

void FAT_READDIR(PAGEINRQ *readdir, struct readdir_buffer *buf)
{
	FATPAGEINRQ *r = (FATPAGEINRQ *)readdir;
	r->fn = FAT_READDIR_AST;
	if (buf) {
		r->cookie = *(int *)buf->cookie;
	} else {
		r->cookie = 0;
	}
	r->dir_off = r->cookie;
	r->dirptr = FAT_CLUSTER_2_SECTOR((FATFS *)((FATFNODE *)r->fnode)->fs, ((FATFNODE *)r->fnode)->start_cluster);
	r->n_slots = 0;
	r->c[1] = 0;
	r->current_fat = 0;
	CALL_AST(r);
}

static int copy_name(__u8 *chr, FATPAGEINRQ *rq)
{
	if ((!chr[0] && !chr[1]) || (chr[0] == 0xff && chr[1] == 0xff)) {
		if (__unlikely(rq->longnamelen)) return 1;
		return 0;
	}
	if (__unlikely(rq->longnamelen == 255)) return 1;
	/* !!! TODO: translate unicode here */
	rq->longname[rq->longnamelen] = chr[0];
	if (not_allowed_char(rq->longname[rq->longnamelen])) rq->longname[rq->longnamelen] += 0x80;
	rq->longnamelen++;
	return 0;
}

/* return 0 - eof
	  1 - nothing
	  2 - got name
*/

static int FAT_GET_NEXT_NAME(struct fat_dir_entry *dirblk, FATPAGEINRQ *rq)
{
#define dirslot	((struct fat_dir_slot *)dirblk)
	if (__unlikely(!dirblk->name[0])) return 0;
	if (dirblk->name[0] == DELETED_FLAG) {
		skip:
		rq->n_slots = 0;
		return 1;
	}
	if (dirblk->attr == ATTR_EXT && rq->fs->flags & FAT_VFAT) {
		if (__unlikely(dirslot->id & DIRSLOT_INVALID)) goto skip;
		if (__unlikely(!(dirslot->id & DIRSLOT_SLOTS))) goto skip;
		if (dirslot->id & DIRSLOT_START) {
			rq->n_slots = rq->longnameslots = dirslot->id & DIRSLOT_SLOTS;
			rq->longnamelen = 0;
			rq->shortnamesum = dirslot->alias_checksum;
		} else {
			if (__unlikely(!rq->n_slots)) goto skip;
			if (__unlikely(rq->shortnamesum != dirslot->alias_checksum)) goto skip;
			rq->longnameslots--;
			if (__unlikely(rq->longnameslots != (dirslot->id & DIRSLOT_SLOTS))) goto skip;
		}
		if (__unlikely(copy_name(&dirslot->name11_12[2], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name11_12[0], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name5_10[10], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name5_10[8], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name5_10[6], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name5_10[4], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name5_10[2], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name5_10[0], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name0_4[8], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name0_4[6], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name0_4[4], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name0_4[2], rq))) goto skip;
		if (__unlikely(copy_name(&dirslot->name0_4[0], rq))) goto skip;
		return 1;
	}
	if (dirblk->attr & ATTR_VOLUME) goto skip;
	if (!memcmp(dirblk->name, ".          ", 11) || !memcmp(dirblk->name, "..         ", 11)) goto skip;
	if (rq->longnameslots != 1) goto no_long;
	if (rq->n_slots && __likely(rq->longnamelen)) {
		int i;
		if (shortnamesum(dirblk->name) != rq->shortnamesum) goto no_long;
		for (i = (rq->longnamelen - 1) / 2; i >= 0; i--) {
			char c = rq->longname[i];
			rq->longname[i] = rq->longname[rq->longnamelen - i - 1];
			rq->longname[rq->longnamelen - i - 1] = c;
		}
	} else {
		int i;
		no_long:;
		rq->n_slots = 0;
		for (i = 0; i < 8; i++) rq->longname[i] = dirblk->name[i];
		rq->longnamelen = 8;
		if (rq->longname[0] == DELETED_REPLACE) rq->longname[0] = DELETED_FLAG;
		for (i = 7; i >= 0; i--) if (rq->longname[i] == ' ') rq->longnamelen--;
					 else break;
		if (__likely(dirblk->ext[0] != ' ') || __unlikely(dirblk->ext[1] != ' ') || __unlikely(dirblk->ext[2] != ' ')) {
			rq->longname[rq->longnamelen++] = '.';
			for (i = 0; i < 3; i++) rq->longname[rq->longnamelen++] = dirblk->ext[i];
			for (i = 2; i >= 0; i--) if (dirblk->ext[i] == ' ') rq->longnamelen--;
						 else break;
		}
	}
	/* {
		int i;
		for (i = 0; i < rq->longnamelen; i++) if (rq->longname[i] >= 'a' && rq->longname[i] <= 'z') rq->longname[i] -= 0x20;
	} */
	rq->longname[rq->longnamelen] = 0;
	rq->shortnamesum = 256;
	return 2;
#undef dirslot
}

DECL_AST(FAT_READDIR_AST, SPL_FS, FATPAGEINRQ)
{
	struct fat_dir_entry *dirblk;
	IORQ *caller;
	long error;
	int r;
	IO_DISABLE_CHAIN_CANCEL(SPL_FS, RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status < 0)) {
		if (__unlikely(++RQ->current_fat >= ((FATFS *)RQ->fs)->fats)) {
			error = RQ->status;
			goto err;
		}
		RQ->status = 0;
	}
	if (0) {
		again:
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
			WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
			RETURN;
		}
	}
	if (RQ->dir_off >= BIO_SECTOR_SIZE / FAT_DIRENT_SIZE) {
		unsigned long sector;
		if (__unlikely(FAT_GET_NEXT_SECTOR((FATFS *)RQ->fs, &RQ->dirptr, &sector, RQ->current_fat))) {
			VFS$READ_BUFFER(sector & -((FATFS *)RQ->fs)->sectors_per_sector, (PAGEINRQ *)RQ, FAT_READAHEAD((FATFS *)RQ->fs));
			RETURN;
		}
		RQ->current_fat = 0;
		if (!RQ->dirptr) goto eof;
		if (__unlikely(FAT_STOP_CYCLES(RQ->dirptr, &RQ->c))) goto cycle;
		RQ->dir_off -= BIO_SECTOR_SIZE / FAT_DIRENT_SIZE;
		goto again;
	}
	dirblk = VFS$GET_BUFFER(RQ->fs, RQ->dirptr, 1, (void *)&KERNEL$LIST_END, RQ->tag.proc);
	if (__unlikely(!dirblk)) {
		RQ->current_fat = MAXINT - 1;
		VFS$READ_BUFFER(RQ->dirptr & -((FATFS *)RQ->fs)->sectors_per_sector, (PAGEINRQ *)RQ, 0);
		RETURN;
	}
	RQ->current_fat = 0;
	again_2:
	r = FAT_GET_NEXT_NAME(&dirblk[RQ->dir_off], RQ);
	if (r != 2) {
		if (r == 1) {
			RQ->dir_off++;
			RQ->cookie++;
			next_entry:
			if (RQ->dir_off >= BIO_SECTOR_SIZE / FAT_DIRENT_SIZE) {
				VFS$PUT_BUFFER(dirblk);
				goto again;
			}
			goto again_2;
		}
		VFS$PUT_BUFFER(dirblk);
		eof:
		{
			void *map;
			void (*unmap)(void *ptr);
			map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, RQ->fs, (void *)&KERNEL$LIST_END, &unmap);
			if (__unlikely(!map)) {
				goto pagefault;
			}
			if (__unlikely(map == (void *)1)) {
				error = 0;
				goto err;
			}
			if (__unlikely(__IS_ERR(map))) {
				error = __PTR_ERR(map);
				goto err;
			}
			((struct readdir_buffer *)map)->end = 1;
			VFS$UNMAP_READDIR(map, unmap);
		}
		if (VFS$DONE_READDIR((PAGEINRQ *)RQ)) goto repost;
		error = 0;
		goto err;
	} else {
		void *map;
		unsigned maplen;
		void (*unmap)(void *ptr);
		map = VFS$MAP_READDIR((IOCTLRQ *)RQ->caller, RQ->wr, RQ->fs, &maplen, &unmap);
		if (__unlikely(!map)) {
			VFS$PUT_BUFFER(dirblk);
			goto pagefault;
		}
		if (__unlikely(map == (void *)1)) {
			error = 0;
			goto err_put;
		}
		if (__unlikely(__IS_ERR(map))) {
			error = __PTR_ERR(map);
			goto err_put;
		}
		error = VFS$DO_READDIR_NAME((PAGEINRQ *)RQ, map, maplen, RQ->longname, RQ->longnamelen, dirblk[RQ->dir_off].attr & ATTR_DIR ? DT_DIR : DT_REG);
		if (__unlikely(error != 0)) {
			VFS$UNMAP_READDIR(map, unmap);
			goto err_put;
		}
		RQ->dir_off++;
		RQ->cookie++;
		*(int *)((struct readdir_buffer *)map)->cookie = RQ->cookie;
		VFS$UNMAP_READDIR(map, unmap);
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS)) {
			if (__unlikely(RQ->tag.proc == &KERNEL$PROC_KERNEL) || __unlikely(RQ->wr < 0)) {
				VFS$PUT_BUFFER(dirblk);
				WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
				RETURN;
			} else {
				error = -EINTR;
				goto err_put;
			}
		}
		goto next_entry;
	}


	err_put:
	VFS$PUT_BUFFER(dirblk);
	err:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	caller->status = error;
	RETURN_AST(caller);

	repost:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);

	pagefault:
	caller = VFS$FREE_PAGEIN((PAGEINRQ *)RQ);
	DO_PAGEIN(caller, &((IOCTLRQ *)caller)->v, PF_RW);

	cycle:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, RQ->fs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (READDIR)", RQ->dirptr);
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, RQ->fs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (READDIR %s)", RQ->dirptr, RQ->fnode->name);
	error = -EFSERROR;
	goto err;
}

extern AST_STUB FAT_LOOKUP_AST;

void FAT_LOOKUP(PAGEINRQ *p)
{
	FATPAGEINRQ *r = (FATPAGEINRQ *)p;
	r->fn = FAT_LOOKUP_AST;
	r->dirptr = FAT_CLUSTER_2_SECTOR((FATFS *)((FATFNODE *)r->fnode)->fs, ((FATFNODE *)r->fnode)->start_cluster);
	r->dir_off = 0;
	r->n_slots = 0;
	r->c[1] = 0;
	r->current_fat = 0;
	CALL_AST(r);
}

DECL_AST(FAT_LOOKUP_AST, SPL_FS, FATPAGEINRQ)
{
	IORQ *caller;
	long error;
	int i, r;
	struct fat_dir_entry *dirblk;
	IO_DISABLE_CHAIN_CANCEL(SPL_FS, RQ->caller);
	SWITCH_PROC_ACCOUNT(RQ->tag.proc, SPL_X(SPL_FS));
	if (__unlikely(RQ->status < 0)) {
		if (__unlikely(++RQ->current_fat >= ((FATFS *)RQ->fs)->fats)) {
			error = RQ->status;
			goto err;
		}
		RQ->status = 0;
	}
	if (RQ->dir_off) {
		unsigned long sector;
		fat_next:
		if (__unlikely(FAT_GET_NEXT_SECTOR((FATFS *)RQ->fs, &RQ->dirptr, &sector, RQ->current_fat))) {
			VFS$READ_BUFFER(sector & -((FATFS *)RQ->fs)->sectors_per_sector, (PAGEINRQ *)RQ, FAT_READAHEAD((FATFS *)RQ->fs));
			RETURN;
		}
		RQ->current_fat = 0;
		if (!RQ->dirptr) goto eof;
		if (__unlikely(FAT_STOP_CYCLES(RQ->dirptr, &RQ->c))) goto cycle;
		RQ->dir_off = 0;
		goto next_sector;
	}
	if (0) {
		subdir:
		next_sector:
		if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
			WQ_WAIT_PAGEINRQ(&KERNEL$LOCKUP_EVENTS, (PAGEINRQ *)RQ);
			RETURN;
		}
	}
	dirblk = VFS$GET_BUFFER(RQ->fs, RQ->dirptr, 1, (void *)&KERNEL$LIST_END, RQ->tag.proc);
	if (__unlikely(!dirblk)) {
		RQ->current_fat = MAXINT - 1;
		VFS$READ_BUFFER(RQ->dirptr & -((FATFS *)RQ->fs)->sectors_per_sector, (PAGEINRQ *)RQ, 0);
		RETURN;
	}
	RQ->current_fat = 0;
	for (i = 0; i < BIO_SECTOR_SIZE / FAT_DIRENT_SIZE; i++) {
		FATFNODE *fn;
		int was_dir;
		r = FAT_GET_NEXT_NAME(&dirblk[i], RQ);
		if (r != 2) {
			if (!r) {
				VFS$PUT_BUFFER(dirblk);
				goto eof;
			}
			continue;
		}
		fn = (FATFNODE *)RQ->new_fnode;
		if (RQ->longnamelen != fn->namelen) continue;
		if (_memcasecmp(RQ->longname, fn->name, RQ->longnamelen)) continue;
		was_dir = 0;
		if (fn->flags & FNODE_DIRECTORY) {
			was_dir = 1;
			if (__unlikely(!(dirblk[i].attr & ATTR_DIR))) {
				error = -ENOTDIR;
				goto err_put;
			}
		}
		RQ->fnode->readers--;
		WQ_WAKE_ALL(&RQ->fnode->wait);
		if (__unlikely(RQ->fnode->readers < 0))
			KERNEL$SUICIDE("FAT_LOOKUP_AST: %s: LOCK UNDERRUN: %d", VFS$FNODE_DESCRIPTION(RQ->fnode), RQ->fnode->readers);
		FAT_GET_FNODE(fn, &dirblk[i], RQ->n_slots, RQ->dirptr, i);
		WQ_WAKE_ALL(&fn->wait);
		VFS$PUT_BUFFER(dirblk);
		if (was_dir) {
			RQ->new_fnode = LIST_STRUCT(fn->u.d.clean.next, FNODE, dirent_entry);
			RQ->fnode = (FNODE *)fn;
			RQ->fnode->readers++;
			RQ->dirptr = FAT_CLUSTER_2_SECTOR((FATFS *)fn->fs, fn->start_cluster);
			RQ->c[1] = 0;
			goto subdir;
		}
		if (__likely((RQ->wr & (O_CREAT | O_EXCL)) != (O_CREAT | O_EXCL))) {
			caller = VFS$FREE_EMPTY_PAGEIN((PAGEINRQ *)RQ);
			RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);
		} else {
			caller = VFS$FREE_EMPTY_PAGEIN((PAGEINRQ *)RQ);
			caller->status = -EEXIST;
			RETURN_AST(caller);
		}
	}
	RQ->dir_off = 1;
	VFS$PUT_BUFFER(dirblk);
	goto fat_next;

	eof:
	error = -ENOENT;
	goto err;

	err_put:
	VFS$PUT_BUFFER(dirblk);

	err:
	if (__unlikely(error != -ENOENT) || __likely(!(RQ->wr & (O_CREAT | _O_RENAME)))) {
		caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, error == -ENOENT);
		caller->status = error;
		RETURN_AST(caller);
	}
	caller = VFS$FREE_LOOKUP_PAGEIN((PAGEINRQ *)RQ, 1);
	RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);

	cycle:
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, RQ->fs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (LOOKUP)", RQ->dirptr);
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR | __SYSLOG_SECRET, RQ->fs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (LOOKUP %s/%s)", RQ->dirptr, RQ->fnode->name, RQ->new_fnode->name);
	error = -EFSERROR;
	goto err;
}


