#include <SPAD/SYSLOG.H>
#include <ENDIAN.H>

#include "FAT.H"
#include "STRUCT.H"

static void SET_DATE(FATFNODE *f, struct fat_dir_entry *e)
{
	unsigned short time, date;
	fat_date_unix2dos(f->mtime, &time, &date);
	e->time = time, e->date = date;
	if (((FATFS *)f->fs)->flags & FAT_VFAT) {
		fat_date_unix2dos(f->ctime, &time, &date);
		e->ctime = time, e->cdate = date;
	}
}

int FAT_CREATE_FNODE(FNODE *f_)
{
#define f ((FATFNODE *)f_)
#define fs ((FATFS *)f->fs)
	f->dirent_sec = 0;
	f->dirent_pos = 0;
	f->n_slots = 0;
	f->start_cluster = 0;
	f->run_length = 0;
	if (__unlikely(f->flags & FNODE_DIRECTORY)) {
		struct fat_dir_entry *dirblk;
		int i;
		unsigned long len = 1;
		unsigned long sec;
		unsigned long cl = FAT_ALLOC(fs, ((FATFNODE *)f->parent)->start_cluster, &len, 1);
		if (!cl) return -ENOSPC;
		sec = FAT_CLUSTER_2_SECTOR(fs, cl);
		dirblk = VFS$ALLOC_BUFFER_SYNC((FS *)fs, sec, fs->sectors_per_sector, (void *)&KERNEL$LIST_END);
		if (__IS_ERR(dirblk)) {
			FAT_FREE(fs, cl);
			return __PTR_ERR(dirblk);
		}
		f->start_cluster = cl;
		memcpy(dirblk[0].name, ".          ", 11);
		dirblk[0].attr = ATTR_DIR;
		dirblk[0].start = __16CPU2LE(cl);
		dirblk[0].starthi = __16CPU2LE(cl >> 16);
		SET_DATE(f, &dirblk[0]);
		memcpy(dirblk[1].name, "..         ", 11);
		dirblk[1].attr = ATTR_DIR;
		dirblk[1].start = __16CPU2LE(((FATFNODE *)f->parent)->start_cluster);
		dirblk[1].starthi = __16CPU2LE(((FATFNODE *)f->parent)->start_cluster >> 16);
		SET_DATE((FATFNODE *)f->parent, &dirblk[1]);
		VFS$PUT_BUFFER(dirblk);
		for (i = 1; i < fs->sectors_per_cluster; i += fs->sectors_per_sector) {
			void *p = VFS$ALLOC_BUFFER_SYNC((FS *)fs, sec + i, fs->sectors_per_sector, (void *)&KERNEL$LIST_END);
			if (__likely(!__IS_ERR(p))) {
				VFS$PUT_BUFFER(p);
			}
		}
	}
	return 0;
#undef f
#undef fs
}

static void FAT_SET_DIRENT(FATFNODE *f, struct fat_dir_entry *e)
{
	FATFS *fs = (FATFS *)f->fs;
	e->start = __16CPU2LE(f->start_cluster);
	if (fs->fat_bits == 32) e->starthi = __16CPU2LE(f->start_cluster >> 16);
	if (!(f->flags & FNODE_DIRECTORY)) e->size = __32CPU2LE(f->disk_size);
	SET_DATE(f, e);
}

int FAT_WRITE_FNODE_DATA(FNODE *f_)
{
#define f ((FATFNODE *)f_)
#define fs ((FATFS *)f->fs)
	struct fat_dir_entry *e;
	int r;
	unsigned long fsize = f->size;
	if (__likely((f->flags & FNODE_FILE) != 0) && fsize != f->disk_size) {
		long l = FAT_SIZE_2_CLUSTERS(fs, fsize) - FAT_SIZE_2_CLUSTERS(fs, f->disk_size);
		int r = 0;
		f->disk_size = fsize;
		if (l > 0) r = FAT_GROW(f, l);
		else if (l < 0) r = FAT_SHRINK(f, FAT_SIZE_2_CLUSTERS(fs, fsize));
		if (r) return r;
	}
	if (__unlikely(r = VFS$WRITEPAGES((FNODE *)f))) return r;
	if (!f->dirent_sec) return 0;
	e = FAT_READ_BUFFER_SYNC(fs, f->dirent_sec, 1);
	if (__IS_ERR(e)) return __PTR_ERR(e);
	FAT_SET_DIRENT(f, &e[f->dirent_pos]);
	VFS$PUT_BUFFER(e);
	return 0;
#undef f
#undef fs
}

int FAT_ADD_FNODE_TO_DIRECTORY(FNODE *f_)
{
#define f ((FATFNODE *)f_)
#define fs ((FATFS *)f->fs)
	unsigned long c[2];
	unsigned long one;
	unsigned long cl, ncl;
	unsigned long sec, nsec, run_start_sector = 0;	/* warning go away */
	int run_start_pos = 0;				/* ----- "" ----- */
	int run_length;
	struct fat_dir_entry slots[32];
	struct fat_dir_entry *e;
	int i;
	int extend = 0;
	int eof = 0;

	int n_slots = FAT_CREATE_SLOTS_FROM_NAME((FATFNODE *)f->parent, f->name, slots);
	if (__unlikely(n_slots < 0)) return n_slots;
	slots[n_slots].attr = f->flags & FNODE_DIRECTORY ? ATTR_DIR : ATTR_ARCH;
	if (f->name[0] == '.') slots[n_slots].attr |= ATTR_HIDDEN;
	FAT_SET_DIRENT(f, &slots[n_slots]);

	sec = FAT_CLUSTER_2_SECTOR(fs, ((FATFNODE *)f->parent)->start_cluster);
	c[1] = 0;
	run_length = 0;
	next_sector:
	e = FAT_READ_BUFFER_SYNC(fs, sec, 0);
	if (__IS_ERR(e)) return __PTR_ERR(e);
	for (i = 0; i < BIO_SECTOR_SIZE / FAT_DIRENT_SIZE; i++) {
		if (__unlikely(!e[i].name[0])) eof = 1;
		if (__unlikely(eof) || __unlikely(e[i].name[0] == DELETED_FLAG)) {
			if (!run_length) {
				run_start_sector = sec;
				run_start_pos = i;
			}
			run_length++;
			if (run_length == n_slots + 1) {
				VFS$PUT_BUFFER(e);
				goto found;
			}
		} else run_length = 0;
}
	VFS$PUT_BUFFER(e);
	nsec = FAT_GET_NEXT_SECTOR_SYNC(fs, sec);
	if (nsec) {
		sec = nsec;
		if (__unlikely(FAT_STOP_CYCLES(sec, &c))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (ADD TO DIRECTORY)", sec);
			return -EFSERROR;
		}
		goto next_sector;
	}
	if (!((FATFNODE *)f->parent)->start_cluster) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ROOT DIRECTORY IS FULL");
		return -ENOSPC;
	}
	cl = FAT_SECTOR_2_CLUSTER(fs, sec);
	if (!cl) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FAT CHAIN POINTING TO ROOT");
		return -EFSERROR;
	}
	eof = 1;
	one = 1;
	if (!(ncl = FAT_ALLOC(fs, cl, &one, 1))) return -EFSERROR;
	FAT_WRITE(fs, cl, ncl);
	sec = FAT_CLUSTER_2_SECTOR(fs, ncl);
	for (i = 0; i < fs->sectors_per_cluster; i += fs->sectors_per_sector) {
		void *p = VFS$ALLOC_BUFFER_SYNC((FS *)fs, sec + i, fs->sectors_per_sector, (void *)&KERNEL$LIST_END);
		if (__likely(!__IS_ERR(p))) {
			VFS$PUT_BUFFER(p);
		}
	}
	extend++;
	if (extend <= 2) goto next_sector;
	KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNABLE TO EXTEND DIRECTORY");
	return -EFSERROR;

	found:

	i = 0;
	write_sector:
	e = FAT_READ_BUFFER_SYNC(fs, run_start_sector, 1);
	if (__IS_ERR(e)) return __PTR_ERR(e);
	write_slot:
	memcpy(&e[run_start_pos], &slots[i], FAT_DIRENT_SIZE);
	i++;
	if (i > n_slots) {
		if (eof) {
			if (__unlikely(run_start_pos + 1 >= BIO_SECTOR_SIZE / FAT_DIRENT_SIZE)) {
				unsigned long nrss;
				VFS$PUT_BUFFER(e);
				nrss = FAT_GET_NEXT_SECTOR_SYNC(fs, run_start_sector);
				if (__unlikely(!nrss)) goto done_zero;
				e = FAT_READ_BUFFER_SYNC(fs, nrss, 1);
				if (__IS_ERR(e)) return __PTR_ERR(e);
				e[0].name[0] = 0;
				VFS$PUT_BUFFER(e);
				goto done_zero;
			}
			e[run_start_pos + 1].name[0] = 0;
		}
		VFS$PUT_BUFFER(e);
		done_zero:
		f->dirent_pos = run_start_pos;
		f->dirent_sec = run_start_sector;
		f->n_slots = n_slots;
		goto ok;
	}
	run_start_pos++;
	if (__unlikely(run_start_pos >= BIO_SECTOR_SIZE / FAT_DIRENT_SIZE)) {
		VFS$PUT_BUFFER(e);
		run_start_sector = FAT_GET_NEXT_SECTOR_SYNC(fs, run_start_sector);
		if (!run_start_sector) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIRECTORY WAS TRUNCATED UNDER OPERATION");
			return -EFSERROR;
		}
		run_start_pos = 0;
		goto write_sector;
	}
	goto write_slot;
	ok:
	if (__unlikely(f->flags & FNODE_DIRECTORY)) {
		e = FAT_READ_BUFFER_SYNC(fs, FAT_CLUSTER_2_SECTOR(fs, f->start_cluster), 1);
		if (__unlikely(__IS_ERR(e))) return __PTR_ERR(e);
		if (memcmp(e[1].name, "..         ", 11)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIRECTORY DOES NOT CONTAIN .. ENTRY");
			VFS$PUT_BUFFER(e);
			return -EFSERROR;
		} else {
			e[1].start = __16CPU2LE(((FATFNODE *)f->parent)->start_cluster);
			e[1].starthi = __16CPU2LE(((FATFNODE *)f->parent)->start_cluster >> 16);
			SET_DATE((FATFNODE *)f->parent, &e[1]);
		}
		VFS$PUT_BUFFER(e);
	}
	return 0;
#undef f
#undef fs
}

int FAT_DELETE_FNODE(FNODE *f_)
{
#define f ((FATFNODE *)f_)
	if (f->start_cluster) FAT_FREE((FATFS *)f->fs, f->start_cluster);
	return 0;
#undef f
}

int FAT_REMOVE_FNODE_FROM_DIRECTORY(FNODE *f_)
{
#define f ((FATFNODE *)f_)
#define fs ((FATFS *)f->fs)
	struct fat_dir_entry *dirblk;
	int i;
	if (__unlikely(!f->dirent_sec)) return -EFSERROR;
	if (__unlikely(f->n_slots >= BIO_SECTOR_SIZE / FAT_DIRENT_SIZE) ||
	    __unlikely(f->dirent_pos >= BIO_SECTOR_SIZE / FAT_DIRENT_SIZE))
		KERNEL$SUICIDE("FAT_REMOVE_FNODE_FROM_DIRECTORY: %s: FNODE POSITION: %u,%u", VFS$FNODE_DESCRIPTION((FNODE *)f), f->n_slots, f->dirent_pos);
	if (__unlikely(f->n_slots > f->dirent_pos)) {
		unsigned long osec = 0;
		unsigned long sec = FAT_CLUSTER_2_SECTOR(fs, ((FATFNODE *)f->parent)->start_cluster);
		unsigned long c[2];
		c[1] = 0;
		while (sec != f->dirent_sec) {
			osec = sec;
			sec = FAT_GET_NEXT_SECTOR_SYNC(fs, sec);
			if (__unlikely(!sec)) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "UNABLE TO FIND ENTRY IN PARENT DIRECTORY");
				goto x;
			}
			if (__unlikely(FAT_STOP_CYCLES(sec, &c))) {
				KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "CYCLE DETECTED ON BLOCK %08lX (REMOVE FROM DIRECTORY)", sec);
				return -EFSERROR;
			}
		}
		if (__unlikely(!osec)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "PARENT DIRECTORY TOO SMALL");
			goto x;
		}
		dirblk = FAT_READ_BUFFER_SYNC(fs, osec, 1);
		if (__unlikely(__IS_ERR(dirblk))) goto x;
		for (i = f->dirent_pos - f->n_slots + BIO_SECTOR_SIZE / FAT_DIRENT_SIZE; i < BIO_SECTOR_SIZE / FAT_DIRENT_SIZE; i++) {
			dirblk[i].name[0] = DELETED_FLAG;
		}
		VFS$PUT_BUFFER(dirblk);
		x:
		f->n_slots = f->dirent_pos;
	}
	dirblk = FAT_READ_BUFFER_SYNC(fs, f->dirent_sec, 1);
	if (__unlikely(__IS_ERR(dirblk))) return __PTR_ERR(dirblk);
	for (i = f->dirent_pos - f->n_slots; i <= f->dirent_pos; i++) {
		dirblk[i].name[0] = DELETED_FLAG;
	}
	VFS$PUT_BUFFER(dirblk);
	f->dirent_sec = 0;
	f->dirent_pos = 0;
	f->n_slots = 0;
	return 0;
#undef f
#undef fs
}


