#include "HPFS.H"

#include <SPAD/ALLOC.H>
#include <SYS/TIME.H>

static __const__ FSOPS hpfsops = {
	DEFAULT_VFSID,
	"HPFS",
	0,
	sizeof(HPFSFS),
	sizeof(HPFSFNODE),
	VFS$FNODE_CTOR,
	sizeof(HPFSPAGEINRQ),
	VFS$PAGEINRQ_CTOR,
	HPFS_PROCESS_OPTION,
	HPFS_MOUNT,
	HPFS_COUNT_BITMAPS,
	HPFS_UMOUNT,
	HPFS_MARK_DIRTY,
	HPFS_VALIDATE_FILENAME,
	HPFS_LOOKUP,
	256,
	HPFS_INIT_READDIR_COOKIE,
	HPFS_READDIR,
	VFS$DEFAULT_WRITEPAGES,
	VFS$DEFAULT_COMMIT,
	HPFS_INIT_FNODE,
	HPFS_CREATE_FNODE,
	HPFS_WRITE_FNODE_DATA,
	HPFS_ADD_FNODE_TO_DIRECTORY,
	HPFS_DELETE_FNODE,
	HPFS_REMOVE_FNODE_FROM_DIRECTORY,
	VFS$NULL_FNODE_CALL,
	HPFS_BMAP,
	HPFS_SYNC_BMAP,
	VFS$INIT_LAST_PAGE,
	HPFS_ACCOUNT,
	HPFS_UNACCOUNT,
	HPFS_STAT,
	HPFS_STATFS,
};

int main(int argc, char *argv[])
{
	return VFS$MOUNT(argc, argv, &hpfsops, "HPFS.SYS");
}

int HPFS_PROCESS_OPTION(FS *fs_, char *opt, char *optend, char *str)
{
	HPFSFS *fs = (HPFSFS *)fs_;
	/*if (!__strcasexcmp("CHECK", opt, optend) && !str) fs->flags |= HPFS_CHECK;
	else*/
	if (!__strcasexcmp("ERRORS", opt, optend) && str) {
		if (!_strcasecmp(str, "RO")) fs->flags &= ~HPFS_ERROR_CONT;
		else if (!_strcasecmp(str, "CONTINUE")) fs->flags |= HPFS_ERROR_CONT;
		else return 1;
	} else return 1;
	return 0;
}

void HPFS_ERROR(HPFSFS *fs)
{
	if (__unlikely(!(fs->flags & (HPFS_ERROR_CONT | FS_RO)))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "REMOUNTING READ-ONLY");
		fs->flags |= FS_RO;
	}
	fs->flags |= HPFS_WAS_ERROR;
}

void HPFS_MARK_DIRTY(FS *fs_, int dirty)
{
	HPFSFS *fs = (HPFSFS *)fs_;
	BUFFER *b;
	struct hpfs_spare_block *spare;
	if (__unlikely(fs->flags & HPFS_WAS_ERROR)) dirty = 1;
	if (__unlikely(__IS_ERR(spare = VFS$READ_BUFFER_SYNC((FS *)fs, SP_SECTOR, 1, 0, &b)))) return;
	if (__unlikely(__32LE2CPU(spare->magic) != SP_MAGIC)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD HPFS SPARE BLOCK SIGNATURE: %08X", (unsigned)__32LE2CPU(spare->magic));
		VFS$PUT_BUFFER(spare);
		return;
	}
	if (!dirty) {
		if (__unlikely(!(spare->flags1 & SP_1_DIRTY))) {
			VFS$PUT_BUFFER(spare);
			return;
		}
		spare->flags1 &= ~SP_1_DIRTY;
	} else {
		if (__unlikely(spare->flags1 & SP_1_DIRTY)) {
			VFS$PUT_BUFFER(spare);
			return;
		}
		spare->flags1 |= SP_1_DIRTY;
	}
	VFS$MARK_BUFFER_DIRTY(b, SP_SECTOR & __SECTORS_PER_PAGE_CLUSTER_MINUS_1, 1);
	VFS$WRITE_BUFFER(b, fs->syncproc);
	VFS$PUT_BUFFER_AND_WAIT(spare);
	VFS$SYNC_DISK((FS *)fs);
	return;
}

int HPFS_MOUNT(FS *fs_)
{
	HPFSFS *fs = (HPFSFS *)fs_;
	HPFSFNODE *f;
	int r;
	struct hpfs_super_block *super;
	struct hpfs_spare_block *spare;
	struct code_page_directory *cpd;
	struct code_page_data *cpa;
	CONTIG_AREA_REQUEST car;
	int i;
	unsigned sec, idx;
	struct timezone tz;
	fs->block_mask = 0;
	fs->pageio_mask = 511;
	fs->pageio_bits = 9;
	fs->max_filesize = 0x80000000U - __PAGE_CLUSTER_SIZE;
	if (fs->max_filesize > 0x80000000U - 2048) fs->max_filesize = 0x80000000U - 2048;
	if (__likely(!gettimeofday(NULL, &tz))) {
		fs->timeshift = tz.tz_minuteswest * 60 - (tz.tz_dsttime ? 3600 : 0);
	} else {
		fs->timeshift = 0;
	}
	if (__unlikely(__IS_ERR(super = VFS$READ_BUFFER_SYNC((FS *)fs, SB_SECTOR, 1, 0, (void *)&KERNEL$LIST_END)))) {
		r = __PTR_ERR(super);
		goto ret0;
	}
	if (__unlikely(__32LE2CPU(super->magic) != SB_MAGIC)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD HPFS SUPERBLOCK SIGNATURE: %08X", (unsigned)__32LE2CPU(super->magic));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: HPFS SUPERBLOCK SIGNATURE NOT FOUND", fs->filesystem_name);
		VFS$PUT_BUFFER(super);
		r = -EINVAL;
		goto ret0;
	}
	if (__unlikely(super->version != 2)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BAD HPFS VERSION", fs->filesystem_name);
		badv:
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD HPFS VERSION: %d.%d", (unsigned)super->version, (unsigned)super->funcversion);
		VFS$PUT_BUFFER(super);
		r = -EINVAL;
		goto ret0;
	}
	if (__likely(!(fs->flags & FS_RO)) && __unlikely(super->funcversion > 3)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: BAD HPFS VERSION (PROBABLY HPFS386), ONLY READ-ONLY MOUNT ALLOWED", fs->filesystem_name);
		goto badv;
	}
	fs->root_fnode = __32LE2CPU(super->root);
	fs->size = __32LE2CPU(super->n_sectors);
	if (__unlikely(!fs->size)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "ZERO FILESYSTEM SIZE");
		goto isup;
	}
	fs->bitmap_directory = __32LE2CPU(super->bitmaps);
	fs->n_dir_band = __32LE2CPU(super->n_dir_band);
	fs->dir_band_start = __32LE2CPU(super->dir_band_start);
	if (__unlikely(fs->dir_band_start + fs->n_dir_band - 1 != __32LE2CPU(super->dir_band_end))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "DIR_BAND_END(%08X) DOES NOT MATCH DIR_BAND_START(%08X)+N_DIR_BAND(%08X)-1", (unsigned)__32LE2CPU(super->dir_band_end), (unsigned)fs->dir_band_start, (unsigned)fs->n_dir_band);
		isup:
		VFS$PUT_BUFFER(super);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID SUPERBLOCK", fs->filesystem_name);
		r = -EFSERROR;
		goto ret0;
	}
	sec = __32LE2CPU(super->dir_band_bitmap);
	if (__unlikely(!sec) || __unlikely(sec & 3) || __unlikely(sec >= fs->size)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID DIRECTORY BITMAP POINTER %08X", sec);
		goto isup;
	}
	fs->dir_band_bitmap = sec;
	VFS$PUT_BUFFER(super);
	if (__unlikely(__IS_ERR(spare = VFS$READ_BUFFER_SYNC((FS *)fs, SP_SECTOR, 1, 0, (void *)&KERNEL$LIST_END)))) {
		r = __PTR_ERR(spare);
		goto ret0;
	}
	if (__unlikely(__32LE2CPU(spare->magic) != SP_MAGIC)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "BAD HPFS SPARE BLOCK SIGNATURE: %08X", (unsigned)__32LE2CPU(spare->magic));
		r = -EFSERROR;
		sp_err:
		VFS$PUT_BUFFER(spare);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SPARE BLOCK ERROR", fs->filesystem_name);
		goto ret0;
	}
	if (__unlikely(spare->flags1 & SP_1_DIRTY)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "FILESYSTEM WAS NOT CLEANLY UNMOUNTED");
		HPFS_ERROR(fs);
	}
	if (__unlikely(spare->flags1 & SP_1_HOTFIX_USED)) {
		if (__likely(!(fs->flags & FS_RO))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "HOTFIXES USED, REFUSING TO MOUNT READ-WRITE, RUN CHKDSK TO FIX IT");
			r = -EINVAL;
			goto sp_err;
		}
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "HOTFIXES USED, YOU MAY READ INCORRECT DATA, RUN CHKDSK TO FIX IT");
		HPFS_ERROR(fs);
	}
	if (__unlikely(spare->flags2 & SP_2_MULTIMEDIA_ACTIVE)) {
		if (__likely(!(fs->flags & FS_RO))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "MULTIMEDIA FORMAT ACTIVE, REFUSING TO MOUNT READ-WRITE, USE MMUTIL TO REMOVE IT");
			r = -EINVAL;
			goto sp_err;
		}
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "MULTIMEDIA FORMAT ACTIVE, YOU MAY READ INCORRECT DATA, USE MMUTIL TO REMOVE IT");
		HPFS_ERROR(fs);
	}
	fs->code_page_dir = __32LE2CPU(spare->code_page_dir);
	fs->n_code_pages = __32LE2CPU(spare->n_code_pages);
	VFS$PUT_BUFFER(spare);
	for (i = 0; i < 128; i++) fs->upcase[i] = i + 128;
	if (__unlikely(!fs->n_code_pages)) goto skip_cp;
	if (__unlikely(__IS_ERR(cpd = VFS$READ_BUFFER_SYNC((FS *)fs, fs->code_page_dir, 1, 0, (void *)&KERNEL$LIST_END)))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_NONFATAL_ERROR, fs->filesystem_name, "CAN'T READ CODE PAGE DIRECTORY AT %08X: %s, ASSUMING NO CODEPAGE", fs->code_page_dir, strerror(-__PTR_ERR(cpd)));
		goto skip_cp;
	}
	if (__unlikely(__32LE2CPU(cpd->magic) != CP_DIR_MAGIC)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_NONFATAL_ERROR, fs->filesystem_name, "BAD MAGIC ON CODE PAGE DIRECTORY AT %08X: %08X, ASSUMING NO CODEPAGE", fs->code_page_dir, (unsigned)__32LE2CPU(cpd->magic));
		VFS$PUT_BUFFER(cpd);
		goto skip_cp;
	}
	if (__unlikely(!cpd->n_code_pages)) {
		VFS$PUT_BUFFER(cpd);
		goto skip_cp;
	}
	sec = cpd->array[0].code_page_data;
	idx = cpd->array[0].index;
	VFS$PUT_BUFFER(cpd);
	if (__unlikely(idx >= 3)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_NONFATAL_ERROR, fs->filesystem_name, "BAD CODEPAGE INDEX: %08X, ASSUMING NO CODEPAGE", idx);
		goto skip_cp;
	}
	if (__unlikely(__IS_ERR(cpa = VFS$READ_BUFFER_SYNC((FS *)fs, sec, 1, 0, (void *)&KERNEL$LIST_END)))) {
		KERNEL$SYSLOG(__SYSLOG_DATA_NONFATAL_ERROR, fs->filesystem_name, "CAN'T READ CODE PAGE DATA AT %08X: %s, ASSUMING NO CODEPAGE", sec, strerror(-__PTR_ERR(cpa)));
		goto skip_cp;
	}
	if (__unlikely(__32LE2CPU(cpa->magic) != CP_DATA_MAGIC)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_NONFATAL_ERROR, fs->filesystem_name, "BAD MAGIC ON CODE PAGE DATA AT %08X: %08X, ASSUMING NO CODEPAGE", sec, (unsigned)__32LE2CPU(cpa->magic));
		VFS$PUT_BUFFER(cpa);
		goto skip_cp;
	}
	memcpy(fs->upcase, cpa->code_page[idx].map, 128);
	VFS$PUT_BUFFER(cpa);
	
	skip_cp:
	fs->n_bitmaps = (fs->size + 16383) >> 14;
	car.nclusters = (sizeof(struct bmp_desc) * fs->n_bitmaps + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS;
	car.flags = CARF_DATA;
	car.align = 0;
	SYNC_IO(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (__unlikely(car.status < 0)) {
		r = car.status;
		goto ret0;
	}
	fs->bmp_desc = car.ptr;
	fs->bmp_desc_clusters = car.nclusters;
	memset(fs->bmp_desc, 0, fs->n_bitmaps * sizeof(struct bmp_desc));
	for (i = 0; i < fs->n_bitmaps; i++) {
		__u32 *bmp_pos;
		unsigned idx = i & 127;
		unsigned sec = fs->bitmap_directory + (i >> 7);
		unsigned bmpsec;
		if (__unlikely(__IS_ERR(bmp_pos = VFS$READ_BUFFER_SYNC((FS *)fs, sec, 1, 0, (void *)&KERNEL$LIST_END)))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "CAN'T READ BITMAP POINTER FOR BITMAP %u AT %08X: %s", i, sec, strerror(-__PTR_ERR(bmp_pos)));
			if (!(fs->flags & FS_RO)) {
				r = __PTR_ERR(bmp_pos);
				goto ret0;
			}
			i = ((i + 128) & ~127) - 1;
			continue;
		}
		bmpsec = __32LE2CPU(bmp_pos[idx]);
		VFS$PUT_BUFFER(bmp_pos);
		if (__unlikely(!bmpsec) || __unlikely(bmpsec & 3) || __unlikely(bmpsec >= fs->size)) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, fs->filesystem_name, "INVALID BITMAP POINTER FOR BITMAP %u: %08X", i, bmpsec);
			HPFS_ERROR(fs);
			if (!(fs->flags & FS_RO)) {
				r = -EFSERROR;
				goto ret0;
			}
			continue;
		}
		fs->bmp_desc[i].sector = bmpsec;
	}
	f = (HPFSFNODE *)VFS$GET_ROOT_FNODE((FS *)fs);
	f->flags = FNODE_DIRECTORY;
	VFS_INIT_DIR((FNODE *)f);
	f->flags1 = 0;
	f->flags2 = 0;
	f->fnode = fs->root_fnode;
	f->ea_size = 0;

	return 0;

	ret0:
	if (__unlikely(fs->bmp_desc_clusters)) KERNEL$VM_RELEASE_CONTIG_AREA(fs->bmp_desc, fs->bmp_desc_clusters);
	return r;
}

void HPFS_COUNT_BITMAPS(FS *fs_)
{
	HPFSFS *fs = (HPFSFS *)fs_;
	int i, j, bit_set;
	__u8 *bmp;
	for (i = 0; i < fs->n_bitmaps; i++) {
		/*__debug_printf("hpfs count(%d of %d)\n", i, fs->n_bitmaps);*/
		if (__unlikely(fs->flags & FS_SHUTTING_DOWN)) {
			fs->bmp_desc[i].max_alloc = 0x3ffc;
			fs->free_space += 0x4000;
			fs->free_dnodes += 0x4000 / 4;
			continue;
		}
		if (__unlikely(!(bmp = HPFS_MAP_BITMAP(fs, i, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END)))) {
			fs->bmp_desc[i].max_alloc = 0;
			continue;
		}
		bit_set = 0;
		for (j = 0; j < 2048; j++) {
			__u8 b = bmp[j];
			if (__likely(b == 0xff)) {
				fs->free_dnodes += 2;
				fs->free_space += 8;
				bit_set = 1;
				continue;
			}
			if (__likely(!b)) continue;
			if ((b & 0x0f) == 0x0f) fs->free_dnodes++;
			if ((b & 0xf0) == 0xf0) fs->free_dnodes++;
			while (b) {
				if (b & 1) fs->free_space++, bit_set = 1;
				b >>= 1;
			}
		}
		if (bit_set) WQ_WAKE_ALL(&fs->sync_done_wait);
		fs->bmp_desc[i].max_alloc = FIND_MAX_RUN(bmp);
		VFS$PUT_BUFFER(bmp);
	}
	if (__unlikely(!(bmp = HPFS_MAP_BITMAP(fs, -1, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END)))) goto skip_dmap;
	for (j = 0; j < 2048; j++) {
		__u8 b = bmp[j];
		while (b) {
			if (b & 1) {
				fs->free_dnodes++;
				fs->dir_band_free++;
			}
			b >>= 1;
		}
	}
	VFS$PUT_BUFFER(bmp);

	skip_dmap:;
}

void HPFS_UMOUNT(FS *fs_, int nosync)
{
	HPFSFS *fs = (HPFSFS *)fs_;
	if (!nosync && (__unlikely(fs->overallocated_space != 0) || __unlikely(fs->overallocated_dnodes != 0) || __unlikely(fs->overallocated_average != 0))) KERNEL$SUICIDE("HPFS_UMOUNT: %s: ACCOUNTING SKEW (%u SECTORS, %u DNODES LEAKED, %u AVERAGE SECTORS)", fs->filesystem_name, fs->overallocated_space, fs->overallocated_dnodes, fs->overallocated_average);
	if (__unlikely(fs->bmp_desc_clusters)) KERNEL$VM_RELEASE_CONTIG_AREA(fs->bmp_desc, fs->bmp_desc_clusters);
}

#define PAGE_ACCOUNT_BLOCKS	(__SECTORS_PER_PAGE_CLUSTER + 5 + __SECTORS_PER_PAGE_CLUSTER / 40)
#define PAGE_ACCOUNT_DNODES	PAGE_ACCOUNT_BLOCKS
#define PAGE_ACCOUNT_AVERAGE	__SECTORS_PER_PAGE_CLUSTER
#define FILE_ACCOUNT_DNODES	12
#define FILE_ACCOUNT_BLOCKS	1
#define FILE_ACCOUNT_AVERAGE	1
#define DIR_ACCOUNT_DNODES	(FILE_ACCOUNT_DNODES + 1)
#define DIR_ACCOUNT_BLOCKS	1
#define DIR_ACCOUNT_AVERAGE	5

void *HPFS_ACCOUNT(FNODE *f, int dir, PAGE *p)
{
	HPFSFS *fs = (HPFSFS *)f->fs;
	if (__likely(dir == ACCOUNT_PAGE)) {
		if (__unlikely(fs->overallocated_dnodes + PAGE_ACCOUNT_DNODES > fs->free_dnodes)) goto ovr;
		if (__unlikely(fs->overallocated_space + PAGE_ACCOUNT_BLOCKS > fs->free_space)) goto ovr;
		fs->overallocated_dnodes += PAGE_ACCOUNT_DNODES;
		fs->overallocated_space += PAGE_ACCOUNT_BLOCKS;
		fs->overallocated_average += PAGE_ACCOUNT_AVERAGE;
		return NULL;
	} else if (__likely(dir == ACCOUNT_FILE)) {
		if (__unlikely(fs->overallocated_dnodes + FILE_ACCOUNT_DNODES + FILE_ACCOUNT_BLOCKS > fs->free_dnodes)) goto ovr;
		if (__unlikely(fs->overallocated_space + FILE_ACCOUNT_DNODES * 4 + FILE_ACCOUNT_BLOCKS > fs->free_space)) goto ovr;
		fs->overallocated_dnodes += FILE_ACCOUNT_DNODES + FILE_ACCOUNT_BLOCKS;
		fs->overallocated_space += FILE_ACCOUNT_DNODES * 4 + FILE_ACCOUNT_BLOCKS;
		fs->overallocated_average += FILE_ACCOUNT_AVERAGE;
		return NULL;
	} else {
		if (__unlikely(fs->overallocated_dnodes + DIR_ACCOUNT_DNODES + DIR_ACCOUNT_BLOCKS > fs->free_dnodes)) goto ovr;
		if (__unlikely(fs->overallocated_space + DIR_ACCOUNT_DNODES * 4 + DIR_ACCOUNT_BLOCKS > fs->free_space)) goto ovr;
		fs->overallocated_dnodes += DIR_ACCOUNT_DNODES + DIR_ACCOUNT_BLOCKS;
		fs->overallocated_space += DIR_ACCOUNT_DNODES * 4 + DIR_ACCOUNT_BLOCKS;
		fs->overallocated_average += DIR_ACCOUNT_AVERAGE;
		return NULL;
	}
	ovr:
	return VFS$OVERACCOUNT(fs->root);
}

void HPFS_UNACCOUNT(FNODE *f, int dir, PAGE *p)
{
	HPFSFS *fs = (HPFSFS *)f->fs;
	if (__likely(dir == ACCOUNT_PAGE)) {
		fs->overallocated_dnodes -= PAGE_ACCOUNT_DNODES;
		fs->overallocated_space -= PAGE_ACCOUNT_BLOCKS;
		fs->overallocated_average -= PAGE_ACCOUNT_AVERAGE;
	} else if (__likely(dir == ACCOUNT_FILE)) {
		fs->overallocated_dnodes -= FILE_ACCOUNT_DNODES + FILE_ACCOUNT_BLOCKS;
		fs->overallocated_space -= FILE_ACCOUNT_DNODES * 4 + FILE_ACCOUNT_BLOCKS;
		fs->overallocated_average -= FILE_ACCOUNT_AVERAGE;
	} else {
		fs->overallocated_dnodes -= DIR_ACCOUNT_DNODES + DIR_ACCOUNT_BLOCKS;
		fs->overallocated_space -= DIR_ACCOUNT_DNODES * 4 + DIR_ACCOUNT_BLOCKS;
		fs->overallocated_average -= DIR_ACCOUNT_AVERAGE;
	}
}

void HPFS_STAT(FNODE *f_, struct stat *stat)
{
	if (__likely(!S_ISDIR(stat->st_mode))) stat->st_blocks++;
	else stat->st_blocks = 5;
}

WQ *HPFS_STATFS(FS *fs_, struct statfs *stat)
{
	HPFSFS *fs = (HPFSFS *)fs_;
	if (__unlikely(fs->flags & FS_BACKGROUND_INIT)) return &fs->sync_done_wait;
	stat->f_blocks = fs->size;
	stat->f_bfree = __likely(fs->free_space > fs->overallocated_average) ? (__u64)(fs->free_space - fs->overallocated_average) : 0;
	stat->f_bavail = stat->f_bfree;
	stat->f_files = fs->n_dir_band >> 2;
	stat->f_ffree = fs->dir_band_free;
	stat->f_favail = stat->f_ffree;
	stat->f_namelen = 255;
	return NULL;
}

int HPFS_STOP_CYCLES(unsigned key, unsigned (*c)[2])
{
	if (__likely((*c)[1]) && __unlikely((*c)[0] == key)) return -1;
	(*c)[1]++;
	if (!(((*c)[1] - 1) & (*c)[1])) (*c)[0] = key;
	return 0;
}

