#include <VALUES.H>
#include <SPAD/SYNC.H>
#include <SPAD/SLAB.H>
#include <SPAD/LIBC.H>
#include <SPAD/HASH.H>
#include <SPAD/WQ.H>
#include <SPAD/DEV.H>
#include <SPAD/DL.H>
#include <ARCH/SETUP.H>

#include "VFS.H"

#define RESERVE_FNODES		(MAX_DEPTH * 2)
#define RESERVE_NAMES		(MAX_DEPTH * 2)
#define RESERVE_PAGEINRQ	2
#define RESERVE_SPAGES		1
#define RESERVE_SMALL_HASH	(MAX_DEPTH * 2)
#define RESERVE_BUFFERS		RESERVE_BZ
#define RESERVE_BRQ		(__unlikely(mem <= 16777216) ? 0 : 1)
#define RESERVE_WPAGES		2
#define RESERVE_Z		(__unlikely(mem < 12582912) ? 2 : __unlikely(mem <= 33554432) ? 4 : 16)
#define RESERVE_BZ		(__unlikely(mem < 12582912) ? 2 : __unlikely(mem <= 33554432) ? 4 : 16)

AST_STUB SYNCER_END;
static void VFS_INIT_ROOT(HANDLE *h, void *fs_);
static void REAP_ALL_FS_DATA(FS *fs);
static int VFS_UMOUNT(void *p, void **release, char *argv[]);

char *VFS$FNODE_DESCRIPTION(FNODE *f)
{
#if __DEBUG_INSECURE >= 1
	static char fnode_description[__MAX_STR_LEN * 3];
	if (f->parent) _snprintf(fnode_description, sizeof fnode_description, "%s:%s/%s", f->fs->filesystem_name, f->parent->name, f->name);
	else _snprintf(fnode_description, sizeof fnode_description, "%s:%s", f->fs->filesystem_name, f->name);
	return fnode_description;
#else
	return f->fs->filesystem_name;
#endif
}

char *VFS$HIDE_NAME(char *name)
{
#if __DEBUG_INSECURE >= 1
	return name;
#else
	return "-";
#endif
}

static void small_hash_ctor(void *g, void *o)
{
	XLIST_HEAD *h = o;
	int i;
	/* constructor can't construct first pointer of an object */
	for (i = 1; i < SMALL_HASH_SIZE; i++)
		INIT_XLIST(&h[i]);
}

int VFS$MOUNT(int argc, char *argv[], __const__ FSOPS *f, char *driver_name)
{
	int r;
	FS *fs;
	int i;
	MALLOC_REQUEST mrq;
	OPENRQ openrq;
	CHHRQ ch;
	DEVICE_REQUEST devrq;
	IOCTLRQ ioctlrq;
	char **arg;
	int state;
	struct __param_table params[] = {
		"RO", __PARAM_BOOL, FS_RO, FS_RO, NULL,
		"SYNC_TIME", __PARAM_INT, 0, MAXINT / JIFFIES_PER_SECOND, NULL,
		"BUFFER_FLUSH_TIME", __PARAM_INT, 0, MAXINT / JIFFIES_PER_SECOND, NULL,
		"", __PARAM_STRING, 1, __MAX_STR_LEN - 1, NULL,
		"", __PARAM_STRING, 1, __MAX_STR_LEN - 1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char *opt, *optend, *val;
	char *dev, *name;
	void *p;
	PAGEZONE *z;
	PAGE *up;
	__u64 mem;
	int syncsecs = -2;
	static __const__ VFSID vfsid = DEFAULT_VFSID;

	if (__unlikely(memcmp(&vfsid, &f->vfsid, sizeof(VFSID)))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: VFS.DLL VERSION MISMATCH", f->name);
		r = -EPROTO;
		goto err0;
	}

	mem = KERNEL$GET_MEMORY_SIZE(VM_TYPE_WIRED_MAPPED);
	mrq.size = f->sizeof_FS + (__PAGES_PER_CLUSTER - 1) * sizeof(XLIST_HEAD) + __SECTORS_PER_PAGE_CLUSTER;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (mrq.status < 0) {
		if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: OUT OF MEMORY", f->name);
		r = mrq.status;
		goto err0;
	}
	RAISE_SPL(SPL_FS);
	fs = mrq.ptr;
	memset(fs, 0, f->sizeof_FS);
	fs->slist = (XLIST_HEAD *)((char *)fs + f->sizeof_FS);
	fs->tmp_dirtmap = (char *)fs + f->sizeof_FS + (__PAGES_PER_CLUSTER - 1) * sizeof(XLIST_HEAD);

	fs->buffer_flush_time = VFS_DEFAULT_BUFFER_FLUSHTIME;

	params[0].__p = &fs->flags;
	params[1].__p = &syncsecs;
	params[2].__p = &fs->buffer_flush_time;
	params[3].__p = &dev;
	params[4].__p = &name;

	if (f->flags & FS_RO_ONLY) fs->flags |= FS_RO;
	
	dev = name = NULL;

	arg = argv;
	state = 0;
	while (__parse_params(&arg, &state, params, &opt, &optend, &val)) {
		if (f->process_option) {
			if ((r = f->process_option(fs, opt, optend, val))) {
				if (r > 0) bads: r = -EBADSYN, _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SYNTAX ERROR", f->name);
				goto err1;
			}
		} else goto bads;
	}
	if (!dev || !name) goto bads;

	fs->buffer_flush_time *= JIFFIES_PER_SECOND;

	strcpy(fs->device_name, dev);
	/* it is guaranteed that it is at least 1 and at most __MAX_STR_LEN - 2 characters long */
	if (stpcpy(fs->filesystem_name, name)[-1] != ':') strcat(fs->filesystem_name, ":");

	__upcase(fs->filesystem_name);
	__upcase(fs->device_name);
#ifdef VFS_INO
	fs->st_dev = 0;
	dev = fs->device_name;
	quickcasehash(dev, *dev, fs->st_dev);
#endif

	openrq.flags = (fs->flags & FS_RO ? O_RDONLY : O_RDWR) | O_DIRECT;
	openrq.path = fs->device_name;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR OPENING DEVICE %s: %s", f->name, fs->device_name, strerror(-openrq.status));
		r = openrq.status;
		goto err1;
	}
	fs->disk_handle_num = openrq.status;
	dev = KERNEL$HANDLE_PATH(fs->disk_handle_num);
	if (dev && strlen(dev) < sizeof(fs->device_name)) strcpy(fs->device_name, dev);
	ch.h = fs->disk_handle_num;
	ch.option = "DIRECT";
	ch.value = "";
	SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);

	/***** init any static non-allocated items of FS here !! *****/
	INIT_LIST(&fs->synclist);
	WQ_INIT(&fs->syncer_wait, "VFS$SYNCER_WAIT");
	WQ_INIT(&fs->sync_done_wait, "VFS$SYNC_DONE_WAIT");
	WQ_INIT(&fs->sync_buf_done, "VFS$SYNC_BUF_DONE");
	fs->wantfree.status = RQS_PROCESSING;
	INIT_XLIST(&fs->wantfree_pages);
	INIT_XLIST(&fs->wantfree_fnodes);
	INIT_TIMER(&fs->sync_timer);
	VOID_LIST_ENTRY(&fs->sync_timer.list);
	fs->size = -1;
	fs->fsops = f;
	fs->readdir_cookie_size = f->readdir_cookie_size > INTERNAL_READDIR_COOKIE_SIZE ? f->readdir_cookie_size : INTERNAL_READDIR_COOKIE_SIZE;

	VFS$ZINIT(&fs->z, VM_TYPE_CACHED_UNMAPPED, "PAGE");
	VFS$ZINIT(&fs->bz, VM_TYPE_CACHED_UNMAPPED, "BUFFER");
	KERNEL$SLAB_INIT(&fs->fnodes, f->sizeof_FNODE, 0, VM_TYPE_CACHED_MAPPED, f->ctor_FNODE, fs, NULL, "VFS$FNODE");
	KERNEL$SLAB_INIT(&fs->names, MAX_NAMELEN, 0, VM_TYPE_CACHED_MAPPED, NULL, NULL, NULL, "VFS$NAME");
	KERNEL$SLAB_INIT(&fs->pageinrq, f->sizeof_PAGEINRQ, 0, VM_TYPE_CACHED_MAPPED, f->ctor_PAGEINRQ, fs, NULL, "VFS$PAGEINRQ");
	KERNEL$SLAB_INIT(&fs->spages, sizeof(SPAGES) + (sizeof(SPAGE_ENTRY) << __PAGES_PER_CLUSTER_BITS), 0, VM_TYPE_CACHED_MAPPED, VFS_SPAGE_CTOR, fs, NULL, "VFS$SPAGES");
	KERNEL$SLAB_INIT(&fs->small_hash, sizeof(XLIST_HEAD) * SMALL_HASH_SIZE, 0, VM_TYPE_CACHED_MAPPED, small_hash_ctor, fs, NULL, "VFS$SMALL_HASH");
	KERNEL$SLAB_INIT(&fs->buffers, sizeof(BUFFER) + ((3 * BUFFER_BITMAP_SIZE / 8 + sizeof(long) - 1) & ~(sizeof(long) - 1)), 0, VM_TYPE_CACHED_MAPPED, BUFFER_CTOR, fs, NULL, "VFS$BUFFER");
	KERNEL$SLAB_INIT(&fs->brq, sizeof(BRQ), 0, VM_TYPE_CACHED_MAPPED, BRQ_CTOR, fs, NULL, "VFS$BRQ");
	KERNEL$SLAB_INIT(&fs->wpages, sizeof(WPAGE), 0, VM_TYPE_CACHED_MAPPED, WPAGE_CTOR, fs, NULL, "VFS$WPAGE");
	
	WQ_INIT(&fs->freemem, "VFS$FREEMEM");
	WQ_INIT(&fs->pageinrq_wait, "VFS$PAGEINRQ_WAIT");
	WQ_INIT(&fs->freebuffer, "VFS$FREEBUFFER");

	INIT_LIST(&fs->clean_buffers);
	INIT_LIST(&fs->dirty_buffers);

	INIT_XLIST(&fs->wpages_prep);
	INIT_XLIST(&fs->wpages_io);
	WQ_INIT(&fs->wpage_wait, "VFS$WPAGE_WAIT");

	for (i = 0; i < __PAGES_PER_CLUSTER - 1; i++) INIT_XLIST(&fs->slist[i]);

	fs->syncproc = &KERNEL$PROC_KERNEL;

	ioctlrq.h = fs->disk_handle_num;
	ioctlrq.ioctl = IOCTL_BIO_GET_OPTIMAL_REQUEST_SIZE;
	ioctlrq.param = PARAM_BIO_GET_OPTIMAL_REQUEST_SIZE_READ;
	ioctlrq.v.ptr = 0;
	ioctlrq.v.len = 0;
	ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_CANCELABLE(&ioctlrq, KERNEL$IOCTL);
	if (ioctlrq.status >= 0) {
		fs->bio_request_size = ioctlrq.status;
		fs->buffer_readahead = (ioctlrq.status << BIO_SECTOR_SIZE_BITS) / __PAGE_CLUSTER_SIZE - 1;
		fs->page_readahead = ioctlrq.status;
	} else {
		fs->bio_request_size = -ENOOP;
		fs->buffer_readahead = 0;
		fs->page_readahead = __SECTORS_PER_PAGE_CLUSTER;
	}
	if (__unlikely(fs->buffer_readahead < 0)) fs->buffer_readahead = 0;
	if (__unlikely(fs->buffer_readahead >= VFS_BUFFER_READAHEAD / __PAGE_CLUSTER_SIZE)) fs->buffer_readahead = VFS_BUFFER_READAHEAD / __PAGE_CLUSTER_SIZE - 1;
	if (__unlikely(fs->page_readahead < 1)) fs->page_readahead = 1;
	if (__unlikely(fs->page_readahead > VFS_PAGE_READAHEAD >> BIO_SECTOR_SIZE_BITS)) fs->page_readahead = VFS_PAGE_READAHEAD >> BIO_SECTOR_SIZE_BITS;

	if (__unlikely(r = KERNEL$SLAB_RESERVE(&fs->fnodes, RESERVE_FNODES))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->names, RESERVE_NAMES))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->pageinrq, RESERVE_PAGEINRQ))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->spages, RESERVE_SPAGES))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->small_hash, RESERVE_SMALL_HASH))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->buffers, RESERVE_BUFFERS))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->brq, RESERVE_BRQ))
	 || __unlikely(r = KERNEL$SLAB_RESERVE(&fs->wpages, RESERVE_WPAGES))
	) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: OUT OF MEMORY", f->name);
		goto err2;
	}

	next_page:
	z = NULL;
	if (fs->z.reserved < RESERVE_Z) z = &fs->z;
	else if (fs->bz.reserved < RESERVE_BZ) z = &fs->bz;
	if (z) {
		up = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED);
		if (up) {
			VFS$ZRESERVE(z, up);
			goto next_page;
		}
		memwait:
		if (__unlikely(r = KERNEL$MEMWAIT_SYNC(__PAGE_CLUSTER_SIZE))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: OUT OF MEMORY", f->name);
			goto err2;
		}
		goto next_page;
	}
	if (!fs->bhash) {
		p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
		if (!p) goto memwait;
		fs->bhash = p;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) INIT_XLIST(&fs->bhash[i]);
	}
	if (!fs->ihash) {
		p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
		if (!p) goto memwait;
		fs->ihash = p;
		for (i = 0; i < IHASH_SIZE; i++) INIT_XLIST(&fs->ihash[i]);
	}
	if (!fs->ehash) {
		p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
		if (!p) goto memwait;
		fs->ehash = p;
		for (i = 0; i < IHASH_SIZE; i++) INIT_XLIST(&fs->ehash[i]);
	}
	if (RESERVE_PAGEINRQ > 1) {
		fs->dummy_pageinrq = __slalloc(&fs->pageinrq);
		if (__unlikely(!fs->dummy_pageinrq))
			KERNEL$SUICIDE("VFS$MOUNT: %s: CAN'T ALLOCATE DUMMY PAGEINRQ", fs->filesystem_name);
	}
	if (RESERVE_BRQ > 0) {
		fs->dummy_brq = __slalloc(&fs->brq);
		if (__unlikely(!fs->dummy_brq))
			KERNEL$SUICIDE("VFS$MOUNT: %s: CAN'T ALLOCATE DUMMY BRQ", fs->filesystem_name);
	}
	if (RESERVE_WPAGES > 1) {
		fs->dummy_wpage = __slalloc(&fs->wpages);
		if (__unlikely(!fs->dummy_wpage))
			KERNEL$SUICIDE("VFS$MOUNT: %s: CAN'T ALLOCATE DUMMY WPAGE", fs->filesystem_name);
	}
	r = f->mount(fs);
	if (r < 0) {
		if (!*KERNEL$ERROR_MSG()) if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR MOUNTING FILESYSTEM: %s", fs->filesystem_name, strerror(-r));
		goto err2;
	}
	if (!fs->max_filesize || (fs->max_filesize & __PAGE_CLUSTER_SIZE_MINUS_1))
		KERNEL$SUICIDE("VFS$MOUNT: %s: MAX_FILESIZE %016"__64_format"X RETURNED BY FSD", fs->filesystem_name, (__u64)fs->max_filesize);
	if (fs->pageio_mask != ((1 << fs->pageio_bits) - 1) || __unlikely(fs->pageio_bits < BIO_SECTOR_SIZE_BITS))
		KERNEL$SUICIDE("VFS$MOUNT: %s: PAGEIO_MASK %d, PAGEIO_BITS %d", fs->filesystem_name, fs->pageio_mask, fs->pageio_bits);
	fs->mounted = 1;

	fs->syncer.fn = SYNCER_END;
	fs->syncer.thread_main = SYNCER;
	fs->syncer.p = fs;
	fs->syncer.error = NULL;
	fs->syncer.cwd = NULL;
	fs->syncer.std_in = -1;
	fs->syncer.std_out = -1;
	fs->syncer.std_err = -1;
	fs->syncer.dlrq = NULL;
	fs->syncer.thread = NULL;
	fs->syncer.spawned = 0;
	CALL_IORQ(&fs->syncer, KERNEL$THREAD);
	WQ_WAIT_SYNC(&fs->syncer_wait);
	if (!fs->syncer.spawned) {
		r = fs->syncer.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR SPAWNING SYNCER THREAD: %s", fs->filesystem_name, strerror(-r));
		goto err3;
	}
	WQ_WAKE_ALL_PL(&fs->syncer_wait);
	fs->mounted = 2;

	devrq.name = fs->filesystem_name;
	devrq.driver_name = driver_name;
	devrq.flags = 0;
	devrq.init_root_handle = VFS_INIT_ROOT;
	devrq.dev_ptr = fs;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = VFS_UMOUNT;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		r = devrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: ERROR REGISTERING DEVICE", fs->filesystem_name);
		goto err4;
	}
	if (syncsecs == -2) syncsecs = VFS_DEFAULT_SYNCTIME;
	if (syncsecs > 0) {
		fs->sync_time = syncsecs * JIFFIES_PER_SECOND / 2;
		fs->sync_timer.fn = SYNC_TIMER;
		KERNEL$DEL_TIMER(&fs->sync_timer);
		KERNEL$SET_TIMER(fs->sync_time, &fs->sync_timer);
	}
	fs->lnte = devrq.lnte;
	strlcpy(KERNEL$ERROR_MSG(), fs->filesystem_name, __MAX_STR_LEN);
	fs->dlrq = KERNEL$TSR_IMAGE();
	return 0;

	err4:
	err3:
	err2:
	REAP_ALL_FS_DATA(fs);
	err1:
	KERNEL$UNIVERSAL_FREE(fs);
	err0:
	return r;
}

DECL_AST(SYNCER_END, SPL_FS, THREAD_RQ)
{
	FS *fs = RQ->p;
	if (!RQ->spawned) {
		WQ_WAKE_ALL_PL(&fs->syncer_wait);
		RETURN;
	}
	if (!fs->terminate_syncer)
		KERNEL$SUICIDE("SYNCER_END: %s: UNEXPECTED SYNCER RETURN", fs->filesystem_name);
	fs->terminate_syncer = 2;
	WQ_WAKE_ALL_PL(&fs->syncer_wait);
	RETURN;
}

static void VFS_INIT_ROOT(HANDLE *h, void *fs_)
{
	FS *fs = fs_;
	h->fnode = fs->root;
	if (__likely(fs->root->flags & FNODE_DIRECTORY))
		h->op = &VFS_DIR_OPERATIONS;
	else
		h->op = &VFS_FILE_OPERATIONS;
	h->flags = 0;
	h->flags2 = CAP_ALL;
}

static void REAP_ALL_FS_DATA(FS *fs)
{
	CLOSERQ closerq;
	int i;
	KERNEL$DEL_TIMER(&fs->sync_timer);
	if (fs->mounted == 2) {
		fs->terminate_syncer = 1;
		while (fs->terminate_syncer == 1) {
			WQ_WAKE_ALL_PL(&fs->syncer_wait);
			KERNEL$SLEEP(1);
		}
	}
	if (fs->mounted) {
		VFS$WRITE_BUFFERS(fs, 1);
		VFS$WAIT_FOR_BUFFERS(fs);
		VFS$SYNC_DISK(fs);
		if (!(fs->flags & FS_RO)) {
			fs->flags &= ~FS_DIRTY;
			fs->fsops->set_dirty(fs, 0);
		}
		fs->fsops->umount(fs, 0);
		VFS$WRITE_BUFFERS(fs, 1);
		VFS$WAIT_FOR_BUFFERS(fs);
		VFS$SYNC_DISK(fs);
	}
	if (fs->root) {
		if (fs->root->flags & (FNODE_FILE | FNODE_DIRECTORY)) VFS_FREE_FNODE_DATA(fs->root);
		VFS_FREE_EMPTY_FNODE(fs->root);
	}
	CLOSE_BUFFERS(fs);
	closerq.h = fs->disk_handle_num;
	SYNC_IO(&closerq, KERNEL$CLOSE);
	if (fs->dummy_pageinrq) __slow_slfree(fs->dummy_pageinrq);
	if (fs->dummy_brq) __slow_slfree(fs->dummy_brq);
	if (fs->dummy_wpage) __slow_slfree(fs->dummy_wpage);
	KERNEL$SLAB_DESTROY(&fs->fnodes);
	KERNEL$SLAB_DESTROY(&fs->names);
	KERNEL$SLAB_DESTROY(&fs->pageinrq);
	KERNEL$SLAB_DESTROY(&fs->spages);
	KERNEL$SLAB_DESTROY(&fs->small_hash);
	KERNEL$SLAB_DESTROY(&fs->buffers);
	KERNEL$SLAB_DESTROY(&fs->brq);
	KERNEL$SLAB_DESTROY(&fs->wpages);
	VFS$ZDONE(&fs->z);
	VFS$ZDONE(&fs->bz);
	if (fs->bhash) {
		int i;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) if (!XLIST_EMPTY(&fs->bhash[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: NOT EMPTY BHASH[%d]", fs->filesystem_name, i);
		KERNEL$FREE_KERNEL_PAGE(fs->bhash, VM_TYPE_WIRED_MAPPED);
	}
	if (fs->ihash) {
		int i;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) if (!XLIST_EMPTY(&fs->ihash[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: NOT EMPTY IHASH[%d]", fs->filesystem_name, i);
		KERNEL$FREE_KERNEL_PAGE(fs->ihash, VM_TYPE_WIRED_MAPPED);
	}
	if (fs->ehash) {
		int i;
		for (i = 0; i < BUFFER_HASH_SIZE; i++) if (!XLIST_EMPTY(&fs->ehash[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: NOT EMPTY EHASH[%d]", fs->filesystem_name, i);
		KERNEL$FREE_KERNEL_PAGE(fs->ehash, VM_TYPE_WIRED_MAPPED);
	}
	if (!LIST_EMPTY(&fs->clean_buffers)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: CLEAN BUFFER XLIST NOT EMPTY", fs->filesystem_name);
	if (!LIST_EMPTY(&fs->dirty_buffers)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: DIRTY BUFFER XLIST NOT EMPTY", fs->filesystem_name);
	if (!XLIST_EMPTY(&fs->wpages_prep)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WPAGES_PREP LIST NOT EMPTY", fs->filesystem_name);
	if (!XLIST_EMPTY(&fs->wpages_io)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WPAGES_IO LIST NOT EMPTY", fs->filesystem_name);
	if (!LIST_EMPTY(&fs->synclist)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: SYNCLIST NOT EMPTY", fs->filesystem_name);
	if (!XLIST_EMPTY(&fs->wantfree_pages)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WANTFREE_PAGES LIST NOT EMPTY", fs->filesystem_name);
	if (!XLIST_EMPTY(&fs->wantfree_fnodes)) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: WANTFREE_FNODES LIST NOT EMPTY", fs->filesystem_name);
	if (fs->n_buffers != 0) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: BUFFER COUNT LEAKED: %ld", fs->filesystem_name, fs->n_buffers);
	if (fs->n_bighash != 0) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: BIGHASH LEAKED: %ld", fs->filesystem_name, fs->n_bighash);
	for (i = 0; i < __PAGES_PER_CLUSTER - 1; i++) if (!XLIST_EMPTY(&fs->slist[i])) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: SPAGE ENTRY LEAKED AT POSITION %d", fs->filesystem_name, i);
	if (fs->total_dirty) KERNEL$SUICIDE("REAP_ALL_FS_DATA: %s: TOTAL DIRTY COUNTER LEAKED: %lu", fs->filesystem_name, fs->total_dirty);
	WQ_WAKE_ALL(&fs->freemem);
	WQ_WAKE_ALL(&fs->pageinrq_wait);
	WQ_WAKE_ALL(&fs->freebuffer);
	WQ_WAKE_ALL(&fs->wpage_wait);
}

static void *DFS_WRITE_FNODE(FNODE *f, void *unused)
{
	WRITE_FNODE(f);
	return NULL;
}

static int VFS_UMOUNT(void *p, void **release, char *argv[])
{
	FNODE barrier;
	WQ *wq;
	FS *fs = p;
	int r;
	int state = 0;
	int flags = 0;
	struct __param_table params[] = {
		"BLOCK", __PARAM_BOOL, DELETE_LOGICAL_BLOCK, DELETE_LOGICAL_BLOCK, NULL,
		"UNUSED", __PARAM_BOOL, DELETE_LOGICAL_UNUSED, DELETE_LOGICAL_UNUSED, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **arg = argv;
	params[0].__p = &flags, params[1].__p = &flags;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: SYNTAX ERROR", fs->filesystem_name);
		return -EBADSYN;
	}
	RAISE_SPL(SPL_FS);
	if (__likely(!(fs->flags & FS_BACKGROUND_INIT))) {
/* this is not required, just minimize time spent in unmounted state */
		VFS$DO_FOR_SUBTREE(fs->root, NULL, DFS_PREORDER, DFS_WRITE_FNODE);
		SYNC(fs, &barrier, &KERNEL$PROC_KERNEL);
		if ((r = WQ_WAIT_SYNC_CANCELABLE(&barrier.wait))) {
			CANCEL_SYNC(fs, &barrier);
			LOWER_SPL(SPL_ZERO);
			return r;
		}
	}
	LOWER_SPL(SPL_ZERO);
	if ((r = KERNEL$UNREGISTER_DEVICE(fs->lnte, flags))) return r;
	RAISE_SPL(SPL_FS);
	fs->flags |= FS_SHUTTING_DOWN;

	RAISE_SPL(SPL_VSPACE);
	if (fs->dummy_pageinrq) __slow_slfree(fs->dummy_pageinrq), fs->dummy_pageinrq = NULL;
	while (!KERNEL$SLAB_EMPTY(&fs->pageinrq)) {
		WQ_WAIT_SYNC(&fs->pageinrq_wait);
	}
	LOWER_SPL(SPL_FS);

	while ((wq = VFS$FREE_FNODE(fs->root))) {
		WQ_WAIT_SYNC(wq);
		LOWER_SPL(SPL_FS);
	}

	SYNC(fs, &barrier, &KERNEL$PROC_KERNEL);
	WQ_WAIT_SYNC(&barrier.wait);
	REAP_ALL_FS_DATA(fs);
	*release = fs->dlrq;
	KERNEL$UNIVERSAL_FREE(fs);
	return 0;
}

DECL_IOCALL(DLL$LOAD, SPL_FS, DLINITRQ)
{
	if (BUFFER_INIT_GLOBAL()) {
		RQ->status = -ENFILE;
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER BUFFER VM ENTITY");
		ret1:
		RETURN_AST(RQ);
	};
	if (FNODE_INIT_GLOBAL()) {
		RQ->status = -ENFILE;
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER FNODE VM ENTITY");
		ret2:
		BUFFER_TERM_GLOBAL();
		goto ret1;
	};
	if (PAGE_INIT_GLOBAL()) {
		RQ->status = -ENFILE;
		_snprintf(RQ->error, __MAX_STR_LEN, "CAN'T REGISTER PAGE VM ENTITY");
		FNODE_TERM_GLOBAL();
		goto ret2;
	};
	RQ->status = 0;
	RETURN_AST(RQ);
}

DECL_IOCALL(DLL$UNLOAD, SPL_FS, DLINITRQ)
{
	PAGE_TERM_GLOBAL();
	FNODE_TERM_GLOBAL();
	BUFFER_TERM_GLOBAL();
	RQ->status = 0;
	RETURN_AST(RQ);
}

