#include "VFS.H"

#include <SPAD/SYSLOG.H>

extern AST_STUB PAGES_WROTE;

void WPAGE_CTOR(void *fs_, void *wpage_)
{
	WPAGE *wpage = wpage_;
	FS *fs = fs_;
	wpage->rq.h = fs->disk_handle_num;
	wpage->desc.v.vspace = &KERNEL$PHYSICAL;
	wpage->fs = fs;
}

static __finline__ void join_wpages(FS *fs, WPAGE *w1, WPAGE *w2)
{
	w1->last->next = w2;
	w1->last->desc.next = &w2->desc;
	w1->last = w2->last;
	w1->rq.nsec += w2->rq.nsec;
	DEL_FROM_LIST(&w1->ehash_entry);
	ADD_TO_XLIST(&fs->ehash[((unsigned)w1->rq.sec + w1->rq.nsec) & IHASH_MASK], &w1->ehash_entry);
}

int VFS_WRITEPAGE(FS *fs, PAGE *p, unsigned sec, unsigned nsec, __d_off blk)
{
	WPAGE *wpage, *w;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_FS)))
		KERNEL$SUICIDE("VFS_WRITEPAGE AT SPL %08X", KERNEL$SPL);
#endif
	if (__unlikely(fs->write_error)) {
		WQ_WAKE_ALL(&p->wait);
		return 0;
	}
	if (__unlikely(!(wpage = __slalloc(&fs->wpages)))) {
		dowr:
		VFS_DO_WRITE(fs);
		return 1;
	}
	wpage->desc.v.ptr = KERNEL$PAGE_2_PHYS(p) | (sec << BIO_SECTOR_SIZE_BITS);
	wpage->desc.v.len = nsec << BIO_SECTOR_SIZE_BITS;
	wpage->desc.next = NULL;
	wpage->p = p;
	wpage->last = wpage;
	wpage->next = NULL;
	wpage->rq.nsec = nsec;
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely((p->flags & PAGE_WRITECOUNT) == PAGE_WRITECOUNT)) {
		LOWER_SPL(SPL_FS);
		__slow_slfree(wpage);
		goto dowr;
	}
	p->flags += PAGE_WRITECOUNT_1;
	LOWER_SPL(SPL_FS);
	((FNODE *)p->fnode)->pending_writes++;
	wpage->rq.proc = fs->syncproc;
	KERNEL$ACQUIRE_WRITEBACK_TAG(wpage->rq.proc);
	wpage->rq.fault_sec = -1;
	XLIST_FOR_EACH(w, &fs->ehash[(unsigned)blk & IHASH_MASK], WPAGE, ehash_entry)
		if (__likely(w->rq.sec + (unsigned)w->rq.nsec == blk)) {
			if (__unlikely(nsec + w->rq.nsec > VFS_MAX_PAGE_WRITE_CLUSTER >> BIO_SECTOR_SIZE_BITS)) break;
			join_wpages(fs, w, wpage);
			wpage = w;
			goto joint;
		}
	wpage->rq.fn = PAGES_WROTE;
	wpage->rq.sec = blk;
	wpage->rq.flags = BIO_WRITE;
	wpage->rq.desc = &wpage->desc;
	ADD_TO_XLIST(&fs->ihash[(unsigned)blk & IHASH_MASK], &wpage->ihash_entry);
	ADD_TO_XLIST(&fs->ehash[((unsigned)blk + nsec) & IHASH_MASK], &wpage->ehash_entry);
	ADD_TO_XLIST(&fs->wpages_prep, &wpage->list_entry);
	fs->n_wpages_prep++;
	joint:
	XLIST_FOR_EACH_UNLIKELY(w, &fs->ihash[((unsigned)blk + nsec) & IHASH_MASK], WPAGE, ihash_entry)
		if (__likely(w->rq.sec == blk + nsec)) {
			if (__unlikely(nsec + w->rq.nsec > VFS_MAX_PAGE_WRITE_CLUSTER >> BIO_SECTOR_SIZE_BITS)) break;
			DEL_FROM_LIST(&w->ihash_entry);
			DEL_FROM_LIST(&w->ehash_entry);
			DEL_FROM_LIST(&w->list_entry);
			fs->n_wpages_prep--;
			join_wpages(fs, wpage, w);
			break;
		}
	return 0;
}

void VFS_DO_WRITE(FS *fs)
{
	WPAGE *w, *ww;
	if (!XLIST_EMPTY(&fs->wpages_io)) return;
	MOVE_XLIST_HEAD(&fs->wpages_io, &fs->wpages_prep);
	INIT_XLIST(&fs->wpages_prep);
	fs->n_wpages_prep = 0;
	ww = NULL;
	XLIST_FOR_EACH(w, &fs->wpages_io, WPAGE, list_entry) {
		if (!ww || w->rq.sec < ww->rq.sec) ww = w;
	}
	if (__unlikely(!ww)) return;
	/*__debug_printf("wp1: %p,%Ld-%d.", &ww->rq, ww->rq.sec, ww->rq.nsec);*/
	ww->sec_cpy = ww->rq.sec;
	ww->nsec_cpy = ww->rq.nsec;
	CALL_IORQ(&ww->rq, KERNEL$BIO);
	XLIST_FOR_EACH(w, &fs->wpages_io, WPAGE, list_entry) {
		DEL_FROM_LIST(&w->ihash_entry);
		DEL_FROM_LIST(&w->ehash_entry);
		if (w != ww) {
			/*__debug_printf("wp: %p,%Ld-%d.", &w->rq, w->rq.sec, w->rq.nsec);*/
			w->sec_cpy = w->rq.sec;
			w->nsec_cpy = w->rq.nsec;
			CALL_IORQ(&w->rq, KERNEL$BIO);
		}
	}
}

void VFS$DEFAULT_WRITEPAGES(FS *fs, int syn)
{
	VFS_DO_WRITE(fs);
	if (__unlikely(syn)) {	/* must not block if syn is 0 */
		unsigned seq = fs->wpage_seq;
/* Prevent starvation when wpage operations directly from async iorqs will be
   implemented. After 2 wrap-arounds we are sure that all initially sumbitted
   wpages ended. */
		while (fs->wpage_seq - seq < 2 && !XLIST_EMPTY(&fs->wpages_io)) {
			WQ_WAIT_SYNC(&fs->wpage_wait);
		}
	}
}

DECL_AST(PAGES_WROTE, SPL_FS, BIORQ)
{
	WPAGE *ww;
	WPAGE *w = GET_STRUCT(RQ, WPAGE, rq);
	PAGE *p;
	FS *fs;
	VFS$INVALIDATE_FILEBUFFERS(w->fs, w->sec_cpy, w->nsec_cpy);
	if (__unlikely(RQ->status < 0)) {
		KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, w->fs->filesystem_name, "ERROR WRITING PAGE: %s", strerror(-RQ->status));
		w->fs->write_error = 1;
	}

	/*__debug_printf("done: %p.", &w->rq);*/
	DEL_FROM_LIST(&w->list_entry);
	loop:
	KERNEL$RELEASE_WRITEBACK_TAG(w->rq.proc);
	p = w->p;
	if (!--((FNODE *)p->fnode)->pending_writes)
		WQ_WAKE_ALL(&((FNODE *)p->fnode)->pending_write_wait);
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(!(p->flags & PAGE_WRITECOUNT)))
		KERNEL$SUICIDE("PAGES_WROTE: %s: IO ON NON-LOCKED PAGE, FLAGS %X", w->fs->filesystem_name, p->flags);
	p->flags -= PAGE_WRITECOUNT_1;
	if (!(p->flags & PAGE_WRITECOUNT)) WQ_WAKE_ALL(&p->wait);
	LOWER_SPL(SPL_FS);
	ww = w->next;
	fs = w->fs;
	__slfree(w);
	w = ww;
	if (w) goto loop;

	WQ_WAKE_ALL(&fs->wpage_wait);
	if (XLIST_EMPTY(&fs->wpages_io)) {
		fs->wpage_seq++;
		VFS_DO_WRITE(fs);
		if (XLIST_EMPTY(&fs->wpages_io) && FS_BUFFERS_DONE(fs)) WQ_WAKE_ALL(&fs->syncer_wait);
	}
	RETURN;
}

