#include <SPAD/VFS.H>
#include <SPAD/HASH.H>

#include "SPADFS.H"
#include "STRUCT.H"
#include "SFSFNODE.H"

void SPAD_INIT_FNODE(FNODE *f_)
{
	SPADFNODE *f = (SPADFNODE *)f_;
	f->fnode_address = 0;
	f->ea_size = 0;
}

int SPAD_CREATE_FNODE(FNODE *f)
{
#define sf ((SPADFNODE *)f)
#define par ((SPADFNODE *)f->parent)
#define sfs ((SPADFS *)f->fs)
	sf->fnode_address = 0;
	sf->fnode_off = 0;
	sf->blk1 = 0;
	sf->blk1_n = 0;
	sf->blk2 = 0;
	sf->blk2_n = 0;
	sf->root = 0;
	if (__likely(!(sf->flags & FNODE_DIRECTORY))) {
		sf->run_length = 0;
		/* there's no need to create anything ... */

		/* test for prealloc. The first test is to avoid quadratic
		   complexity when the disk is out of space (i.e. do not test
		   FNODES that have been already tested) */
		if (__unlikely(!(sf->flags & FNODE_UNCOMMITTED_PREALLOC_HINT))) {
			SPAD_DISCARD_PREALLOC(sfs);	/* it may be possible that prealloc was left from processing previous directory (if files were deleted before allocating them) --- discard it, because we need different group now */
			VFS$PREPARE_CONTIG_ALLOC((FNODE *)sf, SPAD_TEST_CONTIG_ALLOC);
			SPAD_DO_PREALLOC(sfs, sf);
		}
	} else {
		struct fnode_block *b;
		unsigned dirblks;
		sf->blk1_n = par->blk1_n;
		sf->blk2_n = par->blk2_n;
		dirblks = sfs->sectors_per_block;
			/* estimate directory size */
		while (dirblks < sfs->sectors_per_fnodepage && sf->u.d.n_dirty > dirblks * 6) dirblks *= 2;
		if (__unlikely(!SPAD_CAN_OVERALLOC(sf, dirblks - sfs->sectors_per_block)))
			alloc_1: dirblks = sfs->sectors_per_block;
		if (__unlikely(__IS_ERR(b = SPAD_ALLOC_LEAF_PAGE(sf, par->root, dirblks, ALLOC_SKIP_ALMOST_FULL, &sf->root)))) {
			if (dirblks > sfs->sectors_per_block) goto alloc_1;
			if (__unlikely(__PTR_ERR(b) == -ENOSPC)) {
				SPADFS_ERROR((SPADFS *)sf->fs, TXFLAGS_FS_ERROR, "OUT OF SPACE FOR DIRECTORY");
			}
			return __PTR_ERR(b);
		}
		VFS$PUT_BUFFER(b);
	}
	return 0;
#undef sf
#undef par
#undef sfs
}

int SPAD_DELETE_FNODE(FNODE *fnode)
{
#define fs ((SPADFS *)fnode->fs)
	int r;
	if (fnode->flags & FNODE_DIRECTORY) {
		r = SPAD_REMOVE_RECURSIVE((SPADFNODE *)fnode, ((SPADFNODE *)fnode)->root, 0);
	} else {
		r = SPAD_TRUNCATE((SPADFNODE *)fnode, 0);
	}
	/* memset(fnode + sizeof(FNODE), 0, sizeof(SPADFNODE) - sizeof(FNODE)); jsem PRASE !! --- tohle tady nechavam jen jako ukazku, co jsem napsal :-( */
	return r;
#undef fs
}

static __finline__ int REQUESTED_FNODE_SIZE(SPADFNODE *sf)
{
	return FNODE_SIZE(sf->namelen, sf->ea_size);
}

static __finline__ void SET_SPAD_FNODE(SPADFNODE *sf, struct fnode *fnode, int part)
{
	fnode->ctime = sf->ctime;
	fnode->mtime = sf->mtime;
	if (__likely(!(sf->flags & FNODE_DIRECTORY))) {
		fnode->flags = 0;
		fnode->run10 = MAKE_PART_0(sf->blk1);
		fnode->run11 = MAKE_PART_1(sf->blk1);
		fnode->run1n = sf->blk1_n;
		fnode->run20 = MAKE_PART_0(sf->blk2);
		fnode->run21 = MAKE_PART_1(sf->blk2);
		fnode->run2n = sf->blk2_n;
		fnode->anode0 = MAKE_PART_0(sf->root);
		fnode->anode1 = MAKE_PART_1(sf->root);
		if (!part) fnode->size0 = sf->disk_size;
		else fnode->size1 = sf->disk_size;
	} else {
		fnode->flags = FNODE_FLAGS_DIR;
		if (!part) {
			fnode->run10 = MAKE_PART_0(sf->root);
			fnode->run11 = MAKE_PART_1(sf->root);
		} else {
			fnode->run20 = MAKE_PART_0(sf->root);
			fnode->run21 = MAKE_PART_1(sf->root);
		}
		fnode->run1n = sf->blk1_n;
		fnode->run2n = sf->blk2_n;
		if (!part) fnode->size0 = sf->disk_size;
		else fnode->size1 = sf->disk_size;
	}
}

static __finline__ void SWAP_SPAD_FNODE(SPADFNODE *sf, struct fnode *fnode)
{
	__u64 s;
	if (__unlikely(sf->flags & FNODE_DIRECTORY)) {
		__u32 r0 = fnode->run10;
		__u16 r1 = fnode->run11;
		fnode->run10 = fnode->run20;
		fnode->run11 = fnode->run21;
		fnode->run20 = r0;
		fnode->run21 = r1;
	}
	s = fnode->size0;
	fnode->size0 = fnode->size1;
	fnode->size1 = s;
}

#define DO_BLOCK_CHECKSUM						\
do {									\
	if (!(sfs->flags & SPADFS_DONT_MAKE_CHECKSUMS)) {		\
		fnode_block->flags |= FNODE_BLOCK_CHECKSUM_VALID;	\
		fnode_block->checksum ^= CHECKSUM_BASE ^ __byte_sum(fnode_block, FNODE_BLOCK_SIZE);\
	} else fnode_block->flags &= ~FNODE_BLOCK_CHECKSUM_VALID;	\
} while (0)

#define DO_BLOCK_DIRTY							\
do {									\
	DO_BLOCK_CHECKSUM;						\
	VFS$MARK_BUFFER_DIRTY(b, (((unsigned long)fnode_block >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1) & ~(sfs->sectors_per_block - 1), sfs->sectors_per_block);\
} while (0)

static __finline__ struct fnode *SPAD_READ_FNODE(SPADFNODE *f, struct fnode_block **fb, BUFFER **b)
{
#define sfs ((SPADFS *)f->fs)
	struct fnode *fnode;
	*fb = VFS$READ_BUFFER_SYNC(f->fs, f->fnode_address, 1, f->fs->buffer_readahead, b);
	if (__unlikely(__IS_ERR(*fb))) return (void *)*fb;
	if (__unlikely((*fb)->magic != FNODE_BLOCK_MAGIC) && __unlikely((*fb)->magic != FIXED_FNODE_BLOCK_MAGIC)) {
		SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "BAD POINTER WHEN UPDATING FNODE: %012"__d_off_format"X/%04X", f->fnode_address, f->fnode_off);
		SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "BAD POINTER WHEN UPDATING FNODE %s: %012"__d_off_format"X/%04X", f->name, f->fnode_address, f->fnode_off);
		VFS$PUT_BUFFER(*fb);
		return __ERR_PTR(-EFSERROR);
	}
	fnode = (struct fnode *)((char *)*fb + f->fnode_off);
	return fnode;
#undef sfs
}

static int SPAD_UPDATE_FNODE(SPADFNODE *f)
{
#define sfs ((SPADFS *)f->fs)
	BUFFER *b;
	struct fnode_block *fnode_block;
	struct fnode *fnode;
	fnode = SPAD_READ_FNODE(f, &fnode_block, &b);
	if (__unlikely(__IS_ERR(fnode))) return __PTR_ERR(fnode);
	if (!CC_CURRENT(sfs, fnode->cc, fnode->txc)) {
		CC_SET_CURRENT(sfs, &fnode->cc, &fnode->txc);
		fnode->next &= ~FNODE_NEXT_FREE;
	}
	SET_SPAD_FNODE(f, fnode, !CC_VALID(sfs, fnode->cc, fnode->txc));
	DO_BLOCK_DIRTY;
	VFS$PUT_BUFFER(fnode_block);
	return 0;
#undef sfs
}

static void MOVE_FNODE_PTR(SPADFNODE *sf, int namelen, __const__ char *name, __d_off src_sec, unsigned src_off, __d_off dst_sec, unsigned dst_off)
{
	SPADFNODE *f2;
	__const__ char *e;
	int hash;
	/*__debug_printf("MOVE_PTR: %.*s %08LX/%04X->%08LX/%04X   ", namelen > 10 ? 10 : namelen, name, src_sec, src_off, dst_sec, dst_off);*/
	if (__unlikely(!(sf->flags & FNODE_DIRECTORY)))
		KERNEL$SUICIDE("MOVE_FNODE_PTR: %s: NOT A DIRECTORY (%08X)", VFS$FNODE_DESCRIPTION((FNODE *)sf), (int)sf->flags);
	if (__unlikely(!sf->u.d.hash)) return;
	e = name;
	hash = 0;
	quickcasehash(e, e < name + namelen, hash);
	hash &= sf->u.d.hash_mask;
	XLIST_FOR_EACH(f2, &sf->u.d.hash[hash], SPADFNODE, hash_entry) if (__likely(f2->namelen == namelen) && __likely(!_memcasecmp(f2->name, name, namelen))) {
		if (__unlikely(!(f2->flags & (FNODE_FILE | FNODE_DIRECTORY))) /*|| __unlikely((fs->flags & FNODE_UNCOMMITTED) != 0)*/) break; /* there should be no uncommitted fnode of that name */
		if (__unlikely(f2->fnode_address != src_sec) || __unlikely(f2->fnode_off) != src_off) {
			SPADFS_ERROR((SPADFS *)sf->fs, TXFLAGS_FS_ERROR, "FNODE POINTER MISMATCH: MEMORY %012"__d_off_format"X/%04X, DISK %012"__d_off_format"X/%04X, NEW %012"__d_off_format"X/%04X", f2->fnode_address, f2->fnode_off, src_sec, src_off, dst_sec, dst_off);
			SPADFS_ERROR((SPADFS *)sf->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "FNODE %s/%s POINTER MISMATCH: MEMORY %012"__d_off_format"X/%04X, DISK %012"__d_off_format"X/%04X, NEW %012"__d_off_format"X/%04X", sf->name, f2->name, f2->fnode_address, f2->fnode_off, src_sec, src_off, dst_sec, dst_off);
		}
		f2->fnode_address = dst_sec;
		f2->fnode_off = dst_off;
		/*__debug_printf("HIT: %.10s   ", f2->name);*/
		return;
	}
	return;
}

static void COPY_FNODES_TO_BLOCK(SPADFNODE *sf, struct fnode_block *fnode_block, struct fnode_block *src_block, __d_off dst_sec, __d_off src_sec, int (*condition)(struct fnode *f, unsigned x), unsigned x)
{
#define sfs ((SPADFS *)sf->fs)
	struct fnode *dest = fnode_block->fnodes;
	struct fnode *fnode;
	int size, dsize;
	fnode = src_block->fnodes;
	next_fnode:
	size = fnode->next & FNODE_NEXT_SIZE;
	dsize = -1;
	VALIDATE_FNODE(sfs, src_block, fnode, size, ok, skip, bad_fnode);
	ok:
	if (condition && !condition(fnode, x)) goto skip;
	dsize = dest->next & FNODE_NEXT_SIZE;
	if (dsize >= size) {
		memcpy(dest, fnode, size);
		MOVE_FNODE_PTR(sf, fnode->namelen, FNODE_NAME(fnode), src_sec, (char *)fnode - (char *)src_block, dst_sec, (char *)dest - (char *)fnode_block);
		if (dsize > size) {
			dest = (struct fnode *)((char *)dest + size);
			dest->next = (dsize - size) | FNODE_NEXT_FREE;
			dest->cc = 0;
			dest->txc = 0;
		}
		skip:
		fnode = (struct fnode *)((char *)fnode + size);
		if (__unlikely(!((unsigned long)fnode & (FNODE_BLOCK_SIZE - 1)))) {
			next_block:
			if (__unlikely((src_block->flags & FNODE_BLOCK_LAST) != 0)) goto end;
			src_sec += FNODE_BLOCK_SIZE / BIO_SECTOR_SIZE;
			src_block = (struct fnode_block *)((char *)src_block + FNODE_BLOCK_SIZE);
			fnode = src_block->fnodes;
		}
		if (__unlikely(dsize == size)) goto next_dest;
		goto next_fnode;
	} else {
		next_dest:
		if (fnode_block->flags & FNODE_BLOCK_LAST) {
			SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "OUT OF SPACE WHEN COPYING FNODE PAGE %012"__d_off_format"X -> %012"__d_off_format"X", src_sec, dst_sec);
			goto end;
		}
		DO_BLOCK_CHECKSUM;
		dst_sec += FNODE_BLOCK_SIZE / BIO_SECTOR_SIZE;
		fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
		dest = fnode_block->fnodes;
		goto next_fnode;
	}
	end:
	DO_BLOCK_CHECKSUM;
	return;

	bad_fnode:
	SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012" __d_off_format "X", src_sec);
	goto next_block;
#undef sfs
}

#define HASH_1_SHIFT	16
#define HASH_1		(1 << HASH_1_SHIFT)
#define HASH_VAL	(HASH_1 - 1)

static int TEST_HASH_BIT(struct fnode *fnode, unsigned hash_bits)
{
	hash_t hash = name_len_hash(FNODE_NAME(fnode), fnode->namelen);
	return ((hash >> (hash_bits & HASH_VAL)) ^ (hash_bits >> HASH_1_SHIFT) ^ 1) & 1;
}

static int TEST_HASH(SPADFS *sfs, struct fnode_block *fnode_block, int hash_bits)
{
	hash_t hash;
	int found = 0;
	struct fnode *fnode;
	int size;
	next_block:
	fnode = fnode_block->fnodes;
	next_fnode:
	size = fnode->next & FNODE_NEXT_SIZE;
	VALIDATE_FNODE(sfs, fnode_block, fnode, size, used, free, bad_fnode);
	used:
	hash = name_len_hash(FNODE_NAME(fnode), fnode->namelen);
	found |= 1 << ((hash >> hash_bits) & 1);
	if (found == 3) return 3;
	free:
	fnode = (struct fnode *)((char *)fnode + size);
	if (__likely(((unsigned long)fnode & (FNODE_BLOCK_SIZE - 1)) != 0)) goto next_fnode;
	if (!(fnode_block->flags & FNODE_BLOCK_LAST)) {
		fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
		goto next_block;
	}
	return found;
	bad_fnode:
	return -1;
}

static int IS_BLOCK_EMPTY(SPADFS *fs, struct fnode_block *fnode_block, __d_off blk)
{
	struct fnode *fnode;
	int size;

	next_block:
	if (__unlikely(fnode_block->magic != FNODE_BLOCK_MAGIC)) goto bad_block;
	fnode = fnode_block->fnodes;
	next_fnode:
	size = fnode->next & FNODE_NEXT_SIZE;
	VALIDATE_FNODE(fs, fnode_block, fnode, size, used, free, bad_fnode);
	used:
	return 0;
	free:
	fnode = (struct fnode *)((char *)fnode + size);
	if (__likely(((unsigned long)fnode & (FNODE_BLOCK_SIZE - 1)) != 0)) goto next_fnode;
	if (!(fnode_block->flags & FNODE_BLOCK_LAST)) {
		fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
		if (__unlikely(!((unsigned long)fnode_block & __PAGE_CLUSTER_SIZE_MINUS_1))) goto eob;
		blk += FNODE_BLOCK_SIZE / BIO_SECTOR_SIZE;
		goto next_block;
	}
	return 1;

	bad_fnode:
	SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012" __d_off_format "X", blk);
	return 0;
	eob:
	SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "OVERFLOW ON BLOCK %012" __d_off_format "X", blk);
	return 0;
	bad_block:
	SPADFS_ERROR(fs, TXFLAGS_FS_ERROR, "BAD FNODE BLOCK ON BLOCK %012" __d_off_format "X", blk);
	return 0;
}

int SPAD_ADD_FNODE_TO_DIRECTORY(FNODE *f)
{
#define sf ((SPADFNODE *)f)
#define sfs ((SPADFS *)f->fs)
	__d_off c[2];
	int hash_bits, splitable;
	int r;
	int pass2;
	__d_off fnode_blk_sector;
	__d_off dnode;
	int size, rsize;
	struct fnode *fnode;
	struct dnode_page *dnode_page;
	BUFFER *b;
	struct fnode_block *fnode_block;
	struct fnode_block *first_block;
	hash_t hash = name_hash(f->name);
	rsize = REQUESTED_FNODE_SIZE(sf);
	c[1] = 0;
	total_restart:
	fnode_block = SPAD_FIND_LEAF_PAGE_FOR_HASH((SPADFNODE *)sf->parent, ((SPADFNODE *)f->parent)->root, hash, &b, &fnode_blk_sector, &dnode, &hash_bits, &splitable);
	/*__debug_printf("ADD: '%.10s->%.10s'   ", f->name, f->parent->name);*/
	pass2 = 0;
	next_fnode_page:
	if (__unlikely(__IS_ERR(fnode_block))) return __PTR_ERR(fnode_block);
	first_block = fnode_block;
	next_block:
	fnode = fnode_block->fnodes;
	next_fnode:
	retry_fnode:
	size = fnode->next & FNODE_NEXT_SIZE;
	VALIDATE_FNODE(sfs, fnode_block, fnode, size, used, free, bad_fnode);
	free:
	if (!CC_CURRENT(sfs, fnode->cc, fnode->txc)) {
		if (size >= rsize) {
			sf->fnode_address = b->blk + (((unsigned long)fnode_block >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1);
			sf->fnode_off = (char *)fnode - (char *)fnode_block;
			if (rsize < size) {
				struct fnode *afterfnode = (struct fnode *)((char *)fnode + rsize);
				afterfnode->next = (size - rsize) | FNODE_NEXT_FREE;
				afterfnode->cc = 0;
				afterfnode->txc = 0;
				fnode->next = rsize | FNODE_NEXT_FREE;
			}
			CC_SET_CURRENT_INVALID(sfs, &fnode->cc, &fnode->txc);
			fnode->size0 = 0;
			SET_SPAD_FNODE(sf, fnode, 1);
			fnode->namelen = sf->namelen;
			memcpy(FNODE_NAME(fnode), sf->name, sf->namelen);
			memset(FNODE_NAME(fnode) + sf->namelen, 0, FNODE_EA_POS(sf->namelen) - FNODE_NAME_POS - sf->namelen);
			memcpy((char *)fnode + FNODE_EA_POS(sf->namelen), sf->ea, sf->ea_size);
			DO_BLOCK_DIRTY;
			VFS$PUT_BUFFER(fnode_block);
			/*__debug_printf("mam to!   ");*/
			return 0;
		} else {
			struct fnode *new_fnode = (struct fnode *)((char *)fnode + size);
			int new_size;
			if (__unlikely(!((unsigned long)new_fnode & (FNODE_BLOCK_SIZE - 1)))) goto end_of_block;
			new_size = new_fnode->next & FNODE_NEXT_SIZE;
			VALIDATE_FNODE(sfs, fnode_block, new_fnode, new_size, s_used, s_free, bad_fnode);
			s_free:
			if (__likely(!CC_CURRENT(sfs, new_fnode->cc, new_fnode->txc))) {
				fnode->next += new_size;
				DO_BLOCK_DIRTY;
				goto retry_fnode;
			}
			s_used:
			fnode = new_fnode;
			size = new_size;
		}
	}
	used:
	fnode = (struct fnode *)((char *)fnode + size);
	if (__likely(((unsigned long)fnode & (FNODE_BLOCK_SIZE - 1)) != 0)) goto next_fnode;
	end_of_block:
	if (!(fnode_block->flags & FNODE_BLOCK_LAST)) {

			/* GO TO THE NEXT FNODE BLOCK IN PAGE */

		fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
		goto next_block;
	}
	if (CC_VALID(sfs, fnode_block->cc, fnode_block->txc)) {

			/* GO DOWN THE CHAIN */

		int dummy1, dummy2;
		fnode_blk_sector = MAKE_D_OFF(fnode_block->next0, fnode_block->next1);
		VFS$PUT_BUFFER(fnode_block);
		if (__unlikely(SPAD_STOP_CYCLES(fnode_blk_sector, &c))) goto cycle;
		read_new_block:
		pass2 = 1;
		read_new_block_x:
		fnode_block = SPAD_FIND_LEAF_PAGE_FOR_HASH((SPADFNODE *)sf->parent, fnode_blk_sector, hash, &b, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, &dummy1, &dummy2);
		goto next_fnode_page;
	}
	if (__unlikely(CC_CURRENT(sfs, fnode_block->cc, fnode_block->txc))) {

			/* RESURRECT FNODE PREVIOUSLY ALLOCATED IN CHAIN */

		fnode_blk_sector = MAKE_D_OFF(fnode_block->next0, fnode_block->next1);
		fnode_block->cc = 0;
		fnode_block->txc = 0;
		DO_BLOCK_DIRTY;
		VFS$PUT_BUFFER(fnode_block);
		if (__unlikely((r = SPAD_RESURRECT_BLOCKS((SPADFNODE *)f->parent, fnode_blk_sector, sfs->sectors_per_block))))
			return r;
		goto read_new_block;
	}
	/*__debug_printf("a nic nemam...   ");*/
	if (__likely(!pass2)) {
		__d_off new_dnode;
		unsigned fn_sectors = ((char *)fnode_block - (char *)first_block + FNODE_BLOCK_SIZE) >> BIO_SECTOR_SIZE_BITS;
		__d_off fnode_block_0_sector;
		__d_off fnode_block_1_sector;
		struct dnode_page_entry *e;
		int bits, data;

		/* 1. REALLOC CURRENT BLOCK IF IT'S ROOT & SMALL */

		if (!dnode && fn_sectors < sfs->sectors_per_fnodepage) {
			__d_off new_sector;
			unsigned new_sectors = FNODE_BLOCK_SIZE / BIO_SECTOR_SIZE;
			while (new_sectors <= fn_sectors) new_sectors <<= 1;
			if (__unlikely(!SPAD_CAN_OVERALLOC((SPADFNODE *)f->parent, new_sectors - 1))) goto alloc_chain;
			memcpy(sfs->scratch_page_2, first_block, fn_sectors << BIO_SECTOR_SIZE_BITS);
			VFS$PUT_BUFFER(fnode_block);
			fnode_block = SPAD_ALLOC_LEAF_PAGE((SPADFNODE *)f->parent, fnode_blk_sector, new_sectors, 0, &new_sector);
			if (__unlikely(__IS_ERR(fnode_block))) {
				goto read_new_block;
			}
			COPY_FNODES_TO_BLOCK((SPADFNODE *)f->parent, fnode_block, (struct fnode_block *)sfs->scratch_page_2, new_sector, fnode_blk_sector, NULL, 0);
			VFS$PUT_BUFFER(fnode_block);
			((SPADFNODE *)f->parent)->root = new_sector;
			r = SPAD_UPDATE_FNODE((SPADFNODE *)f->parent);
			if (__unlikely(r)) return r;
			VFS$CLEAN_BUFFERS_SYNC((FS *)sfs, fnode_blk_sector, fn_sectors);
			r = SPAD_FREE_BLOCKS((SPADFNODE *)f->parent, fnode_blk_sector, fn_sectors);
			if (__unlikely(r)) return r;
			fnode_blk_sector = new_sector;
			goto read_new_block;
		}

		/* 2. IF IT'S END OF HASH, ALLOC CHAIN */

		if (__unlikely(hash_bits >= HASH_BITS)) goto alloc_chain;

		/* 3. IF THE DNODE IS FULL, ALLOC NEW DNODE */

		if (splitable == -1) {
			VFS$PUT_BUFFER(fnode_block);
			if (__unlikely(!SPAD_CAN_OVERALLOC((SPADFNODE *)f->parent, sfs->dnode_page_sectors))) goto alloc_chain_x;
			if (__unlikely(SPAD_ALLOC_DNODE_PAGE((SPADFNODE *)f->parent, fnode_blk_sector, &new_dnode, dnode, fnode_blk_sector))) goto alloc_chain_x;
			if (dnode) {
				int data, i;
				struct dnode_page_entry *e;
				dnode_page = VFS$READ_BUFFER_SYNC((FS *)sfs, dnode, sfs->dnode_page_sectors, sfs->buffer_readahead, &b);
				if (__IS_ERR(dnode_page)) return __PTR_ERR(dnode_page);
				data = DNODE_ENTRY_OFFSET;
				if (!CC_VALID(sfs, dnode_page->cc, dnode_page->txc)) data += sfs->dnode_data_size;
				data = SPAD_BEGIN_MODIFY_DNODE(sfs, dnode_page, data);
				e = (struct dnode_page_entry *)((char *)dnode_page + data);
				for (i = 1 << sfs->dnode_hash_bits; i; i--) {
					__d_off ex = MAKE_D_OFF3(e->b0, e->b1, e->b2);
					if (ex == fnode_blk_sector) {
						e->b0 = MAKE_PART_30(new_dnode);
						e->b1 = MAKE_PART_31(new_dnode);
						e->b2 = MAKE_PART_32(new_dnode);
					}
					e++;
				}
				SPAD_END_MODIFY_DNODE(b, dnode_page, data);
				VFS$PUT_BUFFER(dnode_page);
			} else {
				((SPADFNODE *)f->parent)->root = new_dnode;
				r = SPAD_UPDATE_FNODE((SPADFNODE *)f->parent);
				if (__unlikely(r)) return r;
			}
			dnode = new_dnode;
			splitable = 0;
			goto read_new_block_x;
		}

		/* 4. TEST SPLITABILITY OF FNODE PAGE */

		/* go through fnodes and test (name_hash >> hash_bits) & 1 */

		r = TEST_HASH(sfs, first_block, hash_bits);
		if (__unlikely(r == -1)) goto bad_fnode;

		/* 5. ALLOC NEW 2 FNODE PAGES AND SPLIT FNODE */

		if (__unlikely(!SPAD_CAN_OVERALLOC((SPADFNODE *)f->parent, sfs->sectors_per_fnodepage << 1))) goto alloc_chain;
		memcpy(sfs->scratch_page_2, first_block, fn_sectors << BIO_SECTOR_SIZE_BITS);
		VFS$PUT_BUFFER(fnode_block);
		if (r & 1) {
			fnode_block = SPAD_ALLOC_LEAF_PAGE((SPADFNODE *)f->parent, fnode_blk_sector, sfs->sectors_per_fnodepage, 0, &fnode_block_0_sector);
			if (__IS_ERR(fnode_block)) goto read_new_block;
			/* don't copy it now, because allocation of page 1 might fail */
			/*COPY_FNODES_TO_BLOCK((SPADFNODE *)f->parent, fnode_block, (struct fnode_block *)sfs->scratch_page_2, fnode_block_0_sector, fnode_blk_sector, TEST_HASH_BIT, hash_bits);*/
			VFS$PUT_BUFFER(fnode_block);
		} else fnode_block_0_sector = 0;
		if (r & 2) {
			fnode_block = SPAD_ALLOC_LEAF_PAGE((SPADFNODE *)f->parent, fnode_blk_sector, sfs->sectors_per_fnodepage, 0, &fnode_block_1_sector);
			if (__IS_ERR(fnode_block)) {
				if (fnode_block_0_sector) {
					int rr;
					VFS$CLEAN_BUFFERS_SYNC((FS *)sfs, fnode_block_0_sector, sfs->sectors_per_fnodepage);
					rr = SPAD_FREE_BLOCKS((SPADFNODE *)f->parent, fnode_block_0_sector, sfs->sectors_per_fnodepage);
					if (__unlikely(rr)) {
						VFS$PUT_BUFFER(fnode_block);
						return rr;
					}
				}
				goto read_new_block;
			}
			COPY_FNODES_TO_BLOCK((SPADFNODE *)f->parent, fnode_block, (struct fnode_block *)sfs->scratch_page_2, fnode_block_1_sector, fnode_blk_sector, TEST_HASH_BIT, hash_bits | HASH_1);
			VFS$PUT_BUFFER(fnode_block);
		} else fnode_block_1_sector = 0;
		if (r & 1) {
			BUFFER *b;
			fnode_block = VFS$READ_BUFFER_SYNC((FS *)sfs, fnode_block_0_sector, sfs->sectors_per_fnodepage, sfs->buffer_readahead, &b);
			if (__IS_ERR(fnode_block))
				return __PTR_ERR(fnode_block);
			COPY_FNODES_TO_BLOCK((SPADFNODE *)f->parent, fnode_block, (struct fnode_block *)sfs->scratch_page_2, fnode_block_0_sector, fnode_blk_sector, TEST_HASH_BIT, hash_bits);
			VFS$PUT_BUFFER(fnode_block);
		}

		/* 6. READ PARENT DNODE */

		dnode_page = VFS$READ_BUFFER_SYNC((FS *)sfs, dnode, sfs->dnode_page_sectors, sfs->buffer_readahead, &b);
		if (__IS_ERR(dnode_page)) return __PTR_ERR(dnode_page);

		/* 7. SPLIT POINTERS ON DNODE */
		bits = hash_bits % (unsigned)sfs->dnode_hash_bits;
		data = DNODE_ENTRY_OFFSET;
		if (!CC_VALID(sfs, dnode_page->cc, dnode_page->txc)) data += sfs->dnode_data_size;
		data = SPAD_BEGIN_MODIFY_DNODE(sfs, dnode_page, data);
		e = (struct dnode_page_entry *)((char *)dnode_page + data);
		splitable &= (1 << bits) - 1;
		do {
			e[splitable].b0 = MAKE_PART_30(fnode_block_0_sector);
			e[splitable].b1 = MAKE_PART_31(fnode_block_0_sector);
			e[splitable].b2 = MAKE_PART_32(fnode_block_0_sector);
			splitable += 1 << bits;
			e[splitable].b0 = MAKE_PART_30(fnode_block_1_sector);
			e[splitable].b1 = MAKE_PART_31(fnode_block_1_sector);
			e[splitable].b2 = MAKE_PART_32(fnode_block_1_sector);
			splitable += 1 << bits;
		} while (splitable < 1 << sfs->dnode_hash_bits);
		SPAD_END_MODIFY_DNODE(b, dnode_page, data);
		VFS$PUT_BUFFER(dnode_page);
		VFS$CLEAN_BUFFERS_SYNC((FS *)sfs, fnode_blk_sector, fn_sectors);
		r = SPAD_FREE_BLOCKS((SPADFNODE *)f->parent, fnode_blk_sector, fn_sectors);
		if (__unlikely(r)) return r;
		goto total_restart;
	} else {

			/* ALLOCATE A NEW FNODE IN CHAIN */

		__d_off last_blk_sector;
		__d_off new_chain;
		alloc_chain:
		VFS$PUT_BUFFER(fnode_block);
		alloc_chain_x:
		last_blk_sector = fnode_blk_sector + (((char *)fnode_block - (char *)first_block) >> BIO_SECTOR_SIZE_BITS);
		fnode_block = SPAD_ALLOC_LEAF_PAGE((SPADFNODE *)f->parent, last_blk_sector, sfs->sectors_per_block, 0, &new_chain);
		if (__unlikely(__IS_ERR(fnode_block))) {
			if (__PTR_ERR(fnode_block) == -ENOSPC) SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "OUT OF SPACE FOR DIRECTORY CHAIN PAGE");
			return __PTR_ERR(fnode_block);
		}
		fnode = fnode_block->fnodes;
		size = fnode->next & FNODE_NEXT_SIZE;
		sf->fnode_address = new_chain;
		sf->fnode_off = (char *)fnode - (char *)fnode_block;
		if (__unlikely(rsize > size)) KERNEL$SUICIDE("SPAD_ADD_FNODE_TO_DIRECTORY: %s: FNODE SIZE %d > %d", VFS$FNODE_DESCRIPTION(f), rsize, size);
		if (rsize < size) {
			struct fnode *afterfnode = (struct fnode *)((char *)fnode + rsize);
			afterfnode->next = (size - rsize) | FNODE_NEXT_FREE;
			afterfnode->cc = 0;
			afterfnode->txc = 0;
			fnode->next = rsize | FNODE_NEXT_FREE;
		}
		CC_SET_CURRENT_INVALID(sfs, &fnode->cc, &fnode->txc);
		fnode->size0 = 0;
		SET_SPAD_FNODE(sf, fnode, 1);
		fnode->namelen = sf->namelen;
		memcpy(FNODE_NAME(fnode), sf->name, sf->namelen);
		fnode_block->prev0 = MAKE_PART_0(last_blk_sector);
		fnode_block->prev1 = MAKE_PART_1(last_blk_sector);
		DO_BLOCK_CHECKSUM;
		VFS$PUT_BUFFER(fnode_block);
		fnode_block = VFS$READ_BUFFER_SYNC((FS *)sfs, last_blk_sector & ~(__d_off)(sfs->sectors_per_block - 1), sfs->sectors_per_block, sfs->buffer_readahead, &b);
		if (__unlikely(__IS_ERR(fnode_block)))
			return __PTR_ERR(fnode_block);
		fnode_block = (struct fnode_block *)((char *)fnode_block + ((last_blk_sector & (sfs->sectors_per_block - 1)) << BIO_SECTOR_SIZE_BITS));
		fnode_block->next0 = MAKE_PART_0(new_chain);
		fnode_block->next1 = MAKE_PART_1(new_chain);
		fnode_block->cc = sfs->cc;
		fnode_block->txc = sfs->txc;
		DO_BLOCK_DIRTY;
		VFS$PUT_BUFFER(fnode_block);
		return 0;
	}
	VFS$PUT_BUFFER(fnode_block);
	return -EFSERROR;

	bad_fnode:
	VFS$PUT_BUFFER(fnode_block);
	SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012" __d_off_format "X", b->blk | (((unsigned long)fnode_block >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
	return -EFSERROR;

	cycle:
	SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "CYCLE DETECTED ON BLOCK %012" __d_off_format "X", fnode_blk_sector);
	return -EFSERROR;
#undef sf
#undef sfs
}

int SPAD_REMOVE_FNODE_FROM_DIRECTORY(FNODE *f)
{
#define sf ((SPADFNODE *)f)
#define sfs ((SPADFS *)f->fs)
#define dnode_page ((struct dnode_page *)fnode_block)
	__d_off c[2];
	int dummy1, dummy2;
	hash_t hash = name_hash(f->name);
	BUFFER *b;
	int size;
	__d_off fnode_blk_sector, dnode;
	struct fnode *fnode;
	struct fnode_block *first_block;
	struct fnode_block *fnode_block;
	c[1] = 0;
	fnode_block = SPAD_FIND_LEAF_PAGE_FOR_HASH((SPADFNODE *)sf->parent, ((SPADFNODE *)sf->parent)->root, hash, &b, &fnode_blk_sector, &dnode, &dummy1, &dummy2);
	/*__debug_printf("remove: '%.10s/%.10s' ", f->parent->name, f->name);*/
	next_fnode_page:
	if (__unlikely(__IS_ERR(fnode_block))) return __PTR_ERR(fnode_block);
	first_block = fnode_block;
	next_block:
	fnode = fnode_block->fnodes;
	next_fnode:
	size = fnode->next & FNODE_NEXT_SIZE;
	VALIDATE_FNODE(sfs, fnode_block, fnode, size, used, free, bad_fnode);
	used:
	if (fnode->namelen == sf->namelen && (__likely(!(sfs->flags & FS_MORE_SAME_NAMES)) ? !memcmp(FNODE_NAME(fnode), sf->name, sf->namelen) : !_memcasecmp(FNODE_NAME(fnode), sf->name, sf->namelen))) {
		__d_off prev;
		unsigned nsec;

		int empty;
		struct dnode_page_entry *e;
		int r;
		unsigned i;

		if (CC_CURRENT(sfs, fnode->cc, fnode->txc) && fnode->next & FNODE_NEXT_FREE) {
			fnode->cc = 0;
			fnode->txc = 0;
		} else {
			/*if (CC_VALID(sfs, fnode->cc, fnode->txc)) SET_SPAD_FNODE(sf, fnode, 1);*/
			if (CC_VALID(sfs, fnode->cc, fnode->txc) ^ CC_CURRENT(sfs, fnode->cc, fnode->txc)) SWAP_SPAD_FNODE(sf, fnode);
			fnode->cc = sfs->cc;
			fnode->txc = sfs->txc;
			fnode->next |= FNODE_NEXT_FREE;
		}
		DO_BLOCK_DIRTY;
		sf->fnode_address = 0;
		sf->fnode_off = 0;
		if (__likely(!IS_BLOCK_EMPTY(sfs, first_block, fnode_blk_sector))) {
			put_ret:
			VFS$PUT_BUFFER(fnode_block);
			return 0;
		}
		while (!(fnode_block->flags & FNODE_BLOCK_LAST))
			fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
		if (CC_VALID(sfs, fnode_block->cc, fnode_block->txc)) goto put_ret;
		nsec = ((char *)fnode_block - (char *)first_block + FNODE_BLOCK_SIZE) >> BIO_SECTOR_SIZE_BITS;
		free_back_chain:
		prev = MAKE_D_OFF(first_block->prev0, first_block->prev1);
		if (!prev && !dnode) goto put_ret;
		VFS$PUT_BUFFER(fnode_block);
		VFS$CLEAN_BUFFERS_SYNC((FS *)sfs, fnode_blk_sector, nsec);
		r = SPAD_FREE_BLOCKS((SPADFNODE *)sf->parent, fnode_blk_sector, nsec);
		if (__unlikely(r)) return r;
		if (prev) {
			nsec = 1;
			fnode_blk_sector = prev;
			read_more:
			fnode_block = VFS$READ_BUFFER_SYNC((FS *)sfs, fnode_blk_sector, nsec, sfs->buffer_readahead, &b);
			if (__IS_ERR(fnode_block)) return __PTR_ERR(fnode_block);
			if (nsec == 1) {
				CC_SET_CURRENT_INVALID(sfs, &fnode_block->cc, &fnode_block->txc);
				DO_BLOCK_DIRTY;
			}
			if (!(fnode_block->flags & FNODE_BLOCK_FIRST)) {
				VFS$PUT_BUFFER(fnode_block);
				if (!(fnode_blk_sector & __SECTORS_PER_PAGE_CLUSTER_MINUS_1)) {
					SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "FNODE BLOCK %012"__d_off_format"X CROSSES PAGE", fnode_blk_sector);
					return -EFSERROR;
				}
				fnode_blk_sector--;
				nsec++;
				goto read_more;
			}
			if (__likely(!IS_BLOCK_EMPTY(sfs, fnode_block, fnode_blk_sector))) goto put_ret;
			first_block = fnode_block;
			goto free_back_chain;
		}
		zap_up_dnode:
		fnode_block = VFS$READ_BUFFER_SYNC((FS *)sfs, dnode, sfs->dnode_page_sectors, sfs->buffer_readahead, &b);
		if (__IS_ERR(fnode_block)) return __PTR_ERR(fnode_block);
		r = SPAD_CHECK_DFNODE(fnode_block, b);
		if (r < 0) {
			VFS$PUT_BUFFER(fnode_block);
			return r;
		}
		if (r <= 1) goto bad_fnode;
		r = SPAD_BEGIN_MODIFY_DNODE(sfs, dnode_page, r);
		e = (struct dnode_page_entry *)((char *)dnode_page + r);
		empty = 1;
		for (i = 0; i < 1 << sfs->dnode_hash_bits; i++) {
			__d_off ee = MAKE_D_OFF3(e[i].b0, e[i].b1, e[i].b2);
			if (ee == fnode_blk_sector) {
				e[i].b0 = MAKE_PART_30(0);
				e[i].b1 = MAKE_PART_31(0);
				e[i].b2 = MAKE_PART_32(0);
			} else if (ee) {
				empty = 0;
			}
		}
		fnode_blk_sector = dnode;
		dnode = dnode_page->up_dnode;
		SPAD_END_MODIFY_DNODE(b, dnode_page, r);
		VFS$PUT_BUFFER(fnode_block);
		if (dnode && empty) {
			VFS$CLEAN_BUFFERS_SYNC((FS *)sfs, fnode_blk_sector, sfs->dnode_page_sectors);
			r = SPAD_FREE_BLOCKS((SPADFNODE *)sf->parent, fnode_blk_sector, sfs->dnode_page_sectors);
			if (__unlikely(r)) return r;
			goto zap_up_dnode;
		}
		return 0;
	}
	free:
	fnode = (struct fnode *)((char *)fnode + size);
	if (__likely(((unsigned long)fnode & (FNODE_BLOCK_SIZE - 1)) != 0)) goto next_fnode;
	if (!(fnode_block->flags & FNODE_BLOCK_LAST)) {

			/* GO TO THE NEXT FNODE BLOCK IN PAGE */

		fnode_block = (struct fnode_block *)((char *)fnode_block + FNODE_BLOCK_SIZE);
		goto next_block;
	}
	if (CC_VALID(sfs, fnode_block->cc, fnode_block->txc)) {

			/* GO DOWN THE CHAIN */

		int dummy1, dummy2;
		fnode_blk_sector = MAKE_D_OFF(fnode_block->next0, fnode_block->next1);
		VFS$PUT_BUFFER(fnode_block);
		if (__unlikely(SPAD_STOP_CYCLES(fnode_blk_sector, &c))) goto cycle;
		fnode_block = SPAD_FIND_LEAF_PAGE_FOR_HASH((SPADFNODE *)sf->parent, fnode_blk_sector, hash, &b, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END, &dummy1, &dummy2);
		goto next_fnode_page;
	}
	VFS$PUT_BUFFER(fnode_block);
	SPADFS_ERROR((SPADFS *)sf->fs, TXFLAGS_FS_ERROR, "FNODE NOT FOUND IN DIRECTORY %012"__d_off_format"X", ((SPADFNODE *)sf->parent)->root);
	SPADFS_ERROR((SPADFS *)sf->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "FNODE %s/%s NOT FOUND IN DIRECTORY %012"__d_off_format"X", sf->parent->name, sf->name, ((SPADFNODE *)sf->parent)->root);
	sf->fnode_address = 0;
	sf->fnode_off = 0;
	return -EFSERROR;

	bad_fnode:
	VFS$PUT_BUFFER(fnode_block);
	SPADFS_ERROR(sfs, TXFLAGS_FS_ERROR, "BAD FNODE ON BLOCK %012" __d_off_format "X", b->blk | (((unsigned long)fnode_block >> BIO_SECTOR_SIZE_BITS) & __SECTORS_PER_PAGE_CLUSTER_MINUS_1));
	return -EFSERROR;

	cycle:
	SPADFS_ERROR((SPADFS *)sf->fs, TXFLAGS_FS_ERROR, "CYCLE DETECTED IN DIRECTORY %012"__d_off_format"X", ((SPADFNODE *)sf->parent)->root);
	SPADFS_ERROR((SPADFS *)sf->fs, TXFLAGS_FS_ERROR | SPADFS_ERROR_SECRET, "CYCLE DETECTED WHEN REMOVING FNODE %s/%s FROM DIRECTORY %012"__d_off_format"X", sf->parent->name, sf->name, ((SPADFNODE *)sf->parent)->root);
	return -EFSERROR;
#undef dnode_page
#undef sfs
#undef sf
}

int SPAD_WRITE_FNODE_DATA(FNODE *f_)
{
#define sfs ((SPADFS *)f->fs)
	SPADFNODE *f = (SPADFNODE *)f_;
	int r;
	__u64 fsize = f->size;
	if (__likely((f->flags & FNODE_FILE) != 0) && f->disk_size != fsize) {
		struct fnode_block *fb;
		struct fnode *fnode;
		BUFFER *b;
		__u64 dsize;
		if (!f->fnode_address) dsize = 0;
		else {
			fnode = SPAD_READ_FNODE(f, &fb, &b);
			if (__unlikely(__IS_ERR(fnode))) return __PTR_ERR(fnode);
			if (CC_VALID(sfs, fnode->cc, fnode->txc) ^ CC_CURRENT(sfs, fnode->cc, fnode->txc))
				dsize = fnode->size0;
			 else
				dsize = fnode->size1;
			VFS$PUT_BUFFER(fnode);
		}
		if (__unlikely(fsize < f->disk_size)) {
			r = SPAD_TRUNCATE(f, fsize);
			if (__unlikely(r)) return r;
		} else {
			if (__unlikely(dsize > f->disk_size)) {
				r = SPAD_RESURRECT(f, fsize < dsize ? fsize : dsize);
				if (__unlikely(r)) return r;
			}
			if (fsize > f->disk_size) {
				r = SPAD_EXTEND(f, fsize);
				if (__unlikely(r)) return r;
			}
		}
	}
	if (__unlikely(r = VFS$WRITEPAGES((FNODE *)f))) return r;
	if (!f->fnode_address) return 0;
	return SPAD_UPDATE_FNODE(f);
#undef sfs
}



