#ifndef _SPADFS_SPADFS_H
#define _SPADFS_SPADFS_H

#include <SYS/TYPES.H>
#include "STRUCT.H"
#include <SPAD/VFS.H>
#include <SPAD/ALLOC.H>
#include <SYS/STATFS.H>
#include <SYS/STAT.H>

#define SPAD_MAX_GROUP_PART_ALLOC_BITS	3	/* max rq size: 1/8 of a group */
#define SPAD_AVG_FREE_DIVIDE_OTHERZONE	2	/* 1/2 avg free space in other zones */
#define SPAD_PROPAGATE_HINT_TIME	3600	/* propagate allocation hint on parent directories, if created in last 1 hour */

/* fs->flags */
#define SPADFS_DONT_MAKE_CHECKSUMS	0x00000001
#define SPADFS_DONT_CHECK_CHECKSUMS	0x00000002

#define MAKE_D_OFF(l,h) (sizeof(__d_off) <= 4 ? (l) : __make64(l, h))
#define MAKE_PART_0(l)	((__u32)(l))
#define MAKE_PART_1(l)	(sizeof(__d_off) <= 4 ? 0 : ((__d_off)(l) >> 32))

#define MAKE_D_OFF3(l,h,hh) (sizeof(__d_off) <= 4 ? (l) | ((h) << 16) : __make64((l) | ((h) << 16), hh))
#define MAKE_PART_30(l)	((__u16)(l))
#define MAKE_PART_31(l)	((__u16)((__u32)(l) >> 16))
#define MAKE_PART_32(l)	(sizeof(__d_off) <= 4 ? 0 : ((__d_off)(l) >> 32))

struct fnode;
struct apage_info;

struct zone {
	__d_off freespace;
	__u16 grp_start;
	__u16 grp_n;
};

typedef struct {
	FS_HEAD;
	__s32 *cct;	/* crash count table -- array __s32[65536] */
	__u16 cc;
	__s32 txc;	/* == cct[cc] */

	__s32 a_txc;
	__u16 a_cc;

	unsigned n_groups;
	struct group_info *group_info;		/* size is n_groups */
	struct apage_info *apage_info;		/* size is n_apages */

	int min_apage_invalid_checksum;
	int max_apage_invalid_checksum;

	unsigned n_apages;			/* size of apage_index */
	unsigned n_active_apages;
	struct apage_index_entry *apage_index;	/* size is n_apages */

	__u8 *scratch_page;
	__u8 *scratch_page_2;

	unsigned blocksize;
	__u8 sectors_per_block_bits;
	unsigned sectors_per_block;

		/* !!! FIXME: remove redundant entries ? */
	__u8 sectors_per_cluster_bits;
	unsigned sectors_per_cluster_mask;
	unsigned clustersize;
	__u32 cluster_threshold;

	__u8 sectors_per_page_bits;
	unsigned sectors_per_page;
	__u8 pagesize_bits;
	unsigned pagesize;

	__u8 sectors_per_fnodepage_bits;
	unsigned sectors_per_fnodepage;

	unsigned dnode_page_sectors;
	unsigned dnode_data_size;
	__u8 dnode_hash_bits;

	__d_off prealloc_sector;
	unsigned prealloc_length;

	__u8 sectors_per_group_bits;
	__d_off group_mask;

	unsigned max_alloc;

	__d_off reserve_sectors;

	__d_off free;
	__d_off free_allowed;
	__d_off free_at_commit;
	__d_off overaccounted_avg;

	struct zone zones[3];

	__d_off txb_sec;
	__d_off apage_index0_sec;
	__d_off apage_index1_sec;
	__d_off cct_sec;
	__d_off root_sec;

	unsigned n_info_pages;		/* number of pages allocated in group_info and apage_info */
	void *apage_index_addr;	
	unsigned n_apage_index_pages;		/* number of pages allocated in apage index */

	__u32 txflags;
	__u32 txflags_new;

	int dont_make_checksums_set:1;
	int dont_check_checksums_set:1;

} SPADFS;

typedef struct {
	FNODE_HEAD;
	__d_off fnode_address;
	unsigned fnode_off;
	__d_off blk1;
	__d_off blk2;
	unsigned blk1_n;	/* for directories: small zone group hint */
	unsigned blk2_n;	/* for directories: big zone group hint */
	__d_off root;		/* root dnode/anode */
	unsigned ea_size;
	__u8 ea[FNODE_MAX_EA_SIZE];
} SPADFNODE;

typedef struct {
	hash_t hash;
	__d_off chain;
	unsigned fnode_offset;
} SPADREADDIRCOOKIE;

typedef struct {
	PAGEINRQ_HEAD;
	__d_off dirptr;
	hash_t hash;
	__d_off c[2];
	SPADREADDIRCOOKIE cookie;
	int hash_bits;
	hash_t next_hash;
	__d_off chain_skip;
	int depth_now;
	int depth_total;
} SPADPAGEINRQ;

struct group_info {
	__d_off freespace;
	struct zone *zone;
};

struct apage_info {	/* note: this is zeroed when splitting apage */
	char invalid_checksum;
};

#define ALLOC_METADATA		0x00010000	/* do not cross page */
#define ALLOC_SMALL_FILE	0x00020000
#define ALLOC_BIG_FILE		0x00040000	/* do not cross cluster */
#define ALLOC_PARTIAL_AT_GOAL	0x00080000	/* alloc partial block exactly at goal */
#define ALLOC_SKIP_ALMOST_FULL	0x00100000
#define ALLOC_NO_CHARGE		0x01000000
#define ALLOC_NEW_GROUP_HINT	0x02000000	/* returned */

#define ALLOC_FREE_FROM		0x80000000	/* used internally, DO NOT SET ! */

#define ALLOC_MASK_MASK		0x0000FFFF
#define ALLOC_MASK_1		0x00000001
#define ALLOC_MASK(flags)	(((unsigned)(flags) / ALLOC_MASK_1) & ALLOC_MASK_MASK)

/*	in STRUCT.H
#define MAX_ALLOC_MASK_BITS	16
*/

struct alloc {
	__d_off sector;
	__d_off top;		/* used internally, let it be */
	unsigned n_sectors;
	unsigned flags;
};

#define SPADFS_ERROR_SECRET	__INT_SGN_BIT
void __PRINTF_ATTR__(3,4) SPADFS_ERROR(SPADFS *fs, unsigned flags, char *msg, ...);
struct txblock *SPAD_READ_TX_BLOCK(SPADFS *fs, BUFFER **bb);
void SPAD_TX_BLOCK_CHECKSUM(struct txblock *tb);

void SPAD_APAGE_FIX_CHECKSUMS(SPADFS *fs);
int SPAD_FREE_BLOCKS_(SPADFNODE *fnode, __d_off start, __d_off n_blocks, int acct);
int SPAD_ALLOC_BLOCKS(SPADFNODE *fnode, struct alloc *al);
static __finline__ int SPAD_FREE_BLOCKS(SPADFNODE *fnode, __d_off start, __d_off n_blocks)
{
	return SPAD_FREE_BLOCKS_(fnode, start, n_blocks, 1);
}
int SPAD_RESURRECT_BLOCKS(SPADFNODE *fnode, __d_off start, __u32 n_blocks);
void SPAD_CHARGE_QUOTA(SPADFNODE *fnode, unsigned n_blocks);
int SPAD_COUNT_FREE_SPACE(SPADFS *fs);

void SPAD_FNODE_CTOR(void *fs, void *fnode);
void SPAD_PAGEINRQ_CTOR(void *fs, void *pageinrq);
void SPAD_GET_FNODE(SPADFNODE *f, struct fnode *fd, __d_off addr, unsigned pos);
void SPAD_LOOKUP(PAGEINRQ *p);
void SPAD_INIT_READDIR_COOKIE(struct readdir_buffer *buf, FNODE *f);
void SPAD_READDIR(PAGEINRQ *readdir, struct readdir_buffer *buf);
extern AST_STUB SPAD_LOOKUP_AST;

int SPAD_CHECK_DFNODE(struct fnode_block *f, BUFFER *b);
struct fnode_block *SPAD_ALLOC_LEAF_PAGE(SPADFNODE *fn, __d_off hint, unsigned sectors, unsigned flags, __d_off *result);
int SPAD_ALLOC_DNODE_PAGE(SPADFNODE *fn, __d_off hint, __d_off *result, __d_off parent, __d_off ptr_init);
struct fnode_block *SPAD_FIND_LEAF_PAGE_FOR_HASH(SPADFNODE *fn, __d_off root, hash_t hash, BUFFER **bb, __d_off *fnode_sector, __d_off *prev_dnode, int *hash_bits, int *splitable);
int SPAD_BEGIN_MODIFY_DNODE(SPADFS *fs, struct dnode_page *d, int valid);
void SPAD_END_MODIFY_DNODE(BUFFER *b, struct dnode_page *d, int valid);
int SPAD_REMOVE_RECURSIVE(SPADFNODE *fn, __d_off root, int depth);

void SPAD_COMMIT(FS *fs);
void SPAD_INIT_FNODE(FNODE *f);
int SPAD_CREATE_FNODE(FNODE *f);
int SPAD_WRITE_FNODE_DATA(FNODE *f);
int SPAD_DELETE_FNODE(FNODE *f);
int SPAD_REMOVE_FNODE_FROM_DIRECTORY(FNODE *f);
int SPAD_ADD_FNODE_TO_DIRECTORY(FNODE *f);

#define is_bmap_invalid_nozerotest(f)	\
	(__unlikely(((unsigned long)(f)->disk_blk | (unsigned long)(f)->file_blk | (unsigned long)(f)->run_length) & (((SPADFS *)(f)->fs)->sectors_per_block - 1)) || __unlikely(((f)->disk_blk & ~SIZE_MASK) != 0) || __unlikely((f)->disk_blk + (f)->run_length > ((SPADFS *)(f)->fs)->size))

#define is_bmap_invalid(f)		\
	(is_bmap_invalid_nozerotest(f) || __unlikely(!(f)->run_length))

void SPAD_INVALID_BMAP_MSG(SPADFNODE *f);

int SPAD_SYNC_BMAP(FNODE *f, __d_off off, int try);
void SPAD_BMAP(PAGEINRQ *pgin_);
int SPAD_EXTEND(SPADFNODE *f, _u_off_t size);
int SPAD_RESURRECT(SPADFNODE *f, _u_off_t size);
int SPAD_TRUNCATE(SPADFNODE *f, _u_off_t size);
void SPAD_INIT_LAST_PAGE(FNODE *f, char *ptr, unsigned len);

int SPAD_TEST_CONTIG_ALLOC(FNODE *f);
void SPAD_DO_PREALLOC_(SPADFNODE *sf);
void SPAD_DISCARD_PREALLOC_(SPADFS *sfs);

static __finline__ void SPAD_DO_PREALLOC(SPADFS *sfs, SPADFNODE *sf)
{
	if (__likely(sfs->prealloc_length != 0)) SPAD_DO_PREALLOC_(sf);
}

static __finline__ int SPAD_DISCARD_PREALLOC(SPADFS *sfs)
{
	if (__unlikely(sfs->prealloc_length != 0)) {
		SPAD_DISCARD_PREALLOC_(sfs);
		return 1;
	}
	return 0;
}

void *SPAD_ACCOUNT(FNODE *f, int acct, PAGE *p);
void SPAD_UNACCOUNT(FNODE *f, int acct, PAGE *p);
int SPAD_CAN_OVERALLOC(SPADFNODE *f, __u32 blocks);

int SPAD_STOP_CYCLES(__d_off key, __d_off (*c)[2]);

void SPAD_STAT(FNODE *f_, struct stat *stat);
WQ *SPAD_STATFS(FS *fs_, struct statfs *stat);

/*
static __finline__ int CC_VALID(SPADFS *fs, __u16 cc, __s32 txc)
{
	return (__s32)((__u32)fs->cct[cc] - (__u32)txc) >= 0;
}

static __finline__ int CC_CURRENT(SPADFS *fs, __u16 cc, __s32 txc)
{
	return !((cc ^ fs->cc) | ((txc ^ fs->txc) & 0x7fffffff));
}
*/

/* gcc has inefficient inline functions returning logical expressions */

#define CC_VALID(fs_, cc_, txc_)	((__s32)((__u32)(fs_)->cct[(__u16)(cc_)] - (__u32)(txc_)) >= 0)

#define CC_CURRENT(fs_, cc_, txc_)	(!(((__u16)(cc_) ^ (fs_)->cc) | (((txc_) ^ (fs_)->txc) & 0x7fffffff)))

static __finline__ void CC_SET_CURRENT(SPADFS *fs, __u16 *cc, __s32 *txc)
{
	*txc = (~((__u32)fs->cct[*cc] - (__u32)*txc) & 0x80000000) | fs->txc;
	*cc = fs->cc;
}

static __finline__ void CC_SET_CURRENT_INVALID(SPADFS *fs, __u16 *cc, __s32 *txc)
{
	*txc = fs->txc ^ 0x80000000;
	*cc = fs->cc;
}

#endif
