#include <SYS/TYPES.H>
#include <SPAD/KEYBOARD.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/RANDOM.H>
#include <SPAD/TTY.H>
#include <SPAD/TIMER.H>
#include <SYS/KD.H>
#include <SPAD/VM.H>
#include <SPAD/TIMER.H>
#include <SPAD/SLAB.H>
#include <VALUES.H>

#define FROM_KBD
#include "KEYMAP/DEFAULT.I"

/*
 * Files PCKBD.C PCKBD.INC and KBD.INC contain these parts:
 *	PCKBD.C - all PC-controller specific things of a driver, passes scancodes to PCKBD.INC
 *	PCKBD.INC - all PC-keyboard specific things, passes keycodes (KEYBOARD.H) to KBD.INC
 *	KBD.INC - common code for all keyboards, translates keycodes to ascii codes
 */

#define H_FLAGS_VC_MASK		31
#define MAX_VCS			12
#define PASTE_BUFFER_SIZE	65536
#define PASTE_BURST		128
#define MOUSE_QUEUE_SIZE	32		/* must be power of 2 */
#define DOUBLE_CLICK_TIME	(JIFFIES_PER_SECOND / 2)

#define DEFAULT_AUTOREPEAT_DELAY        (JIFFIES_PER_SECOND / 4)
#define DEFAULT_AUTOREPEAT_RATE		(JIFFIES_PER_SECOND / 30)

/* WARNING: do not leave trace in processor caches or branch target buffer
   about pressed keys ... that's why the following functions are so
   complicated */

#define KEYTABLE_SLOT		(sizeof(unsigned long) * 8)
#define KEYTABLE_MASK		(KEYTABLE_SLOT - 1)
#define KEYTABLE_SLOT_SHIFT	(__BSR_CONST(KEYTABLE_SLOT))

static unsigned long keytable[(KC_MAX + KEYTABLE_MASK) / KEYTABLE_SLOT] = {0};

#define TEST_KEYTABLE_CONST_1(kc)	((keytable[(kc) >> KEYTABLE_SLOT_SHIFT] >> ((kc) & KEYTABLE_MASK)) & 1)
#define TEST_KEYTABLE_CONST(kc)		(keytable[(kc) >> KEYTABLE_SLOT_SHIFT] & (1 << ((kc) & KEYTABLE_MASK)))

static int TEST_SET_KEYTABLE_VAR(unsigned kc, unsigned set)
{
	unsigned i;
	int result = 0;
	for (i = 0; i < sizeof(keytable) / sizeof(*keytable); i++) {
		unsigned this_slot = (((kc >> KEYTABLE_SLOT_SHIFT) ^ i) - 1) >> (sizeof(unsigned) * 8 - 1);
		__u32 data = keytable[i];
		result |= (data >> (kc & KEYTABLE_MASK)) & this_slot;
		this_slot &= ~(set >> 1);
		data &= ~(this_slot << (kc & KEYTABLE_MASK));
		data |= (set & this_slot) << (kc & KEYTABLE_MASK);
		keytable[i] = data;
	}
	return result;
}

static void test_leds(void);
static void kbd_dovcswitch(int vc);
static void kbd_vcrel(int move);
static void kbd_vclast(void);
static void kbd_scroll_halfpages(int dir);
static void kbd_scroll_lines(int dir);
static void kbd_unscroll(void);
static void kbd_display_mouse(void);
static void start_paste_operation(void);
static int max_vcs;
static int last_vc;
static int active_vc;
static int scroll_halfpages;
static int scroll_lines;
static char scrolled;
static char want_mouse;
static char need_erase_clipboard;

static char paste_buffer[PASTE_BUFFER_SIZE];
static int paste_size;
static int paste_position;
static char pasting;

struct keymap_struct {
	unsigned refcount;
	__u8 name[32];
	unsigned n_entries;
	__const__ struct keymap_entry *entries;
};

__const__ struct keymap_struct default_keymap = {
	0, "US", sizeof(keymap), keymap,
};

static unsigned short numpad_ascii;
static __u32 mods[MAX_VCS];
static int forced_leds[MAX_VCS];
struct keymap_struct *keymaps[MAX_VCS];
static char pri_sec[MAX_VCS];


static TIMER autorepeat;
static int autorepeat_key;

static char keyboard_mode[MAX_VCS];
static struct tty_keyboard_params rate[MAX_VCS];
static struct tty_keyboard_params current_rate;
static jiffies_lo_t autorepeat_delay_jiffies;
static jiffies_lo_t autorepeat_rate_jiffies;

static struct mouse_state mouse;
static struct mouse_info mouse_info = { 3, 1, 2 };

static int drag_start_x;
static int drag_start_y;
static char dragging;
static char dbl_clk;
static u_jiffies_t last_click;

static struct vc_mouse_desc {
	int mode;
	struct mouse_state queue[MOUSE_QUEUE_SIZE];
	int qfirst, qlast;
	int signaled;
	int make_new_event;
	WQ wait;
} vc_mouse[MAX_VCS];

static int mouse_accel_thresh = 50;
static int mouse_speed = 100;
static int mouse_scrollback = 7;

static int console_h;

static int flags;

#define KBD_2W				1
#define KBD_NUMLOCK_INIT		2

#if __DEBUG >= 2
static AST heap_check_ast;
static volatile int heap_check_active;
DECL_AST(heap_check, SPL_MALLOC, AST)
{
	heap_check_active = 2;
	KERNEL$HEAP_CHECK("KEYBOARD");
	__critical_printf("HEAP OK\n");
	heap_check_active = 0;
	RETURN;
}
#endif

static __const__ unsigned char ctrl_table[64] = {
	0, 27, 0, 0, 0, 0, 0, 30,
	0, 0, 0, 0, 31, 0, 0, 0,
	17, 23, 5, 18, 20, 25, 21, 9,
	15, 16, 27, 29, 0, 0, 1, 19,
	4, 6, 7, 8, 10, 11, 12, 0,
	0, 0, 0, 28, 26, 24, 3, 22,
	2, 14, 13, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
};

static char *special_key(unsigned keycode, unsigned down)
{
#define DOWN_REPEAT	MAXUINT
	unsigned ct;
	int shift;
	static RANDOM_CTX key_ctx;
#if __DEBUG >= 2
	static int last_sysrq = 0;
#endif
	static unsigned last_keycode = MAXUINT;

	if (!down) {
		KERNEL$ADD_RANDOMNESS(&key_ctx, &keycode, sizeof keycode);
		last_keycode = MAXUINT;
		return "";
	}
	if (__likely(keycode != last_keycode)) {
		KERNEL$ADD_RANDOMNESS(&key_ctx, &keycode, sizeof keycode);
		last_keycode = keycode;
	} else {
		down = DOWN_REPEAT;
	}

#if __DEBUG >= 2
	if (__unlikely(keycode == KC_SYSRQ)) {
		last_sysrq = 1;
		return NULL;
	}
	if (__unlikely(last_sysrq | TEST_KEYTABLE_CONST(KC_SYSRQ))) {
		if (__unlikely(TEST_KEYTABLE_CONST(KC_SHIFT) | TEST_KEYTABLE_CONST(KC_RIGHT_SHIFT) | TEST_KEYTABLE_CONST(KC_CTRL) | TEST_KEYTABLE_CONST(KC_RIGHT_CTRL) | TEST_KEYTABLE_CONST(KC_ALT) | TEST_KEYTABLE_CONST(KC_RIGHT_ALT) | TEST_KEYTABLE_CONST(KC_ESC))) goto no_sysrq;
		last_sysrq = 0;
		if (__likely(keycode == KC_M)) {
			KERNEL$MEMSTAT_DUMP();
		} else if (__likely(keycode == KC_S)) {
			KERNEL$STACK_DUMP();
		} else if (__likely(keycode == KC_W)) {
			WQ_DUMP();
		} else if (__likely(keycode == KC_L)) {
			KERNEL$SLABS_DUMP();
		} else if (__likely(keycode == KC_H)) {
			if (__likely(!heap_check_active)) {
				__critical_printf("CHECKING HEAP...\n");
				heap_check_active = 1;
				heap_check_ast.fn = heap_check;
				CALL_AST(&heap_check_ast);
			} else if (heap_check_active == 1) {
				__critical_printf("HEAP CHECK ALREADY PENDING (STUCK ON SPL_MALLOC?)\n");
			} else {
				__critical_printf("HEAP CHECK ALREADY RUNNING (INFINITE LOOP IN KERNEL$HEAP_CHECK?)\n");
			}
		} else {
			__critical_printf("SYSRQ: M - MEMORY USAGE, S - STACK DUMP, W - WAIT QUEUES, L - SLABS, H - HEAP CHECK\n");
		}
		return NULL;
	}
	no_sysrq:
	last_sysrq = 0;
#endif
	if (__unlikely((ct = __secure_lookup_8(ctrl_table, 64, keycode & 63)) & -(((unsigned)(keycode >> 6) - 1) >> (sizeof(unsigned) * 8 - 1)) & -((TEST_KEYTABLE_CONST_1(KC_CTRL) | TEST_KEYTABLE_CONST_1(KC_RIGHT_CTRL)) ^ (((unsigned)(keycode ^ KC_ESC) - 1) >> (sizeof(unsigned) * 8 - 1))) & ((TEST_KEYTABLE_CONST_1(KC_SHIFT) | TEST_KEYTABLE_CONST_1(KC_RIGHT_SHIFT)) - 1))) {
		static char str[3] = "\e\0\0";
		str[1] = ct;
		return str + 1 - (TEST_KEYTABLE_CONST_1(KC_ALT) | TEST_KEYTABLE_CONST_1(KC_RIGHT_ALT));
	}

	if (__unlikely(keycode == KC_SCROLL_LOCK)) {
		if (__unlikely(down == DOWN_REPEAT) || __unlikely(keyboard_mode[active_vc] != K_XLATE)) return NULL;
		pri_sec[active_vc] ^= 1;
		mods[active_vc] &= ~MOD_NOREINIT;
		return NULL;
	}
	if (__unlikely(console_h == -1)) goto noswitch;
	if (__unlikely(TEST_KEYTABLE_CONST(KC_ALT) | TEST_KEYTABLE_CONST(KC_RIGHT_ALT))) {
		int vc;
		if (__likely(keycode >= KC_F1) && __likely(keycode <= KC_F10)) {
			if (__unlikely(down == DOWN_REPEAT)) return NULL;
			vc = keycode - KC_F1 + 0;
			goto switchvc;
		}
		if (__unlikely(keycode >= KC_F11) && __unlikely(keycode <= KC_F12)) {
			if (__unlikely(down == DOWN_REPEAT)) return NULL;
			vc = keycode - KC_F11 + 10;
			goto switchvc;
		}
		if (__unlikely(keycode == KC_LEFT)) {
			if (__unlikely(down == DOWN_REPEAT)) return NULL;
			kbd_vcrel(-1);
			return NULL;
		}
		if (__unlikely(keycode == KC_RIGHT)) {
			if (__unlikely(down == DOWN_REPEAT)) return NULL;
			kbd_vcrel(1);
			return NULL;
		}
		if (__unlikely(keycode == KC_DOWN)) {
			if (__unlikely(down == DOWN_REPEAT)) return NULL;
			kbd_vclast();
			return NULL;
		}
		goto noswitch;
		switchvc:
		vc += 12 * (TEST_KEYTABLE_CONST_1(KC_SHIFT) | TEST_KEYTABLE_CONST_1(KC_RIGHT_SHIFT) | TEST_KEYTABLE_CONST_1(KC_RIGHT_ALT));
		kbd_dovcswitch(vc);
		return NULL;
	}
	if (__unlikely(keycode == KC_MSLW)) {
		if (__unlikely(down == DOWN_REPEAT)) return NULL;
		if (__unlikely(flags & KBD_2W)) kbd_vcrel(-1);
		else kbd_vclast();
		return NULL;
	}
	if (__unlikely(keycode == KC_MSRW)) {
		if (__unlikely(down == DOWN_REPEAT)) return NULL;
		if (__unlikely(flags & KBD_2W)) kbd_vcrel(1);
		else kbd_vcrel(-1);
		return NULL;
	}
	if (__unlikely(keycode == KC_MSTM)) {
		if (__unlikely(down == DOWN_REPEAT)) return NULL;
		kbd_vcrel(1);
		return NULL;
	}
/* leaking shift is worse than leaking alt ... but it can leak anyway via interrupt timing :-/ */
	shift = TEST_KEYTABLE_CONST_1(KC_SHIFT) | TEST_KEYTABLE_CONST_1(KC_RIGHT_SHIFT);
	if (__unlikely(!((keycode ^ KC_PAGE_UP) | (shift - 1))) && __likely(keyboard_mode[active_vc] == K_XLATE)) {
		kbd_scroll_halfpages(1);
		return NULL;
	}
	if (__unlikely(!((keycode ^ KC_PAGE_DOWN) | (shift - 1))) && __likely(keyboard_mode[active_vc] == K_XLATE)) {
		kbd_scroll_halfpages(-1);
		return NULL;
	}
	noswitch:
	return "";
}

/* don't leak information about pressed key --- that's why this
   function is coded in such a strange way */

#define RET_PAD		3	/* at least 1 */

static __const__ char *interpret_key(unsigned keycode)
{
	__const__ struct keymap_entry *progend;
	__const__ struct keymap_entry *p;
	__const__ struct keymap_struct *desc;
	unsigned long sel;
	unsigned mod = mods[active_vc];
	unsigned quit = 0;
	unsigned quitpass = -1;
	unsigned kc;
	unsigned dont_extract_numpad, prev_mod, numpad_add, dont_change_numpad, ret_nonnull;
	static __u32 ret[((RET_PAD + MAX_KEYMAP_SEQUENCE + 1) + 3) / 4] = { 0 };
	((__u8 *)ret)[RET_PAD - 1] = '\e';
	((__u8 *)ret)[RET_PAD] = 0;
	sel = keyboard_mode[active_vc] - K_XLATE;
	sel = -((sel | -sel) >> (sizeof(unsigned long) * 8 - 1));
	sel |= (unsigned long)pri_sec[active_vc] - 1;
	desc = (__const__ struct keymap_struct *)(((unsigned long)&default_keymap & sel) | ((unsigned long)keymaps[active_vc] & ~sel));
	progend = (struct keymap_entry *)((char *)desc->entries + desc->n_entries);
	for (p = desc->entries; p < progend; p++) {
		unsigned inactive, active, x;
		quitpass |= (((((unsigned)p->seq & SEQ_MASK) ^ quitpass) - 1) >> (sizeof(unsigned) * 8 - 1)) - 1;
		inactive = (~quitpass & (SEQ_MASK * 2 + 1)) | quit | ((mod & p->if_mod_set) ^ p->if_mod_set) | (mod & p->if_mod_reset);
		if (__unlikely(p->if_key & (KEY_IS_DOWN | KEY_IS_UP))) {
			if (__unlikely((p->if_key & KEY_NONEMPTY_OUTPUT) == KEY_NONEMPTY_OUTPUT)) {
				inactive |= ((unsigned)(((__u8 *)ret)[RET_PAD] - 1) >> (sizeof(unsigned) * 8 - 1)) ^ (p->if_key & 1);
			} else {
				unsigned key = p->if_key & KEY_CODE_MASK;
				if (__unlikely(key >= KC_MAX)) inactive = 1;
				else inactive |= TEST_KEYTABLE_CONST_1(key) ^ (((unsigned)p->if_key / KEY_IS_DOWN) & 1);
			}
		} else {
			inactive |= (((keycode ^ p->if_key) & KEY_CODE_MASK) | (((keycode & (unsigned)p->if_key & (KEY_DOWN_NO_UP | KEY_DOWN | KEY_REPEAT | KEY_UP)) - 1) >> (sizeof(unsigned) * 8 - 1))) & ((((unsigned)p->if_key - 1) >> (sizeof(unsigned) * 8 - 1)) - 1);
		}
		active = -((inactive - 1) >> (sizeof(unsigned) * 8 - 1));
		mod |= p->do_mod_set & active;
		mod &= ~(p->do_mod_reset & active);
		quit |= p->seq & SEQ_QUIT & active;
		quitpass &= ((((unsigned)p->seq / SEQ_QUIT_PASS) & 1 & active) - 1) | (p->seq & SEQ_MASK);

#if MAX_KEYMAP_SEQUENCE != 5
	FIX THIS
#endif
		x = *(__u32 *)&p->do_sequence[1];
		active &= -(((x - VAL_NOCHANGE) ^ (VAL_NOCHANGE - x)) >> (sizeof(unsigned) * 8 - 1));
		ret[(RET_PAD + 1) / 4] = (ret[(RET_PAD + 1) / 4] & ~active) | (x & active);
		((__u8 *)ret)[RET_PAD + 0] = (((__u8 *)ret)[RET_PAD + 0] & ~active) | (p->do_sequence[0] & active);
	}
	prev_mod = mods[active_vc];
	mods[active_vc] = mod;

	/* MOD_ALT_NUM was cleared AND empty output AND
		nonzero numpad_ascii AND numpad_ascii < 256
	0: do extract, -1: don't */
	dont_extract_numpad = (((prev_mod & (mod ^ MOD_ALT_NUM)) / MOD_ALT_NUM) & (((unsigned)((__u8 *)ret)[RET_PAD + 0] - 1) >> 8) & (1 - (((unsigned)numpad_ascii - 1) >> (sizeof(unsigned) * 8 - 1))) & ((((unsigned)numpad_ascii >> 8) - 1) >> (sizeof(unsigned) * 8 - 1))) - 1;
	((__u8 *)ret)[RET_PAD + 0] = (((__u8 *)ret)[RET_PAD + 0] & dont_extract_numpad) | (numpad_ascii & ~dont_extract_numpad);
	ret[(RET_PAD + 1) / 4] &= dont_extract_numpad;
	mod &= ~MOD_ADD_ESC | dont_extract_numpad;

	kc = keycode & KEY_CODE_MASK;

/* 0 if ALT-NUMPAD pressed and needs to be processed, -1 otherwise */
	/* don't need to test (mod & MOD_ALT_NUM) now, numpad_ascii will be
	   erased anyway if not valid */
	dont_change_numpad = (((keycode / KEY_DOWN_NO_UP) | (keycode / KEY_DOWN)) & (
		(((kc ^ KC_NUM_DOT) - 1) |
		((kc ^ KC_NUM_0) - 1) |
		((kc ^ KC_NUM_1) - 1) |
		((kc ^ KC_NUM_2) - 1) |
		((kc ^ KC_NUM_3) - 1) |
		((kc ^ KC_NUM_4) - 1) |
		((kc ^ KC_NUM_5) - 1) |
		((kc ^ KC_NUM_6) - 1) |
		((kc ^ KC_NUM_7) - 1) |
		((kc ^ KC_NUM_8) - 1) |
		((kc ^ KC_NUM_9) - 1) |
		((unsigned)((__u8 *)ret)[RET_PAD + 0] - 1)) >> (sizeof(unsigned) * 8 - 1)
	)) - 1;
	numpad_ascii &= ((((kc ^ KC_NUM_DOT) - 1) >> (sizeof(unsigned) * 8 - 1)) - 1) | dont_change_numpad;
	numpad_add = (
		(((kc ^ KC_NUM_0) - 1) & 0x0000) |
		(((kc ^ KC_NUM_1) - 1) & 0x1000) |
		(((kc ^ KC_NUM_2) - 1) & 0x2000) |
		(((kc ^ KC_NUM_3) - 1) & 0x3000) |
		(((kc ^ KC_NUM_4) - 1) & 0x4000) |
		(((kc ^ KC_NUM_5) - 1) & 0x5000) |
		(((kc ^ KC_NUM_6) - 1) & 0x6000) |
		(((kc ^ KC_NUM_7) - 1) & 0x7000) |
		(((kc ^ KC_NUM_8) - 1) & 0x8000) |
		(((kc ^ KC_NUM_9) - 1) & 0x9000)
		) >> 12;
	numpad_ascii = (numpad_ascii & dont_change_numpad) | ((numpad_ascii * 10 + numpad_add) & ~dont_change_numpad);
	numpad_ascii |= (((unsigned)numpad_ascii & 0xf000) >> 4) & ~dont_change_numpad;

/* 1 if returning nonempty string, 0 otherwise */
	ret_nonnull = ((unsigned)((__u8 *)ret)[RET_PAD + 0] + 255) >> 8;
	numpad_ascii &= (ret_nonnull - 1) & -((mod / MOD_ALT_NUM) & 1);
	return (__u8 *)ret + RET_PAD - ((mod / MOD_ADD_ESC) & ret_nonnull);
}

#define KBD_BUFFER_SIZE		4096

struct kbd_buffer {
	char data[KBD_BUFFER_SIZE];
	int start;
	int ptr;
	WQ blocked;
};

static struct kbd_buffer data_buffer[MAX_VCS];
static struct kbd_buffer halfraw_buffer[MAX_VCS];
static struct kbd_buffer raw_buffer[MAX_VCS];

static void put_buffer(struct kbd_buffer *buffer, __const__ char *str, unsigned len)
{
	int p = buffer->ptr;
	while (len--) {
		buffer->data[p++] = *str++;
		if ((p &= KBD_BUFFER_SIZE - 1) == buffer->start) return;
	}
	buffer->ptr = p;
	WQ_WAKE_ALL_PL(&buffer->blocked);
}

static void SET_REPEAT(jiffies_lo_t tim);

/*
 * down: 0 - up
 *	 1 - down
 *	 2 - down & keyboard won't send up code
 *	 3 - down & software autorepeat
 */

static void handle_keycode(unsigned keycode, unsigned down)
{
	int pressed;
	char c[2];
	unsigned len = 0;
	__const__ char *str;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_KEYBOARD), spl)))
		KERNEL$SUICIDE("handle_keycode AT SPL %08X", spl);
	RAISE_SPL(SPL_KEYBOARD);
	if (__unlikely(current_rate.soft) && down == KBD_KEY_DOWN) down = KBD_KEY_DOWN_SOFT_REPEAT;
	if (__unlikely((unsigned)keycode >= KC_MAX)) goto ret;
	if (__likely(down != KBD_KEY_DOWN_NO_UP)) {
		c[0] = (keycode | 0x80) ^ (down << 7);
		len = 1;
		if (down == KBD_KEY_DOWN_SOFT_REPEAT) {
			if (__unlikely(TEST_SET_KEYTABLE_VAR(keycode, 2))) goto ret;
			autorepeat_key = keycode;
			KERNEL$DEL_TIMER(&autorepeat);
			SET_REPEAT(autorepeat_delay_jiffies);
		} else if (!down) {
			len = TEST_SET_KEYTABLE_VAR(keycode, 2);
			if (__likely(keycode == autorepeat_key)) {
				KERNEL$DEL_TIMER(&autorepeat);
				VOID_LIST_ENTRY(&autorepeat.list);
			}
		}
	} else {
		c[0] = keycode;
		c[1] = keycode | 0x80;
		len = 2;
	}
	str = special_key(keycode, down);
	if (__unlikely(!str)) goto special_processed;
	put_buffer(&halfraw_buffer[active_vc], c, len);
	pressed = TEST_SET_KEYTABLE_VAR(keycode, down & 1);
	if (down) {
		if (__unlikely(pressed)) keycode |= KEY_REPEAT;
		else keycode |= KEY_DOWN_NO_UP + (down & 1) * (KEY_DOWN - KEY_DOWN_NO_UP);
	} else {
		if (__unlikely(!pressed)) goto ret;
		keycode |= KEY_UP;
	}
	if (__likely(!*str)) {
		str = interpret_key(keycode);
	}
	if (*str) {
		kbd_unscroll();
		put_buffer(&data_buffer[active_vc], str, strlen(str));
	} else if (__unlikely(keyboard_mode[active_vc] != K_XLATE)) {
		kbd_unscroll();
	}
	special_processed:
	test_leds();
	ret:
	LOWER_SPLX(spl);
}

static void autorepeat_timer(TIMER *t)
{
	char soft;
	LOWER_SPL(SPL_KEYBOARD);
	SET_REPEAT(autorepeat_rate_jiffies);
	soft = current_rate.soft;
	current_rate.soft = 0;
	handle_keycode(autorepeat_key, KBD_KEY_DOWN);
	current_rate.soft = soft;
}

static void SET_REPEAT(jiffies_lo_t tim)
{
	if (__unlikely(tim < 0)) {
		VOID_LIST_ENTRY(&autorepeat.list);
		return;
	}
	KERNEL$SET_TIMER((u_jiffies_lo_t)tim, &autorepeat);
}

static jiffies_lo_t calculate_jiffies(int ms, jiffies_lo_t deflt);
static void set_soft_autorepeat(void);

static void test_autorepeat(void)
{
	if (__unlikely(memcmp(&current_rate, &rate[active_vc], sizeof(struct tty_keyboard_params)))) {
		hw_kbd_set_autorepeat(&rate[active_vc]);
		memcpy(&current_rate, &rate[active_vc], sizeof(struct tty_keyboard_params));
		set_soft_autorepeat();
	}
}

static void set_soft_autorepeat(void)
{
	autorepeat_delay_jiffies = calculate_jiffies(current_rate.repeat_delay, DEFAULT_AUTOREPEAT_DELAY);
	autorepeat_rate_jiffies = calculate_jiffies(current_rate.repeat_rate, DEFAULT_AUTOREPEAT_RATE);
}

static jiffies_lo_t calculate_jiffies(int ms, jiffies_lo_t deflt)
{
	jiffies_lo_t tim;
	if (ms < 0) return -1;
	if (__likely(!ms)) return deflt;
	tim = ((__u64)(unsigned)ms * JIFFIES_PER_SECOND + 500) / 1000;
	if (__likely(tim)) tim--;
	return tim;
}

static __finline__ int get_leds_vc(int vc)
{
	int leds = (mods[vc] & MOD_CAPS_LOCK) / MOD_CAPS_LOCK * LED_CAPS_LOCK + (mods[vc] & MOD_NUM_LOCK) / MOD_NUM_LOCK * LED_NUM_LOCK + pri_sec[vc] * LED_SCROLL_LOCK;
	int fl = forced_leds[vc];
	leds = (leds & -((unsigned)fl >> (sizeof(unsigned) * 8 - 1))) | (fl & (((unsigned)fl >> (sizeof(unsigned) * 8 - 1)) - 1));
	return leds;
}

static void test_leds(void)
{
	static int leds_cache = -1;
	int leds = get_leds_vc(active_vc);
	if (__unlikely(leds != leds_cache)) {
		leds_cache = leds;
		hw_kbd_set_leds(leds);
	}
}

static void kbd_send_op(void);

static void kbd_dovcswitch(int vc)
{
	char *ptr;
	unsigned val, st;
	char keystr[KC_MAX + 1];
	if (__unlikely(console_h == -1)) return;
	if (__unlikely(vc >= max_vcs)) return;
	if (__unlikely(vc == active_vc)) return;

	ptr = keystr;
	do {
		unsigned i, cnt;
		st = 0;
		val = 0;
		cnt = ptr - keystr;
		for (i = 0; i < KC_MAX; i++) {
			unsigned ncnt = cnt - TEST_KEYTABLE_CONST_1(i);
			unsigned sel = -((cnt ^ ncnt) >> (sizeof(unsigned) * 8 - 1));
			cnt = ncnt;
			val |= i & sel;
			st |= sel;
		}
		*ptr++ = val;
	} while (st);
	put_buffer(&halfraw_buffer[vc], keystr, ptr - keystr);
	for (ptr = keystr; *ptr; ptr++) *ptr ^= 0x80;
	put_buffer(&halfraw_buffer[active_vc], keystr, ptr - keystr);

	last_vc = active_vc;
	active_vc = vc;
	test_autorepeat();
	test_leds();
	scroll_halfpages = 0;
	scroll_lines = 0;
	scrolled = 1;		/* new console may already be scrolled */
	interpret_key(0);	/* fix flags for SHIFT, CTRL, ALT & possibly others */
	kbd_send_op();
}

static void kbd_vcrel(int move)
{
	int vc = active_vc + move;
	if (__unlikely(vc < 0)) vc += max_vcs;
	else if (__unlikely(vc >= max_vcs)) vc -= max_vcs;
	kbd_dovcswitch(vc);
}

static void kbd_vclast(void)
{
	kbd_dovcswitch(last_vc);
}

static void kbd_scroll_halfpages(int dir)
{
	if (dir > 0) scrolled = 1;
	scroll_halfpages += dir;
	kbd_send_op();
}

static void kbd_scroll_lines(int dir)
{
	if (dir > 0) scrolled = 1;
	scroll_lines += dir;
	kbd_send_op();
}

static void kbd_unscroll(void)
{
	if (__unlikely(scrolled)) {
		scrolled = 0;
		scroll_halfpages = -1024;
		scroll_lines = 0;
		kbd_send_op();
	}
}

static void kbd_display_mouse(void)
{
	if (__likely(!want_mouse)) want_mouse = 1;
	kbd_send_op();
}

static int pending = 0;
static IOCTLRQ io;
static struct console_mouse_request cmr;
static struct scroll_request sr;

DECL_AST(RET, SPL_KEYBOARD, IOCTLRQ)
{
	pending = 0;
	if (__likely(RQ->ioctl == IOCTL_CONSOLE_MOUSE) && __likely(RQ->status > 0)) {
		if (__unlikely(mouse.x < cmr.min_x)) mouse.x = cmr.min_x;
		if (__unlikely(mouse.x >= cmr.max_x)) mouse.x = cmr.max_x - 1;
		if (__unlikely(mouse.y < cmr.min_y)) mouse.y = cmr.min_y;
		if (__unlikely(mouse.y >= cmr.max_y)) mouse.y = cmr.max_y - 1;
	}
	if (__unlikely(RQ->ioctl == IOCTL_CONSOLE_GET_CLIPBOARD)) {
		if (RQ->status >= 0) {
			paste_size = RQ->status;
			paste_position = 0;
			start_paste_operation();
		} else pasting = 0;
	}
	kbd_send_op();
	RETURN;
}

static void kbd_send_op(void)
{
	static int cons_vc = -1;
	if (__unlikely(pending)) return;
	if (__unlikely(console_h == -1)) return;
	io.fn = RET;
	io.h = console_h;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	if (need_erase_clipboard) {
		io.ioctl = IOCTL_CONSOLE_ERASE_CLIPBOARD;
		CALL_IORQ(&io, KERNEL$IOCTL);
		pending = 1;
		need_erase_clipboard = 0;
		return;
	}
	if (active_vc != cons_vc) {
		io.ioctl = IOCTL_CONSOLE_SWITCH;
		io.param = active_vc;
		CALL_IORQ(&io, KERNEL$IOCTL);
		pending = 1;
		cons_vc = active_vc;
		return;
	}
	if (scroll_halfpages) {
		sr.flags = SCROLL_HALFPAGES;
		sr.n = scroll_halfpages;
		scroll_halfpages = 0;
		do_scroll:
		if (dragging) want_mouse = 2, sr.flags |= SCROLL_DRAGGING;
		io.ioctl = IOCTL_CONSOLE_SCROLL;
		io.v.ptr = (unsigned long)&sr;
		io.v.len = sizeof sr;
		CALL_IORQ(&io, KERNEL$IOCTL);
		pending = 1;
		return;
	}
	if (scroll_lines) {
		sr.flags = 0;
		sr.n = scroll_lines;
		scroll_lines = 0;
		goto do_scroll;
	}
	if (want_mouse) {
		io.ioctl = IOCTL_CONSOLE_MOUSE;
		io.v.ptr = (unsigned long)&cmr;
		io.v.len = sizeof cmr;
		cmr.x = mouse.x;
		cmr.y = mouse.y;
		cmr.start_x = drag_start_x;
		cmr.start_y = drag_start_y;
		cmr.drag = dragging;
		cmr.update = want_mouse > 1;
		CALL_IORQ(&io, KERNEL$IOCTL);
		pending = 1;
		want_mouse = 0;
		return;
	}
	if (pasting == 1) {
		io.ioctl = IOCTL_CONSOLE_GET_CLIPBOARD;
		io.param = 0;
		io.v.ptr = (unsigned long)paste_buffer;
		io.v.len = sizeof paste_buffer;
		CALL_IORQ(&io, KERNEL$IOCTL);
		pending = 1;
		pasting = 2;
		return;
	}
}

static TIMER paste_timer;

static void paste(void)
{
	if (__unlikely(pasting)) return;
	pasting = 1;
	kbd_send_op();
}

static void start_paste_operation(void)
{
	unsigned l;
	if (paste_position >= paste_size) {
		pasting = 0;
		return;
	}
	l = paste_size - paste_position;
	if (l >= PASTE_BURST) {
		int i;
		l = PASTE_BURST;
		for (i = PASTE_BURST - 1; i >= PASTE_BURST / 2; i--) {
			if (paste_buffer[paste_position + i] == '\n') {
				l = i + 1;
				break;
			}
		}
	}
	put_buffer(&data_buffer[active_vc], paste_buffer + paste_position, l);
	paste_position += l;
	KERNEL$SET_TIMER(1, &paste_timer);
}

static void paste_next(TIMER *t)
{
	start_paste_operation();
}

static __const__ HANDLE_OPERATIONS keyboard_operations;

DECL_IOCALL(keyboard_read, SPL_TTY, SIORQ)
{
	long s;
	int vc;
	if (__unlikely(RQ->handle->op != &keyboard_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(RQ->handle->name_addrspace, SPL_X(SPL_TTY));
	RAISE_SPL(SPL_KEYBOARD);
	vc = RQ->handle->flags & H_FLAGS_VC_MASK;
	if (__unlikely(!RQ->v.len)) goto ret;
	if (data_buffer[vc].start != data_buffer[vc].ptr) {
		while (RQ->v.len && data_buffer[vc].start != data_buffer[vc].ptr) {
			VBUF vbuf;
			s = RQ->v.len;
			if (data_buffer[vc].start <= data_buffer[vc].ptr && data_buffer[vc].start + s > data_buffer[vc].ptr) s = data_buffer[vc].ptr - data_buffer[vc].start;
			else if (data_buffer[vc].start + s > KBD_BUFFER_SIZE) s = KBD_BUFFER_SIZE - data_buffer[vc].start;
			vbuf.ptr = data_buffer[vc].data + data_buffer[vc].start;
			vbuf.len = s;
			vbuf.spl = SPL_X(SPL_TTY);
			if (SPLX_BELOW(SPL_X(SPL_VSPACE), SPL_X(SPL_KEYBOARD))) LOWER_SPL(SPL_VSPACE);
			else if (SPLX_BELOW(SPL_X(SPL_KEYBOARD), SPL_X(SPL_VSPACE))) RAISE_SPL(SPL_VSPACE);
			if (__unlikely(!(s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf)))) goto pf;
			RAISE_SPL(SPL_KEYBOARD);
			RQ->progress += s;
			data_buffer[vc].start = (data_buffer[vc].start + s) & (KBD_BUFFER_SIZE - 1);
			LOWER_SPL(SPL_TTY);
			SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(RQ->handle->name_addrspace, SPL_X(SPL_TTY));
			TEST_LOCKUP_LOOP(RQ, RETURN);
			RAISE_SPL(SPL_KEYBOARD);
		}
		ret:
		if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
		else RQ->status = -EOVERFLOW;
		RETURN_AST(RQ);
	}

	if (RQ->progress) goto ret;
	WQ_WAIT_F(&data_buffer[vc].blocked, RQ);
	RETURN;

	pf:
	DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
}

DECL_IOCALL(keyboard_ioctl, SPL_TTY, IOCTLRQ)
{
	if (__unlikely(RQ->handle->op != &keyboard_operations))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(RQ->handle->name_addrspace, SPL_X(SPL_TTY));
	switch (RQ->ioctl) {
		case IOCTL_CONSOLE_VC: {
			RQ->status = RQ->handle->flags & H_FLAGS_VC_MASK;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}


static void kbd_detach(HANDLE *h)
{
	int vc = h->flags & H_FLAGS_VC_MASK;
	WQ_WAKE_ALL(&data_buffer[vc].blocked);
}

static void *kbd_lookup(HANDLE *h, char *str, int open_flags)
{
	char *e;
	e = strchr(str, '=');
	if (__unlikely(!e)) return __ERR_PTR(-EBADMOD);
	if (__likely(!__strcasexcmp("^VC", str, e))) {
		long vc;
		if (__unlikely(__get_number(e + 1, e + strlen(e), 0, &vc)) || __unlikely((unsigned long)--vc >= max_vcs)) return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~H_FLAGS_VC_MASK) | vc;
	} else return __ERR_PTR(-EBADMOD);
	return NULL;
}

static __const__ HANDLE_OPERATIONS keyboard_operations = {
	SPL_X(SPL_TTY),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,	/* clone -- default */
	kbd_lookup,	/* lookup -- none */
	NULL,	/* create -- none */
	NULL,	/* delete -- none */
	NULL,	/* rename -- none */
	NULL,	/* lookup io -- none */
	NULL,	/* instantiate */
	NULL,	/* leave */
	kbd_detach,	/* detach */
	NULL,	/* open */
	NULL,	/* close */
	keyboard_read,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
	keyboard_ioctl,

	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
};

static void keyboard_init_root(HANDLE *h, void *dummy)
{
	h->op = &keyboard_operations;
	h->flags = 0;
}

/* warning: in case of HW failure, shutdown won't be called */
static int kbd_init(int argc, char *argv[])
{
	char **arg;
	int state;
	char *opt, *optend, *val;
	int i;
	char *scr = NULL;
	struct __param_table params[] = {
		"SCR", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"2W", __PARAM_BOOL, KBD_2W, KBD_2W, NULL,
		"NUMLOCK_ON", __PARAM_BOOL, KBD_NUMLOCK_INIT, KBD_NUMLOCK_INIT, NULL,
		"NUMLOCK_OFF", __PARAM_BOOL, KBD_NUMLOCK_INIT, 0, NULL,
#ifdef HAVE_PSAUX
		"NOPS2MOUSE", __PARAM_BOOL, ~0, 0, NULL,
#endif
		NULL, 0, 0, 0, NULL,
	};
	for (i = 0; i < sizeof(keymap) / sizeof(*keymap) - 1; i++)
		if ((keymap[i].seq & SEQ_MASK) > (keymap[i + 1].seq & SEQ_MASK))
			KERNEL$SUICIDE("kbd_init: NON-MONOTONIC KEYMAP SEQUENCE NUMBERS AT OFFSET %d", i);
	flags = 0;
	if (hw_kbd_get_default_leds() & LED_NUM_LOCK) flags |= KBD_NUMLOCK_INIT;
	params[0].__p = &scr;
	params[1].__p = &flags;
	params[2].__p = &flags;
	params[3].__p = &flags;
#ifdef HAVE_PSAUX
	params[4].__p = &init_psaux;
#endif
	arg = argv;
	state = 0;
	while (__unlikely(__parse_params(&arg, &state, params, &opt, &optend, &val))) {
		if (hwkbd_parse_flag(opt, optend, val)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, KBD_NAME": SYNTAX ERROR");
			return -EBADSYN;
		}
	}
	console_h = -1;
	max_vcs = 1;
	if (scr) {
		union {
			OPENRQ o;
			IOCTLRQ io;
		} u;
		u.o.flags = O_WRONLY;
		u.o.path = scr;
		u.o.cwd = KERNEL$CWD();
		SYNC_IO_CANCELABLE(&u.o, KERNEL$OPEN);
		if (u.o.status < 0) {
			if (u.o.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, KBD_NAME": UNABLE TO OPEN SCREEN %s: %s", scr, strerror(-u.o.status));
			return u.o.status;
		}
		console_h = u.o.status;
		u.io.h = console_h;
		u.io.ioctl = IOCTL_CONSOLE_NUM_VCS;
		u.io.param = 0;
		u.io.v.ptr = 0;
		u.io.v.len = 0;
		u.io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&u.io, KERNEL$IOCTL);
		if (u.io.status > 0) max_vcs = u.io.status;
		if (max_vcs > MAX_VCS) max_vcs = MAX_VCS;
	}
	memset(data_buffer, 0, sizeof data_buffer);
	memset(halfraw_buffer, 0, sizeof halfraw_buffer);
	memset(raw_buffer, 0, sizeof raw_buffer);
	for (i = 0; i < MAX_VCS; i++) {
		keyboard_mode[i] = K_XLATE;
		WQ_INIT(&data_buffer[i].blocked, "KBD$DATA_BUFFER_BLOCKED");
		WQ_INIT(&halfraw_buffer[i].blocked, "KBD$HALFRAW_BUFFER_BLOCKED");
		WQ_INIT(&raw_buffer[i].blocked, "KBD$RAW_BUFFER_BLOCKED");
	}
	active_vc = 0;
	last_vc = 0;
	scroll_halfpages = 0;
	scroll_lines = 0;
	scrolled = 0;
	want_mouse = 0;
	need_erase_clipboard = 0;
	pasting = 0;
	for (i = 0; i < MAX_VCS; i++) {
		mods[i] = flags & KBD_NUMLOCK_INIT ? MOD_NUM_LOCK : 0;
		forced_leds[i] = -1;
		keymaps[i] = (struct keymap_struct *)&default_keymap;
		pri_sec[i] = 0;
	}
	INIT_TIMER(&autorepeat);
	VOID_LIST_ENTRY(&autorepeat.list);
	memset(&rate, 0, sizeof rate);
	memset(&current_rate, 0, sizeof current_rate);
	set_soft_autorepeat();
	autorepeat.fn = autorepeat_timer;
	memset(&mouse, 0, sizeof mouse);
	mouse.x = (80 / 2) * 8;
	mouse.y = 80 * 8 * 3 / 4 / 25 * 25 / 2;
	dragging = 0;
	dbl_clk = 0;
	last_click = 0;
	memset(&vc_mouse, 0, sizeof vc_mouse);
	for (i = 0; i < MAX_VCS; i++) WQ_INIT(&vc_mouse[i].wait, "KBD$VC_MOUSE_WAIT");
	INIT_TIMER(&paste_timer);
	paste_timer.fn = paste_next;
	return 0;
}

static void kbd_do_shutdown(void)
{
	int i;
	while (pending | pasting) KERNEL$SLEEP(1);
	KERNEL$DEL_TIMER(&autorepeat);
	for (i = 0; i < MAX_VCS; i++) {
		WQ_WAKE_ALL(&data_buffer[i].blocked);
		WQ_WAKE_ALL(&halfraw_buffer[i].blocked);
		WQ_WAKE_ALL(&raw_buffer[i].blocked);
		WQ_WAKE_ALL(&vc_mouse[i].wait);
	}
}

static int COMPARE_KEYMAPS(struct keymap_struct *k1, struct keymap_struct *k2);
static void UNREF_KEYMAP(struct keymap_struct *k);

static WQ *SET_KEYMAP(int vc, PROC *proc, FBLOB *blob_addr, size_t blob_size, IORQ *rq)
{
	vspace_unmap_t *unmap;
	VDESC vdesc;
	struct keymap_blob *keymap_blob;
	struct keymap_struct *keymap, *prev;
	unsigned i;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_TTY)))
		KERNEL$SUICIDE("SET_KEYMAP AT SPL %08X", KERNEL$SPL);
	if (!proc) {
		do_default:
		RAISE_SPL(SPL_KEYBOARD);
		if (keymaps[vc] != &default_keymap) mods[active_vc] &= ~MOD_NOREINIT;
		prev = keymaps[vc];
		keymaps[vc] = (struct keymap_struct *)&default_keymap;
		lower_unref_prev:
		LOWER_SPL(SPL_TTY);
		unref_prev:
		UNREF_KEYMAP(prev);
		return NULL;
	}
	if (__unlikely(blob_size < sizeof(struct keymap_blob))) {
		if (__likely(blob_size == sizeof(FBLOB))) goto do_default;
		return __ERR_PTR(-EINVAL);
	}
	vdesc.ptr = (unsigned long)blob_addr;
	vdesc.len = blob_size;
	vdesc.vspace = KERNEL$PROC_VSPACE(proc);
	RAISE_SPL(SPL_VSPACE);
	keymap_blob = vdesc.vspace->op->vspace_map(&vdesc, PF_READ, &unmap);
	LOWER_SPL(SPL_TTY);
	if (__unlikely(!keymap_blob)) {
		DO_PAGEIN_NORET(rq, &vdesc, PF_READ);
		return (void *)1;
	}
	if (__unlikely(__IS_ERR(keymap_blob))) {
		return (void *)keymap_blob;
	}
	keymap = malloc(sizeof(struct keymap_struct) + blob_size - sizeof(struct keymap_blob));
	if (__unlikely(!keymap)) {
		RAISE_SPL(SPL_VSPACE);
		unmap(keymap_blob);
		LOWER_SPL(SPL_TTY);
		KERNEL$MEMWAIT(rq, (IO_STUB *)rq->tmp1, sizeof(struct keymap_struct) + blob_size);
		return (void *)1;
	}
	keymap->refcount = 1;
	keymap->n_entries = blob_size - sizeof(struct keymap_blob);
	keymap->entries = (struct keymap_entry *)(keymap + 1);
	strlcpy(keymap->name, keymap_blob->keymap, sizeof(keymap->name) < sizeof(keymap_blob->keymap) ? sizeof(keymap->name) : sizeof(keymap_blob->keymap));
	memcpy((struct keymap_entry *)keymap->entries, keymap_blob + 1, keymap->n_entries);
	RAISE_SPL(SPL_VSPACE);
	unmap(keymap_blob);
	LOWER_SPL(SPL_TTY);
	RAISE_SPL(SPL_KEYBOARD);
	if (!COMPARE_KEYMAPS(keymaps[vc], keymap)) {
		LOWER_SPL(SPL_TTY);
		free(keymap);
		return NULL;
	}
	mods[active_vc] &= ~MOD_NOREINIT;
	for (i = 0; i < MAX_VCS; i++) if (__likely(i != vc)) {
		if (COMPARE_KEYMAPS(keymaps[i], keymap)) continue;
		keymaps[i]->refcount++;
		prev = keymaps[vc];
		keymaps[vc] = keymaps[i];
		LOWER_SPL(SPL_TTY);
		free(keymap);
		goto unref_prev;
	}
	prev = keymaps[vc];
	keymaps[vc] = keymap;
	goto lower_unref_prev;
}

static int COMPARE_KEYMAPS(struct keymap_struct *k1, struct keymap_struct *k2)
{
	if (k1->n_entries != k2->n_entries) return 1;
	if (__unlikely(strcmp(k1->name, k2->name))) return 1;
	return memcmp(k1->entries, k2->entries, k1->n_entries);
}

static void UNREF_KEYMAP(struct keymap_struct *k)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_TTY)))
		KERNEL$SUICIDE("UNREF_KEYMAP AT SPL %08X", KERNEL$SPL);
	if (k == &default_keymap) return;
	if (__unlikely(!k->refcount)) KERNEL$SUICIDE("UNREF_KEYMAP: UNREFERENCING KEYMAP WITH NO REFERENCES");
	if (!--k->refcount) free(k);
}

static char locked[MAX_VCS] = { 0 };
static int mouse_lock_count = 0;
static int keyboard_lock_count = 0;
static int kbd_shutting_down = 0;

static long kbdmode_call(int cmd, int kc, ...);
static void kbd_mouse_event(struct mouse_state *m);

static int kbd_raw_dcall(void *ptr, __const__ char *dcall_type, int cmd, void *param)
{
	int r;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_KEYBOARD), spl)))
		KERNEL$SUICIDE("kbd_raw_call AT SPL %08X", spl);
	RAISE_SPL(SPL_KEYBOARD);
	if (cmd >= KEYBOARD_CMD_GET && cmd < KEYBOARD_CMD_GET + max_vcs) {
		if (kbd_shutting_down) return -EBUSY;
		if (locked[cmd - KEYBOARD_CMD_GET]) return -EBUSY;
		locked[cmd - KEYBOARD_CMD_GET] = 0;
		*(kbdmode_call_t **)param = kbdmode_call;
		r = 0;
	} else if (cmd == KEYBOARD_MOUSE_GET) {
		if (kbd_shutting_down) return -EBUSY;
		*(kbdmouse_call_t **)param = kbd_mouse_event;
		mouse_lock_count++;
		r = 0;
	} else if (cmd == KEYBOARD_MOUSE_PUT) {
		if (!mouse_lock_count) KERNEL$SUICIDE("kbd_raw_dcall: UNLOCKING UNLOCKED MOUSE");
		mouse_lock_count--;
		r = 0;
	} else if (cmd == KEYBOARD_EXTKBD_GET) {
		if (kbd_shutting_down) return -EBUSY;
		*(kbdext_call_t **)param = handle_keycode;
		keyboard_lock_count++;
		r = 0;
	} else if (cmd == KEYBOARD_EXTKBD_PUT) {
		if (!keyboard_lock_count) KERNEL$SUICIDE("kbd_raw_dcall: UNLOCKING UNLOCKED KEYBOARD");
		keyboard_lock_count--;
		r = 0;
	} else r = -EINVAL;
	LOWER_SPLX(spl);
	return r;
}

static long kbdmode_call(int cmd, int kc, ...)
{
	long r;
	char *buf;
	int size, mode;
	struct kbd_buffer *buffer;
	struct mouse_state *m;
	struct mouse_info *mi;
	struct vc_mouse_desc *vcm;
	struct tty_keyboard_params *k;
	IORQ *rq;
	va_list args;
	PROC *proc;
	FBLOB *blob_addr;
	size_t blob_size;
	unsigned slock;
	int leds;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_KEYBOARD), spl)))
		KERNEL$SUICIDE("kbdmode_call AT SPL %08X", spl);
	RAISE_SPL(SPL_KEYBOARD);
	switch (cmd) {
		case KBD_RELEASE:
			locked[kc] = 0;
			r = 0;
			break;
		case KBD_READ:
			va_start(args, kc);
			buf = va_arg(args, char *);
			size = va_arg(args, int);
			mode = va_arg(args, int);
			va_end(args);
			if (__likely(mode == K_MEDIUMRAW)) buffer = &halfraw_buffer[kc];
			else if (__likely(mode == K_RAW)) buffer = &raw_buffer[kc];
			else {
				r = -ENXIO;
				break;
			}
			if (buffer->start <= buffer->ptr && buffer->start + size > buffer->ptr) size = buffer->ptr - buffer->start;
			else if (buffer->start + size > KBD_BUFFER_SIZE) size = KBD_BUFFER_SIZE - buffer->start;
			memcpy(buf, buffer->data + buffer->start, size);
			buffer->start = (buffer->start + size) & (KBD_BUFFER_SIZE - 1);
			r = size;
			break;
		case KBD_WAIT:
			va_start(args, kc);
			mode = va_arg(args, int);
			va_end(args);
				/* this must not lower SPL */
			if (__likely(mode == K_MEDIUMRAW)) return (long)&halfraw_buffer[kc].blocked;
			else if (__likely(mode == K_RAW)) return (long)&raw_buffer[kc].blocked;
			else return (long)&data_buffer[kc].blocked;
		case KBD_CLEAR_BUFFER:
			va_start(args, kc);
			mode = va_arg(args, int);
			va_end(args);
			if (__likely(mode == K_MEDIUMRAW)) buffer = &halfraw_buffer[kc];
			else if (__likely(mode == K_RAW)) buffer = &raw_buffer[kc];
			else {
				r = -ENXIO;
				break;
			}
			buffer->start = buffer->ptr = 0;
			r = 0;
			break;
		case KBD_SET_MODE:
			va_start(args, kc);
			mode = va_arg(args, int);
			va_end(args);
			keyboard_mode[kc] = mode;
			r = 0;
			break;
		case KBD_SET_MOUSE_MODE:
			va_start(args, kc);
			mode = va_arg(args, int);
			va_end(args);
			vc_mouse[kc].mode = mode;
				/* fallthrough */
		case KBD_CLEAR_MOUSE_QUEUE:
			vc_mouse[kc].qfirst = vc_mouse[kc].qlast;
			vc_mouse[kc].signaled = 0;
			vc_mouse[kc].make_new_event = 0;
			r = 0;
			break;
		case KBD_GET_MOUSE_STATE:
			va_start(args, kc);
			m = va_arg(args, struct mouse_state *);
			va_end(args);
			vcm = &vc_mouse[kc];
			memcpy(m, &vcm->queue[vcm->qfirst], sizeof(struct mouse_state));
			r = 0;
			break;
		case KBD_PULL_MOUSE_STATE:
			vcm = &vc_mouse[kc];
			if (vcm->qfirst != vcm->qlast) {
				vcm->qfirst = (vcm->qfirst + 1) & (MOUSE_QUEUE_SIZE - 1);
			} else {
				vcm->make_new_event = 0;
				vcm->signaled = 0;
			}
			r = 0;
			break;
		case KBD_WAIT_MOUSE:
			va_start(args, kc);
			rq = va_arg(args, IORQ *);
			mode = va_arg(args, int);
			va_end(args);
			if (vc_mouse[kc].signaled) {
				r = 0;
			} else {
				if (mode) {
					WQ_WAIT_F(&vc_mouse[kc].wait, rq);
					r = 1;
				} else {
					r = -EWOULDBLOCK;
				}
			}
			break;
		case KBD_GET_MOUSE_INFO:
			va_start(args, kc);
			mi = va_arg(args, struct mouse_info *);
			va_end(args);
			memcpy(mi, &mouse_info, sizeof(struct mouse_info));
			r = 0;
			break;
		case KBD_ERASE_CLIPBOARD:
			need_erase_clipboard = 1;
			kbd_send_op();
			r = 0;
			break;
		case KBD_SET_KEYBOARD_RATE:
			va_start(args, kc);
			k = va_arg(args, struct tty_keyboard_params *);
			va_end(args);
			memcpy(&rate[kc], k, sizeof(struct tty_keyboard_params));
			test_autorepeat();
			r = 0;
			break;
		case KBD_SET_KEYMAP:
			va_start(args, kc);
			proc = va_arg(args, PROC *);
			blob_addr = va_arg(args, FBLOB *);
			blob_size = va_arg(args, size_t);
			rq = va_arg(args, IORQ *);
			va_end(args);
			if (__unlikely(SPLX_BELOW(SPL_X(SPL_TTY), spl)))
				KERNEL$SUICIDE("kbdmode_call KBD_SET_KEYMAP AT SPL %08X", spl);
			LOWER_SPL(SPL_TTY);
			r = (long)SET_KEYMAP(kc, proc, blob_addr, blob_size, rq);
			break;
		case KBD_SWITCH_KEYBOARD:
			va_start(args, kc);
			slock = va_arg(args, unsigned);
			va_end(args);
			if (__unlikely(slock >= 2)) {
				r = -EINVAL;
				break;
			}
			pri_sec[kc] = slock;
			test_leds();
			r = 0;
			break;
		case KBD_SET_LEDS:
			va_start(args, kc);
			leds = va_arg(args, int);
			va_end(args);
			forced_leds[kc] = leds;
			test_leds();
			r = 0;
			break;
		case KBD_GET_LEDS:
			r = get_leds_vc(kc);
			break;
		default:
			r = -EINVAL;
			break;
	}
	LOWER_SPLX(spl);
	return r;
}

static void kbd_mouse_event(struct mouse_state *m)
{
	struct vc_mouse_desc *vcm;
	int x, y, z, ax, ay, az, accel_factor;
	int send_ev = 1;
	int spl;
	static RANDOM_CTX mouse_ctx;
	KERNEL$ADD_RANDOMNESS(&mouse_ctx, m, sizeof(struct mouse_state));
	spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_KEYBOARD), spl)))
		KERNEL$SUICIDE("kbd_mouse_event AT SPL %08X", spl);
	RAISE_SPL(SPL_KEYBOARD);
	mouse_info.buttons = m->x;
	mouse_info.wheels = m->y;
	mouse_info.axes = m->z;
	mouse.rx += (unsigned)m->rx;
	mouse.ry += (unsigned)m->ry;
	mouse.rz += (unsigned)m->rz;
	mouse.wx += (unsigned)m->wx;
	mouse.wy += (unsigned)m->wy;
	mouse.wz += (unsigned)m->wz;
	x = m->rx;
	y = m->ry;
	z = m->rz;
	if (x > 0) x = (x - 1) * mouse_speed + 100, ax = x;
	else if (x < 0) x = (x + 1) * mouse_speed - 100, ax = -x;
	else ax = 0;
	if (y > 0) y = (y - 1) * mouse_speed + 100, ay = y;
	else if (y < 0) y = (y + 1) * mouse_speed - 100, ay = -y;
	else ay = 0;
	if (z > 0) z = (z - 1) * mouse_speed + 100, az = z;
	else if (z < 0) z = (z + 1) * mouse_speed - 100, az = -z;
	else az = 0;
	if (__likely(mouse_accel_thresh != 0)) accel_factor = 100 + (ax + ay + az) / mouse_accel_thresh;
	else accel_factor = 100;
	x = x * accel_factor / 10000;
	y = y * accel_factor / 10000;
	z = z * accel_factor / 10000;
	mouse.x += (unsigned)x;
	mouse.y += (unsigned)y;
	mouse.z += (unsigned)z;
/*__debug_printf("%x\n", m->buttons);*/
	if (vc_mouse[active_vc].mode != TTYF_MOUSE_RAW) {
		if ((mouse.buttons ^ 8) & m->buttons & 8) {
			kbd_vcrel(-1);
		}
		if ((mouse.buttons ^ 16) & m->buttons & 16) {
			kbd_vcrel(1);
		}
		if ((mouse.buttons ^ 32) & m->buttons & 32) {
			kbd_vclast();
		}
		if (__likely(vc_mouse[active_vc].mode ^ TTYF_MOUSE_CHAR) | (mods[active_vc] & MOD_MOUSE_PASTE)) {
			if (__unlikely(m->wy != 0)) {
				kbd_scroll_lines(-m->wy * mouse_scrollback);
			}
			if ((mouse.buttons ^ 1) & m->buttons & 1) {
				u_jiffies_t j = KERNEL$GET_JIFFIES();
				if (__likely(j - last_click > TIMEOUT_JIFFIES(DOUBLE_CLICK_TIME))) dbl_clk = 0;
				drag_start_x = mouse.x;
				drag_start_y = mouse.y;
				dragging = ++dbl_clk;
				if (__unlikely(dragging > 3)) dragging = 1, dbl_clk = 1;
				last_click = j;
			}
			if (!(m->buttons & 1)) {
				dragging = 0;
			}
			if ((mouse.buttons ^ 2) & m->buttons & 2) {
				paste();
			}
			send_ev = 0;
		} else {
			dragging = 0;
			kbd_unscroll();
		}
		if (m->rx | m->ry | (m->buttons & 1)) {
			kbd_display_mouse();
		}
	}
	if (send_ev) {
		vcm = &vc_mouse[active_vc];
		if (__unlikely(vcm->make_new_event))
			if (__likely(((vcm->qlast + 1) & (MOUSE_QUEUE_SIZE - 1)) != vcm->qfirst))
				vcm->qlast = (vcm->qlast + 1) & (MOUSE_QUEUE_SIZE - 1);
		/*__debug_printf("event: buttons %x, last buttons %x, make new %d, signaled %d\n", m->buttons, mouse.buttons, vcm->make_new_event, vcm->signaled);*/
		vcm->make_new_event = mouse.buttons != m->buttons;
		mouse.buttons = m->buttons;
		memcpy(&vcm->queue[vcm->qlast], &mouse, sizeof(struct mouse_state));
		if (vc_mouse[active_vc].mode != TTYF_MOUSE_RAW) {
			vcm->queue[vcm->qlast].buttons &= ~(8 | 16 | 32);
		}
		vcm->signaled = 1;
		WQ_WAKE_ALL_PL(&vcm->wait);
	} else {
		mouse.buttons = m->buttons;
	}
	LOWER_SPLX(spl);
}

static char *kbd_prepare_shutdown(void)
{
	int i;
	static char msg[__MAX_STR_LEN];
	for (i = 0; i < MAX_VCS; i++) if (locked[i]) {
		_snprintf(msg, __MAX_STR_LEN, "TTY ON CONSOLE %d USED", i);
		return msg;
	}
	if (mouse_lock_count && keyboard_lock_count) return "KEYBOARD AND MOUSE USED";
	if (mouse_lock_count) return "MOUSE USED";
	if (keyboard_lock_count) return "KEYBOARD USED";
	kbd_shutting_down = 1;
	return NULL;
}
