#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <ARCH/IRQ.H>
#include <ARCH/TIME.H>
#include <ARCH/IO.H>
#include <SPAD/DEV.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/SYNC.H>
#include <SPAD/VM.H>
#include <SPAD/TTY.H>

static int hwkbd_parse_flag(char *opt, char *optend, char *val);
static void hw_kbd_set_autorepeat(struct tty_keyboard_params *k);
#define LED_SCROLL_LOCK		1
#define LED_NUM_LOCK		2
#define LED_CAPS_LOCK		4
static void hw_kbd_set_leds(int leds);
static int hw_kbd_get_default_leds(void);
static int init_psaux = 1;
static int kbd_sync = 0;

#define KBD_NAME "PCKBD"
#define HAVE_PSAUX

#include "PCKBD.I"
#include "PCKBD.H"

#define KEYBOARD_IRQ		1
#define MOUSE_IRQ		12
#define KEYBOARD_TIMEOUT	(2 * JIFFIES_PER_SECOND)
#define MOUSE_TIMEOUT		(JIFFIES_PER_SECOND / 20)

/*
 * Parts of this file are taken from Linux:
 *
 * linux/drivers/char/pc_keyb.c
 *
 * Separation of the PC low-level part by Geert Uytterhoeven, May 1997
 * See keyboard.c for the whole history.
 *
 * Major cleanup by Martin Mares, May 1997
 *
 * Combined the keyboard and PS/2 mouse handling into one file,
 * because they share the same hardware.
 * Johan Myreen <jem@iki.fi> 1998-10-08.
 *
 * Code fixes to handle mouse ACKs properly.
 * C. Scott Ananian <cananian@alumni.princeton.edu> 1999-01-29.
 */

static int keyboard_initialized = 0;

static void handle_mouse_code(int scancode);

static unsigned char handle_kbd_event(void)
{
	unsigned char status;
	u_jiffies_lo_t timeout = KERNEL$GET_JIFFIES_LO(), tt;
	while (tt = KERNEL$GET_JIFFIES_LO(), (status = io_inb(KBD_STATUS_REG)) & KBD_STAT_OBF) {
		unsigned char scancode = io_inb(KBD_DATA_REG);
		if (__likely(!(status & (KBD_STAT_GTO | KBD_STAT_PERR)))) {
			if (__likely(keyboard_initialized)) {
				if (status & KBD_STAT_MOUSE_OBF) {
					if (__likely(init_psaux)) handle_mouse_code(scancode);
				} else {
					handle_scancode(scancode);
				}
			}
		}
		if (tt - timeout > TIMEOUT_JIFFIES(KEYBOARD_TIMEOUT)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, KBD_NAME, "CONTROLLER JAMMED, STATUS %02X", (int)status);
			break;
		}
	}
	return status;
}

static void kbd_wait(void)
{
	u_jiffies_lo_t timeout = KERNEL$GET_JIFFIES_LO();
	while (1) {
		u_jiffies_lo_t tt = KERNEL$GET_JIFFIES_LO();
		unsigned char status = handle_kbd_event();
		if (!(status & KBD_STAT_IBF)) break;
		if (tt - timeout > TIMEOUT_JIFFIES(KEYBOARD_TIMEOUT)) {
			KERNEL$SYSLOG(__SYSLOG_HW_ERROR, KBD_NAME, "TIMEOUT");
			break;
		}
		if (!kbd_sync) KERNEL$UDELAY(1000);
		else KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);
	}
}

static void kbd_write_command_w(int cmd)
{
	kbd_wait();
	io_outb(KBD_CNTL_REG, cmd);
}

#define KBD_NO_DATA     (-1)    /* No data */
#define KBD_BAD_DATA    (-2)    /* Parity or other error */

static int kbd_read_data(void)
{
	int retval = KBD_NO_DATA;
	unsigned char status;
	status = io_inb(KBD_STATUS_REG);
	if (status & KBD_STAT_OBF) {
		retval = io_inb(KBD_DATA_REG);
		if (__unlikely(status & KBD_STAT_MOUSE_OBF)) retval = KBD_NO_DATA;
		else if (__unlikely(status & (KBD_STAT_GTO | KBD_STAT_PERR))) retval = KBD_BAD_DATA;
	}
	return retval;
}

static int kbd_wait_for_input(void)
{
	u_jiffies_lo_t timeout = KERNEL$GET_JIFFIES_LO();
	while (1) {
		u_jiffies_lo_t tt = KERNEL$GET_JIFFIES_LO();
		int retval = kbd_read_data();
		if (retval >= 0) return retval;
		if (__unlikely(tt - timeout > TIMEOUT_JIFFIES(KEYBOARD_TIMEOUT))) break;
		if (!kbd_sync) KERNEL$UDELAY(1000);
		else KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);
	}
	return -1;
}

static void kbd_write_output_w(int cmd)
{
	kbd_wait();
	io_outb(KBD_DATA_REG, cmd);
}

static int kbd_write_output_ack(int cmd, char *msg, char *err)
{
	u_jiffies_lo_t timeout = KERNEL$GET_JIFFIES_LO();
	unsigned char status;
	do {
		u_jiffies_lo_t tt = KERNEL$GET_JIFFIES_LO();
		kbd_write_output_w(cmd);
		status = kbd_wait_for_input();
		if (status == KBD_REPLY_ACK) return 0;
		if (tt - timeout > TIMEOUT_JIFFIES(KEYBOARD_TIMEOUT)) {
			if (err) _snprintf(err, __MAX_STR_LEN, "KEYBOARD: RESEND TIMEOUT ON %s", msg);
			return -EIO;
		}
	} while (status == KBD_REPLY_RESEND);
	if (err) _snprintf(err, __MAX_STR_LEN, "KEYBOARD: NO ACK ON %s: %d", msg, (int)status);
	return -EIO;
}

static void mouse_write_output(int data)
{
	kbd_write_command_w(KBD_CCMD_WRITE_MOUSE);
	kbd_write_output_w(data);
}

static int mouse_read_data(void)
{
	int retval = KBD_NO_DATA;
	unsigned char status;
	status = io_inb(KBD_STATUS_REG);
	if (status & KBD_STAT_OBF) {
		retval = io_inb(KBD_DATA_REG);
		if (__unlikely(!(status & KBD_STAT_MOUSE_OBF))) retval = KBD_NO_DATA;
		else if (__unlikely(status & (KBD_STAT_GTO | KBD_STAT_PERR))) retval = KBD_BAD_DATA;
	}
	return retval;
}

static int mouse_wait_for_input(void)
{
	int retval;
	u_jiffies_lo_t timeout = KERNEL$GET_JIFFIES_LO(), tt;
	do {
	/* SiS legacy USB support wants big timeout --- 1ms is not enough */
		if (!kbd_sync) KERNEL$UDELAY(10000);
		else KERNEL$SLEEP(JIFFIES_PER_SECOND / 100 + 1);
		tt = KERNEL$GET_JIFFIES_LO();
		retval = mouse_read_data();
		if (__likely(retval >= 0)) return retval;
	} while (!(tt - timeout > TIMEOUT_JIFFIES(MOUSE_TIMEOUT)));
	return retval;
}

static AST KBD_INTR_AST;
static IRQ_CONTROL KBD_INTR_CTRL;

static AST MOUSE_INTR_AST;
static IRQ_CONTROL MOUSE_INTR_CTRL;

DECL_AST(KBD_INTR, SPL_KEYBOARD, AST)
{
	handle_kbd_event();
	KBD_INTR_CTRL.eoi();
	RETURN;
}

DECL_AST(MOUSE_INTR, SPL_KEYBOARD, AST)
{
	handle_kbd_event();
	MOUSE_INTR_CTRL.eoi();
	RETURN;
}

#include "PS2MOUSE.I"

#define PCKBD_DEFAULT_DELAY	0
#define PCKBD_DEFAULT_RATE	0

static __const__ short int pckbd_delays[4] = {
	250, 500, 750, 1000
};

static __const__ short int pckbd_rates[32] = {
	1000 / 30.0,  1000 / 26.7,  1000 / 24.0,  1000 / 21.8,
	1000 / 20.7,  1000 / 18.5,  1000 / 17.1,  1000 / 16.0,
	1000 / 15.0,  1000 / 13.3,  1000 / 12.0,  1000 / 10.9,
	1000 / 10.0,  1000 /  9.2,  1000 /  8.6,  1000 /  8.0,
	1000 /  7.5,  1000 /  6.7,  1000 /  6.0,  1000 /  5.5,
	1000 /  5.0,  1000 /  4.6,  1000 /  4.3,  1000 /  4.0,
	1000 /  3.7,  1000 /  3.3,  1000 /  3.0,  1000 /  2.7,
	1000 /  2.5,  1000 /  2.3,  1000 /  2.1,  1000 /  2.0,
};

static __finline__ int absolute(int i)
{
	if (__likely(i >= 0)) return i;
	return -i;
}

static int hw_kbd_find_nearest_rate(__const__ short int array[], int size, int value, int deflt)
{
	int i;
	int best_idx;
	if (!value) return deflt;
	best_idx = 0;
	for (i = 1; i < size; i++) {
		if (absolute(value - array[best_idx]) > absolute(value - array[i])) best_idx = i;
	}
	return best_idx;
}

static char err_sink[__MAX_STR_LEN];

static void hw_kbd_set_autorepeat(struct tty_keyboard_params *k)
{
	int delay_idx, rate_idx;
	if (__unlikely((k->repeat_delay | k->repeat_rate) < 0)) {
		k->soft = 1;
	}
	if (!k->soft) {
		delay_idx = hw_kbd_find_nearest_rate(pckbd_delays, 4, k->repeat_delay, PCKBD_DEFAULT_DELAY);
		rate_idx = hw_kbd_find_nearest_rate(pckbd_rates, 32, k->repeat_rate, PCKBD_DEFAULT_RATE);
	} else {
		delay_idx = 3;
		rate_idx = 31;
	}
	if (__unlikely(kbd_write_output_ack(KBD_CMD_SET_RATE, "SET RATE", err_sink))) {
		k->soft = 1;
		return;
	}
	if (__unlikely(kbd_write_output_ack((delay_idx << 5) | rate_idx, "SET RATE VALUE", err_sink))) {
		k->soft = 1;
		return;
	}
}

static void hw_kbd_set_leds(int leds)
{
	if (__unlikely(kbd_write_output_ack(KBD_CMD_SET_LEDS, "SET LEDS", err_sink))) return;
	if (__unlikely(kbd_write_output_ack(leds, "SET LEDS VALUE", err_sink))) return;
}

static int hw_kbd_get_default_leds(void)
{
	return (*(KERNEL$ZERO_BANK + 0x417) >> 4) & (LED_SCROLL_LOCK | LED_NUM_LOCK | LED_CAPS_LOCK);
}

static int hw_kbd_init(char *err)
{
	int mode;
	char *e;
	int m;
	kbd_sync = 1;
	m = 100;
	while (m--) if (kbd_read_data() == KBD_NO_DATA) break;
	/*
	This test fails on SiS chipset with legacy-emulated USB.
	*/
	/*
	kbd_write_command_w(KBD_CCMD_SELF_TEST);
	if ((r = kbd_wait_for_input()) != 0x55) {
		_snprintf(err, __MAX_STR_LEN, KBD_NAME": SELF TEST FAILED: %d", r);
		return -EIO;
	}
	*/
	/*
	This test fails on my 486 with return 3. I have no idea why.
	When I comment it out, the keyboard works.
	*/
	/*
	kbd_write_command_w(KBD_CCMD_KBD_TEST);
	if ((r = kbd_wait_for_input()) != 0x00) {
		_snprintf(err, __MAX_STR_LEN, KBD_NAME": INTERFACE TEST FAILED: %d", r);
		return -EIO;
	}
	*/
	kbd_write_command_w(KBD_CCMD_KBD_ENABLE);
	/*
	if (kbd_write_output_ack(KBD_CMD_RESET, "RESET", err)) return -EIO;
	if ((r = kbd_wait_for_input()) != KBD_REPLY_POR) {
		_snprintf(err, __MAX_STR_LEN, KBD_NAME": NO POR: %d", r);
		return -EIO;
	}
	*/
	kbd_write_output_ack(KBD_CMD_DISABLE, "DISABLE", err);
	kbd_write_command_w(KBD_CCMD_WRITE_MODE);
	kbd_write_output_w(KBD_MODE_KBD_INT | KBD_MODE_SYS | KBD_MODE_DISABLE_MOUSE | KBD_MODE_KCC);
	kbd_write_command_w(KBD_CCMD_READ_MODE);
	mode = kbd_wait_for_input();
	if (__likely(mode != -1) && __unlikely(!(mode & KBD_MODE_KCC))) {
		/*
		 * If the controller does not support conversion,
		 * Set the keyboard to scan-code set 1.
		 */
		kbd_write_output_w(0xF0);
		kbd_wait_for_input();
		kbd_write_output_w(0x01);
		kbd_wait_for_input();
	}
	kbd_write_output_ack(KBD_CMD_ENABLE, "ENABLE", err);
	kbd_write_output_ack(KBD_CMD_SET_RATE, "SET RATE", err);
	kbd_write_output_ack((PCKBD_DEFAULT_DELAY << 5) | PCKBD_DEFAULT_RATE, "SET RATE VALUE", err);
	if (init_psaux) {
		kbd_write_command_w(KBD_CCMD_MOUSE_ENABLE);
		/*mouse_write_output(AUX_ENABLE_DEV);*/
		kbd_write_command_w(KBD_CCMD_WRITE_MODE);
		kbd_write_output_w(KBD_MODE_KBD_INT | KBD_MODE_MOUSE_INT | KBD_MODE_SYS | KBD_MODE_KCC);
		kbd_write_command_w(KBD_CCMD_READ_MODE);
		mode = kbd_wait_for_input();
		if ((mode & (KBD_MODE_MOUSE_INT | KBD_MODE_DISABLE_MOUSE)) != KBD_MODE_MOUSE_INT) {
			goto skip_mouse;
		}
		init_ps2_mouse();
		kbd_sync = 0;
		MOUSE_INTR_AST.fn = &MOUSE_INTR;
		if ((e = KERNEL$REQUEST_IRQ(MOUSE_IRQ, IRQ_AST_HANDLER | IRQ_SHARED, NULL, &MOUSE_INTR_AST, &MOUSE_INTR_CTRL, KBD_NAME))) {
			KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, "PS2MOUSE", "COULD NOT GET MOUSE IRQ %d: %s", MOUSE_IRQ, e);
			skip_mouse:
			init_psaux = 0;
			kbd_write_command_w(KBD_CCMD_WRITE_MODE);
			kbd_write_output_w(KBD_MODE_KBD_INT | KBD_MODE_SYS | KBD_MODE_DISABLE_MOUSE | KBD_MODE_KCC);
		}
	}
	kbd_sync = 0;
	keyboard_initialized = 1;
	KBD_INTR_AST.fn = &KBD_INTR;
	if ((e = KERNEL$REQUEST_IRQ(KEYBOARD_IRQ, IRQ_AST_HANDLER, NULL, &KBD_INTR_AST, &KBD_INTR_CTRL, KBD_NAME))) {
		if (init_psaux) KERNEL$RELEASE_IRQ(&MOUSE_INTR_CTRL, 0);
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, KBD_NAME, "COULD NOT GET KEYBOARD IRQ %d: %s", KEYBOARD_IRQ, e);
		_snprintf(err, __MAX_STR_LEN, KBD_NAME": COULD NOT GET IRQ");
		return -EBUSY;
	}
	return 0;
}

static int hwkbd_parse_flag(char *opt, char *optend, char *val)
{
	return -1;
}

static int unload(void *p, void **release, char *argv[]);
static void *lnte, *dlrq;
static IO_RANGE range;

int main(int argc, char *argv[])
{
	DEVICE_REQUEST keyboard_dev;
	int r;
	if ((r = kbd_init(argc, argv))) {
		return r;
	}
	range.start = KBD_REGSPACE_START;
	range.len = KBD_REGSPACE_LEN;
	range.name = KBD_NAME;
	if ((r = KERNEL$REGISTER_IO_RANGE(&range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, KBD_NAME": COULD NOT REGISTER PORT RANGE: %s", strerror(-r));
		return r;
	}
	if ((r = hw_kbd_init(KERNEL$ERROR_MSG()))) {
		KERNEL$UNREGISTER_IO_RANGE(&range);
		return r;
	}
	RAISE_SPL(SPL_KEYBOARD);
	test_leds();
	LOWER_SPL(SPL_ZERO);
	keyboard_dev.flags = 0;
	keyboard_dev.init_root_handle = keyboard_init_root;
	keyboard_dev.name = KBD_NAME;
	keyboard_dev.driver_name = "PCKBD.SYS";
	keyboard_dev.dcall = kbd_raw_dcall;
	keyboard_dev.dcall_type = "KEYBOARD";
	keyboard_dev.dctl = NULL;
	keyboard_dev.unload = unload;
	SYNC_IO_CANCELABLE(&keyboard_dev, KERNEL$REGISTER_DEVICE);
	if (keyboard_dev.status < 0) {
		if (keyboard_dev.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, KBD_NAME": COULD NOT REGISTER DEVICE");
		KERNEL$UNREGISTER_IO_RANGE(&range);
		return keyboard_dev.status;
	}
	lnte = keyboard_dev.lnte;
	strlcpy(KERNEL$ERROR_MSG(), KBD_NAME, __MAX_STR_LEN);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;
}

static int unload(void *p, void **release, char *argv[])
{
	int r;
	char *e;
	RAISE_SPL(SPL_KEYBOARD);
	if ((e = kbd_prepare_shutdown())) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD: %s", e);
		return -EBUSY;
	}
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	KBD_INTR_CTRL.mask();
	if (init_psaux) MOUSE_INTR_CTRL.mask();
	RAISE_SPL(SPL_KEYBOARD);
	if (__unlikely(ps2_reset_timer.fn != NULL)) KERNEL$DEL_TIMER(&ps2_reset_timer), ps2_reset_timer.fn = NULL;
	LOWER_SPL(SPL_ZERO);
	KERNEL$RELEASE_IRQ(&KBD_INTR_CTRL, IRQ_RELEASE_MASKED);
	if (init_psaux) KERNEL$RELEASE_IRQ(&MOUSE_INTR_CTRL, IRQ_RELEASE_MASKED);
	RAISE_SPL(SPL_KEYBOARD);
	kbd_do_shutdown();
	LOWER_SPL(SPL_ZERO);
	KERNEL$UNREGISTER_IO_RANGE(&range);
	*release = dlrq;
	return 0;
}
