#include <SPAD/LIBC.H>
#include <SPAD/LIST.H>
#include <SPAD/IOCTL.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/AC.H>
#include <ENDIAN.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/ALLOC.H>
#include <VALUES.H>

#include "EAP.H"

static void END_CALLBACK(EAP *eap, int status);
static void EAP_RESET_METHOD(EAP *eap);
static void EAP_RESET_OFFERS(EAP *eap);
static void EAP_WAKE_ALL_OFFERS(EAP *eap);
static void EAP_CLEAR_VOID_OFFERS(EAP *eap);

typedef struct {
	LIST_ENTRY list;
	WQ wq;
	__u64 magic;
	unsigned priority;
	unsigned n_offers;
	struct {	/* must match EAP_OFFER_REQUEST */
		__u32 vendor;
		__u32 type;
		char name[EAP_MAXNAMELEN];
	} offer[1];
} EAP_OFFER;

#define NEEDS_EXPANDED(vendor, type)	(!(__likely(!(vendor)) && __likely((type) < 256)))

static __u8 *EAP_CREATE_REPLY(EAP *eap, unsigned len, __u32 vendor, __u32 type)
{
	EAP_FRAME *ef;
	if (__unlikely(len > EAP_MAX_DATA_LENGTH))
		KERNEL$SUICIDE("EAP_CREATE_REPLY: LENGTH %u > %d", len, EAP_MAX_DATA_LENGTH);
	eap->can_repeat_packet = 1;
	ef = (EAP_FRAME *)eap->packet;
	ef->code = EAP_CODE_RESPONSE;
	ef->id = eap->last_id;
	if (__likely(!eap->was_expanded)) {
		eap->packet_length = len + EAP_REQ_HEADER_SIZE;
		ef->length = htons(eap->packet_length);
		ef->type = type;
		return eap->packet + EAP_REQ_HEADER_SIZE;
	} else {
		eap->packet_length = len + EAP_EXP_HEADER_SIZE;
		ef->length = htons(eap->packet_length);
		ef->type = EAP_TYPE_EXPANDED;
		ef->vendor[0] = vendor >> 16;
		ef->vendor[1] = vendor >> 8;
		ef->vendor[2] = vendor;
		ef->exp_type = htonl(type);
		return eap->packet + EAP_EXP_HEADER_SIZE;
	}
}

static EAP_OFFER *EAP_SELECT_OFFER(EAP *eap, __u32 vendor, __u32 type, unsigned *idx)
{
	unsigned i;
	EAP_OFFER *offer, *best = NULL;
	XLIST_FOR_EACH(offer, &eap->offers, EAP_OFFER, list) {
		if (__unlikely(offer->priority > eap->top_priority)) continue;
		if (__unlikely(!offer->n_offers)) continue;
		if (__likely(!best) || offer->priority > best->priority) best = offer;
	}
	offer = best;
	if (!offer) return NULL;
	eap->top_priority = offer->priority;
	for (i = 0; i < offer->n_offers; i++) if (offer->offer[i].vendor == vendor && offer->offer[i].type == type) {
		*idx = i;
		return offer;
	}
	return NULL;
}

static void EAP_SEND_NAK(EAP *eap)
{
	EAP_OFFER *offer, *best = NULL;
	XLIST_FOR_EACH(offer, &eap->offers, EAP_OFFER, list) {
		if (__likely(!eap->was_expanded)) {
			unsigned i;
			for (i = 0; i < offer->n_offers; i++) if (!NEEDS_EXPANDED(offer->offer[i].vendor, offer->offer[i].type)) goto ok;
			continue;
		}
		ok:
		if (offer->priority > eap->top_priority) continue;
		if (__likely(!best) || offer->priority > best->priority) best = offer;
	}
	if (__unlikely(!best)) {
		EAP_RESET_OFFERS(eap);
		END_CALLBACK(eap, EAP_END_IDLE);
		return;
	}
	offer = best;
	eap->top_priority = offer->priority;
	if (__likely(!eap->was_expanded)) {
		unsigned i;
		unsigned n = 0;
		__u8 *data;
		for (i = 0; i < offer->n_offers; i++) if (!NEEDS_EXPANDED(offer->offer[i].vendor, offer->offer[i].type)) n++;
		data = EAP_CREATE_REPLY(eap, n, 0, EAP_TYPE_NAK);
		for (i = 0; i < offer->n_offers; i++) if (!NEEDS_EXPANDED(offer->offer[i].vendor, offer->offer[i].type)) *data++ = offer->offer[i].type;
	} else {
		unsigned i;
		__u8 *data = EAP_CREATE_REPLY(eap, offer->n_offers * 8, 0, EAP_TYPE_NAK);
		for (i = 0; i < offer->n_offers; i++) {
			*data++ = EAP_TYPE_EXPANDED;
			*data++ = offer->offer[i].vendor >> 16;
			*data++ = offer->offer[i].vendor >> 8;
			*data++ = offer->offer[i].vendor;
			*data++ = offer->offer[i].type >> 24;
			*data++ = offer->offer[i].type >> 16;
			*data++ = offer->offer[i].type >> 8;
			*data++ = offer->offer[i].type;
		}
	}
	debug(("nak\n"));
	eap->progress_callback(eap);
	eap->send_packet(eap);
}

void EAP_REQUEST(EAP *eap, EAP_FRAME *ef)
{
	static char err[__MAX_STR_LEN];
	__u8 *data;
	unsigned len;
	__u32 vendor, type;
	if (__unlikely(ef->id == eap->last_id) && eap->can_repeat_packet) {
		debug(("repeating...\n"));
		eap->send_packet(eap);
		return;
	}
	eap->last_id = ef->id;
	eap->can_repeat_packet = 0;
	if (__likely(ef->type != EAP_TYPE_EXPANDED)) {
		vendor = 0;
		type = ef->type;
		data = (__u8 *)ef + EAP_REQ_HEADER_SIZE;
		len = ntohs(ef->length) - EAP_REQ_HEADER_SIZE;
		eap->was_expanded = 0;
	} else {
		if (__unlikely(ntohs(ef->length) < EAP_EXP_HEADER_SIZE)) {
			_snprintf(err, sizeof err, "TOO SMALL EXPANDED REQUEST PACKET (%d < %d)", ntohs(ef->length), EAP_EXP_HEADER_SIZE);
			log_error_end:
			eap->log_callback(eap, EAP_LOG_ERROR, err);
			END_CALLBACK(eap, EAP_END_PROTOCOL_VIOLATION);
			return;
		}
		vendor = (ef->vendor[2] << 16) | (ef->vendor[1] << 8) | ef->vendor[0];
		type = ntohl(ef->exp_type);
		data = (__u8 *)ef + EAP_EXP_HEADER_SIZE;
		len = ntohs(ef->length) - EAP_EXP_HEADER_SIZE;
		eap->was_expanded = 1;
	}
	/*__debug_printf("eap: %02x\n", type);*/
	if (!vendor && type == EAP_TYPE_IDENTITY) {
		EAP_RESET_METHOD(eap);
		debug(("request identity\n"));
		len = strlen(eap->identity);
		if (__unlikely(len > EAP_MAX_DATA_LENGTH)) len = EAP_MAX_DATA_LENGTH;
		memcpy(EAP_CREATE_REPLY(eap, len, 0, EAP_TYPE_IDENTITY), eap->identity, len);
		eap->send_packet(eap);
		if (!EAP_HAVE_OFFERS(eap)) {
			EAP_RESET_OFFERS(eap);
			END_CALLBACK(eap, EAP_END_IDLE);
		}
		return;
	}
	if (!vendor && type == EAP_TYPE_NOTIFICATION) {
		_snprintf(err, sizeof err, "%.*s", (int)len, data);
		eap->log_callback(eap, EAP_LOG_WARNING, err);
		EAP_CREATE_REPLY(eap, 0, 0, EAP_TYPE_NOTIFICATION);
		eap->send_packet(eap);
		return;
	}
	if (!vendor && type == EAP_TYPE_NAK) {
		_snprintf(err, sizeof err, "NAK RECEIVED IN REQUEST");
		goto log_error_end;
	}
	if (__unlikely(len > EAP_MAX_RECEIVED_DATA_LENGTH)) {
		_snprintf(err, sizeof err, "TOO LONG REQUEST, LENGTH %u", len);
		goto log_error_end;
	}
	if (__unlikely(eap->waiting_for_userspace)) {
		if (__likely(eap->received_data_length == len) && __likely(!memcmp(eap->received_data, data, len)) && __likely(eap->received_data_code == EAP_CODE_REQUEST)) return;
		_snprintf(err, sizeof err, "PACKET RECEIVED WHILE WAITING FOR USERSPACE, VENDOR %X, TYPE %X", (unsigned)vendor, (unsigned)type);
		eap->log_callback(eap, EAP_LOG_ERROR, err);
		EAP_RESET_METHOD(eap);
	}
	eap->something_received = 1;
	eap->received_data_code = EAP_CODE_REQUEST;
	eap->received_data_length = len;
	memcpy(eap->received_data, data, len);
	if (eap->method_vendor == -1) {
		unsigned idx;
		EAP_OFFER *offer;
		IOCTLRQ *rq;
		select_again:
		EAP_CLEAR_VOID_OFFERS(eap);
		offer = EAP_SELECT_OFFER(eap, vendor, type, &idx);
		if (__unlikely(!offer)) {
			debug(("not supported\n"));
			EAP_SEND_NAK(eap);
			return;
		}
		rq = (IOCTLRQ *)WQ_GET_ONE(&offer->wq);
		if (__unlikely(!rq)) goto select_again;
		eap->method_vendor = vendor;
		eap->method_type = type;
		memcpy(eap->method_name, offer->offer[idx].name, EAP_MAXNAMELEN);
		eap->method_name[EAP_MAXNAMELEN] = 0;
		eap->magic = offer->magic;
		debug(("selected offer: %d\n", idx));
		rq->status = idx;
		CALL_AST(rq);
	}
	if (__unlikely(vendor != eap->method_vendor) || __unlikely(type != eap->method_type)) {
		_snprintf(err, sizeof err, "REQUEST TYPE DOESN'T MATCH. THIS REQUEST (%d/%d), LAST REQUEST (%d/%d)", vendor, type, eap->method_vendor, eap->method_type);
		goto log_error_end;
	}
	eap->waiting_for_userspace = 1;
	WQ_WAKE_ALL_PL(&eap->userspace_wait);
}

static void EAP_FINAL_PACKET(EAP *eap, unsigned code)
{
	static char err[__MAX_STR_LEN];
	if (__unlikely(eap->waiting_for_userspace)) {
		if (code == EAP_CODE_FAILURE) goto reset;	/* this can actually happen, do not log protocol error */
		_snprintf(err, sizeof err, "%s PACKET RECEIVED WHILE WAITING FOR USERSPACE", code == EAP_CODE_SUCCESS ? "SUCCESS" : "FAILURE");
		log_reset:
		eap->log_callback(eap, EAP_LOG_ERROR, err);
		reset:
		EAP_RESET_METHOD(eap);
		return;
	}
	if (__unlikely(eap->method_vendor == -1)) {
		if (code == EAP_CODE_FAILURE) goto reset;	/* this can actually happen, do not log protocol error */
		_snprintf(err, sizeof err, "%s PACKET RECEIVED WHILE NO METHOD NEGOTIATED", code == EAP_CODE_SUCCESS ? "SUCCESS" : "FAILURE");
		goto log_reset;
	}
	eap->received_data_code = code;
	eap->received_data_length = 0;
	eap->waiting_for_userspace = 1;
	WQ_WAKE_ALL_PL(&eap->userspace_wait);
}

void EAP_SUCCESS(EAP *eap)
{
	eap->something_received = 1;
	debug(("success\n"));
	EAP_FINAL_PACKET(eap, EAP_CODE_SUCCESS);
}

void EAP_FAILURE(EAP *eap)
{
	eap->something_received = 1;
	debug(("failure\n"));
	if (eap->method_vendor == -1) {
		if (!eap->top_priority) EAP_RESET_OFFERS(eap);
		else eap->top_priority = eap->top_priority - 1;
	} else {
		EAP_FINAL_PACKET(eap, EAP_CODE_FAILURE);
	}
}

static void END_CALLBACK(EAP *eap, int status)
{
	char *msg = NULL;
	if (status == EAP_END_PROTOCOL_VIOLATION) {
		msg = "PROTOCOL VIOLATION";
	} else if (status == EAP_END_FAILURE_RETRY || status == EAP_END_FAILURE_HOLD) {
		msg = "FAILURE";
	}
	if (msg) {
		_snprintf(eap->failure_reason, sizeof eap->failure_reason, "%s%s%s", msg, eap->method_vendor == -1 ? "" : " IN METHOD ", eap->method_vendor == -1 ? "" : eap->method_name);
	} else {
		eap->failure_reason[0] = 0;
	}
	eap->end_callback(eap, status);
}

static void EAP_RESET_METHOD(EAP *eap)
{
	eap->method_vendor = -1;
	eap->method_type = 0;
	eap->waiting_for_userspace = 0;
	WQ_WAKE_ALL(&eap->userspace_wait);
}

static void EAP_RESET_OFFERS(EAP *eap)
{
	eap->top_priority = MAXINT;
	eap->offers_seq++;
	EAP_WAKE_ALL_OFFERS(eap);
}

void EAP_RESET(EAP *eap)
{
	/*__debug_printf("reset\n");*/
	eap->can_repeat_packet = 0;
	eap->something_received = 0;
	EAP_RESET_METHOD(eap);
}

void EAP_INIT(EAP *eap)
{
	WQ_INIT(&eap->userspace_wait, "EAP$USERSPACE_WAIT");
	INIT_XLIST(&eap->offers);
	eap->offers_seq = 0;
	eap->failure_reason[0] = 0;
	EAP_RESET_OFFERS(eap);
	EAP_RESET(eap);
}

int EAP_HAVE_OFFERS(EAP *eap)
{
	EAP_CLEAR_VOID_OFFERS(eap);
	return !XLIST_EMPTY(&eap->offers);
}

int EAP_HAVE_NONEMPTY_OFFERS(EAP *eap)
{
	EAP_OFFER *offer;
	EAP_CLEAR_VOID_OFFERS(eap);
	XLIST_FOR_EACH(offer, &eap->offers, EAP_OFFER, list) if (offer->n_offers) return 1;
	return 0;
}

static void EAP_WAKE_ALL_OFFERS(EAP *eap)
{
	EAP_OFFER *offer;
	XLIST_FOR_EACH(offer, &eap->offers, EAP_OFFER, list) WQ_WAKE_ALL_PL(&offer->wq);
	EAP_CLEAR_VOID_OFFERS(eap);
}

static void EAP_CLEAR_VOID_OFFERS(EAP *eap)
{
	EAP_OFFER *offer;
	XLIST_FOR_EACH(offer, &eap->offers, EAP_OFFER, list) if (__unlikely(WQ_EMPTY(&offer->wq))) {
		EAP_OFFER *prev = LIST_STRUCT(offer->list.prev, EAP_OFFER, list);
		DEL_FROM_LIST(&offer->list);
		free(offer);
		offer = prev;
	}
}

void EAP_IOCTL(EAP *eap, IOCTLRQ *rq)
{
	int r;
	static EAP_OFFER_REQUEST tmp_offer;
	static EAP_PACKET_REQUEST tmp_packet;
	switch (rq->ioctl) {
		case IOCTL_EAP_OFFER_METHODS: {
			unsigned n, size;
			EAP_OFFER *offer;
			EAP_CLEAR_VOID_OFFERS(eap);
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &tmp_offer, sizeof tmp_offer))) {
				get_ioctl_failure:
				if (r == 1) {
					DO_PAGEIN_NORET(rq, &rq->v, PF_READ);
				} else {
					rq->status = r;
					CALL_AST(rq);
				}
				return;
			}
			if (__unlikely(tmp_offer.offers_seq != eap->offers_seq)) {
				tmp_offer.offers_seq = eap->offers_seq;
				if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(rq, &tmp_offer, sizeof tmp_offer))) {
					put_ioctl_failure:
					if (r == 1) {
						DO_PAGEIN_NORET(rq, &rq->v, PF_WRITE);
					} else {
						rq->status = r;
						CALL_AST(rq);
					}
					return;
				}
				rq->status = -ESTALE;
				CALL_AST(rq);
				return;
			}
			for (n = 0; n < EAP_MAX_OFFERS; n++) if (!tmp_offer.offers[n].vendor && !tmp_offer.offers[n].type) break;
			size = sizeof(EAP_OFFER) + n * sizeof(tmp_offer.offers[0]);
			offer = malloc(size);
			if (__unlikely(!offer)) {
				KERNEL$MEMWAIT((IORQ *)rq, KERNEL$WAKE_IOCTL, size);
				return;
			}
			WQ_INIT(&offer->wq, "EAP$OFFER");
			WQ_WAIT_F(&offer->wq, rq);
			offer->magic = tmp_offer.magic;
			offer->priority = rq->param;
			offer->n_offers = n;
			memcpy(offer->offer, tmp_offer.offers, n * sizeof(tmp_offer.offers[0]));
			ADD_TO_XLIST(&eap->offers, &offer->list);
			eap->offer_callback(eap);
			return;
		}
		case IOCTL_EAP_GET_PACKET: {
			if (__unlikely(eap->method_vendor == -1)) {
				econnreset:
				rq->status = -ECONNRESET;
				CALL_AST(rq);
				return;
			}
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &tmp_packet, sizeof tmp_packet))) goto get_ioctl_failure;
			if (__unlikely(tmp_packet.vendor != eap->method_vendor) || __unlikely(tmp_packet.type != eap->method_type) || __unlikely(tmp_packet.magic != eap->magic)) goto econnreset;
			if (__unlikely(!eap->waiting_for_userspace)) {
				WQ_WAIT_F(&eap->userspace_wait, rq);
				return;
			}
			tmp_packet.code = eap->received_data_code;
			tmp_packet.data_length = eap->received_data_length;
			memcpy(tmp_packet.data, eap->received_data, eap->received_data_length);
			memset(tmp_packet.data + eap->received_data_length, 0, sizeof tmp_packet.data - eap->received_data_length);
			if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(rq, &tmp_packet, sizeof tmp_packet))) goto put_ioctl_failure;
			rq->status = 0;
			CALL_AST(rq);
			return;
		}
		case IOCTL_EAP_SEND_PACKET: {
			__u8 *data;
			if (__unlikely(eap->method_vendor == -1)) goto econnreset;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(rq, &tmp_packet, sizeof tmp_packet))) goto get_ioctl_failure;
			if (__unlikely(tmp_packet.vendor != eap->method_vendor) || __unlikely(tmp_packet.type != eap->method_type) || __unlikely(tmp_packet.magic != eap->magic)) goto econnreset;
			if (__unlikely(tmp_packet.code != EAP_CODE_RESPONSE)) {
				if (__unlikely(tmp_packet.code != EAP_END_SUCCESS) &&
				    __unlikely(tmp_packet.code != EAP_END_FAILURE_RETRY) &&
				    __unlikely(tmp_packet.code != EAP_END_FAILURE_HOLD) &&
				    __unlikely(tmp_packet.code != EAP_END_PROTOCOL_VIOLATION) &&
				    __unlikely(tmp_packet.code != EAP_END_IDLE)) {
					rq->status = -EINVAL;
					CALL_AST(rq);
					return;
				}
				END_CALLBACK(eap, tmp_packet.code);
				rq->status = 0;
				CALL_AST(rq);
				return;
			}
			if (__unlikely(!eap->waiting_for_userspace)) {
				WQ_WAIT_F(&eap->userspace_wait, rq);
				return;
			}
			if (__unlikely(tmp_packet.data_length > EAP_MAX_DATA_LENGTH)) {
				rq->status = -EMSGSIZE;
				CALL_AST(rq);
				return;
			}
			data = EAP_CREATE_REPLY(eap, tmp_packet.data_length, eap->method_vendor, eap->method_type);
			memcpy(data, tmp_packet.data, tmp_packet.data_length);
			eap->waiting_for_userspace = 0;
			eap->progress_callback(eap);
			eap->send_packet(eap);
			rq->status = 0;
			CALL_AST(rq);
			return;
		}
		case IOCTL_EAP_GET_STATUS: {
			int st;
			static char status[__MAX_STR_LEN];
			memset(status, 0, sizeof status);
			st = eap->status_callback(eap, status);
			if (st == STATE_AUTHENTICATING) {
				if (!eap->something_received) {
					if (eap->failure_reason[0]) goto failure_desc;
					_snprintf(status, __MAX_STR_LEN, "WAITING FOR RADIUS SERVER");
				} else if (eap->method_vendor == -1) {
					_snprintf(status, __MAX_STR_LEN, "NEGOTIATING METHOD");
				} else {
					_snprintf(status, __MAX_STR_LEN, "AUTHENTICATING USING %s", eap->method_name);
				}
			} else if (st == STATE_AUTHENTICATED) {
				if (eap->method_vendor != -1) {
					_snprintf(status, __MAX_STR_LEN, "AUTHENTICATED USING %s", eap->method_name);
				}
			} else if (eap->failure_reason[0]) {
				failure_desc:
				_snprintf(status, __MAX_STR_LEN, eap->failure_reason);
			}
			if (__unlikely(rq->v.len > __MAX_STR_LEN)) {
				rq->status = -ERANGE;
				CALL_AST(rq);
				return;
			}
			if (__unlikely(rq->v.len <= strlen(status))) {
				rq->status = -EMSGSIZE;
				CALL_AST(rq);
				return;
			}
			if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(rq, status, rq->v.len))) goto put_ioctl_failure;
			rq->status = 0;
			CALL_AST(rq);
			return;
		}
		default: {
			rq->status = -ENOOP;
			CALL_AST(rq);
			return;
		}
	}
}

void EAP_DONE(EAP *eap)
{
	WQ_WAKE_ALL(&eap->userspace_wait);
	while (!XLIST_EMPTY(&eap->offers)) {
		EAP_OFFER *offer = LIST_STRUCT(eap->offers.next, EAP_OFFER, list);
		WQ_WAKE_ALL(&offer->wq);
		DEL_FROM_LIST(&offer->list);
		free(offer);
	}
}

