#include <SPAD/ALLOC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/PKT.H>
#include <SPAD/ETHERNET.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/IOCTL.H>
#include <SPAD/TIMER.H>
#include <SPAD/LIST.H>
#include <STRING.H>
#include <FCNTL.H>
#include <UNISTD.H>

#include "EAPOL.H"
 
static __const__ char eapol_multicast[ETHERNET_ADDRLEN] = EAPOL_MULTICAST_ADDRESS;

static DECL_XLIST(EAPOL_IFACES);

typedef struct {
	LIST_ENTRY list;
	int h;
	char errorlevel;
	char state;
	long outstanding;
	EAP eap;
	IF_TYPE type;
	TIMER eap_start_timer;
	u_jiffies_lo_t eap_start_jiffies;
	char use_private_address;
	__u8 server_address[MAX_ADDR_LEN];
	IOCTLRQ mcast_ioctl;
	char mcast_ioctl_posted;
	IOCTLRQ linkchg_ioctl;
	char linkchg_ioctl_posted;
	IOCTLRQ linknotify_ioctl;
	char linknotify_ioctl_posted;
	char shutdown;
	LINK_STATE_PHYS link_state_phys;
	char dev_name[__MAX_STR_LEN];
	void *dlrq;
	void *lnte;
} EAPOL;

static void REMOVE_FROM_LIST(EAPOL *eapol);
static int EAPOL_UNLOAD(void *p, void **release, char *argv[]);
extern IO_STUB EAPOL_IOCTL;
static void MONITOR_LINK_CHANGE(EAPOL *eapol, int get);
static void NOTIFY_LINK_CHANGE(EAPOL *eapol);

static __const__ HANDLE_OPERATIONS EAPOL_OPERATIONS = {
	SPL_X(SPL_NET),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	EAPOL_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	KERNEL$NO_OPERATION,	/* pktio */
};

static void EAPOL_SET_TIMEOUT(EAPOL *eapol, u_jiffies_lo_t timeout)
{
	eapol->eap_start_jiffies = EAP_START_MIN_JIFFIES;
	KERNEL$DEL_TIMER(&eapol->eap_start_timer);
	KERNEL$SET_TIMER(timeout, &eapol->eap_start_timer);
}

static void EAPOL_SEND_PACKET(EAPOL *eapol, __u8 type, void *data, unsigned length)
{
	PACKET *p;
	retry_alloc:
	ALLOC_PACKET(p, EAPOL_HEADER_SIZE, &NET$PKTPOOL, SPL_NET, {
		if (!NET$OOM()) goto retry_alloc;
		return;
	});
	p->fn = NET$FREE_PACKET;
	(*(long *)(p->sender_data = &eapol->outstanding))++;
	p->data_length = EAPOL_HEADER_SIZE;
	p->addrlen = eapol->type.addrlen;
	if (__unlikely(eapol->use_private_address)) memcpy(PKT_DSTADDR(p, p->addrlen), eapol->server_address, p->addrlen);
	else if (__likely(p->addrlen == ETHERNET_ADDRLEN)) memcpy(PKT_DSTADDR(p, p->addrlen), eapol_multicast, p->addrlen);
	else SET_BROADCAST_DST(p);
	memcpy(PKT_SRCADDR(p, eapol->type.addrlen), eapol->type.addr, eapol->type.addrlen);
	PKT_PROTOCOL(p) = NET_PROTOCOL_EAPOL;
	PKT2EAPOL(p)->version = EAPOL_VERSION;
	PKT2EAPOL(p)->type = type;
	PKT2EAPOL(p)->length = htons(length);
	p->v.ptr = (unsigned long)data;
	p->v.len = length;
	p->v.vspace = &KERNEL$VIRTUAL;
	p->h = eapol->h;
	p->id = IDTYPE_LO;
	CALL_IORQ(p, KERNEL$PKTIO);
}

static void EAP_SEND_PACKET(EAP *eap)
{
	EAPOL *eapol = GET_STRUCT(eap, EAPOL, eap);
	EAPOL_SEND_PACKET(eapol, EAPOL_TYPE_EAP, eapol->eap.packet, eapol->eap.packet_length);
}

static void EAPOL_INPUT(PACKET *p)
{
	static char a1[__MAX_STR_LEN], a2[__MAX_STR_LEN];
	EAPOL *eapol;
	XLIST_FOR_EACH(eapol, &EAPOL_IFACES, EAPOL, list) if (__likely(p->id == eapol->type.id)) goto found_iface;
	goto drop;
	found_iface:
	if (__likely(p->addrlen == ETHERNET_ADDRLEN) && __likely(!memcmp(PKT_DSTADDR(p, p->addrlen), eapol_multicast, ETHERNET_ADDRLEN))) goto address_match;
	if (__likely(p->addrlen == eapol->type.addrlen) && !memcmp(PKT_DSTADDR(p, p->addrlen), eapol->type.addr, p->addrlen)) {
		eapol->use_private_address = 1;
		memcpy(eapol->server_address, PKT_SRCADDR(p, p->addrlen), p->addrlen);
		goto address_match;
	}
	TEST_BROADCAST_PACKET(p, goto address_match);
	if (eapol->errorlevel >= 2)
		KERNEL$SYSLOG(__SYSLOG_NET_WARNING, eapol->dev_name, "UNKNOWN DESTINATION ADDRESS %s (SOURCE ADDRESS %s)", NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen));
	goto drop;
	address_match:
	if (__unlikely(p->data_length < EAPOL_HEADER_SIZE)) {
		if (eapol->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "TOO SHORT EAPOL PACKET, LENGTH %d, SRCADDR %s, DSTADDR %s", p->data_length, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
		goto drop;
			
	}
	if (__unlikely(PKT2EAPOL(p)->version != EAPOL_VERSION)) {
		if (eapol->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, eapol->dev_name, "BAD EAPOL VERSION %02X, SRCADDR %s, DSTADDR %s", PKT2EAPOL(p)->version, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
		goto drop;
	}
	if (__unlikely(ntohs(PKT2EAPOL(p)->length) + EAPOL_HEADER_SIZE > p->data_length)) {
		if (eapol->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "BAD EAPOL PACKET LENGTH, %d < %d, SRCADDR %s, DSTADDR %s, EAPOL TYPE %02X", p->data_length, ntohs(PKT2EAPOL(p)->length) + EAPOL_HEADER_SIZE, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), PKT2EAPOL(p)->type);
		goto drop;
	}
	p->data_length = ntohs(PKT2EAPOL(p)->length) + EAPOL_HEADER_SIZE;
	switch (PKT2EAPOL(p)->type) {
		default:
			if (eapol->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "UNKNOWN EAPOL TYPE %02X, SRCADDR %s, DSTADDR %s", PKT2EAPOL(p)->type, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
			goto drop;
		case EAPOL_TYPE_EAP:
			if (__unlikely(p->data_length < EAPOL_HEADER_SIZE + EAP_HEADER_SIZE)) {
				if (eapol->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "TOO SHORT EAP PACKET, LENGTH %d, SRCADDR %s, DSTADDR %s", p->data_length, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
				goto drop;
			}
			if (__unlikely(ntohs(PKT2EAP(p)->length) + EAPOL_HEADER_SIZE > p->data_length)) {
				if (eapol->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "BAD EAP PACKET LENGTH, %d < %d, SRCADDR %s, DSTADDR %s, EAP CODE %02X", p->data_length, ntohs(PKT2EAP(p)->length) + EAPOL_HEADER_SIZE, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), PKT2EAP(p)->code);
				goto drop;
			}
			p->data_length = ntohs(PKT2EAP(p)->length) + EAPOL_HEADER_SIZE;
			/*__debug_printf("eapol: %02x\n", PKT2EAP(p)->code);*/
			switch (PKT2EAP(p)->code) {
				default:
					if (eapol->errorlevel >= 1)
						KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "UNKNOWN EAP CODE %02X, SRCADDR %s, DSTADDR %s", PKT2EAP(p)->code, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
					goto drop;
				case EAP_CODE_REQUEST:
					if (eapol->state == STATE_CONNECTING || __unlikely(eapol->state == STATE_IDLE) || __unlikely(eapol->state == STATE_AUTHENTICATED)) {
						EAPOL_SET_TIMEOUT(eapol, EAP_AUTH_JIFFIES);
						EAP_RESET(&eapol->eap);
					}
					eapol->state = STATE_AUTHENTICATING;
					if (__unlikely(p->data_length < EAPOL_HEADER_SIZE + EAP_REQ_HEADER_SIZE)) {
						too_short_request_response:
						if (eapol->errorlevel >= 1)
							KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "TOO SHORT EAP PACKET, LENGTH %d, SRCADDR %s, DSTADDR %s, CODE %02X", p->data_length, NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen), PKT2EAP(p)->code);
						goto drop;
					}
	debug(("received request...: %02x, %x\n", PKT2EAP(p)->type, ntohs(PKT2EAP(p)->length)));
					EAP_REQUEST(&eapol->eap, PKT2EAP(p));
					goto drop;
				case EAP_CODE_RESPONSE:
					if (__unlikely(p->data_length < EAPOL_HEADER_SIZE + EAP_REQ_HEADER_SIZE)) goto too_short_request_response;
					goto for_server;
				case EAP_CODE_SUCCESS:
					if (__unlikely(eapol->state == STATE_CONNECTING) || __unlikely(eapol->state == STATE_IDLE)) {
						NOTIFY_LINK_CHANGE(eapol);
						/*
						if (eapol->errorlevel >= 1)
							KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "CANNED SUCCESS PACKET, SRCADDR %s, DSTADDR %s", NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
						*/
						goto drop;
					}
					if (__unlikely(eapol->state == STATE_AUTHENTICATED)) goto drop;
					EAP_SUCCESS(&eapol->eap);
					goto drop;
				case EAP_CODE_FAILURE:
					if (__unlikely(eapol->state == STATE_CONNECTING) || __unlikely(eapol->state == STATE_IDLE)) {
						goto drop;
					}
					if (__unlikely(eapol->state == STATE_AUTHENTICATED)) {
						/*
						this does happen on plug-unplug
						if (eapol->errorlevel >= 1)
							KERNEL$SYSLOG(__SYSLOG_NET_ERROR, eapol->dev_name, "FAILURE RECEIVED IN AUTHENTICATED STATE, SRCADDR %s, DSTADDR %s", NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
						*/
						EAPOL_SET_TIMEOUT(eapol, EAP_HOLD_JIFFIES);
						goto drop;
					}
					EAP_FAILURE(&eapol->eap);
					goto drop;
			}
		case EAPOL_TYPE_START:
		case EAPOL_TYPE_LOGOFF:
			goto for_server;
		case EAPOL_TYPE_KEY:
			goto drop;
		case EAPOL_TYPE_ASF_ALERT:
			if (eapol->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_NET_WARNING, eapol->dev_name, "ASF ALERT RECEIVED, SRCADDR %s, DSTADDR %s", NET$PRINT_HWADDR(a2, PKT_SRCADDR(p, p->addrlen), p->addrlen), NET$PRINT_HWADDR(a1, PKT_DSTADDR(p, p->addrlen), p->addrlen));
			goto drop;
	}
	for_server:
	drop:
	p->status = 0;
	CALL_PKT(p);
}

static void EAP_START_TIMEOUT(TIMER *t)
{
	EAPOL *eapol = GET_STRUCT(t, EAPOL, eap_start_timer);
	LOWER_SPL(SPL_NET);
	EAP_RESET(&eapol->eap);
	if (__unlikely(!EAP_HAVE_OFFERS(&eapol->eap)) || __unlikely(!(eapol->link_state_phys.flags & (LINK_STATE_UP | LINK_STATE_UNKNOWN)))) {
		VOID_LIST_ENTRY(&eapol->eap_start_timer.list);
		if (eapol->state != STATE_AUTHENTICATED) eapol->state = STATE_IDLE;
		return;
	}
	debug(("sent start...\n"));
	eapol->state = STATE_CONNECTING;
	KERNEL$SET_TIMER(eapol->eap_start_jiffies, &eapol->eap_start_timer);
	EAPOL_SEND_PACKET(eapol, EAPOL_TYPE_START, NULL, 0);
	eapol->eap_start_jiffies <<= 1;
	if (eapol->eap_start_jiffies > EAP_START_MAX_JIFFIES)
		eapol->eap_start_jiffies = EAP_START_MAX_JIFFIES;
}

static void EAP_PROGRESS_CALLBACK(EAP *eap)
{
	EAPOL *eapol = GET_STRUCT(eap, EAPOL, eap);
	EAPOL_SET_TIMEOUT(eapol, EAP_AUTH_JIFFIES);
}

static void EAP_END_CALLBACK(EAP *eap, int status)
{
	EAPOL *eapol = GET_STRUCT(eap, EAPOL, eap);
	if (status == EAP_END_SUCCESS) {
		eapol->state = STATE_AUTHENTICATED;
		NOTIFY_LINK_CHANGE(eapol);
		clr_timer:
		KERNEL$DEL_TIMER(&eapol->eap_start_timer);
		VOID_LIST_ENTRY(&eapol->eap_start_timer.list);
	} else if (status == EAP_END_IDLE) {
		eapol->state = STATE_IDLE;
		goto clr_timer;
	} else {
		eapol->state = STATE_CONNECTING;
		if (status == EAP_END_FAILURE_RETRY || EAP_HAVE_NONEMPTY_OFFERS(&eapol->eap)) EAPOL_SET_TIMEOUT(eapol, EAP_RETRY_JIFFIES);
		else EAPOL_SET_TIMEOUT(eapol, EAP_HOLD_JIFFIES);
	}
}

static void EAP_OFFER_CALLBACK(EAP *eap)
{
	EAPOL *eapol = GET_STRUCT(eap, EAPOL, eap);
	if (eapol->state == STATE_IDLE) {
		eapol->state = STATE_CONNECTING;
		EAPOL_SET_TIMEOUT(eapol, EAP_RETRY_JIFFIES);
	}
}

static void EAP_LOG_CALLBACK(EAP *eap, int level, __const__ char *msg)
{
	EAPOL *eapol = GET_STRUCT(eap, EAPOL, eap);
	if (level > eapol->errorlevel) return;
	KERNEL$SYSLOG(level >= 2 ? __SYSLOG_NET_WARNING : __SYSLOG_NET_ERROR, eapol->dev_name, "EAP ERROR: %s", msg);
}

static int EAP_STATUS_CALLBACK(EAP *eap, char msg[__MAX_STR_LEN])
{
	EAPOL *eapol = GET_STRUCT(eap, EAPOL, eap);
	if (eapol->state == STATE_IDLE) {
		_snprintf(msg, __MAX_STR_LEN, "IDLE");
	} else if (eapol->state == STATE_CONNECTING) {
		_snprintf(msg, __MAX_STR_LEN, "CONNECTING");
	} else if (eapol->state == STATE_AUTHENTICATING) {
		_snprintf(msg, __MAX_STR_LEN, "AUTHENTICATING");
	} else if (eapol->state == STATE_AUTHENTICATED) {
		_snprintf(msg, __MAX_STR_LEN, "AUTHENTICATED");
	} else KERNEL$SUICIDE("EAP_STATUS_CALLBACK: STATE %d", (int)eapol->state);
	return eapol->state;
}

DECL_IOCALL(EAPOL_IOCTL, SPL_NET, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	EAPOL *eapol;
	if (__unlikely(h->op != &EAPOL_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_NET));
	eapol = h->fnode;
	EAP_IOCTL(&eapol->eap, RQ);
	RETURN;
}

static void EAPOL_INIT_ROOT(HANDLE *h, void *eapol_)
{
	EAPOL *eapol = eapol_;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = eapol;
	h->op = &EAPOL_OPERATIONS;
}

DECL_AST(MCAST_IOCTL_AST, SPL_NET, IOCTLRQ)
{
	EAPOL *eapol = GET_STRUCT(RQ, EAPOL, mcast_ioctl);
	if (eapol->shutdown) {
		eapol->mcast_ioctl_posted = 0;
		RETURN;
	}
	eapol->mcast_ioctl.status = RQS_PROCESSING;
	WQ_WAIT(&KERNEL$LOGICAL_WAIT, &eapol->mcast_ioctl, KERNEL$IOCTL);
	RETURN;
}

DECL_AST(LINKCHG_IOCTL_AST, SPL_NET, IOCTLRQ)
{
	EAPOL *eapol = GET_STRUCT(RQ, EAPOL, linkchg_ioctl);
	eapol->linkchg_ioctl_posted = 0;
	if (eapol->shutdown) RETURN;
	if (__unlikely(eapol->linkchg_ioctl.status < 0)) {
		eapol->linkchg_ioctl_posted = 1;
		eapol->linkchg_ioctl.status = RQS_PROCESSING;
		WQ_WAIT(&KERNEL$LOGICAL_WAIT, &eapol->linkchg_ioctl, KERNEL$IOCTL);
		RETURN;
	}
	if (eapol->linkchg_ioctl.ioctl == IOCTL_IF_WAIT_LINK) {
		MONITOR_LINK_CHANGE(eapol, 1);
	} else {
		if (__unlikely(eapol->link_state_phys.flags & (LINK_STATE_UP | LINK_STATE_UNKNOWN))) {
			if (eapol->state == STATE_CONNECTING) {
				EAPOL_SET_TIMEOUT(eapol, EAP_LINKUP_JIFFIES);
			}
		}
		MONITOR_LINK_CHANGE(eapol, 0);
		/*{
			char buf[__MAX_STR_LEN];
			__debug_printf("link state: %s\n", NET$PRINT_LINK_STATE(buf, &eapol->link_state_phys));
		}*/
	}
	RETURN;
}

static void MONITOR_LINK_CHANGE(EAPOL *eapol, int get)
{
	if (__unlikely(eapol->linkchg_ioctl_posted)) return;
	eapol->linkchg_ioctl.fn = LINKCHG_IOCTL_AST;
	eapol->linkchg_ioctl.h = eapol->h;
	eapol->linkchg_ioctl.ioctl = get ? IOCTL_IF_GET_LINK : IOCTL_IF_WAIT_LINK;
	eapol->linkchg_ioctl.param = 0;
	eapol->linkchg_ioctl.v.ptr = (unsigned long)&eapol->link_state_phys;
	eapol->linkchg_ioctl.v.len = sizeof eapol->link_state_phys;
	eapol->linkchg_ioctl.v.vspace = &KERNEL$VIRTUAL;
	eapol->linkchg_ioctl_posted = 1;
	CALL_IORQ(&eapol->linkchg_ioctl, KERNEL$IOCTL);
}

DECL_AST(LINKNOTIFY_IOCTL_AST, SPL_NET, IOCTLRQ)
{
	EAPOL *eapol = GET_STRUCT(RQ, EAPOL, linknotify_ioctl);
	char v = eapol->linknotify_ioctl_posted;
	eapol->linknotify_ioctl_posted = 0;
	if (__unlikely(eapol->shutdown)) RETURN;
	if (__unlikely(v == 2)) NOTIFY_LINK_CHANGE(eapol);
	RETURN;
}

static void NOTIFY_LINK_CHANGE(EAPOL *eapol)
{
	if (__unlikely(eapol->linknotify_ioctl_posted)) {
		eapol->linknotify_ioctl_posted = 2;
		return;
	}
	eapol->linknotify_ioctl.fn = LINKNOTIFY_IOCTL_AST;
	eapol->linknotify_ioctl.h = eapol->h;
	eapol->linknotify_ioctl.ioctl = IOCTL_IF_SIGNAL_CHANGED_LINK;
	eapol->linknotify_ioctl.param = 0;
	eapol->linknotify_ioctl.v.ptr = 0;
	eapol->linknotify_ioctl.v.len = 0;
	eapol->linknotify_ioctl.v.vspace = &KERNEL$VIRTUAL;
	eapol->linknotify_ioctl_posted = 1;
	CALL_IORQ(&eapol->linknotify_ioctl, KERNEL$IOCTL);
}

int main(int argc, char **argv)
{
	int errorlevel = 0;
	char *if_name = NULL;
	char *identity = EAP_DEFAULT_IDENTITY;
	struct __param_table params[] = {
		"", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2, NULL,
		"IDENTITY", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		NULL, 0, 0, 0, NULL,
	};
	unsigned i, j;
	int r;
	int list_empty;
	MALLOC_REQUEST mrq;
	DEVICE_REQUEST devrq;
	OPENRQ openrq;
	IOCTLRQ io;
	EAPOL *eapol;
	char **arg = argv;
	int state = 0;

	PKT_CHECK_VERSION;

	params[0].__p = &if_name;
	params[1].__p = &errorlevel;
	params[2].__p = &errorlevel;
	params[3].__p = &identity;
	if (__unlikely(__parse_params(&arg, &state, params, NULL, NULL, NULL)) || __unlikely(!if_name)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "EAPOL: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}
	mrq.size = sizeof(EAPOL);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "EAPOL: COULD NOT ALLOCATE MEMORY: %s", strerror(-mrq.status));
		r = mrq.status;
		goto ret0;
	}
	eapol = mrq.ptr;
	memset(eapol, 0, sizeof(EAPOL));

	eapol->errorlevel = errorlevel;

	_snprintf(eapol->dev_name, __MAX_STR_LEN, "AUTH$EAP@%s", if_name);
	for (i = 0, j = 0; eapol->dev_name[i]; i++) {
		if (eapol->dev_name[i] != ':' &&
		    eapol->dev_name[i] != '/' &&
		    eapol->dev_name[i] != '=' &&
		    eapol->dev_name[i] != '.')
			eapol->dev_name[j++] = eapol->dev_name[i];
	}
	eapol->dev_name[j] = 0;

	eapol->state = STATE_IDLE;

	eapol->eap.send_packet = EAP_SEND_PACKET;
	eapol->eap.offer_callback = EAP_OFFER_CALLBACK;
	eapol->eap.progress_callback = EAP_PROGRESS_CALLBACK;
	eapol->eap.end_callback = EAP_END_CALLBACK;
	eapol->eap.log_callback = EAP_LOG_CALLBACK;
	eapol->eap.status_callback = EAP_STATUS_CALLBACK;
	strlcpy(eapol->eap.identity, identity, __MAX_STR_LEN);
	EAP_INIT(&eapol->eap);

	openrq.flags = O_RDWR;
	openrq.path = if_name;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (__unlikely(openrq.status < 0)) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "EAPOL: CAN'T OPEN INTERFACE %s: %s", if_name, strerror(-openrq.status));
		r = openrq.status;
		goto ret1;
	}
	eapol->h = openrq.status;

	io.h = eapol->h;
	io.ioctl = IOCTL_IF_GET_TYPE;
	io.param = 0;
	io.v.ptr = (unsigned long)&eapol->type;
	io.v.len = sizeof(eapol->type);
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "EAPOL: CAN'T GET TYPE ON INTERFACE %s: %s", if_name, strerror(-io.status));
		r = io.status;
		goto ret2;
	}

	eapol->eap_start_jiffies = EAP_START_MIN_JIFFIES;
	eapol->eap_start_timer.fn = EAP_START_TIMEOUT;
	VOID_LIST_ENTRY(&eapol->eap_start_timer.list);

	RAISE_SPL(SPL_NET);
	list_empty = XLIST_EMPTY(&EAPOL_IFACES);
	ADD_TO_XLIST(&EAPOL_IFACES, &eapol->list);
	LOWER_SPL(SPL_ZERO);
	if (list_empty) {
		if (__unlikely((r = NET$REGISTER_PROTOCOL(EAPOL_INPUT, NET_PROTOCOL_EAPOL)))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "EAPOL: COULD NOT REGISTER PROTOCOL EAPOL");
			goto ret3;
		}
	}

	devrq.name = eapol->dev_name;
	devrq.driver_name = "EAPOL.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = EAPOL_INIT_ROOT;
	devrq.dev_ptr = eapol;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = EAPOL_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (__unlikely(devrq.status < 0)) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", eapol->dev_name);
		r = devrq.status;
		goto ret4;
	}

	eapol->mcast_ioctl.fn = MCAST_IOCTL_AST;
	eapol->mcast_ioctl.h = eapol->h;
	eapol->mcast_ioctl.ioctl = IOCTL_IF_SET_MULTICAST;
	eapol->mcast_ioctl.param = PARAM_MULTICAST_ADDRESS;
	eapol->mcast_ioctl.v.ptr = (unsigned long)&eapol_multicast;
	eapol->mcast_ioctl.v.len = ETHERNET_ADDRLEN;
	eapol->mcast_ioctl.v.vspace = &KERNEL$VIRTUAL;
	eapol->mcast_ioctl_posted = 1;
	CALL_IORQ(&eapol->mcast_ioctl, KERNEL$IOCTL);
	MONITOR_LINK_CHANGE(eapol, 1);

	eapol->lnte = devrq.lnte;
	eapol->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), eapol->dev_name);
	return 0;

	ret4:
	REMOVE_FROM_LIST(eapol);
	ret3:
	KERNEL$DEL_TIMER(&eapol->eap_start_timer);
	ret2:
	close(eapol->h);
	ret1:
	EAP_DONE(&eapol->eap);
	KERNEL$UNIVERSAL_FREE(eapol);
	ret0:
	return r;
}

static void REMOVE_FROM_LIST(EAPOL *eapol)
{
	int list_empty;
	RAISE_SPL(SPL_NET);
	DEL_FROM_LIST(&eapol->list);
	list_empty = XLIST_EMPTY(&EAPOL_IFACES);
	LOWER_SPL(SPL_ZERO);
	if (list_empty) {
		NET$UNREGISTER_PROTOCOL(NET_PROTOCOL_EAPOL);
	}
}

static int EAPOL_UNLOAD(void *eapol_, void **release, char *argv[])
{
	IOCTLRQ io;
	int r;
	EAPOL *eapol = eapol_;
	if ((r = KERNEL$DEVICE_UNLOAD(eapol->lnte, argv))) return r;

	eapol->shutdown = 1;
	while (eapol->mcast_ioctl_posted) {
		KERNEL$CIO((IORQ *)&eapol->mcast_ioctl);
		KERNEL$SLEEP(1);
	}
	while (eapol->linkchg_ioctl_posted) {
		KERNEL$CIO((IORQ *)&eapol->linkchg_ioctl);
		KERNEL$SLEEP(1);
	}
	while (eapol->linknotify_ioctl_posted) {
		KERNEL$CIO((IORQ *)&eapol->linknotify_ioctl);
		KERNEL$SLEEP(1);
	}
	KERNEL$DEL_TIMER(&eapol->eap_start_timer);

	io.h = eapol->h;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	io.ioctl = IOCTL_IF_RESET_MULTICAST;
	io.param = PARAM_MULTICAST_ALL;
	SYNC_IO_CANCELABLE(&io, KERNEL$IOCTL);

	EAP_DONE(&eapol->eap);

	REMOVE_FROM_LIST(eapol);
	while (eapol->outstanding) KERNEL$SLEEP(1);
	close(eapol->h);
	*release = eapol->dlrq;
	KERNEL$UNIVERSAL_FREE(eapol);
	return 0;
}

