#include <STDLIB.H>
#include <UNISTD.H>
#include <SPAD/DEV.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <PTHREAD.H>

#include "EAP.H"

#include "ROOTER.H"

struct algorithm {
	char *name;
	__u32 eap_vendor;
	__u32 eap_type;
	int flags;
	int (*function)(struct instance *inst);
};

#define ALG_PLAIN_TEXT				0x01
#define ALG_HASH_PASSWORD			0x02
#define ALG_SERVER_CERTIFICATE			0x04
#define ALG_SERVER_CERTIFICATE_HASH_PASSWORD	0x08

struct algorithm algorithms[] = {
	"EAP-TTLS", 0, EAP_TYPE_TTLS, ALG_PLAIN_TEXT | ALG_SERVER_CERTIFICATE, eap_tls,		/* hash not yet supported */
	NULL, 0, 0, 0, NULL
};

static void destroy_instance(struct instance *inst)
{
	debug(("destroy instance\n"));
	lock_ssl();
	close(inst->h);
	free(inst->blacklist_config_seqs);
	DEL_FROM_LIST(&inst->list);
	free(inst);
	unlock_ssl();
}

static int algorithm_suitable_for_config_entry(struct instance *inst, struct algorithm *alg, struct config *cfg)
{
	unsigned i;
	for (i = 0; i < inst->n_blacklist_config_seqs; i++)
		if (cfg->seq == inst->blacklist_config_seqs[i]) return 0;
	if (!XLIST_EMPTY(&cfg->devices)) {
		struct stringlist *device;
		XLIST_FOR_EACH(device, &cfg->devices, struct stringlist, list) {
			char *d = strchr(device->string, ':');
			char *l = strchr(device->string, '/');
			if ((!d || !d[1]) && !l) {
				unsigned len = strcspn(device->string, ":");
				if (len < strlen(inst->name) && !_memcasecmp(device->string, inst->name, len) && inst->name[len] == ':') goto fount_dev;
			} else {
				if (!_strcasecmp(device->string, inst->name)) goto fount_dev;
			}
		}
		return 0;
		fount_dev:;
	}
	if (!XLIST_EMPTY(&cfg->algorithms)) {
		struct stringlist *alg_str;
		XLIST_FOR_EACH(alg_str, &cfg->algorithms, struct stringlist, list) if (!_strcasecmp(alg_str->string, alg->name)) goto found_alg;
		return 0;
		found_alg:;
	}
	if (cfg->cert_store && cfg->hash) {
		if (!(alg->flags & ALG_SERVER_CERTIFICATE_HASH_PASSWORD)) return 0;
	} else if (cfg->cert_store) {
		if (!(alg->flags & ALG_SERVER_CERTIFICATE)) return 0;
	} else if (cfg->hash) {
		if (!(alg->flags & ALG_HASH_PASSWORD)) return 0;
	}
	return 1;
}

static void blacklist_config_entry(struct instance *inst, struct config *c)
{
	unsigned i;
	debug(("blacklist entry: \"%s\"\n", c->user));
	for (i = 0; i < inst->n_blacklist_config_seqs; i++) if (__unlikely(inst->blacklist_config_seqs[i] == c->seq)) return;
	inst->blacklist_config_seqs = xrealloc(inst->blacklist_config_seqs, (inst->n_blacklist_config_seqs + 1) * sizeof(__u64));
	inst->blacklist_config_seqs[inst->n_blacklist_config_seqs++] = c->seq;
}

static void get_offers(struct instance *inst, EAP_OFFER_REQUEST *offer)
{
	unsigned i, j;
	memset(offer, 0, sizeof(EAP_OFFER_REQUEST));
	offer->magic = inst->magic;
	offer->offers_seq = inst->offers_seq;
	for (i = 0, j = 0; algorithms[i].name && j < EAP_MAX_OFFERS; i++) {
		struct config *cfg;
		lock_ssl();
		LIST_FOR_EACH(cfg, &config, struct config, list) if (algorithm_suitable_for_config_entry(inst, &algorithms[i], cfg)) {
			offer->offers[j].vendor = algorithms[i].eap_vendor;
			offer->offers[j].type = algorithms[i].eap_type;
			strncpy(offer->offers[j].name, algorithms[i].name, EAP_MAXNAMELEN);
			j++;
			break;
		}
		unlock_ssl();
	}
}

void *do_instance(void *inst_)
{
	struct instance *inst = inst_;
	EAP_OFFER_REQUEST offer;
	IOCTLRQ io;
	unsigned i;

	inst->blacklist_config_seqs = NULL;
	inst->n_blacklist_config_seqs = 0;
	inst->offers_seq = 0;
	get_offers_again:
	inst->magic = ((__u64)arc4random() << 32) | arc4random();
	inst->seq = -1;
	get_offers(inst, &offer);
	if (__likely(!offer.offers[0].vendor) && __unlikely(!offer.offers[0].type) && __unlikely(!inst->n_blacklist_config_seqs)) goto no_offers;
	io.h = inst->h;
	io.ioctl = IOCTL_EAP_OFFER_METHODS;
	io.param = priority;
	io.v.ptr = (unsigned long)&offer;
	io.v.len = sizeof offer;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (__likely(io.status == -ESTALE)) {
			inst->offers_seq = offer.offers_seq;
			__slow_free(inst->blacklist_config_seqs);
			inst->blacklist_config_seqs = NULL;
			inst->n_blacklist_config_seqs = 0;
	debug(("release blacklist\n"));
			goto get_offers_again;
		}
		goto no_offers;
	}
	for (i = 0; algorithms[i].name; i++) if (algorithms[i].eap_vendor == offer.offers[io.status].vendor && algorithms[i].eap_type == offer.offers[io.status].type) {
		int status, r, res;
		inst->vendor = algorithms[i].eap_vendor;
		inst->type = algorithms[i].eap_type;
		inst->algorithm = &algorithms[i];
		status = algorithms[i].function(inst);
		if (__likely(status == AUTH_SUCCESS)) {
			notify_eap_success:
			/* return status doesn't matter */
			send_final_packet(inst, &inst->packet, EAP_END_SUCCESS);
			goto get_offers_again;
		}
		if (__unlikely(status == AUTH_FAILURE)) {
			notify_eap_failure:
			res = EAP_END_FAILURE_RETRY;
			goto blacklist_config_line;
		}
		if (__unlikely(status == AUTH_PROTOCOL_VIOLATION)) {
			struct config *c;
			notify_eap_protocol_violation:
			res = EAP_END_PROTOCOL_VIOLATION;
			blacklist_config_line:
			lock_ssl();
			c = get_config(inst, NULL);
			if (__likely(c != NULL)) {
				blacklist_config_entry(inst, c);
			} else {
				LIST_FOR_EACH(c, &config, struct config, list) blacklist_config_entry(inst, c);
			}
			unlock_ssl();
			if (__likely(res == EAP_END_FAILURE_RETRY)) {
				get_offers(inst, &offer);
				if (__likely(!offer.offers[0].vendor) && __unlikely(!offer.offers[0].type)) res = EAP_END_FAILURE_HOLD;
			}
			/* return status doesn't matter */
			send_final_packet(inst, &inst->packet, res);
			goto get_offers_again;
		}
		if (__unlikely(status != AUTH_DONT_KNOW)) {
			KERNEL$SUICIDE("do_instance: INVALID STATUS RETURNED BY PROTOCOL: %d", status);
		}
		r = get_final_packet(inst, &inst->packet);
		if (__unlikely(r < 0)) goto notify_eap_protocol_violation;
		if (__likely(!r)) goto notify_eap_success;
		goto notify_eap_failure;
	}
	KERNEL$SUICIDE("do_instance: IOCTL_EAP_OFFER_METHODS RETURNED INVALID INDEX %ld", io.status);

	no_offers:
	destroy_instance(inst);
	return NULL;
}

struct config *get_config(struct instance *inst, int (*is_suitable)(struct instance *inst, struct config *config))
{
	assert_ssl_lock("get_config");
	if (inst->seq != -1) {
		struct config *c;
		LIST_FOR_EACH(c, &config, struct config, list) if (c->seq == inst->seq) return c;
		return NULL;
	} else {
		struct config *c;
		if (!is_suitable) return NULL;
		LIST_FOR_EACH(c, &config, struct config, list) {
			if (!algorithm_suitable_for_config_entry(inst, inst->algorithm, c)) continue;
			if (is_suitable(inst, c)) {
				inst->seq = c->seq;
				return c;
			}
		}
		return NULL;
	}
}

int get_general_packet(struct instance *inst, EAP_PACKET_REQUEST *packet)
{
	IOCTLRQ io;
	memset(packet, 0, sizeof(EAP_PACKET_REQUEST));
	packet->magic = inst->magic;
	packet->vendor = inst->vendor;
	packet->type = inst->type;
	io.h = inst->h;
	io.ioctl = IOCTL_EAP_GET_PACKET;
	io.param = priority;
	io.v.ptr = (unsigned long)packet;
	io.v.len = sizeof(EAP_PACKET_REQUEST);
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, inst->name, "ERROR GETTING PACKET: %s", strerror(-io.status));
		return io.status;
	}
	return 0;
}

int get_packet(struct instance *inst, EAP_PACKET_REQUEST *packet)
{
	int r;
	r = get_general_packet(inst, packet);
	if (__unlikely(r)) return r;
	if (__unlikely(packet->code != EAP_CODE_REQUEST)) {
		if (__unlikely(packet->code != EAP_CODE_FAILURE)) {
			if (errorlevel >= 1) {
				if (packet->code != EAP_CODE_SUCCESS) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, inst->name, "GOT PACKET WITH CODE %02X", packet->code);
				else KERNEL$SYSLOG(__SYSLOG_NET_ERROR, inst->name, "PREMATURE SUCCESS PACKET");
			}
		}
		return -EPROTO;
	}
	return 0;
}

int get_final_packet(struct instance *inst, EAP_PACKET_REQUEST *packet)
{
	int r;
	r = get_general_packet(inst, packet);
	if (__unlikely(r)) return r;
	if (__likely(packet->code == EAP_CODE_SUCCESS)) return 0;
	if (__likely(packet->code == EAP_CODE_FAILURE)) return 1;
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, inst->name, "EXPECTING SUCCESS OR FAILURE PACKET, GOT REQUEST PACKET");
	return -EPROTO;
}

int send_general_packet(struct instance *inst, EAP_PACKET_REQUEST *packet)
{
	IOCTLRQ io;
	packet->magic = inst->magic;
	packet->vendor = inst->vendor;
	packet->type = inst->type;
	io.h = inst->h;
	io.ioctl = IOCTL_EAP_SEND_PACKET;
	io.param = priority;
	io.v.ptr = (unsigned long)packet;
	io.v.len = sizeof(EAP_PACKET_REQUEST);
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_NET_WARNING, inst->name, "ERROR SENDING PACKET: %s", strerror(-io.status));
		return io.status;
	}
	return 0;
}

int send_packet(struct instance *inst, EAP_PACKET_REQUEST *packet)
{
	packet->code = EAP_CODE_RESPONSE;
	return send_general_packet(inst, packet);
}

int send_final_packet(struct instance *inst, EAP_PACKET_REQUEST *packet, unsigned code)
{
	memset(packet, 0, sizeof(EAP_PACKET_REQUEST));
	packet->code = code;
	return send_general_packet(inst, packet);
}
