#include <SYS/TYPES.H>
#include <SPAD/SLAB.H>
#include <SPAD/DL.H>
#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <SPAD/ALLOC.H>
#include <STDLIB.H>
#include <SPAD/ENV.H>
#include <ARCH/IPCHECKS.H>
#include <SPAD/SOCKET.H>
#include <VALUES.H>

#include <SPAD/PKT.H>
#include "NET.H"

#define RESERVED_PACKETS	(mem >= 67108864 ? 128 : mem >= 25165824 ? 16 : 8)
#define MIN_PKT_SIZE		9000

__const__ int NET$PKT_VERSION = PKT_VERSION;
__const__ int NET$PKT_SIZE = sizeof(PACKET);

struct __slhead NET$_PACKET_SLABS[_PACKET_SLOTS];
unsigned NET$MAX_PACKET_LENGTH;
long NET$MEMORY_LIMIT = MAXLONG;
long NET$MEMORY_AVAIL = MAXLONG;

int NET$PKTPOOL_RESERVE(PKTPOOL *pool)
{
	PACKET *p;
	int i, spl;
	__u64 mem = KERNEL$GET_MEMORY_SIZE(VM_TYPE_WIRED_MAPPED);
	if (__unlikely(pool->length > NET$MAX_PACKET_LENGTH)) return -EMSGSIZE;
	for (i = RESERVED_PACKETS; i > 0; i--) {
		again:
		spl = KERNEL$SPL;
		__pkt_slalloc(p, pool->length, spl, {
			TEST_SPLX(SPL_X(SPL_ZERO), spl);
			__pkt_slalloc(p, pool->length, spl, {
				int r;
				if (__unlikely(r = NET$MEMWAIT_SYNC())) {
					NET$PKTPOOL_FREE(pool);
					return r;
				}
				goto again;
			});
		});
		*(PACKET **)p = pool->freelist;
		pool->freelist = p;
		pool->n_total++;
		TEST_LOCKUP_SYNC;
	}
	return 0;
}

void NET$PKTPOOL_FREE(PKTPOOL *pool)
{
	PACKET *p;
	int spl;
	while ((p = pool->freelist)) {
		pool->freelist = *(PACKET **)p;
		pool->n_total--;
		spl = KERNEL$SPL;
		__pkt_slfree(p, spl);
	}
	if (__unlikely(pool->n_total | pool->n_allocated)) KERNEL$SUICIDE("NET$PKTPOOL_FREE: POOL IS FREE, BUT HAS TOTAL COUNT %u, ALLOCATED COUNT %u (PACKET LENGTH %u)", pool->n_total, pool->n_allocated, pool->length);
	WQ_WAKE_ALL(&pool->freepkt_wait);
}

static void NO_UNMAP(void *p)
{
}

static __u8 *MAP_INDIRECT(VBUF *vbuf, VDESC *v, vspace_unmap_t **unmap);
static __u8 *MAP_INDIRECT_COPY(VBUF *vbuf, VDESC *v);

__u8 *NET$MAP_INDIRECT_PACKET(VBUF *vbuf, VDESC *v, vspace_unmap_t **unmap)
{
	*unmap = NO_UNMAP;
	if (__likely(!v->len)) return NULL;
	return MAP_INDIRECT(vbuf, v, unmap);
}

static __u8 *MAP_INDIRECT(VBUF *vbuf, VDESC *v, vspace_unmap_t **unmap)
{
	__u8 *a;
#if __DEBUG >= 1
	if (__unlikely(vbuf->spl != KERNEL$SPL))
		KERNEL$SUICIDE("NET$MAP_INDIRECT_PACKET IBUF SPL (%08X) DOES NOT MATCH KERNEL$SPL (%08X)", vbuf->spl, KERNEL$SPL);
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), KERNEL$SPL)))
		KERNEL$SUICIDE("NET$MAP_INDIRECT_PACKET AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_VSPACE);
	a = v->vspace->op->vspace_map(v, PF_READ, unmap);
	LOWER_SPLX(vbuf->spl);
	if (__unlikely(!a)) return __ERR_PTR(-EVSPACEFAULT);
	if (__unlikely(__PTR_ERR(a) != -EINVAL)) return a;
	return MAP_INDIRECT_COPY(vbuf, v);
}

static __u8 *MAP_INDIRECT_COPY(VBUF *vbuf, VDESC *v)
{
	long c;
	if (__unlikely(v->len > vbuf->len)) return __ERR_PTR(-EMSGSIZE);
	c = 0;
	while (v->len != 0) {
		long s;
		RAISE_SPL(SPL_VSPACE);
		s = v->vspace->op->vspace_get(v, vbuf);
		if (__unlikely(!s)) {
			v->ptr -= c;
			v->len += c;
			vbuf->ptr = (char *)vbuf->ptr - c;
			vbuf->len += c;
			return __ERR_PTR(-EVSPACEFAULT);
		}
		vbuf->ptr = (char *)vbuf->ptr + s;
		vbuf->len -= s;
		c += s;
	}
	v->ptr -= c;
	v->len += c;
	vbuf->len += c;
	return vbuf->ptr = (char *)vbuf->ptr - c;
}

int NET$CHECKSUM_PACKET(PACKET *p)
{
	checksum_t checksum;
	vspace_unmap_t *unmap;
	__u8 *ind;
	static __u8 aux_indirect[MAX_PACKET_LENGTH_TOP];
	static VBUF v = { aux_indirect, MAX_PACKET_LENGTH_TOP, SPL_X(SPL_NET) };
	if (__unlikely(!(p->flags & PKT_OUTPUT_CHECKSUM))) return 0;
	checksum = NET$IP_CHECKSUM(0, p->data + LL_HEADER + CHECKSUM_FROM(p), p->data_length - CHECKSUM_FROM(p));
	if (__unlikely(p->v.len != 0)) {
		ind = NET$MAP_INDIRECT_PACKET(&v, &p->v, &unmap);
		if (__unlikely(__IS_ERR(ind))) {
			p->flags |= PKT_PAGING_ERROR;
			return __PTR_ERR(ind);
		}
		if (__unlikely(p->data_length & 1)) {
			__u8 x[2];
			x[0] = 0;
			x[1] = ind[2];
			checksum = NET$IP_CHECKSUM(checksum, x, 2);
			checksum = NET$IP_CHECKSUM(checksum, ind + 1, p->v.len - 1);
		} else {
			checksum = NET$IP_CHECKSUM(checksum, ind, p->v.len);
		}
		RAISE_SPL(SPL_VSPACE);
		unmap(ind);
		LOWER_SPL(SPL_NET);
	}
	*(__u16 *)&p->data[LL_HEADER + CHECKSUM_POS(p)] = IP_CHECKSUM_FOLD_INVERT(checksum);
	p->flags &= ~PKT_OUTPUT_CHECKSUM;
	return 0;
}

static void SET_MEMORY_LIMIT(ENV_CALLBACK *e)
{
	__u64 l;
	int spl;
	char *m = getenv("@NET$MEMORY_LIMIT");
	if (__likely(!m) || __unlikely(__get_64_number(m, m + strlen(m), 0, &l))) {
		__u64 ll = KERNEL$GET_MEMORY_SIZE(VM_TYPE_WIRED_UNMAPPED) >> 2;
		l = KERNEL$GET_MEMORY_SIZE(VM_TYPE_WIRED_MAPPED) >> 1;
		if (ll < l) l = ll;
	}
	l >>= __PAGE_CLUSTER_BITS;
	if (__unlikely(l > MAXLONG)) l = MAXLONG;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_PACKET);
	NET$MEMORY_AVAIL += l - NET$MEMORY_LIMIT;
	NET$MEMORY_LIMIT = l;
	LOWER_SPLX(spl);
}

PKTPOOL NET$PKTPOOL;
PKTPOOL NET$PKTPOOL_LONG_TERM;

static ENV_CALLBACK env = { { NULL, NULL }, &SET_MEMORY_LIMIT };

int PACKET_INIT(void)
{
	int r;
	int i;
	LOWER_SPL(SPL_DEV);
	KERNEL$REGISTER_ENV_CALLBACK(&env);
	RAISE_SPL(SPL_NET);
	SET_MEMORY_LIMIT(NULL);
	for (i = 0; i < _PACKET_SLOTS; i++) {
		static char pkt_msg[_PACKET_SLOTS][20];
		static long NO_MEMORY = 0;
		if (KERNEL$SLAB_ENTRIES_PER_PAGE(sizeof(PACKET) + _SLOT_SIZE(i), __CPU_CACHELINE_ALIGN) >= (i > _PACKET_SLOT(MIN_PKT_SIZE) ? 2 : 1)) {
			_snprintf(pkt_msg[i], 20, "NET$PACKET-%d", _SLOT_SIZE(i));
			KERNEL$SLAB_INIT(&NET$_PACKET_SLABS[i], sizeof(PACKET) + _SLOT_SIZE(i), __CPU_CACHELINE_ALIGN, VM_TYPE_WIRED_MAPPED, NULL, NULL, &NET$MEMORY_AVAIL, pkt_msg[i]);
			NET$MAX_PACKET_LENGTH = _SLOT_SIZE(i) - _PKT_REDZONE_BYTES;
		} else {
			_snprintf(pkt_msg[i], 20, "NET$EMPTY-%d", _SLOT_SIZE(i));
			KERNEL$SLAB_INIT(&NET$_PACKET_SLABS[i], 1, 0, VM_TYPE_WIRED_MAPPED, NULL, NULL, &NO_MEMORY, pkt_msg[i]);
		}
		TEST_LOCKUP_SYNC;
	}
	INIT_PKTPOOL(&NET$PKTPOOL, NET$MAX_PACKET_LENGTH);
	INIT_PKTPOOL(&NET$PKTPOOL_LONG_TERM, NET$MAX_PACKET_LENGTH);
	if (__unlikely(r = NET$PKTPOOL_RESERVE(&NET$PKTPOOL)) || 
	    __unlikely(r = NET$PKTPOOL_RESERVE(&NET$PKTPOOL_LONG_TERM))) {
		PACKET_DONE();
		return r;
	}
	return 0;
}

void PACKET_DONE(void)
{
	int i;
	NET$PKTPOOL_FREE(&NET$PKTPOOL);
	NET$PKTPOOL_FREE(&NET$PKTPOOL_LONG_TERM);
	for (i = 0; i < _PACKET_SLOTS; i++)
		KERNEL$SLAB_DESTROY(&NET$_PACKET_SLABS[i]);
	LOWER_SPL(SPL_DEV);
	KERNEL$UNREGISTER_ENV_CALLBACK(&env);
	RAISE_SPL(SPL_NET);
}

