#include <SPAD/PCI.H>
#include <SPAD/PKT.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/ALLOC.H>
#include <SPAD/ETHERNET.H>
#include <SPAD/PKT.H>
#include <SPAD/IOCTL.H>
#include <STRING.H>
#include <VALUES.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>

#include "3C.H"
#include "3CREG.H"

static __const__ struct pci_id_s pci_cards[] = {
	{ 0x10B7, 0x9000, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900-TPO ETHERLINK XL", BOOMERANG },
	{ 0x10B7, 0x9001, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900-COMBO ETHERLINK XL", BOOMERANG },
	{ 0x10B7, 0x9004, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-TPO ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9005, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-COMBO ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9006, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-TPC ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x900A, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C900B-FL ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9050, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905-TX FAST ETHERLINK XL", BOOMERANG | MII },
	{ 0x10B7, 0x9051, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905-T4 FAST ETHERLINK XL", BOOMERANG | MII },
	{ 0x10B7, 0x9055, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-TX FAST ETHERLINK XL", CYCLONE | NWAY | MII_PREAMBLE },
	{ 0x10B7, 0x9056, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-T4 FAST ETHERLINK XL", CYCLONE | NWAY | MII_PREAMBLE },
	{ 0x10B7, 0x9058, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-COMBO FAST ETHERLINK XL", CYCLONE | NWAY },
	{ 0x10B7, 0x905A, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905B-FX/SC FAST ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9200, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C905C-TX FAST ETHERLINK XL", TORNADO | NWAY | MII_PREAMBLE },
	{ 0x10B7, 0x9201, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C920B-EMB INTEGRATED FAST ETHERLINK XL", TORNADO | NWAY },
	{ 0x10B7, 0x9202, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C920B-EMB-WNM INTEGRATED FAST ETHERLINK XL", TORNADO | MII },
	{ 0x10B7, 0x9210, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C920B-EMB-WNM INTEGRATED FAST ETHERLINK XL", TORNADO | NWAY },
	{ 0x10B7, 0x9800, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C980 FAST ETHERLINK XL", CYCLONE },
	{ 0x10B7, 0x9805, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C980C FAST ETHERLINK XL", CYCLONE | NWAY },
	{ 0x10B7, 0x7646, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3CSOHO100-TX OFFICECONNECT", CYCLONE | NWAY },
	{ 0x10B7, 0x4500, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C450-TX HOMECONNECT", TORNADO | NWAY },
	{ 0x10B7, 0x5055, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C555 FAST ETHERLINK XL", CYCLONE | EEPROM_8BIT },
	{ 0x10B7, 0x6055, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C556 FAST ETHERLINK XL", TORNADO | NWAY | EEPROM_8BIT | CARDBUS | INVERT_MII_PWR | NO_EEPROM_RELOAD },
	{ 0x10B7, 0x6056, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C556B FAST ETHERLINK XL", TORNADO | NWAY | EEPROM_OFFSET_30 | CARDBUS | INVERT_MII_PWR | WNO_XCVR_PWR | NO_EEPROM_RELOAD },
	{ 0x10B7, 0x5B57, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575TX FAST ETHERLINK XL", BOOMERANG | MII | EEPROM_8BIT },
	{ 0x10B7, 0x5057, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575TX FAST ETHERLINK XL", BOOMERANG | MII | EEPROM_8BIT },
	{ 0x10B7, 0x5157, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575B FAST ETHERLINK XL", CYCLONE | NWAY | CARDBUS | EEPROM_8BIT | INVERT_LED_PWR },
	{ 0x10B7, 0x5257, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C575C FAST ETHERLINK XL", TORNADO | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | MAX_COLLISION_RESET },
	{ 0x10B7, 0x6560, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656 FAST ETHERLINK XL", CYCLONE | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | INVERT_LED_PWR },
	{ 0x10B7, 0x6562, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656B FAST ETHERLINK XL", CYCLONE | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | INVERT_LED_PWR },
	{ 0x10B7, 0x6564, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656C FAST ETHERLINK XL", TORNADO | NWAY | CARDBUS | EEPROM_8BIT | INVERT_MII_PWR | MAX_COLLISION_RESET },
	{ 0x10B7, 0x6564, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C656C FAST ETHERLINK XL", 0 },
	{ 0x10B7, 0x1201, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C982 HYDRA PORT A", TORNADO | NWAY },
	{ 0x10B7, 0x1202, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "3COM 3C982 HYDRA PORT B", TORNADO | NWAY },
	{ 0, }
};

#define MEDIA_V2		1
#define MEDIA_NEED_MII		2
#define MEDIA_INTERNAL_MII	4
#define MEDIA_NEED_COAX_CMD	8
#define MEDIA_CAPABLE_FDX	16
#define MEDIA_LINK_DETECT	32
#define MEDIA_100		64

static __const__ struct {
	char *name;
	int flags;
	unsigned internal_config;
	unsigned media_options;
	unsigned media_bits;
} media[] = {
	{
		"AUTO-NEGOTIATION",
		MEDIA_V2 | MEDIA_NEED_MII | MEDIA_INTERNAL_MII | MEDIA_CAPABLE_FDX | MEDIA_LINK_DETECT,
		X2_InternalConfig_XcvrSelect_ANE,
		X1_W3_ResetOptions_BaseTXAvailable,
		XL_W4_MediaStatus_LinkBeatEnable
	}, {
		"10BASE-T",
		MEDIA_CAPABLE_FDX | MEDIA_LINK_DETECT,
		X1_InternalConfig_XcvrSelect_10BASET,
		X2_W3_MediaOptions_10bTAvailable,
		XL_W4_MediaStatus_JabberGuardEnable | XL_W4_MediaStatus_LinkBeatEnable
	}, {
		"100BASE-FX",
		MEDIA_CAPABLE_FDX | MEDIA_LINK_DETECT | MEDIA_100,
		X1_InternalConfig_XcvrSelect_100BASEFX,
		X2_W3_MediaOptions_BaseFXAvailable,
		XL_W4_MediaStatus_LinkBeatEnable
	}, {
		"10BASE5",
		MEDIA_CAPABLE_FDX,
		X1_InternalConfig_XcvrSelect_10AUI,
		X2_W3_MediaOptions_AuiAvailable,
		XL_W4_MediaStatus_EnableSqeStats
	}, {
		"10BASE2",
		MEDIA_NEED_COAX_CMD,
		X1_InternalConfig_XcvrSelect_10BASE2,
		X2_W3_MediaOptions_CoaxAvailable,
		0
	}, {
		"MII",
		MEDIA_NEED_MII | MEDIA_CAPABLE_FDX,
		X1_InternalConfig_XcvrSelect_MII,
		X2_W3_MediaOptions_MiiConnector | X2_W3_MediaOptions_BaseT4Available,
		0
	}, {
		"MII-EXTERNAL",
		MEDIA_V2 | MEDIA_NEED_MII | MEDIA_CAPABLE_FDX,
		X2_InternalConfig_XcvrSelect_MII_EXT,
		X2_W3_MediaOptions_MiiConnector | X2_W3_MediaOptions_BaseT4Available,
		0
	},
};

#define N_MEDIA			(sizeof(media) / sizeof(*media))

#define CLEAR_MEDIA_BITS	(XL_W4_MediaStatus_EnableSqeStats | XL_W4_MediaStatus_JabberGuardEnable | XL_W4_MediaStatus_LinkBeatEnable)

#define V2(xl)	(__likely((xl)->flags & (CYCLONE | TORNADO)))

static __u8 XL_READ_8(XL *xl, unsigned i)
{
	if (__likely(!(xl->flags & FLAG_PIO))) return mmio_inb(xl->u.mem + i);
	else return io_inb((xl)->u.io + i);
}

static __u16 XL_READ_16(XL *xl, unsigned i)
{
	if (__likely(!(xl->flags & FLAG_PIO))) return mmio_inw(xl->u.mem + i);
	else return io_inw((xl)->u.io + i);
}

static __u32 XL_READ_32(XL *xl, unsigned i)
{
	if (__likely(!(xl->flags & FLAG_PIO))) return mmio_inl(xl->u.mem + i);
	else return io_inl((xl)->u.io + i);
}

static void XL_WRITE_8(XL *xl, unsigned i, __u8 v)
{
	if (__likely(!(xl->flags & FLAG_PIO))) return mmio_outb(xl->u.mem + i, v);
	else return io_outb((xl)->u.io + i, v);
}

static void XL_WRITE_16(XL *xl, unsigned i, __u16 v)
{
	if (__likely(!(xl->flags & FLAG_PIO))) return mmio_outw(xl->u.mem + i, v);
	else return io_outw((xl)->u.io + i, v);
}

static void XL_WRITE_32(XL *xl, unsigned i, __u32 v)
{
	if (__likely(!(xl->flags & FLAG_PIO))) return mmio_outl(xl->u.mem + i, v);
	else return io_outl((xl)->u.io + i, v);
}

static int FIND_MEDIA_BY_INTERNAL_CONFIG(XL *xl, __u32 internal_config)
{
	int i;
	__u32 media_type = internal_config & (!V2(xl) ? X1_InternalConfig_XcvrSelect : X2_InternalConfig_XcvrSelect);
	for (i = 0; i < N_MEDIA; i++) {
		if ((V2(xl) || !(media[i].flags & MEDIA_V2)) && media[i].internal_config == media_type) return i;
	}
	return -1;
}

static int AUTOSELECT_MEDIA(XL *xl, int last_media)
{
	int wrap = 0;
	int i = last_media;
	while (1) {
		i++;
		if (__unlikely(i >= N_MEDIA)) {
			i = 0, wrap++;
			if (__unlikely(wrap == 2)) return -1;
		}
		if (xl->media_options & media[i].media_options) return i;
	}
}

#include "3CFN.I"

#include "MII.I"

static __finline__ void MII_DELAY(XL *xl)
{
	XL_READ_16(xl, XL_W4_PhysicalMgmt);
}

static void MII_SYNC(XL *xl)
{
	int i;
	for (i = 0; i < 32; i++) {
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtData | XL_W4_PhysicalMgmt_MgmtDir);
		MII_DELAY(xl);
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtData | XL_W4_PhysicalMgmt_MgmtDir | XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
}

__u16 MII_READ(void *p, unsigned idx)
{
	unsigned cmd, ret;
	int i;
	XL *xl = p;
	if (__unlikely(xl->phy_id < 0)) return 0xffff;
	XL_WINDOW(xl, 4);
	if (__unlikely(xl->flags & MII_PREAMBLE_RUNTIME)) MII_SYNC(xl);
	cmd = (0xf6 << 10) | (xl->phy_id << 5) | idx;
	for (i = 14; i >= 0; i--) {
		unsigned val = (cmd & (1 << i) ? XL_W4_PhysicalMgmt_MgmtData : 0) | XL_W4_PhysicalMgmt_MgmtDir;
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, val);
		MII_DELAY(xl);
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, val | XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
	ret = 0;
	for (i = 19; i > 0; i--) {
		unsigned rd;
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, 0);
		MII_DELAY(xl);
		rd = XL_READ_16(xl, XL_W4_PhysicalMgmt);
		ret = (ret << 1) | !!(rd & XL_W4_PhysicalMgmt_MgmtData);
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
	if (__unlikely(ret & 0x20000)) return 0xffff;
	return ret >> 1;
}

void MII_WRITE(void *p, unsigned idx, __u16 val)
{
	unsigned cmd;
	int i;
	XL *xl = p;
	if (__unlikely(xl->phy_id < 0)) return;
	XL_WINDOW(xl, 4);
	if (__unlikely(xl->flags & MII_PREAMBLE_RUNTIME)) MII_SYNC(xl);
	cmd = 0x50020000 | (xl->phy_id << 23) | (idx << 18) | val;
	for (i = 31; i >= 0; i--) {
		unsigned val = (cmd & (1 << i) ? XL_W4_PhysicalMgmt_MgmtData : 0) | XL_W4_PhysicalMgmt_MgmtDir;
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, val);
		MII_DELAY(xl);
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, val | XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
	for (i = 1; i >= 0; i--) {
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, 0);
		MII_DELAY(xl);
		XL_WRITE_16(xl, XL_W4_PhysicalMgmt, XL_W4_PhysicalMgmt_MgmtClk);
		MII_DELAY(xl);
	}
}

static int MII_PROBE(XL *xl)
{
	__u16 bmsr;
	xl->flags |= MII_PREAMBLE_RUNTIME;
	bmsr = MII_READ(xl, MII_BMSR);
	if (!bmsr || bmsr == 0xffff) return 0;
	if (!(xl->flags & MII_PREAMBLE) && bmsr & 0x40) xl->flags &= ~MII_PREAMBLE_RUNTIME;
	return 1;
}

static void SET_MAC_CONTROL(XL *xl, int fdx);
static long XL_RESET_THREAD(void *xl_);
extern AST_STUB XL_RESET_THREAD_DONE;
static void XL_CHECK_MEDIA(XL *xl);

void XL_EXTRACT_ADDRESS(XL *xl, unsigned offset)
{
	xl->address[0] = xl->eeprom[offset] >> 8;
	xl->address[1] = xl->eeprom[offset];
	xl->address[2] = xl->eeprom[offset + 1] >> 8;
	xl->address[3] = xl->eeprom[offset + 1];
	xl->address[4] = xl->eeprom[offset + 2] >> 8;
	xl->address[5] = xl->eeprom[offset + 2];
}

static void XL_RESET_1(XL *xl)
{
	XL_WAIT(xl, NULL);
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_SetInterruptEnable);
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_GlobalReset | (xl->flags & NO_EEPROM_RELOAD ? XL_Cmd_GlobalReset_AismReset : 0) | XL_Cmd_GlobalReset_NetworkReset);
}

static int XL_RESET_2(XL *xl)
{
	KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);
	if (XL_WAIT(xl, "RESET")) return 1;
	return 0;
}

static void XL_RESET_3(XL *xl, int loading)
{
	if (__likely(!(xl->flags & MEDIA_UNKNOWN))) {
		__u32 internal_config;
		XL_WINDOW(xl, 3);
		internal_config = XL_READ_32(xl, XL_W3_InternalConfig);
		if (!V2(xl)) internal_config &= ~X1_InternalConfig_XcvrSelect;
		else internal_config &= ~X2_InternalConfig_XcvrSelect;
		internal_config |= media[xl->media].internal_config;
		XL_WRITE_32(xl, XL_W3_InternalConfig, internal_config);
	}
	if (__unlikely(xl->flags & CARDBUS)) {
		__u16 ro;
		XL_WINDOW(xl, 2);
		ro = XL_READ_16(xl, X2_W2_ResetOptions) & ~(X2_W2_ResetOptions_CB_LedPwr | X2_W2_ResetOptions_CB_MiiPwr);
		if (!!(xl->flags & INVERT_MII_PWR) ^ (loading == -1)) ro |= X2_W2_ResetOptions_CB_MiiPwr;
		if (!!(xl->flags & INVERT_LED_PWR) ^ (loading == -1)) ro |= X2_W2_ResetOptions_CB_LedPwr;
		XL_WRITE_16(xl, X2_W2_ResetOptions, ro);
		if (xl->flags & WNO_XCVR_PWR) {
			XL_WINDOW(xl, 0);
			XL_WRITE_16(xl, 0, 0x0800);
		}
		KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);
	}
	if (__likely(!(xl->flags & MEDIA_UNKNOWN))) {
		__u16 media_status;
		XL_WINDOW(xl, 4);
		media_status = XL_READ_16(xl, XL_W4_MediaStatus);
		media_status &= ~CLEAR_MEDIA_BITS;
		media_status |= media[xl->media].media_bits;
		XL_WRITE_16(xl, XL_W4_MediaStatus, media_status);
		if (__unlikely(media[xl->media].flags & MEDIA_NEED_COAX_CMD)) {
			XL_WRITE_16(xl, XL_IntStatusCmd, X2_Cmd_EnableDcConverter);
			KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);
			XL_WAIT(xl, "ENABLE DC CONVERTER");
		}
	}
	xl->phy_id = -1;
	if (__unlikely(xl->flags & MEDIA_UNKNOWN) || media[xl->media].flags & MEDIA_NEED_MII /*|| xl->flags & NWAY*/) {
		int i;
		if (__likely(!(xl->flags & MEDIA_UNKNOWN)) && media[xl->media].flags & MEDIA_INTERNAL_MII) {
			xl->phy_id = INTERNAL_PHY_ID;
			if (__likely(MII_PROBE(xl))) goto have_mii;
		}
		for (i = 0; i < N_PHY; i++) {
			xl->phy_id = i;
			if (__likely(MII_PROBE(xl))) goto have_mii;
		}
		xl->phy_id = -1;
		have_mii:;
	}
	SET_MAC_CONTROL(xl, xl->bmcr & BMCR_FULLDPLX && (xl->flags & MEDIA_UNKNOWN || media[xl->media].flags & MEDIA_CAPABLE_FDX));
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_RxReset | XL_Cmd_RxReset_NetworkRxReset);
	KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);
	XL_WAIT(xl, "RX RESET");
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_TxReset | XL_Cmd_TxReset_NetworkTxReset);
	KERNEL$SLEEP(JIFFIES_PER_SECOND / 1000 + 1);
	XL_WAIT(xl, "TX RESET");
	KERNEL$SLEEP(JIFFIES_PER_SECOND / 10 + 1);
	if (xl->phy_id != -1) {
		MII_WRITE(xl, MII_BMCR, xl->bmcr);

		/*__debug_printf("bmcr: %04x\n", MII_READ(xl, MII_BMCR));
		MII_WRITE(xl, MII_ADVERTISE, 0x5e1);
		__debug_printf("advertise: %04x\n", MII_READ(xl, MII_ADVERTISE));
		__debug_printf("expansion: %04x\n", MII_READ(xl, MII_EXPANSION));
		__debug_printf("quick status: %04x\n", MII_READ(xl, 31));*/
	}
	SET_MAC_CONTROL(xl, xl->bmcr & BMCR_FULLDPLX && (xl->flags & MEDIA_UNKNOWN || media[xl->media].flags & MEDIA_CAPABLE_FDX));
}

static void SET_MAC_CONTROL(XL *xl, int fdx)
{
	__u16 mac_ctrl;
	XL_WINDOW(xl, 3);
	mac_ctrl = XL_READ_16(xl, XL_W3_MacControl) & ~(XL_W3_MacControl_FullDuplexEnable | X2_W3_MacControl_FlowControlEnable | XL_W3_MacControl_DeferExtendedEnable | XL_W3_MacControl_DeferTimerSelect | X2_W3_MacControl_ExtendAfterCollision);
	if (fdx) {
		xl->flags |= FULL_DUPLEX_ENABLED;
		mac_ctrl |= XL_W3_MacControl_FullDuplexEnable;
		if (V2(xl) && xl->flags & FLOW_CONTROL) {
			mac_ctrl |= X2_W3_MacControl_FlowControlEnable;
		}
	} else {
		xl->flags &= ~FULL_DUPLEX_ENABLED;
		if (xl->defer_timer) {
			mac_ctrl |= XL_W3_MacControl_DeferExtendedEnable;
			mac_ctrl |= ((xl->defer_timer >> 4) << __BSF_CONST(XL_W3_MacControl_DeferTimerSelect)) & XL_W3_MacControl_DeferTimerSelect;
		}
		if (V2(xl) && xl->flags & DEFER_ONLY_COLLISION) {
			mac_ctrl |= X2_W3_MacControl_ExtendAfterCollision;
		}
	}
	XL_WRITE_16(xl, XL_W3_MacControl, mac_ctrl);
}

static void XL_READ_EEPROM(XL *xl)
{
	unsigned i, w;
	__u16 eeprom_cmd, checksum;
	if (xl->flags & EEPROM_8BIT) eeprom_cmd = (XL_W0_EepromCommand_ReadRegister << 2) | XL_W0_EepromCommand_DataOFfset;
	else if (xl->flags & EEPROM_OFFSET_30) eeprom_cmd = XL_W0_EepromCommand_ReadRegister | XL_W0_EepromCommand_DataOFfset;
	else eeprom_cmd = XL_W0_EepromCommand_ReadRegister;
	XL_WINDOW(xl, 0);
	for (i = 0; i < EEPROM_SIZE; i++) {
		XL_WRITE_16(xl, XL_W0_EepromCommand, eeprom_cmd + i);
		for (w = 0; w < 15; w++) {
			KERNEL$UDELAY(162);
			if (!(XL_READ_16(xl, XL_W0_EepromCommand) & XL_W0_EepromCommand_EepromBusy)) break;
		}
		xl->eeprom[i] = XL_READ_16(xl, XL_W0_EepromData);
	}
	checksum = 0;
	for (i = 0; i < 0x18; i++) checksum ^= xl->eeprom[i];
	checksum ^= checksum >> 8;
	checksum &= 0xff;
	if (checksum) {
		for (i = 0x18; i < 0x21; i++) checksum ^= xl->eeprom[i];
		checksum ^= checksum >> 8;
		checksum &= 0xff;
	}
	if (checksum && !(xl->flags & TORNADO))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "INVALID EEPROM CHECKSUM");
}

static void XL_CONTINUE_RESET(XL *xl)
{
	unsigned i;
	unsigned int_mask;
	XL_WINDOW(xl, 2);
	for (i = 0; i < ETHERNET_ADDRLEN; i++)
		XL_WRITE_8(xl, XL_W2_StationAddressLo + i, xl->address[i]);
	for (i = 0; i < ETHERNET_ADDRLEN; i += 2)
		XL_WRITE_16(xl, XL_W2_StationMaskLo + i, 0);
	
	for (i = 0; i < N_RX_DESCS; i++) {
		RX_DESC(xl, i)->upd.UpNextPtr = __32CPU2LE(xl->desc_dmaaddr + RX_DESCS_OFFSET + ((i + 1) & (N_RX_DESCS - 1)) * SIZEOF_XL_RXDESC);
		RX_DESC(xl, i)->upd.UpPktStatus = __32CPU2LE(XL_UPS_UpComplete);
		RX_DESC(xl, i)->frags[0].UpFragLen = __32CPU2LE(PKT_SIZE | XL_UFL_UpFragLast);
	}
	for (i = 0; i < xl->n_recv; i++) {
		RX_DESC(xl, (xl->first_recv + i) & (N_RX_DESCS - 1))->upd.UpPktStatus = __32CPU2LE(0);
	}
	XL_DESTROY_SENT_PACKETS(xl);
	xl->first_sent = 0;
	xl->n_sent = 0;
	if (!V2(xl)) XL_WRITE_8(xl, X1_DM_TxFreeThresh, (PKT_SIZE + 255) >> 8);
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_SetTxStartThresh | xl->tx_thresh);
	if (V2(xl)) XL_WRITE_16(xl, XL_IntStatusCmd, X2_Cmd_SetTxReclaimThresh | xl->tx_reclaim_thresh);
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_SetRxEarlyThresh | ((PKT_SIZE + 3) >> 2));

	XL_UPDATE_FILTER(xl);

	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_DnStall);
	XL_WAIT(xl, "DOWNLOAD STALL");
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_UpStall);
	XL_WAIT(xl, "UPLOAD STALL");

	XL_WRITE_32(xl, XL_DM_DmaCtrl, XL_DM_DmaCtrl_UpEarlyRxEnable);
	if (V2(xl) && !(xl->eeprom[EE_XL_Software_Information_2] & EE_XL_Software_Information_2_FixedMWIBug)) {
		XL_WRITE_32(xl, XL_DM_DmaCtrl, XL_READ_32(xl, XL_DM_DmaCtrl) | X2_DM_DmaCtrl_DefeatMWI);
	}

	if (XMIT_POLLING(xl)) {
		TX_DESC(xl, N_TX_DESCS - 1)->upd.DnNextPtr = __32CPU2LE(0);
		TX_DESC(xl, N_TX_DESCS - 1)->upd.FrameStartHeader = __32CPU2LE(X2_FSH_DpdEmpty);
		XL_WRITE_32(xl, XL_DM_DnListPtr, xl->desc_dmaaddr + TX_DESCS_OFFSET + (N_TX_DESCS - 1) * SIZEOF_XL_TXDESC);
	} else {
		XL_WRITE_32(xl, XL_DM_DnListPtr, 0);
	}
	XL_WRITE_32(xl, XL_DM_UpListPtr, xl->desc_dmaaddr + RX_DESCS_OFFSET + xl->first_recv * SIZEOF_XL_RXDESC);

	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_AcknowledgeInterrupt | XL_Cmd_AcknowledgeInterrupt_InterruptLatchAck | XL_Cmd_AcknowledgeInterrupt_RxEarlyAck | XL_Cmd_AcknowledgeInterrupt_IntRequestedAck | XL_Cmd_AcknowledgeInterrupt_DnCompleteAck | XL_Cmd_AcknowledgeInterrupt_UpCompleteAck);

	int_mask = XL_IntStatus_HostError | XL_IntStatus_TxComplete | XL_IntStatus_DnComplete | XL_IntStatus_UpComplete;
	xl->flags &= ~LINK_INTERRUPT;
	if (V2(xl) && __likely(!(xl->flags & (NO_LINK_INTR | MEDIA_UNKNOWN))) && __likely(media[xl->media].flags & MEDIA_INTERNAL_MII) && __likely(xl->phy_id == INTERNAL_PHY_ID)) {
		int_mask |= XL_IntStatus_LinkEvent;
		xl->flags |= LINK_INTERRUPT;
	}
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_SetIndicationEnable | int_mask);
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_SetInterruptEnable | int_mask);

	if (RECV_POLLING(xl)) XL_WRITE_8(xl, X2_DM_UpPoll, RECV_POLL_INTERVAL);
	if (XMIT_POLLING(xl)) XL_WRITE_8(xl, X2_DM_DnPoll, XMIT_POLL_INTERVAL);

	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_DnUnStall);
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_UpUnStall);
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_TxEnable);
	XL_WAIT(xl, "TX ENABLE");
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_RxEnable);
	XL_WAIT(xl, "RX ENABLE");
	THE_PCMCIA_PEOPLE_ARE_IDIOTS(xl);

	XL_CHECK_MEDIA(xl);

	KERNEL$DEL_TIMER(&xl->media_timer);
	KERNEL$SET_TIMER(INITIAL_LINK_EXPIRE_TIME > LINK_CHECK_INITIAL_TIME ? LINK_CHECK_INITIAL_TIME : INITIAL_LINK_EXPIRE_TIME, &xl->media_timer);

	xl->flags &= ~(RECEIVED_PACKET | RECEIVED_PACKET_2);
	xl->change_media_time = KERNEL$GET_JIFFIES() + INITIAL_LINK_EXPIRE_TIME;
}

void XL_UPDATE_FILTER(XL *xl)
{
	unsigned i, j;
	unsigned filter = XL_Cmd_SetRxFilter_ReceiveIndividual | XL_Cmd_SetRxFilter_ReceiveBroadcast;
	if (__unlikely((xl->flags & (MEDIA_AUTOSELECT | RECEIVED_PACKET)) == MEDIA_AUTOSELECT) && xl->phy_id == -1 && !(media[xl->media].flags & MEDIA_LINK_DETECT)) goto promisc;
	if (__unlikely(!WQ_EMPTY(&xl->mcast_table[MCAST_PROMISC]))) {
		promisc:
		filter |= XL_Cmd_SetRxFilter_ReceiveAllFrames | XL_Cmd_SetRxFilter_ReceiveMulticast;
		xl->mcast_state = STATUS_MULTICAST_PROMISC;
		goto set_filter;
	}
	if (__unlikely(!WQ_EMPTY(&xl->mcast_table[MCAST_ALL]))) {
		all_mc:
		filter |= XL_Cmd_SetRxFilter_ReceiveMulticast;
		xl->mcast_state = STATUS_MULTICAST_ALL;
		goto set_filter;
	}
	xl->mcast_state = STATUS_MULTICAST_NONE;
	for (i = 0; i < N_HW_MCAST; i++) if (__unlikely(!WQ_EMPTY(&xl->mcast_table[i]))) {
		if (!V2(xl)) goto all_mc;
		if (!(filter & X2_Cmd_SetRxFilter_ReceiveMulticastHash)) {
/* Temporarily receive all multicasts, while we clear and reinitialize the hash.
We don't know if the hash is 6-bit or 8-bit, so we must do it this way. */
			XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_SetRxFilter | filter | XL_Cmd_SetRxFilter_ReceiveMulticast);
			for (j = 0; j < N_HW_MCAST; j++) {
				XL_WRITE_16(xl, XL_IntStatusCmd, X2_Cmd_ClearHashFilterBit | j);
			}
			xl->mcast_state = STATUS_MULTICAST_SOME;
		}
		filter |= X2_Cmd_SetRxFilter_ReceiveMulticastHash;
		XL_WRITE_16(xl, XL_IntStatusCmd, X2_Cmd_SetHashFilterBit | i);
	}
	set_filter:
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_SetRxFilter | filter);
}

static int XL_INIT(XL *xl)
{
	char a1[ETHERNET_ADDRESS_STR_LEN];

	xl->tx_thresh = XL_INIT_TX_THRESH;
	xl->tx_reclaim_thresh = XL_INIT_TX_RECLAIM_THRESH;

	XL_RESET_1(xl);
	if (XL_RESET_2(xl)) return -EIO;
	XL_READ_EEPROM(xl);

	if (!(xl->flags & ADDRESS_OVERRIDE)) {
		XL_EXTRACT_ADDRESS(xl, EE_XL_OEM_Node_Address_0);
		if (__unlikely(!NET$VALID_ETHERNET_ADDRESS(xl->address))) {
			XL_EXTRACT_ADDRESS(xl, EE_XL_3Com_Node_Address_0);
			if (__unlikely(!NET$VALID_ETHERNET_ADDRESS(xl->address))) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "INVALID ETHERNET ADDRESS: %s", NET$PRINT_ETHERNET_ADDRESS(a1, xl->address));
				return -EIO;
			}
		}
	}
	/*{
		int i;
		for (i = 0; i < 0x40; i++) {
			__debug_printf("%04x ", xl->eeprom[i]);
		}
		__debug_printf("\n");
	}*/
	return 0;
}

static void XL_RESET_DEQUEUE(XL *xl)
{
	if (__unlikely(xl->flags & (FLAG_RESETTING | FLAG_DEAD))) return;
	xl->flags |= FLAG_RESETTING;
	XL_RESET_1(xl);
	xl->reset_thread.fn = XL_RESET_THREAD_DONE;
	xl->reset_thread.thread_main = XL_RESET_THREAD;
	xl->reset_thread.p = xl;
	xl->reset_thread.error = NULL;
	xl->reset_thread.cwd = NULL;
	xl->reset_thread.std_in = -1;
	xl->reset_thread.std_out = -1;
	xl->reset_thread.std_err = -1;
	xl->reset_thread.dlrq = NULL;
	xl->reset_thread.spawned = 0;
	/* xl->reset_thread.thread already set */
	CALL_IORQ(&xl->reset_thread, KERNEL$THREAD);
}

static long XL_RESET_THREAD(void *xl_)
{
	XL *xl = xl_;
	RAISE_SPL(SPL_XL);
	XL_RESET_2(xl);
	XL_RESET_3(xl, 1);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

DECL_AST(XL_RESET_THREAD_DONE, SPL_XL, THREAD_RQ)
{
	XL *xl = GET_STRUCT(RQ, XL, reset_thread);
	XL_CONTINUE_RESET(xl);
	if (__unlikely(xl->flags & FLAG_RESETTING_INTR)) {
		xl->irq_ctrl.eoi();
	}
	xl->flags &= ~(FLAG_RESETTING | FLAG_RESETTING_INTR);
	if (__unlikely(xl->flags & FLAG_PIO)) PIO_DEQUEUE(xl);
	else MMIO_DEQUEUE(xl);
	RETURN;
}

static void XL_CHECK_MEDIA(XL *xl)
{
	LINK_STATE_PHYS ls;
	if (xl->flags & FLAG_DEAD) {
		MII_GET_MODE(NULL, &ls);
	} else {
		if (xl->phy_id != -1) {
			MII_GET_MODE(xl, &ls);
			if (ls.desc && xl->media_options & X2_W3_MediaOptions_BaseT4Available) {
				memset(ls.desc, 0, sizeof ls.desc);
				strlcpy(ls.desc, "100BASE-T4", sizeof ls.desc);
			}
			SET_MAC_CONTROL(xl, ls.flags & LINK_STATE_FULL_DUPLEX);
	/* Tx/Rx should be reset here, but neither Linux nor FreeBSD do it.
	   Unfortunatelly, the manual is unclear about what bits should be set
	   in reset mask (resetting the whole card is not good idea, because it
	   would change media status and loop back here).
	   Don't reset for now, and see if someone reports problems.
	*/
		} else {
			memset(&ls, 0, sizeof ls);
			if (xl->flags & MEDIA_UNKNOWN) {
				ls.flags |= LINK_STATE_UNKNOWN;
			} else {
				strlcpy(ls.desc, media[xl->media].name, sizeof(ls.desc));
				if (media[xl->media].flags & MEDIA_LINK_DETECT) {
					__u16 media_status;
					XL_WINDOW(xl, 4);
					media_status = XL_READ_16(xl, XL_W4_MediaStatus);
					if (media_status & XL_W4_MediaStatus_LinkDetect) ls.flags |= LINK_STATE_UP;
				} else {
					ls.flags |= LINK_STATE_UNKNOWN;
				}
				if (media[xl->media].flags & MEDIA_100) {
					ls.speed = 100000000;
				} else {
					ls.speed = 10000000;
				}
			}
			if (xl->flags & FULL_DUPLEX_ENABLED) ls.flags |= LINK_STATE_FULL_DUPLEX;
			else ls.flags |= LINK_STATE_HALF_DUPLEX;
		}
		if (xl->flags & MEDIA_AUTOSELECT) ls.flags |= LINK_STATE_MEDIA_AUTOSELECT;
	}
	if (__unlikely(memcmp(&xl->link_state, &ls, sizeof(LINK_STATE_PHYS)))) {
		memcpy(&xl->link_state, &ls, sizeof(LINK_STATE_PHYS));
		WQ_WAKE_ALL_PL(&xl->link_state_wait);
	}
}

static void AUTOSELECT_LINK_CHANGE(XL *xl)
{
	if (__unlikely(xl->flags & FLAG_DEAD)) return;
	xl->media = AUTOSELECT_MEDIA(xl, xl->media);
	XL_RESET_DEQUEUE(xl);
}

static void XL_MEDIA_TIMER(TIMER *t)
{
	XL *xl = GET_STRUCT(t, XL, media_timer);
	u_jiffies_lo_t j, tm;
	LOWER_SPL(SPL_XL);
	j = KERNEL$GET_JIFFIES_LO();
	if (j - xl->n_sent_decay >= N_SENT_DECAY_TIME) {
		xl->n_sent_decay = j;
		if (xl->max_sent > DEFAULT_SENT) xl->max_sent--;
	}
	tm = LINK_CHECK_TIME(xl);
	if (__unlikely(xl->flags & FLAG_RESETTING)) goto skip_check;
	XL_CHECK_MEDIA(xl);
	if (!(xl->flags & MEDIA_AUTOSELECT)) goto skip_check;
	if (__likely(xl->link_state.flags & LINK_STATE_UP)) {
		xl->change_media_time = j + MONITORED_LINK_EXPIRE_TIME;
	} else if (__unlikely(!(xl->link_state.flags & LINK_STATE_UNKNOWN))) {
		if ((jiffies_lo_t)(j - xl->change_media_time) >= 0) {
			AUTOSELECT_LINK_CHANGE(xl);
			goto skip_check;
		}
	} else if (__unlikely(!(xl->flags & RECEIVED_PACKET))) {
		if ((jiffies_lo_t)(j - xl->change_media_time) >= 0) {
			AUTOSELECT_LINK_CHANGE(xl);
			goto skip_check;
		}
	} else {
		if (__likely(xl->flags & RECEIVED_PACKET_2)) {
			xl->flags &= ~RECEIVED_PACKET_2;
			xl->change_media_time = j + UNMONITORED_LINK_PROMISC_TIME;
		} else if ((jiffies_lo_t)(j - xl->change_media_time) >= 0) {
			xl->flags &= ~RECEIVED_PACKET;
			XL_UPDATE_FILTER(xl);
			xl->change_media_time = j + UNMONITORED_LINK_EXPIRE_TIME;
		}
	}
	if (xl->change_media_time - j < tm) tm = xl->change_media_time - j;
	skip_check:
	KERNEL$SET_TIMER(tm, &xl->media_timer);
}

int XL_IRQ_SPECIAL(XL *xl, __u16 status)
{
	if (__unlikely(status == 0xffff)) {
		xl->flags |= FLAG_DEAD;
		return 1;
	}
	if (__likely(status & XL_IntStatus_LinkEvent)) {
		MII_READ(xl, MII_EXPANSION);	/* ack for 3C905B */
		XL_CHECK_MEDIA(xl);
		XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_AcknowledgeInterrupt | XL_Cmd_AcknowledgeInterrupt_InterruptLatchAck | X3_Cmd_AcknowledgeInterrupt_LinkEventAck);	/* ack for 3C905C */
	}
	if (__unlikely(status & XL_IntStatus_HostError)) {
		__u32 dmactrl = XL_READ_32(xl, XL_DM_DmaCtrl);
		if ((dmactrl & (XL_DM_DmaCtrl_TargetAbort | XL_DM_DmaCtrl_MasterAbort)) == XL_DM_DmaCtrl_TargetAbort) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, xl->dev_name, "PCI TARGET ABORT");
		if ((dmactrl & (XL_DM_DmaCtrl_TargetAbort | XL_DM_DmaCtrl_MasterAbort)) == XL_DM_DmaCtrl_MasterAbort) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, xl->dev_name, "PCI MASTER ABORT");
		if (dmactrl != 0xffffffff) KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "HOST ERROR, DMACTRL %08X", dmactrl);
		reset_it:
		XL_RESET_DEQUEUE(xl);
		return 1;
	}
	if (__likely(status & XL_IntStatus_TxComplete)) {
		int r = 0;
		__u8 tx_status = XL_READ_8(xl, XL_DM_TxStatus);
		XL_WRITE_8(xl, XL_DM_TxStatus, 0);
		if (tx_status & XL_DM_TxStatus_TxReclaimError) {
			if (xl->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "OUT OF WINDOW COLLISION");
			r = 1;
		}
		if (tx_status & XL_DM_TxStatus_TxStatusOverflow) {
			if (xl->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_HW_WARNING, xl->dev_name, "TX STATUS STACK OVERFLOW");
			r = 1;
		}
		if (tx_status & XL_DM_TxStatus_MaxCollisions) {
			if (xl->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_NET_WARNING, xl->dev_name, "EXCESSIVE COLLISIONS");
			r = 1;
		}
		if (tx_status & XL_DM_TxStatus_TxUnderrun) {
			if (xl->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_HW_WARNING, xl->dev_name, "TX FIFO UNDERRUN");
			r = 1;
		}
		if (tx_status & XL_DM_TxStatus_TxJabber) {
			if (xl->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "JABBER DETECTED");
			r = 1;
		}
		if (tx_status & XL_DM_TxStatus_InterruptRequested) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "UNEXPECTED INTERRUPT REQUESTED");
			r = 1;
		}
		if (__unlikely(!r)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "UNKNOWN TX ERROR, STATUS %02X", tx_status);
			goto reset_it;
		}
		if (tx_status & XL_DM_TxStatus_TxUnderrun) {
			xl->tx_thresh += XL_INCREMENT_TX_THRESH;
			if (xl->tx_thresh > (PKT_SIZE + 3) / 4) xl->tx_thresh = (PKT_SIZE + 3) / 4;
		}
		if (tx_status & XL_DM_TxStatus_TxReclaimError) {
			xl->tx_reclaim_thresh += XL_INCREMENT_TX_RECLAIM_THRESH;
			if (xl->tx_reclaim_thresh > XL_MAX_TX_RECLAIM_THRESH) xl->tx_reclaim_thresh = XL_MAX_TX_RECLAIM_THRESH;
			if (V2(xl)) {
				XL_WRITE_16(xl, XL_IntStatusCmd, X2_Cmd_SetTxReclaimThresh | xl->tx_reclaim_thresh);
				XL_WAIT(xl, "SET TX RECLAIM THRESH");
			}
		}
		if (tx_status & (XL_DM_TxStatus_TxUnderrun | XL_DM_TxStatus_TxJabber)) goto reset_it;
		if (tx_status & XL_DM_TxStatus_MaxCollisions && xl->flags & MAX_COLLISION_RESET) {
			XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_TxReset | XL_Cmd_TxReset_FifoTxReset | XL_Cmd_TxReset_DnTxReset);
			KERNEL$UDELAY(1000);
			XL_WAIT(xl, "TX RESET (EXCEPT FIFO AND DOWNLOAD)");
		}
		XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_TxEnable);
		XL_WAIT(xl, "TX ENABLE");
	}
	return 0;
}

void XL_RECV_ERROR(XL *xl, __u32 up_status)
{
	int r = 0;
	up_status = __32LE2CPU(up_status);
	if (up_status & XL_UPS_UpOverrun) {
		if (xl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_HW_WARNING, xl->dev_name, "RECEIVE FIFO OVERRUN");
		r = 1;
	}
	if (up_status & XL_UPS_RuntFrame) {
		if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "RUNT PACKET");
		r = 1;
	}
	if (up_status & XL_UPS_AlignmentError) {
		if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "ALIGNMENT ERROR");
		r = 1;
	}
	if (up_status & XL_UPS_CrcError) {
		if (xl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "CRC ERROR");
		r = 1;
	}
	if (up_status & (XL_UPS_OversizedFrame | XL_UPS_UpOverflow)) {
		if (xl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, xl->dev_name, "TOO LONG PACKET");
		r = 1;
	}
	if (__unlikely(!r)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "UNKNOWN RECEIVE ERROR, STATUS %08X", up_status);
	}
}

void XL_TX_TIMEOUT(XL *xl)
{
	if (__unlikely(XL_READ_16(xl, XL_IntStatusCmd) == 0xffff)) xl->flags |= FLAG_DEAD;
	if (!(xl->flags & FLAG_DEAD))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, xl->dev_name, "TRANSMIT TIMEOUT");
	XL_RESET_DEQUEUE(xl);
}

DECL_IOCALL(XL_IOCTL, SPL_XL, IOCTLRQ)
{
	int r;
	HANDLE *h = RQ->handle;
	XL *xl;
	if (__unlikely(h->op != &PIO_OPERATIONS) && __unlikely(h->op != &MMIO_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_XL));
	xl = h->fnode;
	if (__unlikely(xl->flags & FLAG_RESETTING)) {
		WQ_WAIT_F(&xl->link_state_wait, RQ);
		RETURN;
	}
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			ETHERNET_FILL_IF_TYPE(&t);
			t.id = IDTYPE_PCI | (xl->id & IDTYPE_MASK);
			memcpy(t.addr, xl->address, ETHERNET_ADDRLEN);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_WAIT_LINK: {
			LINK_STATE ls;
			if (!RQ->v.len) {
				WQ_WAIT(&xl->link_state_wait, RQ, KERNEL$SUCCESS);
				XL_CHECK_MEDIA(xl);
				RETURN;
			}
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, &ls, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			XL_CHECK_MEDIA(xl);
			if (!memcmp(&xl->link_state, &ls, RQ->v.len)) {
				WQ_WAIT_F(&xl->link_state_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_LINK: {
			XL_CHECK_MEDIA(xl);
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &xl->link_state, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SIGNAL_CHANGED_LINK: {
			xl->link_state.seq++;
			WQ_WAKE_ALL_PL(&xl->link_state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_MULTICAST: {
			if (__unlikely(RQ->v.len != 0)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			RQ->status = xl->mcast_state;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SET_MULTICAST: {
			__u8 mcast[ETHERNET_ADDRLEN];
			unsigned bit;
			int need_refresh;
			if (__unlikely(RQ->param == PARAM_MULTICAST_PROMISC)) {
				bit = MCAST_PROMISC;
				goto have_bit;
			} else if (__unlikely(RQ->param == PARAM_MULTICAST_ALL)) {
				bit = MCAST_ALL;
				goto have_bit;
			}
			if (__unlikely(RQ->param != PARAM_MULTICAST_ADDRESS)) {
				RQ->status = -ENOOP;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, mcast, ETHERNET_ADDRLEN)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			bit = NET$ETHERNET_CRC(mcast, ETHERNET_ADDRLEN) & (N_HW_MCAST - 1);
			have_bit:
			need_refresh = WQ_EMPTY(&xl->mcast_table[bit]);
			WQ_WAIT_F(&xl->mcast_table[bit], RQ);
			if (need_refresh) XL_UPDATE_FILTER(xl);
			RETURN;
		}
		case IOCTL_IF_RESET_MULTICAST: {
			XL_UPDATE_FILTER(xl);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
	RETURN;
}

static void XL_DESTROY_RECV_PACKETS(XL *xl)
{
	RAISE_SPL(SPL_XL);
	while (xl->n_recv) {
		PACKET *p = RX_DESC(xl, xl->first_recv)->p;
		XL_UNMAP_RECV_PACKET(xl, p, xl->first_recv);
		FREE_PACKET(p, NULL, SPL_XL);
		xl->first_recv = (xl->first_recv + 1) & (N_RX_DESCS - 1);
		xl->n_recv--;
	}
	LOWER_SPL(SPL_ZERO);
#if __DEBUG >= 2
	if (__unlikely(xl->recv_maps))
		KERNEL$SUICIDE("XL_DESTROY_RECV_PACKETS: RECEIVE MAPS LEAKED: %d", xl->recv_maps);
#endif
}

void XL_DESTROY_SENT_PACKETS(XL *xl)
{
	while (xl->n_sent) {
		PACKET *p;
		XL_UNMAP_SENT_PACKET(xl, TX_DESC(xl, xl->first_sent));
		p = TX_DESC(xl, xl->first_sent)->p;
		p->status = -EAGAIN;
		CALL_PKT(p);
		xl->first_sent = (xl->first_sent + 1) & (N_RX_DESCS - 1);
		xl->n_sent--;
	}
}

static void XL_INIT_ROOT(HANDLE *h, void *data)
{
	XL *xl = data;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = xl;
	h->op = xl->flags & FLAG_PIO ? &PIO_OPERATIONS : &MMIO_OPERATIONS;
}

static int XL_UNLOAD(void *p, void **release, char *argv[]);
static void XL_DESTROY_RECV_PACKETS(XL *xl);

int main(int argc, char *argv[])
{
	int errorlevel = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	char *net_input = NET_INPUT_DEVICE ":";
	char *address_str = NULL;
	int pio = 0;
	char *media_string = NULL;
	int bmcr = BMCR_ANENABLE;
	int defer_timer = 0;
	int extra_flags = 0;
	char *opt, *optend, *str;
	int r;
	char dev_name[__MAX_STR_LEN];
	char a1[__MAX_STR_LEN];
	char *chip_name;
	unsigned long flags;
	__u8 *mem = NULL;	/* against warning */
	io_t io = 0;		/* against warning */
	int irq;
	CONTIG_AREA_REQUEST car;
	OPENRQ openrq;
	DEVICE_REQUEST devrq;
	VDESC desc;
	VDMA dma;
	char *e;
	int i;
	XL *xl;
	struct __param_table params[] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2, NULL,
		"INPUT", __PARAM_STRING, 1, MAXINT, NULL,
		"ADDRESS", __PARAM_STRING, 1, MAXINT, NULL,
		"PIO", __PARAM_BOOL, ~0, 1, NULL,
		"MEDIA", __PARAM_STRING, 1, MAXINT, NULL,
		"FULL_DUPLEX", __PARAM_BOOL, BMCR_ANENABLE | BMCR_FULLDPLX, BMCR_FULLDPLX, NULL,
		"10M", __PARAM_BOOL, BMCR_ANENABLE | BMCR_SPEED100, 0, NULL,
		"100M", __PARAM_BOOL, BMCR_ANENABLE | BMCR_SPEED100, BMCR_SPEED100, NULL,
		"AUTO", __PARAM_BOOL, BMCR_ANENABLE | BMCR_SPEED100 | BMCR_FULLDPLX, BMCR_ANENABLE, NULL,
		"DEFER_TIMER", __PARAM_INT, 0, (XL_W3_MacControl_DeferTimerSelect >> __BSF_CONST(XL_W3_MacControl_DeferTimerSelect) << 4) + 15 + 1, NULL,
		"DEFER_ONLY_AFTER_COLLISION", __PARAM_BOOL, DEFER_ONLY_COLLISION, DEFER_ONLY_COLLISION, NULL,
		"PAUSE_RECEIVE", __PARAM_BOOL, FLOW_CONTROL, FLOW_CONTROL, NULL,
		"NO_LINK_INTERRUPT", __PARAM_BOOL, NO_LINK_INTR, NO_LINK_INTR, NULL,
		"NO_TX_CHECKSUM", __PARAM_BOOL, NO_TX_CHECKSUM, NO_TX_CHECKSUM, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **arg = argv;
	int state = 0;

	PKT_CHECK_VERSION;

	params[0].__p = &errorlevel;
	params[1].__p = &errorlevel;
	params[2].__p = &net_input;
	params[3].__p = &address_str;
	params[4].__p = &pio;
	params[5].__p = &media_string;
	params[6].__p = &bmcr;
	params[7].__p = &bmcr;
	params[8].__p = &bmcr;
	params[9].__p = &bmcr;
	params[10].__p = &defer_timer;
	params[11].__p = &extra_flags;
	params[12].__p = &extra_flags;
	params[13].__p = &extra_flags;
	params[14].__p = &extra_flags;

	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}

	if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	_snprintf(dev_name, sizeof(dev_name), "PKT$3C@" PCI_ID_FORMAT, id);

	if (!(flags & (CYCLONE | TORNADO))) pio = 1;
	if (!pio) {
		PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
		mem = PCI$MAP_MEM_RESOURCE(id, 1, REGSPACE(flags));
		if (!mem) pio = 2;
		if (__IS_ERR(mem)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(mem)));
			r = __PTR_ERR(mem);
			goto ret1;
		}
	}
	if (pio) {
		PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
		io = PCI$READ_IO_RESOURCE(id, 0);
		if (!io) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO%s RESOURCE", pio == 2 ? " OR MEMORY" : "");
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO RESOURCE", dev_name);
			r = -ENXIO;
			goto ret1;
		}
		flags |= FLAG_PIO;
	}
	irq = PCI$READ_INTERRUPT_LINE(id);

	car.nclusters = XL_CLUSTERS;
	car.align = 0;
	car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (car.status < 0) {
		if (car.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE DMA DESCRIPTORS", dev_name);
		r = car.status;
		goto ret2;
	}
	xl = car.ptr;
	memset(xl, 0, sizeof(XL));
	xl->flags = flags | extra_flags;
	xl->errorlevel = errorlevel;
	xl->id = id;
	strcpy(xl->dev_name, dev_name);
	if (pio) {
		xl->u.io = io;
		xl->range.start = io;
		xl->range.len = REGSPACE(flags);
		xl->range.name = xl->dev_name;
		if (__unlikely(r = KERNEL$REGISTER_IO_RANGE(&xl->range))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, xl->range.start, xl->range.start + xl->range.len - 1, strerror(-r));
			goto ret25;
		}
	} else {
		xl->u.mem = mem;
	}
	WQ_INIT(&xl->link_state_wait, "3C$LINK_STATE_WAIT");
	for (i = 0; i < N_MCAST; i++) WQ_INIT(&xl->mcast_table[i], "RTL$MCAST_TABLE");
	xl->max_sent = DEFAULT_SENT;
	INIT_TIMER(&xl->timer);
	xl->timer.fn = pio ? PIO_TIMER : MMIO_TIMER;
	VOID_LIST_ENTRY(&xl->timer.list);
	INIT_TIMER(&xl->media_timer);
	xl->media_timer.fn = XL_MEDIA_TIMER;
	VOID_LIST_ENTRY(&xl->media_timer.list);
	xl->phy_id = -1;
	xl->bmcr = bmcr;
	xl->defer_timer = defer_timer;
	if (!V2(xl)) xl->flags |= NO_TX_CHECKSUM;

	desc.ptr = (unsigned long)DESCS_START(xl);
	desc.len = DESCS_LEN;
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	dma = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &xl->desc_dmaunlock);
	LOWER_SPL(SPL_ZERO);
	xl->desc_dmaaddr = dma.ptr;

	xl->reset_thread.thread = KERNEL$ALLOC_THREAD_SYNC();
	if (__IS_ERR(xl->reset_thread.thread)) {
		r = __PTR_ERR(xl->reset_thread.thread);
		xl->reset_thread.thread = NULL;
		goto ret3;
	}

	if (address_str) {
		if (NET$PARSE_ETHERNET_ADDRESS(address_str, xl->address) || !NET$VALID_ETHERNET_ADDRESS(xl->address)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "3C: INVALID ADDRESS: %s", address_str);
			r = -EBADSYN;
			goto ret3;
		}
		xl->flags |= ADDRESS_OVERRIDE;
	}

	if (xl->flags & CARDBUS) {
/* it is expected that if xl->flags & CARDBUS then xl->cb_mem is accessible */
		__u8 *cb_mem = PCI$MAP_MEM_RESOURCE(id, 2, REGSPACE_CB);
		if (!cb_mem) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO CARDBUS MEMORY RESOURCE");
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO CARDBUS MEMORY RESOURCE", dev_name);
			r = -ENXIO;
			goto ret3;
		}
		if (__IS_ERR(cb_mem)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP CARDBUS MEMORY RESOURCE: %s", dev_name, strerror(-__PTR_ERR(cb_mem)));
			r = __PTR_ERR(cb_mem);
			goto ret3;
		}
		xl->cb_mem = cb_mem;
	}

	if ((r = NETQUE$ALLOC_QUEUE(&xl->queue))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", dev_name, strerror(-r));
		goto ret3;
	}

	if ((r = KERNEL$RESERVE_BOUNCE(dev_name, N_RX_DESCS * N_RX_FRAGS + N_TX_DESCS * N_TX_FRAGS, PKT_SIZE))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT RESERVE BOUNCE BUFFERS: %s", dev_name, strerror(-r));
		goto ret4;
	}

	xl->first_recv = 0;
	xl->n_recv = 0;
	xl->first_sent = 0;
	xl->n_sent = 0;
	for (i = 0; i < N_RX_DESCS; i++) {
		PACKET *p;
		RAISE_SPL(SPL_XL);
		alloc_again:
		ALLOC_PACKET(p, PKT_SIZE - ETHERNET_HEADER, NULL, SPL_XL, {
			if ((r = NET$MEMWAIT_SYNC())) {
				if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE PACKETS", dev_name);
				goto ret5;
			}
			goto alloc_again;
		});
		XL_MAP_RECV_PACKET(xl, p, i);
		xl->n_recv = i + 1;
		LOWER_SPL(SPL_ZERO);
	}

	openrq.flags = O_WRONLY;
	openrq.path = net_input;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", dev_name, net_input, strerror(-openrq.status));
		r = openrq.status;
		goto ret5;
	}
	xl->packet_input = openrq.status;
	xl->outstanding = 0;

	r = XL_INIT(xl);
	if (r) goto ret6;

	if (!(xl->eeprom[EE_XL_Software_Information_2] & EE_XL_Software_Information_2_FixedMWIBug)) {
		__u16 cmd = PCI$READ_CONFIG_WORD(id, PCI_COMMAND);
		if (cmd & PCI_COMMAND_INVALIDATE) {
			cmd &= ~PCI_COMMAND_INVALIDATE;
			PCI$WRITE_CONFIG_WORD(id, PCI_COMMAND, cmd);
		}
	}

	if (media_string) {
		if (!_strcasecmp(media_string, "AUTOSELECT")) goto media_auto;
		if (!_strcasecmp(media_string, "DEFAULT")) goto media_unknown;
		for (i = 0; i < N_MEDIA; i++) {
			if (!_strcasecmp(media[i].name, media_string)) {
				xl->media = i;
				goto media_ok;
			}
		}
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID MEDIA STRING \"%s\"", dev_name, media_string);
		r = -EBADSYN;
		goto ret6;
	} else {
		__u32 internal_config;
		media_auto:
		XL_WINDOW(xl, 3);
		internal_config = XL_READ_32(xl, XL_W3_InternalConfig);
		if (!(internal_config & XL_InternalConfig_AutoSelect)) {

			xl->media = FIND_MEDIA_BY_INTERNAL_CONFIG(xl, internal_config);
			if (xl->media != -1) goto media_ok;
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "UNKNOWN MEDIA IN INTERNALCONFIG REGISTER (%08X), ASSUMING AUTOSELECT", internal_config);
		}
		xl->media_options = XL_READ_16(xl, X2_W3_MediaOptions);
		if (!(xl->media_options & (X2_W3_MediaOptions_BaseT4Available | X2_W3_MediaOptions_BaseTXAvailable | X2_W3_MediaOptions_BaseFXAvailable | X2_W3_MediaOptions_10bTAvailable | X2_W3_MediaOptions_CoaxAvailable | X2_W3_MediaOptions_AuiAvailable | X2_W3_MediaOptions_MiiConnector)))
			xl->media_options = X2_W3_MediaOptions_MiiConnector;
		if (xl->media_options & X2_W3_MediaOptions_BaseTXAvailable)
			xl->media_options &= ~X2_W3_MediaOptions_10bTAvailable;
		xl->flags |= MEDIA_AUTOSELECT;
		xl->media = AUTOSELECT_MEDIA(xl, -1);
		if (xl->media != -1) {
			if (xl->media == AUTOSELECT_MEDIA(xl, xl->media)) xl->flags &= ~MEDIA_AUTOSELECT;
		} else {
			xl->flags &= ~MEDIA_AUTOSELECT;
			xl->media = FIND_MEDIA_BY_INTERNAL_CONFIG(xl, internal_config);
			if (xl->media != -1) {
				KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "AUTOSELECT COULDN'T FIND MEDIA, ASSUMING %s", media[xl->media].name);
				goto media_ok;
			}
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "AUTOSELECT COULDN'T FIND MEDIA, LEAVING MEDIA SETTINGS UNCHANGED");
			media_unknown:
			xl->media = 0;
			xl->flags |= MEDIA_UNKNOWN;
		}
	}
	media_ok:

	XL_RESET_3(xl, 1);

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(a1, id));
	_printf("%s: %s %"__64_format"X, IRQ %d\n", dev_name, pio ? "IO" : "MEM", pio ? (__u64)io : PCI$READ_MEM_RESOURCE(id, 1, NULL), irq);
	_printf("%s: ADDRESS %s%s\n", dev_name, NET$PRINT_ETHERNET_ADDRESS(a1, xl->address), xl->flags & ADDRESS_OVERRIDE ? " (OVERRIDEN)" : "");

	RAISE_SPL(SPL_XL);
	xl->irq_ast.fn = pio ? PIO_IRQ : MMIO_IRQ;
	if ((e = KERNEL$REQUEST_IRQ(irq, IRQ_AST_HANDLER | IRQ_SHARED, NULL, &xl->irq_ast, &xl->irq_ctrl, dev_name))) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, e);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		r = -EINVAL;
		goto ret6;
	}
	XL_CONTINUE_RESET(xl);
	LOWER_SPL(SPL_ZERO);

	devrq.name = dev_name;
	devrq.driver_name = "3C.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = XL_INIT_ROOT;
	devrq.dev_ptr = xl;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = XL_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = devrq.status;
		goto ret7;
	}
	xl->lnte = devrq.lnte;
	xl->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;

	ret7:
	xl->irq_ctrl.mask();
	RAISE_SPL(SPL_XL);
	while (xl->flags & FLAG_RESETTING) KERNEL$SLEEP(1);
	xl->flags |= FLAG_DEAD;
	KERNEL$DEL_TIMER(&xl->timer);
	KERNEL$DEL_TIMER(&xl->media_timer);
	XL_RESET_1(xl);
	XL_RESET_2(xl);
	XL_RESET_3(xl, -1);
	XL_DESTROY_SENT_PACKETS(xl);
	KERNEL$RELEASE_IRQ(&xl->irq_ctrl, IRQ_RELEASE_MASKED);
	LOWER_SPL(SPL_ZERO);
	ret6:
	while (xl->outstanding) KERNEL$SLEEP(1);
	KERNEL$FAST_CLOSE(xl->packet_input);
	ret5:
	XL_DESTROY_RECV_PACKETS(xl);
	KERNEL$UNRESERVE_BOUNCE(xl->dev_name, N_RX_DESCS * N_RX_FRAGS + N_TX_DESCS * N_TX_FRAGS, PKT_SIZE);
	ret4:
	NETQUE$FREE_QUEUE(xl->queue);
	ret3:
	if (xl->reset_thread.thread) KERNEL$FREE_THREAD(xl->reset_thread.thread);
	if (xl->cb_mem) PCI$UNMAP_MEM_RESOURCE(id, xl->cb_mem, REGSPACE_CB);
	RAISE_SPL(SPL_VSPACE);
	xl->desc_dmaunlock(xl->desc_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	WQ_WAKE_ALL(&xl->link_state_wait);
	if (xl->flags & FLAG_PIO) KERNEL$UNREGISTER_IO_RANGE(&xl->range);
	ret25:
	KERNEL$VM_RELEASE_CONTIG_AREA(xl, XL_CLUSTERS);
	ret2:
	if (!pio) PCI$UNMAP_MEM_RESOURCE(id, mem, REGSPACE(flags));
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int XL_UNLOAD(void *p, void **release, char *argv[])
{
	unsigned i;
	int r;
	XL *xl = p;
	if ((r = KERNEL$DEVICE_UNLOAD(xl->lnte, argv))) return r;
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&xl->mcast_table[i]);
	xl->irq_ctrl.mask();
	RAISE_SPL(SPL_XL);
	while (xl->flags & FLAG_RESETTING) KERNEL$SLEEP(1);
	xl->flags |= FLAG_DEAD;
	KERNEL$DEL_TIMER(&xl->timer);
	KERNEL$DEL_TIMER(&xl->media_timer);
	XL_RESET_1(xl);
	XL_RESET_2(xl);
	XL_RESET_3(xl, -1);
	XL_DESTROY_SENT_PACKETS(xl);
	KERNEL$RELEASE_IRQ(&xl->irq_ctrl, IRQ_RELEASE_MASKED);
	LOWER_SPL(SPL_ZERO);
	while (xl->outstanding) KERNEL$SLEEP(1);
	KERNEL$FAST_CLOSE(xl->packet_input);
	XL_DESTROY_RECV_PACKETS(xl);
	KERNEL$UNRESERVE_BOUNCE(xl->dev_name, N_RX_DESCS * N_RX_FRAGS + N_TX_DESCS * N_TX_FRAGS, PKT_SIZE);
	NETQUE$FREE_QUEUE(xl->queue);
	if (xl->cb_mem) PCI$UNMAP_MEM_RESOURCE(xl->id, xl->cb_mem, REGSPACE_CB);
	RAISE_SPL(SPL_VSPACE);
	xl->desc_dmaunlock(xl->desc_dmaaddr);
	LOWER_SPL(SPL_ZERO);
	if (!(xl->flags & FLAG_PIO)) PCI$UNMAP_MEM_RESOURCE(xl->id, xl->u.mem, REGSPACE(xl->flags));
	PCI$FREE_DEVICE(xl->id);
	*release = xl->dlrq;
	WQ_WAKE_ALL(&xl->link_state_wait);
	KERNEL$FREE_THREAD(xl->reset_thread.thread);
	if (xl->flags & FLAG_PIO) KERNEL$UNREGISTER_IO_RANGE(&xl->range);
	KERNEL$VM_RELEASE_CONTIG_AREA(xl, XL_CLUSTERS);
	return 0;
}
