#ifndef __NIC_3C_H
#define __NIC_3C_H

#include <SYS/TYPES.H>
#include <SPAD/WQ.H>
#include <SPAD/LIBC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/PCI.H>
#include <SPAD/ETHERNET.H>
#include <SPAD/PKT.H>
#include <ARCH/IRQ.H>
#include <ARCH/BARRIER.H>

#include "3CREG.H"

#define XL_COMMAND_TIMEOUT		40000
#define XL_INIT_TX_THRESH		64	/* * 4 */
#define XL_INCREMENT_TX_THRESH		16
#define XL_INIT_TX_RECLAIM_THRESH	8	/* * 16 */
#define XL_INCREMENT_TX_RECLAIM_THRESH	1
#define XL_MAX_TX_RECLAIM_THRESH	32
#define N_RX_DESCS			256	/* must be power of 2 */
#define N_TX_DESCS			256	/* must be power of 2 */
#define N_TX_DESCS_INTR			16	/* must be power of 2 */
#define DEFAULT_SENT			N_TX_DESCS_INTR
#define MAX_SENT			(N_TX_DESCS - 1 - 1)
#define PKT_SIZE			(ETHERNET_HEADER + ETHERNET_MTU)
#define RECV_POLL_INTERVAL		X2_DM_UpPoll_Mask
#define XMIT_POLL_INTERVAL		X2_DM_DnPoll_Mask
#define TX_TIMEOUT			(JIFFIES_PER_SECOND * 2)

#define N_SENT_DECAY_TIME		(JIFFIES_PER_SECOND * 30)
#define LINK_CHECK_INITIAL_TIME		(JIFFIES_PER_SECOND * 3)
#define LINK_CHECK_SHORT_TIME		(JIFFIES_PER_SECOND * 10)
#define LINK_CHECK_LONG_TIME		(JIFFIES_PER_SECOND * 30)

#define INITIAL_LINK_EXPIRE_TIME	(JIFFIES_PER_SECOND * 10)
#define MONITORED_LINK_EXPIRE_TIME	(JIFFIES_PER_SECOND * 30)
#define UNMONITORED_LINK_PROMISC_TIME	(JIFFIES_PER_SECOND * 60 * 5)
#define UNMONITORED_LINK_EXPIRE_TIME	(JIFFIES_PER_SECOND * 60 * 5)

#define LINK_CHECK_TIME(xl)		(__likely((xl)->flags & LINK_INTERRUPT) ? LINK_CHECK_LONG_TIME : LINK_CHECK_SHORT_TIME)

/* loaded from PCI IDs (from Linux, some are unused) */
#define BOOMERANG		0x00000001
#define CYCLONE			0x00000002
#define TORNADO			0x00000004

#define MII			0x00000010
#define NWAY			0x00000020
#define MII_PREAMBLE		0x00000040
#define NO_EEPROM_RELOAD	0x00000100
#define EEPROM_8BIT		0x00000200
#define EEPROM_OFFSET_30	0x00000400
#define MAX_COLLISION_RESET	0x00000800

#define CARDBUS			0x00001000
#define INVERT_MII_PWR		0x00002000
#define INVERT_LED_PWR		0x00004000
#define WNO_XCVR_PWR		0x00008000

/* set by user */
#define FLOW_CONTROL		0x00010000
#define DEFER_ONLY_COLLISION	0x00020000
#define NO_LINK_INTR		0x00040000
#define NO_TX_CHECKSUM		0x00080000

/* internal state */
#define RECEIVED_PACKET		0x00100000
#define RECEIVED_PACKET_2	0x00200000
#define FULL_DUPLEX_ENABLED	0x00400000
#define LINK_INTERRUPT		0x00800000
#define FLAG_PIO		0x01000000
#define ADDRESS_OVERRIDE	0x02000000
#define MEDIA_AUTOSELECT	0x04000000
#define MEDIA_UNKNOWN		0x08000000
#define MII_PREAMBLE_RUNTIME	0x10000000
#define FLAG_DEAD		0x20000000
#define FLAG_RESETTING_INTR	0x40000000
#define FLAG_RESETTING		0x80000000

#define MCAST_ALL		N_HW_MCAST
#define MCAST_PROMISC		(N_HW_MCAST + 1)
#define N_MCAST			(N_HW_MCAST + 2)

typedef struct {
	union {
		io_t io;
		__u8 *mem;
	} u;
	unsigned flags;
	AST irq_ast;
	IRQ_CONTROL irq_ctrl;
	int packet_input;
	unsigned first_recv;
	unsigned n_recv;
#if __DEBUG >= 2
	int recv_maps;
	int sent_maps;
#endif
	unsigned max_sent;
	unsigned first_sent;
	unsigned n_sent;
	int outstanding;
	PKTQUEUE *queue;
	TIMER timer;

	__u8 *cb_mem;

	unsigned tx_thresh;
	unsigned tx_reclaim_thresh;

	WQ mcast_table[N_MCAST];
	int mcast_state;

	LINK_STATE link_state;
	WQ link_state_wait;
	TIMER media_timer;
	u_jiffies_lo_t change_media_time;
	u_jiffies_lo_t n_sent_decay;
	__u16 media_options;
	int media;
	char phy_id;
	__u16 bmcr;
	int defer_timer;
	THREAD_RQ reset_thread;

	__u8 address[ETHERNET_ADDRLEN];
	__u16 eeprom[EEPROM_SIZE];

	__u32 desc_dmaaddr;
	vspace_dmaunlock_t *desc_dmaunlock;
	void *lnte;
	void *dlrq;

	int errorlevel;
	pci_id_t id;
	IO_RANGE range;
	char dev_name[__MAX_STR_LEN];
} XL;

#define N_RX_FRAGS		1
#define N_TX_FRAGS		MAX_PACKET_FRAGMENTS(PKT_SIZE - ETHERNET_MTU)
#define DESC_ALIGN		8

typedef struct {
	UPD upd;
	UPD_ENTRY frags[N_RX_FRAGS];
	PACKET *p;
} XL_RXDESC;

typedef struct {
	DPD upd;
	DPD_ENTRY frags[N_TX_FRAGS];
	PACKET *p;
	vspace_dmaunlock_t *dmaunlock[N_TX_FRAGS];
} XL_TXDESC;

#define SIZEOF_XL_RXDESC	((sizeof(XL_RXDESC) + (DESC_ALIGN - 1)) & ~(DESC_ALIGN - 1))
#define SIZEOF_XL_TXDESC	((sizeof(XL_TXDESC) + (DESC_ALIGN - 1)) & ~(DESC_ALIGN - 1))

#define RX_DESC(xl, i)		((XL_RXDESC *)((char *)(xl) + ((sizeof(XL) + DESC_ALIGN - 1) & ~(DESC_ALIGN - 1)) + (i) * SIZEOF_XL_RXDESC))
#define TX_DESC(xl, i)		((XL_TXDESC *)((char *)RX_DESC(xl, N_RX_DESCS) + (i) * SIZEOF_XL_TXDESC))
#define RX_DESCS_OFFSET		0
#define TX_DESCS_OFFSET		(N_RX_DESCS * SIZEOF_XL_RXDESC)

#define DESCS_START(xl)		RX_DESC(xl, 0)
#define DESCS_LEN		(N_RX_DESCS * SIZEOF_XL_RXDESC + N_TX_DESCS * SIZEOF_XL_TXDESC)

#define REGSPACE(flags)		((flags) & BOOMERANG ? REGSPACE_3C90X : REGSPACE_3C90XB)
#define XL_CLUSTERS		(((unsigned long)DESCS_START(NULL) + DESCS_LEN + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS)

void XL_EXTRACT_ADDRESS(XL *xl, unsigned offset);
void XL_UPDATE_FILTER(XL *xl);
void XL_RECV_ERROR(XL *xl, __u32 up_status);
int XL_IRQ_SPECIAL(XL *xl, __u16 status);
void XL_TX_TIMEOUT(XL *xl);
extern IO_STUB XL_IOCTL;
void XL_DESTROY_SENT_PACKETS(XL *xl);

extern __const__ HANDLE_OPERATIONS PIO_OPERATIONS;
extern __const__ HANDLE_OPERATIONS MMIO_OPERATIONS;

extern AST_STUB PIO_IRQ;
extern AST_STUB MMIO_IRQ;

void PIO_TIMER(TIMER *t);
void MMIO_TIMER(TIMER *t);

void PIO_DEQUEUE(XL *xl);
void MMIO_DEQUEUE(XL *xl);

static __finline__ void XL_MAP_RECV_PACKET(XL *xl, PACKET *p, unsigned idx)
{
	VDESC desc;
	VDMA dma;
	RX_DESC(xl, idx)->p = p;
#if __DEBUG >= 2
	xl->recv_maps++;
	if (__unlikely(xl->recv_maps > N_RX_DESCS))
		KERNEL$SUICIDE("XL_MAP_RECV_PACKET: MAPS LEAKED");
	if (__unlikely(p->length < PKT_SIZE - ETHERNET_HEADER))
		KERNEL$SUICIDE("XL_MAP_RECV_PACKET: RECEIVE PACKET TOO SHORT (%d)", p->length);
#endif
	p->addrlen = ETHERNET_ADDRLEN;
	desc.ptr = (unsigned long)p->data + LL_HEADER - sizeof(struct eth_header);
	desc.len = ETHERNET_ADDRLEN + sizeof(struct eth_header);
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	dma = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_WRITE, (vspace_dmaunlock_t **)&p->sender_data);
	LOWER_SPL(SPL_XL);
	RX_DESC(xl, idx)->frags[0].UpFragAddr = __32CPU2LE(dma.ptr);
	__write_barrier();
	RX_DESC(xl, idx)->upd.UpPktStatus = __32CPU2LE(0);
}

static __finline__ void XL_UNMAP_RECV_PACKET(XL *xl, PACKET *p, unsigned idx)
{
#if __DEBUG >= 2
	xl->recv_maps--;
	if (__unlikely(xl->recv_maps < 0))
		KERNEL$SUICIDE("XL_UNMAP_RECV_PACKET: MAP COUNT UNDERFLOW");
#endif
	RAISE_SPL(SPL_VSPACE);
	((vspace_dmaunlock_t *)p->sender_data)(__32LE2CPU(RX_DESC(xl, idx)->frags[0].UpFragAddr));
	LOWER_SPL(SPL_XL);
}

static __finline__ void XL_UNMAP_SENT_PACKET(XL *xl, XL_TXDESC *desc)
{
	unsigned i;
#if __DEBUG >= 2
	xl->sent_maps--;
	if (__unlikely(xl->sent_maps < 0))
		KERNEL$SUICIDE("XL_UNMAP_SENT_PACKET: MAP COUNT UNDERFLOW");
#endif
	RAISE_SPL(SPL_VSPACE);
	desc->dmaunlock[0](__32LE2CPU(desc->frags[0].DnFragAddr));
	i = 0;
	while (__unlikely(!desc->frags[i].DnFragLen & __32CPU2LE(XL_DFL_DnFragLast))) {
		i++;
		desc->dmaunlock[i](__32LE2CPU(desc->frags[i].DnFragAddr));
	}
	LOWER_SPL(SPL_XL);
}

#endif
