#include <ARCH/IO.H>
#include <ARCH/MOV.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/PKT.H>
#include <SPAD/IOCTL.H>

#include "3C.H"
#include "3CREG.H"

#ifdef PIO

#define M(X)	PIO_##X
#define V2(xl)	(__unlikely((xl)->flags & (CYCLONE | TORNADO)))

#define XL_READ_8(xl, i)	io_inb((xl)->u.io + (i))
#define XL_READ_16(xl, i)	io_inw((xl)->u.io + (i))
#define XL_READ_32(xl, i)	io_inl((xl)->u.io + (i))
#define XL_WRITE_8(xl, i, v)	io_outb((xl)->u.io + (i), (v))
#define XL_WRITE_16(xl, i, v)	io_outw((xl)->u.io + (i), (v))
#define XL_WRITE_32(xl, i, v)	io_outl((xl)->u.io + (i), (v))

#else

#define M(X)	MMIO_##X
#define V2(xl)	1

#define XL_READ_8(xl, i)	mmio_inb((xl)->u.mem + (i))
#define XL_READ_16(xl, i)	mmio_inw((xl)->u.mem + (i))
#define XL_READ_32(xl, i)	mmio_inl((xl)->u.mem + (i))
#define XL_WRITE_8(xl, i, v)	mmio_outb((xl)->u.mem + (i), (v))
#define XL_WRITE_16(xl, i, v)	mmio_outw((xl)->u.mem + (i), (v))
#define XL_WRITE_32(xl, i, v)	mmio_outl((xl)->u.mem + (i), (v))

#endif

static void XL_SEND(XL *xl, PACKET *p);
static void XL_CLEANUP_SENT_NONEMPTY(XL *xl);
extern IO_STUB M(PACKET);

#define XL_CAN_SEND(xl)		(xl->n_sent < xl->max_sent)

__const__ HANDLE_OPERATIONS M(OPERATIONS) = {
	SPL_X(SPL_XL),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	XL_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	M(PACKET),		/* pktio */
};

#include "3CFN.I"

#ifdef PIO
DECL_IOCALL(PIO_PACKET, SPL_XL, PACKET)
#else
DECL_IOCALL(MMIO_PACKET, SPL_XL, PACKET)
#endif
{
	XL *xl;
	HANDLE *h;
#ifdef PIO
	__CHECK_PKT(RQ, "PIO_PACKET");
#else
	__CHECK_PKT(RQ, "MMIO_PACKET");
#endif
	h = RQ->handle;
	if (__unlikely(h->op != &M(OPERATIONS))) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_XL));
	xl = h->fnode;
	if (__unlikely(xl->flags & (FLAG_RESETTING | FLAG_DEAD))) {
		goto enq;
	}
	if (__unlikely(RQ->data_length + RQ->v.len > PKT_SIZE - ETHERNET_HEADER)) {
		RQ->status = -EMSGSIZE;
		RETURN_PKT(RQ);
	}
	RQ->status = 0;
	if (XL_CAN_SEND(xl)) {
		do_send:
		XL_SEND(xl, RQ);
	} else {
		XL_CLEANUP_SENT_NONEMPTY(xl);
		if (XL_CAN_SEND(xl)) goto do_send;
		else {
			enq:
			NETQUE$ENQUEUE_PACKET(xl->queue, RQ);
		}
	}
	RETURN;
}

static void XL_SEND(XL *xl, PACKET *p)
{
	XL_TXDESC *tx;
	VDESC desc;
	VDMA dma;
	__u32 new_ptr;
	tx = TX_DESC(xl, (xl->first_sent + xl->n_sent) & (N_TX_DESCS - 1));
	__PREFETCHNTA(tx - 1);
	__PREFETCHNTA_IF64((char *)(tx - 1) + 64);
	if (__likely(!(xl->flags & NO_TX_CHECKSUM))) {
		if (__likely(p->flags & PKT_OUTPUT_CHECKSUM_TCP)) {
			tx->upd.FrameStartHeader = __32CPU2LE(X2_FSH_RndupDefeat | XL_FSH_DnIndicate | X2_FSH_AddTcpChecksum);
			goto hw_checksum;
		}
		if (__likely(p->flags & PKT_OUTPUT_CHECKSUM_UDP)) {
			tx->upd.FrameStartHeader = __32CPU2LE(X2_FSH_RndupDefeat | XL_FSH_DnIndicate | X2_FSH_AddUdpChecksum);
			goto hw_checksum;
		}
	}
	CHECKSUM_PACKET(p, {
		p->status = r_;
		CALL_PKT(p);
		return;
	});
	if (!V2(xl)) tx->upd.FrameStartHeader = __32CPU2LE((p->data_length + p->v.len + ETHERNET_HEADER) + XL_FSH_DnIndicate);
	else tx->upd.FrameStartHeader = __32CPU2LE(X2_FSH_RndupDefeat | XL_FSH_DnIndicate);
	hw_checksum:
	desc.ptr = (unsigned long)p->data + LL_HEADER - ETHERNET_HEADER;
	desc.len = p->data_length + ETHERNET_HEADER;
	tx->frags[0].DnFragLen = __32CPU2LE(desc.len | XL_DFL_DnFragLast);
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	dma = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ, &tx->dmaunlock[0]);
	tx->frags[0].DnFragAddr = __32CPU2LE(dma.ptr);
	LOWER_SPL(SPL_XL);
	tx->upd.DnNextPtr = __32CPU2LE(0);
	tx->p = p;
	if (__unlikely(p->v.len != 0)) {
		unsigned i = 1;
		tx->frags[0].DnFragLen &= ~__32CPU2LE(XL_DFL_DnFragLast);
		next_part:
		RAISE_SPL(SPL_VSPACE);
		dma = p->v.vspace->op->vspace_dmalock(&p->v, PF_READ, &tx->dmaunlock[i]);
		tx->frags[i].DnFragAddr = __32CPU2LE(dma.ptr);
		tx->frags[i].DnFragLen = __32CPU2LE(dma.len);
		LOWER_SPL(SPL_XL);
		if (__unlikely(dma.len != p->v.len)) {
			if (__unlikely(!dma.len)) {
				p->flags |= PKT_PAGING_ERROR;
				p->status = -EVSPACEFAULT;
				do {
					i--;
					tx->dmaunlock[i](__32LE2CPU(tx->frags[i].DnFragAddr));
				} while (i);
				CALL_PKT(p);
				return;
			}
			p->v.ptr += dma.len;
			p->v.len -= dma.len;
			goto next_part;
		}
		tx->frags[i].DnFragLen |= __32CPU2LE(XL_DFL_DnFragLast);
	}
	tx = TX_DESC(xl, (xl->first_sent + xl->n_sent - 1) & (N_TX_DESCS - 1));
	new_ptr = xl->desc_dmaaddr + TX_DESCS_OFFSET + ((xl->first_sent + xl->n_sent) & (N_TX_DESCS - 1)) * SIZEOF_XL_TXDESC;
	if (!XMIT_POLLING(xl)) {
		XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_DnStall);
		XL_WAIT(xl, "DOWNLOAD STALL");
		if (__likely((xl->first_sent + xl->n_sent) & (N_TX_DESCS_INTR - 1)))
			tx->upd.FrameStartHeader &= ~__32CPU2LE(XL_FSH_DnIndicate);
		__barrier();
		tx->upd.DnNextPtr = __32CPU2LE(new_ptr);
		if (!XL_READ_32(xl, XL_DM_DnListPtr)) XL_WRITE_32(xl, XL_DM_DnListPtr, new_ptr);
		XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_DnUnStall);
	} else {
		if (__likely((xl->first_sent + xl->n_sent) & (N_TX_DESCS_INTR - 1)))
			__LOCK_AND32(&tx->upd.FrameStartHeader, ~__32CPU2LE(XL_FSH_DnIndicate));
		__barrier();
		tx->upd.DnNextPtr = __32CPU2LE(new_ptr);
	}
	xl->n_sent++;
#if __DEBUG >= 2
	xl->sent_maps++;
	if (__unlikely(xl->sent_maps > N_TX_DESCS))
		KERNEL$SUICIDE("XL_SEND: MAPS LEAKED");
#endif
	KERNEL$DEL_TIMER(&xl->timer);
	KERNEL$SET_TIMER(TX_TIMEOUT, &xl->timer);
}

static void XL_CLEANUP_SENT_NONEMPTY(XL *xl)
{
	while (xl->n_sent) {
		XL_TXDESC *tx = TX_DESC(xl, xl->first_sent);
		if (V2(xl)) {
			if (__unlikely(!(tx->upd.FrameStartHeader & __32CPU2LE(X2_FSH_DnComplete)))) break;
		} else {
			__u32 dma_addr = XL_READ_32(xl, XL_DM_DnListPtr);
			if (__unlikely(dma_addr != xl->desc_dmaaddr + TX_DESCS_OFFSET + xl->first_sent * SIZEOF_XL_TXDESC)) break;
		}
		XL_UNMAP_SENT_PACKET(xl, tx);
		CALL_PKT(tx->p);
		xl->first_sent = (xl->first_sent + 1) & (N_TX_DESCS - 1);
		xl->n_sent--;
	}
	while (__likely(XL_CAN_SEND(xl))) {
		PACKET *p = NETQUE$DEQUEUE(xl->queue);
		if (__unlikely(!p)) break;
		if (__unlikely(!xl->n_sent)) {
			if (__likely(xl->max_sent < MAX_SENT)) xl->max_sent++;
		}
		XL_SEND(xl, p);
	}
}

static __finline__ void XL_CLEANUP_SENT(XL *xl)
{
	if (xl->n_sent) XL_CLEANUP_SENT_NONEMPTY(xl);
}

static void XL_INSERT_RECV_PACKET(XL *xl, PACKET *p)
{
	XL_MAP_RECV_PACKET(xl, p, (xl->first_recv + xl->n_recv) & (N_RX_DESCS - 1));
	xl->n_recv++;
}

static void XL_START_RECV(XL *xl)
{
	if (!RECV_POLLING(xl)) {
		__barrier();
		XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_UpUnStall);
	}
}

#ifdef PIO
DECL_AST(PIO_PACKET_RETURNED, SPL_XL, PACKET)
#else
DECL_AST(MMIO_PACKET_RETURNED, SPL_XL, PACKET)
#endif
{
	XL *xl;
	PKT_AST_ENTER(RQ);
	xl = RQ->sender_data;
	xl->outstanding--;
	if (__likely(xl->n_recv == N_RX_DESCS) || __unlikely(xl->flags & FLAG_DEAD)) {
		FREE_PACKET(RQ, NULL, SPL_XL);
	} else {
		RQ->flags &= PKT_REUSE_FLAGS;
		XL_INSERT_RECV_PACKET(xl, RQ);
		XL_START_RECV(xl);
	}
	RETURN;
}

static __finline__ void XL_DO_RECEIVE(XL *xl)
{
	XL_RXDESC *desc;
	while (__likely(xl->n_recv) && (desc = RX_DESC(xl, xl->first_recv))->upd.UpPktStatus & __32CPU2LE(XL_UPS_UpComplete)) {
		PACKET *p = desc->p;
		__read_barrier();
		XL_UNMAP_RECV_PACKET(xl, p, xl->first_recv);
		PKT_PREFETCH_1(p);
		p->fn = M(PACKET_RETURNED);
		p->sender_data = xl;
		p->h = xl->packet_input;
		p->id = IDTYPE_PCI | (xl->id & IDTYPE_MASK);
		xl->outstanding++;
		p->data_length = __32LE2CPU(desc->upd.UpPktStatus) & XL_UPS_UpPktLen;
		if (__unlikely(desc->upd.UpPktStatus & __32CPU2LE(XL_UPS_UpError | XL_UPS_UpOverflow))) {
			p->status = -EIO;
			CALL_PKT(p);
			XL_RECV_ERROR(xl, desc->upd.UpPktStatus);
			goto dropped;
		}
		if (V2(xl)) {
			if (__likely((desc->upd.UpPktStatus & __32CPU2LE(X2_UPS_IpChecksumError | X2_UPS_IpChecksumChecked)) == __32CPU2LE(X2_UPS_IpChecksumChecked))) p->flags |= PKT_IPHDR_CHECKSUM_OK;
			if (__likely(((desc->upd.UpPktStatus ^ __32CPU2LE(X2_UPS_TcpChecksumError | X2_UPS_UdpChecksumError)) * (X2_UPS_TcpChecksumChecked / X2_UPS_TcpChecksumError)) & __32CPU2LE(X2_UPS_TcpChecksumChecked | X2_UPS_UdpChecksumChecked))) p->flags |= PKT_TCPUDP_CHECKSUM_OK;
		}
		PKT_PREFETCH_2(p);
		/*__debug_printf("recv\n");*/
		CALL_IORQ(p, KERNEL$PKTIO);
		xl->flags |= RECEIVED_PACKET_2;
		if (__unlikely(!(xl->flags & RECEIVED_PACKET))) {
			xl->flags |= RECEIVED_PACKET;
			if (__unlikely(xl->mcast_state == STATUS_MULTICAST_PROMISC))
				XL_UPDATE_FILTER(xl);
		}
		dropped:
		xl->first_recv = (xl->first_recv + 1) & (N_RX_DESCS - 1);
		xl->n_recv--;
	}
	if (__likely(xl->n_recv != N_RX_DESCS)) {
		do {
			PACKET *p;
			ALLOC_PACKET(p, PKT_SIZE - ETHERNET_HEADER, NULL, SPL_XL, break;);
			XL_INSERT_RECV_PACKET(xl, p);
		} while (xl->n_recv != N_RX_DESCS);
		XL_START_RECV(xl);
	}
}

#ifdef PIO
DECL_AST(PIO_IRQ, SPL_XL, AST)
#else
DECL_AST(MMIO_IRQ, SPL_XL, AST)
#endif
{
	XL *xl = GET_STRUCT(RQ, XL, irq_ast);
	__u16 status = XL_READ_16(xl, XL_IntStatusCmd);
	if (__unlikely(!(status & XL_IntStatus_InterruptLatch))) goto end;
	if (__unlikely(xl->flags & (FLAG_RESETTING | FLAG_DEAD))) {
		if (xl->flags & FLAG_RESETTING) {
			xl->flags |= FLAG_RESETTING_INTR;
			goto ret;
		}
		goto end;
	}
	if (__unlikely(status & (XL_IntStatus_HostError | XL_IntStatus_TxComplete | XL_IntStatus_LinkEvent))) {
		if (XL_IRQ_SPECIAL(xl, status)) goto end;
	}
	XL_WRITE_16(xl, XL_IntStatusCmd, XL_Cmd_AcknowledgeInterrupt | XL_Cmd_AcknowledgeInterrupt_InterruptLatchAck | XL_Cmd_AcknowledgeInterrupt_DnCompleteAck | XL_Cmd_AcknowledgeInterrupt_UpCompleteAck);
	THE_PCMCIA_PEOPLE_ARE_IDIOTS(xl);
	XL_CLEANUP_SENT(xl);	/* this should be before XL_DO_RECEIVE to allow faster retrasmits (packets that are not cleaned up can't be retransmitted) */
	if (__likely(status & XL_IntStatus_UpComplete)) XL_DO_RECEIVE(xl);
	end:
	xl->irq_ctrl.eoi();
	ret:
	RETURN;
}

void M(TIMER)(TIMER *t)
{
	XL *xl = GET_STRUCT(t, XL, timer);
	LOWER_SPL(SPL_XL);
	VOID_LIST_ENTRY(&xl->timer.list);
	if (__unlikely(xl->flags & (FLAG_RESETTING | FLAG_DEAD))) return;
	if (__unlikely(xl->n_sent)) XL_TX_TIMEOUT(xl);
}

void M(DEQUEUE)(XL *xl)
{
	XL_CLEANUP_SENT_NONEMPTY(xl);
}
