#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DL.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/PCI.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <ARCH/TIME.H>
#include <SPAD/TIMER.H>
#include <ENDIAN.H>
#include <SPAD/PKT.H>
#include <SPAD/ETHERNET.H>
#include <SPAD/IOCTL.H>
#include <VALUES.H>
/*#include <SPAD/RANDOM.H>*/

#include "NEREG.H"

static int pci_loaded = 0;

#define MCAST_ALL	64
#define MCAST_PROMISC	65
#define N_MCAST		66

typedef struct __ne NE;

struct __ne {
	io_t io;
	AST irq_ast;
	IRQ_CONTROL irq_ctrl;
	int irq;
	unsigned long flags;
	int errorlevel;
	int tx_start_page;
	int stop_page;
	int rx_start_page;

	char txing;
	unsigned short tx1;
	unsigned short tx2;

	unsigned w;
	int wpos;

	TIMER timer;

	void (*prepare_output)(NE *ne, int page, int count);
	void (*output)(NE *ne, __u8 *ptr, int count);
	void (*done_output)(NE *ne);
	void (*input)(NE *ne, int page, __u8 *ptr, int count);

	/*RANDOM_CTX rnd_ctx;*/

	int packet_input;
	int outstanding;
	PKTQUEUE *queue;

	PKTPOOL pool;

	VBUF vbuf;
	__u8 packet_tmp[ETHERNET_MTU];
	__u8 padding[ETHERNET_ZLEN];

	__u8 address[ETHERNET_ADDRLEN];
	LINK_STATE link_state;
	__u8 c0, c2, c3, cx;
	WQ link_state_wait;
	TIMER media_timer;
	WQ mcast_table[N_MCAST];
	int mcast_state;
	int use_pci;
	pci_id_t id;
	char dev_name[__MAX_STR_LEN];
	void *lnte;
	void *dlrq;
	IO_RANGE range;
};

extern IO_STUB NE_PACKET;
extern IO_STUB NE_IOCTL;
static int NE_UNLOAD(void *p, void **release, char *argv[]);
static void NS8390_INIT(NE *ne, int start);
static void RESET(NE *ne);

static __const__ HANDLE_OPERATIONS NE_OPERATIONS = {
	SPL_X(SPL_NE),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	NE_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	NE_PACKET,		/* pktio */
};

#define ONLY_8BIT_IO		1
#define ONLY_16BIT_IO		2
#define ONLY_32BIT_IO		4
#define STOP_PG_0x60		8
#define FORCE_FDX		16
#define FORCE_HDX		32
#define FORCE_FLOW_CONTROL	64
#define FORCE_NO_FLOW_CONTROL	128
#define MEDIA_CHECK		256
#define REALTEK			512

static __const__ struct pci_id_s pci_cards[] = {
	{ 0x10ec, 0x8029, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "REALTEK RTL-8029", REALTEK },
	{ 0x1050, 0x0940, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "WINBOND 89X940", 0 },
	{ 0x11f6, 0x1401, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "COMPEX RL2000", 0 },
	{ 0x8e2e, 0x3000, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "KTI ET32P2", 0 },
	{ 0x4a14, 0x5000, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "NETVIN NV5000SC", REALTEK },
	{ 0x1106, 0x0926, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "VIA 86C926", ONLY_16BIT_IO },
	{ 0x10bd, 0x0e34, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SURECOM NE34", 0 },
	{ 0x1050, 0x5a5a, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "WINBOND 89C940F", 0 },
	{ 0x8c4a, 0x1980, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "WINBOND 89C940 MISPROGRAMMED", 0 },
	{ 0x12c3, 0x0058, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "HOLTEK HT80232", ONLY_16BIT_IO /*| HOLTEK_FDX*/ },
	{ 0x12c3, 0x5598, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "HOLTEK HT80229", ONLY_32BIT_IO /*| HOLTEK_FDX*/ | STOP_PG_0x60 },
	{ 0, }
};

static void RESET(NE *ne)
{
	jiffies_t start, ss;
	io_t io = ne->io;
	io_outb(io + NE_RESET, io_inb(io + NE_RESET));
	ne->txing = 0;
	ne->tx1 = 0;
	ne->tx2 = 0;
	start = KERNEL$GET_JIFFIES();
	while (ss = KERNEL$GET_JIFFIES(), !(io_inb_p(io + NE0_ISR) & NE_ISR_RST)) {
		if (__unlikely(ss - start > TIMEOUT_JIFFIES(RST_TIMEOUT))) {
			KERNEL$SYSLOG(__SYSLOG_DATA_ERROR, ne->dev_name, "FAILED TO ACK RESET");
			return;
		}
	}
	io_outb_p(io + NE0_ISR, NE_ISR_RST);
}

static void NS8390_REFRESH_MULTICAST(NE *ne);

static void NS8390_INIT(NE *ne, int start)
{
	int m, i, endcfg;
	io_t io = ne->io;
	/*
	Linux contains this to enable full-duplex, but it's completely bogus
	if (ne->flags & FORCE_FDX) {
		if (ne->flags & REALTEK_FDX) {
			io_outb(io + NE_CMD, NE_CMD_PS0 | NE_CMD_PS1 | NE_CMD_RD2);
			io_outb(io + 0x20, io_inb(io + 0x20) | 0x80);
			io_outb(io + NE_CMD, NE_CMD_RD2);
		} else if (ne->flags & HOLTEK_FDX) {
			io_outb(io + 0x20, io_inb(io + 0x20) | 0x80);
		} else {
			KERNEL$SUICIDE("NS8390_INIT: FORCE FDX, BUT NO FDX, FLAGS %08lX", ne->flags);
		}
	}
	*/
	if (ne->flags & REALTEK) {
		/* reload eeprom to clear previous duplex/flowctrl override */
		io_outb(io + NE_CMD, NE_CMD_PS0 | NE_CMD_PS1 | NE_CMD_RD2 | NE_CMD_STP);
		io_outb(io + NE3_RTL_9346CR, NE3_RTL_9346CR_EEM0);
		KERNEL$UDELAY(500);
		i = 0;
		while (1) {
			__u8 cr;
			io_outb(io + NE_CMD, NE_CMD_PS0 | NE_CMD_PS1 | NE_CMD_RD2 | NE_CMD_STP);
			cr = io_inb(io + NE3_RTL_9346CR);
			if (!(cr & NE3_RTL_9346CR_EEM0) && cr) break;
			KERNEL$UDELAY(10);
			if (__unlikely(++i >= 10000)) {
				io_outb(io + NE3_RTL_9346CR, 0);
				break;
			}
		}
		/* optionally override duplex mode and flow control */
		if (ne->flags & (FORCE_HDX | FORCE_FDX | FORCE_FLOW_CONTROL | FORCE_NO_FLOW_CONTROL)) {
			__u8 c2, c3;
			io_outb(io + NE_CMD, NE_CMD_PS0 | NE_CMD_PS1 | NE_CMD_RD2 | NE_CMD_STP);
			io_outb(io + NE3_RTL_9346CR, NE3_RTL_9346CR_EEM0 | NE3_RTL_9346CR_EEM1);
			c3 = io_inb(io + NE3_RTL_CONFIG3);
			if (ne->flags & FORCE_HDX) c3 &= ~NE3_RTL_CONFIG3_FUDUP;
			if (ne->flags & FORCE_FDX) c3 |= NE3_RTL_CONFIG3_FUDUP;
			io_outb(io + NE3_RTL_CONFIG3, c3);
			if (c3 & NE3_RTL_CONFIG3_FUDUP && ne->flags & (FORCE_FLOW_CONTROL | FORCE_NO_FLOW_CONTROL)) {
				c2 = io_inb(io + NE3_RTL_CONFIG2);
				if (ne->flags & FORCE_NO_FLOW_CONTROL) c2 &= ~NE3_RTL_CONFIG2_FCE;
				if (ne->flags & FORCE_FLOW_CONTROL) c2 |= NE3_RTL_CONFIG2_FCE;
				io_outb(io + NE3_RTL_CONFIG2, c2);
			}
			io_outb(io + NE3_RTL_9346CR, 0);
		}
		/*__debug_printf("cr: %02x, c0: %02x, c2: %02x, c3: %02x\n", io_inb(io + NE3_RTL_9346CR), io_inb(io + NE3_RTL_CONFIG0), io_inb(io + NE3_RTL_CONFIG2), io_inb(io + NE3_RTL_CONFIG3));*/
		io_outb(io + NE_CMD, NE_CMD_RD2 | NE_CMD_STP);
		ne->cx = 1;
	}
	endcfg = !(ne->flags & ONLY_8BIT_IO) ? (0x48 | NE_DCR_WTS
#ifdef __BIG_ENDIAN
		| NE_DCR_BOS
#endif
		) : 0x48;
	io_outb_p(io + NE_CMD, NE_CMD_RD2 | NE_CMD_STP);
	io_outb_p(io + NE0_DCR, endcfg);
	io_outb_p(io + NE0_RBCR0, 0x00);
	io_outb_p(io + NE0_RBCR1, 0x00);
	io_outb_p(io + NE0_RCR, NE_RCR_AR);
	io_outb_p(io + NE0_TCR, 0x02);
	io_outb_p(io + NE0_TPSR, ne->tx_start_page);
	ne->tx1 = ne->tx2 = 0;
	ne->txing = 0;
	io_outb_p(io + NE0_PSTART, ne->rx_start_page);
	io_outb_p(io + NE0_BNRY, ne->stop_page - 1);
	io_outb_p(io + NE0_PSTOP, ne->stop_page);
	io_outb_p(io + NE0_ISR, 0xff);
	io_outb_p(io + NE0_IMR, 0x00);
	io_outb_p(io + NE_CMD, NE_CMD_RD2 + NE_CMD_PS0 + NE_CMD_STP);
	m = 0;
	for (i = 0; i < ETHERNET_ADDRLEN; i++) {
		io_outb_p(io + NE1_PAR0 + i, ne->address[i]);
		if (!m && io_inb_p(io + NE1_PAR0 + i) != ne->address[i]) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "ETHERNET ADDRESS COULD NOT BE WRITTEN");
			m = 1;
		}
	}
	io_outb_p(io + NE1_CURR, ne->rx_start_page);
	io_outb_p(io + NE_CMD, NE_CMD_RD2 + NE_CMD_STP);

	if (__likely(start)) {
		io_outb_p(io + NE0_ISR, 0xff);
		io_outb_p(io + NE0_IMR, NE_IMR_PRXE | NE_IMR_PTXE | NE_IMR_RXEE | NE_IMR_TXEE | NE_IMR_OVWE | NE_IMR_CNTE);
		io_outb_p(io + NE_CMD, NE_CMD_RD2 + NE_CMD_STA);
		io_outb_p(io + NE0_TCR, 0);
		NS8390_REFRESH_MULTICAST(ne);
	}
}

static void NS8390_REFRESH_MULTICAST(NE *ne)
{
	unsigned i, j;
	int some_mcast_set;
	io_t io = ne->io;
	io_outb(io + NE_CMD, NE_CMD_RD2);
	if (__unlikely(!WQ_EMPTY(&ne->mcast_table[MCAST_PROMISC]))) {
		io_outb_p(io + NE0_RCR, NE_RCR_AB | NE_RCR_PRO);
		ne->mcast_state = STATUS_MULTICAST_PROMISC;
		return;
	}
	io_outb_p(io + NE0_RCR, NE_RCR_AB);
	io_outb_p(io + NE_CMD, NE_CMD_RD2 + NE_CMD_PS0);
	ne->mcast_state = STATUS_MULTICAST_SOME;
	some_mcast_set = 0;
	for (i = 0; i < 8; i++) {
		__u8 bits = 0;
		if (__unlikely(!WQ_EMPTY(&ne->mcast_table[MCAST_ALL]))) {
			bits = 0xff;
			ne->mcast_state = STATUS_MULTICAST_ALL;
		} else for (j = 0; j < 8; j++) if (__unlikely(!WQ_EMPTY(&ne->mcast_table[i * 8 + j]))) bits |= 1 << j;
		if (__unlikely(bits)) some_mcast_set = 1;
		io_outb_p(io + NE1_MAR0 + i, bits);
	}
	io_outb(io + NE_CMD, NE_CMD_RD2);
	if (some_mcast_set) io_outb_p(io + NE0_RCR, NE_RCR_AB | NE_RCR_AM);
	else ne->mcast_state = STATUS_MULTICAST_NONE;
}

static void NE_ISA_PREPARE_OUTPUT(NE *ne, int page, int count)
{
	io_t io = ne->io;
	if (__likely(!(ne->flags & ONLY_8BIT_IO))) count = (count + 1) & ~1;
	io_outb_p(io + NE0_ISR, NE_ISR_RDC);
	io_outb_p(io + NE0_RBCR0, count & 0xff);
	io_outb_p(io + NE0_RBCR1, count >> 8);
	io_outb_p(io + NE0_RSAR0, 0);
	io_outb_p(io + NE0_RSAR1, page);
	io_outb_p(io + NE_CMD, NE_CMD_RD1 | NE_CMD_STA);
	ne->wpos = 0;
}

static void NE_PCI_PREPARE_OUTPUT(NE *ne, int page, int count)
{
	io_t io = ne->io;
	if (__likely(!(ne->flags & ONLY_16BIT_IO))) count = (count + 3) & ~3;
	else count = (count + 1) & ~1;
	io_outb(io + NE0_ISR, NE_ISR_RDC);
	io_outb(io + NE0_RBCR0, count & 0xff);
	io_outb(io + NE0_RBCR1, count >> 8);
	io_outb(io + NE0_RSAR0, 0);
	io_outb(io + NE0_RSAR1, page);
	io_outb(io + NE_CMD, NE_CMD_RD1 | NE_CMD_STA);
	ne->wpos = 0;
}

static void NE_OUTPUT_8(NE *ne, __u8 *start, int count)
{
	io_outsb(ne->io + NE_DATAPORT, start, count);
}

static void NE_OUTPUT_16(NE *ne, __u8 *start, int count)
{
	unsigned w = ne->w;
	switch (ne->wpos) case 0: while (1) {
#ifdef __BIG_ENDIAN
			w = *start << 8;
#else
			w = *start;
#endif
			start++;
			if (__unlikely(!--count)) {
				ne->wpos = 1;
				ne->w = w;
				return;
			}
		case 1:
#ifdef __BIG_ENDIAN
			w |= *start;
#else
			w |= *start << 8;
#endif
			io_outw(ne->io + NE_DATAPORT, w);
			start++;
			if (__unlikely(!--count)) {
				ne->wpos = 0;
				return;
			}
	}
	KERNEL$SUICIDE("NE_OUTPUT_16: WPOS == %d", ne->wpos);
}

static void NE_OUTPUT_32(NE *ne, __u8 *start, int count)
{
	unsigned w = ne->w;
	switch (ne->wpos) case 0: while (1) {
#ifdef __BIG_ENDIAN
			w = *start << 24;
#else
			w = *start;
#endif
			start++;
			if (__unlikely(!--count)) {
				ne->wpos = 1;
				ne->w = w;
				return;
			}
		case 1:
#ifdef __BIG_ENDIAN
			w |= *start << 16;
#else
			w |= *start << 8;
#endif
			start++;
			if (__unlikely(!--count)) {
				ne->wpos = 2;
				ne->w = w;
				return;
			}
		case 2:
#ifdef __BIG_ENDIAN
			w |= *start << 8;
#else
			w |= *start << 16;
#endif
			start++;
			if (__unlikely(!--count)) {
				ne->wpos = 3;
				ne->w = w;
				return;
			}
		case 3:
#ifdef __BIG_ENDIAN
			w |= *start;
#else
			w |= *start << 24;
#endif
			io_outl(ne->io + NE_DATAPORT, w);
			start++;
			if (__unlikely(!--count)) {
				ne->wpos = 0;
				return;
			}
	}
	KERNEL$SUICIDE("NE_OUTPUT_32: WPOS == %d", ne->wpos);
}

static void NE_ISA_DONE_OUTPUT(NE *ne)
{
	u_jiffies_t start, ss;
	if (__unlikely(ne->wpos)) {
		io_outw(ne->io + NE_DATAPORT, ne->w);
	}
	ss = start = KERNEL$GET_JIFFIES();
	while (!(io_inb_p(ne->io + NE0_ISR) & NE_ISR_RDC)) {
		if (__unlikely(ss - start > TIMEOUT_JIFFIES(WR_TIMEOUT))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "TIMEOUT WAITING FOR TX RDC");
			RESET(ne);
			NS8390_INIT(ne, 1);
			return;
		}
		ss = KERNEL$GET_JIFFIES();
	}
	io_outb_p(ne->io + NE0_ISR, NE_ISR_RDC);
}

static void NE_PCI_DONE_OUTPUT(NE *ne)
{
	u_jiffies_t start, ss;
	if (__unlikely(ne->wpos)) {
		if (__likely(!(ne->flags & ONLY_16BIT_IO))) {
			io_outl(ne->io + NE_DATAPORT, ne->w);
		}
		else io_outw(ne->io + NE_DATAPORT, ne->w);
	}
	ss = start = KERNEL$GET_JIFFIES();
	while (!(io_inb(ne->io + NE0_ISR) & NE_ISR_RDC)) {
		if (__unlikely(ss - start > TIMEOUT_JIFFIES(WR_TIMEOUT))) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "TIMEOUT WAITING FOR TX RDC");
			RESET(ne);
			NS8390_INIT(ne, 1);
			return;
		}
		ss = KERNEL$GET_JIFFIES();
	}
	io_outb(ne->io + NE0_ISR, NE_ISR_RDC);
}

static void NE_ISA_INPUT(NE *ne, int page, __u8 *ptr, int count)
{
	io_t io = ne->io;
	if (__likely(!(ne->flags & ONLY_8BIT_IO))) count = (count + 1) & ~1;
	io_outb_p(io + NE0_RBCR0, count & 0xff);
	io_outb_p(io + NE0_RBCR1, count >> 8);
	io_outb_p(io + NE0_RSAR0, 0);
	io_outb_p(io + NE0_RSAR1, page);
	io_outb_p(io + NE_CMD, NE_CMD_RD0 | NE_CMD_STA);
	if (__unlikely((ne->flags & ONLY_8BIT_IO) != 0)) {
		io_insb(io + NE_DATAPORT, ptr, count);
	} else {
		io_insw(io + NE_DATAPORT, ptr, count >> 1);
	}
	io_outb(io + NE0_ISR, NE_ISR_RDC);
}

static void NE_PCI_INPUT(NE *ne, int page, __u8 *ptr, int count)
{
	io_t io = ne->io;
	if (__likely(!(ne->flags & ONLY_16BIT_IO))) count = (count + 3) & ~3;
	else count = (count + 1) & ~1;
	io_outb(io + NE0_RBCR0, count & 0xff);
	io_outb(io + NE0_RBCR1, count >> 8);
	io_outb(io + NE0_RSAR0, 0);
	io_outb(io + NE0_RSAR1, page);
	io_outb(io + NE_CMD, NE_CMD_RD0 | NE_CMD_STA);
	if (__unlikely((ne->flags & ONLY_16BIT_IO) != 0)) {
		io_insw(io + NE_DATAPORT, ptr, count >> 1);
	} else {
		count >>= 2;
		while (count) {
			__u32 u = io_inl(io + NE_DATAPORT);
			*(__u16 *)ptr = __16LE2CPU(u);
			*(__u16 *)(ptr + 2) = __16LE2CPU(u >> 16);
			ptr += 4;
			count--;
		}
	}
	io_outb(io + NE0_ISR, NE_ISR_RDC);
}

static void NS8390_TRIGGER_SEND(NE *ne, int length, int start_page)
{
	io_t io = ne->io;
	if (__unlikely(io_inb_p(io + NE_CMD) & NE_CMD_TXP)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "TRANSMITTER BUSY, STATUS %02X", io_inb_p(io + NE_CMD));
		return;
	}
	io_outb_p(io + NE0_TBCR0, length & 0xff);
	io_outb_p(io + NE0_TBCR1, length >> 8);
	io_outb_p(io + NE0_TPSR, start_page);
	io_outb_p(io + NE_CMD, NE_CMD_RD2 | NE_CMD_TXP | NE_CMD_STA);
	KERNEL$DEL_TIMER(&ne->timer);
	KERNEL$SET_TIMER(TX_TIMEOUT, &ne->timer);
}

static int NS8390_XMIT(NE *ne, PACKET *pkt)
{
	int send_len, output_page;
	__u8 *ind;
	vspace_unmap_t *unmap;
#ifdef EI_PINGPONG
	if (ne->tx1 && ne->tx2) return -1;
#else
	if (ne->txing) return -1;
#endif
	if (__unlikely(pkt->data_length + pkt->v.len > ETHERNET_MTU)) {
		pkt->status = -EMSGSIZE;
		return 1;
	}
	CHECKSUM_PACKET(pkt, {
		pkt->status = r_;
		return 1;
	});
	ind = NET$MAP_INDIRECT_PACKET(&ne->vbuf, &pkt->v, &unmap);
	if (__unlikely(__IS_ERR(ind))) {
		pkt->flags |= PKT_PAGING_ERROR;
		pkt->status = __PTR_ERR(ind);
		return 1;
	}
#ifdef EI_PINGPONG
	if (!ne->tx1) {
		output_page = ne->tx_start_page;
	} else {
		output_page = ne->tx_start_page + TX_1X_PAGES;
	}
#else
	output_page = ne->tx_start_page;
#endif
	if (__unlikely((send_len = pkt->data_length + pkt->v.len + ETHERNET_HEADER) < ETHERNET_ZLEN))
		send_len = ETHERNET_ZLEN;
	ne->prepare_output(ne, output_page, send_len);
	ne->output(ne, &pkt->data[LL_HEADER - ETHERNET_HEADER], ETHERNET_HEADER + pkt->data_length);
	if (__unlikely(pkt->v.len != 0)) ne->output(ne, ind, pkt->v.len);
	if (__unlikely(pkt->data_length + pkt->v.len + ETHERNET_HEADER < ETHERNET_ZLEN))
		ne->output(ne, ne->padding, ETHERNET_ZLEN - ETHERNET_HEADER - pkt->data_length - pkt->v.len);
	ne->done_output(ne);
	RAISE_SPL(SPL_VSPACE);
	unmap(ind);
	LOWER_SPL(SPL_NE);
#ifdef EI_PINGPONG
	if (!ne->tx1) {
		ne->tx1 = send_len;
		if (!ne->txing) {
			ne->txing = 1;
			NS8390_TRIGGER_SEND(ne, send_len, output_page);
		}
	} else {
		ne->tx2 = send_len;
		if (!ne->txing) {
			ne->txing = 2;
			NS8390_TRIGGER_SEND(ne, send_len, output_page);
		}
	}
#else
	ne->txing = 1;
	NS8390_TRIGGER_SEND(ne, send_len, output_page);
#endif
	return 0;
}

static void NS8390_DEQUEUE(NE *ne)
{
	int r;
	PACKET *p;
	again:
	p = NETQUE$DEQUEUE(ne->queue);
	if (__unlikely(!p)) return;
	__CHECK_PKT(p, "NS8390_DEQUEUE");
	if (__unlikely((r = NS8390_XMIT(ne, p)))) {
		if (__unlikely(r < 0)) KERNEL$SUICIDE("NS8390_DEQUEUE: %s: TRANSMIT NOT ALLOWED", ne->dev_name);
		else {
			CALL_PKT(p);
			goto again;
		}
	}
	CALL_PKT(p);
}

static void NS8390_TX_TIMEOUT(NE *ne)
{
	io_t io = ne->io;
	int tsr = io_inb(io + NE0_TSR);
	int isr = io_inb(io + NE0_ISR);
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "TRANSMIT TIMEOUT, TSR=%02X, ISR=%02X%s", tsr, isr, tsr & NE_TSR_ABT ? " (EXCESS COLLISIONS)" : isr ? " (LOST INTERRUPT)" : "");
	RESET(ne);
	NS8390_INIT(ne, 1);
	NS8390_DEQUEUE(ne);
}

static void NS8390_TX_INTR(NE *ne)
{
	io_t io = ne->io;
	/*int status = io_inb(io + NE0_TSR);*/
	io_outb_p(io + NE0_ISR, NE_ISR_PTX);
	if (__unlikely(!ne->txing)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "TX INTERRUPT WHILE NOT TRANSMITTING");
		return;
	}
#ifdef EI_PINGPONG
	if (ne->txing == 1) {
		ne->tx1 = 0;
		ne->txing = 0;
		if (__likely(ne->tx2)) {
			NS8390_TRIGGER_SEND(ne, ne->tx2, ne->tx_start_page + TX_1X_PAGES);
			ne->txing = 2;
		}
	} else {
		ne->tx2 = 0;
		ne->txing = 0;
		if (__likely(ne->tx1)) {
			NS8390_TRIGGER_SEND(ne, ne->tx1, ne->tx_start_page);
			ne->txing = 1;
		}
	}
#else
	ne->txing = 0;
#endif
	NS8390_DEQUEUE(ne);
}

static void NS8390_TX_ERR(NE *ne)
{
	io_t io = ne->io;
	int txsr = io_inb_p(io + NE0_TSR);
	io_outb_p(io + NE0_ISR, NE_ISR_TXE);
	if (txsr & NE_TSR_ABT) if (ne->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, ne->dev_name, "EXCESSIVE COLLISIONS");
	if (txsr & NE_TSR_CRS) if (ne->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ne->dev_name, "CARRIER SENSE LOST");
	if (txsr & NE_TSR_FU) KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "FIFO UNDERRUN ON SEND");
	if (txsr & NE_TSR_CDH) if (ne->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ne->dev_name, "CD HEARTBEAT FAILURE");
	if (txsr & NE_TSR_OWC) if (ne->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ne->dev_name, "OUT OF WINDOW COLLISION");
	if (txsr & (NE_TSR_ABT | NE_TSR_FU)) NS8390_TX_INTR(ne);
}

DECL_AST(NS8390_PACKET_RETURNED, SPL_NE, PACKET)
{
	NE *ne;
	PKT_AST_ENTER(RQ);
	ne = RQ->sender_data;
	FREE_PACKET(RQ, &ne->pool, SPL_NE);
	ne->outstanding--;
	RETURN;
}

static void NS8390_RECEIVE(NE *ne)
{
	struct header_8390 rx_frame;
	io_t io = ne->io;
	int rx_pkt_count = 0;
	int num_rx_pages, rxing_page, this_frame, next_frame;
	num_rx_pages = ne->stop_page - ne->rx_start_page;
	while (++rx_pkt_count < 128) {
		/* note: we must clear receive interrupt before starting copying
		   packet. Linux got it wrong --- cleared interrupt after
		   this loop and it led to transmit timeouts under load
		   (I don't know how this can lead to a transmit timeout but it
		   did). */
		io_outb_p(io + NE0_ISR, NE_ISR_PRX | NE_ISR_RXE);
		io_outb_p(io + NE_CMD, NE_CMD_RD2 | NE_CMD_PS0);
		rxing_page = io_inb_p(io + NE1_CURR);
		io_outb_p(io + NE_CMD, NE_CMD_RD2);
		this_frame = io_inb_p(io + NE0_BNRY) + 1;
		if (__unlikely(this_frame >= ne->stop_page))
			this_frame = ne->rx_start_page;
		if (__unlikely(this_frame == rxing_page)) break;
		ne->input(ne, this_frame, (__u8 *)&rx_frame, sizeof(rx_frame));
		next_frame = this_frame + 1 + ((rx_frame.count) >> 8);
		if (__unlikely(rx_frame.next != next_frame) &&
		    __unlikely(rx_frame.next != next_frame + 1) &&
		    __unlikely(rx_frame.next != next_frame - num_rx_pages) &&
		    __unlikely(rx_frame.next != next_frame + 1 - num_rx_pages)) {
			io_outb(io + NE0_BNRY, rxing_page - 1);
			continue;
		}
		if (__unlikely(rx_frame.count < 64) || __unlikely(rx_frame.count > 1518)) {
			if (ne->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ne->dev_name, "INVALID PACKET LENGTH %d", rx_frame.count - 4);
		} else if (__likely((rx_frame.status & 0x0f) == NE_RSR_PRX)) {
			PACKET *p;
			ALLOC_PACKET(p, rx_frame.count - 4 - ETHERNET_HEADER + 3, &ne->pool, SPL_NE, {
				if (ne->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_SW_WARNING, ne->dev_name, "OUT OF MEMORY FOR PACKET");
				goto skip;
			});
			ne->input(ne, this_frame, p->data + LL_HEADER - 4 - ETHERNET_HEADER, rx_frame.count);
			p->data_length = rx_frame.count - 4 - ETHERNET_HEADER;
			p->addrlen = ETHERNET_ADDRLEN;
			p->fn = NS8390_PACKET_RETURNED;
			p->sender_data = ne;
			p->h = ne->packet_input;
			p->id = IDTYPE_IOPORT | (ne->io & IDTYPE_MASK);
			ne->outstanding++;
			CALL_IORQ(p, KERNEL$PKTIO);
			skip:;
		} else {
			if (rx_frame.status & NE_RSR_FAE) if (ne->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, ne->dev_name, "FRAME ALIGNMENT ERROR");
			if ((rx_frame.status & (NE_RSR_FAE | NE_RSR_CRC)) == NE_RSR_CRC) if (ne->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, ne->dev_name, "CRC ERROR");
			if (rx_frame.status & NE_RSR_FO) KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "FIFO OVERRUN ON RECEIVE");
		}
		next_frame = rx_frame.next;
		if (__unlikely(next_frame >= ne->stop_page)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, ne->dev_name, "NEXT FRAME INCONSISTENCY: %02X", next_frame);
			next_frame = ne->rx_start_page;
		}
		io_outb_p(io + NE0_BNRY, next_frame - 1);
	}
}

static void NS8390_RX_OVERRUN(NE *ne)
{
	io_t io;
	int must_resend;
	int was_txing;
	if (ne->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_SW_WARNING, ne->dev_name, "RECEIVE OVERRUN");
	io = ne->io;
	must_resend = 0;
	was_txing = io_inb_p(io + NE_CMD) & NE_CMD_TXP;
	io_outb_p(io + NE_CMD, NE_CMD_RD2 | NE_CMD_STP);
	KERNEL$UDELAY(2000);
	io_outb_p(io + NE0_RBCR0, 0x00);
	io_outb_p(io + NE0_RBCR1, 0x00);
	if (was_txing) {
		if (!(io_inb_p(io + NE0_ISR) & (NE_ISR_PTX | NE_ISR_TXE))) must_resend = 1;
	}
	io_outb_p(io + NE0_TCR, 0x02);
	io_outb_p(io + NE_CMD, NE_CMD_RD2 | NE_CMD_STA);
	NS8390_RECEIVE(ne);
	io_outb_p(io + NE0_ISR, NE_ISR_OVW);
	io_outb_p(io + NE0_TCR, 0x00);
	if (must_resend) io_outb_p(io + NE_CMD, NE_CMD_RD2 | NE_CMD_STA | NE_CMD_TXP);
}

DECL_AST(NE_IRQ, SPL_NE, AST)
{
	NE *ne = GET_STRUCT(RQ, NE, irq_ast);
	io_t io = ne->io;
	int interrupts, n_int = 0;
	while ((interrupts = io_inb_p(io + NE0_ISR)) && ++n_int < MAX_SERVICE) {
		if (__unlikely(interrupts & (NE_ISR_OVW | NE_ISR_TXE | NE_ISR_CNT | NE_ISR_RDC))) {
			if (interrupts & NE_ISR_OVW) NS8390_RX_OVERRUN(ne);
			if (interrupts & NE_ISR_TXE) NS8390_TX_ERR(ne);
			if (interrupts & NE_ISR_CNT) {
				io_inb_p(io + NE0_CNTR0);
				io_inb_p(io + NE0_CNTR1);
				io_inb_p(io + NE0_CNTR2);
				io_outb_p(io + NE0_ISR, NE_ISR_CNT);
			}
			if (__unlikely(interrupts & NE_ISR_RDC)) {
				io_outb_p(io + NE0_ISR, NE_ISR_RDC);
			}
		}
		if (__likely(interrupts & (NE_ISR_PRX | NE_ISR_RXE))) NS8390_RECEIVE(ne);
		if (interrupts & NE_ISR_PTX) NS8390_TX_INTR(ne);
		io_outb_p(io + NE_CMD, NE_CMD_RD2 | NE_CMD_STA);
	}
	if (__unlikely(n_int == MAX_SERVICE)) {
		if (ne->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, ne->dev_name, "TOO MUCH WORK ON INTERRUPT");
		/*io_outb_p(io + NE0_ISR, interrupts);*/
	}
	/*KERNEL$ADD_RANDOMNESS(&ne->rnd_ctx, NULL, 0);*/
	ne->irq_ctrl.eoi();
	RETURN;
}

DECL_IOCALL(NE_PACKET, SPL_NE, PACKET)
{
	HANDLE *h = RQ->handle;
	NE *ne;
	__CHECK_PKT(RQ, "NE_PACKET");
	if (__unlikely(h->op != &NE_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_NE));
	ne = h->fnode;
	RQ->status = 0;
	if (__likely(NS8390_XMIT(ne, RQ) >= 0)) {
		RETURN_PKT(RQ);
	}
	NETQUE$ENQUEUE_PACKET(ne->queue, RQ);
	RETURN;
}

static void NE_TIMER(TIMER *t)
{
	NE *ne = GET_STRUCT(t, NE, timer);
	LOWER_SPL(SPL_NE);
	VOID_LIST_ENTRY(&ne->timer.list);
	if (__unlikely(ne->txing)) NS8390_TX_TIMEOUT(ne);
}

static void NE_CHECK_MEDIA(NE *ne)
{
	LINK_STATE_PHYS ls;
	if (__likely(ne->flags & REALTEK)) {
		__u8 c0, c2, c3;
		io_t io = ne->io;
		io_outb(io + NE_CMD, NE_CMD_PS0 | NE_CMD_PS1 | NE_CMD_RD2);
		c0 = io_inb(io + NE3_RTL_CONFIG0);
		c2 = io_inb(io + NE3_RTL_CONFIG2);
		c3 = io_inb(io + NE3_RTL_CONFIG3);
		io_outb(io + NE_CMD, NE_CMD_RD2);
		if (__likely(!((c0 ^ ne->c0) | (c2 ^ ne->c2) | (c3 ^ ne->c3) | ne->cx))) return;
		ne->c0 = c0;
		ne->c2 = c2;
		ne->c3 = c3;
		ne->cx = 0;
		memset(&ls, 0, sizeof(LINK_STATE_PHYS));
		ls.speed = 10000000;
		switch (c2 & (NE3_RTL_CONFIG2_PL0 | NE3_RTL_CONFIG2_PL1)) {
			case 0:
				ne->flags |= MEDIA_CHECK;
				if (c0 & NE3_RTL_CONFIG0_BNC) {
					ls.flags |= LINK_STATE_UNKNOWN;
					strlcpy(ls.desc, "10BASE2", sizeof ls.desc);
				} else {
					ls.flags |= LINK_STATE_UP;
					strlcpy(ls.desc, "10BASE-T", sizeof ls.desc);
				}
				ls.flags |= LINK_STATE_MEDIA_AUTOSELECT;
				break;
			case NE3_RTL_CONFIG2_PL0:
				ls.flags |= LINK_STATE_UNKNOWN;
				strlcpy(ls.desc, "10BASE-T", sizeof ls.desc);
				break;
			case NE3_RTL_CONFIG2_PL1:
				ls.flags |= LINK_STATE_UNKNOWN;
				strlcpy(ls.desc, "10BASE5", sizeof ls.desc);
				break;
			case NE3_RTL_CONFIG2_PL0 | NE3_RTL_CONFIG2_PL1:
				ls.flags |= LINK_STATE_UNKNOWN;
				strlcpy(ls.desc, "10BASE2", sizeof ls.desc);
				break;
		}
		if (__unlikely(c3 & NE3_RTL_CONFIG3_FUDUP)) ls.flags |= LINK_STATE_FULL_DUPLEX;
		else ls.flags |= LINK_STATE_HALF_DUPLEX;
	} else {
		memset(&ls, 0, sizeof(LINK_STATE_PHYS));
		ls.speed = 10000000;
		ls.flags |= LINK_STATE_UNKNOWN;
	}
	if (memcmp(&ls, &ne->link_state, sizeof(LINK_STATE_PHYS))) WQ_WAKE_ALL_PL(&ne->link_state_wait);
	memcpy(&ne->link_state, &ls, sizeof(LINK_STATE_PHYS));
}

static void NE_MEDIA_TIMER(TIMER *t)
{
	NE *ne = GET_STRUCT(t, NE, media_timer);
	LOWER_SPL(SPL_NE);
	NE_CHECK_MEDIA(ne);
	KERNEL$SET_TIMER(MEDIA_CHECK_TIME, &ne->media_timer);
}

DECL_IOCALL(NE_IOCTL, SPL_NE, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	NE *ne;
	int r;
	if (__unlikely(h->op != &NE_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_NE));
	ne = h->fnode;
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			ETHERNET_FILL_IF_TYPE(&t);
			t.id = IDTYPE_IOPORT | (ne->io & IDTYPE_MASK);
			memcpy(t.addr, ne->address, ETHERNET_ADDRLEN);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_WAIT_LINK: {
			LINK_STATE ls;
			if (!RQ->v.len) {
				WQ_WAIT(&ne->link_state_wait, RQ, KERNEL$SUCCESS);
				NE_CHECK_MEDIA(ne);
				RETURN;
			}
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, &ls, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			NE_CHECK_MEDIA(ne);
			if (!memcmp(&ne->link_state, &ls, RQ->v.len)) {
				WQ_WAIT_F(&ne->link_state_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_LINK: {
			NE_CHECK_MEDIA(ne);
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &ne->link_state, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SIGNAL_CHANGED_LINK: {
			ne->link_state.seq++;
			WQ_WAKE_ALL_PL(&ne->link_state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_MULTICAST: {
			if (__unlikely(RQ->v.len != 0)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			RQ->status = ne->mcast_state;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SET_MULTICAST: {
			__u8 mcast[ETHERNET_ADDRLEN];
			unsigned bit;
			int need_refresh;
			if (__unlikely(RQ->param == PARAM_MULTICAST_PROMISC)) {
				bit = MCAST_PROMISC;
				goto have_bit;
			} else if (__unlikely(RQ->param == PARAM_MULTICAST_ALL)) {
				bit = MCAST_ALL;
				goto have_bit;
			}
			if (__unlikely(RQ->param != PARAM_MULTICAST_ADDRESS)) {
				RQ->status = -ENOOP;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, mcast, ETHERNET_ADDRLEN)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			bit = NET$ETHERNET_CRC(mcast, ETHERNET_ADDRLEN) >> 26;
			have_bit:
			need_refresh = WQ_EMPTY(&ne->mcast_table[bit]);
			WQ_WAIT_F(&ne->mcast_table[bit], RQ);
			if (need_refresh) NS8390_REFRESH_MULTICAST(ne);
			RETURN;
		}
		case IOCTL_IF_RESET_MULTICAST: {
			NS8390_REFRESH_MULTICAST(ne);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static void ne_init_root(HANDLE *h, void *data)
{
	NE *ne = data;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = ne;
	h->op = &NE_OPERATIONS;
}

int main(int argc, char *argv[])
{
	int r;
	unsigned i;
	int use_pci = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	char *chip_name;
	unsigned long flags = ONLY_16BIT_IO;
	char dev_name[__MAX_STR_LEN];
	char gstr[__MAX_STR_LEN];
	long io = -1;
	int irq = -1;
	int duplex = -1;
	int fce = -1;
	int errorlevel = 0;
	__u8 address[ETHERNET_ADDRLEN];
	char *address_str = NULL;
	char *net_input = NET_INPUT_DEVICE ":";
	struct __param_table params[] = {
		"IO", __PARAM_UNSIGNED_LONG, 0, IO_SPACE_LIMIT - 14, NULL,
		"IRQ", __PARAM_INT, 0, IRQ_LIMIT + 1, NULL,
		"FULL_DUPLEX", __PARAM_BOOL, ~0, 1, NULL,
		"HALF_DUPLEX", __PARAM_BOOL, ~0, 0, NULL,
		"PAUSE_RECEIVE", __PARAM_BOOL, ~0, 1, NULL,
		"NO_PAUSE_RECEIVE", __PARAM_BOOL, ~0, 0, NULL,
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~1, 2, NULL,
		"ADDRESS", __PARAM_STRING, 1, MAXINT, NULL,
		"INPUT", __PARAM_STRING, 1, MAXINT, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **arg;
	int state;
	char *opt, *optend, *str;
	char *e;
	union {
		MALLOC_REQUEST mrq;
		OPENRQ openrq;
		DEVICE_REQUEST devrq;
	} u;

	int start_page, stop_page;
	__u8 SA_prom[32];
	NE *ne;

	PKT_CHECK_VERSION;

	params[0].__p = &io;
	params[1].__p = &irq;
	params[2].__p = &duplex;
	params[3].__p = &duplex;
	params[4].__p = &fce;
	params[5].__p = &fce;
	params[6].__p = &errorlevel;
	params[7].__p = &errorlevel;
	params[8].__p = &address_str;
	params[9].__p = &net_input;
	arg = argv;
	state = 0;
	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (!__strcasexcmp("PCI", opt, optend)) {
			if (!pci_loaded) {
				DLLZRQ lz;
				lz.handle = KERNEL$DLRQ();
				lz.caller = NULL;
				strcpy(lz.modname, "PCI");
				SYNC_IO_CANCELABLE(&lz, KERNEL$DL_LOAD_LAZY);
				if (__unlikely(lz.status < 0)) {
					if (lz.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: PCI NOT ENABLED: %s", lz.error);
					r = lz.status;
					goto ret0;
				}
				pci_loaded = 1;
			}
			use_pci = 1;
		}
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
	/* write pnp initialization here ... when it'll be implemented ... */
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: SYNTAX ERROR");
			badsm:
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (address_str) if (NET$PARSE_ETHERNET_ADDRESS(address_str, address) || !NET$VALID_ETHERNET_ADDRESS(address)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: INVALID ADDRESS: %s", address_str);
		r = -EBADSYN;
		goto ret0;
	}
	if (use_pci) {
		if (io != -1 || irq != -1) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: CAN'T SPECIFY IO OR IRQ WHEN CARD IS ON PCI");
			goto badsm;
		}
		if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: NO PCI DEVICE FOUND");
			r = -ENODEV;
			goto ret0;
		}
		PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO);
		_snprintf(dev_name, sizeof(dev_name), "PKT$NE@" PCI_ID_FORMAT, id);
		io = PCI$READ_IO_RESOURCE(id, 0);
		if (!io) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO RESOURCE");
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", dev_name);
			r = -ENXIO;
			goto ret1;
		}
		irq = PCI$READ_INTERRUPT_LINE(id);
	} else {
		if (io == -1 && irq == -1) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: SPECIFY /IO AND /IRQ OR /PCI");
			goto badsm;
		}
		if (io == -1) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: IO NOT SPECIFIED");
			goto badsm;
		}
		if (irq == -1) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NE: IRQ NOT SPECIFIED");
			goto badsm;
		}
		_snprintf(dev_name, sizeof(dev_name), "PKT$NE@" IO_ID, (io_t)io);
	}

	u.mrq.size = sizeof(NE);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE MEMORY: %s", dev_name, strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret1;
	}
	ne = u.mrq.ptr;
	memset(ne, 0, sizeof(NE));
	strcpy(ne->dev_name, dev_name);
	ne->io = io;
	ne->irq = irq;
	WQ_INIT(&ne->link_state_wait, "NE$LINK_STATE_WAIT");
	INIT_PKTPOOL(&ne->pool, ETHERNET_MTU);

	ne->range.start = io;
	ne->range.len = NE_REGSPACE;
	ne->range.name = ne->dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&ne->range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, ne->range.start, ne->range.start + ne->range.len - 1, strerror(-r));
		goto ret15;
	}

	if (io_inb(io) == 0xff) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "PORT "IO_FORMAT" EMPTY", (io_t)io);
		nodev1:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CARD NOT FOUND", dev_name);
		r = -ENODEV;
		goto ret2;
	}

	io_outb(io + NE_CMD, NE_CMD_RD2 | NE_CMD_PS0 | NE_CMD_STP);
	io_outb(io + NE1_MAR5, 0xff);
	io_outb(io + NE_CMD, NE_CMD_RD2);
	io_inb(io + NE0_CNTR0);
	if (io_inb(io + NE0_CNTR0)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO 8390 AT %lX", io);
		goto nodev1;
	}

	{
		jiffies_t reset, ss;

		io_outb(io + NE_RESET, io_inb(io + NE_RESET));
		reset = KERNEL$GET_JIFFIES();
		while (ss = KERNEL$GET_JIFFIES(), !(io_inb_p(io + NE0_ISR) & NE_ISR_RST)) if (ss - reset > TIMEOUT_JIFFIES(RST_TIMEOUT)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "8390 FAILED TO ACK RESET");
			goto nodev1;
		}
		io_outb_p(io + NE0_ISR, 0xff);
	}

	{
		int i;
		struct {__u8 val; __u8 off;} program_seq[] = {
			{NE_CMD_RD2 | NE_CMD_STP, NE_CMD},
			{0x48, NE0_DCR},
			{0x00, NE0_RBCR0},
			{0x00, NE0_RBCR1},
			{0x00, NE0_IMR},
			{0xFF, NE0_ISR},
			{0x20, NE0_RCR},
			{0x02, NE0_TCR},
			{0x20, NE0_RBCR0},
			{0x00, NE0_RBCR1},
			{0x00, NE0_RSAR0},
			{0x00, NE0_RSAR1},
			{NE_CMD_RD0 | NE_CMD_STA, NE_CMD},
		};
		if (use_pci) program_seq[1].val = 0x49;
		for (i = 0; i < sizeof(program_seq) / sizeof(*program_seq); i++)
			io_outb(io + program_seq[i].off, program_seq[i].val);
	}

	if (use_pci) {
		if (flags & ONLY_32BIT_IO) {	
			int i;
			for (i = 0; i < 4; i++)
				((__u32 *)SA_prom)[i] = __32LE2CPU(io_inl(io + NE_DATAPORT));
			__barrier_ptr(SA_prom);
		} else {
			int i;
			for (i = 0; i < 32; i++)
				SA_prom[i] = io_inb(io + NE_DATAPORT);
		}
	} else {
		int i;
		for (i = 0; i < 32; i += 2) {
			SA_prom[i] = io_inb(io + NE_DATAPORT);
			SA_prom[i + 1] = io_inb(io + NE_DATAPORT);
			if (SA_prom[i] != SA_prom[i + 1])
				flags &= ~ONLY_16BIT_IO, flags |= ONLY_8BIT_IO;
		}
		if (!(flags & ONLY_8BIT_IO)) {
			io_outb_p(io + NE0_DCR, 0x49);
			for (i = 0; i < 16; i++)
				SA_prom[i] = SA_prom[i + i];
		}
	}

	if (!(flags & ONLY_8BIT_IO)) {
		start_page = NESM_START_PG;
		stop_page = flags & STOP_PG_0x60 ? 0x60 : NESM_STOP_PG;
	} else {
		start_page = NE1SM_START_PG;
		stop_page = NE1SM_STOP_PG;
	}

	if (SA_prom[0] == 0x00 && SA_prom[1] == 0x00 && SA_prom[2] == 0x1d) {
		start_page = 0x1;
		stop_page = flags & ONLY_8BIT_IO ? 0x20 : 0x40;
	}

	if (duplex == 1) {
		if (!(flags & REALTEK)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: FULL DUPLEX NOT SUPPORTED BY THIS CARD", dev_name);
			r = -EINVAL;
			goto ret2;
		}
		flags |= FORCE_FDX;
	} else if (!duplex) {
		flags |= FORCE_HDX;
	}
	if (fce == 1) {
		flags |= FORCE_FLOW_CONTROL;
	} else if (!fce) {
		flags |= FORCE_NO_FLOW_CONTROL;
	}

	ne->flags = flags;
	ne->errorlevel = errorlevel;
	ne->tx_start_page = start_page;
	ne->stop_page = stop_page;
	ne->rx_start_page = start_page + TX_PAGES;
	ne->use_pci = use_pci;
	ne->id = id;
	ne->vbuf.ptr = ne->packet_tmp;
	ne->vbuf.len = ETHERNET_MTU;
	ne->vbuf.spl = SPL_X(SPL_NE);
	{
		unsigned i;
		for (i = 0; i < N_MCAST; i++) WQ_INIT(&ne->mcast_table[i], "NE$MCAST_TABLE");
	}
	if (address_str) memcpy(ne->address, address, ETHERNET_ADDRLEN);
	else memcpy(ne->address, SA_prom, ETHERNET_ADDRLEN);
	if (use_pci) ne->prepare_output = NE_PCI_PREPARE_OUTPUT, ne->done_output = NE_PCI_DONE_OUTPUT, ne->input = NE_PCI_INPUT;
	else ne->prepare_output = NE_ISA_PREPARE_OUTPUT, ne->done_output = NE_ISA_DONE_OUTPUT, ne->input = NE_ISA_INPUT;
	if (flags & ONLY_8BIT_IO) ne->output = NE_OUTPUT_8;
	else if (flags & ONLY_16BIT_IO) ne->output = NE_OUTPUT_16;
	else ne->output = NE_OUTPUT_32;

	if (!NET$VALID_ETHERNET_ADDRESS(ne->address)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INVALID ETHERNET ADDRESS: %s", NET$PRINT_ETHERNET_ADDRESS(gstr, ne->address));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID ETHERNET ADDRESS", dev_name);
		r = -EIO;
		goto ret2;
	}

	if (use_pci) _printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, id));
	_printf("%s: IO "IO_FORMAT", IRQ %d\n", dev_name, (io_t)io, irq);
	_printf("%s: ADDRESS %s%s\n", dev_name, NET$PRINT_ETHERNET_ADDRESS(gstr, ne->address), address_str ? " (OVERRIDEN)" : "");

	if ((r = NET$PKTPOOL_RESERVE(&ne->pool))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET POOL: %s", dev_name, strerror(-r));
		goto ret2;
	}

	if ((r = NETQUE$ALLOC_QUEUE(&ne->queue))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", dev_name, strerror(-r));
		goto ret3;
	}

	u.openrq.flags = O_WRONLY;
	u.openrq.path = net_input;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", dev_name, net_input, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto ret4;
	}
	ne->packet_input = u.openrq.status;
	ne->outstanding = 0;

	RAISE_SPL(SPL_NE);
	ne->irq_ast.fn = NE_IRQ;
	if ((e = KERNEL$REQUEST_IRQ(irq, IRQ_AST_HANDLER | (use_pci ? IRQ_SHARED : 0), NULL, &ne->irq_ast, &ne->irq_ctrl, dev_name))) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, e);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		r = -EINVAL;
		goto ret5;
	}
	NS8390_INIT(ne, 1);
	NE_CHECK_MEDIA(ne);
	INIT_TIMER(&ne->media_timer);
	ne->media_timer.fn = NE_MEDIA_TIMER;
	if (!(ne->flags & MEDIA_CHECK)) VOID_LIST_ENTRY(&ne->media_timer.list);
	else KERNEL$SET_TIMER(MEDIA_CHECK_TIME, &ne->media_timer);
	INIT_TIMER(&ne->timer);
	ne->timer.fn = NE_TIMER;
	VOID_LIST_ENTRY(&ne->timer.list);
	LOWER_SPL(SPL_ZERO);

	u.devrq.name = dev_name;
	u.devrq.driver_name = "NE.SYS";
	u.devrq.flags = 0;
	u.devrq.init_root_handle = ne_init_root;
	u.devrq.dev_ptr = ne;
	u.devrq.dcall = NULL;
	u.devrq.dcall_type = NULL;
	u.devrq.dctl = NULL;
	u.devrq.unload = NE_UNLOAD;
	SYNC_IO_CANCELABLE(&u.devrq, KERNEL$REGISTER_DEVICE);
	if (u.devrq.status < 0) {
		if (u.devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = u.devrq.status;
		goto ret6;
	}
	ne->lnte = u.devrq.lnte;
	ne->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;

	ret6:
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&ne->mcast_table[i]);
	ne->irq_ctrl.mask();
	RAISE_SPL(SPL_NE);
	NS8390_INIT(ne, 0);
	KERNEL$RELEASE_IRQ(&ne->irq_ctrl, IRQ_RELEASE_MASKED);
	KERNEL$DEL_TIMER(&ne->timer);
	KERNEL$DEL_TIMER(&ne->media_timer);
	LOWER_SPL(SPL_ZERO);
	while (ne->outstanding) KERNEL$SLEEP(1);
	ret5:
	KERNEL$FAST_CLOSE(ne->packet_input);
	ret4:
	NETQUE$FREE_QUEUE(ne->queue);
	ret3:
	NET$PKTPOOL_FREE(&ne->pool);
	ret2:
	KERNEL$UNREGISTER_IO_RANGE(&ne->range);
	ret15:
	WQ_WAKE_ALL(&ne->link_state_wait);
	KERNEL$UNIVERSAL_FREE(ne);
	ret1:
	if (use_pci) PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int NE_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	unsigned i;
	NE *ne = p;
	if ((r = KERNEL$DEVICE_UNLOAD(ne->lnte, argv))) return r;
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&ne->mcast_table[i]);
	ne->irq_ctrl.mask();
	RAISE_SPL(SPL_NE);
	NS8390_INIT(ne, 0);
	KERNEL$RELEASE_IRQ(&ne->irq_ctrl, IRQ_RELEASE_MASKED);
	KERNEL$DEL_TIMER(&ne->timer);
	KERNEL$DEL_TIMER(&ne->media_timer);
	LOWER_SPL(SPL_ZERO);
	while (ne->outstanding) KERNEL$SLEEP(1);
	KERNEL$FAST_CLOSE(ne->packet_input);
	NETQUE$FREE_QUEUE(ne->queue);
	NET$PKTPOOL_FREE(&ne->pool);
	if (ne->use_pci) PCI$FREE_DEVICE(ne->id);
	*release = ne->dlrq;
	KERNEL$UNREGISTER_IO_RANGE(&ne->range);
	WQ_WAKE_ALL(&ne->link_state_wait);
	KERNEL$UNIVERSAL_FREE(ne);
	return 0;
}

