#include <SPAD/LIBC.H>
#include <STDLIB.H>
#include <VALUES.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/IOCTL.H>
#include <SPAD/THREAD.H>
#include <SPAD/SYSLOG.H>

#include <SPAD/PIO.H>
#include <SPAD/PKT.H>
#include <SPAD/ETHERNET.H>

#define FAST_TRIES			10
#define DEFAULT_NIBBLE_TIMEOUT		3000
#define DEFAULT_TRIGGER_TIMEOUT		100000
#define COLLISION_BACKOFF		200
#define TRANSMIT_BACKOFF		300
#define RECEIVE_LIVELOCK		25
#define ERROR_BACKOFF			100000

typedef struct __plip PLIP;

struct __plip {
	struct pio_client client;
	
	IORQ event;

	int (*dispatcher)(PLIP *);

	unsigned tries;

	unsigned rx_nibble_count;
	unsigned rx_length;
	PACKET *rx_packet;

	unsigned tx_nibble_count;
	PACKET *tx_packet;

	unsigned short mtu;
	__u8 checksum;
	char send_timed;

	int packet_input;
	int outstanding;
	MTX *irq_event;
	PKTQUEUE *queue;
	PKTPOOL pool;

	u_jiffies_lo_t schedule_timeout;

	unsigned nibble_timeout;
	unsigned trigger_timeout;

	unsigned receive_livelock;

	u_jiffies_lo_t error_time;

	char errorlevel;
	char terminate;
	char acquired;

	__u8 address[ETHERNET_ADDRLEN];
	char dev_name[__MAX_STR_LEN];
	char pio_name[__MAX_STR_LEN];
	void *lnte;
	void *dlrq;
};

#define WAIT_NONE		-1
#define WAIT_IRQ		-2
#define WAIT_YIELD		-3
#define WAIT_RESET		-4

extern IO_STUB plip_ioctl;
extern IO_STUB plip_packet;

static __const__ HANDLE_OPERATIONS PLIP_OPERATIONS = {
	SPL_X(SPL_PLIP),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	plip_ioctl,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	plip_packet,		/* pktio */
};

static int dispatcher_receive(PLIP *plip);
static int dispatcher_receive_length(PLIP *plip);
static int dispatcher_receive_data(PLIP *plip);
static int dispatcher_receive_data_dummy(PLIP *plip);
static int dispatcher_send(PLIP *plip);
static void set_nibble(PLIP *plip, __u8 data);
static int dispatcher_send_data(PLIP *plip);
extern AST_STUB plip_packet_returned;
static int plip_error(PLIP *plip);
static int plip_error_dispatcher(PLIP *plip);

static void free_packets(PLIP *plip)
{
	if (__unlikely(plip->rx_packet != NULL)) {
		plip->rx_packet->status = -EIO;
		CALL_PKT(plip->rx_packet);
		plip->rx_packet = NULL;
	}
	if (__unlikely(plip->tx_packet != NULL)) {
		plip->tx_packet->status = -EIO;
		CALL_PKT(plip->tx_packet);
		plip->tx_packet = NULL;
	}
}

static int dispatcher_idle(PLIP *plip)
{
	int r;
	__u8 st;
	if (__unlikely(((unsigned long)plip->rx_packet | (unsigned long)plip->tx_packet) != 0)) free_packets(plip);
	if (__unlikely(!plip->acquired)) {
		plip->irq_event = plip->client.op->acquire(&plip->client);
		/*__debug_printf("attempt acquire %p\n", plip->irq_event);*/
		if (__unlikely(plip->irq_event != NULL)) goto discard_queue;
		plip->acquired = 1;
	}
	if (__unlikely(r = plip->client.op->set_mode(&plip->client, PIO_ECR_MODE_STANDARD | PIO_MODE_NONBLOCK))) {
		KERNEL$SUICIDE("dispatcher_idle: CAN'T SET STANDARD MODE: %d", r);
	}
	if (__unlikely(plip->client.op->want_release(&plip->client))) {
		/*__debug_printf("release\n");*/
		plip->client.op->set_data(&plip->client, 0xff);
		plip->client.op->release(&plip->client);
		plip->acquired = 0;
		plip->irq_event = plip->client.op->acquire(&plip->client);
	/*__debug_printf("attempt acquire 2 %p\n", plip->irq_event);*/
		if (__likely(plip->irq_event != NULL)) goto discard_queue;

	}
	plip->client.op->set_data(&plip->client, 0x00);
	plip->irq_event = plip->client.op->enable_irq(&plip->client);
	st = plip->client.op->get_status(&plip->client);
	if (st == 0x40) {
		if (__unlikely(++plip->receive_livelock > RECEIVE_LIVELOCK)) {
			if (!NETQUE$QUEUE_EMPTY(plip->queue) && plip->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE LIVELOCK");
			NETQUE$DISCRAD_PACKETS(plip->queue, -EIO);
			plip->receive_livelock = 0;
		}
		plip->tries = 0;
		plip->dispatcher = dispatcher_receive;
		return dispatcher_receive(plip);
	} else if (!st) {
		PACKET *p;
		if (__likely((p = NETQUE$DEQUEUE(plip->queue)) != NULL)) {
			int r;
			__CHECK_PKT(p, "dispatcher_idle");
			DEFRAG_PACKET(p, {
				r = -ENOMEM;
				return_pkt:
				p->status = r;
				CALL_PKT(p);
				return r;
			});
			CHECKSUM_PACKET(p, {
				r = r_;
				goto return_pkt;
			});
			plip->tx_packet = p;
			plip->tries = 0;
			plip->dispatcher = dispatcher_send;
			return dispatcher_send(plip);
		}
	} else {
		discard_queue:
		plip->receive_livelock = 0;
		NETQUE$DISCRAD_PACKETS(plip->queue, -ENETDOWN);
	}
	return WAIT_IRQ;
}

static int dispatcher_receive(PLIP *plip)
{
	/* Receive Linux weirdness:
		Linux sets 0x08 on data lines and waits for seeing 0x01.
		If it temporarily times out, it sets 0x00 and expects 0x00.

		After 10 temporary timeouts, it reports error and abandons
		the packet.

		We must reproduce it here, fiddling with lines like Linux wants.
		When we are here, we never know if Linux started sending us data
		or if it just temporarily timed out. When it sets something
		different from 0x00 and 0x08, we are sure that it is data.

		TODO: should we use both IRQ and YIELD to wait?
		--- doesn't seem necessary --- in 10 retries we eventually get
		in sync with the party.
	*/
	__u8 st = plip->client.op->get_status(&plip->client);
	if (st == 0x40) {
		plip->client.op->set_data(&plip->client, 0x01);
	} else if (!st) {
		plip->client.op->set_data(&plip->client, 0x00);
	} else {
		plip->tries = 0;
		plip->rx_length = 0;
		plip->rx_nibble_count = 0;
		plip->dispatcher = dispatcher_receive_length;
		return dispatcher_receive_length(plip);
	}
	KERNEL$UDELAY(1);
	plip->tries++;
	if (__unlikely(plip->tries >= plip->trigger_timeout)) {
		if (plip->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE TRIGGER TIMEOUT");
		return plip_error(plip);
	}
	return WAIT_NONE;
}

static int receive_nibble(PLIP *plip)
{
	__u8 st;
	while (!((((st = plip->client.op->get_status(&plip->client)) >> 7) ^ plip->rx_nibble_count) & 1)) {
		plip->tries++;
		if (__unlikely(plip->tries >= FAST_TRIES)) {
			if (__unlikely(plip->tries >= plip->nibble_timeout)) {
				if (plip->errorlevel >= 2) {
					if (plip->rx_nibble_count >= 4) KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE TIMEOUT (NIBBLE %u OF %u)", plip->rx_nibble_count, (plip->rx_length << 1) + 6);
					else KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "RECEIVE TIMEOUT (NIBBLE %u)", plip->rx_nibble_count);
				}
				return plip_error(plip);
			}
			KERNEL$UDELAY(1);
			return WAIT_NONE;
		}
	}
	plip->client.op->set_data(&plip->client, (~plip->rx_nibble_count & 1) << 4);
	plip->tries = 0;
	return (st >> 3) & 0x0f;
}

static int dispatcher_receive_length(PLIP *plip)
{
	int r;
	PACKET *p;
	if (__unlikely((r = receive_nibble(plip)) < 0)) return r;
	plip->rx_length |= r << (plip->rx_nibble_count++ << 2);
	if (__unlikely(plip->rx_nibble_count < 4)) return WAIT_NONE;
	if (__unlikely(plip->rx_length - ETHERNET_HEADER > plip->mtu)) {
		if (plip->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, plip->dev_name, "INVALID PACKET LENGTH: %04X", plip->rx_length);
		return plip_error(plip);
	}
	ALLOC_PACKET(p, plip->rx_length - ETHERNET_HEADER + 1, &plip->pool, SPL_PLIP, {
		plip->dispatcher = dispatcher_receive_data_dummy;
		return WAIT_NONE;
	});
	p->data_length = plip->rx_length - ETHERNET_HEADER;
	p->addrlen = ETHERNET_ADDRLEN;
	p->fn = plip_packet_returned;
	p->sender_data = plip;
	p->h = plip->packet_input;
	p->id = plip->client.net_id;
	plip->outstanding++;
	plip->dispatcher = dispatcher_receive_data;
	plip->rx_packet = p;
	plip->checksum = 0;
	return WAIT_NONE;
}

static int dispatcher_receive_data(PLIP *plip)
{
	int r;
	PACKET *p = plip->rx_packet;
	if (__unlikely((r = receive_nibble(plip)) < 0)) return r;
	if (!(plip->rx_nibble_count & 1)) {
		p->data[LL_HEADER - ETHERNET_HEADER + (plip->rx_nibble_count >> 1) - 2] = r;
	} else {
		__u8 byte;
		byte = p->data[LL_HEADER - ETHERNET_HEADER + (plip->rx_nibble_count >> 1) - 2] |= r << 4;
		if (__unlikely(plip->rx_nibble_count == (plip->rx_length << 1) + 5)) {
			plip->client.op->set_data(&plip->client, 0x00);
			if (__unlikely(plip->checksum != byte)) {
				if (plip->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "CHECKSUM MISMATCH, COMPUTED %02X, RECEIVED %02X", plip->checksum, byte);
				return WAIT_RESET;
			}
			/*{
				int i;
				__debug_printf("receive:");
				for (i = -ETHERNET_HEADER; i < p->data_length; i++) __debug_printf(" %02x", p->data[LL_HEADER + i]);
				__debug_printf("\n");
			}*/
			CALL_IORQ(p, KERNEL$PKTIO);
			plip->rx_packet = NULL;
			return WAIT_RESET;
		} else {
			plip->checksum += byte;
		}
	}
	plip->rx_nibble_count++;
	return WAIT_NONE;
}

static int dispatcher_receive_data_dummy(PLIP *plip)
{
	int r;
	if (__unlikely((r = receive_nibble(plip)) < 0)) return r;
	if (__unlikely(plip->rx_nibble_count == (plip->rx_length << 1) + 5)) {
		plip->client.op->set_data(&plip->client, 0x00);
		return WAIT_RESET;
	}
	plip->rx_nibble_count++;
	return WAIT_NONE;
}

DECL_IOCALL(plip_event, SPL_PLIP, IORQ)
{
	PLIP *plip = GET_STRUCT(RQ, PLIP, event);
	int r;
	u_jiffies_lo_t last_block = KERNEL$GET_JIFFIES_LO();
	retry:
	plip->irq_event = NULL;
	while ((r = plip->dispatcher(plip)) == WAIT_NONE) {
		if (__unlikely(KERNEL$GET_JIFFIES_LO() - last_block > plip->schedule_timeout) /*&& KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_KERNEL_IORQS*/) {
			/*if (plip->errorlevel >= 2)
				KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "YIELDING");*/
			r = WAIT_YIELD;
			break;
		}
	}
	if (__unlikely(plip->terminate)) {
		plip->terminate = 2;
		RETURN;
	}
	if (r == WAIT_IRQ) {
		MTX_WAIT(plip->irq_event, &plip->event, plip_event);
	} else if (r == WAIT_RESET) {
		plip->dispatcher = dispatcher_idle;
		if (1 || __unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ONE_PASS))
			goto wait_lockup;
		goto retry;
	} else {
		wait_lockup:
		WQ_WAIT(&KERNEL$LOCKUP_EVENTS, &plip->event, plip_event);
	}
	RETURN;
}

DECL_IOCALL(plip_packet, SPL_PLIP, PACKET)
{
	HANDLE *h = RQ->handle;
	PLIP *plip;
	__CHECK_PKT(RQ, "plip_packet");
	if (__unlikely(h->op != &PLIP_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_PLIP));
	plip = h->fnode;
	NETQUE$ENQUEUE_PACKET(plip->queue, RQ);
	if (__likely(plip->irq_event != NULL)) MTX_SET(plip->irq_event, 0);
	RETURN;
}

int dispatcher_send(PLIP *plip)
{
	__u8 st;
	if (!plip->tries) {
		st = plip->client.op->get_status(&plip->client);
		if (__unlikely(st == 0x40)) goto maybe_retry;
		plip->client.op->set_data(&plip->client, 0x08);
	}
	KERNEL$UDELAY(1);
	st = plip->client.op->get_status(&plip->client);
	if (__likely(st & 0x08)) {
		st = plip->client.op->get_status(&plip->client);
		if (!(st & 0x08)) goto unstable_status;
		plip->send_timed = 0;
		plip->receive_livelock = 0;
		plip->tx_nibble_count = 0;
		plip->checksum = 0;
		set_nibble(plip, plip->tx_packet->data_length + ETHERNET_HEADER);
		plip->dispatcher = dispatcher_send_data;
		return WAIT_NONE;
	}
	unstable_status:
	if (__unlikely(st == 0x40)) {
		plip->client.op->set_data(&plip->client, 0x00);
		/*if (plip->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "TRANSMIT COLLISION");*/
		KERNEL$UDELAY(random() % COLLISION_BACKOFF);
		maybe_retry:
		/* Retry packet once timeout */
		if (__likely(plip->send_timed ^= 1)) {
			NETQUE$REQUEUE_DEQUEUED_PACKET(plip->queue, plip->tx_packet);
			plip->tx_packet = NULL;
		}
		plip->client.op->set_data(&plip->client, 0x00);
		return WAIT_RESET;
	}
	plip->tries++;
	if (__unlikely(plip->tries >= plip->trigger_timeout)) {
		if (plip->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "TRANSMIT TRIGGER TIMEOUT");
		return plip_error(plip);
	}
	return WAIT_NONE;
}

static void set_nibble(PLIP *plip, __u8 data)
{
	__u8 d;
	plip->tries = 0;
	d = (data & 0x0f) | ((plip->tx_nibble_count & 1) << 4);
	plip->client.op->set_data(&plip->client, d);
	plip->client.op->set_data(&plip->client, d ^ 0x10);
}

static int wait_nibble(PLIP *plip)
{
	__u8 st;
	while (!((((st = plip->client.op->get_status(&plip->client)) >> 7) ^ plip->tx_nibble_count) & 1)) {
		plip->tries++;
		if (__unlikely(plip->tries >= FAST_TRIES)) {
			if (__unlikely(plip->tries >= plip->nibble_timeout)) {
				if (plip->errorlevel >= 2)
					KERNEL$SYSLOG(__SYSLOG_SW_WARNING, plip->dev_name, "TRANSMIT TIMEOUT (NIBBLE %u OF %u)", plip->tx_nibble_count, (plip->tx_packet->length + ETHERNET_HEADER + 3) << 1);
				return plip_error(plip);
			}
			KERNEL$UDELAY(1);
			return WAIT_NONE;
		}
	}
	plip->tx_nibble_count++;
	return 0;
}

static int dispatcher_send_data(PLIP *plip)
{
	__u8 ni;
	int r;
	if (__unlikely((r = wait_nibble(plip)) < 0)) return r;
	if (__unlikely(plip->tx_nibble_count <= 4)) {
		if (plip->tx_nibble_count < 4) {
			ni = (plip->tx_packet->data_length + ETHERNET_HEADER) >> (plip->tx_nibble_count << 2);
		} else {
			ni = plip->tx_packet->data[LL_HEADER - ETHERNET_HEADER];
		/* Linux IP stack hates packets with multicast bit set */
			if (__likely(PKT_PROTOCOL(plip->tx_packet) == NET_PROTOCOL_IP)) ni &= 0xfe;
			plip->checksum = ni;
		}
	} else if (__likely(plip->tx_nibble_count < (plip->tx_packet->data_length + ETHERNET_HEADER + 2) << 1)) {
		ni = plip->tx_packet->data[LL_HEADER - ETHERNET_HEADER + (plip->tx_nibble_count >> 1) - 2];
		plip->checksum += ni & ((plip->tx_nibble_count & 1) - 1);
		ni >>= ((plip->tx_nibble_count & 1) << 2);
	} else if (plip->tx_nibble_count < (plip->tx_packet->data_length + ETHERNET_HEADER + 3) << 1) {
		ni = plip->checksum >> ((plip->tx_nibble_count & 1) << 2);
	} else {
		plip->client.op->set_data(&plip->client, 0x00);
		plip->tx_packet->status = 0;
		CALL_PKT(plip->tx_packet);
		plip->tx_packet = NULL;
			/* Give other party a chance to send */
		KERNEL$UDELAY(TRANSMIT_BACKOFF);
		return WAIT_RESET;
	}
	set_nibble(plip, ni);
	return WAIT_NONE;
}

/*
static void plip_fast_irq(struct pio_client *client)
{
	PLIP *plip = GET_STRUCT(client, PLIP, client);
	__u8 st = plip->client.op->get_status(&plip->client);
	if (__likely(st == 0x40)) {
		plip->intr_receive = 1;
		plip->client.op->set_data(&plip->client, 0x01);
	}
}
*/

DECL_AST(plip_packet_returned, SPL_PLIP, PACKET)
{
	PLIP *plip;
	PKT_AST_ENTER(RQ);
	plip = RQ->sender_data;
	FREE_PACKET(RQ, &plip->pool, SPL_PLIP);
	plip->outstanding--;
	RETURN;
}

/* Error recovery on Linux is tricky --- Linux kernel sets 0x00 and waits in
   error state until it sees 0x00 on data lines --- unfortunatelly we don't know
   when it noticed our 0x00 and left error state.

   If we start sending too early, Linux kernel will notice 0x08 on data lines
   and will wait in error state.

   So wait for 1/20s and hope that Linux noticed us.
*/

static int plip_error(PLIP *plip)
{
	plip->client.op->set_data(&plip->client, 0x00);
	plip->dispatcher = plip_error_dispatcher;
	plip->error_time = KERNEL$GET_JIFFIES_LO();
	return WAIT_NONE;
}

static int plip_error_dispatcher(PLIP *plip)
{
	u_jiffies_lo_t t = KERNEL$GET_JIFFIES_LO() - plip->error_time;
	u_jiffies_lo_t eb;
	USEC_2_JIFFIES_LO(ERROR_BACKOFF, eb);
	if (__unlikely(t > TIMEOUT_JIFFIES(eb))) return WAIT_RESET;
	return WAIT_YIELD;
}

DECL_IOCALL(plip_ioctl, SPL_PLIP, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	PLIP *plip;
	int r;
	if (__unlikely(h->op != &PLIP_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_PLIP));
	plip = h->fnode;
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			t.mtu = plip->mtu;
			t.addrlen = ETHERNET_ADDRLEN;
			t.arptype = htons(1);
			strlcpy(t.type, "PLIP", sizeof(t.type));
			t.id = plip->client.net_id;
			memcpy(t.addr, plip->address, ETHERNET_ADDRLEN);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static void plip_init_root(HANDLE *h, void *data)
{
	PLIP *plip = data;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = plip;
	h->op = &PLIP_OPERATIONS;
}

static int PLIP_UNLOAD(void *p, void **release, char *argv[]);

int main(int argc, char *argv[])
{
	int r;
	char *lpt = NULL;
	int errorlevel = 0;
	char *address_str = NULL;
	int mtu = ETHERNET_MTU;
	int nibble_timeout = DEFAULT_NIBBLE_TIMEOUT;
	int trigger_timeout = DEFAULT_TRIGGER_TIMEOUT;
	int schedule_timeout = -1;
	char *net_input = NET_INPUT_DEVICE ":";
	PLIP *plip;
	char **arg;
	int state;
	union {
		MALLOC_REQUEST mrq;
		OPENRQ openrq;
		DEVICE_REQUEST devrq;
	} u;
	struct __param_table params[] = {
		"LPT", __PARAM_STRING, 1, __MAX_STR_LEN, NULL,
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~1, 2, NULL,
		"ADDRESS", __PARAM_STRING, 1, MAXINT, NULL,
		"MTU", __PARAM_INT, 576, ETHERNET_JUMBO_MTU+1, NULL,
		"INPUT", __PARAM_STRING, 1, MAXINT, NULL,
		"TIMEOUT", __PARAM_INT, FAST_TRIES, 1000001, NULL,
		"TRIGGER_TIMEOUT", __PARAM_INT, FAST_TRIES, 1000001, NULL,
		"SCHEDULE_TIMEOUT", __PARAM_INT, 0, 1000001, NULL,
		NULL, 0, 0, 0, NULL,
	};

	PKT_CHECK_VERSION;

	params[0].__p = &lpt;
	params[1].__p = &errorlevel;
	params[2].__p = &errorlevel;
	params[3].__p = &address_str;
	params[4].__p = &mtu;
	params[5].__p = &net_input;
	params[6].__p = &nibble_timeout;
	params[7].__p = &trigger_timeout;
	params[8].__p = &schedule_timeout;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}

	if (schedule_timeout < 0) {
		/* schedule_timeout = mtu * 30; */
		schedule_timeout = 0;
	}

	u.mrq.size = sizeof(PLIP);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: COULD NOT ALLOCATE MEMORY: %s", strerror(-u.mrq.status));
		r = u.mrq.status;
		goto ret0;
	}
	plip = u.mrq.ptr;
	memset(plip, 0, sizeof(PLIP));
	plip->mtu = mtu;
	plip->errorlevel = errorlevel;
	plip->nibble_timeout = nibble_timeout;
	plip->trigger_timeout = trigger_timeout;
	USEC_2_JIFFIES_LO(schedule_timeout, plip->schedule_timeout);
	plip->schedule_timeout = TIMEOUT_JIFFIES(plip->schedule_timeout);

	if (address_str) {
		if (NET$PARSE_ETHERNET_ADDRESS(address_str, plip->address) || !NET$VALID_ETHERNET_ADDRESS(plip->address)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: INVALID ADDRESS: %s", address_str);
			r = -EBADSYN;
			goto ret1;
		}
	} else {
		memset(plip->address, 0xfe, sizeof plip->address);
	}

	plip->client.size_of_pio_operations = sizeof(struct pio_operations);
	plip->client.size_of_pio_client = sizeof(struct pio_client);
	if (!lpt) {
		size_t n;
		LOGICAL_LIST_REQUEST lrq;
		lrq.prefix = "PIO$";
		SYNC_IO_CANCELABLE(&lrq, KERNEL$LIST_LOGICALS);
		if (lrq.status < 0) {
			if (lrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: CAN'T LIST LOGICAL NAMES");
			r = lrq.status;
			goto ret1;
		}
		for (n = 0; n < lrq.n_entries; n++) {
			if (!(r = KERNEL$DCALL(lrq.entries[n]->name, "PIO", PIO_CMD_ATTACH_MOST_TIME_NOACQ, &plip->client))) {
				strcpy(plip->pio_name, lrq.entries[n]->name);
				KERNEL$FREE_LOGICAL_LIST(&lrq);
				goto ok;
			}
			if (r == -EINTR) {
				KERNEL$FREE_LOGICAL_LIST(&lrq);
				goto ret1;
			}
		}
		if (!lrq.n_entries) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: NO PARALLEL PORTS"), r = -ENXIO;
		else _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ALL PARALLEL PORTS ARE LOCKED"), r = -EBUSY;
		KERNEL$FREE_LOGICAL_LIST(&lrq);
		goto ret1;
	} else {
		if ((r = KERNEL$DCALL(lpt, "PIO", PIO_CMD_ATTACH_MOST_TIME_NOACQ, &plip->client))) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PLIP: CAN'T ATTACH TO PARALLEL PORT DRIVER %s: %s", lpt, strerror(-r));
			goto ret1;
		}
		strcpy(plip->pio_name, lpt);
		__check_logical_name(plip->pio_name, 1);
	}
	ok:
	_snprintf(plip->dev_name, sizeof(plip->dev_name), "PKT$PLIP@%s", plip->pio_name);

	/*if (fastirq) plip->fast_irq = plip_fast_irq;*/
	/*if ((r = plip->client.op->set_mode(&plip->client, PIO_ECR_MODE_STANDARD))) {
		KERNEL$SYSLOG(__SYSLOG_HW_INCAPABILITY, plip->dev_name, "CAN'T SET STANDARD MODE: %s", strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T SET STANDARD MODE", plip->dev_name);
		goto ret2;
	}*/
	INIT_PKTPOOL(&plip->pool, mtu);
	if ((r = NET$PKTPOOL_RESERVE(&plip->pool))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET POOL: %s", plip->dev_name, strerror(-r));
		goto ret2;
	}
	if ((r = NETQUE$ALLOC_QUEUE(&plip->queue))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", plip->dev_name, strerror(-r));
		goto ret3;
	}
	u.openrq.flags = O_WRONLY;
	u.openrq.path = net_input;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", plip->dev_name, net_input, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto ret4;
	}
	plip->packet_input = u.openrq.status;

	u.devrq.name = plip->dev_name;
	u.devrq.driver_name = "PLIP.SYS";
	u.devrq.flags = 0;
	u.devrq.init_root_handle = plip_init_root;
	u.devrq.dev_ptr = plip;
	u.devrq.dcall = NULL;
	u.devrq.dcall_type = NULL;
	u.devrq.dctl = NULL;
	u.devrq.unload = PLIP_UNLOAD;
	SYNC_IO_CANCELABLE(&u.devrq, KERNEL$REGISTER_DEVICE);
	if (u.devrq.status < 0) {
		if (u.devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", plip->dev_name);
		r = u.devrq.status;
		goto ret6;
	}
	/*__debug_printf("status: %02x\n", plip->client.op->get_status(&plip->client));*/

	plip->dispatcher = dispatcher_idle;
	CALL_IORQ(&plip->event, plip_event);

	plip->lnte = u.devrq.lnte;
	plip->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), plip->dev_name);
	return 0;

	ret6:
	KERNEL$FAST_CLOSE(plip->packet_input);
	ret4:
	NETQUE$FREE_QUEUE(plip->queue);
	ret3:
	NET$PKTPOOL_FREE(&plip->pool);
	ret2:
	if (plip->acquired) plip->client.op->set_data(&plip->client, 0xff);
	KERNEL$DCALL(plip->pio_name, "PIO", PIO_CMD_DETACH, &plip->client);
	ret1:
	KERNEL$UNIVERSAL_FREE(plip);
	ret0:
	return r;
}

static int PLIP_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	PLIP *plip = p;
	if ((r = KERNEL$DEVICE_UNLOAD(plip->lnte, argv))) return r;
	RAISE_SPL(SPL_PLIP);
	plip->terminate = 1;
	if (__likely(plip->irq_event != NULL)) MTX_SET(plip->irq_event, 0);
	while (plip->terminate == 1) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SLEEP(1);
		RAISE_SPL(SPL_PLIP);
	}
	free_packets(plip);
	while (plip->outstanding) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SLEEP(1);
		RAISE_SPL(SPL_PLIP);
	}
	LOWER_SPL(SPL_ZERO);
	KERNEL$FAST_CLOSE(plip->packet_input);
	NETQUE$FREE_QUEUE(plip->queue);
	NET$PKTPOOL_FREE(&plip->pool);
	if (plip->acquired) plip->client.op->set_data(&plip->client, 0xff);
	KERNEL$DCALL(plip->pio_name, "PIO", PIO_CMD_DETACH, &plip->client);
	*release = plip->dlrq;
	KERNEL$UNIVERSAL_FREE(plip);
	return 0;
}
