#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/AC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DL.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/PCI.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <ARCH/TIME.H>
#include <SPAD/TIMER.H>
#include <ENDIAN.H>
#include <SPAD/PKT.H>
#include <SPAD/ETHERNET.H>
#include <SPAD/IOCTL.H>
#include <ARCH/IPCHECKS.H>
#include <VALUES.H>
/*#include <SPAD/RANDOM.H>*/

#include "RTLREG.H"

#include "MII.I"

#define USE_MMIO

#define RTL_TCP_SUM_START	DEFAULT_CHECKSUM_POS
#define RTL_RX_CHECKSUM
#define RTL_TX_CHECKSUM

#define MCAST_ALL	64
#define MCAST_PROMISC	65
#define N_MCAST		66

typedef struct __rtl RTL;

struct __rtl {
#ifndef USE_MMIO
	io_t io;
#else
	__u8 *mem;
#endif
	AST irq_ast;
	IRQ_CONTROL irq_ctrl;

	int packet_input;
	int outstanding;
	PKTQUEUE *queue;
	/*RANDOM_CTX rnd_ctx;*/
	PKTPOOL pool;

	int irq;
	unsigned long flags;
	int errorlevel;
	unsigned chipid;

	unsigned cur_rx;
	unsigned cur_tx;
	unsigned dirty_tx;

	__u8 *rx_virt_buffer;
	__u32 rx_vdma_buffer;
	__u8 *tx_virt_buffer[4];
	__u32 tx_vdma_buffer[4];
	vspace_dmaunlock_t *dmaunlock;
	int clusters_allocated;

	TIMER timer;

	__u32 tcr;
	__u32 rcr;
	__u16 bmcr;	/* when I set bit autonegotiation restart, it stops
			   receiving packets until the first packet is sent.
			   Why?
			   --- maybe it's because I didn't enable linkchg
			   interrupt, now it works
			 */
	__u8 msr;
	__u8 msr_mask;
	__u32 tsd;
	int phy_id;

	__u8 address[ETHERNET_ADDRLEN];

	LINK_STATE link_state;
	WQ link_state_wait;
	TIMER media_timer;

	WQ mcast_table[N_MCAST];
	int mcast_state;

	VBUF vbuf;
	__u8 packet_tmp[ETHERNET_MTU];
	
	pci_id_t id;
	void *lnte;
	void *dlrq;
#ifndef USE_MMIO
	IO_RANGE range;
#endif
	char dev_name[__MAX_STR_LEN];
};

static void RTL_RESET_DEQUEUE(RTL *rtl);
static int RTL_UNLOAD(void *p, void **release, char *argv[]);

#define RTL_RING_BUFFER_SIZE	32768
#define RTL_DEFAULT_BURST	512
#define RTL_MAX_PACKET_LENGTH	1516	/* must be multiple of 4 */
#define RTL_EXTRA_TRIES		3
#define RTL_FIFO_THRESH		64
#define TX_TIMEOUT		(JIFFIES_PER_SECOND / 2)
#define MEDIA_CHECK_TIME	(JIFFIES_PER_SECOND * 10)

#ifndef USE_MMIO
#define RTL_READ8(s, i)		io_inb((s)->io + (i))
#define RTL_READ16(s, i)	io_inw((s)->io + (i))
#define RTL_READ32(s, i)	io_inl((s)->io + (i))
#define RTL_WRITE8(s, i, v)	io_outb((s)->io + (i), (v))
#define RTL_WRITE16(s, i, v)	io_outw((s)->io + (i), (v))
#define RTL_WRITE32(s, i, v)	io_outl((s)->io + (i), (v))
#define RTL_WRITE8_F		RTL_WRITE8
#define RTL_WRITE16_F		RTL_WRITE16
#define RTL_WRITE32_F		RTL_WRITE32
#else
#define RTL_READ8(s, i)		mmio_inb((s)->mem + (i))
#define RTL_READ16(s, i)	__16LE2CPU(mmio_inw((s)->mem + (i)))
#define RTL_READ32(s, i)	__32LE2CPU(mmio_inl((s)->mem + (i)))
#define RTL_WRITE8(s, i, v)	mmio_outb((s)->mem + (i), v)
#define RTL_WRITE16(s, i, v)	mmio_outw((s)->mem + (i), __16CPU2LE(v))
#define RTL_WRITE32(s, i, v)	mmio_outl((s)->mem + (i), __32CPU2LE(v))
#define RTL_WRITE8_F(s, i, v)	RTL_WRITE8(s, i, v), RTL_READ8(s, i)
#define RTL_WRITE16_F(s, i, v)	RTL_WRITE16(s, i, v), RTL_READ16(s, i)
#define RTL_WRITE32_F(s, i, v)	RTL_WRITE32(s, i, v), RTL_READ32(s, i)
#endif

#define RTL_8129		0x00000001
#define RTL_DEAD		0x00000002
#define RTL_ADDRESS_OVERRIDE	0x00010000

static __const__ struct pci_id_s pci_cards[] = {
	{0x10ec, 0x8129, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "REALTEK RTL8129", RTL_8129 },
	{0x10ec, 0x8139, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "REALTEK RTL8139", 0 },
	{0x10ec, 0x8138, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "REALTEK RTL8139 CARDBUS", 0 },
	{0x10ec, 0x8100, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "REALTEK RTL8100", 0 },
	{0x1113, 0x1211, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ACCTRON MPX 5030/5038", 0 },
	{0x1500, 0x1360, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "DELTA ELECTRONICS 8139", 0 },
	{0x4033, 0x1360, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "ADDTRON TECHNOLGY 8139", 0 },
	{0x1186, 0x1300, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DFE-530TX+", 0 },
	{0x1186, 0x1340, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "D-LINK DFE-690TXD", 0 },
	{0x13d1, 0xab06, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "EDIMAX EP-4103DL CARDBUS", 0 },
	{0x1259, 0xa117, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "COREGA FETHER CB-TXD", 0 },
	{0x1259, 0xa11e, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "COREGA FETHERII CB-TXD", 0 },
	{0x14ea, 0xab06, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PLANEX RTL8139 CLONE", 0 },
	{0x14ea, 0xab07, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PLANEX FNW-3800-TX", 0 },
	{0x11db, 0x1234, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "SEGA BBA", 0 },
	{0x1432, 0x9130, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "UNKNOWN RTL8139 CLONE", 0 },
	{0x02ac, 0x1012, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "UNKNOWN RTL8139 CLONE", 0 },
	{0x018a, 0x0106, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "LEVELONE FPC-0106TX", 0 },
	{0x126c, 0x1211, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "NORTEL NETWORKS 10/100BASETX", 0 },
	{0x1743, 0x8139, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "PEPPERCON AG ROL-F", 0 },
	{0x021b, 0x8139, PCI_ANY_ID, PCI_ANY_ID, 0, 0, "COMPAQ HNE-300", 0 },
	{PCI_ANY_ID, 0x8139, 0x10ec, 0x8139, 0, 0, "UNKNOWN RTL8139 CLONE", 0 },
	{PCI_ANY_ID, 0x8139, 0x1186, 0x1300, 0, 0, "UNKNOWN RTL8139 CLONE", 0 },
	{PCI_ANY_ID, 0x8139, 0x13d1, 0xab06, 0, 0, "UNKNOWN RTL8139 CLONE", 0 },
	{ 0 },
};

extern IO_STUB RTL_IOCTL;
extern IO_STUB RTL_PACKET;

static __const__ HANDLE_OPERATIONS RTL_OPERATIONS = {
	SPL_X(SPL_RTL),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL, 			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	RTL_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	RTL_PACKET,		/* pktio */
};

static __const__ unsigned rtl_mii[7] = { RTL_BMCR, RTL_BMSR, 0, 0, RTL_ANAR, RTL_ANLPAR, RTL_ANER };

static __finline__ void MII_DELAY(RTL *rtl)
{
	RTL_READ8(rtl, RTL_CONFIG4);
}

static void MII_SYNC(RTL *rtl)
{
	int i;
	for (i = 32; i >= 0; i--) {
		RTL_WRITE8(rtl, RTL_CONFIG4, MDIO_WRITE1);
		MII_DELAY(rtl);
		RTL_WRITE8(rtl, RTL_CONFIG4, MDIO_WRITE1 | MDIO_CLK);
		MII_DELAY(rtl);
	}
}

__u16 MII_READ(void *p, unsigned idx)
{
	RTL *rtl = p;
	if (__likely(rtl->phy_id < 0)) {
		unsigned reg;
		if (__unlikely(idx >= 7)) return 0;
		if (__unlikely(!(reg = rtl_mii[idx]))) return 0;
		return RTL_READ16(rtl, reg);
	} else {
		unsigned mii_cmd = (0xf6 << 10) | (rtl->phy_id << 5) | idx;
		unsigned retval;
		int i;
		MII_SYNC(rtl);
		for (i = 15; i >= 0; i--) {
			unsigned dataval = (mii_cmd & (1 << i)) ? MDIO_DATA_OUT : 0;
			RTL_WRITE8(rtl, RTL_CONFIG4, MDIO_DIR | dataval);
			MII_DELAY(rtl);
			RTL_WRITE8(rtl, RTL_CONFIG4, MDIO_DIR | dataval | MDIO_CLK);
			MII_DELAY(rtl);
		}
		retval = 0;
		for (i = 19; i >= 0; i--) {
			RTL_WRITE8(rtl, RTL_CONFIG4, 0);
			MII_DELAY(rtl);
			retval <<= 1;
			if (RTL_READ8(rtl, RTL_CONFIG4) & MDIO_DATA_IN) retval |= 1;
			RTL_WRITE8(rtl, RTL_CONFIG4, MDIO_CLK);
			MII_DELAY(rtl);
		}
		return (retval >> 1) & 0xffff;
	}
}

void MII_WRITE(void *p, unsigned idx, __u16 val)
{
	RTL *rtl = p;
	if (__likely(rtl->phy_id < 0)) {
		unsigned reg;
		if (__unlikely(idx >= 7)) return;
		if (__unlikely(!(reg = rtl_mii[idx]))) return;
		RTL_WRITE16(rtl, reg, val);
		return;
	} else {
		unsigned mii_cmd = (0x5002 << 16) | (rtl->phy_id << 23) | (idx << 18) | val;
		int i;
		MII_SYNC(rtl);
		for (i = 31; i >= 0; i--) {
			unsigned dataval = (mii_cmd & (1 << i)) ? MDIO_WRITE1 : MDIO_WRITE0;
			RTL_WRITE8(rtl, RTL_CONFIG4, dataval);
			MII_DELAY(rtl);
			RTL_WRITE8(rtl, RTL_CONFIG4, dataval | MDIO_CLK);
			MII_DELAY(rtl);
		}
		for (i = 2; i > 0; i--) {
			RTL_WRITE8(rtl, RTL_CONFIG4, 0);
			MII_DELAY(rtl);
			RTL_WRITE8(rtl, RTL_CONFIG4, MDIO_CLK);
			MII_DELAY(rtl);
		}
	}
}

static __finline__ void EEPROM_DELAY(RTL *rtl)
{
	RTL_READ32(rtl, RTL_9346CR);
}

static unsigned EEPROM_READ(RTL *rtl, unsigned addr, unsigned addr_len)
{
	unsigned read_cmd = (EE_READ_CMD << addr_len) | addr;
	unsigned retval;
	int i;
	RTL_WRITE8(rtl, RTL_9346CR, RTL_9346CR_EEM1);
	RTL_WRITE8(rtl, RTL_9346CR, RTL_9346CR_EEM1 | RTL_9346CR_EECS);
	EEPROM_DELAY(rtl);
	for (i = addr_len + 4; i >= 0; i--) {
		unsigned dataval = (read_cmd & (1 << i)) ? RTL_9346CR_EEDI : 0;
		dataval |= RTL_9346CR_EEM1 | RTL_9346CR_EECS;
		RTL_WRITE8(rtl, RTL_9346CR, dataval);
		EEPROM_DELAY(rtl);
		RTL_WRITE8(rtl, RTL_9346CR, dataval | RTL_9346CR_EESK);
		EEPROM_DELAY(rtl);
	}
	RTL_WRITE8(rtl, RTL_9346CR, RTL_9346CR_EEM1 | RTL_9346CR_EECS);
	EEPROM_DELAY(rtl);
	retval = 0;
	for (i = 16; i > 0; i--) {
		RTL_WRITE8(rtl, RTL_9346CR, RTL_9346CR_EEM1 | RTL_9346CR_EECS | RTL_9346CR_EESK);
		EEPROM_DELAY(rtl);
		retval <<= 1;
		if (RTL_READ8(rtl, RTL_9346CR) & RTL_9346CR_EEDO) retval |= 1;
		RTL_WRITE8(rtl, RTL_9346CR, RTL_9346CR_EEM1 | RTL_9346CR_EECS);
		EEPROM_DELAY(rtl);
	}
	RTL_WRITE8_F(rtl, RTL_9346CR, 0);
	EEPROM_DELAY(rtl);
	return retval;
}

static void RTL_REFRESH_MULTICAST(RTL *rtl);

static int RTL_RESET(RTL *rtl)
{
	int i;
	RTL_WRITE8_F(rtl, RTL_CR, RTL_CR_RST);
	i = 0;
	while (RTL_READ8(rtl, RTL_CR) & RTL_CR_RST) {
		KERNEL$UDELAY(10);
		if (__unlikely(++i >= 10000)) return -1;
	}
	if (__unlikely((rtl->flags & RTL_ADDRESS_OVERRIDE) != 0)) {
		__u32 addr[2] = { 0, 0 };
		memcpy(addr, rtl->address, ETHERNET_ADDRLEN);
		RTL_WRITE8_F(rtl, RTL_9346CR, RTL_9346CR_EEM0 | RTL_9346CR_EEM1);
		RTL_WRITE32_F(rtl, RTL_IDR0, addr[0]);
		RTL_WRITE32_F(rtl, RTL_IDR4, addr[1]);
		RTL_WRITE8_F(rtl, RTL_9346CR, 0);
		/*
		__debug_printf("%08x, %08x\n", RTL_READ32(rtl, RTL_IDR0), RTL_READ32(rtl, RTL_IDR4));
		__debug_printf("%08x, %08x\n", addr[0], addr[1]);
		*/
	}
	if (rtl->msr_mask) RTL_WRITE8_F(rtl, RTL_MSR, (RTL_READ8(rtl, RTL_MSR) & ~rtl->msr_mask) | rtl->msr);
	RTL_WRITE32_F(rtl, RTL_TSAD0, rtl->tx_vdma_buffer[0]);
	RTL_WRITE32_F(rtl, RTL_TSAD1, rtl->tx_vdma_buffer[1]);
	RTL_WRITE32_F(rtl, RTL_TSAD2, rtl->tx_vdma_buffer[2]);
	RTL_WRITE32_F(rtl, RTL_TSAD3, rtl->tx_vdma_buffer[3]);
	RTL_WRITE32_F(rtl, RTL_RBSTART, rtl->rx_vdma_buffer);
	rtl->cur_rx = 0;
	rtl->cur_tx = 0;
	rtl->dirty_tx = 0;
	RTL_READ16(rtl, RTL_ISR);
	RTL_WRITE8_F(rtl, RTL_CR, RTL_CR_TE | RTL_CR_RE);
	RTL_WRITE32_F(rtl, RTL_TCR, rtl->tcr);
	RTL_WRITE32_F(rtl, RTL_RCR, rtl->rcr);
	RTL_WRITE16_F(rtl, RTL_MULINT, 0);
	RTL_WRITE16_F(rtl, RTL_IMR, RTL_IMR_ROK | RTL_IMR_RER | RTL_IMR_TOK | RTL_IMR_TER | RTL_IMR_RXOVW | RTL_IMR_FOVW | RTL_IMR_SERR | RTL_IMR_PUN_LinkChg);
	RTL_WRITE32_F(rtl, RTL_MPC, 0);	/* don't know why is this needed, but FreeBSD has it in its driver and there's comment around it, that it starts the receiver and transmitter */
	RTL_WRITE8_F(rtl, RTL_CR, RTL_CR_TE | RTL_CR_RE); /* I don't know why enabling it second time ? */
	/*
	MII_WRITE(rtl, MII_BMCR, BMCR_RESET);
	i = 0;
	while (RTL_READ16(rtl, RTL_BMCR) & RTL_BMCR_RESET) {
		KERNEL$UDELAY(10);
		if (__unlikely(++i >= 10000)) break;
	}
	*/
	/*KERNEL$UDELAY(1000000);*/
	MII_WRITE(rtl, MII_BMCR, rtl->bmcr);
	/*__debug_printf("bmcr: %04x\n", MII_READ(rtl, MII_BMCR));
	__debug_printf("advertise: %04x\n", MII_READ(rtl, MII_ADVERTISE));
	__debug_printf("expansion: %04x\n", MII_READ(rtl, MII_EXPANSION));*/
	MII_GET_MODE(__likely(!(rtl->flags & RTL_DEAD)) ? rtl : NULL, (LINK_STATE_PHYS *)&rtl->link_state);
	WQ_WAKE_ALL(&rtl->link_state_wait);
	RTL_REFRESH_MULTICAST(rtl);
	return 0;
}

static void RTL_REFRESH_MULTICAST(RTL *rtl)
{
	unsigned i;
	__u32 multicast_hash[2];
	rtl->rcr &= ~(RTL_RCR_AAP | RTL_RCR_AM);
	if (__unlikely(!WQ_EMPTY(&rtl->mcast_table[MCAST_PROMISC]))) {
		rtl->rcr |= RTL_RCR_AAP;
		rtl->mcast_state = STATUS_MULTICAST_PROMISC;
		goto set_rcr;
	}
	multicast_hash[0] = multicast_hash[1] = 0;
	if (__unlikely(!WQ_EMPTY(&rtl->mcast_table[MCAST_ALL]))) {
		multicast_hash[0] = multicast_hash[1] = 0xffffffff;
		rtl->rcr |= RTL_RCR_AM;
		rtl->mcast_state = STATUS_MULTICAST_ALL;
		goto set_mar;
	}
	rtl->mcast_state = STATUS_MULTICAST_NONE;
	for (i = 0; i < 64; i++) if (__unlikely(!WQ_EMPTY(&rtl->mcast_table[i]))) {
		multicast_hash[i / 32] |= 1 << (i & 31);
		rtl->rcr |= RTL_RCR_AM;
		rtl->mcast_state = STATUS_MULTICAST_SOME;
	}
	set_mar:
	RTL_WRITE32_F(rtl, RTL_MAR0, multicast_hash[0]);
	RTL_WRITE32_F(rtl, RTL_MAR4, multicast_hash[1]);
	set_rcr:
	RTL_WRITE32_F(rtl, RTL_RCR, rtl->rcr);
}

static int RTL_XMIT(RTL *rtl, PACKET *pkt)
{
	unsigned entry, tl;
	static __u8 *ind;
	static vspace_unmap_t *unmap;
	if (__unlikely(rtl->flags & RTL_DEAD)) {
		pkt->status = -ENODEV;
		return 1;
	}
	if (__unlikely(pkt->data_length + pkt->v.len > ETHERNET_MTU)) {
		pkt->status = -EMSGSIZE;
		return 1;
	}
	if (rtl->cur_tx - 4 == rtl->dirty_tx) {
		return -1;
	}
#ifndef RTL_TX_CHECKSUM
	CHECKSUM_PACKET(pkt, {
		pkt->status = r_;
		return 1;
	});
#endif
	entry = rtl->cur_tx & 3;
#ifdef RTL_TX_CHECKSUM
	if (__likely(pkt->flags & PKT_OUTPUT_CHECKSUM)) {
		checksum_t ch;
		unsigned chf = CHECKSUM_FROM(pkt);
		__u8 *d = rtl->tx_virt_buffer[entry];
		memcpy(d, &pkt->data[LL_HEADER - ETHERNET_HEADER], ETHERNET_HEADER + chf);
		ch = NET$IP_CHECKSUM_COPY(d + ETHERNET_HEADER + chf, &pkt->data[LL_HEADER] + chf, pkt->data_length - chf);
		*(__u16 *)(d + ETHERNET_HEADER + CHECKSUM_POS(pkt)) = IP_CHECKSUM_FOLD_INVERT(ch);
	} else
#endif
	{
		memcpy(rtl->tx_virt_buffer[entry], &pkt->data[LL_HEADER - ETHERNET_HEADER], ETHERNET_HEADER + pkt->data_length);
	}
	if (__unlikely(pkt->v.len != 0)) {
		ind = NET$MAP_INDIRECT_PACKET(&rtl->vbuf, &pkt->v, &unmap);
		if (__unlikely(__IS_ERR(ind))) {
			pkt->flags |= PKT_PAGING_ERROR;
			pkt->status = __PTR_ERR(ind);
			return 1;
		}
#ifdef RTL_TX_CHECKSUM
		if (__likely(pkt->flags & PKT_OUTPUT_CHECKSUM)) {
			checksum_t ch = NET$IP_CHECKSUM_COPY(rtl->tx_virt_buffer[entry] + ETHERNET_HEADER + pkt->data_length, ind, pkt->v.len);
			__u16 *pch = (__u16 *)(rtl->tx_virt_buffer[entry] + ETHERNET_HEADER + CHECKSUM_POS(pkt));
			ch -= *pch - __unlikely(ch < *pch);
			*pch = IP_CHECKSUM_FOLD_INVERT(ch);
		} else
#endif
		{
			memcpy(rtl->tx_virt_buffer[entry] + ETHERNET_HEADER + pkt->data_length, ind, pkt->v.len);
		}
	}
	if ((tl = ETHERNET_HEADER + pkt->data_length + pkt->v.len) < ETHERNET_ZLEN) {
		memset(rtl->tx_virt_buffer[entry] + ETHERNET_HEADER + pkt->data_length + pkt->v.len, 0, ETHERNET_ZLEN - ETHERNET_HEADER + pkt->data_length + pkt->v.len);
		tl = ETHERNET_ZLEN;
	}
	__write_barrier();
	RTL_WRITE32_F(rtl, RTL_TSD0 + (entry << 2), rtl->tsd | tl);
	rtl->cur_tx++;
	KERNEL$DEL_TIMER(&rtl->timer);
	KERNEL$SET_TIMER(TX_TIMEOUT, &rtl->timer);
	if (__unlikely(pkt->v.len != 0)) {
		RAISE_SPL(SPL_VSPACE);
		unmap(ind);
		LOWER_SPL(SPL_RTL);
	}
	return 0;
}

static __finline__ void RTL_DEQUEUE(RTL *rtl)
{
	while (rtl->cur_tx - 4 != rtl->dirty_tx) {
		PACKET *p = NETQUE$DEQUEUE(rtl->queue);
		if (__unlikely(!p)) break;
		__CHECK_PKT(p, "RTL_DEQUEUE");
#if __DEBUG >= 1
		if (__unlikely(RTL_XMIT(rtl, p) < 0)) KERNEL$SUICIDE("RTL_DEQUEUE: %s: TRANSMIT NOT ALLOWED", rtl->dev_name);
#else
		RTL_XMIT(rtl, p);
#endif
		CALL_PKT(p);
	}
}

static void RTL_TX_INTERRUPT(RTL *rtl)
{
	while (rtl->cur_tx != rtl->dirty_tx) {
		unsigned entry = rtl->dirty_tx & 3;
		unsigned txstatus = RTL_READ32(rtl, RTL_TSD0 + (entry << 2));
		if (!(txstatus & (RTL_TSD_TOK | RTL_TSD_TUN | RTL_TSD_TABT))) break;
		/*if (!(txstatus & RTL_TSD_OWN)) break;*/
		if (__unlikely(txstatus & (RTL_TSD_WC | RTL_TSD_TABT | RTL_TSD_TUN))) {
			if (txstatus & RTL_TSD_TUN) {
				if (((rtl->tsd >> RTL_TSD_ERTXTH_SHIFT) & RTL_TSD_ERTXTH_MASK) != RTL_TSD_ERTXTH_MASK) rtl->tsd += 1 << RTL_TSD_ERTXTH_SHIFT;
				else if (rtl->errorlevel >= 2)
					KERNEL$SYSLOG(__SYSLOG_HW_WARNING, rtl->dev_name, "TX FIFO UNDERRUN");
			} else if (txstatus & RTL_TSD_WC) {
				if (rtl->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, rtl->dev_name, "OUT OF WINDOW COLLISION");
			} else if (txstatus & RTL_TSD_TABT) {
				if (rtl->errorlevel >= 2)
					KERNEL$SYSLOG(__SYSLOG_NET_WARNING, rtl->dev_name, "TRANSMIT ABORTED");
			}
		}
		rtl->dirty_tx++;
	}
	RTL_DEQUEUE(rtl);
}

DECL_AST(RTL_PACKET_RETURNED, SPL_RTL, PACKET)
{
	RTL *rtl;
	PKT_AST_ENTER(RQ);
	rtl = RQ->sender_data;
	FREE_PACKET(RQ, &rtl->pool, SPL_RTL);
	rtl->outstanding--;
	RETURN;
}

static void RTL_RX_INTERRUPT(RTL *rtl)
{
	unsigned total_rcvd = 0;
	while (!(RTL_READ8(rtl, RTL_CR) & RTL_CR_BUFE)) {
		PACKET *p;
		unsigned ring_offset;
		unsigned rx_status;
		unsigned rx_size;
		__barrier();
		ring_offset = rtl->cur_rx & (RTL_RING_BUFFER_SIZE - 1);
		rx_status = __32LE2CPU(*(__u32 *)(rtl->rx_virt_buffer + ring_offset));
		rx_size = rx_status >> 16;
		/*if (__unlikely(rx_size == 0xfff0)) break; not needed as
		  explained on http://www.scyld.com/rtl8139.html */
		if (__unlikely(rx_size - 8 > 4 + ETHERNET_HEADER + ETHERNET_MTU - 8)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, rtl->dev_name, "LOST IN RING, POSITION %d, STATUS %04X, SIZE %d", ring_offset, rx_status & 0xffff, rx_size);
			RTL_RESET_DEQUEUE(rtl);
			return;
		}
		if (__unlikely(rx_status & (RTL_RX_FAE | RTL_RX_CRC | RTL_RX_LONG | RTL_RX_RUNT | RTL_RX_ISE))) {
			unsigned cmd;
			if (!(rx_status & (RTL_RX_FAE | RTL_RX_LONG | RTL_RX_RUNT))) {
				if (rtl->errorlevel >= 1) KERNEL$SYSLOG(__SYSLOG_NET_ERROR, rtl->dev_name, "INVALID PACKET RECEIVED, STATUS %04X", rx_status & 0xffff);
			} else {
				if (rtl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, rtl->dev_name, "INVALID PACKET RECEIVED, STATUS %04X", rx_status & 0xffff);
			}
			cmd = RTL_READ8(rtl, RTL_CR);
			RTL_WRITE8(rtl, RTL_CR, cmd & ~RTL_CR_RE);
			RTL_WRITE8(rtl, RTL_CR, cmd);
			RTL_WRITE32(rtl, RTL_RCR, rtl->rcr);
			rtl->cur_rx = 0;
			return;
		}
		if (__unlikely(rx_size < ETHERNET_HEADER + 4) || __unlikely(rx_size > ETHERNET_HEADER + ETHERNET_MTU + 4)) goto skip_packet;
		ALLOC_PACKET(p, rx_size - ETHERNET_HEADER - 4, &rtl->pool, SPL_RTL, {
			if (rtl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_SW_WARNING, rtl->dev_name, "OUT OF MEMORY FOR PACKET");
			goto skip_packet;
		});
		p->data_length = rx_size - ETHERNET_HEADER - 4;
#ifdef RTL_RX_CHECKSUM
		if (__unlikely(rx_size <= 4 + ETHERNET_HEADER + RTL_TCP_SUM_START)) {
#endif
			memcpy(p->data + LL_HEADER - ETHERNET_HEADER, rtl->rx_virt_buffer + ring_offset + 4, rx_size - 4);
#ifdef RTL_RX_CHECKSUM
		} else {
			checksum_t ch;
			__u8 *ptr = rtl->rx_virt_buffer + ring_offset + 4;
			memcpy(p->data + LL_HEADER - ETHERNET_HEADER, ptr, ETHERNET_HEADER + RTL_TCP_SUM_START);
			ch = NET$IP_CHECKSUM_COPY(p->data + LL_HEADER + RTL_TCP_SUM_START, ptr + ETHERNET_HEADER + RTL_TCP_SUM_START, rx_size - 4 - (ETHERNET_HEADER + RTL_TCP_SUM_START));
			p->checksum.u = MKCHECKSUM(RTL_TCP_SUM_START, IP_CHECKSUM_FOLD(ch));
			p->flags |= PKT_INPUT_CHECKSUM;
		}
#endif
		p->addrlen = ETHERNET_ADDRLEN;
		p->fn = RTL_PACKET_RETURNED;
		p->sender_data = rtl;
		p->h = rtl->packet_input;
		p->id = IDTYPE_PCI | (rtl->id & IDTYPE_MASK);
		rtl->outstanding++;
		CALL_IORQ(p, KERNEL$PKTIO);
		skip_packet:
		__barrier();
		RTL_WRITE32(rtl, RTL_CAPR, (rtl->cur_rx = (rtl->cur_rx + rx_size + 4 + 3) & ~3) - 16);
		if (__unlikely((total_rcvd += rx_size) > RTL_RING_BUFFER_SIZE * 3 / 2)) {
			if (rtl->errorlevel >= 2) KERNEL$SYSLOG(__SYSLOG_NET_WARNING, rtl->dev_name, "TOO MUCH RX WORK ON INTERRUPT");
			RTL_RESET_DEQUEUE(rtl);
			return;
		}
	}
}

static void RTL_PCI_ERROR(RTL *rtl);
static void RTL_CHECK_MEDIA(RTL *rtl);

DECL_AST(RTL_IRQ, SPL_RTL, AST)
{
	RTL *rtl = GET_STRUCT(RQ, RTL, irq_ast);
	unsigned status;
	status = RTL_READ16(rtl, RTL_ISR);
	RTL_WRITE16(rtl, RTL_ISR, status);
	if (__unlikely(rtl->flags & RTL_DEAD)) goto end;
	/*__debug_printf("rtl irq: %x.\n", status);*/
	if (__unlikely(status & (RTL_IMR_SERR | RTL_IMR_PUN_LinkChg))) {
		if (__unlikely(status == 0xffff)) {
			rtl->flags |= RTL_DEAD;
			goto end;
		}
		if (__unlikely(status & RTL_IMR_SERR)) {
			RTL_PCI_ERROR(rtl);
			goto end;
		}
		RTL_CHECK_MEDIA(rtl);
		/*__debug_printf("bmcr: %04x, bmsr: %04x, cscr: %04x\n", RTL_READ16(rtl, RTL_BMCR), RTL_READ16(rtl, RTL_BMSR), RTL_READ16(rtl, RTL_CSCR));*/
	}
	if (status & (RTL_IMR_TOK | RTL_IMR_TER)) RTL_TX_INTERRUPT(rtl);
	if (status & (RTL_IMR_ROK | RTL_IMR_RER | RTL_IMR_RXOVW | RTL_IMR_FOVW)) RTL_RX_INTERRUPT(rtl);

	end:
	/*KERNEL$ADD_RANDOMNESS(&rtl->rnd_ctx, NULL, 0);*/
	rtl->irq_ctrl.eoi();
	RETURN;
}

DECL_IOCALL(RTL_PACKET, SPL_RTL, PACKET)
{
	HANDLE *h = RQ->handle;
	RTL *rtl;
	__CHECK_PKT(RQ, "RTL_PACKET");
	if (__unlikely(h->op != &RTL_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_RTL));
	rtl = h->fnode;
	RQ->status = 0;
	if (__likely(RTL_XMIT(rtl, RQ) >= 0)) {
		RETURN_PKT(RQ);
	}
	NETQUE$ENQUEUE_PACKET(rtl->queue, RQ);
	RETURN;
}

static void RTL_TX_TIMEOUT(RTL *rtl)
{
	if (__unlikely(RTL_READ16(rtl, RTL_ISR) == 0xffff)) rtl->flags |= RTL_DEAD;
	if (!(rtl->flags & RTL_DEAD))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, rtl->dev_name, "TRANSMIT TIMEOUT");
	RTL_RESET_DEQUEUE(rtl);
}

static void RTL_TX_TIMEOUT(RTL *rtl);

static void RTL_TIMER(TIMER *t)
{
	RTL *rtl = GET_STRUCT(t, RTL, timer);
	LOWER_SPL(SPL_RTL);
	VOID_LIST_ENTRY(&rtl->timer.list);
	if (__unlikely(rtl->dirty_tx != rtl->cur_tx)) RTL_TX_TIMEOUT(rtl);
}

static void RTL_CHECK_MEDIA(RTL *rtl)
{
	LINK_STATE_PHYS ls;
	MII_GET_MODE(__likely(!(rtl->flags & RTL_DEAD)) ? rtl : NULL, &ls);
	if (__unlikely(memcmp(&rtl->link_state, &ls, sizeof(LINK_STATE_PHYS)))) {
		memcpy(&rtl->link_state, &ls, sizeof(LINK_STATE_PHYS));
		WQ_WAKE_ALL_PL(&rtl->link_state_wait);
	}
}

static void RTL_MEDIA_TIMER(TIMER *t)
{
	RTL *rtl = GET_STRUCT(t, RTL, media_timer);
	LOWER_SPL(SPL_RTL);
	RTL_CHECK_MEDIA(rtl);
	KERNEL$SET_TIMER(MEDIA_CHECK_TIME, &rtl->media_timer);
}

static void RTL_PCI_ERROR(RTL *rtl)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, rtl->dev_name, "PCI SERR INTERRUPT");
	PCI$WRITE_CONFIG_WORD(rtl->id, PCI_STATUS, PCI$READ_CONFIG_WORD(rtl->id, PCI_STATUS));
	RTL_RESET_DEQUEUE(rtl);
}

static void RTL_RESET_DEQUEUE(RTL *rtl)
{
	RTL_RESET(rtl);
	RTL_DEQUEUE(rtl);
}

DECL_IOCALL(RTL_IOCTL, SPL_RTL, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	RTL *rtl;
	int r;
	if (__unlikely(h->op != &RTL_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_RTL));
	rtl = h->fnode;
	switch (RQ->ioctl) {
		case IOCTL_IF_GET_TYPE: {
			IF_TYPE t;
			memset(&t, 0, sizeof t);
			ETHERNET_FILL_IF_TYPE(&t);
			t.id = IDTYPE_PCI | (rtl->id & IDTYPE_MASK);
			memcpy(t.addr, rtl->address, ETHERNET_ADDRLEN);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &t, sizeof(t))) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_WAIT_LINK: {
			LINK_STATE ls;
			if (!RQ->v.len) {
				WQ_WAIT(&rtl->link_state_wait, RQ, KERNEL$SUCCESS);
				RTL_CHECK_MEDIA(rtl);
				RETURN;
			}
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, &ls, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			RTL_CHECK_MEDIA(rtl);
			if (!memcmp(&rtl->link_state, &ls, RQ->v.len)) {
				WQ_WAIT_F(&rtl->link_state_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_LINK: {
			RTL_CHECK_MEDIA(rtl);
			if (__unlikely(RQ->v.len != sizeof(LINK_STATE)) && __unlikely(RQ->v.len != sizeof(LINK_STATE_PHYS))) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &rtl->link_state, RQ->v.len)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SIGNAL_CHANGED_LINK: {
			rtl->link_state.seq++;
			WQ_WAKE_ALL_PL(&rtl->link_state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_GET_MULTICAST: {
			if (__unlikely(RQ->v.len != 0)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			RQ->status = rtl->mcast_state;
			RETURN_AST(RQ);
		}
		case IOCTL_IF_SET_MULTICAST: {
			__u8 mcast[ETHERNET_ADDRLEN];
			unsigned bit;
			int need_refresh;
			if (__unlikely(RQ->param == PARAM_MULTICAST_PROMISC)) {
				bit = MCAST_PROMISC;
				goto have_bit;
			} else if (__unlikely(RQ->param == PARAM_MULTICAST_ALL)) {
				bit = MCAST_ALL;
				goto have_bit;
			}
			if (__unlikely(RQ->param != PARAM_MULTICAST_ADDRESS)) {
				RQ->status = -ENOOP;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$GET_IOCTL_STRUCT(RQ, mcast, ETHERNET_ADDRLEN)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
			if (__unlikely(r < 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			bit = NET$ETHERNET_CRC(mcast, ETHERNET_ADDRLEN) >> 26;
			have_bit:
			need_refresh = WQ_EMPTY(&rtl->mcast_table[bit]);
			WQ_WAIT_F(&rtl->mcast_table[bit], RQ);
			if (need_refresh) RTL_REFRESH_MULTICAST(rtl);
			RETURN;
		}
		case IOCTL_IF_RESET_MULTICAST: {
			RTL_REFRESH_MULTICAST(rtl);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static void RTL_INIT_ROOT(HANDLE *h, void *data)
{
	RTL *rtl = data;
	h->flags = 0;
	h->flags2 = 0;
	h->fnode = rtl;
	h->op = &RTL_OPERATIONS;
}

int main(int argc, char *argv[])
{
	int r;
	char *e;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	unsigned long flags;
	int errorlevel = 0;
	int burst = RTL_DEFAULT_BURST;
	int bmcr = RTL_BMCR_ANE;
	int msr = 0;
	int msr_mask = 0;
	char *net_input = NET_INPUT_DEVICE ":";
	char *chip_name;
	char dev_name[__MAX_STR_LEN];
	char gstr[__MAX_STR_LEN];
#ifndef USE_MMIO
	io_t io;
#else
	__u8 *mem;
#endif
	int irq;
	RTL *rtl;
	char *opt, *optend, *str;
	unsigned chipid;
	char *chiptype;
	MALLOC_REQUEST mrq;
	CONTIG_AREA_REQUEST car;
	OPENRQ openrq;
	DEVICE_REQUEST devrq;
	VDESC desc;
	VDMA dma;
	int i;
	int manual_load;
	__u32 mac[2];
	char *address_str = NULL;
	struct __param_table params[] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1, NULL,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2, NULL,
		"INPUT", __PARAM_STRING, 1, MAXINT, NULL,
		"BURST", __PARAM_INT, 16, 2048+1, NULL,
		"ADDRESS", __PARAM_STRING, 1, MAXINT, NULL,
		"FULL_DUPLEX", __PARAM_BOOL, RTL_BMCR_ANE | RTL_BMCR_FDX, RTL_BMCR_FDX, NULL,
		"10M", __PARAM_BOOL, RTL_BMCR_ANE | RTL_BMCR_SPD_SET, 0, NULL,
		"100M", __PARAM_BOOL, RTL_BMCR_ANE | RTL_BMCR_SPD_SET, RTL_BMCR_SPD_SET, NULL,
		"AUTO", __PARAM_BOOL, RTL_BMCR_ANE | RTL_BMCR_SPD_SET | RTL_BMCR_FDX, RTL_BMCR_ANE, NULL,
		"PAUSE_RECEIVE", __PARAM_BOOL, RTL_MSR_RXFCE, RTL_MSR_RXFCE, NULL,
		"PAUSE_RECEIVE", __PARAM_BOOL, RTL_MSR_RXFCE, RTL_MSR_RXFCE, NULL,
		"NO_PAUSE_RECEIVE", __PARAM_BOOL, RTL_MSR_RXFCE, 0, NULL,
		"NO_PAUSE_RECEIVE", __PARAM_BOOL, RTL_MSR_RXFCE, RTL_MSR_RXFCE, NULL,
		"PAUSE_SEND", __PARAM_BOOL, RTL_MSR_TXFCE, RTL_MSR_TXFCE, NULL,
		"PAUSE_SEND", __PARAM_BOOL, RTL_MSR_TXFCE, RTL_MSR_TXFCE, NULL,
		"NO_PAUSE_SEND", __PARAM_BOOL, RTL_MSR_TXFCE, 0, NULL,
		"NO_PAUSE_SEND", __PARAM_BOOL, RTL_MSR_TXFCE, RTL_MSR_TXFCE, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **arg = argv;
	int state = 0;

	PKT_CHECK_VERSION;

	params[0].__p = &errorlevel;
	params[1].__p = &errorlevel;
	params[2].__p = &net_input;
	params[3].__p = &burst;
	params[4].__p = &address_str;
	params[5].__p = &bmcr;
	params[6].__p = &bmcr;
	params[7].__p = &bmcr;
	params[8].__p = &bmcr;
	params[9].__p = &msr;
	params[10].__p = &msr_mask;
	params[11].__p = &msr;
	params[12].__p = &msr_mask;
	params[13].__p = &msr;
	params[14].__p = &msr_mask;
	params[15].__p = &msr;
	params[16].__p = &msr_mask;
	while (__parse_params(&arg, &state, params, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RTL: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (burst < 32) burst = 0;
	else if (burst < 64) burst = 1;
	else if (burst < 128) burst = 2;
	else if (burst < 256) burst = 3;
	else if (burst < 512) burst = 4;
	else if (burst < 1024) burst = 5;
	else if (burst < RTL_MAX_PACKET_LENGTH + 16) burst = 6;
	else burst = 7;
	if (PCI$FIND_DEVICE(pci_cards, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "RTL: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}
	_snprintf(dev_name, sizeof(dev_name), "PKT$RTL@" PCI_ID_FORMAT, id);
#ifndef USE_MMIO
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);
	io = PCI$READ_IO_RESOURCE(id, 0);
	if (!io) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
#else
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER);
	mem = PCI$MAP_MEM_RESOURCE(id, 1, RTL_REGSPACE);
	if (!mem) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO MEM RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO MEM RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	if (__IS_ERR(mem)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT MAP MEM RESOURCE: %s", dev_name, strerror(-__PTR_ERR(mem)));
		r = __PTR_ERR(mem);
		goto ret1;
	}
#endif
	irq = PCI$READ_INTERRUPT_LINE(id);

	mrq.size = sizeof(RTL);
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (mrq.status < 0) {
		if (mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE MEMORY: %s", dev_name, strerror(-mrq.status));
		r = mrq.status;
		goto ret15;
	}
	rtl = mrq.ptr;
	memset(rtl, 0, sizeof(RTL));
	strcpy(rtl->dev_name, dev_name);
#ifndef USE_MMIO
	rtl->io = io;
#else
	rtl->mem = mem;
#endif
	rtl->irq = irq;
	rtl->flags = flags;
	rtl->errorlevel = errorlevel;
	rtl->id = id;
	rtl->phy_id = -1;
	WQ_INIT(&rtl->link_state_wait, "RTL$LINK_STATE_WAIT");
	{
		unsigned i;
		for (i = 0; i < N_MCAST; i++) WQ_INIT(&rtl->mcast_table[i], "RTL$MCAST_TABLE");
	}
#ifndef USE_MMIO
	rtl->range.start = io;
	rtl->range.len = RTL_REGSPACE;
	rtl->range.name = rtl->dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&rtl->range))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", dev_name, rtl->range.start, rtl->range.start + rtl->range.len - 1, strerror(-r));
		goto ret2;
	}
#endif
	rtl->vbuf.ptr = rtl->packet_tmp;
	rtl->vbuf.len = ETHERNET_MTU;
	rtl->vbuf.spl = SPL_RTL;
	car.nclusters = (RTL_RING_BUFFER_SIZE + RTL_MAX_PACKET_LENGTH + 16 + RTL_MAX_PACKET_LENGTH * 4 + __PAGE_CLUSTER_SIZE_MINUS_1) >> __PAGE_CLUSTER_BITS;
	car.align = 65535;
	car.flags = CARF_DATA | CARF_PCIDMA | CARF_PHYSCONTIG;
	SYNC_IO_CANCELABLE(&car, KERNEL$VM_GRAB_CONTIG_AREA);
	if (car.status < 0) {
		if (car.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T ALLOCATE DMA BUFFER: %s", dev_name, strerror(-car.status));
		r = car.status;
		goto ret25;
	}
	desc.ptr = (unsigned long)car.ptr;
	desc.len = car.nclusters << __PAGE_CLUSTER_BITS;
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	dma = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &rtl->dmaunlock);
	LOWER_SPL(SPL_ZERO);
	rtl->clusters_allocated = car.nclusters;
	rtl->rx_virt_buffer = car.ptr;
	rtl->rx_vdma_buffer = dma.ptr;
	for (i = 0; i < 4; i++) {
		rtl->tx_virt_buffer[i] = rtl->rx_virt_buffer + RTL_RING_BUFFER_SIZE + RTL_MAX_PACKET_LENGTH + 16 + i * RTL_MAX_PACKET_LENGTH;
		rtl->tx_vdma_buffer[i] = rtl->rx_vdma_buffer + RTL_RING_BUFFER_SIZE + RTL_MAX_PACKET_LENGTH + 16 + i * RTL_MAX_PACKET_LENGTH;
	}

	RTL_WRITE8_F(rtl, RTL_HLTCLK, 'R');

	/* load defaults from eeprom */
	manual_load = 0;
	RAISE_SPL(SPL_RTL);
	RTL_WRITE8_F(rtl, RTL_9346CR, RTL_9346CR_EEM0);
	i = 0;
	while (RTL_READ8(rtl, RTL_9346CR) & RTL_9346CR_EEM0) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$UDELAY(10);
		if (__unlikely(++i >= 10000)) {
			RTL_WRITE8_F(rtl, RTL_9346CR, 0);
			manual_load = 1;
			break;
		}
	}
	LOWER_SPL(SPL_ZERO);
	if (!i) manual_load = 1;

	rtl->tcr = RTL_READ32(rtl, RTL_TCR);
	if (__unlikely(rtl->tcr == 0xffffffffu)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "CHIP FAILS TO RESPOND");
		goto enxio;
	}

	if (manual_load) {
			/* if autoload fails, load address manually.
			   I don't know if it is really needed. But Linux and
			   FreeBSD drivers load it this way, so maybe there are
			   chips for which it's required */
		unsigned addr_len = EEPROM_READ(rtl, 0, 8) == 0x8129 ? 8 : 6;
		for (i = 0; i < 3; i++) {
			unsigned addr = EEPROM_READ(rtl, i + 7, addr_len);
			rtl->address[i * 2] = addr;
			rtl->address[i * 2 + 1] = addr >> 8;
		}
		rtl->flags |= RTL_ADDRESS_OVERRIDE;
	}


	chipid = ((rtl->tcr >> 22) & 3) | ((rtl->tcr >> 24) & 0x7c);

	switch (chipid) {
		case 0x60: chiptype = "RTL8139"; break;
		case 0x70: chiptype = "RTL8139A"; break;
		case 0x74: chiptype = "RTL8139A-G"; break;
		case 0x78: chiptype = "RTL8139B"; break;
		case 0x7a: chiptype = "RTL8100"; break;
		case 0x75: chiptype = "RTL8139D"; break;
		case 0x76: chiptype = "RTL8139C+"; break;
		case 0x77: chiptype = "RTL8101"; break;
		default: chiptype = NULL;
	}

	if (rtl->flags & RTL_8129) {
		int i;
		for (i = 0; i < 32; i++) {
			int stat;
			rtl->phy_id = i;
			stat = MII_READ(rtl, MII_BMSR);
			if (stat > 0 && stat < 0xffff) goto phy_ok;
		}
		rtl->phy_id = -1;
		phy_ok:;
	}

	rtl->tcr &= ~0x030707f0;
	rtl->tcr |= RTL_TCR_IFG0 | RTL_TCR_IFG1 | (burst << RTL_TCR_MXDMA_SHIFT) | (RTL_EXTRA_TRIES << RTL_TCR_TXRR_SHIFT);
	rtl->rcr = RTL_READ32(rtl, RTL_RCR);
	rtl->rcr &= ~0x0f03ffbf;
	rtl->rcr |= RTL_RCR_APM | RTL_RCR_AB | RTL_RCR_WRAP | (burst << RTL_RCR_MXDMA_SHIFT) |
#if RTL_RING_BUFFER_SIZE == 8192
		0
#elif RTL_RING_BUFFER_SIZE == 16384
		RTL_RCR_RBLEN0
#elif RTL_RING_BUFFER_SIZE == 32768
		RTL_RCR_RBLEN1
#else
			invalid ring buffer size
#endif
		| (0 << RTL_RCR_RXFTH_SHIFT) | (0 << RTL_RCR_ERTH_SHIFT);

	/*if (bmcr & RTL_BMCR_ANE) bmcr |= RTL_BMCR_RESTART;*/
	rtl->bmcr = bmcr;
	if (rtl->flags & RTL_8129) {
		rtl->msr = 0;
		rtl->msr_mask = 0;
	} else {
		rtl->msr = msr;
		rtl->msr_mask = msr_mask;
	}

	rtl->tsd = (RTL_FIFO_THRESH >> 5) << RTL_TSD_ERTXTH_SHIFT;

	if (address_str) {
		if (NET$PARSE_ETHERNET_ADDRESS(address_str, rtl->address) || !NET$VALID_ETHERNET_ADDRESS(rtl->address)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID ADDRESS: %s", dev_name, address_str);
			r = -EBADSYN;
			goto ret3;
		}
		rtl->flags |= RTL_ADDRESS_OVERRIDE;
	} else {
		if (!manual_load) {
			mac[0] = RTL_READ32(rtl, RTL_IDR0);
			mac[1] = RTL_READ32(rtl, RTL_IDR4);
			memcpy(rtl->address, mac, ETHERNET_ADDRLEN);
		}
		if (!NET$VALID_ETHERNET_ADDRESS(rtl->address)) {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "INVALID ETHERNET ADDRESS: %s", NET$PRINT_ETHERNET_ADDRESS(gstr, rtl->address));
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: INVALID ETHERNET ADDRESS", dev_name);
			r = -EIO;
			goto ret3;
		}
	}

	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, id));
#ifndef USE_MMIO
	_printf("%s: IO "IO_FORMAT", IRQ %d", dev_name, io, irq);
#else
	_printf("%s: MEM %"__64_format"X, IRQ %d", dev_name, PCI$READ_MEM_RESOURCE(id, 1, NULL), irq);
#endif
	if (chiptype) _printf(", CHIP %s", chiptype);
	_printf("\n");
	_printf("%s: ADDRESS %s%s\n", dev_name, NET$PRINT_ETHERNET_ADDRESS(gstr, rtl->address), address_str ? " (OVERRIDEN)" : "");


	INIT_PKTPOOL(&rtl->pool, ETHERNET_MTU);
	if (__unlikely(r = NET$PKTPOOL_RESERVE(&rtl->pool))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET POOL: %s", dev_name, strerror(-r));
		goto ret3;
	}
	if (__unlikely(r = NETQUE$ALLOC_QUEUE(&rtl->queue))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE PACKET QUEUE: %s", dev_name, strerror(-r));
		goto ret4;
	}

	openrq.flags = O_WRONLY;
	openrq.path = net_input;
	openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&openrq, KERNEL$OPEN);
	if (openrq.status < 0) {
		if (openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: UNABLE TO OPEN NETWORK PACKET RECEIVE DEVICE %s: %s", dev_name, net_input, strerror(-openrq.status));
		r = openrq.status;
		goto ret5;
	}
	rtl->packet_input = openrq.status;
	rtl->outstanding = 0;


	RAISE_SPL(SPL_RTL);
	rtl->irq_ast.fn = RTL_IRQ;
	if ((e = KERNEL$REQUEST_IRQ(irq, IRQ_AST_HANDLER | IRQ_SHARED, NULL, &rtl->irq_ast, &rtl->irq_ctrl, dev_name))) {
		LOWER_SPL(SPL_ZERO);
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, e);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		r = -EINVAL;
		goto ret6;
	}

	INIT_TIMER(&rtl->timer);
	rtl->timer.fn = RTL_TIMER;
	VOID_LIST_ENTRY(&rtl->timer.list);
	INIT_TIMER(&rtl->media_timer);
	rtl->media_timer.fn = RTL_MEDIA_TIMER;
	VOID_LIST_ENTRY(&rtl->media_timer.list);

	if (__unlikely(RTL_RESET(rtl))) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "CARD FAILED TO RESET");
		enxio:
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CARD CAN'T BE INITIALIZED", dev_name);
		r = -ENXIO;
		goto ret7;
	}
	if (rtl->flags & RTL_8129) {
		KERNEL$DEL_TIMER(&rtl->media_timer);
		KERNEL$SET_TIMER(MEDIA_CHECK_TIME, &rtl->media_timer);
	}
	LOWER_SPL(SPL_ZERO);
	/*__debug_printf("rtl: 52: %02x, d8: %02x\n", RTL_READ8(rtl, 0x52), RTL_READ8(rtl, 0xd8));*/

	devrq.name = dev_name;
	devrq.driver_name = "RTL.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = RTL_INIT_ROOT;
	devrq.dev_ptr = rtl;
	devrq.dcall = NULL;
	devrq.dcall_type = NULL;
	devrq.dctl = NULL;
	devrq.unload = RTL_UNLOAD;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		r = devrq.status;
		goto ret7;
	}
	rtl->lnte = devrq.lnte;
	rtl->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;
	
	ret7:
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&rtl->mcast_table[i]);
	rtl->irq_ctrl.mask();
	RAISE_SPL(SPL_RTL);
	RTL_WRITE16_F(rtl, RTL_IMR, 0);
	RTL_WRITE16(rtl, RTL_ISR, RTL_READ16(rtl, RTL_ISR));
	RTL_WRITE8_F(rtl, RTL_CR, RTL_CR_RST);
	KERNEL$RELEASE_IRQ(&rtl->irq_ctrl, IRQ_RELEASE_MASKED);
	KERNEL$DEL_TIMER(&rtl->media_timer);
	KERNEL$DEL_TIMER(&rtl->timer);
	LOWER_SPL(SPL_ZERO);
	while (rtl->outstanding) KERNEL$SLEEP(1);
	ret6:
	KERNEL$FAST_CLOSE(rtl->packet_input);
	ret5:
	NETQUE$FREE_QUEUE(rtl->queue);
	ret4:
	NET$PKTPOOL_FREE(&rtl->pool);
	ret3:
	RAISE_SPL(SPL_VSPACE);
	rtl->dmaunlock(rtl->rx_vdma_buffer);
	LOWER_SPL(SPL_ZERO);
	KERNEL$VM_RELEASE_CONTIG_AREA(rtl->rx_virt_buffer, rtl->clusters_allocated);
	ret25:
#ifndef USE_MMIO
	KERNEL$UNREGISTER_IO_RANGE(&rtl->range);
	ret2:
#endif
	WQ_WAKE_ALL(&rtl->link_state_wait);
	KERNEL$UNIVERSAL_FREE(rtl);
	ret15:
#ifdef USE_MMIO
	PCI$UNMAP_MEM_RESOURCE(id, mem, RTL_REGSPACE);
#endif
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

static int RTL_UNLOAD(void *p, void **release, char *argv[])
{
	int r;
	unsigned i;
	RTL *rtl = p;
		/*__debug_printf("bmcr: %04x, bmsr: %04x, anar: %04x, anlpar: %04x, aner: %04x\n", RTL_READ16(rtl, RTL_BMCR), RTL_READ16(rtl, RTL_BMSR), RTL_READ16(rtl, RTL_ANAR), RTL_READ16(rtl, RTL_ANLPAR), RTL_READ16(rtl, RTL_ANER));*/
	if ((r = KERNEL$DEVICE_UNLOAD(rtl->lnte, argv))) return r;
	for (i = 0; i < N_MCAST; i++) WQ_WAKE_ALL(&rtl->mcast_table[i]);
	rtl->irq_ctrl.mask();
	RAISE_SPL(SPL_RTL);
	RTL_WRITE16_F(rtl, RTL_IMR, 0);
	RTL_WRITE16(rtl, RTL_ISR, RTL_READ16(rtl, RTL_ISR));
	RTL_WRITE8_F(rtl, RTL_CR, RTL_CR_RST);
	KERNEL$RELEASE_IRQ(&rtl->irq_ctrl, IRQ_RELEASE_MASKED);
	KERNEL$DEL_TIMER(&rtl->media_timer);
	KERNEL$DEL_TIMER(&rtl->timer);
	LOWER_SPL(SPL_ZERO);
	while (rtl->outstanding) KERNEL$SLEEP(1);
	KERNEL$FAST_CLOSE(rtl->packet_input);
	NETQUE$FREE_QUEUE(rtl->queue);
	NET$PKTPOOL_FREE(&rtl->pool);
	RAISE_SPL(SPL_VSPACE);
	rtl->dmaunlock(rtl->rx_vdma_buffer);
	LOWER_SPL(SPL_ZERO);
	KERNEL$VM_RELEASE_CONTIG_AREA(rtl->rx_virt_buffer, rtl->clusters_allocated);
#ifdef USE_MMIO
	PCI$UNMAP_MEM_RESOURCE(rtl->id, rtl->mem, RTL_REGSPACE);
#endif
	PCI$FREE_DEVICE(rtl->id);
	*release = rtl->dlrq;
#ifndef USE_MMIO
	KERNEL$UNREGISTER_IO_RANGE(&rtl->range);
#endif
	WQ_WAKE_ALL(&rtl->link_state_wait);
	KERNEL$UNIVERSAL_FREE(rtl);
	return 0;
}
