#include <SPAD/SYSLOG.H>

#include "SK.H"

static void SKY2_SET_PUT_IDX(SK0 *sk0, unsigned put_idx_reg, unsigned val);
extern IO_STUB SKY2_PACKET;
static void SKY2_UNMAP_TX_DESC_NOINLINE(TX_QUEUE *q, unsigned idx);
static void SKY2_RECEIVE_INVALID_PORT(SK *sk, Y2_DESC *y);
static void SKY2_RECEIVE_NO_PACKETS(MAC *mac);
static void SKY2_TX_INVALID_DONE_INDEX(TX_QUEUE *q, unsigned done);
static void SKY2_TX_TOO_MUCH_DONE(TX_QUEUE *q, unsigned n_done);
static int SKY2_SPECIAL_IRQ(SK *sk, __u32 isr);

__const__ HANDLE_OPERATIONS SKY2_OPERATIONS = {
	SPL_X(SPL_SK),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	SK_LOOKUP,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* open */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* read */
	KERNEL$NO_OPERATION,	/* write */
	KERNEL$NO_OPERATION,	/* aread */
	KERNEL$NO_OPERATION,	/* awrite */
	SK_IOCTL,		/* ioctl */
	KERNEL$NO_OPERATION,	/* bio */
	SKY2_PACKET,		/* pktio */
};

__finline__ void SKY2_KICK_RECEIVE(MAC *mac)
{
	__barrier();
	if (__likely(!Y2_BUG_42(mac))) SK_WRITE_16(mac, SK_Y2_RX_PUT_IDX + (ADD_MAC2_SK_RX & mac->mac_2), (mac->first_recv + mac->n_recv) & (N_DESCRIPTORS - 1));
	else SKY2_SET_PUT_IDX((SK0 *)mac, SK_Y2_RX_PUT_IDX + (ADD_MAC2_SK_RX & mac->mac_2), mac->first_recv + mac->n_recv);
}

static __finline__ void SKY2_KICK_SEND(TX_QUEUE *q)
{
	__barrier();
	if (__likely(!Y2_BUG_42(q))) SK_WRITE_16(q, q->cs_reg, (q->first_sent + q->n_sent) & (N_DESCRIPTORS - 1));
	else SKY2_SET_PUT_IDX((SK0 *)q, q->cs_reg, q->first_sent + q->n_sent);
}

static void SKY2_SET_PUT_IDX(SK0 *sk0, unsigned put_idx_reg, unsigned val)
{
	unsigned hwput;
	val &= (N_DESCRIPTORS - 1);
	hwput = SK_READ_16(sk0, put_idx_reg);
	if (__likely(hwput) && __likely(val >= hwput)) {
		SK_WRITE_16(sk0, put_idx_reg, val);
	} else {
		unsigned hwget = SK_READ_16(sk0, put_idx_reg - SK_Y2_TX_PUT_IDX + SK_Y2_TX_GET_IDX);
		if (__unlikely(hwget == N_DESCRIPTORS - 1)) {
			SK_WRITE_8(sk0, put_idx_reg - SK_Y2_TX_PUT_IDX + SK_Y2_TX_FIFO_WM, 8);
			SK_WRITE_16(sk0, put_idx_reg, 0);
		} else if (__unlikely(!hwget)) {
			SK_WRITE_8(sk0, put_idx_reg - SK_Y2_TX_PUT_IDX + SK_Y2_TX_FIFO_WM, 0xe0);
			SK_WRITE_16(sk0, put_idx_reg, val);
		} else if (__unlikely(hwput != N_DESCRIPTORS - 1) && __likely(hwput)) {
			SK_WRITE_16(sk0, put_idx_reg, N_DESCRIPTORS - 1);
		}
	}
}

void SKY2_ADD_RCV_PACKET(MAC *mac, PACKET *p)
{
	VDMA64 vdma;
	VDESC desc;
	unsigned idx;
	__u32 hi;
	Y2_DESC *y;
#if __DEBUG >= 2
	mac->rcv_maps++;
	if (__unlikely(mac->rcv_maps > N_DESCRIPTORS))
		KERNEL$SUICIDE("SKY2_ADD_RCV_PACKET: MAPS LEAKED");
	if (__unlikely(p->length < mac->mtu))
		KERNEL$SUICIDE("SKY2_ADD_RCV_PACKET: RECEIVE PACKET TOO SHORT (%d)", p->length);
#endif
	p->addrlen = ETHERNET_ADDRLEN;
	desc.ptr = (unsigned long)p->data + LL_HEADER - sizeof(struct eth_header);
	desc.len = mac->mtu + sizeof(struct eth_header);
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	KERNEL$VIRTUAL.op->vspace_dma64lock(&desc, PF_WRITE, &vdma, (vspace_dma64unlock_t **)&p->sender_data);
	LOWER_SPL(SPL_SK);
	if (__unlikely((hi = vdma.ptr >> 32) != mac->recv_hi_cache)) {
		unsigned idx = (mac->first_recv + mac->n_recv++) & (N_DESCRIPTORS - 1);
		y = &mac->u.y2_recv[idx];
		y->rxhi.hiaddr = hi;
		mac->recv_hi_cache = hi;
		y->rxhi.reserved1 = 0;
		y->rxhi.reserved2 = 0;
		__write_barrier();
		y->rxhi.opcode = Y2_OP_ADDR64 | Y2_OP_OWNER;
	}
	idx = (mac->first_recv + mac->n_recv++) & (N_DESCRIPTORS - 1);
#if __DEBUG >= 2
	if (__unlikely(mac->n_recv > N_DESCRIPTORS))
		KERNEL$SUICIDE("SKY2_ADD_RCV_PACKET: RECEIVE DESCRIPTOR SPACE OVERFLOW, n_recv %d", mac->n_recv);
	if (__unlikely(mac->recv_packets[idx] != NULL))
		KERNEL$SUICIDE("SKY2_ADD_RCV_PACKET: PACKET ALREADY ALLOCATED AT THIS SLOT");
#endif
	mac->recv_packets[idx] = p;
	p->v.ptr = vdma.ptr;
	y = &mac->u.y2_recv[idx];
	y->rxpa.loaddr = (__u32)vdma.ptr;
	y->rxpa.len = mac->mtu + sizeof(struct eth_header);
	y->rxpa.ctrl = 0;
	__write_barrier();
	y->rxpa.opcode = Y2_OP_PACKET | Y2_OP_OWNER;
}

static void __finline__ SKY2_UNMAP_RCV_PACKET(MAC *mac, PACKET *p)
{
#if __DEBUG >= 2
	mac->rcv_maps--;
	if (__unlikely(mac->rcv_maps < 0))
		KERNEL$SUICIDE("SKY2_UNMAP_RCV_PACKET: MAP COUNT UNDERFLOW");
#endif
	RAISE_SPL(SPL_VSPACE);
	((vspace_dma64unlock_t *)p->sender_data)(p->v.ptr);
	LOWER_SPL(SPL_SK);
}

static __finline__ void SKY2_UNMAP_TX_DESC(TX_QUEUE *q, unsigned idx)
{
#if __DEBUG >= 2
	q->snd_maps--;
	if (__unlikely(q->snd_maps < 0))
		KERNEL$SUICIDE("SKY2_UNMAP_TX_DESC: MAP COUNT UNDERFLOW");
#endif
	RAISE_SPL(SPL_VSPACE);
	q->xmit_info[idx].unlock(q->xmit_addr[idx]);
	LOWER_SPL(SPL_SK);
}

#define SKY2_CAN_SEND(q)	((q)->n_sent <= (q)->max_sent)

static void no_unlock(__u64 ptr)
{
}

/* Yukon 2 contains bugs that cause BMU hangs. If the bug happens during send,
   transmit timeout will reset the card. However if it happens while there's
   nothing to send, the card stops receiving packets and you won't ever log in
   to that machine remotely. Try to send useless command each 30 seconds to the
   transmit queue. If BMU hung, the timer will reset it */

static __finline__ void SKY2_PROBE_QUEUE(TX_QUEUE *q)
{
	Y2_DESC *y;
	unsigned idx = q->first_sent;
	if (q->n_sent) return;
	q->n_sent = 1;
	q->xmit_info[idx].p = NULL;
	q->xmit_info[idx].unlock = no_unlock;
	y = &q->u.y2_xmit[idx];
	y->txhi.hiaddr = q->send_hi_cache;
	y->txhi.reserved1 = 0;
	y->txhi.reserved2 = 0;
	__write_barrier();
	y->txhi.opcode = Y2_OP_ADDR64 | Y2_OP_OWNER;
#if __DEBUG >= 2
	q->snd_maps++;
	if (__unlikely(q->snd_maps > N_DESCRIPTORS))
		KERNEL$SUICIDE("SKY2_PROBE_QUEUE: MAPS LEAKED");
#endif
	SKY2_KICK_SEND(q);
	KERNEL$DEL_TIMER(&q->timer);
	KERNEL$SET_TIMER(SKY2_TX_TIMEOUT, &q->timer);
}

static void SKY2_SEND(TX_QUEUE *q, PACKET *p)
{
	VDESC desc;
	VDMA64 vdma;
	vspace_dma64unlock_t *unlock;
	unsigned pos;
	Y2_DESC *y;
	__u32 hi;
#ifndef SK_ENABLE_TX_CHECKSUM
	CHECKSUM_PACKET(p, {
		p->status = r_;
		CALL_PKT(p);
		return;
	});
#endif
	desc.ptr = (unsigned long)p->data + LL_HEADER - sizeof(struct eth_header);
	desc.len = p->data_length + sizeof(struct eth_header);
	desc.vspace = &KERNEL$VIRTUAL;
	RAISE_SPL(SPL_VSPACE);
	KERNEL$VIRTUAL.op->vspace_dma64lock(&desc, PF_READ, &vdma, &unlock);
	LOWER_SPL(SPL_SK);
	if (__unlikely((hi = vdma.ptr >> 32) != q->send_hi_cache)) {
		unsigned idx = (q->first_sent + q->n_sent++) & (N_DESCRIPTORS - 1);
		q->xmit_info[idx].p = NULL;
		q->xmit_info[idx].unlock = no_unlock;
		y = &q->u.y2_xmit[idx];
		y->txhi.hiaddr = hi;
		q->send_hi_cache = hi;
		y->txhi.reserved1 = 0;
		y->txhi.reserved2 = 0;
		__write_barrier();
		y->txhi.opcode = Y2_OP_ADDR64 | Y2_OP_OWNER;
#if __DEBUG >= 2
		q->snd_maps++;
		if (__unlikely(q->snd_maps > N_DESCRIPTORS))
			KERNEL$SUICIDE("SKY2_SEND: MAPS LEAKED");
#endif
	}
#ifdef SK_ENABLE_TX_CHECKSUM
	if (__likely(p->flags & PKT_OUTPUT_CHECKSUM) && __unlikely((p->checksum.u != q->checksum_cache))) {
		__u32 cs = q->checksum_cache = p->checksum.u;
		unsigned idx = (q->first_sent + q->n_sent++) & (N_DESCRIPTORS - 1);
		q->xmit_info[idx].p = NULL;
		q->xmit_info[idx].unlock = no_unlock;
		y = &q->u.y2_xmit[idx];
		y->txch.tcp_sum_write = XPOSCHECKSUM(cs) + sizeof(struct eth_header);
		y->txch.tcp_sum_offset = XFROMCHECKSUM(cs) + sizeof(struct eth_header);
		y->txch.tcp_sum_init_val = 0;
		y->txch.lock = 1;
		__write_barrier();
		y->txch.opcode = Y2_OP_TCPWRITE | Y2_OP_TCPSTART | Y2_OP_TCPINIT | Y2_OP_TCPLCK | Y2_OP_OWNER;
#if __DEBUG >= 2
		q->snd_maps++;
		if (__unlikely(q->snd_maps > N_DESCRIPTORS))
			KERNEL$SUICIDE("SKY2_SEND: MAPS LEAKED");
#endif
	}
#endif
#if __DEBUG >= 2
	q->snd_maps++;
	if (__unlikely(q->snd_maps > N_DESCRIPTORS))
		KERNEL$SUICIDE("SKY2_SEND: MAPS LEAKED");
#endif
	pos = (q->first_sent + q->n_sent) & (N_DESCRIPTORS - 1);
	q->xmit_info[pos].unlock = unlock;
	q->xmit_addr[pos] = vdma.ptr;
	y = &q->u.y2_xmit[pos];
	y->txpa.loaddr = (__u32)vdma.ptr;
	y->txpa.len = vdma.len;
#ifdef SK_ENABLE_TX_CHECKSUM
	y->txpa.ctrl = __likely(p->flags & PKT_OUTPUT_CHECKSUM) ? Y2_CTRL_CALSUM | Y2_CTRL_WR_SUM | Y2_CTRL_INIT_SUM | Y2_CTRL_LOCK_SUM | Y2_CTRL_EOP : Y2_CTRL_EOP;
#else
	y->txpa.ctrl = Y2_CTRL_EOP;
#endif
	if (__unlikely(p->v.len != 0)) {
		unsigned ppos;
		y->txpa.ctrl &= ~Y2_CTRL_EOP;
		q->xmit_info[pos].p = NULL;
		ppos = (pos + 1) & (N_DESCRIPTORS - 1);
		next_part:
		RAISE_SPL(SPL_VSPACE);
		p->v.vspace->op->vspace_dma64lock(&p->v, PF_READ, &vdma, &unlock);
		LOWER_SPL(SPL_SK);
		if (__unlikely(!vdma.len)) {
			p->flags |= PKT_PAGING_ERROR;
			p->status = -EVSPACEFAULT;
			unm_drop:
			while (pos != ppos) {
				SKY2_UNMAP_TX_DESC_NOINLINE(q, pos);
				pos = (pos + 1) & (N_DESCRIPTORS - 1);
			}
			CALL_PKT(p);
			goto kick_send;	/* write at least initial hiaddr */
		}
		if (__unlikely((hi = vdma.ptr >> 32) != q->send_hi_cache)) {
			q->xmit_info[ppos].p = NULL;
			q->xmit_info[ppos].unlock = no_unlock;
			y = &q->u.y2_xmit[ppos];
			y->txhi.hiaddr = hi;
			q->send_hi_cache = hi;
			y->txhi.reserved1 = 0;
			y->txhi.reserved2 = 0;
			y->txhi.opcode = Y2_OP_ADDR64 | Y2_OP_OWNER;
#if __DEBUG >= 2
			q->snd_maps++;
			if (__unlikely(q->snd_maps > N_DESCRIPTORS))
				KERNEL$SUICIDE("SKY2_SEND: MAPS LEAKED");
#endif
			ppos = (ppos + 1) & (N_DESCRIPTORS - 1);
			if (__unlikely(((ppos + 1) & (N_DESCRIPTORS - 1)) == q->first_sent)) {
				unlock(vdma.ptr);
				p->status = -EMSGSIZE;
				goto unm_drop;
			}
		}
		q->xmit_info[ppos].unlock = unlock;
		q->xmit_addr[ppos] = vdma.ptr;
		y = &q->u.y2_xmit[ppos];
		y->txpa.loaddr = (__u32)vdma.ptr;
		y->txpa.len = vdma.len;
		y->txpa.opcode = Y2_OP_BUFFER | Y2_OP_OWNER;
#if __DEBUG >= 2
		q->snd_maps++;
		if (__unlikely(q->snd_maps > N_DESCRIPTORS))
			KERNEL$SUICIDE("SKY2_SEND: MAPS LEAKED");
#endif
		if (__unlikely(vdma.len != p->v.len)) {
			y->txpa.ctrl = 0;
			q->xmit_info[ppos].p = NULL;
			ppos = (ppos + 1) & (N_DESCRIPTORS - 1);
			if (__unlikely(((ppos + 1) & (N_DESCRIPTORS - 1)) == q->first_sent)) {
				p->status = -EMSGSIZE;
				goto unm_drop;
			}
			p->v.ptr += vdma.len;
			p->v.len -= vdma.len;
			goto next_part;
		}
		y->txpa.ctrl = Y2_CTRL_EOP;
		__write_barrier();
		q->xmit_info[ppos].p = p;
		q->u.y2_xmit[pos].txpa.opcode = Y2_OP_PACKET | Y2_OP_OWNER;
		q->n_sent += (ppos + 1 - pos) & (N_DESCRIPTORS - 1);
	} else {
		__write_barrier();
		y->txpa.opcode = Y2_OP_PACKET | Y2_OP_OWNER;
		q->xmit_info[pos].p = p;
		q->n_sent++;
	}
	kick_send:
	SKY2_KICK_SEND(q);
	KERNEL$DEL_TIMER(&q->timer);
	KERNEL$SET_TIMER(SKY2_TX_TIMEOUT, &q->timer);
}

DECL_AST(SKY2_PACKET_RETURNED, SPL_SK, PACKET)
{
	MAC *mac;
	PKT_AST_ENTER(RQ);
	mac = RQ->sender_data;
	mac->outstanding--;
	if (__likely(SK_RCVQ_FULL(mac))) {
		FREE_PACKET(RQ, NULL, SPL_SK);
	} else {
		RQ->flags &= PKT_REUSE_FLAGS;
		SKY2_ADD_RCV_PACKET(mac, RQ);
		SKY2_KICK_RECEIVE(mac);
	}
	RETURN;
}

/* gcc produces shorter code if __finline__ is not used */
static int SKY2_RECEIVE(MAC *mac, Y2_DESC *y)
{
	PACKET *p;
#ifdef SK_ENABLE_RX_CHECKSUM
	SK *sk;
	unsigned chs;
#endif
	unsigned str;
	__u32 stat;
	int len;
	next_rcv:
	if (__unlikely(!y->stpa.len)) {
		SK_BAD_STATUS(mac, y->stpa.rxstat, y->stpa.len);
		return 0;
	}
	if (__unlikely(!mac->n_recv)) {
		SKY2_RECEIVE_NO_PACKETS(mac);
		return 1;
	}
	mac->n_recv--;
	str = mac->first_recv;
	mac->first_recv = (str + 1) & (N_DESCRIPTORS - 1);
	p = mac->recv_packets[str];
	mac->recv_packets[str] = NULL;
	mac->u.y2_recv[str].rxpa.opcode = 0;
	if (__unlikely(!p)) goto next_rcv;
	SKY2_UNMAP_RCV_PACKET(mac, p);
	PKT_PREFETCH_1(p);
	p->fn = SKY2_PACKET_RETURNED;
	p->sender_data = mac;
	p->h = mac->packet_input;
	p->id = (IDTYPE_PCI | (mac->sk->id & IDTYPE_MASK)) ^ mac->mac_2;
	mac->outstanding++;
	stat = y->stpa.rxstat;
	len = y->stpa.len;
	if (__unlikely(GMAC_BAD_STATUS(stat, len))) {
		if (__unlikely(stat == -1)) goto drop;
		SK_BAD_STATUS(mac, stat, len);
		goto drop;
	}
	p->data_length = len - sizeof(struct eth_header);
	if (__unlikely(p->data_length > mac->mtu)) {
		drop:
		p->status = -EIO;
		CALL_PKT(p);
		if (__unlikely(y->stpa.len > mac->mtu + sizeof(struct eth_header))) {
			y->stpa.len -= mac->mtu + sizeof(struct eth_header);
			y->stpa.rxstat = -1;
			goto next_rcv;
		}
		return 0;
	}
#ifdef SK_ENABLE_RX_CHECKSUM
	if (__likely((chs = (sk = mac->sk)->stat_sum_1))) {
		p->checksum.u = MKCHECKSUM(SK_TCP_SUM_START_1, chs);
		p->flags |= PKT_INPUT_CHECKSUM;
	}
	sk->stat_sum_1 = sk->stat_sum_2 = 0;
#endif
	PKT_PREFETCH_2(p);
	CALL_IORQ(p, KERNEL$PKTIO);
	return 0;
}

static void SKY2_REFILL_RCVQ(MAC *mac)
{
	do {
		PACKET *p;
		ALLOC_PACKET(p, mac->mtu, NULL, SPL_SK, break;);
		SKY2_ADD_RCV_PACKET(mac, p);
	} while (!SK_RCVQ_FULL(mac));
	SKY2_KICK_RECEIVE(mac);
}

static void SKY2_TX_COMPLETION(TX_QUEUE *q, unsigned done)
{
	unsigned idx;
	MAC *mac;
	if (__unlikely(done >= N_DESCRIPTORS)) {
		SKY2_TX_INVALID_DONE_INDEX(q, done);
		return;
	}
	idx = q->first_sent;
	done = (done - idx) & (N_DESCRIPTORS - 1);
	if (__unlikely(done > q->n_sent)) {
		SKY2_TX_TOO_MUCH_DONE(q, done);
		return;
	}
	q->n_sent -= done;
	do {
		q->u.y2_xmit[idx].txpa.opcode = 0;
		SKY2_UNMAP_TX_DESC(q, idx);
		if (__likely(q->xmit_info[idx].p != NULL)) CALL_PKT(q->xmit_info[idx].p);
		idx = (idx + 1) & (N_DESCRIPTORS - 1);
		done--;
	} while (done);
	q->first_sent = idx;
	mac = q->mac;
	if (__likely(mac->queues - q->queue_2 == 2)) while (__likely(SKY2_CAN_SEND(q))) {
		PACKET *p = NETQUE$DEQUEUE(mac->queue);
		if (__unlikely(!p)) break;
		if (__unlikely(!q->n_sent)) {
			if (__likely(q->max_sent < SKY2_MAX_SEND)) q->max_sent++;
		}
		SKY2_SEND(q, p);
	}
	KERNEL$DEL_TIMER(&q->timer);
	KERNEL$SET_TIMER(SKY2_TX_TIMEOUT, &q->timer);
}

#ifdef SK_HIGH_INTERRUPT

DECL_RT_IRQ_HANDLER(SKY2_IRQ_RT)
{
	SK *sk = DATA;
	__u32 isr = SK_READ_32(sk, SK_Y2_SISR2) & sk->imr;
	if (__unlikely(sk->isr)) {
		IRQ_RETURN;
	}
	if (__unlikely(!isr)) {
		SK_WRITE_32(sk, SK_Y2_ICR, SK_Y2_ICR_ENABLE);
		IRQ_RETURN;
	}
	sk->isr = isr;
	IRQ_POST_AST(&sk->irq_ast);
}

#endif

DECL_AST(SKY2_IRQ, SPL_SK, AST)
{
	SK *sk = GET_STRUCT(RQ, SK, irq_ast);
	__u8 opc;
	__u32 isr;
	Y2_DESC *y;
	__u32 lotxdone, hitxdone;
#ifndef SK_HIGH_INTERRUPT
	isr = SK_READ_32(sk, SK_Y2_SISR2) & sk->imr;
#else
	isr = sk->isr;
#endif
	if (__unlikely(!isr)) {
		SK_WRITE_32(sk, SK_Y2_ICR, SK_Y2_ICR_ENABLE);
		RETURN;
	}
	if (__unlikely(isr & ~SK_Y2_IRQ_STAT_BMU)) {
		if (SKY2_SPECIAL_IRQ(sk, isr)) goto ret;
	}
	SK_WRITE_32(sk, SK_STAT_CTRL, SK_STAT_CTRL_CLR_IRQ);
	__barrier();
	lotxdone = 0 /* against warning */, hitxdone = 0xffffffff;
	if (__likely((opc = (y = &sk->y2_status[sk->first_stat])->stpa.opcode) & Y2_OP_OWNER)) {
		do {
			__read_barrier();
			sk->first_stat = (sk->first_stat + 1) & (N_STATUS_DESCRIPTORS - 1);
			if (__likely(opc == (Y2_OP_RXSTAT | Y2_OP_OWNER))) {
				MAC *mac;
				if (__unlikely(y->stpa.port >= sk->n_macs)) {
					SKY2_RECEIVE_INVALID_PORT(sk, y);
					goto ret;
				}
				mac = (MAC *)((char *)&sk->macs + (-y->stpa.port & sizeof(MAC)));
				if (__unlikely(SKY2_RECEIVE(mac, y)))
					goto ret;
			} else if (__likely(opc == (Y2_OP_RXCHKS | Y2_OP_OWNER))) {
				chks:
				sk->stat_sum_1 = y->stch.tcp_sum_1;
				sk->stat_sum_2 = y->stch.tcp_sum_2;
			} else if (__likely(opc == (Y2_OP_TXINDEXLE | Y2_OP_OWNER))) {
				lotxdone = y->sttx.lostat;
				hitxdone = y->sttx.histat;
			} else if (__unlikely(opc == (Y2_OP_RXCHKSVLAN | Y2_OP_OWNER))) {
				goto chks;
			}
			/* else if (__likely(opc == (Y2_OP_RSS_HASH | Y2_OP_OWNER))) {
			} else if (__likely(opc == (Y2_OP_RXTIMESTAMP | Y2_OP_OWNER))) {
			} else if (__likely(opc == (Y2_OP_RXTIMEVLAN | Y2_OP_OWNER))) {
			} else if (__likely(opc == (Y2_OP_RXVLAN | Y2_OP_OWNER))) {
			} */
			__rw_barrier();
			y->stpa.opcode = 0;
		} while ((opc = (y = &sk->y2_status[sk->first_stat])->stpa.opcode) & Y2_OP_OWNER);
		if (__likely(!SK_RCVQ_FULL(&sk->macs[0]))) SKY2_REFILL_RCVQ(&sk->macs[0]);
		if (__unlikely(sk->n_macs == 2) && !SK_RCVQ_FULL(&sk->macs[1])) SKY2_REFILL_RCVQ(&sk->macs[1]);
		if (__unlikely(Y2_BUG_43_418(sk))) {
			SK_WRITE_8(sk, SK_STAT_TX_TIMER_CTRL, SK_STAT_TX_TIMER_CTRL_STOP);
			SK_WRITE_8(sk, SK_STAT_TX_TIMER_CTRL, SK_STAT_TX_TIMER_CTRL_START);
		}
	}
	if (!(hitxdone & 0x80000000)) {
		unsigned done = lotxdone & 0xfff;
		if (__likely(done != sk->macs[0].xmit_queue[1].first_sent))
			SKY2_TX_COMPLETION(&sk->macs[0].xmit_queue[1], done);
		if (sk->macs[0].queues == 2) {
			done = (lotxdone >> 12) & 0xfff;
			if (done != sk->macs[0].xmit_queue[0].first_sent)
				SKY2_TX_COMPLETION(&sk->macs[0].xmit_queue[0], done);
		}
		if (sk->n_macs == 2) {
			done = (lotxdone >> 24) | (hitxdone & 0xf);
			if (done != sk->macs[1].xmit_queue[1].first_sent)
				SKY2_TX_COMPLETION(&sk->macs[1].xmit_queue[1], done);
			if (sk->macs[1].queues == 2) {
				done = hitxdone >> 4;
				if (done != sk->macs[1].xmit_queue[0].first_sent)
					SKY2_TX_COMPLETION(&sk->macs[1].xmit_queue[0], done);
			}
		}
	}
	if (__unlikely(Y2_BUG_42(sk))) {
		SKY2_SET_PUT_IDX((SK0 *)sk, SK_Y2_RX_PUT_IDX, sk->macs[0].first_recv + sk->macs[0].n_recv);
		SKY2_SET_PUT_IDX((SK0 *)sk, SK_Y2_TX_PUT_IDX + ADD_ASYNC_SK_TX, sk->macs[0].xmit_queue[1].first_sent + sk->macs[0].xmit_queue[1].n_sent);
		if (__unlikely(sk->macs[0].queues == 2))
			SKY2_SET_PUT_IDX((SK0 *)sk, SK_Y2_TX_PUT_IDX, sk->macs[0].xmit_queue[0].first_sent + sk->macs[0].xmit_queue[0].n_sent);
		if (__unlikely(sk->n_macs == 2)) {
			SKY2_SET_PUT_IDX((SK0 *)sk, SK_Y2_RX_PUT_IDX + ADD_MAC2_SK_TX, sk->macs[1].first_recv + sk->macs[1].n_recv);
			SKY2_SET_PUT_IDX((SK0 *)sk, SK_Y2_TX_PUT_IDX + ADD_ASYNC_SK_TX + ADD_MAC2_SK_TX, sk->macs[1].xmit_queue[1].first_sent + sk->macs[1].xmit_queue[1].n_sent);
			if (__unlikely(sk->macs[1].queues == 2))
				SKY2_SET_PUT_IDX((SK0 *)sk, SK_Y2_TX_PUT_IDX + ADD_MAC2_SK_TX, sk->macs[1].xmit_queue[0].first_sent + sk->macs[1].xmit_queue[0].n_sent);
		}
	}
#ifdef SK_HIGH_INTERRUPT
	sk->isr = 0;
	__barrier();
#endif
	SK_WRITE_32(sk, SK_Y2_ICR, SK_Y2_ICR_ENABLE);
	ret:
#ifndef SK_HIGH_INTERRUPT
	sk->irq_ctrl.eoi();
#endif
	RETURN;
}

DECL_IOCALL(SKY2_PACKET, SPL_SK, PACKET)
{
	HANDLE *h = RQ->handle;
	MAC *mac;
	__CHECK_PKT(RQ, "SKY2_PACKET");
	if (__unlikely(h->op != &SKY2_OPERATIONS)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PKTIO);
	SWITCH_PROC_ACCOUNT_KERNEL_OPTIMIZE(h->name_addrspace, SPL_X(SPL_SK));
	mac = h->fnode;
	if (__unlikely(RQ->data_length + RQ->v.len > mac->mtu)) {
		RQ->status = -EMSGSIZE;
		RETURN_PKT(RQ);
	}
	RQ->status = 0;
	if (mac->queues == 1) {
		if (__likely(SKY2_CAN_SEND(&mac->xmit_queue[1]))) SKY2_SEND(&mac->xmit_queue[1], RQ);
		else NETQUE$ENQUEUE_PACKET(mac->queue, RQ);
	} else if (0) {		/* !!! TODO: classify packet */
		if (__likely(SKY2_CAN_SEND(&mac->xmit_queue[1]))) SKY2_SEND(&mac->xmit_queue[1], RQ);
		else class_slow: if (SKY2_CAN_SEND(&mac->xmit_queue[0])) SKY2_SEND(&mac->xmit_queue[0], RQ);
		else NETQUE$ENQUEUE_PACKET(mac->queue, RQ);
	} else goto class_slow;
	RETURN;
}

void SKY2_TIMER_FN(TIMER *t)
{
	TX_QUEUE *q = GET_STRUCT(t, TX_QUEUE, timer);
	LOWER_SPL(SPL_SK);
	VOID_LIST_ENTRY(&q->timer.list);
	if (__likely(!q->n_sent)) return;
	SK_TX_TIMEOUT(q);
}

void SKY2_START_XMIT(MAC *mac)
{
	TX_QUEUE *q = &mac->xmit_queue[2 - mac->queues];
	while (__likely(SKY2_CAN_SEND(q))) {
		PACKET *p = NETQUE$DEQUEUE(mac->queue);
		if (__unlikely(!p)) break;
		SKY2_SEND(q, p);
	}
}

static void SKY2_UNMAP_TX_DESC_NOINLINE(TX_QUEUE *q, unsigned idx)
{
	SKY2_UNMAP_TX_DESC(q, idx);
}

static void SKY2_TEAR_DOWN_SENT_PACKETS_ON_QUEUE(TX_QUEUE *q)
{
	unsigned j;
	for (j = 0; j < q->n_sent; j++) {
		unsigned idx = (q->first_sent + j) & (N_DESCRIPTORS - 1);
		SKY2_UNMAP_TX_DESC_NOINLINE(q, idx);
		if (__likely(q->xmit_info[idx].p != NULL)) {
			q->xmit_info[idx].p->status = -EAGAIN;
			CALL_PKT(q->xmit_info[idx].p);
		}
	}
	q->first_sent = 0;
	q->n_sent = 0;
}

void SKY2_TEAR_DOWN_SENT_PACKETS(MAC *mac)
{
	unsigned i;
	for (i = 0; i < 2; i++)
		SKY2_TEAR_DOWN_SENT_PACKETS_ON_QUEUE(&mac->xmit_queue[i]);
}

void SKY2_TEAR_DOWN_RECV_PACKETS(MAC *mac, int free)
{
	int j;
	for (j = 0; j < mac->n_recv; j++) {
		PACKET *p;
		if (__likely((p = mac->recv_packets[(mac->first_recv + j) & (N_DESCRIPTORS - 1)]) != NULL)) {
			SKY2_UNMAP_RCV_PACKET(mac, p);
			if (free) FREE_PACKET(p, NULL, SPL_SK);
		}
	}
}

static void SKY2_RECEIVE_INVALID_PORT(SK *sk, Y2_DESC *y)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, sk->dev_name, "RECEIVE FOR INVALID PORT %d", y->stpa.port);
	SK_BASIC_RESET(sk);
	SK_CONTINUE_RESET(sk);
}

static void SKY2_RECEIVE_NO_PACKETS(MAC *mac)
{
	SK *sk = mac->sk;
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, sk->dev_name, "RECEIVE STATUS WHILE NO PACKETS IN RECEIVE QUEUE, PORT %c", PORT_ID(mac));
	SK_BASIC_RESET(sk);
	SK_CONTINUE_RESET(sk);
}

static void SKY2_TX_INVALID_DONE_INDEX(TX_QUEUE *q, unsigned done)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, q->mac->sk->dev_name, "INVALID TX INDEX %d RETURNED, PORT %c, %s QUEUE", done, PORT_ID(q->mac), QUEUE_ID(q));
}

static void SKY2_TX_TOO_MUCH_DONE(TX_QUEUE *q, unsigned n_done)
{
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, q->mac->sk->dev_name, "MORE TX PACKETS RETURNED THAN POSTED, %d > %d, PORT %c, %s QUEUE", n_done, q->n_sent, PORT_ID(q->mac), QUEUE_ID(q));
	SKY2_TEAR_DOWN_SENT_PACKETS_ON_QUEUE(q);
	q->first_sent = (q->first_sent + n_done) & (N_DESCRIPTORS - 1);
}

static int SKY2_SPECIAL_IRQ(SK *sk, __u32 isr)
{
	/*if (isr & ~(SK_Y2_IRQ_HW_ERROR | SK_Y2_IRQ_STAT_BMU)) __debug_printf("special irq: %08x\n", isr);*/
	if (isr & SK_Y2_IRQ_HW_ERROR) {
		int r = 0;
		__u32 hwisr = SK_READ_32(sk, SK_IHESR);
		/*if (hwisr & ~SK_Y2_HWIRQ_TS_TIMER_OVERFLOW) __debug_printf("hw irq: %08x\n", hwisr);*/
		if (hwisr & SK_Y2_HWIRQ_TS_TIMER_OVERFLOW) {
			SK_WRITE_8(sk, SK_GMAC_TS_CTRL, SK_GMAC_TS_CTRL_CLEAR_IRQ);
			r = 1;
		}
		if (hwisr & SK_Y2_HWIRQ_SENSOR) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "SENSOR IRQ"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_MASTER_ERROR) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PCI MASTER ERROR"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_STATUS_EXCEPTION) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "STATUS EXCEPTION"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PCI_EXPRESS) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PCI EXPRESS INTERRUPT"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PCI_EXPRESS_ERROR) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PCI EXPRESS ERROR"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_RD_RAM_2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR READING FROM RAM FOR PORT B"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_WR_RAM_2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR WRITING TO RAM FOR PORT B"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_MAC_2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR ON MAC B"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_RX_2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR IN RECEIVE QUEUE ON PORT B"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_TCP_S2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "TCP SEGMENTATION ERROR IN SYNC QUEUE ON PORT B"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_TCP_A2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "TCP SEGMENTATION ERROR IN ASYNC QUEUE ON PORT B"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_RD_RAM_1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR READING FROM RAM FOR PORT A"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_WR_RAM_1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR WRITING TO RAM FOR PORT A"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_MAC_1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR ON MAC A"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_PAR_RX_1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "PARITY ERROR IN RECEIVE QUEUE ON PORT A"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_TCP_S1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "TCP SEGMENTATION ERROR IN SYNC QUEUE ON PORT A"), r = 2;
		if (hwisr & SK_Y2_HWIRQ_TCP_A1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "TCP SEGMENTATION ERROR IN ASYNC QUEUE ON PORT A"), r = 2;
		if (__likely(r == 1)) goto brk;
		if (__unlikely(!r)) KERNEL$SYSLOG(__SYSLOG_HW_BUG, sk->dev_name, "UNKNOWN HARDWARE ERROR %08X", hwisr);
		SK_BASIC_RESET(sk);
		SK_CONTINUE_RESET(sk);
		return 1;
	}
	brk:
	if (isr & SK_Y2_IRQ_POLL_CHECK) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "POLL UNIT CHECK INTERRUPT");
	if (isr & SK_Y2_IRQ_RX2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "INVALID DESCRIPTOR FOR RX QUEUE ON PORT B");
	if (isr & SK_Y2_IRQ_TXS2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "INVALID DESCRIPTOR FOR SYNC TX QUEUE ON PORT B");
	if (isr & SK_Y2_IRQ_TXA2) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "INVALID DESCRIPTOR FOR ASYNC TX QUEUE ON PORT B");
	if (isr & SK_Y2_IRQ_RX1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "INVALID DESCRIPTOR FOR RX QUEUE ON PORT A");
	if (isr & SK_Y2_IRQ_TXS1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "INVALID DESCRIPTOR FOR SYNC TX QUEUE ON PORT A");
	if (isr & SK_Y2_IRQ_TXA1) KERNEL$SYSLOG(__SYSLOG_HW_ERROR, sk->dev_name, "INVALID DESCRIPTOR FOR ASYNC TX QUEUE ON PORT A");
	if (isr & (SK_Y2_IRQ_POLL_CHECK | SK_Y2_IRQ_RX2 | SK_Y2_IRQ_TXS2 | SK_Y2_IRQ_TXA2 | SK_Y2_IRQ_RX1 | SK_Y2_IRQ_TXS1 | SK_Y2_IRQ_TXA1)) {
		SK_BASIC_RESET(sk);
		SK_CONTINUE_RESET(sk);
		return 1;
	}
	if (isr & SK_Y2_IRQ_ASF) {
		__u32 a = SK_READ_8(sk, SK_ASF_STAT_CMD);
		a |= SK_ASF_STAT_CMD_CLR_HSTI;
		SK_WRITE_8(sk, SK_ASF_STAT_CMD, a);
	}
	if (isr & SK_Y2_IRQ_I2C_READY) {
		SK_WRITE_32(sk, SK_I2C_HW_IRQ, SK_I2C_HW_IRQ_CLEAR);
	}
	if (isr & SK_Y2_IRQ_SW) {
		SK_WRITE_8(sk, SK_CS, SK_CS_CLEAR_IRQ_SW);
	}
	if (isr & SK_Y2_IRQ_TIMER) {
		int i, j;
		/*SK_WRITE_8(sk, SK_TIMER_CONTROL, SK_TIMER_CONTROL_STOP);*/
		SK_WRITE_8(sk, SK_TIMER_CONTROL, SK_TIMER_CONTROL_CLEAR_IRQ);
		SK_TIMER_FN(sk);
		for (i = 0; i < sk->n_macs; i++) for (j = 2 - sk->macs[i].queues; j < 2; j++) SKY2_PROBE_QUEUE(&sk->macs[i].xmit_queue[j]);
	}
	if (isr & SK_Y2_IRQ_PHY1) {
		SK_GPHY_IRQ(&sk->macs[0]);
	}
	if (isr & SK_Y2_IRQ_MAC1) {
		SK_MAC_IRQ(&sk->macs[0]);
	}
	if (isr & SK_Y2_IRQ_PHY2) {
		if (sk->n_macs == 2) SK_GPHY_IRQ(&sk->macs[1]);
		else {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, sk->dev_name, "INTERRUPT FROM NON-PRESENT PHY 2");
			SK_WRITE_32(sk, SK_IMR, SK_READ_32(sk, SK_IMR) & ~SK_Y2_IRQ_PHY2);
			sk->imr &= ~SK_Y2_IRQ_PHY2;
		}
	}
	if (isr & SK_Y2_IRQ_MAC2) {
		if (sk->n_macs == 2) SK_MAC_IRQ(&sk->macs[1]);
		else {
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, sk->dev_name, "INTERRUPT FROM NON-PRESENT MAC 2");
			SK_WRITE_32(sk, SK_IMR, SK_READ_32(sk, SK_IMR) & ~SK_Y2_IRQ_MAC2);
			sk->imr &= ~SK_Y2_IRQ_MAC2;
		}
	}
	return 0;
}

