#include <STDLIB.H>
#include <STRING.H>
#include <SPAD/PKT.H>
#include <ARCH/IPCHECKS.H>
#include <ARCH/TIME.H>
#include <ARCH/BT.H>
#include <KERNEL/TIMER.H>
#include <SPAD/ETHERNET.H>
#include <SPAD/SYSLOG.H>
#include <VALUES.H>
#include "IP.H"

#include <NETINET/DHCP.H>
#include "TCPIP.H"

#define DHCP_TTL		0x10
#define MAX_DHCP_RESPONSES	0x10

__u8 DHCP_APPARRAY[256][256];
DHCP *DHCP_APPOWNER[256];

/* Warning: adding entries may cause packet size overflow. check */
static __const__ __u8 parameter_request_list[] = {
	BOOTP_SUBNET_MASK,
	BOOTP_ROUTERS,
	BOOTP_BROADCAST_ADDRESS,
	BOOTP_DNS_SERVERS,
	BOOTP_HOST_NAME,
	BOOTP_DOMAIN_NAME,

	BOOTP_TIME_OFFSET,
	BOOTP_TIME_SERVERS,
	BOOTP_QUOTE_OF_THE_DAY_SERVERS,
	BOOTP_LPR_SERVERS,
	BOOTP_X_WINDOW_FONT_SERVERS,
	BOOTP_X_WINDOW_DISPLAY_MANAGERS,
	BOOTP_SMTP_SERVERS,
	BOOTP_POP3_SERVERS,
	BOOTP_NNTP_SERVERS,
	BOOTP_WWW_SERVERS,
	BOOTP_FINGER_SERVERS,
	BOOTP_IRC_SERVERS,

	BOOTP_IP_FORWARDING,
	BOOTP_MAXIMUM_REASSEMBLY_SIZE,
	BOOTP_MTU,
	BOOTP_STATIC_ROUTES,
	BOOTP_ARP_CACHE_TIMEOUT,
};

/* do not enlarge because the space in request would overflow */
#define dhcp_id	"SPAD"

struct __dhcp {
	IF_TYPE *if_type;
	LIST_ENTRY list;
	int handle;
	int flags;
	in_addr_t preferred_addr;
	__u32 xid;
	jiffies_t start_time;
	jiffies_t last_discovery_time;
	jiffies_t t1;
	jiffies_t t2;
	jiffies_t lease_end;
	u_jiffies_lo_t timeout;
	int retries;
	void (*packet)(DHCP *, PACKET *);
	TIMER timer;
	__u8 chaddr[16];

	in_addr_t my_addr;
	in_addr_t si_addr;
	__u8 si_hwaddr[MAX_ADDR_LEN];

	int n_responses;
	PACKET *responses[MAX_DHCP_RESPONSES];
	int n_blacklist;
	in_addr_t blacklist[MAX_DHCP_RESPONSES];
};

__const__ unsigned DHCP_STRUCT_SIZE = sizeof(DHCP);

#define FLAGS_HAVE_PREFERRED_ADDR	0x0001
#define FLAGS_ORIGINALLY_ONLY_BOOTP	0x0002
#define FLAGS_ONLY_BOOTP		0x0004
#define FLAGS_SECURE			0x0008
#define FLAGS_DISTRUST_LOCAL		0x0010
#define FLAGS_DISTRUST_REMOTE		0x0020
#define FLAGS_STATE_LONGWAIT		0x0040
#define FLAGS_STATE_SHORTWAIT		0x0080
#define FLAGS_LINK_CHANGED		0x0100

static DECL_XLIST(dhcp_interfaces);

static void DHCP_SEND_DISCOVERY(DHCP *dhcp);
static void DHCP_RETRANSMIT_DISCOVERY(TIMER *t);
static void DHCP_DISCOVERY_REPLY(DHCP *dhcp, PACKET *p);
static int DHCP_CHECK_SECURE(DHCP *dhcp, PACKET *p1, PACKET *p2);
static void DHCP_OFFERS_COMPLETE(TIMER *t);
static void DHCP_WAIT_RESTART(DHCP *dhcp, int longwait);
static void DHCP_FINISHED_WAIT_FOR_RESTART(TIMER *t);
static void DHCP_SEND_REQUEST(DHCP *dhcp);
static int DHCP_CONFIGURE_ACCORDING_TO_OPTIONS(DHCP *dhcp);
static void DHCP_MORE_PACKET_OPTIONS(DHCP *dhcp);
static void DHCP_RETRANSMIT_REQUEST(TIMER *t);
static void DHCP_ACK_NAK_REPLY(DHCP *dhcp, PACKET *p);
static int DHCP_GET_LEASE(DHCP *dhcp);
static void DHCP_T12_EXPIRED(TIMER *t);
static void DHCP_BOUND_REPLY(DHCP *dhcp, PACKET *packet);

static jiffies_t DHCP_TIME_FUZZ(jiffies_t tm)
{
	if (__likely(tm > DHCP_TIMEOUT_FUZZ))
		return tm - DHCP_TIMEOUT_FUZZ + RANDOM_DHCP_XID(NULL) % (2 * DHCP_TIMEOUT_FUZZ + 1);
	return 1 + RANDOM_DHCP_XID(NULL) % ((jiffies_lo_t)tm + DHCP_TIMEOUT_FUZZ);
}

#define init_opt(p)	(o = dhcp(p)->options + 4)
#define opt_byte(b)	(*o++ = (b))
#define opt_word(w)	(opt_byte((w) >> 8), opt_byte(w))
#define opt_dword(w)	(opt_word((w) >> 16), opt_word(w)
#define opt_bytes(b, l)	(o = mempcpy(o, b, l))
#define opt_iaddr(w)	(opt_bytes(&w, 4))

static void DHCP_COMMON_OPTS(DHCP *dhcp, __u8 *o, __u8 *oe)
{
	if (__likely(!(dhcp->flags & FLAGS_ONLY_BOOTP))) {
		opt_byte(DHCP_PARAMETER_REQUEST_LIST),	
		opt_byte(sizeof(parameter_request_list));
		opt_bytes(parameter_request_list, sizeof(parameter_request_list));
		opt_byte(DHCP_MAXIMUM_SIZE);
		opt_byte(2);
		opt_word(__unlikely(dhcp->if_type->mtu > ETHERNET_MTU) ? ETHERNET_MTU : dhcp->if_type->mtu);
		opt_byte(DHCP_VENDOR_CLASS_ID);
		opt_byte(strlen(dhcp_id));
		opt_bytes(dhcp_id, strlen(dhcp_id));
		/* this would cause packet size overflow
		opt_byte(DHCP_CLIENT_ID);
		opt_byte(dhcp->if_type->addrlen + 1);
		opt_byte(ntohs(dhcp->if_type->arptype));
		opt_bytes(dhcp->if_type->addr, dhcp->if_type->addrlen);*/
	}
	opt_byte(BOOTP_END);
	if (__unlikely(o > oe))
		KERNEL$SUICIDE("DHCP_COMMON_OPTS: OVERFLOW BY %ld BYTES", (long)(o - oe));
}

static __u8 *options[256];
static __u8 *options_2[256];

static void SCAN_OPTIONS(DHCP *dhcp, __u8 *start, __u8 *end)
{
	while (start < end) {
		__u8 c = *start++;
		if (__unlikely(c == BOOTP_PAD)) continue;
		if (__unlikely(c == BOOTP_END)) return;
		if (__unlikely(start + *start + 1 > end)) {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_NONFATAL_ERROR, net_name, "DHCP(%s): OPTION CROSSES OPTION AREA END", KERNEL$HANDLE_PATH(dhcp->handle));
			return;
		}
		if (__likely(!options[c])) {
			options[c] = start;
		} else {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_NONFATAL_ERROR, net_name, "DHCP(%s): OPTION %02X SPECIFIED MORE TIMES", KERNEL$HANDLE_PATH(dhcp->handle), c);
		}
		start += *start + 1;
	}
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_NONFATAL_ERROR, net_name, "DHCP(%s): UNTERMINATED OPTIONS", KERNEL$HANDLE_PATH(dhcp->handle));
}

static void DHCP_SCAN_OPTIONS(DHCP *dhcp, PACKET *p)
{
	memset(options, 0, sizeof(options));
	if (__unlikely(*(__u32 *)dhcp(p)->options != htonl(BOOTP_OPTIONS_MAGIC))) return;
	SCAN_OPTIONS(dhcp, dhcp(p)->options + 4, (__u8 *)ip(p) + p->data_length);
	if (__unlikely(options[DHCP_OPTION_OVERLOAD] != NULL)) {
		if (__unlikely(options[DHCP_OPTION_OVERLOAD][0] != 1)) {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_NONFATAL_ERROR, net_name, "DHCP(%s): OVERLOAD OPTION HAS LENGTH %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[DHCP_OPTION_OVERLOAD][0]);
		} else switch (options[DHCP_OPTION_OVERLOAD][1]) {
			case DHCP_OPTION_OVERLOAD_FILE:
				SCAN_OPTIONS(dhcp, dhcp(p)->file, dhcp(p)->file + sizeof(dhcp(p)->file));
				break;
			case DHCP_OPTION_OVERLOAD_FILE_SNAME:
				SCAN_OPTIONS(dhcp, dhcp(p)->file, dhcp(p)->file + sizeof(dhcp(p)->file));
				/* fall through */
			case DHCP_OPTION_OVERLOAD_SNAME:
				SCAN_OPTIONS(dhcp, dhcp(p)->sname, dhcp(p)->sname + sizeof(dhcp(p)->sname));
				break;
			default:
				if (errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_NONFATAL_ERROR, net_name, "DHCP(%s): OVERLOAD OPTION HAS VALUE %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[DHCP_OPTION_OVERLOAD][1]);
		}
	}
}

static void DHCP_FILL_PACKET(DHCP *dhcp, PACKET *p)
{
	p->addrlen = dhcp->if_type->addrlen;
	memcpy(PKT_SRCADDR(p, p->addrlen), dhcp->if_type->addr, p->addrlen);
	SET_BROADCAST_DST(p);
	PKT_PROTOCOL(p) = NET_PROTOCOL_IP;
	dhcp(p)->ip_vhl = IP_VHL;
	dhcp(p)->ip_tos = IPTOS_RELIABILITY;
	dhcp(p)->ip_len = htons(sizeof(struct dhcp));
	dhcp(p)->ip_id = 0;
	dhcp(p)->ip_off = htons(IP_DF);
	dhcp(p)->ip_ttl = DHCP_TTL;
	dhcp(p)->ip_p = IPPROTO_UDP;
	dhcp(p)->ip_sum = 0;
	dhcp(p)->ip_src.s_addr = htonl(INADDR_ANY);
	dhcp(p)->ip_dst.s_addr = htonl(INADDR_BROADCAST);
	dhcp(p)->udp_sport = htons(BOOTP_CLIENT_PORT);
	dhcp(p)->udp_dport = htons(BOOTP_SERVER_PORT);
	dhcp(p)->udp_len = htons(sizeof(struct dhcp) - sizeof(struct ip));
	dhcp(p)->udp_checksum = htons(0);
	dhcp(p)->op = OP_BOOTREQUEST;
	dhcp(p)->hwtype = ntohs(dhcp->if_type->arptype);
	dhcp(p)->hlen = dhcp->if_type->addrlen;
	dhcp(p)->hops = 0;
	dhcp(p)->xid = dhcp->xid;
	dhcp(p)->secs = htons((KERNEL$GET_JIFFIES_LO() - (u_jiffies_lo_t)dhcp->start_time) / JIFFIES_PER_SECOND);
	dhcp(p)->flags = htons(/*DHCP_FLAG_BROADCAST*/	0);
	dhcp(p)->ciaddr = htonl(0);
	dhcp(p)->yiaddr = htonl(0);
	dhcp(p)->siaddr = htonl(0);
	dhcp(p)->giaddr = htonl(0);
	memcpy(dhcp(p)->chaddr, dhcp->chaddr, sizeof dhcp(p)->chaddr);
	memset(dhcp(p)->sname, 0, sizeof dhcp(p)->sname);
	memset(dhcp(p)->file, 0, sizeof dhcp(p)->file);
	*(__u32 *)dhcp(p)->options = htonl(BOOTP_OPTIONS_MAGIC);
	memset(dhcp(p)->options + 4, 0, sizeof dhcp(p)->options - 4);
}

static void DHCP_FREE_PACKETS(DHCP *dhcp)
{
	unsigned i;
	for (i = 0; i < dhcp->n_responses; i++) {
		PACKET *p = dhcp->responses[i];
		p->status = 0;
		CALL_PKT(p);
	}
	dhcp->n_responses = 0;
}

static void DHCP_BLACKLIST_SERVER(DHCP *dhcp, in_addr_t addr)
{
	if (__likely(dhcp->n_blacklist < MAX_DHCP_RESPONSES)) {
		dhcp->blacklist[dhcp->n_blacklist++] = addr;
		return;
	}
	memmove(dhcp->blacklist, dhcp->blacklist + 1, sizeof(in_addr_t) * (MAX_DHCP_RESPONSES - 1));
	dhcp->blacklist[MAX_DHCP_RESPONSES - 1] = addr;
	return;
}

static void DHCP_RESTART(DHCP *dhcp)
{
	/*IF_DHCP_UNSET(dhcp);*/
	DHCP_FREE_PACKETS(dhcp);
	if (__likely(!(dhcp->flags & FLAGS_ORIGINALLY_ONLY_BOOTP))) dhcp->flags &= ~FLAGS_ONLY_BOOTP;
	dhcp->flags &= ~FLAGS_LINK_CHANGED;
	dhcp->start_time = KERNEL$GET_JIFFIES();
	dhcp->timeout = DHCP_FIRST_TIMEOUT;
	dhcp->retries = 0;
	DHCP_SEND_DISCOVERY(dhcp);
}

static void DHCP_SEND_DISCOVERY(DHCP *dhcp)
{
	PACKET *p;
	__u8 *o;
	dhcp->flags &= ~(FLAGS_STATE_LONGWAIT | FLAGS_STATE_SHORTWAIT);
	dhcp->xid = htonl(RANDOM_DHCP_XID((__u32 *)dhcp->chaddr));
	dhcp->last_discovery_time = KERNEL$GET_JIFFIES();
	retry_alloc:
	ALLOC_PACKET(p, sizeof(struct dhcp), &NET$PKTPOOL, SPL_NET, if (!NET$OOM()) goto retry_alloc; goto set_time;);
	p->data_length = sizeof(struct dhcp);

	DHCP_FILL_PACKET(dhcp, p);

	init_opt(p);
	if (__likely(!(dhcp->flags & FLAGS_ONLY_BOOTP))) {
		opt_byte(DHCP_MESSAGE_TYPE);
		opt_byte(1);
		opt_byte(DHCP_MESSAGE_TYPE_DISCOVER);
		if (dhcp->flags & FLAGS_HAVE_PREFERRED_ADDR) {
			opt_byte(DHCP_REQUESTED_IP_ADDRESS);
			opt_byte(4);
			opt_iaddr(dhcp->preferred_addr);
		}
	}
	DHCP_COMMON_OPTS(dhcp, o, dhcp(p)->options + sizeof dhcp(p)->options);

	IF_SEND_DHCP_PACKET(dhcp, p);
	
	set_time:
	KERNEL$DEL_TIMER(&dhcp->timer);
	dhcp->packet = DHCP_DISCOVERY_REPLY;
	dhcp->timer.fn = DHCP_RETRANSMIT_DISCOVERY;
	KERNEL$SET_TIMER(DHCP_TIME_FUZZ(dhcp->timeout), &dhcp->timer);
}

static void DHCP_RETRANSMIT_DISCOVERY(TIMER *t)
{
	DHCP *dhcp = GET_STRUCT(t, DHCP, timer);
	LOWER_SPL(SPL_NET);
	VOID_LIST_ENTRY(&dhcp->timer.list);
	dhcp->timeout = dhcp->timeout DHCP_TIMEOUT_BACKOFF;
	if (__likely(dhcp->timeout > DHCP_MAX_TIMEOUT)) dhcp->timeout = DHCP_MAX_TIMEOUT;
	dhcp->retries++;
	DHCP_SEND_DISCOVERY(dhcp);
}

static void DHCP_DISCOVERY_REPLY(DHCP *dhcp, PACKET *p)
{
	DHCP_SCAN_OPTIONS(dhcp, p);
	if (__likely(options[DHCP_MESSAGE_TYPE] != NULL)) {
		if (__unlikely(options[DHCP_MESSAGE_TYPE][0] != 1)) {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): MESSAGE TYPE OPTION HAS LENGTH %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[DHCP_MESSAGE_TYPE][0]);
			goto drop;
		}
		if (__unlikely(options[DHCP_MESSAGE_TYPE][1] != DHCP_MESSAGE_TYPE_OFFER)) goto drop;
	}
	/*{
		unsigned i;
		__debug_printf("got packet:\n");
		for (i = 0; i < 256; i++) if (options[i]) __debug_printf("reply: %02x - %.*s\n", i, options[i][0], options[i] + 1);
	}*/
	if (__likely(!dhcp->n_responses)) {
		KERNEL$DEL_TIMER(&dhcp->timer);
		dhcp->timer.fn = DHCP_OFFERS_COMPLETE;
		KERNEL$SET_TIMER(DHCP_WAIT_FOR_OFFERS, &dhcp->timer);
	}
	if (__likely(dhcp->n_responses < MAX_DHCP_RESPONSES)) {
		retry_dup:
		DUP_PACKET(p, &NET$PKTPOOL_LONG_TERM, if (!NET$OOM()) goto retry_dup; goto drop);
		dhcp->responses[dhcp->n_responses++] = p;
		if (dhcp->flags & FLAGS_SECURE) return;
	} else {
		if (dhcp->flags & FLAGS_SECURE) {
			int i;
			for (i = 0; i < MAX_DHCP_RESPONSES - 1; i++) {
				if (__unlikely(DHCP_CHECK_SECURE(dhcp, dhcp->responses[i], dhcp->responses[i + 1]))) {
					DHCP_WAIT_RESTART(dhcp, 1);
					goto drop;
				}
			}
			if (__unlikely(DHCP_CHECK_SECURE(dhcp, dhcp->responses[0], p))) {
				DHCP_WAIT_RESTART(dhcp, 1);
				goto drop;
			}
			goto drop;
		}
		p->status = 0;
		CALL_PKT(p);
	}
	if (__unlikely(dhcp->n_responses == MAX_DHCP_RESPONSES)) {
		KERNEL$DEL_TIMER(&dhcp->timer);
		DHCP_OFFERS_COMPLETE(&dhcp->timer);
	}
	return;
	drop:
	p->status = 0;
	CALL_PKT(p);
}

static DHCP *cmp_dhcp;

#define check_arg(o, cond)						\
do {									\
	if (options[o] && (options[o][0] cond) && !options_2[o]) return -1;\
	if (!options[o] && options_2[o] && (options_2[o][0] cond)) return 1;\
} while (0)

#define check_not_arg(o, cond)						\
do {									\
	if (options[o] && (options[o][0] cond) && !options_2[o]) return 1;\
	if (!options[o] && options_2[o] && (options_2[o][0] cond)) return -1;\
} while (0)

#define check_arg_val(o, cond, op)					\
do {									\
	check_arg(o, cond);						\
	if (options[o] && options_2[o]) {				\
		unsigned l1 = options[o][0];				\
		unsigned l2 = options_2[o][0];				\
		int r = memcmp(options[o] + 1, options_2[o] + 1, l1 < l2 ? l1 : l2);\
		if (r) {						\
			if (r op 0) return -1;				\
			else return 1;					\
		}							\
		if (l1 != l2) {						\
			if (l1 op l2) return -1;			\
			else return 1;					\
		}							\
	}								\
} while (0)

#define check_arg_len(o, cond, op)					\
do {									\
	check_arg(o, cond);						\
	if (options[o] && options_2[o]) {				\
		unsigned l1 = options[o][0];				\
		unsigned l2 = options_2[o][0];				\
		if (l1 != l2) {						\
			if (l1 op l2) return -1;			\
			else return 1;					\
		}							\
	}								\
} while (0)

static int get_dword(DHCP *dhcp, __u8 *opt[256], unsigned option, __u32 *a)
{
	__u8 *o;
	if (__unlikely(!opt[option])) return -ENOENT;
	o = opt[option];
	if (__unlikely(*o < 4)) {
		if (__unlikely(!*o)) return -ENOENT;
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): OPTION %02X IS TOO SMALL (%d BYTES)", KERNEL$HANDLE_PATH(dhcp->handle), option, *o);
		return -EPROTO;
	}
	*a = (o[1] << 24) | (o[2] << 16) | (o[3] << 8) | o[4];
	return 0;
}

static int get_iaddr(DHCP *dhcp, __u8 *opt[256], unsigned option, in_addr_t *a)
{
	__u32 dw;
	int r = get_dword(dhcp, opt, option, &dw);
	*a = htonl(dw);
	return r;
}

static in_addr_t get_sid(DHCP *dhcp, PACKET *p, __u8 *opt[256])
{
	in_addr_t addr;
	if (!get_iaddr(dhcp, opt, DHCP_SERVER_IDENTIFIER, &addr)) return addr;
	return dhcp(p)->siaddr;
}

static int DHCP_COMPARE_RESPONSES(__const__ void *v1, __const__ void *v2)
{
	PACKET *p1 = *(PACKET **)v1;
	PACKET *p2 = *(PACKET **)v2;
	in_addr_t id1, id2;
	int inb1, inb2, i;
	DHCP_SCAN_OPTIONS(cmp_dhcp, p2);
	memcpy(options_2, options, sizeof options);
	DHCP_SCAN_OPTIONS(cmp_dhcp, p1);
	if (__likely(!(cmp_dhcp->flags & FLAGS_ONLY_BOOTP))) {
		check_arg(DHCP_MESSAGE_TYPE, == 1);
	} else {
		check_not_arg(DHCP_MESSAGE_TYPE, == 1);
	}
	check_arg(BOOTP_ROUTERS, >= 4);
	check_arg(BOOTP_DNS_SERVERS, >= 4);
	check_arg(BOOTP_SUBNET_MASK, == 4);
	id1 = get_sid(cmp_dhcp, p1, options);
	id2 = get_sid(cmp_dhcp, p2, options_2);
	inb1 = inb2 = -1;
	for (i = 0; i < cmp_dhcp->n_blacklist; i++) {
		if (cmp_dhcp->blacklist[i] == id1) inb1 = i;
		if (cmp_dhcp->blacklist[i] == id2) inb2 = i;
	}
	if (__unlikely(inb1 != inb2)) {
		return inb1 - inb2;
	}
	if (cmp_dhcp->flags & FLAGS_HAVE_PREFERRED_ADDR) {
		if (dhcp(p1)->yiaddr == cmp_dhcp->preferred_addr && dhcp(p2)->yiaddr != cmp_dhcp->preferred_addr) return -1;
		if (dhcp(p1)->yiaddr != cmp_dhcp->preferred_addr && dhcp(p2)->yiaddr == cmp_dhcp->preferred_addr) return 1;
	}
	check_arg_val(DHCP_LEASE_TIME, == 4, >);
	check_arg(BOOTP_BROADCAST_ADDRESS, == 4);
	check_arg(BOOTP_DOMAIN_NAME, >= 1);
	check_arg(BOOTP_HOST_NAME, >= 1);
	check_arg_val(BOOTP_IP_FORWARDING, == 1, >);
	check_arg_len(BOOTP_STATIC_ROUTES, >= 8, >);
	check_arg_val(BOOTP_MTU, == 2, >);
	check_arg_val(BOOTP_MAXIMUM_REASSEMBLY_SIZE, == 2, >);
	check_arg_val(BOOTP_ARP_CACHE_TIMEOUT, == 4, >);
	check_arg_val(DHCP_SERVER_IDENTIFIER, == 4, <);
	if (ntohl(dhcp(p1)->siaddr) < ntohl(dhcp(p2)->siaddr)) return -1;
	if (ntohl(dhcp(p1)->siaddr) > ntohl(dhcp(p2)->siaddr)) return 1;
	return 0;
}

static int DHCP_CHECK_SECURE(DHCP *dhcp, PACKET *p1, PACKET *p2)
{
	static char a1[16], a2[16];
	static char b1[__MAX_STR_LEN], b2[__MAX_STR_LEN];
	int i;
	if (__unlikely(dhcp(p1)->yiaddr != dhcp(p2)->yiaddr)) goto error;
	DHCP_SCAN_OPTIONS(dhcp, p2);
	memcpy(options_2, options, sizeof options);
	DHCP_SCAN_OPTIONS(dhcp, p1);
	for (i = 0; i < 256; i++) {
		if (__likely(!options[i]) && __likely(!options_2[i])) continue;
		if (i == DHCP_LEASE_TIME ||
		    i == DHCP_OPTION_OVERLOAD ||
		    i == DHCP_SERVER_IDENTIFIER ||
		    i == DHCP_PARAMETER_REQUEST_LIST ||
		    i == DHCP_MESSAGE ||
		    i == DHCP_MAXIMUM_SIZE ||
		    i == DHCP_RENEWAL_TIME ||
		    i == DHCP_REBINDING_TIME ||
		    i == DHCP_VENDOR_CLASS_ID) continue;
		if (__unlikely(!options[i]) || __unlikely(!options_2[i])) goto error;
		if (__unlikely(options[i][0] != options_2[i][0])) goto error;
		if (__unlikely(memcmp(options[i] + 1, options_2[i] + 1, options[i][0]))) goto error;
	}
	return 0;
	error:
	if (!(dhcp->flags & FLAGS_LINK_CHANGED))
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): DIFFERENT OFFERS RECEIVED FROM SERVERS %s(%s) AND %s(%s)", KERNEL$HANDLE_PATH(dhcp->handle), PRINT_IP(a1, ip(p1)->ip_src.s_addr), NET$PRINT_HWADDR(b1, PKT_SRCADDR(p1, p1->addrlen), p1->addrlen), PRINT_IP(a2, ip(p2)->ip_src.s_addr), NET$PRINT_HWADDR(b2, PKT_SRCADDR(p2, p2->addrlen), p2->addrlen));
	return 1;
}

static void DHCP_OFFERS_COMPLETE(TIMER *t)
{
	DHCP *dhcp = GET_STRUCT(t, DHCP, timer);
	PACKET *p;
	int i;
	LOWER_SPL(SPL_NET);
	dhcp->packet = NULL;
	VOID_LIST_ENTRY(&dhcp->timer.list);
	cmp_dhcp = dhcp;
	qsort(dhcp->responses, dhcp->n_responses, sizeof(PACKET *), DHCP_COMPARE_RESPONSES);
	if (dhcp->flags & FLAGS_SECURE) {
		for (i = 0; i < dhcp->n_responses - 1; i++) {
			if (__unlikely(DHCP_CHECK_SECURE(dhcp, dhcp->responses[i], dhcp->responses[i + 1]))) {
				DHCP_WAIT_RESTART(dhcp, 1);
				return;
			}
		}
	}
	p = dhcp->responses[0];
	DHCP_SCAN_OPTIONS(dhcp, p);
	dhcp->my_addr = dhcp(p)->yiaddr;
	dhcp->si_addr = get_sid(dhcp, p, options);
	memcpy(dhcp->si_hwaddr, PKT_SRCADDR(p, p->addrlen), p->addrlen);
	for (i = 0; i < dhcp->n_blacklist; i++) {
		if (dhcp->blacklist[i] == dhcp->si_addr) {
			dhcp->n_blacklist = 0;
			break;
		}
	}
	/*{
		int i;
		for (i = 0; i < dhcp->n_responses; i++)
			__debug_printf("%d: %08x\n", i, ntohl(dhcp(dhcp->responses[i])->siaddr));
	}*/
	if (__unlikely(!options[DHCP_MESSAGE_TYPE]) || __unlikely(dhcp->flags & FLAGS_ONLY_BOOTP)) {
		if (__likely(!DHCP_CONFIGURE_ACCORDING_TO_OPTIONS(dhcp))) {
			dhcp->flags |= FLAGS_ONLY_BOOTP;
			DHCP_MORE_PACKET_OPTIONS(dhcp);
			/*__debug_printf("bootp configured\n");*/
			DHCP_FREE_PACKETS(dhcp);
			dhcp->flags &= ~FLAGS_LINK_CHANGED;
			return;
		} else {
			DHCP_BLACKLIST_SERVER(dhcp, dhcp->si_addr);
			DHCP_WAIT_RESTART(dhcp, 0);
			return;
		}
	}
	dhcp->timeout = DHCP_FIRST_TIMEOUT;
	dhcp->retries = 0;
	DHCP_SEND_REQUEST(dhcp);
}

static int DHCP_CONFIGURE_ACCORDING_TO_OPTIONS(DHCP *dhcp)
{
	int r;
	in_addr_t netmask;
	int valid_netmask = 0;
	in_addr_t broadcast;
	int valid_broadcast = 0;
	if (__likely(!get_iaddr(dhcp, options, BOOTP_SUBNET_MASK, &netmask))) valid_netmask = 1;
	if (__likely(!get_iaddr(dhcp, options, BOOTP_BROADCAST_ADDRESS, &broadcast))) valid_broadcast = 1;
	if (__unlikely(r = IF_DHCP_SETUP(dhcp, dhcp->my_addr, netmask, valid_netmask, broadcast, valid_broadcast))) {
		if (dhcp->flags & FLAGS_HAVE_PREFERRED_ADDR && dhcp->my_addr == dhcp->preferred_addr) dhcp->flags &= ~FLAGS_HAVE_PREFERRED_ADDR;
		if (r > 0) return 0;
		return r;
	}
	return 0;
}

static void DHCP_MORE_PACKET_OPTIONS(DHCP *dhcp)
{
	unsigned u;
	__u32 arptimeout;
	in_addr_t gw;
	if (__likely(!(dhcp->flags & FLAGS_DISTRUST_REMOTE)) && __likely(!get_iaddr(dhcp, options, BOOTP_ROUTERS, &gw))) {
		IF_DHCP_SETUP_GW(dhcp, gw);
	}
	if (__likely(!(dhcp->flags & FLAGS_DISTRUST_LOCAL)) && __unlikely(options[BOOTP_IP_FORWARDING] != NULL)) {
		if (__unlikely(options[BOOTP_IP_FORWARDING][0] != 1)) {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): IP FORWARDING OPTION HAS LENGTH %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[BOOTP_IP_FORWARDING][0]);
		} else {
			IFF_GATE = !!options[BOOTP_IP_FORWARDING][1];
		}
	}
	if (__likely(!(dhcp->flags & FLAGS_DISTRUST_LOCAL)) && __unlikely(options[BOOTP_MAXIMUM_REASSEMBLY_SIZE] != NULL)) {
		if (__unlikely(options[BOOTP_MAXIMUM_REASSEMBLY_SIZE][0] != 2)) {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): MAXIMUM REASSEMBLY SIZE OPTION HAS LENGTH %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[BOOTP_IP_FORWARDING][0]);
		} else {
			IP_MAX_DEFRAG = (options[BOOTP_MAXIMUM_REASSEMBLY_SIZE][1] << 8) + options[BOOTP_MAXIMUM_REASSEMBLY_SIZE][2];
		}
	}
	if (__unlikely(options[BOOTP_MTU] != NULL)) {
		if (__unlikely(options[BOOTP_MTU][0] != 2)) {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): MTU OPTION HAS LENGTH %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[BOOTP_IP_FORWARDING][0]);
		} else {
			unsigned mtu = (options[BOOTP_MTU][1] << 8) + options[BOOTP_MTU][2];
			if (__likely(mtu)) {
				if (__unlikely(mtu < IP_MIN_MTU)) mtu = IP_MIN_MTU;
				if (mtu < dhcp->if_type->mtu) dhcp->if_type->mtu = mtu;
			}
		}
	}
	if (__likely(!(dhcp->flags & FLAGS_DISTRUST_LOCAL)) && __unlikely(options[BOOTP_STATIC_ROUTES] != NULL)) {
		int nr = options[BOOTP_STATIC_ROUTES][0] & ~7;
		int i;
		for (i = nr - 8; i >= 0; i -= 8) {
			in_addr_t dst;
			in_addr_t rtr;
			memcpy(&dst, options[BOOTP_STATIC_ROUTES] + 1 + i, 4);
			memcpy(&rtr, options[BOOTP_STATIC_ROUTES] + 5 + i, 4);
			IF_DHCP_SETUP_ROUTE(dhcp, dst, -1, rtr);
		}
	}
	if (__likely(!(dhcp->flags & FLAGS_DISTRUST_LOCAL)) && __unlikely(!get_dword(dhcp, options, BOOTP_ARP_CACHE_TIMEOUT, &arptimeout))) {
		if (__unlikely(arptimeout > MAXINT / JIFFIES_PER_SECOND)) arptimeout = MAXINT / JIFFIES_PER_SECOND;
		arptimeout *= JIFFIES_PER_SECOND;
		if (__unlikely(!arptimeout)) arptimeout = 1;
		ARP_TIME_TO_LIVE = arptimeout * JIFFIES_PER_SECOND;
	}
	if (__likely(!(dhcp->flags & FLAGS_DISTRUST_REMOTE))) for (u = 0; u < 256; u++)
		if (__unlikely(options[u] != NULL)) {
			memcpy(DHCP_APPARRAY[u], options[u], options[u][0] + 1);
			DHCP_APPOWNER[u] = dhcp;
			/*__debug_printf("options(%d)\n", u);*/
		}
}

static void DHCP_SEND_REQUEST(DHCP *dhcp)
{
	PACKET *p;
	__u8 *o;
	retry_alloc:
	ALLOC_PACKET(p, sizeof(struct dhcp), &NET$PKTPOOL, SPL_NET, if (!NET$OOM()) goto retry_alloc; goto set_time;);
	p->data_length = sizeof(struct dhcp);
	DHCP_FILL_PACKET(dhcp, p);
	init_opt(p);
	opt_byte(DHCP_MESSAGE_TYPE);
	opt_byte(1);
	opt_byte(DHCP_MESSAGE_TYPE_REQUEST);
	opt_byte(DHCP_REQUESTED_IP_ADDRESS);
	opt_byte(4);
	opt_iaddr(dhcp->my_addr);
	opt_byte(DHCP_SERVER_IDENTIFIER);
	opt_byte(4);
	opt_iaddr(dhcp->si_addr);
	DHCP_COMMON_OPTS(dhcp, o, dhcp(p)->options + sizeof dhcp(p)->options);

	IF_SEND_DHCP_PACKET(dhcp, p);
	
	set_time:
	KERNEL$DEL_TIMER(&dhcp->timer);
	dhcp->packet = DHCP_ACK_NAK_REPLY;
	dhcp->timer.fn = DHCP_RETRANSMIT_REQUEST;
	KERNEL$SET_TIMER(DHCP_TIME_FUZZ(dhcp->timeout), &dhcp->timer);
}

static void DHCP_RETRANSMIT_REQUEST(TIMER *t)
{
	DHCP *dhcp = GET_STRUCT(t, DHCP, timer);
	LOWER_SPL(SPL_NET);
	VOID_LIST_ENTRY(&dhcp->timer.list);
	if (__unlikely(dhcp->flags & FLAGS_LINK_CHANGED)) goto restart;
	dhcp->timeout = dhcp->timeout DHCP_TIMEOUT_BACKOFF;
	if (__likely(dhcp->timeout > DHCP_MAX_TIMEOUT)) dhcp->timeout = DHCP_MAX_TIMEOUT;
	dhcp->retries++;
	if (dhcp->retries == DHCP_REQUEST_RETRIES) {
		dhcp->timeout = DHCP_FIRST_TIMEOUT;
	}
	if (dhcp->retries > DHCP_REQUEST_RETRIES) {
		DHCP_BLACKLIST_SERVER(dhcp, dhcp->si_addr);
		restart:
		DHCP_WAIT_RESTART(dhcp, 0);
		return;
	}
	DHCP_SEND_REQUEST(dhcp);
}

static void DHCP_ACK_NAK_REPLY(DHCP *dhcp, PACKET *p)
{
	__u8 type;
	DHCP_SCAN_OPTIONS(dhcp, p);
	if (__unlikely(!options[DHCP_MESSAGE_TYPE])) goto drop;
	if (__unlikely(options[DHCP_MESSAGE_TYPE][0] != 1)) {
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): MESSAGE TYPE OPTION HAS LENGTH %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[DHCP_MESSAGE_TYPE][0]);
		nak:
		DHCP_BLACKLIST_SERVER(dhcp, dhcp->si_addr);
		DHCP_WAIT_RESTART(dhcp, 0);
		goto drop;
	}
	type = options[DHCP_MESSAGE_TYPE][1];
	if (__unlikely(type == DHCP_MESSAGE_TYPE_NAK)) {
		if (!(dhcp->flags & FLAGS_LINK_CHANGED)) {
			char *m;
			if ((m = options[DHCP_MESSAGE])) {
				static char str[__MAX_STR_LEN];
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): SERVER SENT NAK: %s", KERNEL$HANDLE_PATH(dhcp->handle), NET$PRINT_STRING(str, m + 1, m[0]));
			} else {
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): SERVER SENT NAK", KERNEL$HANDLE_PATH(dhcp->handle));
			}
		}
		goto nak;
	}
	if (__unlikely(type != DHCP_MESSAGE_TYPE_ACK)) goto drop;
	
	DHCP_SCAN_OPTIONS(dhcp, dhcp->responses[0]);
	if (__unlikely(DHCP_GET_LEASE(dhcp)) || __unlikely(DHCP_CONFIGURE_ACCORDING_TO_OPTIONS(dhcp))) goto nak;
	dhcp->preferred_addr = dhcp->my_addr;
	dhcp->flags |= FLAGS_HAVE_PREFERRED_ADDR;

	DHCP_MORE_PACKET_OPTIONS(dhcp);

	DHCP_FREE_PACKETS(dhcp);
	dhcp->packet = NULL;
	dhcp->flags &= ~FLAGS_LINK_CHANGED;
	/*__debug_printf("dhcp configured\n");*/
	drop:
	p->status = 0;
	CALL_PKT(p);
}

static int DHCP_GET_LEASE(DHCP *dhcp)
{
	__u32 lts;
	if (__likely(!get_dword(dhcp, options, DHCP_LEASE_TIME, &lts)) && __likely(lts != 0xffffffffU)) {
		__u32 t1s, t2s;
		u_jiffies_t lt, t1, t2, now;
		int t1set = 0, t2set = 0;
		if (__unlikely(!lts)) {
			if (errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): INVALID ZERO LEASE TIME", KERNEL$HANDLE_PATH(dhcp->handle));
			return -EINVAL;
		}
		if (__unlikely(lts > MAXINT / JIFFIES_PER_SECOND)) lts = MAXINT/ JIFFIES_PER_SECOND;
		lt = (u_jiffies_t)lts * JIFFIES_PER_SECOND;
		lt -= lt >> 8;
		if (!get_dword(dhcp, options, DHCP_RENEWAL_TIME, &t1s) && __likely(t1s)) t1 = (u_jiffies_t)t1s * JIFFIES_PER_SECOND, t1set = 1;
		else t1 = lt / 2;
		if (!get_dword(dhcp, options, DHCP_REBINDING_TIME, &t2s) && __likely(t2s)) t2 = (u_jiffies_t)t2s * JIFFIES_PER_SECOND, t2set = 1;
		else t2 = (__u64)lt * 7 / 8;
		if (__unlikely(t1 > lt - JIFFIES_PER_SECOND / 2)) t1 = lt - JIFFIES_PER_SECOND / 2;
		if (__unlikely(t2 > lt - JIFFIES_PER_SECOND / 2)) t2 = lt - JIFFIES_PER_SECOND / 2;
		if (__unlikely(t1 > t2 - JIFFIES_PER_SECOND / 4)) {
			if (t1set && t2set) t1 = t2 - JIFFIES_PER_SECOND / 4;
			if (t1set && !t2set) t2 = t1 + (lt - t1) * 3 / 4;
			if (!t1set && t2set) t1 = t2 / 2;
			if (!t1set && !t2set) t1 = t2 - JIFFIES_PER_SECOND / 4;
		}
		dhcp->lease_end = dhcp->last_discovery_time + lt;
		dhcp->t1 = dhcp->last_discovery_time + t1;
		dhcp->t2 = dhcp->last_discovery_time + t2;
		now = KERNEL$GET_JIFFIES();
		if (__unlikely(now >= dhcp->t1)) dhcp->t1 = now + 1;
		if (__unlikely(dhcp->t1 >= dhcp->t2)) dhcp->t2 = dhcp->t1 + 1;
		if (__unlikely(dhcp->t2 >= dhcp->lease_end)) return -ETIMEDOUT;
		if (__likely(dhcp->t1 - now > DHCP_TIMEOUT_FUZZ * 3) &&
		    __likely(dhcp->t2 - dhcp->t1 > DHCP_TIMEOUT_FUZZ * 3) &&
		    __likely(dhcp->lease_end - dhcp->t2 > DHCP_TIMEOUT_FUZZ * 3)) {
			dhcp->t1 = DHCP_TIME_FUZZ(dhcp->t1);
			dhcp->t2 = DHCP_TIME_FUZZ(dhcp->t2);
		}
		KERNEL$DEL_TIMER(&dhcp->timer);
		dhcp->timer.fn = DHCP_T12_EXPIRED;
		KERNEL$SET_TIMER(dhcp->t1 - now, &dhcp->timer);
	} else {
		KERNEL$DEL_TIMER(&dhcp->timer);
		VOID_LIST_ENTRY(&dhcp->timer.list);
	}
	return 0;
}

static void DHCP_T12_EXPIRED(TIMER *t)
{
	PACKET *p;
	__u8 *o;
	int t2;
	jiffies_t next_time;
	DHCP *dhcp = GET_STRUCT(t, DHCP, timer);
	LOWER_SPL(SPL_NET);
	VOID_LIST_ENTRY(&dhcp->timer.list);
	dhcp->xid = htonl(RANDOM_DHCP_XID((__u32 *)dhcp->chaddr));
	dhcp->last_discovery_time = KERNEL$GET_JIFFIES();
	if (__likely(!dhcp->packet)) dhcp->start_time = dhcp->last_discovery_time;
	t2 = dhcp->last_discovery_time >= dhcp->t2;
	if (__unlikely(dhcp->last_discovery_time >= dhcp->lease_end)) {
		IF_DHCP_UNSET(dhcp);
		DHCP_RESTART(dhcp);
		return;
	}
	if (dhcp->n_responses) {
		DHCP_SCAN_OPTIONS(dhcp, dhcp->responses[0]);
		if (__unlikely(DHCP_GET_LEASE(dhcp)) || __unlikely(DHCP_CONFIGURE_ACCORDING_TO_OPTIONS(dhcp))) {
			DHCP_FREE_PACKETS(dhcp);
			goto set_time;
		}
		DHCP_MORE_PACKET_OPTIONS(dhcp);
		dhcp->packet = NULL;
		DHCP_FREE_PACKETS(dhcp);
		return;
	}
	DHCP_FREE_PACKETS(dhcp);
	retry_alloc:
	ALLOC_PACKET(p, sizeof(struct dhcp), &NET$PKTPOOL, SPL_NET, if (!NET$OOM()) goto retry_alloc; goto set_time;);
	p->data_length = sizeof(struct dhcp);

	DHCP_FILL_PACKET(dhcp, p);
	dhcp(p)->ip_src.s_addr = dhcp->my_addr;
	if (__likely(!t2)) {
		dhcp(p)->ip_dst.s_addr = dhcp->si_addr;
		memcpy(PKT_DSTADDR(p, p->addrlen), dhcp->si_hwaddr, p->addrlen);
	}

	init_opt(p);
	opt_byte(DHCP_MESSAGE_TYPE);
	opt_byte(1);
	opt_byte(DHCP_MESSAGE_TYPE_REQUEST);
	DHCP_COMMON_OPTS(dhcp, o, dhcp(p)->options + sizeof dhcp(p)->options);
	dhcp(p)->ciaddr = dhcp->my_addr;

	IF_SEND_DHCP_PACKET(dhcp, p);

	set_time:
	if (__likely(!t2)) {
		next_time = (u_jiffies_t)(dhcp->t2 - dhcp->last_discovery_time) / 2;
		if (__unlikely(next_time - dhcp->last_discovery_time < DHCP_MIN_RENEW_TIME)) {
			if (dhcp->last_discovery_time + DHCP_MIN_RENEW_TIME < dhcp->t2) {
				next_time = dhcp->last_discovery_time + DHCP_MIN_RENEW_TIME;
			} else {
				next_time = dhcp->t2;
			}
		}
	} else {
		next_time = (u_jiffies_t)(dhcp->lease_end - dhcp->last_discovery_time) / 2;
		if (__unlikely(next_time - dhcp->lease_end < DHCP_MIN_RENEW_TIME)) {
			if (dhcp->last_discovery_time + DHCP_MIN_RENEW_TIME < dhcp->lease_end) {
				next_time = dhcp->last_discovery_time + DHCP_MIN_RENEW_TIME;
			} else if (dhcp->last_discovery_time + DHCP_LAST_TRY_TIME < dhcp->lease_end) {
				next_time = dhcp->lease_end - DHCP_LAST_TRY_TIME;
			} else {
				next_time = dhcp->lease_end;
			}
		}
	}
	KERNEL$DEL_TIMER(&dhcp->timer);
	dhcp->packet = DHCP_BOUND_REPLY;
	dhcp->timer.fn = DHCP_T12_EXPIRED;
	KERNEL$SET_TIMER(next_time - dhcp->last_discovery_time, &dhcp->timer);
}

static void DHCP_BOUND_REPLY(DHCP *dhcp, PACKET *p)
{
	DHCP_SCAN_OPTIONS(dhcp, p);
	if (__unlikely(!options[DHCP_MESSAGE_TYPE])) goto drop;
	if (__unlikely(options[DHCP_MESSAGE_TYPE][0] != 1)) {
		if (errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "DHCP(%s): MESSAGE TYPE OPTION HAS LENGTH %02X", KERNEL$HANDLE_PATH(dhcp->handle), options[DHCP_MESSAGE_TYPE][0]);
		goto drop;
	}
	if (__unlikely(options[DHCP_MESSAGE_TYPE][1] != DHCP_MESSAGE_TYPE_ACK)) goto drop;
	if (dhcp->flags & FLAGS_SECURE) {
		if (!dhcp->n_responses) {
			if (KERNEL$GET_JIFFIES() < dhcp->lease_end - DHCP_WAIT_FOR_OFFERS) {
				dhcp->responses[0] = p;
				dhcp->n_responses = 1;
				KERNEL$DEL_TIMER(&dhcp->timer);
				dhcp->timer.fn = DHCP_T12_EXPIRED;
				KERNEL$SET_TIMER(DHCP_WAIT_FOR_OFFERS, &dhcp->timer);
				return;
			}
			goto drop;
		} else {
			if (__unlikely(DHCP_CHECK_SECURE(dhcp, dhcp->responses[0], p))) {
				IF_DHCP_UNSET(dhcp);
				DHCP_WAIT_RESTART(dhcp, 1);
			}
			goto drop;
		}
	}
	if (__unlikely(DHCP_GET_LEASE(dhcp)) || __unlikely(DHCP_CONFIGURE_ACCORDING_TO_OPTIONS(dhcp))) goto drop;
	DHCP_MORE_PACKET_OPTIONS(dhcp);
	dhcp->packet = NULL;
	dhcp->flags &= ~FLAGS_LINK_CHANGED;
	/*__debug_printf("lease extended\n");*/
	drop:
	p->status = 0;
	CALL_PKT(p);
}

static void DHCP_WAIT_RESTART(DHCP *dhcp, int longwait)
{
	DHCP_FREE_PACKETS(dhcp);
	KERNEL$DEL_TIMER(&dhcp->timer);
	dhcp->packet = NULL;
	dhcp->flags &= ~(FLAGS_STATE_LONGWAIT | FLAGS_STATE_SHORTWAIT);
	if (longwait == -1) dhcp->flags |= FLAGS_STATE_SHORTWAIT;
	if (dhcp->flags & FLAGS_LINK_CHANGED) {
		longwait = -1;
	} else {
		if (longwait == 1) dhcp->flags |= FLAGS_STATE_LONGWAIT;
	}
	dhcp->timer.fn = DHCP_FINISHED_WAIT_FOR_RESTART;
	KERNEL$SET_TIMER(__unlikely(longwait == -1) ? DHCP_LINKCHG_TIME : __likely(longwait == 0) ? DHCP_TIME_FUZZ(DHCP_RESTART_TIME) : DHCP_TIME_FUZZ(DHCP_SEC_RESTART_TIME), &dhcp->timer);
}

static void DHCP_FINISHED_WAIT_FOR_RESTART(TIMER *t)
{
	DHCP *dhcp = GET_STRUCT(t, DHCP, timer);
	LOWER_SPL(SPL_NET);
	VOID_LIST_ENTRY(&dhcp->timer.list);
	if (!(dhcp->flags & FLAGS_STATE_SHORTWAIT)) IF_DHCP_UNSET(dhcp);
	DHCP_RESTART(dhcp);
}

char *DHCP_MODE(DHCP *dhcp)
{
	if (__likely(!(dhcp->flags & FLAGS_ONLY_BOOTP))) return "DHCP";
	else return "BOOTP";
}

char *DHCP_STATUS(DHCP *dhcp)
{
	if (__likely(dhcp->packet == DHCP_DISCOVERY_REPLY)) {
		if (!dhcp->n_responses) return "WAITING FOR REPLY";
		else return "COLLECTING REPLIES";
	}
	if (__unlikely(dhcp->packet == DHCP_ACK_NAK_REPLY)) return "WAITING FOR ACK";
	if (__unlikely(dhcp->flags & FLAGS_STATE_LONGWAIT)) return "WAITING 5 MINUTES BECAUSE OF DIFFERENT RESPONSES";
	return "RESTARTING";
}

void DHCP_INIT(DHCP *dhcp, IF_TYPE *if_type, int handle, int request_addr, in_addr_t addr, int only_bootp, int secure, int distrust)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_NET);
	dhcp->if_type = if_type;
	ADD_TO_XLIST(&dhcp_interfaces, &dhcp->list);
	dhcp->handle = handle;
	dhcp->flags = 0;
	if (request_addr) dhcp->flags |= FLAGS_HAVE_PREFERRED_ADDR, dhcp->preferred_addr = addr;
	if (only_bootp) dhcp->flags |= FLAGS_ONLY_BOOTP | FLAGS_ORIGINALLY_ONLY_BOOTP;
	if (secure) dhcp->flags |= FLAGS_SECURE;
	if (distrust >= 2) dhcp->flags |= FLAGS_DISTRUST_LOCAL | FLAGS_DISTRUST_REMOTE;
	if (distrust >= 1) dhcp->flags |= FLAGS_DISTRUST_LOCAL;
	dhcp->packet = NULL;
	VOID_LIST_ENTRY(&dhcp->timer.list);
	if (__likely(dhcp->if_type->addrlen < sizeof(dhcp->chaddr)))
		memcpy(dhcp->chaddr, dhcp->if_type->addr, dhcp->if_type->addrlen),
		memset(dhcp->chaddr + dhcp->if_type->addrlen, 0, sizeof(dhcp->chaddr) - dhcp->if_type->addrlen);
	else
		memcpy(dhcp->chaddr, dhcp->if_type->addr, sizeof(dhcp->chaddr));
	dhcp->n_responses = 0;
	dhcp->n_blacklist = 0;
	DHCP_RESTART(dhcp);
	LOWER_SPLX(spl);
}

void DHCP_DESTROY(DHCP *dhcp)
{
	int i;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_NET);
	for (i = 0; i < 256; i++) if (__unlikely(DHCP_APPOWNER[i] == dhcp)) {
		DHCP_APPOWNER[i] = NULL;
		memset(DHCP_APPARRAY[i], 0, 256);
	}
	KERNEL$DEL_TIMER(&dhcp->timer);
	DHCP_FREE_PACKETS(dhcp);
	DEL_FROM_LIST(&dhcp->list);
	LOWER_SPLX(spl);
}

void DHCP_INPUT(PACKET *p)
{
	DHCP *dhcp;
	REMOVE_IP_OPTIONS;
	if (__unlikely((*(__u32 *)&ip(p)->ip_id & htonl(IP_MF | IP_OFFMASK)))) goto drop;
	if (ip(p)->ip_p != IPPROTO_UDP) goto drop;
	if (__unlikely(p->data_length < sizeof(struct dhcp))) goto drop;
	if (dhcp(p)->udp_dport != htons(BOOTP_CLIENT_PORT)) goto drop;
	if (__unlikely(ntohs(dhcp(p)->udp_len) < sizeof(struct dhcp) - sizeof(struct ip))) goto drop;
	CHECK_UDP_CHECKSUM;
	if (__unlikely(dhcp(p)->op != OP_BOOTREPLY)) goto drop;
	XLIST_FOR_EACH(dhcp, &dhcp_interfaces, DHCP, list) {
		if (dhcp->xid == dhcp(p)->xid &&
		    __likely(dhcp->if_type->id == p->id) &&
		    __likely(!memcmp(dhcp->chaddr, dhcp(p)->chaddr, sizeof(dhcp->chaddr)))) {
			if (__unlikely(p->addrlen != dhcp->if_type->addrlen)) {
				goto drop;
			}
			if (__likely(dhcp->packet != NULL)) {
				dhcp->packet(dhcp, p);
				return;
			} else {
				goto drop;
			}
		}
	}

	drop:
	p->status = 0;
	CALL_PKT(p);
}

void DHCP_NOTIFY_LINK_CHANGE(DHCP *dhcp)
{
	dhcp->flags |= FLAGS_LINK_CHANGED;
	/*__debug_printf("dhcp linkchg...\n");*/
	if (__likely(dhcp->packet == DHCP_DISCOVERY_REPLY)) {
		if (!dhcp->n_responses) {
			dhcp->timeout = DHCP_FIRST_TIMEOUT;
			KERNEL$DEL_TIMER(&dhcp->timer);
			KERNEL$SET_TIMER(DHCP_LINKCHG_TIME, &dhcp->timer);
		}
		return;
	}
	if (__unlikely(dhcp->packet == DHCP_ACK_NAK_REPLY)) {
		if (__likely(dhcp->retries < DHCP_REQUEST_RETRIES - 1)) {
			dhcp->retries = DHCP_REQUEST_RETRIES - 1;
			dhcp->timeout = DHCP_LINKCHG_TIME;
			KERNEL$DEL_TIMER(&dhcp->timer);
			KERNEL$SET_TIMER(DHCP_LINKCHG_TIME, &dhcp->timer);
		} else if (dhcp->retries < DHCP_REQUEST_RETRIES) {
			dhcp->retries = DHCP_REQUEST_RETRIES;
			dhcp->timeout = DHCP_LINKCHG_TIME;
			KERNEL$DEL_TIMER(&dhcp->timer);
			KERNEL$SET_TIMER(DHCP_LINKCHG_TIME, &dhcp->timer);
		}
		return;
	}
	if (__unlikely(dhcp->flags & FLAGS_ONLY_BOOTP)) {
		dhcp_restart:
		DHCP_WAIT_RESTART(dhcp, -1);
		return;
	}
	if (__likely(dhcp->timer.fn == DHCP_T12_EXPIRED)) {
		jiffies_t now = KERNEL$GET_JIFFIES();
		jiffies_t t;
		t = now + DHCP_LINKCHG_T1;
		if (__likely(t < dhcp->t1)) dhcp->t1 = t;
		t = now + DHCP_LINKCHG_T2;
		if (__likely(t < dhcp->t2)) dhcp->t2 = t;
		t = now + DHCP_LINKCHG_LEASE;
		if (__likely(t < dhcp->lease_end)) dhcp->lease_end = t;
		KERNEL$DEL_TIMER(&dhcp->timer);
		KERNEL$SET_TIMER(DHCP_LINKCHG_TIME, &dhcp->timer);
		return;
	} else if (dhcp->timer.fn == DHCP_FINISHED_WAIT_FOR_RESTART) {
		if (!(dhcp->flags & FLAGS_STATE_SHORTWAIT)) {
			IF_DHCP_UNSET(dhcp);
			goto dhcp_restart;
		}
	} else if (dhcp->timer.fn == DHCP_FINISHED_WAIT_FOR_RESTART) {
		KERNEL$DEL_TIMER(&dhcp->timer);
		KERNEL$SET_TIMER(DHCP_LINKCHG_TIME, &dhcp->timer);
	}
}
