#include <STDLIB.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/PKT.H>
#include <SPAD/IOCTL.H>
#include <NETINET/IGMP.H>
#include "MCAST.H"

#include "TCPIP.H"

struct __mcast_record {
	LIST_ENTRY list;
	struct ip_mreq mreq;
	unsigned refcount;
	IOCTLRQ mcast_ioctl;
	u_jiffies_lo_t expire_time;
	TIMER timer;
	jiffies_t v1_time;
	jiffies_t v2_time;
	char v1_heard;
	char v2_heard;
	char version;
	char posted;
	char sent_report;
	char need_cancel;
	char need_if_chg;
	char was_last;
	__u8 hwaddr[MCAST_PHYS_ADDRLEN];
	IF_TYPE type;
};

typedef struct __mcast_record MCAST_RECORD;

static DECL_XLIST(mcast_records);
static DECL_XLIST(mcast_canceling);

static struct __slhead mcast_slab;

static __finline__ in_addr_t IGMP_XLATE_ADDR(in_addr_t addr)
{
	if (__likely(addr == htonl(INADDR_ANY))) return IP_DEFAULT_MULTICAST_IF();
	return addr;
}

extern AST_STUB IGMP_MCAST_IOCTL_AST;
extern AST_STUB IGMP_MCAST_RESET_AST;
static void IGMP_SEND_REPORT(MCAST_RECORD *mr, unsigned type);
static void IGMP_TIMER_EXPIRE(TIMER *t);

static int IGMP_IS_PRESENT(struct ip_mreq *mreq)
{
	MCAST_RECORD *mr;
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (mr->mreq.imr_multiaddr.s_addr == mreq->imr_multiaddr.s_addr && IGMP_XLATE_ADDR(mr->mreq.imr_interface.s_addr) == IGMP_XLATE_ADDR(mreq->imr_interface.s_addr) && mr->sent_report) {
		return 1;
	}
	return 0;
}

static void IGMP_V1_TIME(MCAST_RECORD *m)
{
	MCAST_RECORD *mr;
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (mr->sent_report && IGMP_XLATE_ADDR(mr->mreq.imr_interface.s_addr) == IGMP_XLATE_ADDR(m->mreq.imr_interface.s_addr)) {
		m->v1_time = mr->v1_time;
		m->v2_time = mr->v2_time;
		m->v1_heard = mr->v1_heard;
		m->v2_heard = mr->v2_heard;
		return;
	}
	m->v1_time = m->v2_time = KERNEL$GET_JIFFIES();
	m->v1_heard = 0;
	m->v2_heard = 0;
}

static int IGMP_REPORT_TYPE(MCAST_RECORD *mr)
{
	if (__unlikely(!mr->sent_report))
		KERNEL$SUICIDE("IGMP_REPORT_TYPE: INACTIVE MCAST RECORD");
	if (mr->version == 2) return IGMP_V2_MEMBERSHIP_REPORT;
	if (mr->v2_heard && !mr->v1_heard) return IGMP_V2_MEMBERSHIP_REPORT;
	if (KERNEL$GET_JIFFIES() - mr->v1_time > IGMP_V1_PRESENT_TIMEOUT) return IGMP_V2_MEMBERSHIP_REPORT;
	return IGMP_V1_MEMBERSHIP_REPORT;
}

static void IGMP_SET_TIMER(MCAST_RECORD *mr, u_jiffies_lo_t ticks, int rnd)
{
	u_jiffies_lo_t j;
	if (__unlikely(!mr->sent_report)) return;
	j = KERNEL$GET_JIFFIES_LO();
	if (__unlikely(mr->timer.fn != NULL)) {
		if (j + ticks > mr->expire_time) return;
		KERNEL$DEL_TIMER(&mr->timer);
	}
	if (__likely(rnd) && __likely(ticks != 0)) ticks = random() % ticks;
	mr->expire_time = j + ticks;
	mr->timer.fn = IGMP_TIMER_EXPIRE;
	KERNEL$SET_TIMER(ticks, &mr->timer);
}

static void IGMP_CANCEL_TIMER(MCAST_RECORD *mr)
{
	if (!mr->timer.fn) return;
	KERNEL$DEL_TIMER(&mr->timer);
	mr->timer.fn = NULL;
}

void IGMP_CHECK_MULTICASTS(void)
{
	MCAST_RECORD *mr;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_NET), spl))) KERNEL$SUICIDE("IGMP_CHECK_MULTICASTS AT SPL %08X", spl);
	RAISE_SPL(SPL_NET);
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (!mr->posted) {
		int h;
		IF_TYPE *type;
		in_addr_t local_addr = IGMP_XLATE_ADDR(mr->mreq.imr_interface.s_addr);
		if (__unlikely(local_addr == htonl(INADDR_ANY))) {
			mr->mcast_ioctl.h = -1;
			continue;
		}
		h = IP_IF_HANDLE(local_addr);
		if (__unlikely(h == -1)) {
			mr->mcast_ioctl.h = -1;
			continue;
		}
		mr->mcast_ioctl.h = h;
		type = IP_IF_TYPE(local_addr);
		memcpy(&mr->type, type, sizeof(IF_TYPE));
		mr->version = IP_IF_IGMP_VERSION(local_addr);
		if (__unlikely(mr->type.addrlen != MCAST_PHYS_ADDRLEN)) goto do_report;
		mr->mcast_ioctl.fn = IGMP_MCAST_IOCTL_AST;
		mr->mcast_ioctl.ioctl = IOCTL_IF_SET_MULTICAST;
		mr->mcast_ioctl.param = PARAM_MULTICAST_ADDRESS;
		mr->mcast_ioctl.v.ptr = (unsigned long)&mr->hwaddr;
		mr->mcast_ioctl.v.len = MCAST_PHYS_ADDRLEN;
		mr->mcast_ioctl.v.vspace = &KERNEL$VIRTUAL;
		MCAST_PHYS_ADDR(mr->mreq.imr_multiaddr.s_addr, mr->hwaddr);
		mr->posted = 1;
		CALL_IORQ(&mr->mcast_ioctl, KERNEL$IOCTL);
		do_report:
		if (__likely(!mr->sent_report) && __likely(!TCPIP_MULTICAST_LOCAL_ADDRESS(mr->mreq.imr_multiaddr.s_addr))) {
			int dont_need = IGMP_IS_PRESENT(&mr->mreq);
			IGMP_V1_TIME(mr);
			mr->sent_report = 1;
			mr->was_last = 1;
			if (!dont_need) IGMP_SEND_REPORT(mr, IGMP_REPORT_TYPE(mr));
			IGMP_SET_TIMER(mr, IGMP_RETRY_TIME, 0);
		}
	}
	LOWER_SPLX(spl);
}

int IGMP_GET_MCAST_RECORD(struct ip_mreq *mreq, IORQ *rq)
{
	MCAST_RECORD *mr;
	struct ip_mreq allh;
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) {
		if (mr->mreq.imr_multiaddr.s_addr == mreq->imr_multiaddr.s_addr && mr->mreq.imr_interface.s_addr == mreq->imr_interface.s_addr) {
			mr->refcount++;
			return 0;
		}
	}
	if (__likely(!TCPIP_MULTICAST_LOCAL_ADDRESS(mreq->imr_multiaddr.s_addr))) {
		allh.imr_multiaddr.s_addr = htonl(INADDR_ALLHOSTS_GROUP);
		allh.imr_interface.s_addr = mreq->imr_interface.s_addr;
		if (__unlikely(IGMP_GET_MCAST_RECORD(&allh, rq))) return 0;
	}
	if (__unlikely(!(mr = __slalloc(&mcast_slab)))) {
		WQ *wq;
		if (__likely(!TCPIP_MULTICAST_LOCAL_ADDRESS(mreq->imr_multiaddr.s_addr)))
			IGMP_RELEASE_MCAST_RECORD(&allh);
		wq = NET$OOM();
		if (!wq) {
			CALL_IORQ_EXPR(rq, (IORQ *)rq->tmp1);
		} else if (__IS_ERR(wq)) {
			rq->status = __PTR_ERR(wq);
			CALL_AST(rq);
		} else {
			WQ_WAIT_F(wq, rq);
		}
		return 1;
	}
	memcpy(&mr->mreq, mreq, sizeof(struct ip_mreq));
	mr->refcount = 1;
	INIT_TIMER(&mr->timer);
	mr->timer.fn = NULL;
	mr->posted = 0;
	mr->sent_report = 0;
	mr->need_cancel = 0;
	mr->need_if_chg = 0;
	mr->was_last = 0;
	mr->mcast_ioctl.h = -1;
	ADD_TO_XLIST(&mcast_records, &mr->list);
	IGMP_CHECK_MULTICASTS();
	return 0;
}

DECL_AST(IGMP_MCAST_IOCTL_AST, SPL_NET, IOCTLRQ)
{
	MCAST_RECORD *mr = GET_STRUCT(RQ, MCAST_RECORD, mcast_ioctl);
	if (mr->need_cancel) {
		mr->mcast_ioctl.fn = IGMP_MCAST_RESET_AST;
		mr->mcast_ioctl.v.ptr = 0;
		mr->mcast_ioctl.v.len = 0;
		mr->mcast_ioctl.ioctl = IOCTL_IF_RESET_MULTICAST;
		mr->mcast_ioctl.param = PARAM_MULTICAST_ALL;
		RETURN_IORQ(&mr->mcast_ioctl, KERNEL$IOCTL);
	}
	if (mr->need_if_chg) {
		mr->posted = 0;
		mr->sent_report = 0;
		mr->was_last = 0;
		mr->need_if_chg = 0;
		IGMP_CHECK_MULTICASTS();
		RETURN;
	}
	mr->mcast_ioctl.status = RQS_PROCESSING;
	WQ_WAIT(&KERNEL$LOGICAL_WAIT, &mr->mcast_ioctl, KERNEL$IOCTL);
	RETURN;
}

DECL_AST(IGMP_MCAST_RESET_AST, SPL_NET, IOCTLRQ)
{
	MCAST_RECORD *mr = GET_STRUCT(RQ, MCAST_RECORD, mcast_ioctl);
	DEL_FROM_LIST(&mr->list);
	__slfree(mr);
	RETURN;
}

void IGMP_RELEASE_MCAST_RECORD(struct ip_mreq *mreq)
{
	MCAST_RECORD *mr;
	struct ip_mreq allh;
	static char a1[16];
	static char a2[16];
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (__likely(mr->mreq.imr_multiaddr.s_addr == mreq->imr_multiaddr.s_addr) && __likely(mr->mreq.imr_interface.s_addr == mreq->imr_interface.s_addr)) {
		if (!--mr->refcount) {
			if (__likely(!TCPIP_MULTICAST_LOCAL_ADDRESS(mr->mreq.imr_multiaddr.s_addr))) {
				allh.imr_multiaddr.s_addr = htonl(INADDR_ALLHOSTS_GROUP);
				allh.imr_interface.s_addr = mreq->imr_interface.s_addr;
				IGMP_RELEASE_MCAST_RECORD(&allh);
			}
			IGMP_CANCEL_TIMER(mr);
			DEL_FROM_LIST(&mr->list);
			if (mr->sent_report && mr->was_last && __likely(!IGMP_IS_PRESENT(&mr->mreq))) {
				IGMP_SEND_REPORT(mr, IGMP_V2_LEAVE_GROUP);
			}
			if (mr->posted) {
				ADD_TO_XLIST(&mcast_canceling, &mr->list);
				mr->need_cancel = 1;
				KERNEL$CIO((IORQ *)&mr->mcast_ioctl);
			} else {
				__slfree(mr);
			}
		}
		return;
	}
	KERNEL$SUICIDE("IGMP_RELEASE_MCAST_RECORD: RELEASING NON-EXISTENT MULTICAST GROUP %s, INTERFACE %s", PRINT_IP(a1, mreq->imr_multiaddr.s_addr), PRINT_IP(a2, mreq->imr_interface.s_addr));
}

static void IGMP_SEND_REPORT(MCAST_RECORD *mr, unsigned type)
{
	PACKET *p;
	in_addr_t local_addr;
	if (__unlikely(TCPIP_MULTICAST_LOCAL_ADDRESS(mr->mreq.imr_multiaddr.s_addr))) return;
	local_addr = IGMP_XLATE_ADDR(mr->mreq.imr_interface.s_addr);
	if (__unlikely(local_addr == htonl(INADDR_ANY))) return;
	ALLOC_PACKET(p, sizeof(struct ip) + 4 + sizeof(struct igmp), &NET$PKTPOOL, SPL_NET, NET$DELAYED_OOM(); return);
	p->fn = NET$FREE_PACKET;
	p->data_length = sizeof(struct ip) + 4 + sizeof(struct igmp);
	ip(p)->ip_vhl = IP_VHL + 1;
	ip(p)->ip_tos = IPTOS_PREC_INTERNETCONTROL | IPTOS_RELIABILITY;
	*(__u32 *)&ip(p)->ip_ttl = htonl(0x01000000 | (IPPROTO_IGMP << 16));
	ip(p)->ip_src.s_addr = local_addr;
	if (__likely(type != IGMP_V2_LEAVE_GROUP)) ip(p)->ip_dst.s_addr = mr->mreq.imr_multiaddr.s_addr;
	else ip(p)->ip_dst.s_addr = htonl(INADDR_ALLRTRS_GROUP);
	*ipopt(p) = htonl(ROUTER_ALERT_OPTION);
	*(__u32 *)&igmpopt(p)->igmp_type = __32CPU2LE(type);
	p->checksum.u = MKCHECKSUM(sizeof(struct ip) + 4, sizeof(struct ip) + 4 + 2);
	p->flags |= PKT_OUTPUT_CHECKSUM | PKT_TCPUDP_CHECKSUM_OK;
	igmpopt(p)->igmp_group.s_addr = mr->mreq.imr_multiaddr.s_addr;
	IP_SEND_PACKET(p);
}

static void IGMP_TIMER_EXPIRE(TIMER *t)
{
	MCAST_RECORD *mr = GET_STRUCT(t, MCAST_RECORD, timer);
	LOWER_SPL(SPL_NET);
	mr->timer.fn = NULL;
	if (__likely(mr->sent_report)) {
		mr->was_last = 1;
		IGMP_SEND_REPORT(mr, IGMP_REPORT_TYPE(mr));
	}
}

static void INVALID_IGMP_PACKET(PACKET *p)
{
	static char a1[16];
	static char a2[16];
	static char a3[16];
	if (errorlevel >= 1)
		KERNEL$SYSLOG(__SYSLOG_NET_ERROR, net_name, "INVALID IGMP PACKET: IP LENGTH %X, SRC %s, DST %s, TYPE %02X, CODE %02X, GROUP %s", p->data_length, PRINT_IP(a1, ip(p)->ip_src.s_addr), PRINT_IP(a2, ip(p)->ip_dst.s_addr), igmp(p)->igmp_type, igmp(p)->igmp_code, PRINT_IP(a3, igmp(p)->igmp_group.s_addr));
}

void IGMP_PACKET(PACKET *p)
{
	MCAST_RECORD *mr;
	if (igmp(p)->igmp_type == IGMP_MEMBERSHIP_QUERY) {
		unsigned ticks;
		jiffies_t j = KERNEL$GET_JIFFIES();
		if (!igmp(p)->igmp_code) ticks = JIFFIES_PER_SECOND * 10;
		else ticks = (int)igmp(p)->igmp_code * JIFFIES_PER_SECOND / 10;
		XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (__likely(mr->mcast_ioctl.h != -1) && __likely(mr->type.id == p->id)) {
			if (__likely(igmp(p)->igmp_group.s_addr == htonl(INADDR_ANY)) || mr->mreq.imr_multiaddr.s_addr == igmp(p)->igmp_group.s_addr) {
				IGMP_SET_TIMER(mr, ticks, 1);
			}
			if (!igmp(p)->igmp_code) {
				mr->v1_time = j;
				mr->v1_heard = 1;
			} else {
				mr->v2_time = j;
				mr->v2_heard = 1;
			}
		}
	} else if (igmp(p)->igmp_type == IGMP_V1_MEMBERSHIP_REPORT || igmp(p)->igmp_type == IGMP_V2_MEMBERSHIP_REPORT) {
		if (__likely(!igmp(p)->igmp_code)) goto drop;
		if (__unlikely(ip(p)->ip_dst.s_addr != igmp(p)->igmp_group.s_addr)) {
			INVALID_IGMP_PACKET(p);
			goto drop;
		}
		XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (__likely(mr->mcast_ioctl.h != -1) && __likely(mr->type.id == p->id) && mr->mreq.imr_multiaddr.s_addr == igmp(p)->igmp_group.s_addr) {
			mr->was_last = 0;
			IGMP_CANCEL_TIMER(mr);
		}
	}
	drop:
	p->status = 0;
	CALL_PKT(p);
}

void IGMP_CANCEL_IF(int h)
{
	MCAST_RECORD *mr;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_NET))) KERNEL$SUICIDE("IGMP_CANCEL_IF AT SPL %08X", KERNEL$SPL);

	test_again:
	XLIST_FOR_EACH(mr, &mcast_canceling, MCAST_RECORD, list) if (mr->mcast_ioctl.h == h) {
		wait_test_again:
		KERNEL$SLEEP(1);
		goto test_again;
	}
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (mr->mcast_ioctl.h == h) {
		mr->sent_report = 0;
		mr->was_last = 0;
		if (mr->posted) {
			mr->need_if_chg = 1;
			KERNEL$CIO((IORQ *)&mr->mcast_ioctl);
			goto wait_test_again;
		}
		mr->mcast_ioctl.h = -1;
	}
}

void IGMP_CANCEL_IF_ASYNC(int h)
{
	MCAST_RECORD *mr;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_NET))) KERNEL$SUICIDE("IGMP_CANCEL_IF_ASYNC AT SPL %08X", KERNEL$SPL);
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (mr->mcast_ioctl.h == h) {
		if (mr->posted) {
			mr->need_if_chg = 1;
			KERNEL$CIO((IORQ *)&mr->mcast_ioctl);
		} else {
			mr->sent_report = 0;
			mr->was_last = 0;
		}
	}
}

void IGMP_NOTIFY_LINK_CHANGE(int h)
{
	MCAST_RECORD *mr;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_NET))) KERNEL$SUICIDE("IGMP_NOTIFY_LINK_CHANGE AT SPL %08X", KERNEL$SPL);
	XLIST_FOR_EACH(mr, &mcast_records, MCAST_RECORD, list) if (mr->mcast_ioctl.h == h) {
		IGMP_CANCEL_TIMER(mr);
		IGMP_SET_TIMER(mr, IGMP_LINKCHG_TIME, 0);
	}
}

void IGMP_INIT(void)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_NET))) KERNEL$SUICIDE("IGMP_INIT AT SPL %08X", KERNEL$SPL);
	KERNEL$SLAB_INIT(&mcast_slab, sizeof(MCAST_RECORD), 0, VM_TYPE_WIRED_MAPPED, NULL, NULL, &NET$MEMORY_AVAIL, "TCPIP$MCAST_RECORD");
}

void IGMP_DONE(void)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_NET))) KERNEL$SUICIDE("IGMP_DONE AT SPL %08X", KERNEL$SPL);
	while (__unlikely(!XLIST_EMPTY(&mcast_canceling))) KERNEL$SLEEP(1);
	if (__unlikely(!XLIST_EMPTY(&mcast_records)))
		KERNEL$SUICIDE("IGMP_DONE: MCAST RECORDS LEAKED");
	KERNEL$SLAB_DESTROY(&mcast_slab);
}

