#ifndef _TCPIP_TCPIP_H
#define _TCPIP_TCPIP_H

#include <SPAD/SOCKET.H>
#include <SPAD/PKT.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/WQ.H>
#include <SPAD/TIMER.H>
#include <SPAD/LIST.H>
#include <SYS/UTSNAME.H>

#include <NETINET/IN.H>
#include <NETINET/IP.H>
#include <NETINET/IP_ICMP.H>
#include <NETINET/TCP.H>
#include <NETINET/UDP.H>
#include <NETINET/ARP.H>
#include <NETINET/DHCP.H>
#include <NETINET/IGMP.H>

#include "SETUP.H"

typedef struct __tcpip_socket TCPIP_SOCKET;

#define SOCK_ECN		0x00000001
#define SOCK_ECE		0x00000002
#define SOCK_CWR		0x00000004
#define SOCK_ECE_IGNORE		0x00000008
#define SOCK_SACK		0x00000010
#define SOCK_WINSCALE		0x00000020
#define SOCK_DUP_ACK		0x00000040
#define SOCK_DUP_ACK_2		0x00000080
#define SOCK_FAST_RECOVERY	0x00000100
#define SOCK_SS_AFTER_RTO	0x00000200
#define SOCK_SHUTDOWN_READ	0x00000400
#define SOCK_SHUTDOWN_WRITE	0x00000800
#define SOCK_FIN_ACKED		0x00001000
#define SOCK_SHOULD_SEND_MORE	0x00002000
#define SOCK_TOS_SET		0x00004000
#define SOCK_NETACL_ACCEPT_IN	0x00008000	/* for TCP, only this is used */
#define SOCK_NETACL_ACCEPT_OUT	0x00010000
#define SOCK_NETACL_TCP_ACCEPT	0x00020000

#define SOCK_NONAGLE		0x01000000
#define SOCK_CORK		0x02000000

#define PKT_OUTSTANDING		0x80000000 /* used by stack, packet is sent */
#define PKT_SACKED		0x40000000 /* used by TCP/IP, SACK */
#define PKT_RETRANSMITTED	0x20000000
#define PKT_FIN			0x10000000
#define PKT_OVER_WINDOW		0x08000000

struct __tcpip_socket {
	SOCKET_HEAD
	LIST_ENTRY hash;
	in_addr_t local_addr;
	in_addr_t remote_addr;
	in_port_t local_port;
	in_port_t remote_port;
	unsigned short mss;
	__u8 ip_tos;
	__u8 mcast_ttl;
	unsigned flags;
	unsigned seq;
	unsigned seq_acked;
	unsigned ack;
	u_jiffies_lo_t cwr_time;
	unsigned read_seq;
	void (*packet)(TCPIP_SOCKET *, PACKET *p);
	unsigned sent_queue_length;
	unsigned out_queue_length;
	unsigned wnd;
	unsigned cwnd;
	unsigned ssthresh;
	unsigned wnd_scale;
	unsigned bytes_acked;
	int offered_window;
	unsigned read_rate;
	unsigned write_rate;
	u_jiffies_lo_t read_rate_time;
	u_jiffies_lo_t write_rate_time;
	LIST_HEAD sent_queue;
	LIST_HEAD out_queue;
	LIST_HEAD in_queue;
	LIST_HEAD in_ooo_queue;
	PACKET *prepared_packet;
	WQ read_wait;
	WQ write_wait;
	int conn_error;
	u_jiffies_lo_t srtt;
	u_jiffies_lo_t rttvar;
	u_jiffies_lo_t rto;
	TIMER timer;
	u_jiffies_lo_t last_time;
	LIST_HEAD backlog_preparing;
	LIST_HEAD backlog_connected;
	LIST_ENTRY backlog_list;
	TCPIP_SOCKET *listener;
	u_jiffies_lo_t linger_start;
	WQ linger_wait;
	TIMER linger_timer;
	in_addr_t multicast_src;
	unsigned n_groups;
	struct ip_mreq groups[SOCKET_MCAST_GROUPS];
	unsigned long used_size;	/* used only during oom killing */
};

/* IPMAIN.C */

extern struct utsname utsname;
extern char *__const__ net_name;
extern int errorlevel;
extern int local_security;

#ifdef __IP_LOG
void iplog(PACKET *p);
#else
#define iplog(x)	do { } while (0)
#endif

/* IPRND.C */

unsigned RANDOM_TCP_SEQ(TCPIP_SOCKET *s);
unsigned RANDOM_DHCP_XID(__u32 *a);
int RANDOM_TCP_INIT(void);
void RANDOM_TCP_DONE(void);

/* IPSOCKET.C */

extern __const__ HANDLE_OPERATIONS TCP_OPS;
extern __const__ HANDLE_OPERATIONS UDP_OPS;

extern LIST_HEAD LOST_PACKETS;

#define LS_TCP	(&TCP_OPS)
#define LS_UDP	(&UDP_OPS)
TCPIP_SOCKET *LOOKUP_SOCKET(in_addr_t local_addr, unsigned local_port, in_addr_t remote_addr, unsigned remote_port, __const__ HANDLE_OPERATIONS *proto);
#define LL_TCP	0
#define LL_UDP	IPPORT_MAX
TCPIP_SOCKET *LOOKUP_LISTEN(in_addr_t local_addr, unsigned local_port, unsigned proto);

void INSERT_INTO_HASH(TCPIP_SOCKET *s);

int CONNECT_BIND(TCPIP_SOCKET *s, int u);

void TCPIP_SOCKETS_INIT(void);
void TCPIP_SOCKETS_DONE(void);
void TCPIP_SOCKETS_CHECK(void);
int TCPIP_SOCKETS_OUSTANDING_PACKETS(int h);

void TCPIP_SOCKET_CTOR(SOCKET *s_);
int TCPIP_INIT_SOCKET(SOCKET *s_, char *opt);
void TCPIP_DESTROY_SOCKET(SOCKET *s_);
int TCPIP_CLOSE_SOCKET(SOCKET *s_, IORQ *rq);
void TCPIP_DUP_SOCKET(SOCKET *s_, SOCKET *os_);
void TCP_RESET_SOCKET(TCPIP_SOCKET *s);
void TCP_END_SOCKET(TCPIP_SOCKET *s);

extern IO_STUB TCP_IOCTL;
extern IO_STUB UDP_IOCTL;

int TCPIP_LINGER(SOCKET *s_, IORQ *rq);

int TCPIP_IOCTL(IOCTLRQ *rq);

/* IPKILL.C */

void TCPIP_DELETE_OFFENSIVE_SOCKETS(LIST_HEAD *socket_list);

/* IPNETACL.C */

void *TCPIP_PARSE_NETACL(SOCKET_NODE *sn, char *str, NETACL **netacl);
void *TCPIP_COPY_NETACL(SOCKET_NODE *sn, NETACL **dest, NETACL *src);
void TCPIP_FREE_NETACL(NETACL *netacl);
void NETACL_INIT(void);
void NETACL_DONE(void);

#define NETACL_ALLADDR		1
#define NETACL_ALL		2
#define NETACL_DEFAULT_RESTRICT	4
#define NETACL_PORT_UDP		IPPORT_MAX
#define NETACL_PORT_IN		(2 * IPPORT_MAX)

int NETACL_SEARCH(SOCKET_NODE *sn, in_addr_t addr, unsigned port);

/* IPIF.C */

typedef struct __dhcp DHCP;

#define TOS_VALID(v)							\
	(!(								\
	__unlikely(((v) & (IPTOS_LOWDELAY | IPTOS_THROUGHPUT)) == (IPTOS_LOWDELAY | IPTOS_THROUGHPUT)) ||						\
	__unlikely(((v) & (IPTOS_LOWDELAY | IPTOS_RELIABILITY)) == (IPTOS_LOWDELAY | IPTOS_RELIABILITY)) ||						\
	__unlikely(((v) & (IPTOS_LOWDELAY | IPTOS_RELIABILITY)) == (IPTOS_LOWDELAY | IPTOS_RELIABILITY)) ||						\
	__unlikely((v) & (IPTOS_CE | IPTOS_ECT))			\
	))

extern int IFF_GATE;

int GET_IP(char *str, in_addr_t *addr, char **end);
int IS_NETMASK(in_addr_t a);
char *PRINT_IP(char str[16], in_addr_t a);
char *PRINT_OPTIONS(char str[__MAX_STR_LEN], __u8 *opt, int len);
in_addr_t IP_DEFAULT_MULTICAST_IF(void);
int IP_IF_HANDLE(in_addr_t addr);
IF_TYPE *IP_IF_TYPE(in_addr_t addr);
int IP_IF_IGMP_VERSION(in_addr_t addr);
void *TCPIP_BINDABLE_ADDRESS(in_addr_t addr);	/* nonzero --- this is address of local interface */
in_addr_t IP_FIND_LOCAL_ADDRESS(in_addr_t addr);
in_addr_t IP_FIND_ROUTE(in_addr_t to);
int IP_FIND_MTU(in_addr_t addr);
int IP_IS_BROADCAST(in_addr_t addr);
int IP_SEND_PACKET(PACKET *p);
int IP_ROUTE_PACKET(PACKET *p);
int IF_SEND_ARP(in_addr_t host, PACKET *p, __u8 *reply, int rlen);
int ARP_CHECK_SECURITY(PACKET *p);
int CHECK_ARP_PACKET(PACKET *p);
void IF_SEND_DHCP_PACKET(DHCP *dhcp, PACKET *p);

int TCPIP_IFCONFIG(char *argv[]);
int TCPIP_ROUTE(char *argv[]);
int TCPIP_GATE(char *argv[]);
void TCPIP_IF_SHUTDOWN(void);

/* IPDHCP.C */

extern __const__ unsigned DHCP_STRUCT_SIZE;

extern __u8 DHCP_APPARRAY[256][256];
extern DHCP *DHCP_APPOWNER[256];

void DHCP_INIT(DHCP *dhcp, IF_TYPE *if_type, int handle, int request_addr, in_addr_t addr, int only_bootp, int secure, int distrust);
void DHCP_DESTROY(DHCP *dhcp);
void DHCP_INPUT(PACKET *p);
void DHCP_NOTIFY_LINK_CHANGE(DHCP *dhcp);

char *DHCP_MODE(DHCP *dhcp);
char *DHCP_STATUS(DHCP *dhcp);

int IF_DHCP_SETUP(DHCP *dhcp, in_addr_t addr, in_addr_t netmask, int valid_netmask, in_addr_t broadcast, int valid_broadcast);
void IF_DHCP_SETUP_GW(DHCP *dhcp, in_addr_t gw);
void IF_DHCP_SETUP_ROUTE(DHCP *dhcp, in_addr_t dst, in_addr_t mask, in_addr_t gw);
void IF_DHCP_UNSET(DHCP *dhcp);

/* IPINPUT.C */

void IP_INPUT(PACKET *p);
void INVALID_IP_OPTIONS(PACKET *p);
void INVALID_TCP_PACKET(PACKET *p);

/* IPARP.C */

extern unsigned ARP_TIME_TO_LIVE;

__u8 *ARP_FIND(in_addr_t host, PACKET *p, int (*call)(PACKET *p));
void ARP_INPUT(PACKET *p);
void ARP_UPDATE(PACKET *p);
int ARP_INIT(void);
void ARP_DONE(void);

/* IPFRAG.C */

extern __u16 IP_MAX_DEFRAG;

void IP_FRAGMENT(PACKET *p, unsigned mtu, long *outstanding);
void IP_DEFRAGMENT(PACKET *p);
void IP_FRAG_INIT(void);
void IP_FRAG_DONE(void);

/* IPICMP.C */

void ICMP_SEND(PACKET *op, __u32 typecode, __u32 extra);
void ICMP_PACKET(PACKET *p);

/* IPIGMP.C */

void IGMP_PACKET(PACKET *p);
int IGMP_GET_MCAST_RECORD(struct ip_mreq *mreq, IORQ *rq);
void IGMP_RELEASE_MCAST_RECORD(struct ip_mreq *mreq);
void IGMP_CHECK_MULTICASTS(void);
void IGMP_CANCEL_IF(int h);
void IGMP_CANCEL_IF_ASYNC(int h);
void IGMP_NOTIFY_LINK_CHANGE(int h);
void IGMP_INIT(void);
void IGMP_DONE(void);

/* IPTCP.C */

void INIT_TCP_SOCKET(TCPIP_SOCKET *s);
void TCP_SEND_SOCKET_RESET(TCPIP_SOCKET *s);
void TCP_NULL_PACKET(TCPIP_SOCKET *s, PACKET *p);
void TCP_NULL_PACKET_1(TCPIP_SOCKET *s, PACKET *p);
void TCP_LISTEN(TCPIP_SOCKET *s, PACKET *p);
void TCP_SYN_SENT(TCPIP_SOCKET *s, PACKET *p);
void TCP_ESTABLISHED(TCPIP_SOCKET *s, PACKET *p);
void TCP_SEND_SYN(TCPIP_SOCKET *s);
extern IO_STUB TCP_READ;
extern IO_STUB TCP_WRITE;
extern IO_STUB TCP_RECVMSG;
extern IO_STUB TCP_SENDMSG;
int TCP_CLOSE(TCPIP_SOCKET *s, int type, IORQ *rq);
int TCP_SEND_MORE(TCPIP_SOCKET *s);
void TCPIP_DELETE_PACKET(TCPIP_SOCKET *s, PACKET *p);

/* IPUDP.C */

void UDP_NULL_PACKET(PACKET *p);
void UDP_PACKET(TCPIP_SOCKET *s, PACKET *p);
extern IO_STUB UDP_READ;
extern IO_STUB UDP_WRITE;
extern IO_STUB UDP_RECVMSG;
extern IO_STUB UDP_SENDMSG;

static __finline__ void SET_TIMEOUT(TCPIP_SOCKET *s)
{
#if __DEBUG >= 1
	if (__unlikely(s->timer.fn != NULL))
		KERNEL$SUICIDE("SET_TIMEOUT: TIMEOUT ALREADY SET");
#endif
	s->last_time = KERNEL$GET_JIFFIES_LO();
	s->conn_error = -ETIMEDOUT;
}

static __finline__ void RESET_TIMEOUT(TCPIP_SOCKET *s)
{
	KERNEL$DEL_TIMER(&s->timer);
	s->timer.fn = NULL;
}

static __finline__ unsigned TCP_READ_BUFFER(TCPIP_SOCKET *s)
{
	unsigned l;
	if (__likely(!s->sock_rcvbuf)) {
		unsigned long ma;
		l = s->read_rate;
		ma = (unsigned long)NET$MEMORY_AVAIL << (__PAGE_CLUSTER_BITS - TCP_READ_BUFFER_FRACT_BITS);
		if (__unlikely(l > ma)) l = ma;
		if (__unlikely(l < TCP_INIT_WINDOW)) l = TCP_INIT_WINDOW;
	} else {
		l = s->sock_rcvbuf;
		if (__unlikely(l < s->mss)) l = s->mss;
	}
	if (__unlikely(l < (1 << TCP_WINDOW_SCALE))) l = 1 << TCP_WINDOW_SCALE;
	else if (l >= TCP_MAX_WINDOW) l = TCP_MAX_WINDOW - 1;
	return l;
}

static __finline__ unsigned TCP_WRITE_BUFFER(TCPIP_SOCKET *s)
{
/* !!! FIXME: NET$MEMORY_AVAIL may change and this confuses select !!! */
	unsigned l;
	if (__likely(!s->sock_sndbuf)) {
		unsigned long ma;
		l = s->write_rate;
		ma = (unsigned long)NET$MEMORY_AVAIL << (__PAGE_CLUSTER_BITS - TCP_WRITE_BUFFER_FRACT_BITS);
		if (__unlikely(l > ma)) l = ma;
		if (__unlikely(l < s->mss << 2)) l = s->mss << 2;
	} else {
		l = s->sock_sndbuf;
		if (__unlikely(l < s->mss)) l = s->mss;
	}
	if (__unlikely(l > TCP_MAX_WRITE_BUFFER)) return TCP_MAX_WRITE_BUFFER;
	return l;
}

#define TCP_CANT_WRITE(s) ((s)->sent_queue_length + (s)->out_queue_length > TCP_WRITE_BUFFER(s))

#define ip(p)		((struct ip *)(&p->data[LL_HEADER]))
#define ipopt(p)	((__u32 *)(ip(p) + 1))
#define icmp(p)		((struct icmp *)(ip(p) + 1))
#define tcp(p)		((struct tcphdr *)(ip(p) + 1))
#define tcpopt(p)	((__u8 *)(tcp(p) + 1))
#define udp(p)		((struct udphdr *)(ip(p) + 1))
#define arp(p)		((struct arphdr *)(&p->data[LL_HEADER]))
#define arpaddr(p)	((__u8 *)(arp(p) + 1))
#define dhcp(p)		((struct dhcp *)(&p->data[LL_HEADER]))
#define igmpopt(p)	((struct igmp *)(ipopt(p) + 1))
#define igmp(p)		((struct igmp *)(ip(p) + 1))

#define TCPIP_MULTICAST_ADDRESS(a)			(((a) & htonl(0xf0000000U)) == htonl(0xe0000000U))
#define TCPIP_MULTICAST_LOCAL_ADDRESS(a)		(((a) & htonl(0xffffff00U)) == htonl(0xe0000000U))	/* it is required that this includes htonl(INADDR_ALLHOSTS_GROUP) ... otherwise crash will happen */
#define TCPIP_MULTICAST_OR_EXPERIMENTAL_ADDRESS(a)	(((a) & htonl(0xe0000000U)) == htonl(0xe0000000U))
#define TCPIP_EXPERIMENTAL_ADDRESS(a)			(((a) & htonl(0xf0000000U)) == htonl(0xf0000000U))

#define INAPPROPRIATE_TCP_ADDRESS(a)			(__unlikely(TCPIP_MULTICAST_OR_EXPERIMENTAL_ADDRESS(a)) || __unlikely(a == htonl(INADDR_ANY)))
#define INAPPROPRIATE_UDP_ADDRESS(a)			((__unlikely(TCPIP_EXPERIMENTAL_ADDRESS(a)) && a != htonl(0xffffffff)) || __unlikely(a == htonl(INADDR_ANY)))
#define INAPPROPRIATE_IF_ADDRESS(a)			INAPPROPRIATE_TCP_ADDRESS(a)

#endif
