#include <SPAD/LIBC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/SYSLOG.H>
#include <VALUES.H>
#include <ERRNO.H>
#include <ARCH/IO.H>
#include <ARCH/IRQ.H>
#include <SPAD/TIMER.H>
#include <SPAD/PKT.H>

#include <SPAD/PIO.H>
#include "PIOREG.H"

#include "PIO.H"

struct pio {
	io_t io[2];
	AST irq_ast;
	IRQ_CONTROL irq_ctrl;
	unsigned char flags;

	__u8 ctl;

	MTX irq_event;

	struct client *active_client;
	LIST_HEAD clients;
	int n_clients;
	int n_want;

	AST find_new_active_client;

	long in_dcall;

	void *lnte;
	void *dlrq;
	IO_RANGE range1;
	IO_RANGE range2;
	IO_RANGE range3;
	char dev_name[__MAX_STR_LEN];
};

struct client {
	LIST_ENTRY list;
	struct pio_client *client;
	unsigned char lock_mode;
	unsigned char want_it;
	MTX activate_event;
};

#define PIO_HAS_REVERSE	0x00000001
#define PIO_HAS_EPP	0x00000002
#define PIO_HAS_ECR	0x00000004

static __finline__ __u8 pio_in(struct pio *pio, unsigned port)
{
	return io_inb(pio->io[port / PIO_ECP_PORTS] + (port & (PIO_ECP_PORTS - 1)));
}

static __finline__ void pio_out(struct pio *pio, unsigned port, __u8 val)
{
	io_outb(pio->io[port / PIO_ECP_PORTS] + (port & (PIO_ECP_PORTS - 1)), val);
}

static void pio_set_ctl(struct pio *pio, unsigned mask, unsigned ctl)
{
	unsigned newctl;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
#if __DEBUG >= 1
	if (__unlikely(ctl & ~mask))
		KERNEL$SUICIDE("pio_set_ctl: INVALID VALUE %02X, MASK %02X", ctl, mask);
#endif
	newctl = (pio->ctl & ~mask) | ctl;
	if (__unlikely(pio->ctl == newctl)) goto ret;
	pio_out(pio, PIO_CTL, newctl);
	pio->ctl = newctl;
	ret:
	LOWER_SPLX(spl);
}

static int epp_clear_timeout(struct pio *pio)
{
	unsigned v;
	if (!(pio_in(pio, PIO_STATUS) & PIO_STATUS_EPP_TIMEOUT)) return 1;
	pio_in(pio, PIO_STATUS);
	v = pio_in(pio, PIO_STATUS);
	pio_out(pio, PIO_STATUS, v | PIO_STATUS_EPP_TIMEOUT);
	pio_out(pio, PIO_STATUS, v & ~PIO_STATUS_EPP_TIMEOUT);
	return !(v & PIO_STATUS_EPP_TIMEOUT);
}

static void ecr_setbits(struct pio *pio, __u8 mask, __u8 bits)
{
	pio_out(pio, PIO_ECR, (pio_in(pio, PIO_ECR) & ~mask) | bits);
}

static void probe_reverse(struct pio *pio)
{
	pio_out(pio, PIO_CTL, PIO_CTL_INIT | PIO_CTL_SLCT | PIO_CTL_BIDI);
	pio_out(pio, PIO_DATA, 0xaa);
	if (pio_in(pio, PIO_DATA) != 0xaa) pio->flags |= PIO_HAS_REVERSE;
	pio_out(pio, PIO_DATA, 0x55);
	if (pio_in(pio, PIO_DATA) != 0x55) pio->flags |= PIO_HAS_REVERSE;
	pio_out(pio, PIO_CTL, PIO_CTL_INIT | PIO_CTL_SLCT);
}

static void probe_epp(struct pio *pio)
{
	if (!epp_clear_timeout(pio)) return;
	pio->flags |= PIO_HAS_EPP;
}

static void probe_ecr(struct pio *pio)
{
	pio_out(pio, PIO_CTL, PIO_CTL_INIT | PIO_CTL_SLCT);
	if ((pio_in(pio, PIO_ECR) & (PIO_ECR_FIFO_EMPTY | PIO_ECR_FIFO_FULL)) != PIO_ECR_FIFO_EMPTY)
		return;
	pio_out(pio, PIO_ECR, 0x34);
	if (pio_in(pio, PIO_ECR) != 0x35)
		return;
	pio->flags |= PIO_HAS_ECR;
	ecr_setbits(pio, PIO_ECR_MODE, PIO_ECR_MODE_STANDARD);
}

DECL_AST(pio_irq, SPL_TOP, AST)
{
	struct pio *pio = GET_STRUCT(RQ, struct pio, irq_ast);
	if (__unlikely(!(pio->ctl & PIO_CTL_IRQ))) goto not_for_us;
	pio_set_ctl(pio, PIO_CTL_IRQ, 0);
	MTX_SET(&pio->irq_event, 0);
	if (__likely(pio->active_client != NULL) && __unlikely(pio->active_client->client->fast_irq_upcall != NULL)) pio->active_client->client->fast_irq_upcall(pio->active_client->client);
	not_for_us:
	pio->irq_ctrl.eoi();
	RETURN;
}

#if __DEBUG >= 1
static __finline__ void ASSERT_ACTIVE(struct pio *pio, struct pio_client *pc, char *fn)
{
	if (__unlikely(pio->active_client != pc->opaque_2))
		KERNEL$SUICIDE("%s: CALLING ON INACTIVE PIO CLIENT, PIO IS %s", fn, pio->active_client ? "ACQUIRED BY ANOTHER CLIENT" : "FREE");
}
#else
#define ASSERT_ACTIVE(pio, pc, fn) do { } while (0)
#endif

static int client_set_mode(struct pio_client *pc, unsigned mode)
{
	struct pio *pio = pc->opaque;
	ASSERT_ACTIVE(pio, pc, "client_set_mode");
/* More will be supported when I'll have some device to test it */
	if (__unlikely((mode & 0xff) != PIO_ECR_MODE_STANDARD)) return -EOPNOTSUPP;
	if (pio->flags & PIO_HAS_ECR) {
		ecr_setbits(pio, PIO_ECR_MODE, mode);
	}
	return 0;
}

static __u8 client_get_status(struct pio_client *pc)
{
	struct pio *pio = pc->opaque;
	ASSERT_ACTIVE(pio, pc, "client_get_status");
	return (pio_in(pio, PIO_STATUS) & (PIO_STATUS_ERROR | PIO_STATUS_SLCT | PIO_STATUS_PE | PIO_STATUS_ACK | PIO_STATUS_BUSY)) ^ PIO_STATUS_BUSY;
}

static void client_set_ctl(struct pio_client *pc, __u8 ctl)
{
	struct pio *pio = pc->opaque;
	ASSERT_ACTIVE(pio, pc, "client_set_ctl");
	pio_set_ctl(pio, PIO_CTL_STROBE | PIO_CTL_AUTO_LF | PIO_CTL_INIT | PIO_CTL_SLCT, ctl ^ (PIO_CTL_STROBE | PIO_CTL_AUTO_LF | PIO_CTL_SLCT));
}

static void client_set_data(struct pio_client *pc, __u8 data)
{
	struct pio *pio = pc->opaque;
	ASSERT_ACTIVE(pio, pc, "client_set_data");
	pio_out(pio, PIO_DATA, data);
}

static MTX *client_enable_irq(struct pio_client *pc)
{
	struct pio *pio = pc->opaque;
	ASSERT_ACTIVE(pio, pc, "client_enable_irq");
	MTX_SET(&pio->irq_event, 1);
	pio_set_ctl(pio, PIO_CTL_IRQ, PIO_CTL_IRQ);
	return &pio->irq_event;
}

struct wait_struct {
	TIMER t;
	MTX *m;
};

static void wait_timeout(TIMER *t)
{
	struct wait_struct *w = GET_STRUCT(t, struct wait_struct, t);
	VOID_LIST_ENTRY(&w->t.list);
	MTX_SET(w->m, 0);
}

static int client_wait_sync(struct pio_client *pc, __u8 status, __u8 mask, jiffies_lo_t timeout)
{
	u_jiffies_lo_t j, jj;
	int r;
	__u8 s;
	struct wait_struct w;
	ASSERT_ACTIVE(pc->opaque, pc, "client_wait_sync");
#if __DEBUG >= 1
	if (__unlikely(status & ~mask))
		KERNEL$SUICIDE("client_wait_sync: INVALID BITS: STATUS %02X, MASK %02X", status, mask);
#endif
	timeout = TIMEOUT_JIFFIES(timeout);
	j = KERNEL$GET_JIFFIES();
	test_status:
	s = client_get_status(pc);
	if (__likely((s & mask) == status)) return s;
	w.m = client_enable_irq(pc);
	jj = KERNEL$GET_JIFFIES();
	s = client_get_status(pc);
	if (__likely((s & mask) == status)) return s;
	if (__unlikely(jj - j > timeout)) {
		return s;
	}
	INIT_TIMER(&w.t);
	w.t.fn = wait_timeout;
	KERNEL$SET_TIMER(0, &w.t);
	r = MTX_WAIT_SYNC_CANCELABLE(w.m);
	KERNEL$DEL_TIMER(&w.t);
	if (__unlikely(r)) return r;
	goto test_status;
}

static int client_write_data_sync(struct pio_client *pc, __u8 *data, int len)
{
	int wrote;
	ASSERT_ACTIVE(pc->opaque, pc, "client_write_data_sync");
	wrote = 0;
	client_set_ctl(pc, PIO_CTL_INIT | PIO_CTL_AUTO_LF | PIO_CTL_STROBE);
	while (wrote < len) {
		int status = client_wait_sync(pc, PIO_STATUS_ERROR | PIO_STATUS_SLCT, PIO_STATUS_ERROR | PIO_STATUS_SLCT | PIO_STATUS_PE | PIO_STATUS_BUSY, JIFFIES_PER_SECOND / 28);
		if (__unlikely(status < 0)) return wrote ? wrote : status;
		if (__unlikely((status & (PIO_STATUS_ERROR | PIO_STATUS_SLCT | PIO_STATUS_PE | PIO_STATUS_BUSY)) != (PIO_STATUS_ERROR | PIO_STATUS_SLCT))) {
			if (wrote) return wrote;
			if (status & PIO_STATUS_BUSY) return -EBUSY;
			if (status & PIO_STATUS_PE) return -ENOSPC;
			if (!(status & PIO_STATUS_SLCT)) return -ENOLINK;
			return -EIO;
		}
		client_set_data(pc, data[wrote]);
		KERNEL$UDELAY(1);
		client_set_ctl(pc, PIO_CTL_INIT | PIO_CTL_AUTO_LF);
		KERNEL$UDELAY(1);
		client_set_ctl(pc, PIO_CTL_INIT | PIO_CTL_AUTO_LF | PIO_CTL_STROBE);
		KERNEL$UDELAY(1);
		wrote++;
	}
	return wrote;
}

static void set_want_it(struct pio *pio, struct client *c, int val)
{
	int diff;
#if __DEBUG >= 1
	if (KERNEL$SPL != SPL_X(SPL_TOP))
		KERNEL$SUICIDE("set_want_it AT SPL %08X", KERNEL$SPL);
#endif
	diff = val - c->want_it;
	c->want_it = val;
	if (c->lock_mode != PIO_CMD_ATTACH_SOMETIMES) return;
	pio->n_want += diff;
#if __DEBUG >= 1
	if (__unlikely(pio->n_want < 0))
		KERNEL$SUICIDE("set_want_it: n_want UNDERFLOW: %d", pio->n_want);
#endif
}

static MTX *client_acquire(struct pio_client *pc)
{
	struct pio *pio = pc->opaque;
	struct client *c = pc->opaque_2;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (__likely(pio->active_client == c)) {
		goto ret_z;
	}
	if (__likely(!pio->active_client) && !pio->find_new_active_client.fn) {
		set_want_it(pio, c, 0);
		pio->active_client = c;
		goto ret_z;
	}
	set_want_it(pio, c, 1);
	MTX_SET(&c->activate_event, 1);
	LOWER_SPLX(spl);
	MTX_SET(&pio->irq_event, 0);
	return &c->activate_event;

	ret_z:
	LOWER_SPLX(spl);
	return NULL;
}

extern AST_STUB find_new_active_client;

static void client_release(struct pio_client *pc)
{
	struct pio *pio = pc->opaque;
	struct client *c = pc->opaque_2;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	if (__unlikely(pio->active_client != c)) {
		if (__unlikely(!c->want_it))
			KERNEL$SUICIDE("client_release: ALREADY RELEASED");
		set_want_it(pio, c, 0);
		goto not_me;
	}
	if (__unlikely(c->want_it))
		KERNEL$SUICIDE("client_release: ACTIVE CLIENT HAS \"want_it\" SET");
	pio->active_client = NULL;
	if (!pio->find_new_active_client.fn) {
		pio->find_new_active_client.fn = find_new_active_client;
		CALL_AST(&pio->find_new_active_client);
	}
	not_me:
	LOWER_SPLX(spl);
}

DECL_AST(find_new_active_client, SPL_DEV, AST)
{
	struct client *c;
	struct pio *pio = GET_STRUCT(RQ, struct pio, find_new_active_client);
	unsigned i;
	RAISE_SPL(SPL_TOP);
	pio->find_new_active_client.fn = NULL;
	for (i = 0; i < pio->n_clients * 2; i++) {
		TEST_SPL(SPL_DEV, SPL_TOP);
		if (__unlikely(pio->active_client != NULL)) goto ret_lo;
		c = LIST_STRUCT(pio->clients.next, struct client, list);
		DEL_FROM_LIST(&c->list);
		ADD_TO_LIST_END(&pio->clients, &c->list);
		c = LIST_STRUCT(pio->clients.next, struct client, list);
		if (c->want_it && (c->lock_mode == PIO_CMD_ATTACH_SOMETIMES || i >= pio->n_clients)) {
			pio->active_client = c;
			set_want_it(pio, c, 0);
			LOWER_SPL(SPL_DEV);
			MTX_SET(&c->activate_event, 0);
			goto ret;
		}
	}
	ret_lo:
	LOWER_SPL(SPL_DEV);
	ret:
	RETURN;
}

static int client_want_release(struct pio_client *pc)
{
	struct pio *pio = pc->opaque;
	int r;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	ASSERT_ACTIVE(pio, pc, "client_want_release");
#if __DEBUG >= 1
	if (__unlikely(pio->n_want < 0))
		KERNEL$SUICIDE("client_want_release: n_want UNDERFLOW: %d", pio->n_want);
#endif
	r = pio->n_want;
	LOWER_SPLX(spl);
	return !!r;
}

static __const__ struct pio_operations pio_ops = {
	client_acquire,
	client_release,
	client_want_release,
	client_set_mode,
	client_get_status,
	client_set_ctl,
	client_set_data,
	client_enable_irq,
	client_wait_sync,
	client_write_data_sync,
};

int PIO_DCALL(void *ptr, __const__ char *dcall_type, int cmd, void *param)
{
	struct pio *pio = ptr;
	struct pio_client *pc = param;
	struct client *c, *cl;
	int r;
	pio->in_dcall++;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("pio_dcall AT SPL %08X", KERNEL$SPL);
#endif
	if (!strcmp(dcall_type, "PIO")) {
		MTX *m;
		int found, max_lock;
		c = NULL;	/* against warning */
		switch (cmd & PIO_CMD_MODE_MASK) {
			case PIO_CMD_ATTACH_PERMANENT:
			case PIO_CMD_ATTACH_MOST_TIME:
			case PIO_CMD_ATTACH_SOMETIMES:
				c = __sync_malloc(sizeof(struct client));
				if (__unlikely(!c)) {
					r = -ENOMEM;
					goto ret;
				}
		}
		found = 0;
		max_lock = 0;
		LIST_FOR_EACH(cl, &pio->clients, struct client, list) {
			if (cl == pc->opaque_2) {
				found = 1;
				break;
			}
			if (cl->lock_mode > max_lock) max_lock = cl->lock_mode;
		}
		if (__unlikely(cmd == PIO_CMD_DETACH != found))
			KERNEL$SUICIDE("PIO_DCALL: BAD CLIENT POITNER, CMD %d", cmd);
		switch (cmd & PIO_CMD_MODE_MASK) {
			case PIO_CMD_ATTACH_PERMANENT:
				if (max_lock) {
					busy:
					r = -EBUSY;
					free_ret:
					__slow_free(c);
					goto ret;
				}
				/* fall through */
			case PIO_CMD_ATTACH_MOST_TIME:
				if (max_lock > PIO_CMD_ATTACH_SOMETIMES) goto busy;
				/* fall through */
			case PIO_CMD_ATTACH_SOMETIMES:
				if (max_lock > PIO_CMD_ATTACH_MOST_TIME) goto busy;
				if (__unlikely(pc->size_of_pio_operations > sizeof(struct pio_operations) || __unlikely(pc->size_of_pio_client > sizeof(struct pio_client)))) {
					__slow_free(c);
					need_new:
					KERNEL$SYSLOG(__SYSLOG_SW_INCOMPATIBILITY, pio->dev_name, "THE DRIVER REQUIRES NEW VERSION OF PIO.SYS");
					r = -EINVAL;
					goto ret;
				}
				if (__unlikely(pio->n_clients == MAXINT)) {
					r = -EMFILE;
					goto free_ret;
				}
				c->client = pc;
				c->lock_mode = cmd & PIO_CMD_MODE_MASK;
				c->want_it = 0;
				MTX_INIT(&c->activate_event, "PIO$ACTIVATE_EVENT");
				pc->opaque = pio;
				pc->opaque_2 = c;
				pc->op = &pio_ops;
				pc->net_id = IDTYPE_IOPORT | (pio->io[0] & IDTYPE_MASK);
				pc->fast_irq_upcall = NULL;
				ADD_TO_LIST(&pio->clients, &c->list);
				pio->n_clients++;
				if (cmd & PIO_CMD_NOACQ_MASK) {
					r = 0;
					goto ret;
				}
				m = client_acquire(pc);
				if (__unlikely(m != NULL)) {
					r = MTX_WAIT_SYNC_CANCELABLE(m);
					if (__unlikely(r)) {
						client_release(pc);
						PIO_DCALL(pio, "PIO", PIO_CMD_DETACH, pc);
						goto ret;
					}
				}
				r = 0;
				goto ret;
			case PIO_CMD_DETACH:
				c = pc->opaque_2;
				RAISE_SPL(SPL_TOP);
				if (pio->active_client == c) {
					client_release(pc);
				}
				LOWER_SPL(SPL_DEV);
				DEL_FROM_LIST(&c->list);
				pio->n_clients--;
#if __DEBUG >= 1
				if (__unlikely(pio->n_clients < 0))
					KERNEL$SUICIDE("PIO_DCALL: CLIENT COUNT UNDERFLOW: %d\n", pio->n_clients);
#endif
				r = 0;
				goto ret;
			default:
				goto need_new;
		}
	}
	KERNEL$SUICIDE("PIO_DCALL: UNKNOWN DCALL TYPE %s", dcall_type);

	ret:
	pio->in_dcall--;
	if (__unlikely(pio->in_dcall < 0))
		KERNEL$SUICIDE("PIO_DCALL: DCALL UNDERFLOW: %ld", pio->in_dcall);
	return r;
}

void *PIO_LP(void *ptr)
{
	struct pio *p = ptr;
	return p + 1;
}

static int pio_unload(void *p, void **release, char *argv[]);
static void pio_check_unload(struct pio *pio);

int main(int argc, char *argv[])
{
	long io = -1;
	int irq = -1;
	int lpt = -1;
	char scan_ports = 0;
	struct __param_table params[] = {
		"IO", __PARAM_UNSIGNED_LONG, 0, IO_SPACE_LIMIT - 1, NULL,
		"IRQ", __PARAM_INT, 0, IRQ_LIMIT + 1, NULL,
		"LPT", __PARAM_INT, 1, BIOS_LPT_PORTS + 1, NULL,
		NULL, 0, 0, 0, NULL,
	};
	char **arg;
	int state;
	struct pio *pio;
	int r;
	char *e;
	MALLOC_REQUEST mrq;
	DEVICE_REQUEST devrq;
	params[0].__p = &io;
	params[1].__p = &irq;
	params[2].__p = &lpt;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PIO: SYNTAX ERROR");
		r = -EBADSYN;
		goto ret0;
	}

	mrq.size = sizeof(struct pio) + LP_SIZE;
	SYNC_IO_CANCELABLE(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status < 0)) {
		r = mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SIO: CAN'T ALLOC MEMORY");
		goto ret0;
	}
	pio = mrq.ptr;
	memset(pio, 0, sizeof(struct pio) + LP_SIZE);
	MTX_INIT(&pio->irq_event, "PIO$IRQ_EVENT");
	INIT_LIST(&pio->clients);

	if (io == -1) {
		if (lpt == -1) {
			lpt = 1;
			scan_ports = 1;
		}
		get_port:
		io = *(__u16 *)(KERNEL$ZERO_BANK + 0x408 + (lpt - 1) * 2);
		if (!io) {
			if (scan_ports) {
				scan_next:
				lpt++;
				if (lpt > BIOS_LPT_PORTS) {
					_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PIO: NO LPT PORTS FOUND");
					r = -ENODEV;
					goto ret1;
				}
				goto get_port;
			}
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PIO: PORT LPT%d NOT ENABLED IN BIOS", lpt);
			r = -ENODEV;
			goto ret1;
		}
	} else if (lpt != -1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PIO: CAN'T SPECIFY BOTH IO PORT AND LPT NUMBER");
		r = -EBADSYN;
		goto ret1;
	}

	if (lpt != -1) {
		_snprintf(pio->dev_name, sizeof pio->dev_name, "PIO$LPT%d", lpt);
	} else {
		_snprintf(pio->dev_name, sizeof pio->dev_name, "PIO$LPT@" IO_ID, (io_t)io);
	}
	pio->io[0] = io;

	pio->range1.start = pio->io[0];
	pio->range1.len = SPP_REGSPACE;
	pio->range1.name = pio->dev_name;
	if ((r = KERNEL$REGISTER_IO_RANGE(&pio->range1))) {
		pio->range1.name = NULL;
		if (scan_ports) goto scan_next;
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER IO RANGE AT "IO_FORMAT" - "IO_FORMAT": %s", pio->dev_name, pio->range1.start, pio->range1.start + pio->range1.len - 1, strerror(-r));
		goto ret1;
	}

	if (io <= IO_SPACE_LIMIT - PIO_ECP_MAXPORT) {
		pio->io[1] = io + PIO_ECP_PORTS;
		pio->range3.start = pio->io[1];
		pio->range3.len = ECP_REGSPACE;
		pio->range3.name = pio->dev_name;
		if (KERNEL$REGISTER_IO_RANGE(&pio->range3)) {
			pio->range3.name = NULL;
			goto skip_ecp;
		}
		probe_ecr(pio);
		if (!(pio->flags & PIO_HAS_ECR)) {
			KERNEL$UNREGISTER_IO_RANGE(&pio->range3);
			pio->range3.name = NULL;
		}
	}
	skip_ecp:

	if (pio->flags & PIO_HAS_ECR) {
		unsigned cfg_b;
		ecr_setbits(pio, PIO_ECR_MODE, PIO_ECR_MODE_CONFIGURATION);
		/*__debug_printf("cfg: %02x %02x\n", pio_in(pio, PIO_CFG_A), pio_in(pio, PIO_CFG_B));*/
		cfg_b = pio_in(pio, PIO_CFG_B);
		if (irq == -1) {
			switch (cfg_b & PIO_CFG_B_IRQ) {
				case PIO_CFG_B_IRQ_7: irq = 7; break;
				case PIO_CFG_B_IRQ_9: irq = 9; break;
				case PIO_CFG_B_IRQ_10: irq = 10; break;
				case PIO_CFG_B_IRQ_11: irq = 11; break;
				case PIO_CFG_B_IRQ_14: irq = 14; break;
				case PIO_CFG_B_IRQ_15: irq = 15; break;
				case PIO_CFG_B_IRQ_5: irq = 5; break;
			}
		}
		ecr_setbits(pio, PIO_ECR_MODE, PIO_ECR_MODE_STANDARD);
	}
	if (irq == -1) {
		if (io == 0x378 || io == 0x3bc) irq = 7;
		else if (io == 0x278) irq = 5;
		else {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "PIO: IRQ NOT SPECIFIED AND CAN'T BE DETERMINED AUTOMATICALLY");
			r = -ENODEV;
			goto ret1;
		}
	}

	if (!(pio->io[0] & 7)) {
		pio->range2.start = pio->io[0] + SPP_REGSPACE;
		pio->range2.len = EPP_REGSPACE;
		pio->range2.name = pio->dev_name;
		if (KERNEL$REGISTER_IO_RANGE(&pio->range2)) {
			pio->range2.name = NULL;
			goto skip_epp;
		}
		if (pio->flags & PIO_HAS_ECR) {
			ecr_setbits(pio, PIO_ECR_MODE, PIO_ECR_MODE_EPP);
			probe_epp(pio);
			ecr_setbits(pio, PIO_ECR_MODE, PIO_ECR_MODE_STANDARD);
		} else {
			probe_epp(pio);
		}
		if (!(pio->flags & PIO_HAS_EPP)) {
			KERNEL$UNREGISTER_IO_RANGE(&pio->range2);
			pio->range2.name = NULL;
		}
	}
	skip_epp:

	if (pio->flags & PIO_HAS_ECR) {
		ecr_setbits(pio, PIO_ECR_MODE, PIO_ECR_MODE_BYTE);
		probe_reverse(pio);
		ecr_setbits(pio, PIO_ECR_MODE, PIO_ECR_MODE_STANDARD);
	} else {
		probe_reverse(pio);
	}

	/* reset printer */
	pio_set_ctl(pio, ~0, PIO_CTL_SLCT);
	KERNEL$UDELAY(50);
	pio_set_ctl(pio, ~0, PIO_CTL_INIT | PIO_CTL_SLCT);

	pio->irq_ast.fn = pio_irq;
	if ((e = KERNEL$REQUEST_IRQ(irq, IRQ_AST_HANDLER, NULL, &pio->irq_ast, &pio->irq_ctrl, pio->dev_name))) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, pio->dev_name, "COULD NOT GET IRQ %d: %s", irq, e);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", pio->dev_name);
		r = -EINVAL;
		goto ret1;
	}

	_printf("%s: IO "IO_FORMAT, pio->dev_name, pio->io[0]);
	if (pio->flags & PIO_HAS_ECR) {
		_printf(", ECRIO "IO_FORMAT, pio->io[1]);
	}
	_printf(", IRQ %d\n", irq);
	_printf("%s: SPP", pio->dev_name);
	if (pio->flags & PIO_HAS_REVERSE)
		_printf(", REVERSE");
	if (pio->flags & PIO_HAS_EPP)
		_printf(", EPP");
	if (pio->flags & PIO_HAS_ECR)
		_printf(", ECP");
	_printf("\n");

	devrq.name = pio->dev_name;
	devrq.driver_name = "PIO.SYS";
	devrq.flags = 0;
	devrq.init_root_handle = PIO_INIT_ROOT;
	devrq.dev_ptr = pio;
	devrq.dcall = PIO_DCALL;
	devrq.dcall_type = "PIO";
	devrq.dctl = NULL;
	devrq.unload = pio_unload;
	SYNC_IO_CANCELABLE(&devrq, KERNEL$REGISTER_DEVICE);
	if (devrq.status < 0) {
		if (devrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", pio->dev_name);
		r = devrq.status;
		goto ret2;
	}
	pio->lnte = devrq.lnte;
	pio->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), pio->dev_name);
	return 0;

	/*__debug_printf("capabilities: %X\n", pio->flags);*/
	/*__debug_printf("io: %X, irq: %d\n", pio->io[0], irq);*/

	ret2:
	pio->irq_ctrl.mask();
	pio_set_ctl(pio, PIO_CTL_IRQ, 0);
	KERNEL$RELEASE_IRQ(&pio->irq_ctrl, IRQ_RELEASE_MASKED);
	ret1:
	pio_check_unload(pio);
	if (pio->range1.name) KERNEL$UNREGISTER_IO_RANGE(&pio->range1);
	if (pio->range2.name) KERNEL$UNREGISTER_IO_RANGE(&pio->range2);
	if (pio->range3.name) KERNEL$UNREGISTER_IO_RANGE(&pio->range3);
	KERNEL$UNIVERSAL_FREE(pio);
	ret0:
	return r;
}

static int pio_unload(void *p, void **release, char *argv[])
{
	int r;
	struct pio *pio = p;
	void *pc;
	RAISE_SPL(SPL_DEV);
	while (pio->in_dcall) KERNEL$SLEEP(1);
	if (pio->n_clients > 1 || (pio->n_clients == 1 && (pc = LIST_STRUCT(pio->clients.next, struct client, list)->client, pc < PIO_LP(pio) || pc >= (void *)((char *)PIO_LP(pio) + LP_SIZE)))) {
		LOWER_SPL(SPL_ZERO);
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: SOME CLIENTS ARE USING THIS PORT", pio->dev_name);
		return -EBUSY;
	}
	if ((r = KERNEL$DEVICE_UNLOAD(pio->lnte, argv))) {
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	LP_CANCEL(pio);
	pio->irq_ctrl.mask();
	pio_set_ctl(pio, PIO_CTL_IRQ, 0);
	KERNEL$RELEASE_IRQ(&pio->irq_ctrl, IRQ_RELEASE_MASKED);
	LOWER_SPL(SPL_ZERO);
	*release = pio->dlrq;
	pio_check_unload(pio);
	KERNEL$UNREGISTER_IO_RANGE(&pio->range1);
	if (pio->range2.name) KERNEL$UNREGISTER_IO_RANGE(&pio->range2);
	if (pio->range3.name) KERNEL$UNREGISTER_IO_RANGE(&pio->range3);
	KERNEL$UNIVERSAL_FREE(pio);
	return 0;
}

static void pio_check_unload(struct pio *pio)
{
	if (__unlikely(pio->n_clients) || __unlikely(pio->n_want) || __unlikely(!LIST_EMPTY(&pio->clients)) || __unlikely(pio->find_new_active_client.fn != NULL))
		KERNEL$SUICIDE("pio_check_unload: DATA LEAKED, n_clients %d, n_want %d, LIST_EMPTY(&clients) %d find_new_active_client %p", pio->n_clients, pio->n_want, LIST_EMPTY(&pio->clients), pio->find_new_active_client.fn);
}
